package com.atlassian.bonnie.index;

import com.atlassian.bonnie.ILuceneConnection;
import com.atlassian.core.util.ProgressMeter;
import com.atlassian.core.util.ProgressWrapper;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;

import java.io.IOException;

/**
 * Performs batch operations on an index.
 */
public interface BatchOpIndexer
{
    /**
     * Indexes objects in batch.
     *
     * @param queue    queue of objects to index
     * @param documentWritingScheme document writing scheme to use
     * @param progress progress meter indicating index progress
     * @param truncate should the original index be truncated?
     * @throws IOException
     */
    public void reindex(ObjectQueue queue, DocumentWritingScheme documentWritingScheme,
                        ProgressMeter progress, boolean truncate) throws IOException;

    /**
     * Truncate the index. Depending on implementations, this can mean deleting the documents one by one,
     * or other more efficient ways of removing documents from the index. Either way, this method results in an
     * empty (but valid) Lucene index.
     *
     * @throws IOException
     */
    void truncateIndex() throws IOException;

    /**
     * Get analyzer used for indexing.
     */
    Analyzer getAnalyzer();

    /**
     * <p>Writes documents. Possible implementations include
     * adding documents to a writer directly or adding them to a queue.</p>
     * <p>{@link ObjectQueue.Processor}s create Documents, and will pass them in to this interface for
     * writing.
     * </p>
     */
    public interface DocumentWritingScheme
    {
        /**
         * Write a created document.
         *
         * @param doc
         */
        void write(Document doc);

        /**
         * Signify that the {@link QueueProcessingRunnableImpl} has completed.
         */
        void runComplete();

        /**
         * Release any open resources.
         */
        void close(ILuceneConnection conn) throws IOException;

        void setProgressWrapper(ProgressWrapper progress);
    }
}
