package com.atlassian.bonnie.index;

import java.util.Collection;

/**
 * <p>Queue of objects that need to be indexed. Partitions the total collection of objects to be
 * indexed into sub-collections that individual threads can grab and process.</p>
 * <p>An ObjectQueue MUST be read-only and MUST NOT be externally modified once it has been created.</p>
 * <p>There is no need for implementations to be thread-safe. Synchronization will be handled by clients.</p>
 */
public interface ObjectQueue
{
    /**
     * Pops a block of objects to index. Returns a collection of objects instead of
     * a single object to minimize lock contention.
     *
     * @return objects to index
     */
    Collection pop();

    /**
     * Are there more objects in the queue?
     *
     * @return
     */
    boolean hasMore();

    /**
     * Size of queue. For Iterator-backed implementations, the size is not known.
     * Clients should only use this method to obtain an approximation of the actual
     * number of objects to index.
     *
     * @return
     */
    int size();

    /**
     * Get this queue's processor. Implementations must either return a thread-safe processor
     * or a thread-local instance of the processor.
     *
     * @return
     */
    Processor getProcessor();

    /**
     * <p>Creates Documents from objects via
     * {@link ObjectToDocumentConverter}, then passes the created Document to a Callback.</p>
     * <p>The premise of this class is that the creation of Documents (NOT their addition to the index) is the bottleneck
     * in an indexing process, and performance gains will be achieved by running this operation in a concurrent fashion.
     * </p>
     */
    interface Processor
    {
        void indexCollection(Collection collection, BatchOpIndexer.DocumentWritingScheme documentWritingScheme);
    }
}
