package com.atlassian.bonnie.index;

import com.atlassian.bonnie.ILuceneConnection;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexWriter;
import org.slf4j.LoggerFactory;
import org.slf4j.Logger;

import java.io.IOException;

/**
 * Each thread has its own IndexWriter which writes to a temp directory.
 */
public class TempDirectoryDocumentWritingScheme extends BaseDocumentWritingScheme
{
    private static final Logger log = LoggerFactory.getLogger(TempDirectoryDocumentWritingScheme.class);
    protected final TempIndexWriter tempIndexWriter;

    public TempDirectoryDocumentWritingScheme(BatchOpIndexer indexer)
    {
        this(indexer, null);
    }

    public TempDirectoryDocumentWritingScheme(BatchOpIndexer indexer, String tmpDir)
    {
        this.tempIndexWriter = new TempIndexWriter(indexer.getAnalyzer(), tmpDir);
    }

    public void write(final Document doc)
    {
        if (doc == null)
        {
            progress.incrementCounter();
            return;
        }

        String key = getWriterKey(doc);
        try
        {
            tempIndexWriter.addDocument(key, doc);
            progress.incrementCounter("Indexed: " + getDocumentTitle(doc) + " - " + progress.progressAsString());
        }
        catch (IOException e)
        {
            progress.incrementCounter("Error indexing: " + getDocumentTitle(doc) + " (" + e.toString() + ") - " + progress.progressAsString());
            log.error("Error encountered", e);
        }
    }

    protected String getWriterKey(Document doc)
    {
        return Thread.currentThread().getName();
    }

    protected String getDocumentTitle(Document doc)
    {
        return doc.get("title");
    }

    public void runComplete()
    {
        // noop.
    }

    /**
     * Merge the temp indices into the final index, deleting the temp directories.
     */
    public void close(final ILuceneConnection luceneConnection) throws IOException
    {
        luceneConnection.withBatchUpdate(new ILuceneConnection.BatchUpdateAction() {
            public void perform() throws Exception {
                luceneConnection.withWriter(new ILuceneConnection.WriterAction()
                {
                    public void perform(IndexWriter writer) throws IOException
                    {
                        tempIndexWriter.merge(writer);
                    }
                });
            }
        });
        // remove the temporary indexes
        tempIndexWriter.closeAll();
    }
}
