package com.atlassian.bonnie.index;

import com.atlassian.bonnie.BonnieConstants;
import com.atlassian.bonnie.ILuceneConnection;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.miscellaneous.LimitTokenCountAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.*;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.util.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Index writer that is able to write to one or more temporary indices. This writer is typically used to:
 * <ul>
 * <li>allow writing index changes to a single temporary location and merging back to the proper location when finished
 * <li>allow parallel construction of a index into multiple temporary indices, and merging all temporary indices back to the proper location.
 * </ul>
 */
public class TempIndexWriter
{
    private static final Logger log = LoggerFactory.getLogger(TempIndexWriter.class);

    private final Map writers = Collections.synchronizedMap(new HashMap());
    private final Analyzer analyzerForIndexing;
    private final String tmpDir;
    private final ILuceneConnection.Configuration configuration;

    public TempIndexWriter(Analyzer analyzerForIndexing, String tmpDir)
    {
        this(analyzerForIndexing, tmpDir, ILuceneConnection.DEFAULT_CONFIGURATION);
    }

    /**
     * Constructs a temp index writer.
     *
     * @param analyzerForIndexing analyzer used by {@link org.apache.lucene.index.IndexWriter}s created and managed by this temp index writer. Required parameter.
     * @param tmpDir directory where temporary directories will be created in. Required parameter.
     * @param configuration allows configuration of the {@link org.apache.lucene.index.IndexWriter}s created and managed
     * by this temp index writer. Only the batch settings will be used (as the assumption here is that
     * {@link com.atlassian.bonnie.index.TempIndexWriter}s will be used for batch indexing purposes only).
     * @throws IllegalArgumentException if analyzerForIndexing, tmpDir or configuration are null
     */
    public TempIndexWriter(Analyzer analyzerForIndexing, String tmpDir, ILuceneConnection.Configuration configuration)
    {
        if (analyzerForIndexing == null)
            throw new IllegalArgumentException("analyzerForIndexing is required.");
        if (tmpDir == null)
            throw new IllegalArgumentException("tmpDir is required.");
        if (configuration == null)
            throw new IllegalArgumentException("configuration is required.");

        this.analyzerForIndexing = analyzerForIndexing;
        this.tmpDir = tmpDir;
        this.configuration = configuration;
    }

    /**
     * Performs a {@link ILuceneConnection.WriterAction}
     * @param key key representing which index to write to. A new writer will be created if no writer exists for the key.
     * @param writerAction the index write action to perform
     */
    public void perform(String key, ILuceneConnection.WriterAction writerAction) throws IOException
    {
        writerAction.perform(getWriterData(key).writer);
    }

    /**
     * Add a document to an index.
     * @param key key representing which index to write to. A new writer will be created if no writer exists for the key.
     * @param doc document to add
     * @throws IOException
     */
    public void addDocument(String key, Document doc) throws IOException
    {
        getWriterData(key).writer.addDocument(doc);
    }

    private WriterData getWriterData(String key) throws IOException
    {
        synchronized (key.intern())
        {
            WriterData writerData = (WriterData) writers.get(key);
            if (writerData == null)
            {
                writerData = createData();
                writers.put(key, writerData);
            }
            return writerData;
        }
    }

    /**
     * Delete the respective temp indices.
     * @param prefix
     * @throws IOException
     */
    public void close(String prefix) throws IOException
    {
        for (Iterator it = writers.keySet().iterator(); it.hasNext();)
        {
            String key = it.next().toString();
            if (key.startsWith(prefix))
            {
                WriterData writerData = (WriterData) writers.get(key);
                if (writerData != null) {
                    delete(writerData.tmpIndexDir);
                }
            }
        }
    }

    /**
     * Delete the respective temp indices.
     * @throws IOException
     */
    public void closeAll() throws IOException
    {
        for (Iterator it = writers.keySet().iterator(); it.hasNext();)
        {
            WriterData writerData = (WriterData) writers.get(it.next());
            delete(writerData.tmpIndexDir);
        }
        writers.clear();
    }

    /**
     * Close all temporary writers, and merge all temporary indices to the writer provided.
     * When the merge is complete, the target index will be optimized. 
     * @param writer
     * @throws IOException
     */
    public void merge(IndexWriter writer) throws IOException
    {
        final List indexDirectories = new LinkedList();
        for (Iterator it = writers.keySet().iterator(); it.hasNext();)
        {
            WriterData writerData = (WriterData) writers.get(it.next());
            writerData.writer.close();
            indexDirectories.add(writerData.dir);
        }
        writer.addIndexes((Directory[]) indexDirectories.toArray(new Directory[indexDirectories.size()]));
    }

    /**
     * Close relevant temporary writers, and merge temporary indices identified by keys which start with the provided prefix to the
     * writer provided.
     * @param prefix key prefix
     * @param writer
     * @throws IOException
     */
    public void merge(String prefix, IndexWriter writer) throws IOException
    {
        final List indexDirectories = new LinkedList();
        for (Iterator it = writers.keySet().iterator(); it.hasNext();)
        {
            String key = it.next().toString();
            if (key.startsWith(prefix))
            {
                WriterData writerData = (WriterData) writers.get(key);
                writerData.writer.close();
                indexDirectories.add(writerData.dir);
            }
        }
        System.out.println("Index directories:" + indexDirectories);
        if (indexDirectories.size() > 0)
            writer.addIndexes((Directory[]) indexDirectories.toArray(new Directory[indexDirectories.size()]));
    }

    protected final boolean delete(File directory)
    {
        File[] files = directory.listFiles();
        for (int i = 0; i < files.length; i++)
        {
            if (!files[i].delete())
            {
                log.error("Failed to delete index file: " + files[i].getAbsolutePath());
                return false;
            }
        }
        return directory.delete();
    }

    /**
     * Synchronized to deal with the situation that the two threads get assigned the same temp directory.
     */
    private synchronized WriterData createData() throws IOException
    {
        // create the temp index directory.
        File tmpIndexDir = getTmpDir() == null ? File.createTempFile("lucene", "index") :
                File.createTempFile("lucene", "index", new File(getTmpDir()));
        if (!tmpIndexDir.delete() || !tmpIndexDir.mkdirs())
        {
            throw new IOException("Unable to create temporary index directory: " + tmpIndexDir);
        }
        Directory dir = FSDirectory.open(tmpIndexDir);
        //TODO: This probably doesn't need to be here. We should just construct the next IndexWriter with create=true
        new IndexWriter(dir, new IndexWriterConfig(BonnieConstants.LUCENE_VERSION, null)).close();

		LimitTokenCountAnalyzer limitTokenCountAnalyzer = new LimitTokenCountAnalyzer(analyzerForIndexing, configuration.getMaxFieldLength());
        IndexWriter writer = new IndexWriter(dir, new IndexWriterConfig(BonnieConstants.LUCENE_VERSION, limitTokenCountAnalyzer));

		LiveIndexWriterConfig config = writer.getConfig();

		config.setMaxBufferedDocs(configuration.getBatchMaxBufferedDocs());

        return new WriterData(writer, dir, tmpIndexDir);
    }

    public String getTmpDir()
    {
        return tmpDir;
    }

    private class WriterData
    {
        final IndexWriter writer;
        final Directory dir;
        final File tmpIndexDir;

        protected WriterData(IndexWriter writer, Directory dir, File tmpIndexDir)
        {
            this.writer = writer;
            this.dir = dir;
            this.tmpIndexDir = tmpIndexDir;
        }
    }
}
