package com.atlassian.bonnie.search;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.atlassian.bonnie.Handle;
import com.atlassian.bonnie.HandleResolver;
import com.atlassian.bonnie.Searchable;

public class BaseDocumentBuilder implements DocumentBuilder
{
	private static final Logger log = LoggerFactory.getLogger(BaseDocumentBuilder.class);

	private final HandleResolver handleResolver;
	private final List<Extractor> extractors;
	private final List<DocumentPostProcessor> postProcessors;
	public static final String TYPE_FIELD = "type";
	public static final String URL_PATH_FIELD = "urlPath";
    private static final int CONTENT_BODY_MAX_SIZE = new ContentBodyMaxSizeSystemProperty().getValue();

    static class ContentBodyMaxSizeSystemProperty
    {
        public static final int DEFAULT = 1048576;
        private final int value;

        public ContentBodyMaxSizeSystemProperty()
        {
            int value;
            final String contentBodyMaxSize = System.getProperty("atlassian.indexing.contentbody.maxsize");

            if (contentBodyMaxSize != null)
            {
                try
                {
                    value = Integer.parseInt(contentBodyMaxSize);
                }
                catch (NumberFormatException e)
                {
                    value = DEFAULT;
                }
            }
            else
            {
                value = DEFAULT;
            }
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }
    }

    /**
	 * This may change, don't consider this part of a public API
	 */
	public static class FieldName
	{
		public static final String HANDLE = "handle";
		public static final String CLASS_NAME = "classname";
		public static final String CONTENT_BODY = "contentBody";
	}

	public BaseDocumentBuilder(HandleResolver handleResolver, List<Extractor> extractors, List<DocumentPostProcessor> postProcessors)
	{
		if (extractors == null)
			throw new IllegalArgumentException("extractors is required.");
		if (postProcessors == null)
			throw new IllegalArgumentException("postProcessors is required.");
		if (handleResolver == null)
			throw new IllegalArgumentException("handleResolver is required.");

		this.handleResolver = handleResolver;
		this.extractors = Collections.unmodifiableList(new ArrayList<Extractor>(extractors));
		this.postProcessors = Collections.unmodifiableList(new ArrayList<DocumentPostProcessor>(postProcessors));
	}

    /**
     * Generates a Lucene document using the extractors defined by construction of BaseDocumentBuilder.
     * <em>Note:</em> The 'contentBody' field will only be used if the data is below the value defined by system property
     * 'atlassian.indexing.contentbody.maxsize'. If this property is not defined, threshold will default to 1 MiB.
     *
     * @param searchable The object to be processed 
     * @return a Lucene Document with extracted data from searchable object
     */
    public Document getDocument(Searchable searchable)
	{
		Document document = getInitialDocument(searchable);
		StringBuffer contentBody = new StringBuffer();

		for (Extractor extractor : extractors)
        {
			try
			{
				extractor.addFields(document, contentBody, searchable);
			}
			catch (RuntimeException e)
			{
				log.error("Error extracting search fields from " + searchable + " using " + extractor  + ": " + e.getMessage(), e);
			}
		}

		if (contentBody.length() > 0)
        {
            Field.Store store;
            if (contentBody.length() > CONTENT_BODY_MAX_SIZE)
                store = Field.Store.NO;
            else
                store = Field.Store.YES;

            document.add(new Field(FieldName.CONTENT_BODY, contentBody.toString(), store, Field.Index.ANALYZED));
        }

		for (DocumentPostProcessor documentPostProcessor : postProcessors)
        {
			documentPostProcessor.process(document);
		}

		return document;
	}

	/**
	 * Get the initial document that will be passed through the chain of extractors
	 *
	 * @param searchable the object the document is being created for
	 * @return a new Document pre-filled with the absolute minimum necessary data for it to
	 *         be added.
	 */
	protected Document getInitialDocument(Searchable searchable)
	{
		Document document = new Document();

		Field handleField = new Field(FieldName.HANDLE, getHandle(searchable).toString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
		document.add(handleField);

        Field classNameField = new Field(FieldName.CLASS_NAME,
            HibernateUnwrapper.getUnderlyingClass(searchable).getName(),
            Field.Store.NO, Field.Index.NOT_ANALYZED);
        document.add(classNameField);

		Object type = getProperty(searchable, "type");
        if (type != null)
        {
            document.add(new Field(TYPE_FIELD, type.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
        }

        Object urlPath = getProperty(searchable, "urlPath");
		if (urlPath != null)
		{
			document.add(new Field(URL_PATH_FIELD, urlPath.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
		}
		return document;
	}

    private Object getProperty(Object object, String field)
    {
		try
		{
			return PropertyUtils.getProperty(object, field);
		}
		catch (NoSuchMethodException e)
		{
			log.info("Unable to find field '" + field + "' on " + object, e);
		}
        catch (IllegalAccessException e)
        {
            log.info("Unable to access field '" + field + "' on " + object, e);
        }
        catch (InvocationTargetException e)
        {
            log.info("Problem accessing field '" + field + "' on " + object, e);
        }
        return null;
    }

	public Handle getHandle(Object obj)
	{
		return handleResolver.getHandle(obj);
	}
}
