package com.atlassian.bonnie.search;


import org.apache.lucene.index.AtomicReader;
import org.apache.lucene.index.AtomicReaderContext;
import org.apache.lucene.search.*;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.DocIdBitSet;
import org.apache.lucene.util.OpenBitSet;

import java.io.IOException;
import java.util.BitSet;
import java.util.WeakHashMap;

public class InvertedQueryFilter extends Filter
{
    private Query query;
    private transient WeakHashMap cache = null;

    /**
     * Constructs a filter which DOES NOT match documents matching
     * <p></p>
     * <code>query</code>.
     */
    public InvertedQueryFilter(Query query)
    {
        this.query = query;
    }

	@Override
    public DocIdSet getDocIdSet(AtomicReaderContext context, Bits acceptDocs) throws IOException
    {
		AtomicReader reader = context.reader();
		if (cache == null)
        {
            cache = new WeakHashMap();
        }

        synchronized (cache)
        {  // check cache
			DocIdSet cached = (DocIdSet) cache.get(reader);
            if (cached != null)
            {
                return cached;
            }
        }

        final OpenBitSet bits = new OpenBitSet(reader.maxDoc());

        new IndexSearcher(reader).search(query, new Collector()
        {
			private int docBase;

			@Override
			public void setScorer(Scorer scorer) throws IOException
			{
				// ignore
			}

			@Override
			public void collect(int doc) throws IOException
			{
				bits.fastSet(doc + docBase);
			}

			@Override
			public void setNextReader(AtomicReaderContext atomicReaderContext) throws IOException
			{
				docBase = atomicReaderContext.docBase;
			}

			@Override
			public boolean acceptsDocsOutOfOrder()
			{
				return true;
			}
		});

        bits.flip(0, reader.maxDoc());    // invert

		synchronized (cache)
        {  // update cache
            cache.put(reader, bits);
        }

        return bits;
    }

    public String toString()
    {
        return "InvertedQueryFilter(" + query + ")";
    }
}