package com.atlassian.bonnie.search;

import com.atlassian.bonnie.ILuceneConnection;
import com.atlassian.bonnie.Indexer;
import com.atlassian.bonnie.Searchable;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import java.io.IOException;

/**
 * This Indexer is responsible for indexing objects to Lucene.
 */
public class LuceneIndexer implements Indexer
{
    private static final Logger log = LoggerFactory.getLogger(LuceneIndexer.class);

    private ILuceneConnection luceneConnection;
    private DocumentBuilder documentBuilder;

    public void index(final Searchable obj)
    {
        if (!obj.isIndexable())
        {
            return;
        }

        unIndex(obj);

        luceneConnection.withWriter(new ILuceneConnection.WriterAction()
        {
            public void perform(IndexWriter indexWriter) throws IOException
            {
                try
                {
                    MDC.put("Indexing", obj.toString());
                    indexWriter.addDocument(documentBuilder.getDocument(obj));
                }
                finally
                {
                    MDC.remove("Indexing");
                }
            }
        });
    }

    public void unIndex(final Searchable obj)
    {
        luceneConnection.withWriter(new ILuceneConnection.WriterAction()
        {
            public void perform(IndexWriter writer) throws IOException
            {
                try
                {
                    MDC.put("Unindexing", obj.toString());
                    Term t = new Term(BaseDocumentBuilder.FieldName.HANDLE, documentBuilder.getHandle(obj).toString());
                    writer.deleteDocuments(t);
                }
                finally
                {
                    MDC.remove("Unindexing");
                }
            }
        });
    }

    public void reIndex(Searchable obj)
    {
        unIndex(obj);
        index(obj);
    }

    public void unIndexAll()
    {
        luceneConnection.truncateIndex();
    }

    public void setLuceneConnection(ILuceneConnection luceneConnection)
    {
        this.luceneConnection = luceneConnection;
    }

    public void setDocumentBuilder(DocumentBuilder documentBuilder)
    {
        this.documentBuilder = documentBuilder;
    }
}