package com.atlassian.bonnie.search;

import com.atlassian.bonnie.BonnieConstants;
import com.atlassian.bonnie.ILuceneConnection;
import com.atlassian.bonnie.LuceneException;
import com.atlassian.bonnie.LuceneUtils;
import com.atlassian.bonnie.Searcher;
import com.atlassian.bonnie.analyzer.LuceneAnalyzerFactory;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.*;
import org.apache.lucene.queryparser.classic.MultiFieldQueryParser;
import org.apache.lucene.queryparser.classic.ParseException;
import org.apache.lucene.queryparser.classic.QueryParser;
import org.apache.lucene.search.*;
import org.apache.lucene.util.BytesRef;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;

public class LuceneSearcher implements Searcher
{
    private static final Logger log = LoggerFactory.getLogger(LuceneSearcher.class);
    private ILuceneConnection luceneConnection;
    private LuceneAnalyzerFactory luceneAnalyzerFactory;
    private static final String[] HANDLE_ONLY_FIELDS = new String[] { BaseDocumentBuilder.FieldName.HANDLE};

    public void setLuceneConnection(ILuceneConnection luceneConnection)
    {
        this.luceneConnection = luceneConnection;
    }

    public void setLuceneAnalyzerFactory(LuceneAnalyzerFactory luceneAnalyzerFactory)
    {
        this.luceneAnalyzerFactory = luceneAnalyzerFactory;
    }

    public void setBooleanQueryMaxClause(int max)
    {
        BooleanQuery.setMaxClauseCount(max);
    }

    /**
     * Different from term query in that the query parameter specified is passed through an analyzer that may
     * remove certain stop words before constructing a Query. Desirable for full text search fields. Undesirable for keyword
     * searches (build a Term query instead).
     *
     * @param searchFields
     * @param query
     */
    public Query buildStandardQuery(String[] searchFields, String query)
    {
        Query myquery;

        try
        {
            QueryParser qp = makeQueryParserForSearchFields(searchFields);
            qp.setDefaultOperator(QueryParser.Operator.AND);
            myquery = qp.parse(query);
        }
        catch (ParseException e)
        {
            throw new LuceneException("Couldn't parse the query successfully:" + e.getMessage());
        }

        return myquery;
    }

    private QueryParser makeQueryParserForSearchFields(String[] searchFields)
    {
        if (searchFields.length == 1)
            return new QueryParser(BonnieConstants.LUCENE_VERSION, searchFields[0], luceneAnalyzerFactory.createAnalyzer());
        else
            return new MultiFieldQueryParser(BonnieConstants.LUCENE_VERSION, searchFields, luceneAnalyzerFactory.createAnalyzer());
    }

    public Query buildStandardQuery(String defaultSearchField, String query)
    {
        return buildStandardQuery(new String[]{defaultSearchField}, query);
    }

    /**
     * Performs a search.
     * @param myquery
     * @return
     *
     * @deprecated since 6.0. This is not a performant method for large indexes. Please issue a search with a limit on the number of results instead.
     */
    public List search(final Query myquery)
    {
		final List result = new LinkedList();
		luceneConnection.withSearch(new ILuceneConnection.SearcherAction()
		{
			public void perform(IndexSearcher searcher) throws IOException
			{
				TopDocs hits = searcher.search(myquery, searcher.getIndexReader().maxDoc());

				for (int i = 0, x = hits.scoreDocs.length; i < x; i++)
				{
					Document doc = searcher.doc(hits.scoreDocs[i].doc);
					String handle = doc.get(BaseDocumentBuilder.FieldName.HANDLE);
					result.add(handle);
				}
			}
		});
		return result;
    }

    /**
     *
     * @param myquery
     * @param sort
     * @return
     *
     * @deprecated since 6.0. This is not a performant method for large indexes. Please issue a search with a limit on the number of results instead.
     */
    public List search(final Query myquery, final Sort sort)
    {
        final List result = new LinkedList();
        luceneConnection.withSearch(new ILuceneConnection.SearcherAction()
        {
            public void perform(IndexSearcher searcher) throws IOException
            {
                TopFieldDocs hits = searcher.search(myquery, searcher.getIndexReader().maxDoc(), sort);

                for (int i = 0, x = hits.scoreDocs.length; i < x; i++)
                {
                    Document doc = searcher.doc(hits.scoreDocs[i].doc);
                    String handle = doc.get(BaseDocumentBuilder.FieldName.HANDLE);
                    result.add(handle);
                }
            }
        });
        return result;
    }

    public Query rewrite(final Query query)
    {
        return (Query) luceneConnection.withReader(new ILuceneConnection.ReaderAction()
        {
            public Object perform(IndexReader reader) throws IOException
            {
                return query.rewrite(reader);
            }
        });
    }

    public String explain(final Query myquery, final int docid)
    {
        final StringBuffer sb = new StringBuffer();
        luceneConnection.withSearch(new ILuceneConnection.SearcherAction()
        {
            public void perform(IndexSearcher searcher) throws IOException
            {
                Explanation e = searcher.explain(myquery, docid);
                sb.append(e.toHtml());
            }
        });
        return sb.toString();
    }

    public int searchCount(final Query query)
    {
		return searchCount(query, null);
    }

    public int searchCount(final Query query, final Filter filter)
    {
		final TotalHitCountCollector totalHitCountCollector = new TotalHitCountCollector();
		luceneConnection.withSearch(new ILuceneConnection.SearcherAction()
		{
			public void perform(IndexSearcher searcher) throws IOException
			{
				searcher.search(query, filter, totalHitCountCollector);
			}
		});
		return totalHitCountCollector.getTotalHits();
    }

    public List searchForFields(Query myquery, Set fieldsToExtract, int startIndex, int numItems)
    {
        return searchForFields(myquery, fieldsToExtract, startIndex, numItems, null, new int[1]);
    }

    public List searchForFields(final Query myquery, Set fieldsToExtract, final int startIndex, final int numItems, final Filter filter, final int[] filteredcount)
    {
        return searchForFields(myquery, fieldsToExtract, startIndex, numItems, null, null, new int[1]);
    }

    public List searchForFields(final Query myquery, Set fieldsToExtract, final int startIndex, final int numItems, final Filter filter, final Sort sort, final int[] filteredcount)
    {
        String[] fieldsToExtractArr;

        if (fieldsToExtract != null && fieldsToExtract.size() > 0)
            fieldsToExtractArr = (String[]) fieldsToExtract.toArray(new String[fieldsToExtract.size()]);
        else
            fieldsToExtractArr = HANDLE_ONLY_FIELDS;

        final String[] fieldsToExtractArr1 = fieldsToExtractArr;

        final ArrayList results = new ArrayList();

        luceneConnection.withSearch(new ILuceneConnection.SearcherAction()
        {
            public void perform(IndexSearcher searcher) throws IOException
            {
				TopDocs hits;

                if (sort == null)
				{
					hits = searcher.search(myquery, filter, searcher.getIndexReader().maxDoc());
				}
				else
				{
                	hits = searcher.search(myquery, filter, searcher.getIndexReader().maxDoc(), sort);
				}

                if (searcher instanceof FilterCountingSearcher)
                {
                    FilterCountingSearcher filterCountingSearcher = ((FilterCountingSearcher) searcher);
                    for (int i = 0; i < filterCountingSearcher.getFilteredCounts().length; i++)
                    {
                        if (i > filteredcount.length - 1)
                        {
                            log.error("Array passed in to store filter counts is too small. Actual: " + filteredcount.length 
                                    + ". Expected: " + filterCountingSearcher.getFilteredCounts().length);
                            break;
                        }

                        filteredcount[i] = filterCountingSearcher.getFilteredCounts()[i];
                    }
                }

                results.ensureCapacity(startIndex + numItems);
                for (int i = 0, x = hits.totalHits; i < x; i++)
                {
                    if (i < startIndex || i >= startIndex + numItems)
                    {
                        results.add(null);
                        continue;
                    }
                    else
                    {
						int docId = hits.scoreDocs[i].doc;
						Document doc = searcher.doc(docId);
                        Map result;
                        if (fieldsToExtractArr1 == null)
                        {
                            result = LuceneUtils.buildMapFromDocument(doc);
                        }
                        else
                        {
                            result = new HashMap(fieldsToExtractArr1.length);
                            for (int j = 0; j < fieldsToExtractArr1.length; j++)
                            {
                                String fieldname = fieldsToExtractArr1[j];
                                result.put(fieldname, doc.get(fieldname));
                            }
                        }
                        result.put("docid", docId);   // add docid so explanation is possible
                        results.add(result);
                    }
                }
            }
        });

        return results;
    }

    public List getAllFieldValues(final String fieldName)
    {
        return (List) luceneConnection.withReader(new ILuceneConnection.ReaderAction()
        {
            public Object perform(IndexReader reader) throws IOException
            {
                List values = new ArrayList();

				Terms terms = MultiFields.getTerms(reader, fieldName);
				if (terms != null)
				{
					TermsEnum termsEnum = terms.iterator(null);

                    while (termsEnum.next() != null)
                    {
                        values.add(termsEnum.term().utf8ToString());
                    }
				}

                return values;
            }
        });
    }

}
