package com.atlassian.bonnie.search;

import com.atlassian.core.util.DateUtils;
import com.atlassian.core.util.InvalidDurationException;
import com.atlassian.bonnie.LuceneUtils;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermRangeQuery;

import java.util.Calendar;
import java.util.Date;

public class SinceDateQueryFactory
{
    public static final String TODAY = "today";
    public static final String YESTERDAY = "yesterday";
    public static final String LAST_WEEK = "lastweek";
    public static final String LAST_MONTH = "lastmonth";

    private String period;
    protected String field;

    public static SinceDateQueryFactory getInstance(String period, String field)
    {
        return new SinceDateQueryFactory(period, field);
    }

    SinceDateQueryFactory(String period, String field)
    {
        this.period = period;
        this.field = field;
    }

    public Query toQuery()
    {
        Calendar c = Calendar.getInstance();
//        c.setTime(new Date());
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        Date startOfToday = c.getTime();

        if (TODAY.equals(period))
        {
		 	return TermRangeQuery.newStringRange(field, LuceneUtils.dateToString(startOfToday), null, true, true);
        }
        if (YESTERDAY.equals(period))
        {
            c.add(Calendar.DAY_OF_YEAR, -1);
            Date startOfYesterday = c.getTime();
            Date beforeMidnightYesterday = new Date(startOfToday.getTime() - 1);

            return createQuery(startOfYesterday, beforeMidnightYesterday);
        }
        else if (LAST_WEEK.equals(period)) // as in past 7 days
        {
            c.add(Calendar.DAY_OF_YEAR, -7);
            Date lastWeekStart = c.getTime();

            return createQuery(lastWeekStart, null);
        }
        else if (LAST_MONTH.equals(period))
        {
            c.add(Calendar.DAY_OF_YEAR, -30);
            Date lastMonthStart = c.getTime();

            return createQuery(lastMonthStart, null);
        }
        else
        {
            long duration = 0;
            try
            {
                duration = DateUtils.getDuration(period);
                return createQuery(new Date(new Date().getTime() - duration * 1000), new Date());
            }
            catch (InvalidDurationException e)
            {
                throw new IllegalArgumentException("Invalid date period: " + period);
            }
        }
    }

    private Query createQuery(Date startOfRange, Date endOfRange)
    {
        // leave the range query open ended if no date is specified.
        String lower = (startOfRange != null) ? LuceneUtils.dateToString(startOfRange) : null;
        String upper = (endOfRange != null) ? LuceneUtils.dateToString(endOfRange) : null;
        return TermRangeQuery.newStringRange(field, lower, upper, true, true);
    }
}
