package com.atlassian.bonnie.search.extractor;

import com.atlassian.bonnie.Searchable;
import com.atlassian.bonnie.search.Extractor;
import com.atlassian.bonnie.search.SearchableAttachment;
import org.apache.commons.io.IOUtils;
import org.apache.lucene.document.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;

public abstract class BaseAttachmentContentExtractor implements Extractor
{
    private static final Logger log = LoggerFactory.getLogger(BaseAttachmentContentExtractor.class);

    public void addFields(Document document, StringBuffer defaultSearchableText, Searchable searchable)
    {
        if (!(searchable instanceof SearchableAttachment)) return;
        if (document.getField("contentBody") != null) return;
        if (defaultSearchableText.length() != 0) return;

        SearchableAttachment attachment = (SearchableAttachment) searchable;
        String fileName = (attachment.getFileName() == null) ? "" : attachment.getFileName().toLowerCase();
        String contentType = attachment.getContentType();
        if (!shouldExtractFrom(fileName, contentType)) return;

        InputStream is = null;
        try
        {
            is = attachment.getContentsAsStream();

			if (is == null)
			{
				log.warn("Encountered attachment with null stream: " + attachment.getFileName());
				return;
			}

            log.debug("Starting to index attachment: " + attachment.getFileName());
			defaultSearchableText.append(extractText(is, attachment));
        }
        catch (IOException e)
        {
            log.warn("Error reading attachment ("  + attachment + ")" , e);
        }
        catch (ExtractorException e)
        {
            log.warn("Error indexing attachment ("  + attachment + ")" , e);
        }
        catch (RuntimeException e)
        {
            log.warn("Error indexing attachment ("  + attachment + ")" , e);
        }
        finally
        {
            IOUtils.closeQuietly(is);
        }

    }

    protected boolean shouldExtractFrom(String fileName, String contentType)
    {
        for (int i = 0; i < getMatchingFileExtensions().length; i++)
        {
            if (fileName.endsWith(getMatchingFileExtensions()[i]))
                return true;
        }

        for (int i = 0; i < getMatchingContentTypes().length; i++)
        {
            String validType = getMatchingContentTypes()[i];
            if (validType.equalsIgnoreCase(contentType))
                return true;
        }

        return false;
    }

    protected String[] getMatchingContentTypes()
    {
        return new String[0];
    }

    protected String[] getMatchingFileExtensions()
    {
        return new String[0];
    }

    /**
     * Package access for unit testing only. Do not use this method directly. Use #addFields().
     *
     * @param is         a stream containing the attachment contents
     * @param attachment contains useful attachment metadata, e.g. filename
     * @return a String with a textual representation of the attachment's contents
     * @throws ExtractorException if there is a problem with converting the attachment content into text. A wrapper
     *                            around the original exception.
     */
    protected abstract String extractText(InputStream is, SearchableAttachment attachment) throws ExtractorException;
}
