/*
 * Copyright (c) 2003 by Atlassian Software Systems Pty. Ltd.
 * All rights reserved.
 */
package com.atlassian.bonnie.search.extractor;

import org.dom4j.io.SAXReader;
import org.dom4j.*;
import org.slf4j.LoggerFactory;
import org.slf4j.Logger;

import java.io.InputStream;
import java.util.*;

import com.atlassian.bonnie.LuceneException;

final class XmlClassConfigurations
{
    private static final Logger log = LoggerFactory.getLogger(XmlClassConfigurations.class);
    //~ Static variables/initializers ----------------------------------------------------------------------------------

    private static Map<String, ClassConfiguration> classConfigurations = new HashMap<String, ClassConfiguration>();
    private static List<Class<?>> unconfiguredClasses = new ArrayList<Class<?>>();

    //~ Methods --------------------------------------------------------------------------------------------------------

    private static void addClassConfiguration(Class<?> clazz, ClassConfiguration classConfiguration)
    {
        classConfigurations.put(clazz.getName(), classConfiguration);
    }

    @SuppressWarnings("unchecked")
    private static ClassConfiguration loadClassConfiguration(Class<?> clazz)
    {
        InputStream configXml = loadConfigFile(clazz);

        if (configXml == null)
        {
            unconfiguredClasses.add(clazz);
            log.debug("Lucene configuration file not found for class: " + clazz.toString());
            return null;
        }

        ClassConfiguration newClassConfig = new ClassConfiguration();

        try
        {
            SAXReader reader = new SAXReader();
            Document doc = reader.read(configXml);
            XPath mapXPath = DocumentHelper.createXPath("/configuration/field");
            List<Element> mappings = (List<Element>) mapXPath.selectNodes(doc);
            for (Element mappingNode : mappings)
            {
                FieldConfiguration mapping = new FieldConfiguration();
                mapping.setType(safeStringAttributeGet(mappingNode, "type"));
                mapping.setFieldName(safeStringAttributeGet(mappingNode, "fieldName"));
                mapping.setAttributeName(safeStringAttributeGet(mappingNode, "attributeName"));
                mapping.setAppendToDefaultSearchableText(safeBooleanAttributeGet(mappingNode,
                        "appendToDefaultSearchableText"));
                newClassConfig.addFieldConfiguration(mapping);
            }

            addClassConfiguration(clazz, newClassConfig);

            return newClassConfig;
        }
        catch (Exception e)
        {
            throw new LuceneException("Couldn't load lucene config file successfully, file=" + clazz, e);
        }
    }

    private static String safeStringAttributeGet(Element mappingNode, String attributeName)
    {
        Attribute attribute = mappingNode.attribute(attributeName);
        if (attribute != null && attribute.getValue() != null && !"".equals(attribute.getValue()))
            return attribute.getValue();
        else
            return null;
    }

    /**
     * 
     * @param mappingNode the element to extract attribute values from
     * @param attributeName the name of the attribute who's value is required.
     * @return the value of the specified attributeName as a boolean. If the attribute doesn't exist or has any value
     *         other than true/TRUE then false will be returned.
     */
    private static boolean safeBooleanAttributeGet(Element mappingNode, String attributeName)
    {
        Attribute attribute = mappingNode.attribute(attributeName);
        if (attribute == null)
            return false;

        return Boolean.valueOf(attribute.getValue());
    }

    private static InputStream loadConfigFile(Class<?> clazz)
    {
        String configFileName = clazz.getName().replace('.', '/') + ".lucene.xml";
        return Thread.currentThread().getContextClassLoader().getResourceAsStream(configFileName);
    }

    public static ClassConfiguration getClassConfiguration(Class<?> clazz)
    {
        ClassConfiguration classConfig;

        synchronized (classConfigurations)
        {
            classConfig = (ClassConfiguration) classConfigurations.get(clazz.getName());

            if (classConfig == null && !unconfiguredClasses.contains(clazz))
                classConfig = loadClassConfiguration(clazz);
        }

        return classConfig;
    }

    //~ Classes --------------------------------------------------------------------------------------------------------

    public static final class ClassConfiguration
    {
        private List<FieldConfiguration> fieldConfigurations = new ArrayList<FieldConfiguration>();

        public void addFieldConfiguration(FieldConfiguration fieldConfiguration)
        {
            fieldConfigurations.add(fieldConfiguration);
        }

        public List<FieldConfiguration> getFieldConfigurations()
        {
            return fieldConfigurations;
        }
    }

    public static final class FieldConfiguration
    {
        public static final String TYPE_TEXT = "Text";
        public static final String TYPE_KEYWORD = "Keyword";
        public static final String TYPE_UNINDEXED = "UnIndexed";
        public static final String TYPE_UNSTORED = "UnStored";
        private String type;
        private String fieldName;
        private String attributeName;
        private boolean appendToDefaultSearchableText;

        public String getType()
        {
            return type;
        }

        public void setType(String type)
        {
            this.type = type;
        }

        public String getFieldName()
        {
            return fieldName;
        }

        public void setFieldName(String fieldName)
        {
            this.fieldName = fieldName;
        }

        public String getAttributeName()
        {
            return attributeName;
        }

        public void setAttributeName(String attributeName)
        {
            this.attributeName = attributeName;
        }

        /**
         * @return true if the attribute(s) specified should be appended to the default searchable text. Note that if
         *         you specify multiple attributes then just as all will be added to the same document field, they will
         *         also all be added to the default searchable text if this is true.
         */
        public boolean isAppendToDefaultSearchableText()
        {
            return appendToDefaultSearchableText;
        }

        public void setAppendToDefaultSearchableText(boolean appendToDefaultContent)
        {
            this.appendToDefaultSearchableText = appendToDefaultContent;
        }
    }
}
