package com.atlassian.bonnie.search.extractor;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;

import com.atlassian.bonnie.LuceneException;
import com.atlassian.bonnie.LuceneUtils;
import com.atlassian.bonnie.Searchable;
import com.atlassian.bonnie.search.Extractor;
import com.atlassian.bonnie.search.HibernateUnwrapper;

/**
 * <p>
 * An indexing extractor that is configured via XML. You should note that earlier versions of this extractor were
 * hard-coded to append the data from the contentBody field to the defaultSearchableText supplied to
 * {@link #addFields(Document, StringBuffer, Searchable)} instead of to the Document. <strong>This behaviour has been
 * changed.</strong>
 * </p>
 * <p>
 * The XmlConfiguredExtractor will now append all extracted content to the Document and will only add to the
 * defaultSearchableText if it has been configured to.
 * </p>
 */
public class XmlConfiguredExtractor implements Extractor
{
    public void addFields(Document document, StringBuffer defaultSearchableText, Searchable searchable)
    {
        XmlClassConfigurations.ClassConfiguration classConfig = XmlClassConfigurations.getClassConfiguration(HibernateUnwrapper.getUnderlyingClass(searchable));

        if (classConfig != null)
        {
            for (XmlClassConfigurations.FieldConfiguration fieldConfiguration : classConfig.getFieldConfigurations())
            {
                String attributeNames[] = fieldConfiguration.getAttributeName().split(",");
                List<String> indexedValues = new ArrayList<String>(attributeNames.length);
                for (String attributeName : attributeNames)
                {
                    String indexedValue;
                    Object o = getContentOfAttribute(searchable, attributeName);
                    if (o.getClass().isArray())
                    {
                        indexedValue = indexArrayField(document, fieldConfiguration, o);
                    }
                    else if (o instanceof Collection)
                    {
                        indexedValue = indexCollectionField(document, fieldConfiguration, (Collection) o);
                    }
                    else if (o instanceof Date
                            && fieldConfiguration.getType().equals(
                                    XmlClassConfigurations.FieldConfiguration.TYPE_KEYWORD))
                    {
                        indexedValue = indexDateField(document, fieldConfiguration, (Date) o);
                    }
                    else
                    {
                        indexedValue = indexStringField(document, fieldConfiguration, String.valueOf(o));
                    }

                    indexedValues.add(indexedValue);
                }

                if (fieldConfiguration.isAppendToDefaultSearchableText() && !indexedValues.isEmpty())
                {
                    defaultSearchableText.append(StringUtils.join(indexedValues, ','));
                }
            }
        }
    }

    /**
     * 
     * @param doc
     * @param fieldConfiguration
     * @param date
     * @return a String representing the data that was indexed
     */
    private String indexDateField(Document doc, XmlClassConfigurations.FieldConfiguration fieldConfiguration, Date date)
    {
        String dateStr = LuceneUtils.dateToString(date);
        Field field = new Field(fieldConfiguration.getFieldName(), dateStr, Field.Store.YES, Field.Index.NOT_ANALYZED);
        doc.add(field);
        return dateStr;
    }

    /**
     * @param doc
     * @param fieldConfiguration
     * @param arr
     * @return a String representing the values indexed, with each separated by a space character.
     */
    private String indexArrayField(Document doc, XmlClassConfigurations.FieldConfiguration fieldConfiguration,
            Object arr)
    {
        int length = Array.getLength(arr);
        List<String> indexedValues = new ArrayList<String>(length);
        for (int i = 0; i < length; i++)
        {
            Object o = Array.get(arr, i);
            indexedValues.add(indexStringField(doc, fieldConfiguration, String.valueOf(o)));
        }

        return StringUtils.join(indexedValues, ' ');
    }

    /**
     * 
     * @param doc
     * @param fieldConfiguration
     * @param collection
     * @return a String representing the values indexed, with each separated by a space character.
     */
    private String indexCollectionField(Document doc, XmlClassConfigurations.FieldConfiguration fieldConfiguration,
            Collection collection)
    {
        List<String> indexedValues = new ArrayList<String>(collection.size());
        for (Iterator it = collection.iterator(); it.hasNext();)
        {
            Object o = it.next();
            indexedValues.add(indexStringField(doc, fieldConfiguration, String.valueOf(o)));
        }

        return StringUtils.join(indexedValues, ' ');
    }

    /**
     * 
     * @param doc
     * @param fieldConfiguration
     * @param strContent
     * @return the String that was indexed.
     */
    private String indexStringField(Document doc, XmlClassConfigurations.FieldConfiguration fieldConfiguration,
            String strContent)
    {
        Field field;
        if (fieldConfiguration.getType().equals(XmlClassConfigurations.FieldConfiguration.TYPE_TEXT))
        {
            field = new Field(fieldConfiguration.getFieldName(), strContent, Field.Store.YES, Field.Index.ANALYZED);
        }
        else if (fieldConfiguration.getType().equals(XmlClassConfigurations.FieldConfiguration.TYPE_KEYWORD))
        {
            field = new Field(fieldConfiguration.getFieldName(), strContent, Field.Store.YES, Field.Index.NOT_ANALYZED);
        }
        else if (fieldConfiguration.getType().equals(XmlClassConfigurations.FieldConfiguration.TYPE_UNINDEXED))
        {
            field = new Field(fieldConfiguration.getFieldName(), strContent, Field.Store.YES, Field.Index.NO);
        }
        else if (fieldConfiguration.getType().equals(XmlClassConfigurations.FieldConfiguration.TYPE_UNSTORED))
        {
            field = new Field(fieldConfiguration.getFieldName(), strContent, Field.Store.NO, Field.Index.ANALYZED);
        }
        else
        {
            throw new LuceneException("Unknown type for a field, fieldName=" + fieldConfiguration.getFieldName());
        }

        doc.add(field);
        return strContent;
    }

    private Object getContentOfAttribute(Object obj, String attributeName)
    {
        try
        {
            String[] attributes = attributeName.split("\\.");
            Object o = null;
            for (int i = 0; i < attributes.length; ++i)
                o = PropertyUtils.getProperty(obj, attributeName);
            return (o == null) ? "" : o;
        }
        catch (IllegalAccessException iae)
        {
            throw new LuceneException("Couldn't get string content of attribute, as property accessor method for " + attributeName + " cannot be accessed");
        }
        catch (NoSuchMethodException e)
        {
            throw new LuceneException("Couldn't get string content of attribute, as no such property accessor method for " + attributeName + " exists");
        }
        catch (InvocationTargetException e)
        {
            throw new LuceneException("Calling property accessor method for attribute " + attributeName + " threw an exception", e);
        }
    }
}
