package com.atlassian.bonnie.search.extractor.util;

/**
 * This is a base class for string builders that limit their length.
 * <p>
 * Instances of this class are not safe for use by multiple threads.
 */
public abstract class AbstractLengthLimitedStringBuilder
{
    protected StringBuilder buffer;
    protected final LIMIT_BEHAVIOUR throwWhenLimitReached;
    protected boolean limitReached;

    AbstractLengthLimitedStringBuilder()
    {
        this(LIMIT_BEHAVIOUR.SILENT);
    }

    AbstractLengthLimitedStringBuilder(final LIMIT_BEHAVIOUR limitBehaviour)
    {
        this.buffer = new StringBuilder();
        this.throwWhenLimitReached = limitBehaviour;
        this.limitReached = false;
    }

    public void setLength(int l)
    {
        buffer.setLength(l);
    }

    public int length()
    {
        return buffer.length();
    }

    public AbstractLengthLimitedStringBuilder append(char[] str, int offset, int len)
    {
        int length = len;
        if (limitReached(len))
        {
            length = remainingLength();
        }

        buffer.append(str, offset, length);

        return this;
    }

    public AbstractLengthLimitedStringBuilder append(char c)
    {
        if (!limitReached(1))
        {
            buffer.append(c);
        }

        return this;
    }

    public AbstractLengthLimitedStringBuilder append(String s)
    {
        int length = s.length();
        if (limitReached(s.length()))
        {
            length = remainingLength();
        }

        buffer.append(s.substring(0, length));
        removeSplitCharacter();

        return this;
    }

    /**
     * Some unicode code points consume two chars.
     * This method checks if we split one of these characters in half and remove it if we did.
     */
    private void removeSplitCharacter()
    {
        int bufferLength = buffer.length();
        if (bufferLength > 1)
        {
            if (Character.isHighSurrogate(buffer.charAt(bufferLength - 1)))
            {
                buffer.setLength(bufferLength - 1);
            }
        }
    }

    protected int remainingLength()
    {
        return capacity() - buffer.length();
    }

    protected abstract int limit();

    /**
     * Checks if appending length characters to the string builder will reach the limit.
     *
     * @param length
     * @return true if the limit is reached, false otherwise
     */
    protected boolean limitReached(int length)
    {
        if ((buffer.length() + length) > capacity())
        {
            int newCapacity = (capacity() + 1) * 2;
            if (newCapacity > limit())
            {
                limitReached = true;
                if (throwWhenLimitReached == LIMIT_BEHAVIOUR.THROW)
                {
                    throw new LimitReachedException();
                }

                return true;
            }
        }

        return false;
    }

    public String toString()
    {
        return buffer.toString();
    }

    public int capacity()
    {
        return buffer.capacity();
    }

    public boolean isLimitReached()
    {
        return limitReached;
    }

    /**
     * Behaviour of the string builder when the limit is reached.
     */
    public static enum LIMIT_BEHAVIOUR
    {
        /**
         * Throw an exception when the limit is reached.
         * Not all the available capacity will be used.
         */
        THROW,

        /**
         * Silently discard appending if it goes above the limit.
         * All the available capacity will be used.
         */
        SILENT
    }
}
