package org.apache.lucene.search;

/**
 * Copyright 2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.store.Directory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

/**
 * Implements search over a single IndexReader, but remains open even if close() is called. This way it can be shared by
 * multiple objects that need to search the index without being aware of the keep-the-index-open-until-it-changes logic.<br>
 * <br>
 * Basic use (works, but can slow all searcher while opening a new Searcher instance:
 * <pre>
 * class SearcherFactory {
 *     SearcherFactory(IndexReader reader){
 *         currentSearcher=new DelayCloseIndexSearcher(reader);
 *     }
 *
 *     public synchronized IndexSearcher createSearcher() {
 *         if (!currentSearcher.isCurrent()) {
 *             currentSearcher.closeWhenDone();
 *             currentSearcher=new DelayCloseIndexSearcher(reader);
 *         }
 *
 *         currentSearcher.open();
 *         return currentSearcher;
 *     }
 *
 *     void synchronized close() {
 *         currentSearcher.closeWhenDone();
 *     }
 *
 *     private final IndexReader reader;
 *
 *     private DelayCloseIndexSearcher currentSearcher;
 * }
 * </pre>
 * <p></p>
 * Objects that need to search the index do the following:<br>
 * <pre>
 * //searcherFactory is created once at startup
 *
 * IndexSearcher indexSearcher=searcherFactory.createSearcher();
 * Hits hist=indexSearcher.search(query, filter, sort);
 * ... // handle results
 * indexSearcher.close();
 *
 * // when the application shuts down
 * searcherFactory.close();
 * </pre>
 *
 * @author Luc Vanlerberghe
 */
public class DelayCloseIndexSearcher extends IndexSearcher
{
    private IndexReader indexReader;

    /**
     * Creates a DelayCloseIndexSearcher searching the index using the provided Reader
     *
     * <b>Important:</b> The passed in IndexReader will be closed once the underlying
     * IndexSearcher is closed.
     *
     * @param indexReader Reader to use for opening the searcher
     * @throws IOException if an I/O error occurs
     */
    public DelayCloseIndexSearcher(IndexReader indexReader) throws IOException
    {
        super(indexReader);
        this.indexReader = indexReader;
        this.usageCount = 0;
        this.shouldCloseWhenDone = false;

        log.debug("<init>");
    }

    /**
     * This should be called whenever this instances is passed as a new IndexSearcher.
     * Only when each call to open() is balanced with a call to close(), and closeWhenDone has been called,
     * will super.close() be called.
     */
    public void open()
    {
        synchronized (this)
        {
            if (shouldCloseWhenDone)
            {
                throw new IllegalStateException("closeWhenDone() already called");
            }

            ++usageCount;
        }

        log.debug("open()");
    }

    /**
     * Signals that this instance may really close when all open() calls have been balanced with a call to close().
     *
     * @throws IOException if an I/O error occurs.
     */
    public void closeWhenDone() throws IOException
    {
        log.debug("closeWhenDone()");

        boolean doClose; // Keep the actual super.close() out of the synchronized block

        synchronized (this)
        {
            if (shouldCloseWhenDone)
            {
                throw new IllegalStateException("closeWhenDone() already called");
            }

            shouldCloseWhenDone = true;

            doClose = (usageCount == 0);
        }

        if (doClose)
        {
            closeInternal();
        }
    }

    /**
     * Returns whether the underlying IndexSearcher instance still works on a current version of the index.
     * If it returns false, closeWhenDone() should be called and another instance created to handle further search requests.
     *
     * @return whether the underlying IndexSearcher instance still works on a current version of the index
     * @throws IOException if an I/O error occurs.
     */
    public boolean isCurrent() throws IOException
    {
		final IndexReader r = getIndexReader();
		assert r instanceof DirectoryReader: "searcher's IndexReader should be a DirectoryReader, but got " + r;
		return ((DirectoryReader) r).isCurrent();
    }

    /**
     * Returns wether the underlying IndexSearcher has really been closed.
     * If it is true, this instance can no longer be used.
     *
     * @return whether the underlying IndexSearcher has really been closed.
     */
    public boolean isClosed()
    {
        synchronized (this)
        {
            return shouldCloseWhenDone && usageCount == 0;
        }
    }

    //
    // IndexSearcher overrides
    //

    /**
     * Should be called once for every call to open().
     * If the usageCount drops to zero and closeWhenDone() was called, super.close() will be called.
     *
     * @throws IOException if an I/O error occurs.
     */
    public void close() throws IOException
    {
        log.debug("close()");

        boolean doClose; // Keep the actual super.close() out of the synchronized block

        synchronized (this)
        {
            if (usageCount <= 0)
            {
                throw new IllegalStateException("usageCount<=0");
            }

            doClose = (--usageCount == 0 && shouldCloseWhenDone);
        }

        if (doClose)
        {
            closeInternal();
        }
    }

    private void closeInternal() throws IOException
    {
        log.debug("Closing underlying index searcher");
//        super.close();

        // need to close the indexReader manually because this IndexSearcher was constructed with the constructor
        // IndexSearcher(IndexReader r) which prevents the passed in the reader from being closed when the searcher is closed.
        log.debug("Closing underlying index reader");
        indexReader.close();
    }

    //
    // private members
    //

    /**
     * The number of open() calls minus the number of close() calls.
     * If this drops to zero and closeWhenDone() is true, super.close() is called.
     */
    private int usageCount;

    /**
     * Indicates if closeWhenDone() was called.
     * If true and usageCount is zero, super.close() is called.
     */
    private boolean shouldCloseWhenDone;

    /**
     * The Logger instance for this Class.
     */
    private static final Logger log = LoggerFactory.getLogger(DelayCloseIndexSearcher.class);
}
