package com.atlassian.diagnostics.internal;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.util.Comparator;

import static java.util.Comparator.comparing;
import static java.util.Comparator.naturalOrder;
import static java.util.Comparator.nullsFirst;

public class PluginId implements Comparable<PluginId> {

    private static final Comparator<PluginId> COMPARATOR =
            comparing(PluginId::getKey, nullsFirst(naturalOrder()))
                .thenComparing(PluginId::getVersion, nullsFirst(naturalOrder()));

    private String key;
    private String version;

    public PluginId(String key, String version) {
        if (StringUtils.isEmpty(key) && StringUtils.isEmpty(version)) {
            throw new IllegalArgumentException("If key and version are empty then the plugin should be represented as null");
        }
        if (StringUtils.isEmpty(key) && StringUtils.isNotEmpty(version)) {
            throw new IllegalArgumentException("Plugin key should be non-empty if a version is supplied");
        }
        this.key = key;
        this.version = version;
    }

    @Override
    public int compareTo(@Nonnull PluginId other) {
        return COMPARATOR.compare(this, other);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PluginId plugin = (PluginId) o;
        return Objects.equal(key, plugin.key) &&
                Objects.equal(version, plugin.version);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(key, version);
    }

    public String getKey() {
        return key;
    }

    public String getVersion() {
        return version;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper("")
                .addValue(key)
                .addValue(version)
                .toString();
    }

    public static PluginId valueOf(String key, String version) {
        return StringUtils.isEmpty(key) && StringUtils.isEmpty(version) ? null : new PluginId(key, version);
    }
}
