package com.atlassian.diagnostics.internal;

import com.atlassian.diagnostics.AlertCount;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.PluginDetails;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;

import java.util.HashMap;
import java.util.Map;

import static java.util.Objects.requireNonNull;

public class SimpleAlertCount implements AlertCount {

    private final Issue issue;
    private final Map<String, Long> countsByNodeName;
    private final PluginDetails pluginDetails;

    private SimpleAlertCount(Builder builder) {
        countsByNodeName = ImmutableMap.copyOf(builder.countsByNodeName);
        issue = builder.issue;
        pluginDetails = builder.pluginDetails;
    }

    @Nonnull
    @Override
    public Map<String, Long> getCountsByNodeName() {
        return countsByNodeName;
    }

    @Nonnull
    @Override
    public Issue getIssue() {
        return issue;
    }

    @Nonnull
    @Override
    public PluginDetails getPlugin() {
        return pluginDetails;
    }

    @Override
    public long getTotalCount() {
        return countsByNodeName.values().stream().mapToLong(Long::longValue).sum();
    }

    public static class Builder {
        private final Issue issue;
        private final PluginDetails pluginDetails;
        private final Map<String, Long> countsByNodeName;

        public Builder(@Nonnull Issue issue, @Nonnull PluginDetails plugin) {
            this.issue = requireNonNull(issue, "issue");
            this.pluginDetails = requireNonNull(plugin, "plugin");

            countsByNodeName = new HashMap<>();
        }

        @Nonnull
        public SimpleAlertCount build() {
            return new SimpleAlertCount(this);
        }

        @Nonnull
        public Builder setCountForNode(@Nonnull String nodeName, long count) {
            countsByNodeName.put(nodeName, count);
            return this;
        }
    }
}
