package com.atlassian.diagnostics.internal.dao;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.AlertCriteria;
import com.atlassian.diagnostics.DiagnosticsConstants;
import com.atlassian.diagnostics.PageRequest;
import com.atlassian.diagnostics.Severity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

public interface AlertEntityDao {

    /**
     * Deletes all {@link AlertEntity alerts} matching the supplied {@link AlertCriteria criteria}.
     * @param criteria to use when choosing which alerts to delete
     */
    void deleteAll(@Nonnull AlertCriteria criteria);

    /**
     * @return all component IDs that any alert entity refers to
     */
    Set<String> findAllComponentIds();

    /**
     * @return all issue IDs that any alert entity refers to, with the corresponding severity
     */
    Map<String, Severity> findAllIssueIds();

    /**
     * @return all node names that any alert entity refers to
     */
    Set<String> findAllNodeNames();

    /**
     * @return all plugin keys that any alert entity refers to (including
     *         {@link DiagnosticsConstants#PLUGIN_NOT_DETECTED_KEY})
     */
    Set<String> findAllPluginKeys();

    /**
     * Retrieves an alert entity by the specified ID
     *
     * @param id the alert ID
     * @return the matching entity, or {@code null} if no matching entity was found
     */
    @Nullable
    AlertEntity getById(long id);

    /**
     * Stores an alert. This should only be used to store newly-created alerts. Supplying an already
     * stored alert will yield undefined results but most likely will result in persisting a duplicate.
     *
     * @param alert the alert to store
     * @return the persisted alert entity
     */
    @Nonnull
    AlertEntity save(@Nonnull Alert alert);

    /**
     * Streams all {@link AlertEntity alerts} matching the supplied {@link AlertCriteria criteria} ordered by timestamp
     * (descending), then ID (descending) until the callback {@link RowCallback#onRow(Object) signals} that it's done
     * or until no more matching alerts are found.
     * <p>
     * The {@link PageRequest#getStart() start offset} on the page request refers to the row at which the requested
     * page starts.
     * <p>
     * For pagination purposes, implementations must attempt to retrieve one extra item following the page of data if
     * an {@link PageRequest#getStart() start} is specified on the page request.
     *
     * @param criteria request describing which alerts to retrieve
     * @param callback the callback to use when streaming matching alerts (including extras) from the data store
     * @param pageRequest request describing which page of results to return
     */
    void streamAll(@Nonnull AlertCriteria criteria, @Nonnull RowCallback<AlertEntity> callback,
                   @Nonnull PageRequest pageRequest);

    /**
     * Streams all {@link AlertEntity alerts} matching the supplied IDs ordered by timestamp (descending), then ID
     * (descending) until the callback {@link RowCallback#onRow(Object) signals} that it's done or until no more
     * matching alerts are found.
     * <p>
     * If a requested ID is not found, no error should be returned.
     *
     * @param ids the IDs of the {@link AlertEntity alerts} to retrieve
     * @param callback the callback to use when streaming matching alerts from the data store
     */
    void streamByIds(@Nonnull Collection<Long> ids, @Nonnull RowCallback<AlertEntity> callback);

    /**
     * Streams {@link AlertMetric alert metrics} for all {@link Alert alerts} matching the supplied
     * {@link AlertCriteria criteria} ordered by timestamp (descending) until the callback
     * {@link RowCallback#onRow(Object) signals} that it's done or all available metrics have been provided.
     *
     * @param criteria the criteria to use when choosing which alerts to stream
     * @param callback the callback to use when streaming matching alerts from the data store
     */
    void streamMetrics(@Nonnull AlertCriteria criteria, @Nonnull RowCallback<AlertMetric> callback,
                       @Nonnull PageRequest pageRequest);

    /**
     * Streams {@link MinimalAlertEntity minimal alerts} matching the supplied {@link AlertCriteria criteria} ordered
     * by timestamp (descending), then ID (descending) until the callback {@link RowCallback#onRow(Object) signals}
     * that it's done or until no more matching alerts are found.
     * <p>
     * The {@link PageRequest#getStart() start offset} on the page request refers to the row at which the requested
     * page starts.
     * <p>
     * For pagination purposes, implementations must attempt to retrieve one extra item following the page of data if
     * an {@link PageRequest#getStart() start} is specified on the page request.
     *
     * @param criteria request describing which alerts to retrieve
     * @param callback the callback to use when streaming matching alerts (including extras) from the data store
     * @param pageRequest request describing which page of results to return
     */
    void streamMinimalAlerts(@Nonnull AlertCriteria criteria, @Nonnull RowCallback<MinimalAlertEntity> callback,
                             @Nonnull PageRequest pageRequest);
}
