package com.atlassian.diagnostics.internal.backdoor;

import com.atlassian.diagnostics.Alert;
import com.atlassian.diagnostics.AlertCriteria;
import com.atlassian.diagnostics.AlertRequest;
import com.atlassian.diagnostics.AlertTrigger;
import com.atlassian.diagnostics.CallbackResult;
import com.atlassian.diagnostics.ComponentMonitor;
import com.atlassian.diagnostics.Issue;
import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.PageCallback;
import com.atlassian.diagnostics.PageRequest;
import com.atlassian.diagnostics.PageSummary;
import com.atlassian.diagnostics.Severity;
import com.google.common.collect.ImmutableMap;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Map;

public class DataService {

    private static final AlertTrigger[] TRIGGERS = new AlertTrigger[] {
            new AlertTrigger.Builder().plugin("System", null).module("PluginSystemLifecycle.class").build(),
            new AlertTrigger.Builder().plugin("com.atlassian.plugin1", "1.2.3").build(),
            new AlertTrigger.Builder().plugin("com.atlassian.plugin2", "2.3.1").module("SomeClass.class").build(),
            new AlertTrigger.Builder().plugin("com.atlassian.plugin2", "2.3.5").module("SomeClass.class").build(),
            new AlertTrigger.Builder().plugin("com.atlassian.plugin3", "1.0.0").module("SomeClass.class").build(),
            new AlertTrigger.Builder().plugin("com.atlassian.plugin4", "1.0.0").module("SomeClass.class").build(),
    };

    private final MonitoringService monitoringService;

    private ComponentMonitor monitorIt1;
    private ComponentMonitor monitorIt2;
    private ComponentMonitor monitorIt3;
    private Map<String, Issue> issues;

    public DataService(MonitoringService monitoringService) {
        this.monitoringService = monitoringService;

        issues = new HashMap<>();

        monitorIt1 = initMonitor("IT1");
        monitorIt2 = initMonitor("IT2");
        monitorIt3 = initMonitor("IT3");
    }

    /**
     * Generates a test data set that can is used for integration testing. The same test data set can easily be used
     * for manual testing as well.
     *
     * @return the timestamp of the newest alert in the test data set
     */
    public synchronized Instant createTestData() {
        PageCallback<Alert, Alert> callback = new PageCallback<Alert, Alert>() {
            private Alert latest;

            @Override
            public CallbackResult onItem(Alert item) {
                latest = item;
                return CallbackResult.DONE;
            }

            @Override
            public Alert onEnd(PageSummary summary) {
                return latest;
            }
        };

        // check whether any test data has already been created
        Alert latest = monitoringService.streamAlerts(
                AlertCriteria.builder().componentIds("IT1", "IT2", "IT3").build(), callback, PageRequest.ofSize(1));

        if (latest != null) {
            // an alert of IT1/IT2/IT3 is found - test data has already been generated. Just return the newest
            // timestamp to allow the caller to set expectations on the test data set
            return latest.getTimestamp();
        }

        // no test data exists yet. time to insert some!
        Instant latestTimestamp = Instant.now();
        for (int i = 0; i < 50 ; i++) {
            Instant timestamp = latestTimestamp.minusSeconds((49 - i) * 5);
            Map<String, String> details = ImmutableMap.of("details", "some-value-" + i);
            monitorIt1.alert(new AlertRequest.Builder(issues.get("IT1-1001"))
                    .timestamp(timestamp)
                    .trigger(TRIGGERS[0])
                    .details(() -> details)
                    .build());

            timestamp = latestTimestamp.minusSeconds((49 - i) * 30);
            monitorIt2.alert(new AlertRequest.Builder(issues.get("IT2-1002"))
                    .timestamp(timestamp)
                    .trigger(TRIGGERS[1])
                    .details(() -> details)
                    .build());

            timestamp = latestTimestamp.minus((49 - i), ChronoUnit.HOURS);
            monitorIt3.alert(new AlertRequest.Builder(issues.get("IT3-1003"))
                    .timestamp(timestamp)
                    .trigger(TRIGGERS[i % 6])
                    .details(() -> details)
                    .build());

            try {
                // Sleep for a bit to prevent alert queue overload
                Thread.sleep(25L);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }

        return latestTimestamp;
    }

    private ComponentMonitor initMonitor(String componentId) {
        ComponentMonitor monitor = monitoringService.createMonitor(
                componentId, componentId.toLowerCase() + ".component.name");

        issues.put(componentId + "-1001", defineIssue(monitor, 1001, Severity.INFO));
        issues.put(componentId + "-1002", defineIssue(monitor, 1002, Severity.WARNING));
        issues.put(componentId + "-1003", defineIssue(monitor, 1003, Severity.ERROR));
        issues.put(componentId + "-1004", defineIssue(monitor, 1004, Severity.ERROR));

        return monitor;
    }

    private Issue defineIssue(ComponentMonitor monitor, int id, Severity severity) {
        String prefix = monitor.getComponent().getId().toLowerCase() + ".issue." + id + ".";
        return monitor.defineIssue(id)
                .severity(severity)
                .summaryI18nKey(prefix + "summary")
                .descriptionI18nKey(prefix + "description")
                .build();
    }
}
