package com.atlassian.diagnostics.internal.platform.monitor.db.pool;

import com.atlassian.diagnostics.internal.platform.ConsecutiveAlertGate;
import com.atlassian.diagnostics.internal.platform.ConsecutiveAlertGateFactory;
import com.atlassian.diagnostics.internal.platform.monitor.db.DatabaseMonitor;
import com.atlassian.diagnostics.internal.platform.monitor.db.DatabaseMonitorConfiguration;
import com.atlassian.diagnostics.internal.platform.poller.DiagnosticPoller;

import javax.annotation.Nonnull;
import java.time.Clock;

public class DatabasePoolPoller extends DiagnosticPoller<DatabaseMonitorConfiguration> {

    private final DatabasePoolDiagnosticProvider databasePoolDiagnosticProvider;
    private final DatabaseMonitor databaseMonitor;
    private final Clock clock;
    private final ConsecutiveAlertGate alertGate;

    public DatabasePoolPoller(@Nonnull final DatabaseMonitorConfiguration config,
                              @Nonnull final DatabaseMonitor databaseMonitor,
                              @Nonnull final DatabasePoolDiagnosticProvider databasePoolDiagnosticProvider,
                              @Nonnull final Clock clock,
                              @Nonnull final ConsecutiveAlertGateFactory alertGateFactory) {
        super(DatabasePoolPoller.class.getName(), config);
        this.databasePoolDiagnosticProvider = databasePoolDiagnosticProvider;
        this.databaseMonitor = databaseMonitor;
        this.clock = clock;
        this.alertGate = alertGateFactory.createAlertGate(config::poolUtilizationTimeWindow, clock);
    }

    @Override
    protected void execute() {
        databasePoolDiagnosticProvider.getDiagnostics().ifPresent(this::raiseAlertIfPoolUtilizationHasExceededThreshold);
    }

    private void raiseAlertIfPoolUtilizationHasExceededThreshold(final DatabasePoolDiagnostic diagnostics) {
        if (alertGate.shouldRaiseAlert(() -> hasReachedDbPoolUtilisationThreshold(diagnostics))) {
            databaseMonitor.raiseAlertForHighPoolUtilization(clock.instant(), diagnostics);
        }
    }

    private boolean hasReachedDbPoolUtilisationThreshold(final DatabasePoolDiagnostic diagnostic) {
        return diagnostic.getMaxConnections() != 0d
                && (((double) diagnostic.getActiveConnections() / diagnostic.getMaxConnections()) * 100) >= monitorConfiguration.poolUtilizationPercentageLimit();
    }

}
