package com.atlassian.diagnostics.internal.platform.monitor.cluster;

import com.atlassian.beehive.core.ManagedClusterLock;
import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.internal.InitializingMonitor;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class ClusterLockMonitor extends InitializingMonitor {

    private static final String KEY_PREFIX = "diagnostics.clusterlock.issue";
    private static final int LONG_LOCK_WAIT = 2001;
    private static final int LARGE_LOCK_QUEUE = 2002;

    private final ClusterLockMonitorConfiguration clusterLockMonitorConfiguration;

    public ClusterLockMonitor(final ClusterLockMonitorConfiguration clusterLockMonitorConfiguration) {
        this.clusterLockMonitorConfiguration = clusterLockMonitorConfiguration;
    }

    @Override
    public void init(@Nonnull final MonitoringService monitoringService) {
        monitor = monitoringService.createMonitor("CLUSTERLOCK", "diagnostics.clusterlock.name", clusterLockMonitorConfiguration);

        defineIssue(KEY_PREFIX, LONG_LOCK_WAIT, Severity.WARNING);
        defineIssue(KEY_PREFIX, LARGE_LOCK_QUEUE, Severity.WARNING);
    }

    /**
     * Raises an alert for locks with a long wait time
     *
     * @param timestamp the timestamp to be used on the alert
     * @param locksWithLongWait the ClusterLocks held over the expected threshold
     */
    public void raiseAlertForLocksWithLongWait(@Nonnull final Instant timestamp, @Nonnull final Collection<ManagedClusterLock> locksWithLongWait) {
        alert(LONG_LOCK_WAIT, builder -> builder
                .timestamp(timestamp)
                .details(() -> lockAlertDetails(locksWithLongWait))
        );
    }

    /**
     * Raises an alert for locks with a large queue
     *
     * @param timestamp the timestamp to be used on the alert
     * @param locksWithLargeQueue the ClusterLocks with a large queue
     */
    public void raiseAlertForLargeQueue(@Nonnull final Instant timestamp, @Nonnull final Collection<ManagedClusterLock> locksWithLargeQueue) {
        alert(LARGE_LOCK_QUEUE, builder -> builder
                .timestamp(timestamp)
                .details(() -> lockAlertDetails(locksWithLargeQueue))
        );
    }

    private Map<Object, Object> lockAlertDetails(@Nonnull final Collection<ManagedClusterLock> longHeldLockDiagnostics) {
        final List<Map<Object, Object>> locks = longHeldLockDiagnostics.stream().map(lock -> {
            final ImmutableMap.Builder<Object, Object> alertBuilder = ImmutableMap.builder()
                    .put("lockedByNode", lock.getClusterLockStatus().getLockedByNode())
                    .put("lockName", lock.getClusterLockStatus().getLockName())
                    .put("lastUpdateTimeInMillis", lock.getClusterLockStatus().getUpdateTime());

            lock.getStatistics().forEach((key, value) -> alertBuilder.put(key.getLabel(), value));

            return alertBuilder.build();
        }).collect(Collectors.toList());

        return ImmutableMap.builder()
                .put("locks", locks)
                .build();
    }

}
