package com.atlassian.diagnostics.internal.platform.monitor.gc;

import java.time.Duration;

public class GCDetailsCalculator {
    private final double percentageOfTimeInGarbageCollection;
    private final Duration timeSpentOnCollection;
    private final long collectionsCount;
    private final Duration timeSinceLastPoll;

    public GCDetailsCalculator(final GCRead previousRead, final GCRead currentRead) {
        this.timeSpentOnCollection = calculateTimeSpentOnGC(previousRead, currentRead);
        this.collectionsCount = calculateCollectionsCount(previousRead, currentRead);
        this.timeSinceLastPoll = calculateTimeSinceLastPoll(previousRead, currentRead);
        this.percentageOfTimeInGarbageCollection = calculatePercentageOfTimeInGarbageCollection();
    }

    public Duration getTimeSpentOnCollection() {
        return timeSpentOnCollection;
    }

    public long getCollectionCount() {
        return collectionsCount;
    }

    public Duration getTimeSinceLastPoll() {
        return timeSinceLastPoll;
    }

    public double getPercentageOfTimeInGarbageCollection() {
        return percentageOfTimeInGarbageCollection;
    }

    private Duration calculateTimeSpentOnGC(GCRead previousRead, GCRead currentRead) {
        return currentRead.getCollectionTime().minus(previousRead.getCollectionTime());
    }

    private long calculateCollectionsCount(GCRead previousRead, GCRead currentRead) {
        return currentRead.getCollectionCount() - previousRead.getCollectionCount();
    }

    private Duration calculateTimeSinceLastPoll(GCRead previousRead, GCRead currentRead) {
        return Duration.between(previousRead.getTimestamp(), currentRead.getTimestamp());
    }

    private double calculatePercentageOfTimeInGarbageCollection() {
        return ((double) timeSpentOnCollection.toMillis() * 100D) / (double) timeSinceLastPoll.toMillis();
    }
}
