package com.atlassian.diagnostics.internal.platform.monitor.gc;

import com.atlassian.diagnostics.MonitoringService;
import com.atlassian.diagnostics.Severity;
import com.atlassian.diagnostics.detail.ThreadDumpProducer;
import com.atlassian.diagnostics.internal.InitializingMonitor;
import com.atlassian.diagnostics.internal.jmx.JmxService;
import com.atlassian.diagnostics.internal.jmx.ThreadMemoryAllocation;
import com.atlassian.diagnostics.internal.jmx.ThreadMemoryAllocationService;
import com.atlassian.diagnostics.internal.platform.plugin.PluginFinder;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryUsage;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;

public class GarbageCollectionMonitor extends InitializingMonitor {

    private static final String KEY_PREFIX = "diagnostics.gc.issue";
    private static final int HIGH_GARBAGE_COLLECTION_TIME_WARNING = 2001;
    private static final int HIGH_GARBAGE_COLLECTION_TIME_ERROR = 3001;
    private static final Map<Severity, Integer> SEVERITY_TO_ID = ImmutableMap.of(
            Severity.WARNING, HIGH_GARBAGE_COLLECTION_TIME_WARNING,
            Severity.ERROR, HIGH_GARBAGE_COLLECTION_TIME_ERROR
    );

    private final GarbageCollectionMonitorConfiguration garbageCollectionMonitorConfiguration;
    private final JmxService jmxService;
    private final ThreadMemoryAllocationService threadMemoryAllocationService;
    private final PluginFinder pluginFinder;
    private final ThreadDumpProducer threadDumpProducer;

    public GarbageCollectionMonitor(final GarbageCollectionMonitorConfiguration garbageCollectionMonitorConfiguration,
                                    final JmxService jmxService,
                                    final ThreadMemoryAllocationService threadMemoryAllocationService,
                                    final PluginFinder pluginFinder,
                                    final ThreadDumpProducer threadDumpProducer) {
        this.garbageCollectionMonitorConfiguration = garbageCollectionMonitorConfiguration;
        this.jmxService = jmxService;
        this.threadMemoryAllocationService = threadMemoryAllocationService;
        this.pluginFinder = pluginFinder;
        this.threadDumpProducer = threadDumpProducer;
    }

    @Override
    public void init(@Nonnull final MonitoringService monitoringService) {
        monitor = monitoringService.createMonitor("GC", "diagnostics.gc.name", garbageCollectionMonitorConfiguration);

        defineIssue(KEY_PREFIX, HIGH_GARBAGE_COLLECTION_TIME_WARNING, Severity.WARNING);
        defineIssue(KEY_PREFIX, HIGH_GARBAGE_COLLECTION_TIME_ERROR, Severity.ERROR);
    }

    public void raiseAlertForHighGarbageCollectionTime(HighGCTimeDetails highGCTimeDetails) {
        alert(SEVERITY_TO_ID.getOrDefault(highGCTimeDetails.getSeverity(), HIGH_GARBAGE_COLLECTION_TIME_WARNING), builder -> builder
                .timestamp(highGCTimeDetails.getTimestamp())
                .details(() -> getHighGarbageCollectionTimeAlertDetails(highGCTimeDetails))
        );
    }

    private Map<String, Object> getHighGarbageCollectionTimeAlertDetails(HighGCTimeDetails highGCTimeDetails) {
        final ImmutableMap.Builder<String, Object> alertDetails = new ImmutableMap.Builder<String, Object>()
                .put("timeWindow", highGCTimeDetails.getTimeWindow().getSeconds())
                .put("garbageCollectionTimeInMillis", highGCTimeDetails.getGarbageCollectionTime().toMillis())
                .put("garbageCollectionCount", highGCTimeDetails.getGarbageCollectionCount())
                .put("garbageCollectorName", highGCTimeDetails.getGarbageCollectorName());

        addMemoryPoolDetails(alertDetails);

        if (garbageCollectionMonitorConfiguration.shouldIncludeTopThreadMemoryAllocationsInDetails()) {
            addTopFiveThreadAllocations(alertDetails);
        }

        return alertDetails.build();
    }

    private void addMemoryPoolDetails(final ImmutableMap.Builder<String, Object> alertDetails) {
        for (final MemoryPoolMXBean memoryPoolMXBean : jmxService.getMemoryPoolMXBeans()) {
            final MemoryUsage memoryUsage = memoryPoolMXBean.getUsage();
            alertDetails.put(memoryPoolMXBean.getName(), memoryUsage);
        }
    }

    private void addTopFiveThreadAllocations(final ImmutableMap.Builder<String, Object> alertDetails) {
        final List<ThreadMemoryAllocation> topFiveThreadMemoryAllocations = threadMemoryAllocationService.getTopThreadMemoryAllocations(5);
        final Map<String, Object> topFiveThreadMemoryAllocationDetails = topThreadMemoryAllocationDetails(topFiveThreadMemoryAllocations);
        if (!topFiveThreadMemoryAllocationDetails.isEmpty()) {
            alertDetails.put("topFiveThreadMemoryAllocations", topFiveThreadMemoryAllocationDetails);
        }
    }

    private ImmutableMap<String, Object> topThreadMemoryAllocationDetails(final List<ThreadMemoryAllocation> threadMemoryAllocations) {
        final ImmutableMap.Builder<String, Object> topThreadAllocationDetails = new ImmutableMap.Builder<>();
        for (final ThreadMemoryAllocation threadMemoryAllocation : threadMemoryAllocations) {
            topThreadAllocationDetails.put(threadMemoryAllocation.getThreadName(), threadMemoryAllocationDetails(threadMemoryAllocation));
        }
        return topThreadAllocationDetails.build();
    }

    private String threadMemoryAllocationDetails(final ThreadMemoryAllocation threadMemoryAllocation) {
        final ImmutableMap.Builder<String, Object> threadMemoryAllocationDetails = new ImmutableMap.Builder<>();
        threadMemoryAllocationDetails.put("memoryAllocationInBytes", threadMemoryAllocation.getMemoryAllocationInBytes());

        final String stackTrace = threadDumpProducer.toStackTraceString(Arrays.asList(threadMemoryAllocation.getStackTrace()));
        if (isNotEmpty(stackTrace)) {
            threadMemoryAllocationDetails.put("stackTrace", stackTrace);
        }

        final Collection<String> plugins = getPluginNamesFromStackTrace(threadMemoryAllocation);
        if (!plugins.isEmpty()) {
            threadMemoryAllocationDetails.put("plugins", String.join(" -> ", plugins));
        }

        return threadMemoryAllocationDetails.build().toString();
    }

    private Collection<String> getPluginNamesFromStackTrace(ThreadMemoryAllocation threadMemoryAllocation) {
        return pluginFinder.getPluginNamesFromStackTrace(threadMemoryAllocation.getStackTrace());
    }
}
