package com.atlassian.diagnostics.internal.platform.monitor.scheduler;

import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.SchedulerHistoryService;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.caesium.impl.CaesiumSchedulerService;
import com.atlassian.scheduler.caesium.spi.CaesiumSchedulerConfiguration;
import com.atlassian.scheduler.config.JobRunnerKey;
import com.atlassian.scheduler.core.SchedulerServiceController;
import com.atlassian.scheduler.status.JobDetails;
import com.atlassian.scheduler.status.RunDetails;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class DefaultSchedulerDiagnosticProvider implements SchedulerDiagnosticProvider {

    private final CaesiumSchedulerConfiguration caesiumSchedulerConfiguration;
    private final SchedulerHistoryService schedulerHistoryService;
    private final SchedulerServiceController schedulerServiceController;
    private final SchedulerService schedulerService;

    private Optional<CaesiumSchedulerService> caesiumSchedulerService = Optional.empty();

    public DefaultSchedulerDiagnosticProvider(final CaesiumSchedulerConfiguration caesiumSchedulerConfiguration,
                                              final SchedulerHistoryService schedulerHistoryService,
                                              final SchedulerServiceController schedulerServiceController,
                                              final SchedulerService schedulerService) {
        this.caesiumSchedulerConfiguration = caesiumSchedulerConfiguration;
        this.schedulerHistoryService = schedulerHistoryService;
        this.schedulerServiceController = schedulerServiceController;
        this.schedulerService = schedulerService;
        if (schedulerService instanceof CaesiumSchedulerService) {
            this.caesiumSchedulerService = Optional.of((CaesiumSchedulerService) schedulerService);
        } else if (schedulerServiceController instanceof CaesiumSchedulerService) {
            this.caesiumSchedulerService = Optional.of((CaesiumSchedulerService) schedulerServiceController);
        }
    }

    @Override
    public SchedulerDiagnostic getDiagnostic() {
        final int workerThreadCount = caesiumSchedulerConfiguration.workerThreadCount();
        return new SchedulerDiagnostic(workerThreadCount, getRunningJobs(), getScheduledJobs());
    }

    private List<RunningJobDiagnostic> getRunningJobs() {
        return schedulerServiceController.getLocallyRunningJobs().stream()
                .map(runningJob -> {
                    final Optional<JobRunner> jobRunner = getJobRunner(runningJob.getJobConfig().getJobRunnerKey());
                    return new RunningJobDiagnostic(runningJob, jobRunner);
                }).collect(Collectors.toList());
    }

    private List<ScheduledJobDiagnostic> getScheduledJobs() {
        return schedulerService.getJobRunnerKeysForAllScheduledJobs().stream()
                .flatMap(key -> Stream.of(schedulerService.getJobsByJobRunnerKey(key)))
                .flatMap(jobsForRunnerKey -> jobsForRunnerKey.stream()
                    .map(jobDetails -> new ScheduledJobDiagnostic(
                            jobDetails,
                            getRunDetails(jobDetails),
                            getJobRunner(jobDetails.getJobRunnerKey())
                    )))
                .collect(Collectors.toList());
    }

    private Optional<RunDetails> getRunDetails(final JobDetails jobDetails) {
        return Optional.ofNullable(schedulerHistoryService.getLastRunForJob(jobDetails.getJobId()));
    }

    private Optional<JobRunner> getJobRunner(final JobRunnerKey jobRunnerKey) {
        return caesiumSchedulerService.map(service -> service.getJobRunner(jobRunnerKey));
    }
}
