/*
 * Copyright 2006-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.consol.citrus.xml.schema;

import com.consol.citrus.exceptions.CitrusRuntimeException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.io.*;
import org.springframework.util.Assert;
import org.springframework.xml.validation.XmlValidator;
import org.springframework.xml.validation.XmlValidatorFactory;
import org.springframework.xml.xsd.SimpleXsdSchema;
import org.xml.sax.SAXException;

import javax.wsdl.WSDLException;
import javax.wsdl.extensions.schema.*;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.util.*;

/**
 * @author Christoph Deppisch
 * @since 2.4
 */
public abstract class AbstractSchemaCollection extends SimpleXsdSchema implements InitializingBean {

    /** List of schema resources */
    protected List<Resource> schemaResources = new ArrayList<>();

    /** Imported schemas */
    protected List<String> importedSchemas = new ArrayList<>();

    /** Official xmlns namespace */
    public static final String WWW_W3_ORG_2000_XMLNS = "http://www.w3.org/2000/xmlns/";
    public static final String W3C_XML_SCHEMA_NS_URI = "http://www.w3.org/2001/XMLSchema";

    @Override
    public XmlValidator createValidator() {
        try {
            return XmlValidatorFactory.createValidator(schemaResources.toArray(new Resource[schemaResources.size()]), W3C_XML_SCHEMA_NS_URI);
        } catch (IOException e) {
            throw new CitrusRuntimeException("Failed to create validator from multi resource schema files", e);
        }
    }

    /**
     * Recursively add all imported schemas as schema resource.
     * This is necessary when schema import are located in jar files. If they are not added immediately the reference to them is lost.
     *
     * @param schema
     */
    protected void addImportedSchemas(Schema schema) throws WSDLException, IOException, TransformerException, TransformerFactoryConfigurationError {
        for (Object imports : schema.getImports().values()) {
            for (SchemaImport schemaImport : (Vector<SchemaImport>)imports) {
                // Prevent duplicate imports
                if (!importedSchemas.contains(schemaImport.getNamespaceURI())) {
                    importedSchemas.add(schemaImport.getNamespaceURI());
                    Schema referencedSchema = schemaImport.getReferencedSchema();

                    if (referencedSchema != null) {
                        ByteArrayOutputStream bos = new ByteArrayOutputStream();
                        Source source = new DOMSource(referencedSchema.getElement());
                        Result result = new StreamResult(bos);

                        TransformerFactory.newInstance().newTransformer().transform(source, result);
                        Resource schemaResource = new ByteArrayResource(bos.toByteArray());

                        addImportedSchemas(referencedSchema);
                        schemaResources.add(schemaResource);
                    }
                }
            }
        }
    }

    /**
     * Recursively add all included schemas as schema resource.
     */
    protected void addIncludedSchemas(Schema schema) throws WSDLException, IOException, TransformerException, TransformerFactoryConfigurationError {
        List<SchemaReference> includes = schema.getIncludes();
        for (SchemaReference schemaReference : includes) {
            String schemaLocation;
            URI locationURI = URI.create(schemaReference.getSchemaLocationURI());
            if (locationURI.isAbsolute()) {
                schemaLocation = schemaReference.getSchemaLocationURI();
            } else {
                schemaLocation = schema.getDocumentBaseURI().substring(0, schema.getDocumentBaseURI().lastIndexOf('/') + 1) + schemaReference.getSchemaLocationURI();
            }

            schemaResources.add(new FileSystemResource(schemaLocation));
        }
    }

    @Override
    public void afterPropertiesSet() throws ParserConfigurationException, IOException, SAXException {
        Resource targetXsd = loadSchemaResources();

        if (targetXsd == null) {
            throw new CitrusRuntimeException("Failed to find target schema xsd file resource");
        }

        Assert.isTrue(!schemaResources.isEmpty(), "At least one schema xsd file resource is required");
        setXsd(targetXsd);

        super.afterPropertiesSet();
    }

    /**
     * Loads all schema resource files from schema locations.
     */
    protected abstract Resource loadSchemaResources();

    /**
     * Gets the schema resources.
     * @return
     */
    public List<Resource> getSchemaResources() {
        return schemaResources;
    }
}
