/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.CloseFuture;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ClusterNameMismatchException;
import com.datastax.driver.core.Connection;
import com.datastax.driver.core.ConnectionException;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.PooledConnection;
import com.datastax.driver.core.PoolingOptions;
import com.datastax.driver.core.SessionManager;
import com.datastax.driver.core.UnsupportedProtocolVersionException;
import com.datastax.driver.core.exceptions.AuthenticationException;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.MoreExecutors;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HostConnectionPool {
    private static final Logger logger = LoggerFactory.getLogger(HostConnectionPool.class);
    private static final int MAX_SIMULTANEOUS_CREATION = 1;
    private static final int MIN_AVAILABLE_STREAMS = 96;
    public final Host host;
    public volatile HostDistance hostDistance;
    private final SessionManager manager;
    final List<PooledConnection> connections;
    private final AtomicInteger open;
    private final Set<Connection> trash = new CopyOnWriteArraySet<Connection>();
    private volatile int waiter = 0;
    private final Lock waitLock = new ReentrantLock(true);
    private final Condition hasAvailableConnection = this.waitLock.newCondition();
    private final Runnable newConnectionTask;
    private final AtomicInteger scheduledForCreation = new AtomicInteger();
    private final AtomicReference<CloseFuture> closeFuture = new AtomicReference();

    public HostConnectionPool(Host host, HostDistance hostDistance, SessionManager manager) throws ConnectionException, UnsupportedProtocolVersionException, ClusterNameMismatchException {
        assert (hostDistance != HostDistance.IGNORED);
        this.host = host;
        this.hostDistance = hostDistance;
        this.manager = manager;
        this.newConnectionTask = new Runnable(){

            @Override
            public void run() {
                HostConnectionPool.this.addConnectionIfUnderMaximum();
                HostConnectionPool.this.scheduledForCreation.decrementAndGet();
            }
        };
        ArrayList<PooledConnection> l = new ArrayList<PooledConnection>(this.options().getCoreConnectionsPerHost(hostDistance));
        try {
            for (int i = 0; i < this.options().getCoreConnectionsPerHost(hostDistance); ++i) {
                l.add(manager.connectionFactory().open(this));
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (RuntimeException e) {
            for (PooledConnection connection : l) {
                connection.closeAsync().force();
            }
            throw e;
        }
        this.connections = new CopyOnWriteArrayList<PooledConnection>(l);
        this.open = new AtomicInteger(this.connections.size());
        logger.trace("Created connection pool to host {}", (Object)host);
    }

    private PoolingOptions options() {
        return this.manager.configuration().getPoolingOptions();
    }

    public PooledConnection borrowConnection(long timeout, TimeUnit unit) throws ConnectionException, TimeoutException {
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getSocketAddress(), "Pool is shutdown");
        }
        if (this.connections.isEmpty()) {
            for (int i = 0; i < this.options().getCoreConnectionsPerHost(this.hostDistance); ++i) {
                this.scheduledForCreation.incrementAndGet();
                this.manager.blockingExecutor().submit(this.newConnectionTask);
            }
            PooledConnection c = this.waitForConnection(timeout, unit);
            c.setKeyspace(this.manager.poolsState.keyspace);
            return c;
        }
        int minInFlight = Integer.MAX_VALUE;
        PooledConnection leastBusy = null;
        for (PooledConnection connection : this.connections) {
            int inFlight = connection.inFlight.get();
            if (inFlight >= minInFlight) continue;
            minInFlight = inFlight;
            leastBusy = connection;
        }
        if (minInFlight >= this.options().getMaxSimultaneousRequestsPerConnectionThreshold(this.hostDistance) && this.connections.size() < this.options().getMaxConnectionsPerHost(this.hostDistance)) {
            this.maybeSpawnNewConnection();
        }
        if (leastBusy == null) {
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getSocketAddress(), "Pool is shutdown");
            }
            leastBusy = this.waitForConnection(timeout, unit);
        } else {
            int inFlight;
            do {
                if ((inFlight = leastBusy.inFlight.get()) < leastBusy.maxAvailableStreams()) continue;
                leastBusy = this.waitForConnection(timeout, unit);
                break;
            } while (!leastBusy.inFlight.compareAndSet(inFlight, inFlight + 1));
        }
        leastBusy.setKeyspace(this.manager.poolsState.keyspace);
        return leastBusy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void awaitAvailableConnection(long timeout, TimeUnit unit) throws InterruptedException {
        this.waitLock.lock();
        ++this.waiter;
        try {
            this.hasAvailableConnection.await(timeout, unit);
        }
        finally {
            --this.waiter;
            this.waitLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void signalAvailableConnection() {
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signal();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void signalAllAvailableConnection() {
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signalAll();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    private PooledConnection waitForConnection(long timeout, TimeUnit unit) throws ConnectionException, TimeoutException {
        if (timeout == 0L) {
            throw new TimeoutException();
        }
        long start = System.nanoTime();
        long remaining = timeout;
        do {
            int inFlight;
            try {
                this.awaitAvailableConnection(remaining, unit);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                timeout = 0L;
            }
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getSocketAddress(), "Pool is shutdown");
            }
            int minInFlight = Integer.MAX_VALUE;
            Connection leastBusy = null;
            for (PooledConnection connection : this.connections) {
                int inFlight2 = connection.inFlight.get();
                if (inFlight2 >= minInFlight) continue;
                minInFlight = inFlight2;
                leastBusy = connection;
            }
            if (leastBusy == null) continue;
            while ((inFlight = ((PooledConnection)leastBusy).inFlight.get()) < leastBusy.maxAvailableStreams()) {
                if (!((PooledConnection)leastBusy).inFlight.compareAndSet(inFlight, inFlight + 1)) continue;
                return leastBusy;
            }
        } while ((remaining = timeout - Cluster.timeSince(start, unit)) > 0L);
        throw new TimeoutException();
    }

    public void returnConnection(PooledConnection connection) {
        if (this.isClosed()) {
            this.close(connection);
            return;
        }
        if (connection.isDefunct()) {
            return;
        }
        int inFlight = connection.inFlight.decrementAndGet();
        if (this.trash.contains(connection)) {
            if (inFlight == 0 && this.trash.remove(connection)) {
                this.close(connection);
            }
        } else if (this.connections.size() > this.options().getCoreConnectionsPerHost(this.hostDistance) && inFlight <= this.options().getMinSimultaneousRequestsPerConnectionThreshold(this.hostDistance)) {
            connection.setTrashTimeIn(this.options().getIdleTimeoutSeconds());
        } else if (connection.maxAvailableStreams() < 96) {
            this.replaceConnection(connection);
        } else {
            connection.cancelTrashTime();
            this.signalAvailableConnection();
        }
    }

    void trashIdleConnections(long now) {
        for (PooledConnection connection : this.connections) {
            if (connection.getTrashTime() >= now) continue;
            this.trashConnection(connection);
        }
    }

    private void replaceConnection(PooledConnection connection) {
        if (connection.markForTrash.compareAndSet(false, true)) {
            this.open.decrementAndGet();
        }
        this.maybeSpawnNewConnection();
        this.doTrashConnection(connection);
    }

    private boolean trashConnection(PooledConnection connection) {
        if (connection.markForTrash.compareAndSet(false, true)) {
            int opened;
            do {
                if ((opened = this.open.get()) > this.options().getCoreConnectionsPerHost(this.hostDistance)) continue;
                connection.markForTrash.set(false);
                connection.cancelTrashTime();
                return false;
            } while (!this.open.compareAndSet(opened, opened - 1));
            this.doTrashConnection(connection);
        }
        return true;
    }

    private void doTrashConnection(PooledConnection connection) {
        this.trash.add(connection);
        this.connections.remove(connection);
        if (connection.inFlight.get() == 0 && this.trash.remove(connection)) {
            this.close(connection);
        }
    }

    private boolean addConnectionIfUnderMaximum() {
        int opened;
        do {
            if ((opened = this.open.get()) < this.options().getMaxConnectionsPerHost(this.hostDistance)) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened + 1));
        if (this.isClosed()) {
            this.open.decrementAndGet();
            return false;
        }
        try {
            this.connections.add(this.manager.connectionFactory().open(this));
            this.signalAvailableConnection();
            return true;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.open.decrementAndGet();
            return false;
        }
        catch (ConnectionException e) {
            this.open.decrementAndGet();
            logger.debug("Connection error to {} while creating additional connection", (Object)this.host);
            return false;
        }
        catch (AuthenticationException e) {
            this.open.decrementAndGet();
            logger.error("Authentication error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
        catch (UnsupportedProtocolVersionException e) {
            this.open.decrementAndGet();
            logger.error("UnsupportedProtocolVersionException error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
        catch (ClusterNameMismatchException e) {
            this.open.decrementAndGet();
            logger.error("ClusterNameMismatchException error while creating additional connection (error is: {})", (Object)e.getMessage());
            return false;
        }
    }

    private void maybeSpawnNewConnection() {
        int inCreation;
        do {
            if ((inCreation = this.scheduledForCreation.get()) < 1) continue;
            return;
        } while (!this.scheduledForCreation.compareAndSet(inCreation, inCreation + 1));
        logger.debug("Creating new connection on busy pool to {}", (Object)this.host);
        this.manager.blockingExecutor().submit(this.newConnectionTask);
    }

    void replaceDefunctConnection(PooledConnection connection) {
        if (connection.markForTrash.compareAndSet(false, true)) {
            this.open.decrementAndGet();
        }
        this.connections.remove(connection);
        connection.closeAsync();
        this.manager.blockingExecutor().submit(new Runnable(){

            @Override
            public void run() {
                HostConnectionPool.this.addConnectionIfUnderMaximum();
            }
        });
    }

    private void close(Connection connection) {
        connection.closeAsync();
    }

    public boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    public CloseFuture closeAsync() {
        CloseFuture future = this.closeFuture.get();
        if (future != null) {
            return future;
        }
        this.signalAllAvailableConnection();
        future = new CloseFuture.Forwarding(this.discardAvailableConnections());
        return this.closeFuture.compareAndSet(null, future) ? future : this.closeFuture.get();
    }

    public int opened() {
        return this.open.get();
    }

    private List<CloseFuture> discardAvailableConnections() {
        if (this.connections == null) {
            return Lists.newArrayList((Object[])new CloseFuture[]{CloseFuture.immediateFuture()});
        }
        ArrayList<CloseFuture> futures = new ArrayList<CloseFuture>(this.connections.size());
        for (final PooledConnection connection : this.connections) {
            CloseFuture future = connection.closeAsync();
            future.addListener(new Runnable(){

                @Override
                public void run() {
                    if (connection.markForTrash.compareAndSet(false, true)) {
                        HostConnectionPool.this.open.decrementAndGet();
                    }
                }
            }, (Executor)MoreExecutors.sameThreadExecutor());
            futures.add(future);
        }
        return futures;
    }

    public void ensureCoreConnections() {
        int opened;
        if (this.isClosed()) {
            return;
        }
        for (int i = opened = this.open.get(); i < this.options().getCoreConnectionsPerHost(this.hostDistance); ++i) {
            this.scheduledForCreation.incrementAndGet();
            this.manager.blockingExecutor().submit(this.newConnectionTask);
        }
    }

    static class PoolState {
        volatile String keyspace;

        PoolState() {
        }

        public void setKeyspace(String keyspace) {
            this.keyspace = keyspace;
        }
    }
}

