/*
 * Decompiled with CFR 0.152.
 */
package com.linuxense.javadbf;

import com.linuxense.javadbf.DBFBase;
import com.linuxense.javadbf.DBFDataType;
import com.linuxense.javadbf.DBFException;
import com.linuxense.javadbf.DBFField;
import com.linuxense.javadbf.DBFHeader;
import com.linuxense.javadbf.DBFMemoFile;
import com.linuxense.javadbf.DBFUtils;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

public class DBFReader
extends DBFBase
implements Closeable {
    private static final long MILLISECS_PER_DAY = 86400000L;
    private static final long TIME_MILLIS_1_1_4713_BC = -210866803200000L;
    private DataInputStream dataInputStream;
    private DBFHeader header;
    private boolean trimRightSpaces = true;
    private boolean showDeletedRows = false;
    private DBFMemoFile memoFile = null;
    private boolean closed = false;

    public DBFReader(InputStream in) {
        this(in, null, false);
    }

    public DBFReader(InputStream in, Boolean showDeletedRows) {
        this(in, null, showDeletedRows);
    }

    public DBFReader(InputStream in, Charset charset) {
        this(in, charset, false);
    }

    public DBFReader(InputStream in, Charset charset, boolean showDeletedRows) {
        try {
            this.dataInputStream = new DataInputStream(in);
            this.header = new DBFHeader();
            this.header.read(this.dataInputStream, charset);
            this.setCharset(this.header.getUsedCharset());
            this.showDeletedRows = showDeletedRows;
            int fieldSize = this.header.getFieldDescriptorSize();
            int tableSize = this.header.getTableHeaderSize();
            int t_dataStartIndex = this.header.headerLength - (tableSize + fieldSize * this.header.fieldArray.length) - 1;
            this.skip(t_dataStartIndex);
        }
        catch (IOException e) {
            DBFUtils.close(this.dataInputStream);
            DBFUtils.close(in);
            throw new DBFException(e.getMessage(), e);
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(128);
        sb.append(this.header.getYear()).append("/");
        sb.append(this.header.getMonth()).append("/");
        sb.append(this.header.getDay()).append("\n");
        sb.append("Total records: ").append(this.header.numberOfRecords).append("\n");
        sb.append("Header length: ").append(this.header.headerLength).append("\n");
        sb.append("Columns:\n");
        for (DBFField field : this.header.fieldArray) {
            sb.append(field.getName());
            sb.append("\n");
        }
        return sb.toString();
    }

    public int getRecordCount() {
        return this.header.numberOfRecords;
    }

    public Date getLastModificationDate() {
        if (this.header != null) {
            return this.header.getLastModificationDate();
        }
        return null;
    }

    public DBFField getField(int index) {
        if (this.showDeletedRows) {
            if (index == 0) {
                return new DBFField("DELETED", DBFDataType.LOGICAL);
            }
            return new DBFField(this.header.fieldArray[index - 1]);
        }
        return new DBFField(this.header.fieldArray[index]);
    }

    public int getFieldCount() {
        return this.header.userFieldArray.length + (this.showDeletedRows ? 1 : 0);
    }

    public Object[] nextRecord() {
        if (this.closed) {
            throw new IllegalArgumentException("this DBFReader is closed");
        }
        ArrayList<Object> recordObjects = new ArrayList<Object>(this.getFieldCount());
        try {
            boolean isDeleted = false;
            do {
                byte t_byte;
                if (isDeleted && !this.showDeletedRows) {
                    this.skip(this.header.recordLength - 1);
                }
                if ((t_byte = this.dataInputStream.readByte()) == 26) {
                    return null;
                }
                boolean bl = isDeleted = t_byte == 42;
            } while (isDeleted && !this.showDeletedRows);
            if (this.showDeletedRows) {
                recordObjects.add(isDeleted);
            }
            for (int i = 0; i < this.header.fieldArray.length; ++i) {
                DBFField field = this.header.fieldArray[i];
                Object o = this.getFieldValue(field);
                if (field.isSystem()) {
                    if (field.getType() != DBFDataType.NULL_FLAGS || !(o instanceof BitSet)) continue;
                    BitSet nullFlags = (BitSet)o;
                    int currentIndex = -1;
                    for (int j = 0; j < this.header.fieldArray.length; ++j) {
                        DBFField field1 = this.header.fieldArray[j];
                        if (field1.isNullable() && nullFlags.get(++currentIndex)) {
                            recordObjects.set(j, null);
                        }
                        if (field1.getType() != DBFDataType.VARBINARY && field1.getType() != DBFDataType.VARCHAR) continue;
                        ++currentIndex;
                        if (!(recordObjects.get(i) instanceof byte[])) continue;
                        byte[] data = (byte[])recordObjects.get(j);
                        int size = field1.getLength();
                        if (!nullFlags.get(currentIndex)) {
                            size = data[data.length - 1];
                        }
                        byte[] newData = new byte[size];
                        System.arraycopy(data, 0, newData, 0, size);
                        Object o1 = newData;
                        if (field1.getType() == DBFDataType.VARCHAR) {
                            o1 = new String(newData, this.getCharset());
                        }
                        recordObjects.set(j, o1);
                    }
                    continue;
                }
                recordObjects.add(o);
            }
        }
        catch (EOFException e) {
            return null;
        }
        catch (IOException e) {
            throw new DBFException(e.getMessage(), e);
        }
        return recordObjects.toArray();
    }

    private Object getFieldValue(DBFField field) throws IOException {
        int bytesReaded = 0;
        switch (field.getType()) {
            case CHARACTER: {
                byte[] b_array = new byte[field.getLength()];
                bytesReaded = this.dataInputStream.read(b_array);
                if (bytesReaded < field.getLength()) {
                    throw new EOFException("Unexpected end of file");
                }
                if (this.trimRightSpaces) {
                    return new String(DBFUtils.trimRightSpaces(b_array), this.getCharset());
                }
                return new String(b_array, this.getCharset());
            }
            case VARCHAR: 
            case VARBINARY: {
                byte[] b_array_var = new byte[field.getLength()];
                bytesReaded = this.dataInputStream.read(b_array_var);
                if (bytesReaded < field.getLength()) {
                    throw new EOFException("Unexpected end of file");
                }
                return b_array_var;
            }
            case DATE: {
                byte[] t_byte_year = new byte[4];
                bytesReaded = this.dataInputStream.read(t_byte_year);
                if (bytesReaded < 4) {
                    throw new EOFException("Unexpected end of file");
                }
                byte[] t_byte_month = new byte[2];
                bytesReaded = this.dataInputStream.read(t_byte_month);
                if (bytesReaded < 2) {
                    throw new EOFException("Unexpected end of file");
                }
                byte[] t_byte_day = new byte[2];
                bytesReaded = this.dataInputStream.read(t_byte_day);
                if (bytesReaded < 2) {
                    throw new EOFException("Unexpected end of file");
                }
                try {
                    GregorianCalendar calendar = new GregorianCalendar(Integer.parseInt(new String(t_byte_year, StandardCharsets.US_ASCII)), Integer.parseInt(new String(t_byte_month, StandardCharsets.US_ASCII)) - 1, Integer.parseInt(new String(t_byte_day, StandardCharsets.US_ASCII)));
                    return calendar.getTime();
                }
                catch (NumberFormatException e) {
                    return null;
                }
            }
            case FLOATING_POINT: 
            case NUMERIC: {
                return DBFUtils.readNumericStoredAsText(this.dataInputStream, field.getLength());
            }
            case LOGICAL: {
                byte t_logical = this.dataInputStream.readByte();
                return DBFUtils.toBoolean(t_logical);
            }
            case LONG: 
            case AUTOINCREMENT: {
                int data = DBFUtils.readLittleEndianInt(this.dataInputStream);
                return data;
            }
            case CURRENCY: {
                int c_data = DBFUtils.readLittleEndianInt(this.dataInputStream);
                String s_data = String.format("%05d", c_data);
                String x1 = s_data.substring(0, s_data.length() - 4);
                String x2 = s_data.substring(s_data.length() - 4);
                this.skip(field.getLength() - 4);
                return new BigDecimal(x1 + "." + x2);
            }
            case TIMESTAMP: 
            case TIMESTAMP_DBASE7: {
                int days = DBFUtils.readLittleEndianInt(this.dataInputStream);
                int time = DBFUtils.readLittleEndianInt(this.dataInputStream);
                if (days == 0 && time == 0) {
                    return null;
                }
                GregorianCalendar calendar = new GregorianCalendar();
                calendar.setTimeInMillis((long)days * 86400000L + -210866803200000L + (long)time);
                ((Calendar)calendar).add(14, -TimeZone.getDefault().getOffset(calendar.getTimeInMillis()));
                return calendar.getTime();
            }
            case MEMO: 
            case GENERAL_OLE: 
            case PICTURE: 
            case BLOB: {
                return this.readMemoField(field);
            }
            case BINARY: {
                if (field.getLength() == 8) {
                    return this.readDoubleField(field);
                }
                return this.readMemoField(field);
            }
            case DOUBLE: {
                return this.readDoubleField(field);
            }
            case NULL_FLAGS: {
                byte[] data1 = new byte[field.getLength()];
                int readed = this.dataInputStream.read(data1);
                if (readed != field.getLength()) {
                    throw new EOFException("Unexpected end of file");
                }
                return BitSet.valueOf(data1);
            }
        }
        this.skip(field.getLength());
        return null;
    }

    private Object readDoubleField(DBFField field) throws IOException {
        byte[] data = new byte[field.getLength()];
        int bytesReaded = this.dataInputStream.read(data);
        if (bytesReaded < field.getLength()) {
            throw new EOFException("Unexpected end of file");
        }
        return ByteBuffer.wrap(new byte[]{data[7], data[6], data[5], data[4], data[3], data[2], data[1], data[0]}).getDouble();
    }

    private Object readMemoField(DBFField field) throws IOException {
        Number nBlock = null;
        nBlock = field.getLength() == 10 ? (Number)DBFUtils.readNumericStoredAsText(this.dataInputStream, field.getLength()) : (Number)DBFUtils.readLittleEndianInt(this.dataInputStream);
        if (this.memoFile != null && nBlock != null) {
            return this.memoFile.readData(nBlock.intValue(), field.getType());
        }
        return null;
    }

    private void skip(int n) throws IOException {
        int skipped;
        for (int i = skipped = (int)this.dataInputStream.skip(n); i < n; ++i) {
            this.dataInputStream.readByte();
        }
    }

    protected DBFHeader getHeader() {
        return this.header;
    }

    public boolean isTrimRightSpaces() {
        return this.trimRightSpaces;
    }

    public void setTrimRightSpaces(boolean trimRightSpaces) {
        this.trimRightSpaces = trimRightSpaces;
    }

    public void setMemoFile(File memoFile) {
        if (this.memoFile != null) {
            throw new IllegalStateException("Memo file is already setted");
        }
        if (!memoFile.exists()) {
            throw new DBFException("Memo file " + memoFile.getName() + " not exists");
        }
        if (!memoFile.canRead()) {
            throw new DBFException("Cannot read Memo file " + memoFile.getName());
        }
        this.memoFile = new DBFMemoFile(memoFile, this.getCharset());
    }

    @Override
    public void close() {
        this.closed = true;
        DBFUtils.close(this.dataInputStream);
    }
}

