/**
 * Copyright (c) 2009-2021 Dan Cunningham
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package com.digitaldan.jomnilinkII.MessageTypes;

import com.digitaldan.jomnilinkII.Message;

import lombok.Value;
import lombok.experimental.NonFinal;

/*
 * OBJECT PROPERTIES
 *
 * This message is sent by the HAI controller in reply to a REQUEST OBJECT
 * PROPERTIES message. The HAI controller reports the properties of the
 * specified object.
 *
 *     Start character      0x21
 *     Message length       number of data bytes + 1
 *     Message Type         0x21
 *     Data 1               object type
 *     Data 2               object number (MSB)
 *     Data 3               object number (LSB)
 *     ...
 *     Data n               last property
 *     CRC 1                varies
 *     CRC 2                varies
 *
 * The object type identifies whether the returned properties are for a zone,
 * unit, button, code, area, thermostat, message, auxiliary sensor, audio zone,
 * audio source, user setting, or access control reader object. The object
 * number identifies the specific object (zone, unit, button, code, area,
 * thermostat, message, auxiliary sensor, audio zone, audio source, user
 * setting, or access control reader object).
 *
 * The object name data specifies the name for the respective object. Each name
 * consists of one or more printable ASCII characters, followed by a terminating
 * zero. Zone and message names can be up to 15 characters long, exclusive of
 * the terminating zero. All other names may be up to 12 characters long. Names
 * are always transferred with a fixed number of data bytes for each name type,
 * as shown in the table below. The terminating zero indicates the actual end of
 * the name. Data bytes following the terminating zero may be filled with any
 * value. If the first character received is zero, the object is not named in
 * the controller.
 *
 * Listed below are the available object types and maximum name length for each
 * object type:
 *
 * -------------------------------------------------------------
 * | Object Type |  Object Description   | Maximum Name Length |
 * |-------------|-----------------------|---------------------|
 * | 1           | Zone                  | 15                  |
 * |-------------|-----------------------|---------------------|
 * | 2           | Unit                  | 12                  |
 * |-------------|-----------------------|---------------------|
 * | 3           | Button                | 12                  |
 * |-------------|-----------------------|---------------------|
 * | 4           | Code                  | 12                  |
 * |-------------|-----------------------|---------------------|
 * | 5           | Area                  | 12                  |
 * |-------------|-----------------------|---------------------|
 * | 6           | Thermostat            | 12                  |
 * |-------------|-----------------------|---------------------|
 * | 7           | Message               | 15                  |
 * |-------------|-----------------------|---------------------|
 * | 8           | Auxiliary Sensor      | 15                  |
 * |-------------|-----------------------|---------------------|
 * | 9           | Audio Source          | 12                  |
 * |-------------|-----------------------|---------------------|
 * | 10          | Audio Zone            | 12                  |
 * |-------------|-----------------------|---------------------|
 * | 13          | User Setting          | 15                  |
 * |-------------|-----------------------|---------------------|
 * | 14          | Access Control Reader | 15                  |
 * -------------------------------------------------------------
 *
 * See NAME DATA for additional information.
 */
@Value
@NonFinal
public class ObjectProperties implements Message {
	public final static int FILTER_1_NONE = 0;
	public final static int FILTER_1_NAMED_UNAMED = 0;
	public final static int FILTER_1_NAMED = 1;
	public final static int FILTER_1_UNAMED = 2;

	public final static int FILTER_2_NONE = 0;
	public final static int[] FILTER_2_AREA_BITMAP = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80};
	public final static int FILTER_2_AREA_1 = 0x01;
	public final static int FILTER_2_AREA_2 = 0x02;
	public final static int FILTER_2_AREA_3 = 0x04;
	public final static int FILTER_2_AREA_4 = 0x08;
	public final static int FILTER_2_AREA_5 = 0x10;
	public final static int FILTER_2_AREA_6 = 0x20;
	public final static int FILTER_2_AREA_7 = 0x40;
	public final static int FILTER_2_AREA_8 = 0x80;
	public final static int FILTER_2_AREA_ALL = 0xFF;

	public final static int FILTER_3_NONE = 0;
	public final static int FILTER_3_ANY_LOAD = 0;
	public final static int FILTER_3_ROOM_LOAD_START = 1;
	public final static int FILTER_3_ROOM_LOAD_END = 31; // load in room filter is 1 though 31
	public final static int FILTER_3_ROOM = 254;
	public final static int FILTER_3_INDEPENDENT_LOAD = 255;

	protected final int objectType;
	protected final int number;
	protected final String name;

	protected ObjectProperties(int objectType, int number, String name) {
		this.objectType = objectType;
		this.number = number;
		this.name = name;
	}

	@Override
	public int getMessageType() {
		return MESG_TYPE_OBJ_PROP;
	}
}
