/**
 * The MIT License (MIT)
 *
 * Copyright (c) 2014-2019 <mickael.jeanroy@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package com.github.mjeanroy.junit.servers.annotations;

import com.github.mjeanroy.junit.servers.client.HttpClientConfiguration;
import com.github.mjeanroy.junit.servers.client.HttpClientConfigurationFactory;
import com.github.mjeanroy.junit.servers.client.HttpClientStrategy;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Annotation that can be used to inject simple http
 * client to query embedded server.
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({
	ElementType.FIELD,
	ElementType.PARAMETER,
	ElementType.ANNOTATION_TYPE,
})
@Documented
@Inherited
public @interface TestHttpClient {

	/**
	 * Get strategy to use to build http client.
	 * Default is {@link HttpClientStrategy#AUTO} and classpath detection
	 * will be used to instantiate appropriate client implementation.
	 *
	 * @return Strategy, default is {@link HttpClientStrategy#AUTO}.
	 */
	HttpClientStrategy strategy() default HttpClientStrategy.AUTO;

	/**
	 * A configuration factory to use to create configuration for given HTTP Client.
	 *
	 * @return The HTTP Client configuration factory.
	 */
	Class<? extends HttpClientConfigurationFactory> configuration() default DefaultHttpClientConfigurationFactory.class;

	/**
	 * The default implementation, that just returns the default configuration.
	 */
	class DefaultHttpClientConfigurationFactory implements HttpClientConfigurationFactory {
		@Override
		public HttpClientConfiguration build() {
			return HttpClientConfiguration.defaultConfiguration();
		}
	}
}
