/*
 * Decompiled with CFR 0.152.
 */
package com.github.msemys.esjc.node.cluster;

import com.github.msemys.esjc.node.EndpointDiscoverer;
import com.github.msemys.esjc.node.NodeEndpoints;
import com.github.msemys.esjc.node.cluster.ClusterException;
import com.github.msemys.esjc.node.cluster.ClusterInfoDto;
import com.github.msemys.esjc.node.cluster.ClusterNodeSettings;
import com.github.msemys.esjc.node.cluster.GossipSeed;
import com.github.msemys.esjc.node.cluster.MemberInfoDto;
import com.github.msemys.esjc.node.cluster.VNodeState;
import com.github.msemys.esjc.util.Preconditions;
import com.github.msemys.esjc.util.Throwables;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.stream.JsonReader;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterEndpointDiscoverer
implements EndpointDiscoverer {
    private static final Logger logger = LoggerFactory.getLogger(ClusterEndpointDiscoverer.class);
    private final ScheduledExecutorService scheduler;
    private final AtomicReference<List<MemberInfoDto>> oldGossip = new AtomicReference();
    private final ClusterNodeSettings settings;
    private final Gson gson;

    public ClusterEndpointDiscoverer(ClusterNodeSettings settings, ScheduledExecutorService scheduler) {
        Preconditions.checkNotNull(settings, "settings is null");
        Preconditions.checkNotNull(scheduler, "scheduler is null");
        this.settings = settings;
        this.scheduler = scheduler;
        this.gson = new GsonBuilder().registerTypeAdapter(Instant.class, (json, type, ctx) -> Instant.parse(json.getAsJsonPrimitive().getAsString())).create();
    }

    @Override
    public CompletableFuture<NodeEndpoints> discover(InetSocketAddress failedTcpEndpoint) {
        CompletableFuture<NodeEndpoints> result = new CompletableFuture<NodeEndpoints>();
        if (this.settings.maxDiscoverAttempts != 0) {
            this.scheduler.execute(() -> this.discover(result, failedTcpEndpoint, 1));
        } else {
            result.completeExceptionally(new ClusterException("Cluster endpoint discover is not enabled."));
        }
        return result;
    }

    private void discover(CompletableFuture<NodeEndpoints> result, InetSocketAddress failedEndpoint, int attempt) {
        String attemptInfo = this.settings.maxDiscoverAttempts != -1 ? String.format("%d/%d", attempt, this.settings.maxDiscoverAttempts) : String.valueOf(attempt);
        try {
            Optional<NodeEndpoints> nodeEndpoints = this.tryDiscover(failedEndpoint);
            if (nodeEndpoints.isPresent()) {
                logger.info("Discovering attempt {} successful: best candidate is {}.", (Object)attemptInfo, (Object)nodeEndpoints.get());
                result.complete(nodeEndpoints.get());
            } else {
                logger.info("Discovering attempt {} failed: no candidate found.", (Object)attemptInfo);
            }
        }
        catch (Exception e) {
            logger.info("Discovering attempt {} failed.", (Object)attemptInfo, (Object)e);
        }
        if (!(result.isDone() || attempt >= this.settings.maxDiscoverAttempts && this.settings.maxDiscoverAttempts != -1)) {
            this.scheduler.schedule(() -> this.discover(result, failedEndpoint, attempt + 1), this.settings.discoverAttemptInterval.toMillis(), TimeUnit.MILLISECONDS);
        } else {
            result.completeExceptionally(new ClusterException(String.format("Failed to discover candidate in %d attempts.", attempt)));
        }
    }

    private Optional<NodeEndpoints> tryDiscover(InetSocketAddress failedEndpoint) {
        List oldGossipCopy = this.oldGossip.getAndSet(null);
        List<GossipSeed> gossipCandidates = oldGossipCopy != null ? this.getGossipCandidatesFromOldGossip(oldGossipCopy, failedEndpoint) : this.getGossipCandidatesFromDns();
        for (GossipSeed gossipCandidate : gossipCandidates) {
            Optional<NodeEndpoints> bestNode;
            Optional<ClusterInfoDto> gossip = this.tryGetGossipFrom(gossipCandidate).filter(c -> c.members != null && !c.members.isEmpty());
            if (!gossip.isPresent() || !(bestNode = this.tryDetermineBestNode(gossip.get().members)).isPresent()) continue;
            this.oldGossip.set(gossip.get().members);
            return bestNode;
        }
        return Optional.empty();
    }

    private List<GossipSeed> getGossipCandidatesFromDns() {
        List<GossipSeed> endpoints = !this.settings.gossipSeeds.isEmpty() ? new ArrayList<GossipSeed>(this.settings.gossipSeeds) : this.resolveDns().stream().map(address -> new GossipSeed(new InetSocketAddress((InetAddress)address, this.settings.externalGossipPort))).collect(Collectors.toList());
        if (endpoints.size() > 1) {
            Collections.shuffle(endpoints);
        }
        return endpoints;
    }

    private List<InetAddress> resolveDns() {
        try {
            InetAddress[] addresses = InetAddress.getAllByName(this.settings.dns);
            if (addresses == null || addresses.length == 0) {
                throw new ClusterException(String.format("DNS entry '%s' resolved into empty list.", this.settings.dns));
            }
            return Arrays.asList(addresses);
        }
        catch (Exception e) {
            throw new ClusterException(String.format("Error while resolving DNS entry '%s'.", this.settings.dns), e);
        }
    }

    private List<GossipSeed> getGossipCandidatesFromOldGossip(List<MemberInfoDto> oldGossip, InetSocketAddress failedTcpEndpoint) {
        List<MemberInfoDto> gossipCandidates = failedTcpEndpoint == null ? oldGossip : oldGossip.stream().filter(m -> {
            try {
                return m.externalTcpPort != failedTcpEndpoint.getPort() || !InetAddress.getByName(m.externalTcpIp).equals(failedTcpEndpoint.getAddress());
            }
            catch (UnknownHostException e) {
                throw Throwables.propagate(e);
            }
        }).collect(Collectors.toList());
        return this.arrangeGossipCandidates(gossipCandidates);
    }

    private List<GossipSeed> arrangeGossipCandidates(List<MemberInfoDto> members) {
        ArrayList managers = new ArrayList();
        ArrayList nodes = new ArrayList();
        members.forEach(m -> {
            InetSocketAddress address = new InetSocketAddress(m.externalHttpIp, m.externalHttpPort);
            if (m.state == VNodeState.Manager) {
                managers.add(new GossipSeed(address));
            } else {
                nodes.add(new GossipSeed(address));
            }
        });
        Collections.shuffle(managers);
        Collections.shuffle(nodes);
        ArrayList<GossipSeed> result = new ArrayList<GossipSeed>();
        result.addAll(nodes);
        result.addAll(managers);
        return result;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Optional<ClusterInfoDto> tryGetGossipFrom(GossipSeed gossipSeed) {
        try {
            URL url = new URL("http://" + gossipSeed.endpoint.getHostString() + ":" + gossipSeed.endpoint.getPort() + "/gossip?format=json");
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setConnectTimeout((int)this.settings.gossipTimeout.toMillis());
            connection.setReadTimeout((int)this.settings.gossipTimeout.toMillis());
            connection.setRequestMethod("GET");
            connection.setRequestProperty("Accept", "application/json");
            if (connection.getResponseCode() != 200) return Optional.empty();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), "UTF-8"));){
                Optional<Object> optional = Optional.of(this.gson.fromJson(new JsonReader((Reader)reader), ClusterInfoDto.class));
                return optional;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return Optional.empty();
    }

    private Optional<NodeEndpoints> tryDetermineBestNode(List<MemberInfoDto> members) {
        Predicate<VNodeState> matchesNotAllowedStates = s -> s == VNodeState.Manager || s == VNodeState.ShuttingDown || s == VNodeState.Shutdown;
        List aliveMembers = members.stream().filter(m -> m.isAlive && !matchesNotAllowedStates.test(m.state)).sorted((a, b) -> a.state.ordinal() > b.state.ordinal() ? -1 : 1).collect(Collectors.toList());
        switch (this.settings.nodePreference) {
            case Random: {
                Collections.shuffle(aliveMembers);
                break;
            }
            case Slave: {
                aliveMembers = aliveMembers.stream().sorted((a, b) -> a.state == VNodeState.Slave ? -1 : 1).collect(Collectors.toList());
                Collections.shuffle(aliveMembers.subList(0, (int)aliveMembers.stream().filter(m -> m.state == VNodeState.Slave).count()));
            }
        }
        return aliveMembers.stream().findFirst().map(n -> {
            InetSocketAddress tcp = new InetSocketAddress(n.externalTcpIp, n.externalTcpPort);
            InetSocketAddress secureTcp = n.externalSecureTcpPort > 0 ? new InetSocketAddress(n.externalTcpIp, n.externalSecureTcpPort) : null;
            logger.info("Discovering: found best choice [{},{}] ({}).", new Object[]{tcp, secureTcp == null ? "n/a" : secureTcp.toString(), n.state});
            return new NodeEndpoints(tcp, secureTcp);
        });
    }
}

