/*
 * Decompiled with CFR 0.152.
 */
package com.github.msemys.esjc.operation.manager;

import com.github.msemys.esjc.ConnectionClosedException;
import com.github.msemys.esjc.Settings;
import com.github.msemys.esjc.operation.manager.OperationItem;
import com.github.msemys.esjc.operation.manager.OperationTimeoutException;
import com.github.msemys.esjc.operation.manager.RetriesLimitReachedException;
import com.github.msemys.esjc.tcp.TcpPackage;
import com.github.msemys.esjc.util.Iterables;
import com.github.msemys.esjc.util.Preconditions;
import io.netty.channel.Channel;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OperationManager {
    private static final Logger logger = LoggerFactory.getLogger(OperationManager.class);
    private final Map<UUID, OperationItem> activeOperations = new ConcurrentHashMap<UUID, OperationItem>();
    private final Queue<OperationItem> waitingOperations = new ConcurrentLinkedQueue<OperationItem>();
    private final Queue<OperationItem> retryPendingOperations = new ConcurrentLinkedQueue<OperationItem>();
    private volatile int totalOperationCount;
    private final Settings settings;

    public OperationManager(Settings settings) {
        this.settings = settings;
    }

    public Optional<OperationItem> getActiveOperation(UUID correlationId) {
        return Optional.ofNullable(this.activeOperations.get(correlationId));
    }

    public int totalOperationCount() {
        return this.totalOperationCount;
    }

    private void updateTotalOperationCount() {
        this.totalOperationCount = this.activeOperations.size() + this.waitingOperations.size();
    }

    public void cleanUp(Throwable cause) {
        ConnectionClosedException exception = new ConnectionClosedException("Connection was closed.", cause);
        Consumer<OperationItem> failOperation = item -> item.operation.fail(exception);
        Iterables.consume(this.activeOperations.values(), failOperation);
        Iterables.consume(this.waitingOperations, failOperation);
        Iterables.consume(this.retryPendingOperations, failOperation);
        this.updateTotalOperationCount();
    }

    public void checkTimeoutsAndRetry(Channel connection) {
        ArrayList retryOperations = new ArrayList();
        ArrayList removeOperations = new ArrayList();
        this.activeOperations.values().forEach(item -> {
            if (connection != null && !item.connectionId.equals(connection.id())) {
                retryOperations.add(item);
            } else if (!item.timeout.isZero() && item.lastUpdated.isElapsed(this.settings.operationTimeout)) {
                String error = String.format("Operation never got response from server. UTC now: %s, operation: %s.", Instant.now(), item.toString());
                logger.debug(error);
                if (this.settings.failOnNoServerResponse) {
                    item.operation.fail(new OperationTimeoutException(error));
                    removeOperations.add(item);
                } else {
                    retryOperations.add(item);
                }
            }
        });
        removeOperations.forEach(this::removeOperation);
        if (connection != null) {
            Iterables.consume(retryOperations, this::scheduleOperationRetry);
            Iterables.consume(this.retryPendingOperations, retryOperations::add);
            retryOperations.stream().sorted().forEach(item -> {
                UUID oldCorrelationId = item.correlationId;
                item.correlationId = UUID.randomUUID();
                ++item.retryCount;
                logger.debug("retrying, old correlationId {}, operation {}.", (Object)oldCorrelationId, (Object)item.toString());
                this.scheduleOperation((OperationItem)item, connection);
            });
            this.scheduleWaitingOperations(connection);
        }
    }

    public void scheduleOperationRetry(OperationItem item) {
        if (this.removeOperation(item)) {
            logger.debug("scheduleOperationRetry for {}", (Object)item);
            if (item.maxRetries >= 0 && item.retryCount >= item.maxRetries) {
                item.operation.fail(new RetriesLimitReachedException(item.toString(), item.retryCount));
            } else {
                this.retryPendingOperations.offer(item);
            }
        }
    }

    public boolean removeOperation(OperationItem item) {
        if (this.activeOperations.remove(item.correlationId) == null) {
            logger.debug("removeOperation FAILED for {}", (Object)item);
            return false;
        }
        logger.debug("removeOperation SUCCEEDED for {}", (Object)item);
        this.updateTotalOperationCount();
        return true;
    }

    public void scheduleWaitingOperations(Channel connection) {
        OperationItem item;
        Preconditions.checkNotNull(connection, "connection is null");
        while (this.activeOperations.size() < this.settings.maxConcurrentOperations && (item = this.waitingOperations.poll()) != null) {
            this.send(item, connection);
        }
        this.updateTotalOperationCount();
    }

    public void enqueueOperation(OperationItem item) {
        logger.debug("enqueueOperation WAITING for {}.", (Object)item);
        this.waitingOperations.offer(item);
    }

    public void scheduleOperation(OperationItem item, Channel connection) {
        Preconditions.checkNotNull(connection, "connection is null");
        logger.debug("scheduleOperation WAITING for {}.", (Object)item);
        this.waitingOperations.offer(item);
        this.scheduleWaitingOperations(connection);
    }

    private void send(OperationItem item, Channel connection) {
        item.connectionId = connection.id();
        item.lastUpdated.update();
        this.activeOperations.put(item.correlationId, item);
        TcpPackage tcpPackage = item.operation.create(item.correlationId);
        logger.debug("send package {}, {}, {}.", new Object[]{tcpPackage.command, tcpPackage.correlationId, item});
        connection.writeAndFlush((Object)tcpPackage);
    }
}

