package de.oehme.xtend.contrib;

import com.google.common.annotations.Beta;
import com.google.common.base.Objects;
import com.google.common.collect.Maps;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.xtend.lib.annotations.FinalFieldsConstructor;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtend.lib.macro.declaration.AnnotationReference;
import org.eclipse.xtend.lib.macro.declaration.MethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableMethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableParameterDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableTypeDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableTypeParameterDeclaration;
import org.eclipse.xtend.lib.macro.declaration.ParameterDeclaration;
import org.eclipse.xtend.lib.macro.declaration.ResolvedMethod;
import org.eclipse.xtend.lib.macro.declaration.ResolvedParameter;
import org.eclipse.xtend.lib.macro.declaration.ResolvedTypeParameter;
import org.eclipse.xtend.lib.macro.declaration.Type;
import org.eclipse.xtend.lib.macro.declaration.TypeParameterDeclaration;
import org.eclipse.xtend.lib.macro.declaration.TypeReference;
import org.eclipse.xtend.lib.macro.declaration.Visibility;
import org.eclipse.xtend.lib.macro.expression.Expression;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Helps you with copying signatures of existing methods.
 * 
 * You should use this as an extension for maximum convenience, e.g.:
 * 
 * <pre>
 * override doTransform(MutableClassDeclaration cls, extension TransformationContext context) {
 * 	val extension transformations = new CommonTransformations(context)
 * 	...
 * }
 * </pre>
 */
@Beta
@FinalFieldsConstructor
@SuppressWarnings("all")
public class SignatureHelper {
  @Extension
  private final TransformationContext delegate;
  
  /**
   * Copies the signature of the given base method so that you only have to add a body in most cases.
   * You are free to modify the default settings, of course, e.g. widening the visibility of the
   * implementing method.
   * @return the new implementation method
   */
  public MutableMethodDeclaration addImplementationFor(final MutableClassDeclaration cls, final ResolvedMethod baseMethod, final StringConcatenationClient implementation) {
    MutableMethodDeclaration _createImplementation = this.createImplementation(cls, baseMethod);
    final Procedure1<MutableMethodDeclaration> _function = new Procedure1<MutableMethodDeclaration>() {
      @Override
      public void apply(final MutableMethodDeclaration it) {
        it.setBody(implementation);
      }
    };
    return ObjectExtensions.<MutableMethodDeclaration>operator_doubleArrow(_createImplementation, _function);
  }
  
  private MutableMethodDeclaration createImplementation(final MutableClassDeclaration cls, final ResolvedMethod baseMethod) {
    MethodDeclaration _declaration = baseMethod.getDeclaration();
    String _simpleName = _declaration.getSimpleName();
    final Procedure1<MutableMethodDeclaration> _function = new Procedure1<MutableMethodDeclaration>() {
      @Override
      public void apply(final MutableMethodDeclaration it) {
        SignatureHelper.this.copySignatureFrom(it, baseMethod);
        it.setAbstract(false);
        SignatureHelper.this.delegate.getPrimarySourceElement(it);
        baseMethod.getDeclaration();
      }
    };
    return cls.addMethod(_simpleName, _function);
  }
  
  /**
   * Moves the body of this method to a new private method with the given name.
   * The original method then gets the newly specified body which can delegate to the inner method.
   * @return the new inner method.
   */
  public MutableMethodDeclaration addIndirection(final MutableMethodDeclaration wrapper, final String innerMethodName, final StringConcatenationClient indirection) {
    MutableMethodDeclaration _xblockexpression = null;
    {
      final MutableMethodDeclaration inner = this.createInnerMethod(wrapper, innerMethodName);
      wrapper.setBody(indirection);
      _xblockexpression = inner;
    }
    return _xblockexpression;
  }
  
  private MutableMethodDeclaration createInnerMethod(final MutableMethodDeclaration wrapper, final String innerMethodName) {
    MutableTypeDeclaration _declaringType = wrapper.getDeclaringType();
    final Procedure1<MutableMethodDeclaration> _function = new Procedure1<MutableMethodDeclaration>() {
      @Override
      public void apply(final MutableMethodDeclaration it) {
        MutableTypeDeclaration _declaringType = wrapper.getDeclaringType();
        TypeReference _newSelfTypeReference = SignatureHelper.this.delegate.newSelfTypeReference(_declaringType);
        Iterable<? extends ResolvedMethod> _declaredResolvedMethods = _newSelfTypeReference.getDeclaredResolvedMethods();
        final Function1<ResolvedMethod, Boolean> _function = new Function1<ResolvedMethod, Boolean>() {
          @Override
          public Boolean apply(final ResolvedMethod it) {
            MethodDeclaration _declaration = it.getDeclaration();
            return Boolean.valueOf(Objects.equal(_declaration, wrapper));
          }
        };
        final ResolvedMethod resolvedMethod = IterableExtensions.findFirst(_declaredResolvedMethods, _function);
        SignatureHelper.this.copySignatureFrom(it, resolvedMethod);
        it.setVisibility(Visibility.PRIVATE);
        Expression _body = wrapper.getBody();
        it.setBody(_body);
        SignatureHelper.this.delegate.setPrimarySourceElement(it, wrapper);
      }
    };
    return _declaringType.addMethod(innerMethodName, _function);
  }
  
  /**
   * Copies the fully resolved signature from the source to this method, including type parameter resolution
   */
  public void copySignatureFrom(final MutableMethodDeclaration it, final ResolvedMethod source) {
    this.copySignatureFrom(it, source, Collections.<TypeReference, TypeReference>unmodifiableMap(CollectionLiterals.<TypeReference, TypeReference>newHashMap()));
  }
  
  /**
   * Copies the fully resolved signature from the source to this method, including type parameter resolution.
   * The class-level type parameters assign each type parameter in the target method to a type parameter in the source method.
   */
  public void copySignatureFrom(final MutableMethodDeclaration it, final ResolvedMethod source, final Map<TypeReference, TypeReference> classTypeParameterMappings) {
    MethodDeclaration _declaration = source.getDeclaration();
    boolean _isAbstract = _declaration.isAbstract();
    it.setAbstract(_isAbstract);
    MethodDeclaration _declaration_1 = source.getDeclaration();
    boolean _isDeprecated = _declaration_1.isDeprecated();
    it.setDeprecated(_isDeprecated);
    MethodDeclaration _declaration_2 = source.getDeclaration();
    boolean _isDefault = _declaration_2.isDefault();
    it.setDefault(_isDefault);
    MethodDeclaration _declaration_3 = source.getDeclaration();
    String _docComment = _declaration_3.getDocComment();
    it.setDocComment(_docComment);
    MethodDeclaration _declaration_4 = source.getDeclaration();
    boolean _isFinal = _declaration_4.isFinal();
    it.setFinal(_isFinal);
    MethodDeclaration _declaration_5 = source.getDeclaration();
    boolean _isNative = _declaration_5.isNative();
    it.setNative(_isNative);
    MethodDeclaration _declaration_6 = source.getDeclaration();
    boolean _isStatic = _declaration_6.isStatic();
    it.setStatic(_isStatic);
    MethodDeclaration _declaration_7 = source.getDeclaration();
    boolean _isStrictFloatingPoint = _declaration_7.isStrictFloatingPoint();
    it.setStrictFloatingPoint(_isStrictFloatingPoint);
    MethodDeclaration _declaration_8 = source.getDeclaration();
    boolean _isSynchronized = _declaration_8.isSynchronized();
    it.setSynchronized(_isSynchronized);
    MethodDeclaration _declaration_9 = source.getDeclaration();
    boolean _isVarArgs = _declaration_9.isVarArgs();
    it.setVarArgs(_isVarArgs);
    MethodDeclaration _declaration_10 = source.getDeclaration();
    Visibility _visibility = _declaration_10.getVisibility();
    it.setVisibility(_visibility);
    final HashMap<TypeReference, TypeReference> typeParameterMappings = Maps.<TypeReference, TypeReference>newHashMap(classTypeParameterMappings);
    Iterable<? extends ResolvedTypeParameter> _resolvedTypeParameters = source.getResolvedTypeParameters();
    final Procedure1<ResolvedTypeParameter> _function = new Procedure1<ResolvedTypeParameter>() {
      @Override
      public void apply(final ResolvedTypeParameter param) {
        TypeParameterDeclaration _declaration = param.getDeclaration();
        String _simpleName = _declaration.getSimpleName();
        Iterable<? extends TypeReference> _resolvedUpperBounds = param.getResolvedUpperBounds();
        final MutableTypeParameterDeclaration copy = it.addTypeParameter(_simpleName, ((TypeReference[])Conversions.unwrapArray(_resolvedUpperBounds, TypeReference.class)));
        TypeParameterDeclaration _declaration_1 = param.getDeclaration();
        TypeReference _newTypeReference = SignatureHelper.this.delegate.newTypeReference(_declaration_1);
        TypeReference _newTypeReference_1 = SignatureHelper.this.delegate.newTypeReference(copy);
        typeParameterMappings.put(_newTypeReference, _newTypeReference_1);
        Iterable<? extends TypeReference> _upperBounds = copy.getUpperBounds();
        final Function1<TypeReference, TypeReference> _function = new Function1<TypeReference, TypeReference>() {
          @Override
          public TypeReference apply(final TypeReference it) {
            return SignatureHelper.this.replace(it, typeParameterMappings);
          }
        };
        Iterable<TypeReference> _map = IterableExtensions.map(_upperBounds, _function);
        copy.setUpperBounds(_map);
      }
    };
    IterableExtensions.forEach(_resolvedTypeParameters, _function);
    Iterable<? extends TypeReference> _resolvedExceptionTypes = source.getResolvedExceptionTypes();
    final Function1<TypeReference, TypeReference> _function_1 = new Function1<TypeReference, TypeReference>() {
      @Override
      public TypeReference apply(final TypeReference it) {
        return SignatureHelper.this.replace(it, typeParameterMappings);
      }
    };
    Iterable<TypeReference> _map = IterableExtensions.map(_resolvedExceptionTypes, _function_1);
    it.setExceptions(((TypeReference[])Conversions.unwrapArray(_map, TypeReference.class)));
    TypeReference _resolvedReturnType = source.getResolvedReturnType();
    TypeReference _replace = this.replace(_resolvedReturnType, typeParameterMappings);
    it.setReturnType(_replace);
    Iterable<? extends ResolvedParameter> _resolvedParameters = source.getResolvedParameters();
    final Procedure1<ResolvedParameter> _function_2 = new Procedure1<ResolvedParameter>() {
      @Override
      public void apply(final ResolvedParameter p) {
        ParameterDeclaration _declaration = p.getDeclaration();
        String _simpleName = _declaration.getSimpleName();
        TypeReference _resolvedType = p.getResolvedType();
        TypeReference _replace = SignatureHelper.this.replace(_resolvedType, typeParameterMappings);
        final MutableParameterDeclaration addedParam = it.addParameter(_simpleName, _replace);
        ParameterDeclaration _declaration_1 = p.getDeclaration();
        Iterable<? extends AnnotationReference> _annotations = _declaration_1.getAnnotations();
        final Procedure1<AnnotationReference> _function = new Procedure1<AnnotationReference>() {
          @Override
          public void apply(final AnnotationReference it) {
            addedParam.addAnnotation(it);
          }
        };
        IterableExtensions.forEach(_annotations, _function);
      }
    };
    IterableExtensions.forEach(_resolvedParameters, _function_2);
  }
  
  private TypeReference replace(final TypeReference target, final Map<? extends TypeReference, ? extends TypeReference> mappings) {
    Set<? extends Map.Entry<? extends TypeReference, ? extends TypeReference>> _entrySet = mappings.entrySet();
    final Function2<TypeReference, Map.Entry<? extends TypeReference, ? extends TypeReference>, TypeReference> _function = new Function2<TypeReference, Map.Entry<? extends TypeReference, ? extends TypeReference>, TypeReference>() {
      @Override
      public TypeReference apply(final TypeReference result, final Map.Entry<? extends TypeReference, ? extends TypeReference> mapping) {
        TypeReference _key = mapping.getKey();
        TypeReference _value = mapping.getValue();
        return SignatureHelper.this.replace(result, _key, _value);
      }
    };
    return IterableExtensions.fold(_entrySet, target, _function);
  }
  
  private TypeReference replace(final TypeReference target, final TypeReference oldType, final TypeReference newType) {
    boolean _equals = Objects.equal(target, oldType);
    if (_equals) {
      return newType;
    }
    List<TypeReference> _actualTypeArguments = target.getActualTypeArguments();
    boolean _isEmpty = _actualTypeArguments.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      Type _type = target.getType();
      List<TypeReference> _actualTypeArguments_1 = target.getActualTypeArguments();
      final Function1<TypeReference, TypeReference> _function = new Function1<TypeReference, TypeReference>() {
        @Override
        public TypeReference apply(final TypeReference it) {
          return SignatureHelper.this.replace(it, oldType, newType);
        }
      };
      List<TypeReference> _map = ListExtensions.<TypeReference, TypeReference>map(_actualTypeArguments_1, _function);
      return this.delegate.newTypeReference(_type, ((TypeReference[])Conversions.unwrapArray(_map, TypeReference.class)));
    }
    boolean _isWildCard = target.isWildCard();
    if (_isWildCard) {
      TypeReference _upperBound = target.getUpperBound();
      TypeReference _object = this.delegate.getObject();
      boolean _notEquals = (!Objects.equal(_upperBound, _object));
      if (_notEquals) {
        TypeReference _upperBound_1 = target.getUpperBound();
        TypeReference _replace = this.replace(_upperBound_1, oldType, newType);
        return this.delegate.newWildcardTypeReference(_replace);
      } else {
        TypeReference _lowerBound = target.getLowerBound();
        boolean _isAnyType = _lowerBound.isAnyType();
        boolean _not_1 = (!_isAnyType);
        if (_not_1) {
          TypeReference _lowerBound_1 = target.getLowerBound();
          TypeReference _replace_1 = this.replace(_lowerBound_1, oldType, newType);
          return this.delegate.newWildcardTypeReferenceWithLowerBound(_replace_1);
        }
      }
    }
    boolean _isArray = target.isArray();
    if (_isArray) {
      TypeReference _arrayComponentType = target.getArrayComponentType();
      TypeReference _replace_2 = this.replace(_arrayComponentType, oldType, newType);
      return this.delegate.newArrayTypeReference(_replace_2);
    }
    return target;
  }
  
  public SignatureHelper(final TransformationContext delegate) {
    super();
    this.delegate = delegate;
  }
}
