/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.lcs.activation;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

import java.util.HashMap;
import java.util.Map;

/**
 * @author Igor Beslic
 */
public class LCSClusterEntryTokenContentAdvisor {

	public LCSClusterEntryTokenContentAdvisor(String content)
		throws IOException {

		if (isNotValid(content)) {
			throw new UnsupportedOperationException(
				"Automatic activation token content must not be empty");
		}

		String[] array = content.split("--");

		if (array.length == 1) {
			_contentStructureVersion = 3;

			processContent(content);
		}
		else if (array.length == 2) {
			_accessSecret = array[1];
			_accessToken = array[0];

			_contentStructureVersion = 1;
		}
		else if (array.length == 3) {
			_accessSecret = array[1];
			_accessToken = array[0];

			_contentStructureVersion = 2;

			processContent(array[2]);
		}
		else {
			throw new UnsupportedEncodingException(
				"Automatic activation token content structure is not " +
					"supported");
		}
	}

	public LCSClusterEntryTokenContentAdvisor(
		String accessSecret, String accessToken,
		Map<String, String> lcsServicesConfiguration,
		String portalPropertiesBlacklist) {

		if (isNotValid(accessSecret) || isNotValid(accessToken)) {
			throw new UnsupportedOperationException(
				"OAuth access secret and token must not be empty");
		}

		_accessSecret = accessSecret;
		_accessToken = accessToken;
		_portalPropertiesBlacklist = portalPropertiesBlacklist;

		_lcsServicesConfiguration.putAll(lcsServicesConfiguration);

		_contentStructureVersion = 3;
	}

	public LCSClusterEntryTokenContentAdvisor(
		String accessSecret, String accessToken, String consumerKey,
		String consumerSecret, String dataCenterHostName,
		String dataCenterHostPort, String dataCenterProtocol,
		Map<String, String> lcsServicesConfiguration,
		String portalPropertiesBlacklist) {

		this(
			accessSecret, accessToken, lcsServicesConfiguration,
			portalPropertiesBlacklist);

		if (isNotValid(consumerKey) || isNotValid(consumerSecret)) {
			throw new UnsupportedOperationException(
				"OAuth consumer key and secret must not be empty");
		}

		_consumerKey = consumerKey;
		_consumerSecret = consumerSecret;
		_dataCenterHostName = dataCenterHostName;
		_dataCenterHostPort = dataCenterHostPort;
		_dataCenterProtocol = dataCenterProtocol;
	}

	public String getAccessSecret() {
		return _accessSecret;
	}

	public String getAccessToken() {
		return _accessToken;
	}

	public String getConsumerKey() {
		return _consumerKey;
	}

	public String getConsumerSecret() {
		return _consumerSecret;
	}

	public String getContentJSONString() {
		ObjectMapper objectMapper = new ObjectMapper();

		ObjectNode rootNode = objectMapper.createObjectNode();

		rootNode.put("accessSecret", _accessSecret);
		rootNode.put("accessToken", _accessToken);

		if (_consumerKey != null) {
			rootNode.put("consumerKey", _consumerKey);
		}

		if (_consumerSecret != null) {
			rootNode.put("consumerSecret", _consumerSecret);
		}

		if (_contentStructureVersion < 3) {
			throw new UnsupportedOperationException(
				"Unsupported content structure version " +
					_contentStructureVersion);
		}

		rootNode.put("contentStructureVersion", _contentStructureVersion);

		if (_dataCenterHostName != null) {
			rootNode.put("dataCenterHostName", _dataCenterHostName);
		}

		if (_dataCenterHostPort != null) {
			rootNode.put("dataCenterHostPort", _dataCenterHostPort);
		}

		if (_dataCenterProtocol != null) {
			rootNode.put("dataCenterProtocol", _dataCenterProtocol);
		}

		ArrayNode arrayNode = rootNode.putArray("lcsServicesConfiguration");

		for (Map.Entry<String, String> entry :
				_lcsServicesConfiguration.entrySet()) {

			if (Boolean.parseBoolean(entry.getValue())) {
				arrayNode.add(entry.getKey());
			}
		}

		if (_portalPropertiesBlacklist != null) {
			rootNode.put(
				"portalPropertiesBlacklist", _portalPropertiesBlacklist);
		}

		return rootNode.toString();
	}

	public int getContentStructureVersion() {
		return _contentStructureVersion;
	}

	public String getDataCenterHostName() {
		return _dataCenterHostName;
	}

	public String getDataCenterHostPort() {
		return _dataCenterHostPort;
	}

	public String getDataCenterProtocol() {
		return _dataCenterProtocol;
	}

	public Map<String, String> getLCSServicesConfiguration() {
		return _lcsServicesConfiguration;
	}

	public String getPortalPropertiesBlacklist() {
		return _portalPropertiesBlacklist;
	}

	protected boolean isNotValid(String value) {
		if (value == null) {
			return true;
		}

		value = value.trim();

		if (value.length() == 0) {
			return true;
		}

		return false;
	}

	protected void processContent(String contentJSONString) throws IOException {
		ObjectMapper objectMapper = new ObjectMapper();

		JsonNode jsonTreeNode = objectMapper.readTree(contentJSONString);

		JsonNode jsonNode = jsonTreeNode.get("accessSecret");

		_accessSecret = jsonNode.asText();

		jsonNode = jsonTreeNode.get("accessToken");

		_accessToken = jsonNode.asText();

		jsonNode = jsonTreeNode.get("consumerKey");

		if (jsonNode != null) {
			_consumerKey = jsonNode.asText();
		}

		jsonNode = jsonTreeNode.get("consumerSecret");

		if (jsonNode != null) {
			_consumerSecret = jsonNode.asText();
		}

		jsonNode = jsonTreeNode.get("contentStructureVersion");

		if (jsonNode != null) {
			_contentStructureVersion = jsonNode.asInt();
		}

		jsonNode = jsonTreeNode.get("dataCenterHostName");

		if (jsonNode != null) {
			_dataCenterHostName = jsonNode.asText();
		}

		jsonNode = jsonTreeNode.get("dataCenterHostPort");

		if (jsonNode != null) {
			_dataCenterHostPort = jsonNode.asText();
		}

		jsonNode = jsonTreeNode.get("dataCenterProtocol");

		if (jsonNode != null) {
			_dataCenterProtocol = jsonNode.asText();
		}

		jsonNode = jsonTreeNode.get("lcsServicesConfiguration");

		if (jsonNode.isArray()) {
			for (JsonNode jsonArrayElement : jsonNode) {
				_lcsServicesConfiguration.put(
					jsonArrayElement.asText(), "true");
			}
		}
		else {
			_lcsServicesConfiguration.put(jsonNode.asText(), "true");
		}

		jsonNode = jsonTreeNode.get("portalPropertiesBlacklist");

		if (jsonNode != null) {
			_portalPropertiesBlacklist = jsonNode.asText();
		}
	}

	private String _accessSecret;
	private String _accessToken;
	private String _consumerKey;
	private String _consumerSecret;
	private int _contentStructureVersion;
	private String _dataCenterHostName;
	private String _dataCenterHostPort;
	private String _dataCenterProtocol;
	private final Map<String, String> _lcsServicesConfiguration =
		new HashMap<String, String>();
	private String _portalPropertiesBlacklist;

}