package com.linkare.jboolexpr;

import java.util.ArrayList;
import java.util.List;

import com.linkare.commons.utils.StringUtils;

/**
 * If the Boolean String Expression is malformed.
 * 
 * @author Adolfo Sanz De Diego
 */
public final class MalformedBooleanException extends Exception {

    /**
     * serialVersionUID
     */
    private static final long serialVersionUID = 1L;

    /**
     * The error indexes.
     */
    private List<Integer> booleanExpressionErrorIndexes;

    /**
     * The Boolean String Expression.
     */
    private String booleanExpression;

    /**
     * The error message.
     */
    private String booleanExpressionErrorMessage;

    /**
     * Constructor.
     * 
     * @param errorMessage
     *            The error message.
     * @param errorIndex
     *            The index with error.
     * @param newBooleanExpression
     *            The boolean expression.
     */
    MalformedBooleanException(final String errorMessage, final int errorIndex, final String newBooleanExpression) {
	this(errorMessage, toList(errorIndex), newBooleanExpression);
    }

    /**
     * Constructor.
     * 
     * @param errorMessage
     *            The error message.
     * @param errorIndexes
     *            The {@link List} with the indexes with error.
     * @param newBooleanExpression
     *            The boolean expression.
     */
    MalformedBooleanException(final String errorMessage, final List<Integer> errorIndexes, final String newBooleanExpression) {
	super(format(errorMessage, errorIndexes, newBooleanExpression));
	this.booleanExpression = newBooleanExpression;
	this.booleanExpressionErrorIndexes = errorIndexes;
	this.booleanExpressionErrorMessage = errorMessage;
    }

    /**
     * Returns the Boolean String Expression.
     * 
     * @return The Boolean String Expression.
     */
    public String getBooleanExpression() {
	return this.booleanExpression;
    }

    /**
     * Returns the error indexes.
     * 
     * @return The error indexes.
     */
    public List<Integer> getBooleanExpressionErrorIndexes() {
	return this.booleanExpressionErrorIndexes;
    }

    /**
     * Returns the error message.
     * 
     * @return The error message.
     */
    public String getBooleanExpressionErrorMessage() {
	return this.booleanExpressionErrorMessage;
    }

    /**
     * Returns a new {@link List} with the supplied error index.
     * 
     * @param errorIndex
     *            The index with error.
     * @return A new {@link List} with the supplied error index.
     */
    private static List<Integer> toList(int errorIndex) {
	List<Integer> errorIndexes = new ArrayList<Integer>();
	errorIndexes.add(errorIndex);
	return errorIndexes;
    }

    /**
     * Returns the exception message formated.
     * 
     * @param errorMessage
     *            The error message.
     * @param errorIndexes
     *            The {@link List} with the indexes with error.
     * @param newBooleanExpression
     *            The boolean expression.
     * @return The error message formated.
     */
    private static String format(final String errorMessage, final List<Integer> errorIndexes, final String newBooleanExpression) {
	validateFormatArgs(errorMessage, errorIndexes, newBooleanExpression);
	StringBuilder error = buildErrorMessage(errorMessage, errorIndexes, newBooleanExpression);
	return error.toString();
    }

    private static StringBuilder buildErrorMessage(final String errorMessage, final List<Integer> errorIndexes, final String newBooleanExpression) {
	StringBuilder error = new StringBuilder(errorMessage);
	error.append(" in [ ");
	int size = errorIndexes.size();
	int lastIndex = 0;
	for (int i = 0; i < size; i++) {
	    int index = ((Integer) errorIndexes.get(i)).intValue();
	    error.append(newBooleanExpression.substring(lastIndex, index));
	    error.append("_");
	    lastIndex = index;
	}
	error.append(newBooleanExpression.substring(lastIndex, newBooleanExpression.length()));
	error.append(" ]");
	if (size == 1) {
	    error.append(" - Index [");
	} else if (size > 1) {
	    error.append(" - Indexes [");
	}
	for (int i = 0; i < size; i++) {
	    error.append(errorIndexes.get(i));
	    if (i != (size - 1)) {
		error.append(", ");
	    }
	}
	if (size > 0) {
	    error.append("]");
	}
	return error;
    }

    private static void validateFormatArgs(final String errorMessage, final List<Integer> errorIndexes, final String newBooleanExpression) {
	if (StringUtils.isBlank(errorMessage)) {
	    throw new IllegalArgumentException("errorMessage is null or void");
	}
	if (errorIndexes == null || errorIndexes.isEmpty()) {
	    throw new IllegalArgumentException("errorIndexes is null or void");
	}
	if (StringUtils.isBlank(newBooleanExpression)) {
	    throw new IllegalArgumentException("newBooleanExpression is null or void");
	}
    }
}