package com.linkare.commons.dao.jpa;

import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import com.linkare.commons.dao.Deletable;
import com.linkare.commons.dao.IGenericDAO;
import com.linkare.commons.dao.Identifiable;
import com.linkare.commons.jpa.exceptions.DomainException;
import com.linkare.commons.utils.BooleanResult;

/**
 * This abstract class provides a set of common methods that should be implemented for any JPA entity.
 * 
 * @author Paulo Zenida - Linkare TI
 * 
 * @param <T>
 *            The entity that instances of this DAO is able to process. All classes that implements the interface <code>Identifiable</code> are able to be
 *            managed by subclasses of this generic DAO. Since the base <code>DomainObject</code> implements this interface, all entity object types extending
 *            from it will be able to benefit from this DAO's interface.
 */
public abstract class GenericDAO<T extends Identifiable<ID> & Deletable, ID extends Serializable> implements IGenericDAO<T, ID> {

    private static final String FIND_ALL_QUERY = ".findAll";

    private static final String COUNT_ALL_QUERY = ".countAll";

    private EntityManager entityManager;

    private Class<T> entityTypeClass;

    public GenericDAO(final EntityManager entityManager) {
	this.entityManager = entityManager;
    }

    /**
     * @return the entityManager
     */
    public final EntityManager getEntityManager() {
	return entityManager;
    }

    /**
     * @param entityManager
     *            the entityManager to set
     */
    public final void setEntityManager(final EntityManager entityManager) {
	this.entityManager = entityManager;
    }

    /**
     * 
     * @param t
     *            the entity to create
     * @throws DomainException
     *             with the message from the result of the invocation to <code>canCreate()</code>, if the entity <code>t</code> cannot be created
     * 
     * @see AbstractDAO#canCreate(Identifiable)
     */
    @Override
    public void create(final T t) {
	final BooleanResult operationResult = canCreate(t);
	if (Boolean.TRUE == operationResult.getResult()) {
	    getEntityManager().persist(t);
	} else {
	    throw new DomainException(operationResult.getMessage());
	}
    }

    /**
     * @param t
     *            the entity we are checking for creation
     * @return true, by default. Subclasses are supposed to override this method whenever there is a logic to check whether <code>t</code> can be created or
     *         not.
     */
    protected BooleanResult canCreate(final T t) {
	return BooleanResult.TRUE_RESULT;
    }

    /**
     * 
     * @param t
     *            the entity to update/edit
     * @return Returns the new entity instance resulting from the merge operation
     * @throws DomainException
     *             with the message from the result of the invocation to <code>canEdit()</code>, if the entity <code>t</code> cannot be edited
     * 
     * @see AbstractDAO#canEdit(Identifiable)
     */
    @Override
    public T edit(final T t) {
	final BooleanResult operationResult = canEdit(t);
	if (Boolean.TRUE == operationResult.getResult()) {
	    return getEntityManager().merge(t);
	} else {
	    throw new DomainException(operationResult.getMessage());
	}
    }

    /**
     * @param t
     *            the entity we are checking for edition
     * @return true, by default. Subclasses are supposed to override this method whenever there is a logic to check whether <code>t</code> can be edited or not.
     */
    protected BooleanResult canEdit(final T t) {
	return BooleanResult.TRUE_RESULT;
    }

    /**
     * 
     * @param t
     *            the entity to remove
     * @throws DomainException
     *             with the message from the result of the invocation to <code>canRemove()</code>, if the entity <code>t</code> cannot be removed
     * 
     * @see AbstractDAO#canRemove(Identifiable)
     */
    @Override
    public void remove(final T t) {
	final BooleanResult operationResult = canRemove(t);
	if (Boolean.TRUE == operationResult.getResult()) {
	    final T mergedEntity = getEntityManager().merge(t);
	    if (mergedEntity.delete()) {
		getEntityManager().remove(mergedEntity);
	    }
	} else {
	    throw new DomainException(operationResult.getMessage());
	}
    }

    /**
     * @param t
     *            the entity we are checking for removing
     * @return true, by default. Subclasses are supposed to override this method whenever there is a logic to check whether <code>t</code> can be removed or
     *         not.
     */
    protected BooleanResult canRemove(final T t) {
	return BooleanResult.TRUE_RESULT;
    }

    @Override
    public T find(final ID id) {
	try {
	    return getEntityManager().find(getEntityTypeClass(), id);
	} catch (NoResultException e) {
	    return null;
	}
    }

    /**
     * 
     * @return Returns the named query, assuming that a query returning all entity objects is named <Entity>.findAll. This method was not made final since it is
     *         possible that, in certain situations, the findAll query is named differently.
     */
    public Query findAllQuery() {
	return getEntityManager().createNamedQuery(getEntityTypeName() + FIND_ALL_QUERY);
    }

    /**
     * 
     * @return Returns the named query, assuming that a query returning all entity objects is named <Entity>.countAll. This method was not made final since it
     *         is possible that, in certain situations, the findAll query is named differently.
     */
    public Query countAllQuery() {
	return getEntityManager().createNamedQuery(getEntityTypeName() + COUNT_ALL_QUERY);
    }

    @SuppressWarnings("unchecked")
    @Override
    public final List<T> find(final boolean all, final int firstResult, final int maxResults) {
	Query q = findAllQuery();
	if (!all) {
	    q.setMaxResults(maxResults);
	    q.setFirstResult(firstResult);
	}
	return q.getResultList();
    }

    @Override
    public final List<T> findRange(final int[] range) {
	return find(false, range[0], range[1]);
    }

    @Override
    public final List<T> findAll() {
	return find(true, -1, -1);
    }

    @Override
    public final int count() {
	final Query query = countAllQuery();
	return ((Long) query.getSingleResult()).intValue();
    }

    /**
     * Initialises the internal <code>entityTypeClass</code> generic type.
     */
    @SuppressWarnings("unchecked")
    private void initGenericType() {
	if (entityTypeClass == null) {
	    ParameterizedType parameterizedType = (ParameterizedType) getClass().getGenericSuperclass();
	    entityTypeClass = ((Class<T>) parameterizedType.getActualTypeArguments()[0]);
	}
    }

    /**
     * 
     * @return Returns the generic type class.
     */
    private Class<T> getEntityTypeClass() {
	initGenericType();
	return entityTypeClass;
    }

    /**
     * 
     * @return Returns the generic type class simple name.
     */
    private String getEntityTypeName() {
	return getEntityTypeClass().getSimpleName();
    }

}