package com.linkare.commons.jpa;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

import javax.persistence.Column;
import javax.persistence.EntityManager;
import javax.persistence.MappedSuperclass;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;
import javax.persistence.Version;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;

import com.linkare.commons.dao.Deletable;
import com.linkare.commons.dao.Identifiable;
import com.linkare.commons.jpa.exceptions.DomainException;
import com.linkare.commons.jpa.utils.EntityManagerLocator;

/**
 * Entity class representing a domain object. This class may be used in any project where domain persisted entities require an auto increment id.
 * 
 * @author Paulo Zenida - Linkare TI
 */
@MappedSuperclass
@XmlAccessorType(XmlAccessType.FIELD)
public abstract class DomainObject implements Serializable, Identifiable<Long>, Deletable, Loggable {

    private static final long serialVersionUID = 1L;

    @Version
    @Column(name = "version")
    private Integer version = 1;

    /**
     * 
     */
    public DomainObject() {
	super();
    }

    /**
     * @return the version
     */
    public Integer getVersion() {
	return version;
    }

    /**
     * @param version
     *            the version to set
     */
    public void setVersion(Integer version) {
	this.version = version;
    }

    /**
     * 
     * @param objectClass
     * @param id
     * @return
     */
    public static <Pk extends Serializable> DomainObject getById(final Class<? extends DomainObject> objectClass, final Pk id) {
	final EntityManager em = EntityManagerLocator.getCurrentEntityManager();
	if (em == null) {
	    throw new DomainException("error.no.entitymanager.found");
	}
	return em.find(objectClass, id);
    }

    @PrePersist
    public void prePersist() {
    }

    @PreUpdate
    public void preUpdate() {
    }

    @PreRemove
    public void preRemove() {
    }

    /**
     * By default, log all fields.
     */
    @Override
    public String serializeToLog() {
	String result = "";
	for (final Field field : this.getClass().getDeclaredFields()) {
	    if (shouldIgnoreField(field)) {
		continue;
	    }
	    final boolean isAccessible = field.isAccessible();
	    field.setAccessible(true);
	    try {
		result += field.getName() + "=" + field.get(this) + ",";
	    } catch (Exception e) {
		result += field.getName() + "=" + e.toString();
	    }
	    field.setAccessible(isAccessible);
	}
	return result.endsWith(",") ? result.substring(0, result.length() - 1) : result;
    }

    private boolean shouldIgnoreField(final Field field) {
	return Modifier.isStatic(field.getModifiers());
    }

    public boolean isDeletable() {
	return true;
    }

    @Override
    public boolean delete() {
	return true;
    }

    public boolean isPersistent() {
	return id() != null;
    }

    @Override
    public abstract boolean equals(final Object other);

    @Override
    public abstract int hashCode();

    @Override
    public abstract String toString();
}