package com.linkare.commons.jpa.audit;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.EntityManager;
import javax.persistence.FetchType;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;
import javax.persistence.Query;
import javax.persistence.Table;

import com.linkare.commons.jpa.DefaultDomainObject;
import com.linkare.commons.jpa.audit.utils.AuditorLocator;
import com.linkare.commons.jpa.exceptions.DomainException;
import com.linkare.commons.jpa.utils.EntityManagerLocator;

// TODO Recheck the Auditable interface. Use checked exceptions instead of DomainException?
/**
 * 
 * An audited object is, by definition, a logged domain object.
 * 
 * @author Paulo Zenida - Linkare TI
 * 
 * @see com.linkare.commons.entities.audit.Auditor
 * @see com.linkare.commons.entities.logging.LoggedDomainObject
 */
@Entity
@Inheritance(strategy = InheritanceType.JOINED)
@Table(name = "audited_object")
@NamedQueries( {
	@NamedQuery(name = "AuditedObject.findAllAuditors", query = "Select auditor from Auditor auditor, AuditOperation operation, AuditedObject audited where operation.auditor = auditor and operation.audited = audited order by auditor.operationDate"),
	@NamedQuery(name = "AuditedObject.findAllAuditorsByOperation", query = "Select auditor from Auditor auditor, AuditOperation operation, AuditedObject audited where operation.auditor = auditor and operation.audited = audited and operation.operation = :operation order by auditor.operationDate"),
	@NamedQuery(name = "AuditedObject.findAllAuditorsByOperationDesc", query = "Select auditor from Auditor auditor, AuditOperation operation, AuditedObject audited where operation.auditor = auditor and operation.audited = audited and operation.operation = :operation order by auditor.operationDate desc") })
public abstract class AuditedObject extends DefaultDomainObject implements Auditable {

    private static final long serialVersionUID = 1L;

    @OrderBy("auditor")
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, mappedBy = "audited")
    private List<AuditOperation> auditOperations;

    public AuditedObject() {
	super();
    }

    /**
     * @return the auditOperations
     */
    public final List<AuditOperation> getAuditOperations() {
	return Collections.unmodifiableList(auditOperations);
    }

    /**
     * 
     * @param auditOperation
     */
    private void addAuditOperation(final AuditOperation auditOperation) {
	if (auditOperations == null) {
	    auditOperations = new ArrayList<AuditOperation>();
	}
	auditOperations.add(auditOperation);
    }

    /**
     * 
     * @param auditOperation
     */
    @SuppressWarnings("unused")
    private void removeAuditOperation(final AuditOperation auditOperation) {
	if (auditOperations == null) {
	    return;
	}
	auditOperations.remove(auditOperation);

    }

    /**
     * Utility method to add an audit operation.
     * 
     * @param operation
     *            The operation to which this audit operation is being performed.
     */
    private void addAuditOperation(final Operation operation) {
	final EntityManager em = EntityManagerLocator.getCurrentEntityManager();
	final Auditor currentAuditor = AuditorLocator.getCurrentAuditor();
	final AuditOperation auditOperation = new AuditOperation(currentAuditor, this, operation);
	em.persist(auditOperation);
	addAuditOperation(auditOperation);
    }

    @PrePersist
    @Override
    public final void prePersist() {
	super.prePersist();
	addAuditOperation(Operation.CREATE);
    }

    @PreUpdate
    @Override
    public final void preUpdate() {
	super.preUpdate();
	addAuditOperation(Operation.UPDATE);
    }

    @PreRemove
    @Override
    public final void preRemove() {
	super.preRemove();
	addAuditOperation(Operation.REMOVE);
    }

    // Implementation of the interface Auditable

    /**
     * @throws DomainException
     *             Any audited object must have one and only one auditor that audits the create operation, since each audited object can be created only once.
     */
    @Override
    public final Auditor getCreationAuditor() {
	final Auditor auditor = AuditedObject.findAllAuditorsByOperationDesc(Operation.CREATE);
	if (auditor == null) {
	    throw new DomainException("error.no.auditor.found.for.audited.object");
	}
	return auditor;
    }

    /**
     * @return Returns the last auditor that audits the update operation for a given persisted entity, when it exists one. Returns null otherwise.
     */
    @Override
    public final Auditor getLastUpdateAuditor() {
	return AuditedObject.findAllAuditorsByOperationDesc(Operation.UPDATE);
    }

    /**
     * @return Returns the auditor that audits the remove operation for a given persisted entity, when it exists one. Returns null otherwise.
     */
    @Override
    public final Auditor getRemoveAuditor() {
	return AuditedObject.findAllAuditorsByOperationDesc(Operation.REMOVE);
    }

    @Override
    public final List<Auditor> getAllAuditors() {
	return findAllAuditors();
    }

    @Override
    public final List<String> getAllModifiers() {
	final List<Auditor> updateAuditors = findAllAuditorsByOperation(Operation.UPDATE);
	if (updateAuditors == null || updateAuditors.isEmpty()) {
	    return null;
	}
	final List<String> result = new ArrayList<String>(updateAuditors.size());
	for (final Auditor auditor : updateAuditors) {
	    result.add(auditor.getPerformedBy());
	}
	return result;
    }

    @Override
    public final Date getCreationDate() {
	final Auditor creationAuditor = getCreationAuditor();
	return creationAuditor == null ? null : creationAuditor.getOperationDate();
    }

    @Override
    public final String getCreator() {
	final Auditor creationAuditor = getCreationAuditor();
	return creationAuditor == null ? null : creationAuditor.getPerformedBy();
    }

    @Override
    public final Date getLastModificationDate() {
	final Auditor lastUpdateAuditor = getLastUpdateAuditor();
	return lastUpdateAuditor == null ? null : lastUpdateAuditor.getOperationDate();
    }

    @Override
    public final String getLastModifier() {
	final Auditor lastUpdateAuditor = getLastUpdateAuditor();
	return lastUpdateAuditor == null ? null : lastUpdateAuditor.getPerformedBy();
    }

    @Override
    public final Date getRemoveDate() {
	final Auditor removeAuditor = getRemoveAuditor();
	return removeAuditor == null ? null : removeAuditor.getOperationDate();
    }

    @Override
    public final String getRemover() {
	final Auditor removeAuditor = getRemoveAuditor();
	return removeAuditor == null ? null : removeAuditor.getPerformedBy();
    }

    // End implementation of the interface Auditable

    // Utility static methods
    @SuppressWarnings("unchecked")
    public static final List<Auditor> findAllAuditors() {
	final EntityManager em = EntityManagerLocator.getCurrentEntityManager();
	final Query query = em.createNamedQuery("AuditedObject.findAllAuditors");
	return query.getResultList();
    }

    @SuppressWarnings("unchecked")
    public static final List<Auditor> findAllAuditorsByOperation(final Operation operation) {
	final EntityManager em = EntityManagerLocator.getCurrentEntityManager();
	final Query query = em.createNamedQuery("AuditedObject.findAllAuditorsByOperation");
	query.setParameter("operation", operation);
	return query.getResultList();
    }

    public static final Auditor findAllAuditorsByOperationDesc(final Operation operation) {
	final EntityManager em = EntityManagerLocator.getCurrentEntityManager();
	final Query query = em.createNamedQuery("AuditedObject.findAllAuditorsByOperationDesc");
	query.setMaxResults(1);
	query.setParameter("operation", operation);
	return (Auditor) query.getSingleResult();
    }
}