package com.mapbox.android.core.location;

import android.content.Intent;
import android.location.Location;
import android.location.LocationManager;
import android.support.annotation.Nullable;
import com.google.android.gms.location.LocationResult;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static com.mapbox.android.core.location.Utils.checkNotNull;
import static com.mapbox.android.core.location.Utils.isOnClasspath;

/**
 * A wrapper class representing location result from the location engine.
 * <p>
 * TODO: Override default equals(), hashCode() and toString()
 *
 * @since 1.0.0
 */
public final class LocationEngineResult {
  private static final String GOOGLE_PLAY_LOCATION_RESULT = "com.google.android.gms.location.LocationResult";

  private final List<Location> locations;

  private LocationEngineResult(List<Location> locations) {
    this.locations = Collections.unmodifiableList(locations);
  }

  /**
   * Creates {@link LocationEngineResult} instance for location.
   *
   * @param location default location added to the result.
   * @return instance of the new location result.
   * @since 1.0.0
   */
  public static LocationEngineResult create(Location location) {
    checkNotNull(location, "location can't be null");
    List<Location> locations = new ArrayList<>();
    locations.add(location);
    return new LocationEngineResult(locations);
  }

  /**
   * Creates {@link LocationEngineResult} instance for given list of locations.
   *
   * @param locations list of locations.
   * @return instance of the new location result.
   * @since 1.0.0
   */
  public static LocationEngineResult create(List<Location> locations) {
    checkNotNull(locations, "locations can't be null");
    return new LocationEngineResult(locations);
  }

  /**
   * Returns most recent location available in this result.
   *
   * @return the most recent location {@link Location} or null.
   * @since 1.0.0
   */
  @Nullable
  public Location getLastLocation() {
    return locations.isEmpty() ? null : locations.get(0);
  }

  /**
   * Returns locations computed, ordered from oldest to newest.
   *
   * @return ordered list of locations.
   * @since 1.0.0
   */
  public List<Location> getLocations() {
    return Collections.unmodifiableList(locations);
  }

  /**
   * Extracts location result from intent object
   *
   * @param intent valid intent object
   * @return location result.
   * @since 1.1.0
   */
  @Nullable
  public static LocationEngineResult extractResult(Intent intent) {
    LocationEngineResult result = null;
    if (isOnClasspath(GOOGLE_PLAY_LOCATION_RESULT)) {
      result = extractGooglePlayResult(intent);
    }
    return result == null ? extractAndroidResult(intent) : result;
  }

  private static LocationEngineResult extractGooglePlayResult(Intent intent) {
    LocationResult result = LocationResult.extractResult(intent);
    return result != null ? LocationEngineResult.create(result.getLocations()) : null;
  }

  private static LocationEngineResult extractAndroidResult(Intent intent) {
    return !hasResult(intent) ? null :
      LocationEngineResult.create((Location) intent.getExtras()
        .getParcelable(LocationManager.KEY_LOCATION_CHANGED));
  }

  private static boolean hasResult(Intent intent) {
    return intent != null && intent.hasExtra(LocationManager.KEY_LOCATION_CHANGED);
  }
}
