/*
 * (C) Copyright 2015-2016 Nuxeo SA (http://nuxeo.com/).
 * This is unpublished proprietary source code of Nuxeo SA. All rights reserved.
 * Notice of copyright on this source code does not indicate publication.
 *
 * Contributors:
 *     Remi Cattiau <rcattiau@nuxeo.com>
 */
package org.nuxeo.ecm.restapi.server.jaxrs.edgecache;

import java.util.List;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.nuxeo.ecm.core.api.NuxeoPrincipal;
import org.nuxeo.ecm.webengine.model.WebObject;
import org.nuxeo.ecm.webengine.model.impl.AbstractResource;
import org.nuxeo.ecm.webengine.model.impl.ResourceTypeImpl;
import org.nuxeo.runtime.api.Framework;

import com.nuxeo.edgecache.model.EdgeServerConfiguration;
import com.nuxeo.edgecache.service.EdgeCacheServer;
import com.nuxeo.edgecache.service.EdgeCacheService;

/**
 * @author loopingz
 *
 */
@WebObject(type="edgecache")
@Produces({ MediaType.APPLICATION_JSON })
public class EdgeCacheServerObject extends AbstractResource<ResourceTypeImpl> {

    private EdgeCacheService edgeCacheService;

    @Override
    protected void initialize(Object... args) {
        edgeCacheService = Framework.getService(EdgeCacheService.class);
    }

    /**
     * Create a proxy token ( anonymous user with Proxy permission )
     * @param deviceId of the Edge Cache server
     * @param deviceDescription used for range ip storage
     * @return the new token
     */
    @POST
    public Response registerServer(@QueryParam("deviceId") String deviceId, @QueryParam("ipRanges") String ipRanges, @QueryParam("url") String url, @QueryParam("ttl") Integer ttl) {
        if (!((NuxeoPrincipal) ctx.getPrincipal()).isAdministrator()) {
            return Response.ok().status(Status.FORBIDDEN).build();
        }
        try {
            EdgeCacheServer token = edgeCacheService.registerServer(deviceId, ipRanges, url, ttl == null ? 30 : ttl);
            return Response.ok(token).status(Response.Status.CREATED).build();
        } catch (IllegalArgumentException e) {
            return Response.ok().status(Status.BAD_REQUEST).build();
        }
    }
    
    /**
     * Create a proxy token ( anonymous user with Proxy permission )
     * @param deviceId of the Edge Cache server
     * @param deviceDescription used for range ip storage
     * @return the new token
     */
    @GET
    public List<EdgeCacheServer> getServers(@QueryParam("deviceId") String deviceId, @QueryParam("ipRanges") String ipRanges, @QueryParam("url") String url) {
        return (List<EdgeCacheServer>) edgeCacheService.getEdgeCacheServers();
    }

    /**
     * Unregister the server
     * @param tokenId of the server to unregister
     */
    @DELETE
    @Path("{uid}")
    public Response unregisterServer(@PathParam("uid") String tokenId) {
        if (tokenId == null) {
            return Response.ok().status(Status.NOT_FOUND).build();
        }
        edgeCacheService.unregisterServer(tokenId);
        return Response.ok().status(Status.NO_CONTENT).build();
    }
    
    /**
     * Set the last TTL for server
     * @param tokenId of the server to reset TTL
     */
    @GET
    @Path("{uid}/ping")
    @Produces("application/json")
    public Response pingServer(@PathParam("uid") String tokenId) {
        EdgeServerConfiguration result;
        if (tokenId == null) {
            return Response.ok().status(Status.NOT_FOUND).build();
        }
        try {
            result = edgeCacheService.pingServer(tokenId);
        } catch (IllegalArgumentException e) {
            return Response.ok().status(Status.BAD_REQUEST).build();
        }
        return Response.ok(result).build();
    }

    /**
     * Update the token
     * @param deviceDescription to update
     */
    @PUT
    @Path("{uid}")
    public Response updateServer(@PathParam("uid") String tokenId, @QueryParam("ipRanges") String ipRanges, @QueryParam("url") String url, @QueryParam("ttl") Integer ttl) {
        if (tokenId == null) {
            return Response.ok().status(Status.NOT_FOUND).build();
        }
        try {
            edgeCacheService.updateServer(tokenId, ipRanges, url, ttl == null ? 30 : ttl);
        } catch (IllegalArgumentException e) {
            return Response.ok().status(Status.BAD_REQUEST).build();
        }
        return Response.ok().status(Status.NO_CONTENT).build();
    }
}
