/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.api;

import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import oracle.kv.impl.api.TopologyInfo;
import oracle.kv.impl.fault.OperationFaultException;
import oracle.kv.impl.topo.Partition;
import oracle.kv.impl.topo.PartitionId;
import oracle.kv.impl.topo.RepGroupId;
import oracle.kv.impl.topo.Topology;
import oracle.kv.impl.topo.change.TopologyChange;

public class TopologyManager {
    private final String kvsName;
    private volatile Topology topology;
    private Topology localTopology;
    private Localizer localizer = null;
    private final List<PreUpdateListener> preUpdateListeners = new LinkedList<PreUpdateListener>();
    private final List<PostUpdateListener> postUpdateListeners = new LinkedList<PostUpdateListener>();
    private final int maxTopoChanges;
    private final Logger logger;

    public TopologyManager(String kvsName, int maxTopoChanges, Logger logger) {
        this.kvsName = kvsName;
        this.maxTopoChanges = maxTopoChanges;
        this.logger = logger;
    }

    public synchronized void addPreUpdateListener(PreUpdateListener listener) {
        if (!this.preUpdateListeners.contains(listener)) {
            this.preUpdateListeners.add(listener);
        }
    }

    public synchronized void addPostUpdateListener(PostUpdateListener listener) {
        if (!this.postUpdateListeners.contains(listener)) {
            this.postUpdateListeners.add(listener);
        }
    }

    public synchronized void removePostUpdateListener(PostUpdateListener listener) {
        this.postUpdateListeners.remove(listener);
    }

    private void invokePreUpdateListeners(Topology newTopology) {
        assert (Thread.holdsLock(this));
        for (PreUpdateListener l : this.preUpdateListeners) {
            l.preUpdate(newTopology);
        }
    }

    private void invokePostUpdateListeners() {
        assert (Thread.holdsLock(this));
        Iterator<PostUpdateListener> itr = this.postUpdateListeners.iterator();
        while (itr.hasNext()) {
            if (!itr.next().postUpdate(this.topology)) continue;
            itr.remove();
        }
    }

    public void setLocalizer(Localizer localizer) {
        this.localizer = localizer;
    }

    public Topology getTopology() {
        return this.topology;
    }

    public Topology getLocalTopology() {
        return this.localTopology == null ? this.topology : this.localTopology;
    }

    public void setLocalTopology(Topology localTopology) {
        this.localTopology = localTopology;
    }

    public synchronized boolean update(Topology newTopology) {
        int currSeqNum;
        if (this.topology != null) {
            if (!this.kvsName.equals(this.topology.getKVStoreName())) {
                throw new IllegalArgumentException("Update topology associated with KVStore: " + this.topology.getKVStoreName() + " expected: " + this.kvsName);
            }
            this.checkTopologyId(this.topology.getId(), newTopology.getId());
            currSeqNum = this.topology.getSequenceNumber();
        } else {
            currSeqNum = 0;
        }
        int newSequenceNumber = newTopology.getSequenceNumber();
        if (currSeqNum >= newSequenceNumber) {
            this.logger.log(Level.INFO, "Topology update skipped. Current seq #: {0} Update seq #: {1}", new Object[]{currSeqNum, newSequenceNumber});
            return true;
        }
        TopologyManager.checkVersion(this.logger, newTopology);
        this.invokePreUpdateListeners(newTopology);
        if (!this.updateLocalTopology(newTopology)) {
            return false;
        }
        this.logger.log(Level.INFO, "Topology updated from seq#: {0} to {1}", new Object[]{currSeqNum, newSequenceNumber});
        this.topology = TopologyManager.pruneChanges(newTopology, Integer.MAX_VALUE, this.maxTopoChanges);
        this.invokePostUpdateListeners();
        return true;
    }

    public void checkPartitionChanges(RepGroupId rgId, Topology newTopo) throws IllegalStateException {
        if (this.topology == null || this.topology.getPartitionMap().size() == 0) {
            return;
        }
        Set<PartitionId> currentPartitions = this.getRGPartitions(rgId, this.topology);
        Set<PartitionId> newPartitions = this.getRGPartitions(rgId, newTopo);
        for (PartitionId npId : newPartitions) {
            Partition lp;
            Partition np = newTopo.get(npId);
            Partition cp = this.topology.get(npId);
            if (np.getRepGroupId().equals(cp.getRepGroupId())) {
                currentPartitions.remove(npId);
                continue;
            }
            Partition partition = lp = this.localTopology != null ? this.localTopology.get(npId) : null;
            if (lp == null) {
                String msg = String.format("%s in the new topology(seq #: %,d) is absent from this shard in the current topology(seq #: %,d) and there is no partition migration in progress.", np, newTopo.getSequenceNumber(), this.topology.getSequenceNumber());
                throw new IllegalStateException(msg);
            }
            if (lp.getRepGroupId().equals(np.getRepGroupId())) continue;
            String msg = String.format("%s in the new topology(seq #: %,d) and %s in the local topology(internal seq#: %,d) are associated with different shards", np, newTopo.getSequenceNumber(), lp, this.localTopology.getSequenceNumber());
            throw new IllegalStateException(msg);
        }
        for (PartitionId cpId : currentPartitions) {
            String msg;
            Partition lp;
            Partition cp = this.topology.get(cpId);
            Partition partition = lp = this.localTopology != null ? this.localTopology.get(cpId) : null;
            if (lp == null) {
                msg = String.format("%s is in the current topology(seq #: %,d) but is absent from the new topology(seq #: %,d) and there is no partition migration in progress.", cp, this.topology.getSequenceNumber(), newTopo.getSequenceNumber());
                throw new IllegalStateException(msg);
            }
            if (!lp.getRepGroupId().equals(cp.getRepGroupId())) continue;
            msg = String.format("%s is associated with the same shard in both the current(seq #: %,d) and local topologies but is associated with a different shard %s in the new topology(seq#: %,d). ", cp, this.topology.getSequenceNumber(), newTopo.get(cpId).getRepGroupId(), this.localTopology.getSequenceNumber());
            throw new IllegalStateException(msg);
        }
    }

    private Set<PartitionId> getRGPartitions(RepGroupId rgId, Topology topo) {
        HashSet<PartitionId> hostedPartitions = new HashSet<PartitionId>(100);
        for (Partition p : topo.getPartitionMap().getAll()) {
            if (!p.getRepGroupId().equals(rgId)) continue;
            hostedPartitions.add((PartitionId)p.getResourceId());
        }
        return hostedPartitions;
    }

    public static void checkVersion(Logger logger, Topology topology) {
        int topoVersion = topology.getVersion();
        if (topoVersion == 1) {
            return;
        }
        if (topoVersion != 0) {
            throw new OperationFaultException("Encountered topology with version: " + topoVersion + " Current topology version: " + 1);
        }
        logger.warning("Using r1 topology, it was not upgraded.");
    }

    synchronized void update(long topologyId, List<TopologyChange> changes) {
        Topology workingCopy = this.topology == null ? new Topology(this.kvsName, topologyId) : this.topology.getCopy();
        this.checkTopologyId(workingCopy.getId(), topologyId);
        int prevSequenceNumber = workingCopy.getSequenceNumber();
        if (!workingCopy.apply(changes)) {
            return;
        }
        this.invokePreUpdateListeners(workingCopy);
        if (!this.updateLocalTopology(workingCopy)) {
            return;
        }
        this.topology = TopologyManager.pruneChanges(workingCopy, changes.get(0).getSequenceNumber(), this.maxTopoChanges);
        this.logger.log(Level.INFO, "Topology incrementally updated from seq#: {0} to {1}", new Object[]{prevSequenceNumber, this.topology.getSequenceNumber()});
        this.invokePostUpdateListeners();
    }

    public synchronized void update(TopologyInfo topoInfo) {
        this.update(topoInfo.getTopoId(), topoInfo.getChanges());
    }

    private void checkTopologyId(long localTopoId, long remoteTopoId) {
        if (localTopoId == remoteTopoId) {
            return;
        }
        if (localTopoId == 0L || remoteTopoId == 0L) {
            return;
        }
        String msg = "Inconsistent use of Topology. An attempt was made to update this topology created on " + new Date(localTopoId) + " with changes originating from a different topology created on " + new Date(remoteTopoId) + ". This exception indicates an application configuration issue.";
        throw new IllegalStateException(msg);
    }

    public static Topology pruneChanges(Topology topo, int limitSeqNum, int maxTopoChanges) {
        int firstChangeSeqNum = topo.getChangeTracker().getFirstChangeSeqNum();
        if (firstChangeSeqNum == -1) {
            return topo;
        }
        int firstRetainedChangeSeqNum = Math.min(topo.getSequenceNumber() - maxTopoChanges + 1, limitSeqNum);
        if (firstRetainedChangeSeqNum <= firstChangeSeqNum) {
            return topo;
        }
        topo.discardChanges(firstRetainedChangeSeqNum - 1);
        return topo;
    }

    public synchronized boolean updateLocalTopology() {
        if (this.topology == null) {
            return true;
        }
        if (!this.updateLocalTopology(this.topology)) {
            return false;
        }
        this.invokePostUpdateListeners();
        return true;
    }

    private boolean updateLocalTopology(Topology newTopology) {
        if (this.localizer == null) {
            return true;
        }
        Topology local = this.localizer.localizeTopology(newTopology);
        if (local == null) {
            this.logger.log(Level.INFO, "Topology update to {0} skipped. Unable to update local topology.", newTopology.getSequenceNumber());
            return false;
        }
        this.localTopology = local;
        return true;
    }

    public boolean inTransit(PartitionId partitionId) {
        if (partitionId.isNull()) {
            return false;
        }
        RepGroupId localGroupId = this.getLocalTopology().getRepGroupId(partitionId);
        RepGroupId currentGroupId = this.getTopology().getRepGroupId(partitionId);
        return localGroupId.getGroupId() != currentGroupId.getGroupId();
    }

    public static interface Localizer {
        public Topology localizeTopology(Topology var1);
    }

    public static interface PreUpdateListener {
        public void preUpdate(Topology var1);
    }

    public static interface PostUpdateListener {
        public boolean postUpdate(Topology var1);
    }
}

