/*
 * Decompiled with CFR 0.152.
 */
package org.xerial.snappy;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.WritableByteChannel;
import org.xerial.snappy.Snappy;
import org.xerial.snappy.SnappyFramed;

public final class SnappyFramedOutputStream
extends OutputStream
implements WritableByteChannel {
    public static final int MAX_BLOCK_SIZE = 65536;
    public static final int DEFAULT_BLOCK_SIZE = 65536;
    public static final double DEFAULT_MIN_COMPRESSION_RATIO = 0.85;
    private final ByteBuffer buffer;
    private final byte[] outputBuffer;
    private final double minCompressionRatio;
    private final OutputStream out;
    private boolean closed;

    public SnappyFramedOutputStream(OutputStream out) throws IOException {
        this(out, 65536, 0.85);
    }

    public SnappyFramedOutputStream(OutputStream out, int blockSize, double minCompressionRatio) throws IOException {
        if (out == null) {
            throw new NullPointerException();
        }
        if (minCompressionRatio <= 0.0 || minCompressionRatio > 1.0) {
            throw new IllegalArgumentException("minCompressionRatio " + minCompressionRatio + " must be in (0,1.0]");
        }
        if (blockSize <= 0 || blockSize > 65536) {
            throw new IllegalArgumentException("block size " + blockSize + " must be in (0, 65536]");
        }
        this.out = out;
        this.minCompressionRatio = minCompressionRatio;
        this.buffer = ByteBuffer.allocate(blockSize);
        this.outputBuffer = new byte[Snappy.maxCompressedLength(blockSize)];
        this.writeHeader(out);
    }

    private void writeHeader(OutputStream out) throws IOException {
        out.write(SnappyFramed.HEADER_BYTES);
    }

    @Override
    public boolean isOpen() {
        return !this.closed;
    }

    @Override
    public void write(int b) throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        if (this.buffer.remaining() <= 0) {
            this.flushBuffer();
        }
        this.buffer.put((byte)b);
    }

    @Override
    public void write(byte[] input, int offset, int length) throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        this.write(ByteBuffer.wrap(input, offset, length));
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        if (this.closed) {
            throw new ClosedChannelException();
        }
        if (this.buffer.remaining() <= 0) {
            this.flushBuffer();
        }
        int srcLength = src.remaining();
        if (this.buffer.remaining() >= src.remaining()) {
            this.buffer.put(src);
            return srcLength;
        }
        int srcEnd = src.position() + src.remaining();
        while (src.position() + this.buffer.remaining() <= srcEnd) {
            src.limit(src.position() + this.buffer.remaining());
            this.buffer.put(src);
            this.flushBuffer();
        }
        src.limit(srcEnd);
        this.buffer.put(src);
        return srcLength;
    }

    @Override
    public final void flush() throws IOException {
        if (this.closed) {
            throw new IOException("Stream is closed");
        }
        this.flushBuffer();
        this.out.flush();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public final void close() throws IOException {
        if (this.closed) {
            return;
        }
        try {
            this.flush();
            this.out.close();
        }
        finally {
            this.closed = true;
        }
    }

    private void flushBuffer() throws IOException {
        if (this.buffer.position() > 0) {
            this.buffer.flip();
            this.writeCompressed(this.buffer);
            this.buffer.clear();
        }
    }

    private void writeCompressed(ByteBuffer buffer) throws IOException {
        byte[] input = buffer.array();
        int length = buffer.remaining();
        int crc32c = this.calculateCRC32C(input, 0, length);
        int compressedLength = Snappy.compress(input, 0, length, this.outputBuffer, 0);
        if ((double)compressedLength / (double)length <= this.minCompressionRatio) {
            this.writeBlock(this.out, this.outputBuffer, 0, compressedLength, true, crc32c);
        } else {
            this.writeBlock(this.out, input, 0, length, false, crc32c);
        }
    }

    private int calculateCRC32C(byte[] data, int offset, int length) {
        return SnappyFramed.maskedCrc32c(data, offset, length);
    }

    private void writeBlock(OutputStream out, byte[] data, int offset, int length, boolean compressed, int crc32c) throws IOException {
        out.write(compressed ? 0 : 1);
        int headerLength = length + 4;
        out.write(headerLength);
        out.write(headerLength >>> 8);
        out.write(headerLength >>> 16);
        out.write(crc32c);
        out.write(crc32c >>> 8);
        out.write(crc32c >>> 16);
        out.write(crc32c >>> 24);
        out.write(data, offset, length);
    }
}

