/**
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */


woodstock4_3._dojo.provide("woodstock4_3._base.body");



woodstock4_3._dojo.provide("woodstock4_3._base.cookie");

/**
 * @class This class contains functions to manipulate cookies.
 * @static
 * @private
 */
woodstock4_3._base.cookie = {
    /**
     * This function will get the cookie value.
     *
     * @return {String} The cookie value.
     * @private
     */
    _get: function() {
        // Get document cookie.
        var cookie = document.cookie;

        // Parse cookie value.
        var pos = cookie.indexOf(this._cookieName + "=");
        if (pos == -1) {
            return null;
        }

        var start = pos + this._cookieName.length + 1;
        var end = cookie.indexOf(";", start);
        if (end == -1) {
            end = cookie.length;
        }

        // return cookie value
        return cookie.substring(start, end);
    },

    /**
     * This function will load the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _load: function() {
        // Get document cookie.
        var cookieVal = this._get();
        if (cookieVal == null) {
            return false;
        }

        // Break cookie into names and values.
        var a = cookieVal.split('&');

        // Break each pair into an array.
        for (var i = 0; i < a.length; i++) {
            a[i] = a[i].split(':');
        }

        // Set name and values for this object.
        for (i = 0; i < a.length; i++) {
            this[a[i][0]] = unescape(a[i][1]);
        }
        return true;
    },

    /**
     * This function will reset the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _reset: function() {
        // Clear cookie value.
        document.cookie = this._cookieName + "=";
        return true;
    },

    /**
     * This function will store the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _store: function() {
        // Create cookie value by looping through object properties
        var cookieVal = "";

        // Since cookies use the equals and semicolon signs as separators,
        // we'll use colons and ampersands for each variable we store.
        for (var prop in this) {
            // Ignore properties that begin with '$' and methods.
            if (prop.charAt(0) == '$' || typeof this[prop] == 'function') {
                continue;
            }
            if (cookieVal != "") {
                cookieVal += '&';
            }
            cookieVal += prop + ':' + escape(this[prop]);
        }
        var cookieString = this._cookieName + "=" + cookieVal;
        if (this._path != null) {
            cookieString += ";path=" + this._path;
        }
        // Store cookie value.
        document.cookie = cookieString;
        return true;
    }
};

/**
 * @class This class contains functionality for scrolling.
 * @constructor This function is used to construct a javascript object for
 * maintaining scroll position via cookie.
 * @param {String} viewId
 * @param {String} path
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3._base.scrollCookie = function(viewId, path) {    
    // All predefined properties of this object begin with '$' because
    // we don't want to store these values in the cookie.
    this._cookieName = viewId;
    this._path = path;

    // Default properties.
    this._left = "0";
    this._top  = "0";

    /**
     * This function will load the cookie and restore scroll position.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._restore = function() {
        // Load cookie value.
        this._load();
        scrollTo(this._left, this._top);
        return true;
    };

    /**
     * This function will set the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._set = function() {
        var documentElement = window.document.documentElement;
        if (documentElement && documentElement.scrollTop) {
            this._left = documentElement.scrollLeft;
            this._top  = documentElement.scrollTop;
        } else {
            this._left = window.document.body.scrollLeft;
            this._top  = window.document.body.scrollTop;
        }
        // if the left and top scroll values are still null
        // try to extract it assuming the browser is IE.
        if (this._left == null && this._top == null) {
            this._left = window.pageXOffset;
            this._top = window.pageYOffset;
        }
        // Store cookie value.
        this._store();
        return true;
    };
    return true;
};

// Inherit cookie properties.
woodstock4_3._base.scrollCookie.prototype = woodstock4_3._base.cookie;
woodstock4_3._dojo.require("woodstock4_3._base.browser");


woodstock4_3._dojo.provide("woodstock4_3._base.common");

woodstock4_3._dojo.require("woodstock4_3.theme.common");

/**
 * @class This class contains functions common to HTML elements.
 * @static
 * @private
 */
woodstock4_3._base.common = {
    /**
     * Variable needed when submitting form so timeout will work properly.
     * @private
     */
    _formToSubmit: null,

    /**
     * Variable needed when submitting form so timeout will work properly.
     * @private
     */
    _submissionComponentId: null,

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // String functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Replace occurences of delimiter with the escapeChar and the
     * delimiter. For example replace "," with "/," if delimiter == "," and
     * escapeChar is "/".
     *
     * @param {String} s The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {String} The escaped string.
     * @private
     */
    _escapeString: function(s, delimiter, escapeChar) {
        if (s == null) {
            return null;
        }
        if (delimiter == null) {
            return s;
        }
        if (escapeChar == null) {
            return null;
        }
        
        // Escape occurrences of delimiter with 
        // escapeChar and the delimiter.
        //
        // First escape the escape char.
        //
        var escape_escapeChar = escapeChar;
        if (escapeChar == "\\") {
            escape_escapeChar = escapeChar + escapeChar;
        }
        
        var rx = new RegExp(escape_escapeChar, "g");
        var s1 = s.replace(rx, escapeChar + escapeChar);
        
        rx = new RegExp(delimiter, "g");
        return s1.replace(rx, escapeChar + delimiter);
    },
    
    /**
     * Replace occurences of a sequence of 2 instances of delimiter 
     * with 1 instance of the delimiter. For example replace ",," with "," if 
     * delimiter == ","
     *
     * @param {String} s The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {String} The unescaped string.
     * @private
     */
    _unescapeString: function(s, delimiter, escapeChar) {
        if (s == null) {
            return null;
        }
        if (delimiter == null) {
            return s;
        }
        if (escapeChar == null) {
            return null;
        }
        
        // UnEscape occurrences of delimiter with 
        // single instance of the delimiter
        //
        var escape_escapeChar = escapeChar;
        if (escapeChar == "\\") {
            escape_escapeChar = escapeChar + escapeChar;
        }
        
        // First unescape the escape char.
        //
        var rx = new RegExp(escape_escapeChar + escape_escapeChar, "g");
        var s1 = s.replace(rx, escapeChar);
        
        // Now replace escaped delimters
        //
        rx = new RegExp(escape_escapeChar + delimiter, "g");
        return s1.replace(rx, delimiter);
    },
    
    /**
     * Return an array of unescaped strings from escapedString
     * where the escaped character is delimiter.
     * If delimiter is "," escapedString might have the form
     * <p><pre>
     * XX\,XX,MM\,MM
     *
     * where "\" is the escape char.
     * 
     * and is returned as an array
     * array[0] == "XX,XX"
     * array[1] == "MM,MM"
     * </pre><p>
     *
     * @param {String} escapedString The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {Array} An array of unescaped strings.
     * @private
     */
    _unescapeStrings: function(escapedString, delimiter, escapeChar) {
        if (escapedString == null || escapedString == "") {
            return null;
        }
        if (delimiter == null || delimiter == "") {
            return escapedString;
        }
        if (escapeChar == null || escapeChar == "") {
            return null;
        }
        
        // Need to do this character by character.
        var selections = new Array();
        var index = 0;
        var escseen = 0;
        var j = 0;
        
        for (var i = 0; i < escapedString.length; ++i) {
            if (escapedString.charAt(i) == delimiter) {
                if (escseen % 2 == 0) {
                    selections[index++] = escapedString.slice(j, i);
                    j = i + 1;
                }
            }
            if (escapedString.charAt(i) == escapeChar) {
                ++escseen;
                continue;
            } else {
                escseen = 0;
            }
        }
        
        // Capture the last split.
        selections[index] = escapedString.slice(j);
        
        // Now unescape each selection
        var unescapedArray = new Array();
        
        // Now replace escaped delimiters
        // i.e.  "\," with ","
        for (i = 0; i < selections.length; ++i) {
            unescapedArray[i] = woodstock4_3._base.common._unescapeString(
            selections[i], delimiter, escapeChar);
        }
        return unescapedArray;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Style functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Use this function add any styleClass to an html tag
     *
     * @param {Node} element the dom html tag element
     * @param {String} styleClass the name of the class to add
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _addStyleClass: function(element, styleClass) {
        // routine protection in javascript
        if (element == null || styleClass == null) {
            return false;
        }
        
        // handle easy case first
        if (element.className == null) {
            element.className = styleClass;
            return true;
        }
        
        // break out style classes into an array  
        var classes = woodstock4_3._base.common._splitStyleClasses(element);
        if (classes == null) {
            return false;
        }
        
        // For each element of classes, compare it to styleClass.
        // If it is not in the array, append it to the end.
        for (var i = 0; i < classes.length; i++) {
            if (classes[i] != null && classes[i] == styleClass) {
                return true;
            }
        }
        element.className = element.className + " " + styleClass;
        return true;
    },
    
    /**
     * Use this function to check if an array has a style class
     *
     * @param {Array} styleArray of style classes to check
     * @param {String} styleClass the styleClass to check
     * @return {Array} An array of classes.
     * @private
     */
    _checkStyleClasses: function(styleArray, styleClass) {
        if (styleArray == null || styleClass == null) {
            return false;
        }
        for (var i = 0; i < styleArray.length; i++) {
            if (styleArray[i] != null && styleArray[i] == styleClass) {
                return true;
            }
        }   
        return false;
    },
    
    /**
     * Use this function to get array of style classes
     *
     * @param {Node} element the dom html tag element
     * @return {Array} An array of classes.
     * @private
     */
    _splitStyleClasses: function(element) {
        if (element != null && element.className != null) {
            return element.className.split(" ");
        } else {
            return null;
        }
    },
    
    /**
     * Use this function remove any styleClass for an html tag
     *
     * @param {Node} element the dom html tag element
     * @param {String} styleClass the name of the class to remove
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _stripStyleClass: function(element, styleClass) {
        // routine protection in javascript
        if (element == null || styleClass == null || element.className == null) {
            return false;
        }
        
        // break out style classes into an array  
        var classes = woodstock4_3._base.common._splitStyleClasses(element);
        if (classes == null) {
            return false;
        }
        
        // For each styleClass, check if it's hidden and remove otherwise write
        // it back out to the class
        for (var i = 0; i < classes.length; i++) {
            if (classes[i] != null && classes[i] == styleClass) {
                classes.splice(i,1);  	
            }
        }
        element.className = classes.join(" ");
        return true;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Submit functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function to insert a hidden field element in the page.
     *
     * @param {String} elementId The element ID of the html tag 
     * @param {String} elementValue The value of the html tag.
     * @param {Node} parentForm The parent form of the html tag.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _insertHiddenField: function(elementId, elementValue, parentForm) {
        // We have to assume that there is only one element
        // with elementId. document.getElementById, returns
        // the first one it finds, which appears to be the 
        // first one created dynamically, if more than one 
        // element is created dynamically with the same id.
        //
        // appendChild just appends even if there is an element
        // with the same id that exists.
        //
        // The assumption is that there should only be 
        // one element in the document with such an id.
        //
        // If the elementId exists just modifiy its value
        // instead of creating and appending.
        //
        
        var element = document.forms[parentForm.id].elements[elementId];
        if (element != null) {
            element.value = elementValue;            
            return true;            
        } 
        
        var newElement = document.createElement('input');
        newElement.type = 'hidden';
        newElement.id = elementId;
        newElement.value = elementValue;
        newElement.name = elementId;
        parentForm.appendChild(newElement);

        return true;
    },
    
    /**
     * Use this function to submit a virtual form.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     * @deprecated Virtual forms only supported by JSF based form component.
     */
    _submitForm: function() {
        // "formToSubmit" is a literal (not virtual) form.
        // "submissionComponentId" is a component id (not client id).
        // the virtual form implementation uses _submissionComponentId
        // to determine which virtual form (if any) was submitted.
        if (woodstock4_3._base.common._formToSubmit == null) {
            return false;
        }
        if (woodstock4_3._base.common._submissionComponentId != null &&
                woodstock4_3._base.common._submissionComponentId.length > 0) {
            woodstock4_3._base.common._insertHiddenField('_submissionComponentId', 
                woodstock4_3._base.common._submissionComponentId,
                woodstock4_3._base.common._formToSubmit);
        }
        woodstock4_3._base.common._formToSubmit.submit();
        return false;
    },
    
    /**
     * Helper function to submit a virtual form.
     *
     * @param {Node} form The HTML form element to submit.
     * @param {String} submissionComponentId The Id of the component submitting the form.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     * @deprecated Virtual forms only supported by JSF based form component.
     */
    _timeoutSubmitForm: function(form, submissionComponentId) {
        woodstock4_3._base.common._formToSubmit = form;
        woodstock4_3._base.common._submissionComponentId = submissionComponentId;
        setTimeout('woodstock4_3._base.common._submitForm()', 0);
        return true;
    },
    
    /**
     * Helper function to submit a virtual form.
     *
     * @param {Node} form The HTML form element to submit.
     * @param {String} submissionComponentId The Id of the component submitting the form.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     * @deprecated Virtual forms only supported by JSF based form component.
     */
    _leaveSubmitterTrace: function(form, submissionComponentId) {
        // This function only needs to be called in the onclick handler of 
        // an ActionSource component that appears within a -standard- table.
        // Under those circumstances, if this function is not called, then when
        // the component is clicked, the virtual form implementation will have 
        // no way of knowing that a virtual form was submitted.
        if (form != null && submissionComponentId != null && 
                submissionComponentId.length > 0) {
            woodstock4_3._base.common._insertHiddenField('_submissionComponentId',
            submissionComponentId, form);
        }
        return true;
    },
    
    /**
     * delete a previously created element by _createSubmittableArray.
     *
     * @param {String} name The element ID of the html tag 
     * @param {Node} form The HTML form element to submit.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     * @deprecated Virtual forms only supported by JSF based form component.
     */
    _deleteSubmittableArray: function(name, parentForm) {
        try {
            var submittableArray  = document.getElementById(name);
            if (submittableArray != null) {
                parentForm.removeChild(submittableArray);
            }
        } catch (e) {}
        return true;
    },
    
    /**
     * This method creates a hidden "select" element with id 
     * and name attributes set name, values taken from the values
     * array argument, and display labels from the labels array.
     * It adds the element to the parentForm argument.
     * <p>
     * The pupose of this method is to create an array of values
     * that can be decoded using "name" as the key from a FacesContext
     * external context's "getRequestParameterValuesMap" as an
     * array of Strings. This reduces the need of rendering hidden input
     * field and delimiting several strings so that a multiple selection
     * can be returned.
     * </p><p>
     * The labels array provides an additional piece of data
     * for use on the client, but it is not contained in the submit.
     * All values added to the select are selected so that the
     * values will be submitted.
     * </p>
     *
     * @param {String} name The element ID of the html tag 
     * @param {Node} form The HTML form element to submit.
     * @param {Array} labels
     * @param {Array} values
     * @return {Node} The newly created select element.
     * @private
     */
    _createSubmittableArray: function(name, parentForm, labels, values) {
        // An attempt is made to remove a possibly previously created element
        // by this name. It always deletes an element of name from parentForm.
        woodstock4_3._base.common._deleteSubmittableArray(name, parentForm);
        
        if (values == null || values.length <= 0) {
            return null;
        }
        
        var selections = document.createElement('select');
        selections.className = woodstock4_3.theme.common.getClassName("HIDDEN");
        selections.name = name;
        selections.id = name;
        selections.multiple = true;
        
        // Start from the end of the array because
        // add puts things in at the head.
        //
        for (var i = 0; i < values.length; ++i) {
            var opt = document.createElement('option');
            opt.value = values[i];
            if (labels != null) {
                opt.label = labels[i];
            }
            opt.defaultSelected = true;
            selections.add(opt, null);
        }
        parentForm.appendChild(selections);
        return selections;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Visible functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function to test if the specified element is visible (i.e., it
     * does not contain the hidden className).
     *
     * @param {String} elementId The element ID of the html tag 
     * @return {boolean} true if visible; otherwise, false
     * @private
     */
    _isVisible: function(elementId) {
        if (elementId == null) {
            return false;
        }
        // Get element.
        var element = document.getElementById(elementId);
        return woodstock4_3._base.common._isVisibleElement(element);
    },
    
    /**
     * Use this function to test if the given element is visible (i.e., it
     * does not contain the hidden className).
     *
     * @param {Node} element The HTML element
     * @return {boolean} true if visible; otherwise, false
     * @private
     */
    _isVisibleElement: function(element) {
        if (element == null) {
            return false;
        }
        // Test for the hidden style class.
        var styleClasses = woodstock4_3._base.common._splitStyleClasses(element); 
        return !woodstock4_3._base.common._checkStyleClasses(styleClasses,
            woodstock4_3.theme.common.getClassName("HIDDEN"));
    },
    
    /**
     * Use this function to show or hide any html element in the page
     *
     * @param {String} elementId The element ID of the html tag 
     * @param {boolean} visible true to make the element visible, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _setVisible: function(elementId, visible) {
        if (elementId == null || visible == null ) {
            return false;
        }
        // Get element.
        var element = document.getElementById(elementId);
        return woodstock4_3._base.common._setVisibleElement(element, visible);
    },
    
    /**
     * Use this function to show or hide any html element in the page
     *
     * @param {Node} element The HTML element
     * @param {boolean} visible true to make the element visible, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _setVisibleElement: function(element, visible) {
        if (element == null || visible == null) {
            return false;
        }
        if (visible) {
            return woodstock4_3._base.common._stripStyleClass(element,
                woodstock4_3.theme.common.getClassName("HIDDEN"));
        } else {
            return woodstock4_3._base.common._addStyleClass(element,
                woodstock4_3.theme.common.getClassName("HIDDEN"));
        }
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.common = {
    escapeString: woodstock4_3._base.common._escapeString,
    unescapeString: woodstock4_3._base.common._unescapeString,
    unescapeStrings: woodstock4_3._base.common._unescapeStrings,
    addStyleClass: woodstock4_3._base.common._addStyleClass,
    checkStyleClasses: woodstock4_3._base.common._checkStyleClasses,
    splitStyleClasses: woodstock4_3._base.common._splitStyleClasses,
    stripStyleClass: woodstock4_3._base.common._stripStyleClass,
    insertHiddenField: woodstock4_3._base.common._insertHiddenField,
    submitForm: woodstock4_3._base.common._submitForm,
    timeoutSubmitForm: woodstock4_3._base.common._timeoutSubmitForm,
    leaveSubmitterTrace: woodstock4_3._base.common._leaveSubmitterTrace,
    deleteSubmittableArray: woodstock4_3._base.common._deleteSubmittableArray,
    createSubmittableArray: woodstock4_3._base.common._createSubmittableArray,
    isVisible: woodstock4_3._base.common._isVisible,
    isVisibleElement: woodstock4_3._base.common._isVisibleElement,
    setVisible: woodstock4_3._base.common._setVisible,
    setVisibleElement: woodstock4_3._base.common._setVisibleElement
};

/**
 * @class This class contains functions used to maintain focus and scroll position.
 * <p>
 * There can be an initial focus element and a default focus element. The
 * initial focus element is identifed by the "focusElementId" argument.
 * This argument is typically null on the first display of the page. If
 * the Body component is not preserving the focus then it may also be null,
 * at other times, since it represents the element id that last received
 * the focus before the request.
 * </p><p>
 * Whenever the page is displayed and "focusElementId" is null
 * "defaultFocusElementId" will receive the focus if it is not null. This id is
 * defined by the application using the Body "focus" attribute. If the
 * application wants to control the focus in all cases then it should set
 * the Body component "preserveFocus" attribute to "false". The application then
 * explicitly sets the Body "focus" attribute to the element id to receive
 * focus on every request/response.
 * </p><p>
 * In order to preserve focus across requests, the "focusElementFieldId"
 * element is used to preserve the id of the element that receives the focus
 * last. It is a hidden field that is submitted in the
 * request. See the "com.sun.webui.jsf.util.FocusManager" class for
 * information on how the focus is managed. This field exists in all
 * forms, since that it is the only way to guarantee that the last
 * element that received the focus is sent to the server. Which form
 * is submitted can never be known.
 * </p>
 * @constructor This function is used to construct a body object.
 * @param {String} viewId Used to name the scroll cookie
 * @param {String} path A member of the scroll cookie
 * @param {String} defaultFocusElementId The HTML element id that will receive focus.
 * @param {String} focusElementId The id of the element to receive the initial focus.
 * @param {String} focusElementFieldId The id of a hidden field to maintain
 * the id of the last element to have the focus.
 * @param {boolean} preserveScroll if true or not defined the scroll position is 
 * maintained else scroll is not maintained.
 * @private
 */
woodstock4_3._base.body = function(viewId, path, defaultFocusElementId, 
	focusElementId, focusElementFieldId, preserveScroll)  {
    /**
     * The id of the HTML element to receive the focus, if the
     * element identified in focusElementFieldId cannot receive the focus.
     * @private
     */
    this._defaultFocusId = defaultFocusElementId;

    /**
     * The id of a hidden input element whose value is the id
     * of the HTML element to receive the focus. It is set by
     * the _focusListener and calls to setFocusBy{Id,Element}.
     * @private
     */
    this._focusElementFieldId = focusElementFieldId;

    /**
     * The element id to receive the preserved, or initial focus.
     * This member should not be referenced once the onload listener
     * has been invoked. After that point the hidden field will have
     * have the element with the focus. We do this so that two locations
     * do not have to be kept in sync. Developers can just set the focus
     * to the element itself and the focus handler will manage the
     * focus persisitence.
     * @private
     */
    this._focusElementId = focusElementId;

    /**
     * Flag indicating to preserve scroll.
     * @private
     */
    this._preserveScroll = (preserveScroll == null)
        ? true : new Boolean(preserveScroll).valueOf();

    /**
     * Create the scroll cookie object.
     */
    if (this._preserveScroll == true) {
	this._scrollCookie = new woodstock4_3._base.scrollCookie(viewId, path);
    }

    /**
     * According to HTML spec only these elements have
     * "onfocus" which we will equate to "focus".
     * A, AREA, BUTTON, INPUT, LABEL, SELECT, TEXTAREA
     * However just check for a non null "focus" or 
     * catch the exception when trying to reference it.
     * Returns true if the element has a "focus" method, is not
     * disabled, and is visible, else false.
     *
     * @param {Node} element The DOM node to have focus.
     * @return {boolean} true if DOM Node can have focus.
     * @private
     */
    this._canAcceptFocus = function(element) {
	var result = false;
	try {
	    result = element != null && element.focus && !element.disabled
		&& element.type != "hidden"
		&& woodstock4_3._base.common._isVisible(element.id);
	} catch(err) {}
	return result;
    };

    /**
     * Record the id of the element that has just receivied focus.
     * This is called whenever an element receives the focus.
     * This is set on the document so that the cursor entering the
     * window does not trigger this listener.
     *
     * @param {Event} event The object generated by the focus event.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._focusListener = function(event) {
	// If it's not an element node just return
	//
	var node = null;
	var isElementNode = false;
	
	// is IE 
	//
	if (document.attachEvent) {
	    node = event.srcElement;
	
	    // We have to hard code "1" as the Node.ELEMENT_NODE in
	    // ie, because ie does not make the constant accessible.
	    //
	    isElementNode = (node == null ? false : node.nodeType == 1);
	} else {
	    node = event.target;
	    isElementNode = node.nodeType == Node.ELEMENT_NODE;
	}

	if (isElementNode) {
	    // Note that there is no reliable way to set
	    // focus to some other element if the event element
	    // deemed to receive the focus can't accept the focus.
	    //
	    this._updateFocusElementField(node);
	}
	return true;
    };

    /**
     * Set the initial focus and the scroll position.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._onLoadListener = function() {
	// register the focus listener first.
	// Then call "_setDefaultFocus" using "setTimeout" to
	// allow javascript processing to complete, probably
	// to allow "onload" to complete. The focus listener
	// will update the hidden focus fields as a result
	// of the call to "focus" in _setDefaultFocus.
	//

	// Add the focus listener, in the onload to prevent
	// recursive calls from calling _setDefaultFocus.
	//
        if (woodstock4_3._base.browser._isIe()) {
            woodstock4_3._dojo.connect(document, "onfocusin", this, "_focusListener");
        } else {
            woodstock4_3._dojo.connect(window, "onfocus", this, "_focusListener");
        }

        // Rely on the focus listener to update the focus
        // hidden fields by catching the 'element.focus()' in
        // _setDefaultFocus
        //
        this._setInitialFocus();

	// Set up the scroll position after the focus has been
	// restored. Need to make sure that this takes into
	// account the default focus that was just set.
	//
	return this._setDefaultScrollPosition();
    };

    /**
     * Update the page's scroll position.
     *
     * @param {Event} event The object generated by the onUnload event.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._onUnloadListener = function(event) {
	return this._storeScrollPosition();
    };

    /**
     * Set the default focus to the application's chosen default focus element.
     * This method should only be called once to prevent recursive
     * calls since it calls "focus()" on the focus element.
     * Called currently from the onload listener.
     * <p>
     * If "this._defaultFocusId" is not null it will receive the focus; 
     * otherwise, no focus is set.
     * </p>
     * @return {boolean} false if a default focus cannot be established, else true.
     * @private
     */
    this._setDefaultFocus = function() {
        // HTML elements may not have been added, yet.
        if (this._defaultFocusId != null) {
            var domNode = document.getElementById(this._defaultFocusId);
            if (domNode == null) {
                var _this = this; // Closure magic.
                return setTimeout(function() { _this._setDefaultFocus(); }, 10);
            }

            // Focus not set try the default.
            //
            if (this._setFocusById(this._defaultFocusId)) {
                return true;
            }
        }

	/* For now it doesn't sound like a good idea to ever set
	 * a "heuristic" default focus. It is better for screen readers to start
	 * from the top of the page which we will assume that that
	 * browser starts from there when focus is not set explicitly.
	 * This code can be removed, but left it in case we want to
	 * for some reason.

	// No previously set focus element and no default.
	// Find the first focusable element in the first available form.
	//
	for each (var f in window.document.forms) {
	    for each (var e in f.elements) {
		if (this._setFocusByElement(e)) {
		    return true;
		}
	    }
	}
	// If there is no form, set on the first available link
	//
	for each (var l in window.document.links) {
	    if (this._setFocusByElement(l)) {
		return true;
	    }
	}
	*/
	return false;
    };

    /**
     * This method is invoked in the onload listener, body._onLoadListener.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._setDefaultScrollPosition = function() {
	if (!this._preserveScroll) {
	    return false;
	}
	// # char found, anchor being used. forego scrolling.
	// CR 6342635. 
	//
        if (window.location.href.indexOf('#') == -1) {
	    this._scrollCookie._restore(); 
	} else {
	    // destroy the recent scroll data
	    //
	    this._scrollCookie._reset();
	}
        return true;
    };

    /**
     * Set the initial focus by restoring the focus from a previous
     * request or to the application's chosen default focus element.
     * This method should only be called once to prevent recursive
     * calls since it calls "focus()" on the focus element.
     * Called currently from the onload listener.
     * <p>
     * If "this._focusElementId" is not null it will receive the focus.
     * If that element can't receive the focus then the application defined 
     * "this._defaultFocusId" receives the focus. If that element cannot receive 
     * the focus, no focus is set.
     * </p>
     * @return {boolean} false if focus cannot be established, else true.
     * @private
     */
    this._setInitialFocus = function() {
        // HTML elements may not have been added, yet.
        if (this._focusElementId != null) {
            var domNode = document.getElementById(this._focusElementId);
            if (domNode == null) {
                var _this = this; // Closure magic.
                return setTimeout(function() { _this._setInitialFocus(); }, 10);
            }

            // Try to set focus to "this._focusElementId". If this fails
            // fallback to the app defined default 
            // "this.defaultFocusElementId", if there is one.
            //
            if (this._setFocusById(this._focusElementId)) {
                return true;
            }
        }
        return this._setDefaultFocus();
    };

    /**
     * Set the focus on "focusElement".
     * If focus can be set returns true else false.
     *
     * @param {Node} focusElement The DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._setFocusByElement = function(focusElement) {
	if (focusElement == null || !this._canAcceptFocus(focusElement)) {
	    return false;
	}

	// _canAcceptFocus tests the existence of the "focus" handler.
	// So it is safe to call it outside of a try/catch statement.
	// This should trigger the focus listener.
        try {
            // Still need try/catch because _canAcceptFocus doesn't account for 
            // when parent is invisible. For example, the table's sort panel 
            // closes during page submit making focus element invisible.
            focusElement.focus();
        } catch(err) {}

	// Assume that this update is performed by the 
	// focus listener. This policy was changed in order to 
	// call "_setDefaultFocus" using "setTimeout" in order for
	// javascript to have time to be evaluated, probably for
	// on load processing to complete.
	//this._updateFocusElementField(focusElement);
	return true;
    };

    /**
     * Set the focus on element with id "fid".
     * If focus can be set returns true else false.
     *
     * @param {String} fid The id of the DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._setFocusById = function(fid) {
	if (fid == null || fid.length == 0) {
	    return false;
	}
	return this._setFocusByElement(document.getElementById(fid));
    };

    /**
     * This method is invoked in the onunload event listener
     * body._onUnloadListener
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._storeScrollPosition = function() {
	if (!this._preserveScroll) {
	    return false;
	}
	try {
	    this._scrollCookie._set(); 
	} catch (e) {
	}
        return true; 
    };

    /** 
     * Update the hidden field that maintains the last element to 
     * receive the focus. If the body has multiple forms every form's
     * hidden field is updated with the "focusElement".
     *
     * @param {Node} focusElement The DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._updateFocusElementField = function(focusElement) {
	// Don't know if we'll have issues if multiple forms contain
	// an element with the same id. I know getElementById gets
	// confused.
	//

        if (focusElement == null) {
	    return false;
	}
	// Get the form that contains the focus element.
	//
	for (var i = 0;  i < document.forms.length; ++i) {
	    var form = document.forms[i];
            var field = null;

	    // Get the hidden field that maintains the focus element id.
	    // If it exists return it. We know its name is the same
	    // as its id.
	    //
	    try {
		field = form.elements[this._focusElementFieldId];
		if (field != null) {
		    field.value = focusElement.id;
		    continue;
		}
	    } catch (e) {
		// the array access of a non existent element
		// probably threw exception so create the field.
	    }
		
	    // If it doesn't exist create it.
	    // and add it to the form.
	    //
	    field = document.createElement('input');
	    field.type = 'hidden';
	    field.id = this._focusElementFieldId;
	    field.name = this._focusElementFieldId;
	    field.value = focusElement.id;
	    form.appendChild(field);
	}
	return true;
    };

    // The focus listener is set on the document so that the cursor 
    // entering the window does not trigger this listener. 
    this._onLoadListener();

    // If we are not preserving scroll don't add the unload listener.
    if (this._preserveScroll == true) {
        woodstock4_3._dojo.addOnUnload(this, "_onUnloadListener");
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.table");

woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3._base.proto");


woodstock4_3._dojo.provide("woodstock4_3.widget.common");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.theme.common");


woodstock4_3._dojo.provide("woodstock4_3.xhr");

/**
 * @class This class contains functions for XMLHTTP requests.
 * @static
 */
woodstock4_3.xhr = {
    /**
     * This function is used to generate a request using the XMLHTTP 
     * request object (XHR) the underlying protocol.
     *
     * @param {String} method Use "GET" on operations that are primarily data
     * retrieval requests (default). Use "POST" on operations that send data to
     * the server.
     * @param {Object} props Key-Value pairs of properties.
     * @config {boolean} async Flag indicating request should asynchronous (default).
     * @config {String} content Optional postable string or DOM object data.
     * @config {Object} headers Key-Value pairs of request header properties.
     * @config {Function} onError The callback function called in an error case.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {Function} onReady The callback function called on a successful response.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {String} url The URL may be either a complete or relative URL.
     * @private
     */
    _doRequest: function(method, props) {
        if (props == null || props.url == null) {
            console.debug("Error: _send has null props"); // See Firebug console.
            return false;
        }
        var xhr = woodstock4_3.xhr._getXhr();
        if (xhr == null) {
            return false;
        }

        // Set callback functions.
        var _onError = props.onError;
        var _onReady = props.onReady;
        xhr.onreadystatechange = function() {
            // State shows "loaded".
            if (xhr.readyState == 4) {
                // Status shows "OK".
                if (xhr.status == 200) {
                    if (typeof _onReady == "function") {
                        _onReady(xhr);
                    }
                } else {
                    if (typeof _onError == "function") {
                        _onError(xhr);
                    }
                }
            }
        }
        // Open XHR.
        xhr.open((method) ? method : "GET", props.url,
            (props.async != null) ? props.async : true);

        // Set default headers -- must set after XHR is open.
        xhr.setRequestHeader("X-Requested-With", "XMLHttpRequest");
        xhr.setRequestHeader("X-Woodstock-Version", "4.3");

        // Set headers.
        if (props.headers) {
            for (var property in props.headers) {
                xhr.setRequestHeader(property, props.headers[property]);
            }
        }
        // Send request.
        xhr.send((props.content) ? props.content : null);
        return true;
    },

    /**
     * This function is used to generate a "GET" request using the XMLHTTP 
     * request object (XHR) the underlying protocol.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {boolean} async Flag indicating request should asynchronous (default).
     * @config {String} content Optional postable string or DOM object data.
     * @config {Object} headers Key-Value pairs of request header properties.
     * @config {Function} onError The callback function called in an error case.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {Function} onReady The callback function called on a successful response.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {String} url The URL may be either a complete or relative URL.
     * @return {boolean} true if successful; otherwise, false.
     */
    get: function(props) {
        return woodstock4_3.xhr._doRequest("GET", props);
    },

    /**
     * Get either an XMLHttpRequest or ActiveXObject object.
     *
     * @private
     */
    _getXhr: function() {
        var xhr = null;
    
        // Use native XMLHttpRequest object, if possible.
        if (window.XMLHttpRequest && !(window.ActiveXObject)) {
            try {
                xhr = new XMLHttpRequest();
            } catch(e) {
                console.debug("Error: XMLHttpRequest not available"); // See Firebug console.
            }
        } else if (window.ActiveXObject) {
            // Use ActiveXObject for IE.
            try {
                xhr = new ActiveXObject("Msxml2.XMLHTTP");
            } catch(e) {
        	try {
                    xhr = new ActiveXObject("Microsoft.XMLHTTP");
        	} catch(e) {
                    console.debug("Error: ActiveXObject not available"); // See Firebug console.
        	}
            }
        }
        return xhr;
    },

    /**
     * This function is used to generate a "POST" request using the XMLHTTP 
     * request object (XHR) the underlying protocol.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {boolean} async Flag indicating request should asynchronous (default).
     * @config {String} content Optional postable string or DOM object data.
     * @config {Object} headers Key-Value pairs of request header properties.
     * @config {Function} onError The callback function called in an error case.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {Function} onReady The callback function called on a successful response.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {String} url The URL may be either a complete or relative URL.
     * @return {boolean} true if successful; otherwise, false.
     */
    post: function(props) {
        return woodstock4_3.xhr._doRequest("POST", props);
    }
};

/**
 * @class This class contains functions common to all widgets.
 * @static
 */
woodstock4_3.widget.common = {
    /**
     * Object used to store onLoad functions.
     * @private
     */
    _loaders: [],

    /**
     * Object used to store private onLoad functions.
     * @private
     */
    _preLoaders: [],

    /**
     * Object used to store widget properties.
     * @private
     */
    _props: new Object(),

    /**
     * This function is used to add a widget, HTML fragment, or static string to
     * the given domNode.
     * <p>
     * Note: If props is a string, it shall be added as the innerHTML of the 
     * given domNode. By default, all strings shall be HTML escaped.
     * <p></p>
     * If props is a JSON object, containing a fragment property instead of
     * widgetType, it shall be added as the innerHTML of the given domNode. A
     * fragment is also a string; however it is evaluated as JavaScript and not
     * HTML escaped.
     * <p></p>
     * If props is a JSON object, containing a widgetType property, a widget
     * shall be created. The newly created widget shall be added as a child of 
     * the given domNode.
     * <p></p>
     * Valid values for the position param consist of "last" or null. In 
     * general, position only applies when creating new widgets; however, if the 
     * position param is null, any existing child nodes are removed from the
     * domNode.
     * </p>
     *
     * @param {Node} domNode The DOM node to add widget.
     * @param {Object} props Key-Value pairs of properties.
     * @param {boolean} position The position to add widget.
     * @param {boolean} escape HTML escape strings (default).
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _addFragment: function(domNode, props, position, escape) {
        if (domNode == null || props == null) {
            return false;
        }
        var common = woodstock4_3.widget.common;

        // If position is null, remove existing nodes. The contents shall be
        // replaced by the newly created widget.
        if (position == null) {
            common._removeChildNodes(domNode);

            // Note: To ensure Dojo does not replace the given domNode, always
            // provide a default position to the _createWidget() function. The
            // domNode may be used as a place holder for later updates.
            position = "last";            
        }

        // Add include, refId, or html object -- do not HTML escape.
        if (common._isFragmentObject(props)) {
            // Add HTML string.
            if (typeof props.html == "string") {
                return common._addFragment(domNode, props.html, position, false);
            } else if (typeof props.include == "string") {
                // Add URL include.
                var _url = props.include;
                woodstock4_3.xhr.get({
                    onReady: function (response) {
                        woodstock4_3.widget.common._addFragment(domNode, 
                            response.responseText, position, false);
                    },
                    url: _url
                });
            } else if (typeof props.refId == "string") {
                // Add widget reference.
                var refIds = props.refId.split(",");
                for (var i = 0; i < refIds.length; i++) {
                    // Ensure nodes have not been added to DOM already.
                    if (document.getElementById(refIds[i]) != null) {
                        continue;
                    }
                    var widget = common.getWidget(refIds[i]);
                    if (widget) {
                        common._addWidget(domNode, widget, position);
                    }   
                }
            }
            return true;
        }
 
        // Add fragment.
        if (typeof props == 'string') {
            // Strip script fragments, set innerHTML property, and
            // eval scripts using a timeout.
            //
            // Note that using Dojo's ContentPane widget would have
            // been more preferable than creating a new dependency
            // based on Prototype. However, as of version .4.1, Dojo
            // still does not use a timeout to eval JavaScript; thus,
            // IE generates errors with innerHTML.
            //
            // "The problem has to do with the browser's poor
            // threading model. Basically, once some JavaScript code
            // is running, all other threads of execution are on hold
            // until the running thread is finished with it's
            // task. This includes whatever thread of execution that
            // updates the DOM model when new content is inserted via
            // innerHTML. For example if you do:
            //
            // foo.innerHTML = '<span id="bar">Bar</span>';
            // var bar = document.getElementById('bar');
            //
            // This code will sometimes fail because although you
            // added the content via innerHTML the DOM may not have
            // been updated and therefore the "bar" element does not
            // exist in the DOM yet. To work around this you can do:
            //
            // foo.innerHTML = '<span id="bar">Bar</span>';
            // setTimeout(function() {
            //     var bar = document.getElementById('bar');
            // }, 10);
            //
            // This will work because in 10 milliseconds whatever
            // event handler is running will have finished, the DOM
            // will update, then the callback will execute and have no
            // problem accessing the DOM element."
            //
            // The full discussion on this topic can be found at:
            //
            // http://www.ruby-forum.com/topic/73990
            //
            if (escape != null && new Boolean(escape).valueOf() == false) {
                // Note: IE does not insert script tags via innerHTML.
                common._appendHTML(domNode, 
                    woodstock4_3._base.proto._stripScripts(props));

                // Ensure _loaded() is not called before widget completion.
                common._props[domNode.id] = "_wait";

                // Evaluate JavaScript.
                setTimeout(function() {
                    var common = woodstock4_3.widget.common;

                    // Eval not required for Mozilla/Firefox, but consistent.
                    woodstock4_3._base.proto._evalScripts(props);
                    if (new Boolean(woodstock4_3._base.config.parseOnLoad).valueOf() == true) {
                        common._parseMarkup(domNode);
                    }
                    delete(common._props[domNode.id]); // Clean up.
                }, 10);
            } else {
                // Static strings must be HTML escaped by default.
                common._appendHTML(domNode,
                    woodstock4_3._base.proto._escapeHTML(props));
            }
        } else {
            // Create widget.
            common._createWidget(domNode, props, position, false);
        }
        return true;
    },

    /**
     * Register a function to be called after the DOM has finished loading 
     * and widgets declared in markup have been instantiated.
     * <p><pre>
     * 
     * woodstock4_3.widget.common.addOnLoad(function);
     * woodstock4_3.widget.common.addOnLoad(object, "name", [arg1, arg2...]);
     *
     * </pre></p>
     * @param {Object|Function} obj A Function or Object containing a function
     * by the given name.
     * @param {String} name The function name to call.
     * @return {boolean} true if successful; otherwise, false.
     */
    addOnLoad: function(obj, name) {
        if (obj == null) {
            return false;
        }

        var common = woodstock4_3.widget.common;
        if (arguments.length == 1){
            common._loaders.push(obj);
        } else if (arguments.length > 1){
            common._loaders.push(common._hitch.apply(this, arguments));
        }
        return true;
    },
 
     /**
     * Register a function to be called after the DOM has finished loading 
     * and widgets declared in markup have been instantiated.
     * <p>
     * Functions are called after widgets have been instantiated, but before
     * loaders set via the public addOnLoad function. This ensures JavaScript
     * APIs are available for JSF components (e.g., file chooser).
     * </p><p><pre>
     * 
     * woodstock4_3.widget.common._addOnLoad(function);
     * woodstock4_3.widget.common._addOnLoad(object, "name", [arg1, arg2...]);
     *
     * </pre></p>
     * @param {Object|Function} obj A Function or Object containing a function
     * by the given name.
     * @param {String} name The function name to call.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _addOnLoad: function(obj, name) {
        if (obj == null) {
            return false;
        }

        var common = woodstock4_3.widget.common;
        if (arguments.length == 1){
            common._preLoaders.push(obj);
        } else if (arguments.length > 1){
            common._preLoaders.push(common._hitch.apply(this, arguments));
        }
        return true;
    },

    /**
     * This function is used to add widgets to given the domNode.
     * <p>
     * Valid values for the position param consist of "before", "last", or null.
     * If the position is "last", the widget is appended as the last child of 
     * the given domNode. If the position is "before", the widget is inserted
     * before the given domNode. If the position is null, the given domNode is 
     * replaced by the resulting HTML.
     * </p>
     *
     * @param {Node} domNode The DOM node to add widget.
     * @param {Node} widget The widget object to add.
     * @param {String} position The position within given domNode.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _addWidget: function(domNode, widget, position) {
        if (domNode == null || widget == null) {
            return false;
        }
        // Add widget to DOM.
        if (position == "last") {
            // Append widget as the last child of the given DOM node.
            domNode.appendChild(widget._domNode);
        } else if (position == "before") {
            // Append widget before given DOM node.
            domNode.parentNode.insertBefore(widget._domNode, domNode);
        } else if (domNode.parentNode) {
            // Replace given DOM node with widget.
            domNode.parentNode.replaceChild(widget._domNode, domNode);
        }
        return true;
    },

    /**
     * This function is used to append HTML strings to the innerHTML property of
     * the given domNode.
     * <p>
     * Note: Concatenating innerHTML with new strings does not always work. When
     * adding multiple HTML elements to domNode, we can get into a situation
     * where domNode.innerHTML may not yet contain all the changes made to 
     * the previously added DOM node. Therefore, we shall wrap new strings in an
     * HTML span element so it may be added as a child of domNode.
     * </p>
     *
     * @param {Node} domNode The DOM node to append string.
     * @param {String} html The HTML string to append.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _appendHTML: function(domNode, html) {
        if (domNode.innerHTML != null && domNode.innerHTML.length > 0) {
            var span = document.createElement('span');            
            span.innerHTML = html;
            domNode.appendChild(span);
        } else {
            // Don't need span when innerHTML is empty.
            domNode.innerHTML = html;
        }
        return true;
    },
    
    /**
     * This function is used to add the context path to a given property.
     * The property could denote an url such as path to an image to be rendered
     * or location of a resource to navigate to when a link is clicked. The
     * property given has to start with a "/". Relative urls that start with 
     * ".." will not have the context path appended. It also checks whether the
     * context path has already been added to the property.
     *
     * @param (String} prefix The context path of the application.
     * @param (String} url The URL to which contextPath is to be appended.
     * @return {String} The property that has the contextPath appended.
     * @private
     */
    _appendPrefix: function(prefix, url) {
        if (prefix == null || url == null) {
            return null;
        }
        return (url.charAt(0) == "/" && url.indexOf(prefix + "/") == -1)
            ? prefix + url : url;
    },

    /**
     * This function is used to create and start a widget. If the parseOnLoad
     * property is true, widget creation is deferred to the window.onLoad event.
     * <p>
     * Typically, an HTML tag is used as a temporary place holder for the newly
     * created widget. This allows widgets to be added to the document in the 
     * proper location. For example, the expected HTML markup to create an image
     * widget looks like so:
     * </p><p><pre>
     * &lt;span id="j_id1"&gt;&lt;/span&gt;
     * &lt;script type="text/javascript"&gt;
     *   woodstock4_3.widget.common.createWidget('j_id1',{
     *     "id": "form:image1",
     *     "height": "32px",
     *     "width": "32px",
     *     "src": "/example/images/dice1.gif",
     *     "widgetType": "image",
     *   });
     * &lt;/script&gt;
     * </pre></p><p>
     * For better lookup performance, it is assumed that script tags are located
     * immediately after an HTML span element. Ultimately, the newly created
     * widget is added as a child of the HTML span element.
     * </p><p>
     * Performance testing shows this approach is quicker than using the 
     * document.getElementById() function or Level 0 DOM syntax, especially for 
     * large HTML tables. The underlying problem appears to be the extra step
     * taken to convert HTML element IDs to a UTF-16 character encoding.
     * </p>
     *
     * @param {String} elementId The HTML element id to replace (may be null).
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The ID of the widget to create.
     * @config {String} widgetType The widget type to create.
     * @return {Object} The widget object if parseOnLoad is false; otherwise, null.
     */
    createWidget: function(elementId, props) {
        if (props == null) {
            return false;
        }
        var common = woodstock4_3.widget.common;

        // Since there is only one window.onLoad event, the ajaxZone tag of JSF
        // Extensions cannot make use of the parseOnLoad feature while 
        // re-rendering widgets. In this case, we shall call 
        // document.getElementById() even though it is not as efficient.
        if (new Boolean(woodstock4_3._base.config.parseOnLoad).valueOf() == false) {
            var domNode = document.getElementById(elementId);
            return common._createWidget(domNode, props, "last");
        }
        // Store widget properties for window.onLoad event.
        common._props[elementId] = props;
        return null;
    },

    /**
     * This function is used to create and start a widget.
     * <p>
     * See the _addNode function for valid position values.
     * </p>
     *
     * @param {Node} domNode The DOM node to add widget.
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The ID of the widget to create.
     * @config {String} widgetType The widget type to create.
     * @param {String} position The widget position within given domNode.
     * @returns {Object} The newly created widget.
     * @private
     */
    _createWidget: function(domNode, props, position) {
        var widget = null;
        if (props == null || props.id == null || props.widgetType == null) {
            console.debug("Error: _createWidget has null props"); // See Firebug console.
            return widget;
        }

        // Destroy previously created widgets, events, etc.
        var common = woodstock4_3.widget.common;
        common.destroyWidget(props.id);

        // Retrieve required module.
        var _widgetType = "woodstock4_3.widget."  + props.widgetType;
        woodstock4_3._dojo.require(_widgetType);
        
        try {
            // Get widget object.
            var obj = woodstock4_3._dojo.getObject(_widgetType);

            // Instantiate widget. 
            // Note: Dojo mixes attributes, if domNode is provided.
            widget = new obj(props);
        } catch (err) {
            var message = "Error: _createWidget falied for id=" + props.id;
	    message = common._getExceptionString(err, message, true);
            console.debug(message); // See Firebug console.
            return null;
        }

        // Add widget to DOM.
        common._addWidget(domNode, widget, position);

        // Start widget.
        widget._startup();
        return widget;
    },
     
    /**
     * This function is used to create and start a widget from parsed markup. 
     * See the _parseMarkup() function.
     * <p>
     * Unlike the _createWidget() function, setTimeout() is called to allow for
     * progressive rendering. Performance testing shows that the download is
     * quicker with the setTimout() call than without. 
     * </p><p>
     * The setTimeout() also helps to display extremely large tables. As an 
     * example, consider a table that displays 1000 rows. Many browsers display 
     * warnings if JavaScript runs longer than 5 seconds. The setTimeout breaks 
     * up the amount of JavaScript run at any given time and completes each 
     * segment within the browser's alloted time.
     * </p>
     *
     * @param {Node} domNode The DOM node to append widget.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _createMarkupWidget: function(domNode) {
        // Ensure a value exists for given domNode id.
        if (domNode == null 
                || woodstock4_3.widget.common._props[domNode.id] == null) {
            return false;
        }
            
        // Set timeout to allow for progressive rendering.
        setTimeout(function() {
            var common = woodstock4_3.widget.common;
            common._createWidget(domNode, common._props[domNode.id], "last");
            delete(common._props[domNode.id]); // Clean up.

            // Test remaining widget properties.
            for (var property in common._props) {
                return; // At least one widget has not been created.
            }
            // Call after all widgets have been created.
            common._loaded();
        }, 0);
        return true;
    },

    /**
     * This function is used to detroy a widget.
     * <p>
     * Note: By default, all descendant widgets are destroyed as well.
     * </p>
     *
     * @param {String} id The widget id to destroy.
     * @return {boolean} true if successful; otherwise, false.
     */
    destroyWidget: function(id) {
        if (id == null) {
            return false;
        }
        // Destroy previously created widgets, events, etc.
        var widget = woodstock4_3.widget.common.getWidget(id);
        if (widget) {
            return widget.destroyRecursive();
        }
        return false;
    },

    /**
     * Return the appropriate event object depending on the browser.
     *
     * @param {Event} event The client side event generated
     * @return {Event} The appropriate event object
     * @private
     */
    _getEvent: function(event) {
        return (event) ? event : ((window.event) ? window.event : null);          
    },
    
    /**
     * Return a formatted string with the information about the exception error.
     * @param {Error} err An exception <code>Error</code> instance.
     * @param {String} synopsis A general message from the error call site.
     * @param {boolean} verbose If true all possible information from the
     * <code>err</code> is formatted into the returned String.
     * @return {String} Formatted information from <code>err</code>.
     * @private
     */
    _getExceptionString: function(err, synopsis, verbose) {
	var msg = (synopsis == null ? "" : synopsis) + "\n";
	msg = msg + (err.name != null ? err.name : "Unnamed Error") + "\n";
	msg = msg + (err.message != null ? err.message : "No message") + "\n";
	if (verbose) {
	    msg = msg + "\n" + 
		(err.fileName != null ? err.fileName : "No filename" + "::") +
		(err.lineNumber != null ? err.lineNumber : "No lineNumber");
	    if (err.stack != null) {
		var stack = err.stack.replace(/()@/g, "\n");
		msg = msg + "\n" + stack;
	    }
	}
	return msg;
    },

    /**
     * This function returns the closest form ancestor of the given DOM node.
     * <p>
     * Note: Traversing the DOM can be slow, but all HTML input elements have a
     * form property. Therefore, avoid using this function when the form can be
     * retrieved via an HTML input element.
     * </p>
     * @param {Node} domNode A DOM node contained in the form.
     * @return {Node} The HTML form element or null if not found.
     * @private
     */
    _getForm: function(domNode) {
        var form = null;
        var obj = domNode;
        while (obj != null) {
            if (obj.tagName == "FORM") {
                form = obj;
                break;
            }
            obj = obj.parentNode;
        }
        return form;
    },

    /**
     * Return the key code of the key which generated the event.
     *
     * @param {Event} event The client side event generated
     * @return {String} The key code of the key which generated the event
     * @private
     */
    _getKeyCode: function(event) {
        if (event == null) {
            return null;
        }
        return (event.keyCode) 
            ? event.keyCode 
            : ((event.which) ? event.which : event.charCode);              
    },

    /**
     * Get array containing the absolute left and top position of the given DOM
     * node relative to the browser window.
     *
     * @param {Node} domNode The DOM node compute position for.
     * @return {Array} Array containing the absolute left and top position.
     * @private
     */
    _getPosition: function(domNode) {
        var leftPos = topPos = 0;
        if (domNode && domNode.offsetParent) {
            leftPos = domNode.offsetLeft;
            topPos = domNode.offsetTop;
            while ((domNode = domNode.offsetParent) != null) {
                leftPos += domNode.offsetLeft;
                topPos += domNode.offsetTop;
            }
        }
        return [leftPos, topPos];
    },

    /**
     * Get the page height, handling standard noise to mitigate browser
     * differences.
     *
     * @return {int} The page height or null if not available.
     * @private
     */
    _getPageHeight: function() {
        // Mozilla browsers.
        if (window.innerHeight) {
            return window.innerHeight;
        }
        // IE strict mode
        if (document.documentElement.clientHeight > 0) {
            return document.documentElement.clientHeight; 
        }
        // IE quirks mode.
        if (document.body.clientHeight) {
            return document.body.clientHeight;
        }
        return null;
    },

    /**
     * Get the page width, handling standard noise to mitigate browser 
     * differences.
     *
     * @return {int} The page height or null if not available.
     * @private
     */
    _getPageWidth: function() {
        // Mozilla browsers.
        if (window.innerWidth) {
            return window.innerWidth;
        }
        // IE strict mode.
        if (document.documentElement.clientWidth > 0) {
            return document.documentElement.clientWidth; 
        }
        // IE quirks mode.
        if (document.body.clientWidth) {
            return document.body.clientWidth;
        }
        return null;
    },

    /**
     * This function is used to obtain a widget.
     *
     * @param {String} id The widget id.
     * @return {Object} The widget object.
     */
    getWidget: function(id) {
        if (id == null) {
            return null;
        }
        return woodstock4_3._dijit.byId(id);
    },

    /**  
     * Create a function call within the given scope. 
     * <p><pre> 
     *  
     * _hitch(object, name, [arg1, arg2...]); 
     * 
     * </pre></p> 
     * @param {Object} scope The scope in which to call the given method. 
     * @param {String} name The function name to call. 
     * @return {Function} The resulting function. 
     * @private 
     */ 
    _hitch: function(scope, name) {
        if (scope == null || name == null) {
            return null;
        }
        // Save given arguments via closure magic. 
        var pre = woodstock4_3._base.proto._toArray(arguments, 2); 
        return function() { 
            // Append new arguments, if any. 
            var args = woodstock4_3._base.proto._toArray(arguments); 
            return scope[name].apply(scope || this, pre.concat(args)); 
        }; 
    },

    /**
     * Return <code>true</code> if <code>props</code> defines a fragment.
     * A fragment may be a string or an object containing known properties.
     * See the _isFragmentObject and _isFragmentWidget functions.
     * 
     * @param {Object} props properties that may define a fragment.
     * @return {boolean} true of <code>props</code> is a fragment
     * else false. If <code>props</code> is null, false is returned.
     * @private
     */
    _isFragment: function(props) {
        var result = false;
        if (props && (typeof props == "string"
                || woodstock4_3.widget.common._isFragmentObject(props)
                || woodstock4_3.widget.common._isFragmentWidget(props))) {
            result = true;
        }
        return result;
    },

    /**
     * Return <code>true</code> if <code>props</code> defines a fragment object.
     * A fragment object defines either an html, include, or refId property and
     * is not a fragment widget -- see _isFragmentWidget.
     * 
     * @param {Object} props properties that may define a fragment object.
     * @return {boolean} true of <code>props</code> is a fragment object
     * else false. If <code>props</code> is null, false is returned.
     * @private
     */
    _isFragmentObject: function(props) {
        var result = false;
        if (props && !woodstock4_3.widget.common._isFragmentWidget(props)
                && (typeof props.html == "string"
                || typeof props.refId == "string"
                || typeof props.include == "string")) {
            result = true;
        }
        return result;
    },

    /**
     * Return <code>true</code> if <code>props</code> defines a fragment widget.
     * A fragment widget is an object which defines <code>widgetType</code> and
     * <code>id</code> properties.  
     * 
     * @param {Object} props properties that may define a fragment widget.
     * @return {boolean} true of <code>props</code> is a fragment widget
     * else false. If <code>props</code> is null, false is returned.
     * @private
     */
    _isFragmentWidget: function(props) {
        var result = false;
        if (props && (props.id && props.id != "")
                && (props.widgetType && props.widgetType != "")) {
            result = true;
        }
        return result;
    },

    /**
     * This function tests for high contrast mode.
     *  
     * @return {boolean} true if high contrast mode.
     * @private
     */
    _isHighContrastMode:  function() {
        var config = woodstock4_3._base.config;
        if (config._isHighContrastMode != undefined) {
            return config._isHighContrastMode;
        }

        // Dojo appends the following div tag in body tag for a11y support.
        //
        // <div style="border-style: solid; border-color: red green; border-width: 1px; 
        //  position: absolute; left: -999px; top: -999px; background-image: 
        //  url(.../blank.gif);" id="a11yTestNode">
        // </div>

        // Currently high contrast mode check is supported for firefox and ie on
        // windows. High contrast mode is not supported for Safari.
        if (woodstock4_3._base.browser._isSafari()) {
            return false;            
        }
        // Get icon properties.
        var props = woodstock4_3.theme.common.getImage("DOT");
        if (props == null) {
            return false;
        }

        // Create div for testing if high contrast mode is on or images are 
        // turned off.
        var domNode = document.createElement("div"); 
        domNode.style.cssText = 'border: 1px solid;' +
            'border-color:red green;' +
            'position: absolute;' +
            'height: 5px;' +
            'top: -999px;' +
            'background-image: url("' + props.src + '");';

        var body = woodstock4_3._dojo.body();
        body.appendChild(domNode);

        // Detect the high contrast mode.
        var bImg = null;
        if (window.getComputedStyle) {
            var styleValue = getComputedStyle(domNode, "");
            bImg = styleValue.getPropertyValue("background-image");
        } else {
            bImg = domNode.currentStyle.backgroundImage;
        }
        if (bImg != null && (bImg == "none" || bImg == "url(invalid-url:)" )) {
            config._isHighContrastMode = true; // High Contrast Mode
        } else {
            config._isHighContrastMode = false;
        }

        // IE throws security exception if domNode isn't removed.
        // This allows widgets to be created before the window.onLoad event.
        body.removeChild(domNode);
        return config._isHighContrastMode;    
    },

    /** 
     * Keyboard key code as stored in browser events 
     *
     * @private
     */
    _keyCodes: woodstock4_3._dojo.keys,
 
     /**
     * This function is called after the DOM has finished loading and widgets
     * declared in markup have been instantiated.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _loaded: function() {
        // After the page has been parsed, there is no need to perform this task
        // again. Setting the parseOnLoad flag to false will allow the ajaxZone
        // tag of JSF Extensions, for example, to re-render widgets properly. 
        // That is, considering there will only ever be one window.onLoad event.
        woodstock4_3._base.config.parseOnLoad = false;

        // Call loaders set via private addOnLoad function.
        var common = woodstock4_3.widget.common;
        for(var i = 0; i < common._preLoaders.length; i++){
            common._preLoaders[i]();
        }
        common._preLoaders = []; // Clean up.

        // Call loaders set via public addOnLoad function.
        for(var i = 0; i < common._loaders.length; i++){
            common._loaders[i]();
        }
        common._loaders = []; // Clean up.
        return true;
    },

    /**
     * This function is used to parse HTML markup in order to create widgets
     * more efficiently. See the createWidget() function.
     *
     * @param {Node} domNode The DOM node containing HTML elements to replace.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _parseMarkup: function(domNode) {
        if (domNode == null) {
            return false;
        }
        // Note: Using document.getElementById() results in poor perfromance. 
        var nodes = domNode.getElementsByTagName("script");
        var common = woodstock4_3.widget.common;

        // If dealing with JSF facet fragments, we must search for span 
        // elements because IE removes HTML script tags from strings.
        if (nodes.length == 0) {
            nodes = domNode.getElementsByTagName("span");
            if (nodes.length == 0) {
                return false;
            }
            // Match span id with props.
            for (var i = 0; i < nodes.length; i++) {
                common._createMarkupWidget(nodes[i]);
            }
        } else {
            // Match script parent id with props.
            for (var i = 0; i < nodes.length; i++) {
                common._createMarkupWidget(nodes[i].parentNode);
            }
        }
        return true;
    },

    /**
     * Publish an event topic.
     *
     * @param {String} topic The event topic to publish.
     * @param {Object} props Key-Value pairs of properties. This will be applied
     * to each topic subscriber.
     * @return {boolean} true if successful; otherwise, false.
     */
    publish: function(topic, props) {
        // Publish an event for custom AJAX implementations to listen for.
        woodstock4_3._dojo.publish(topic, props);
        return true;
    },

    /**
     * This function is used to remove child nodes from given DOM node.
     * <p>
     * Note: Child nodes may be cleared using the innerHTML property. However,
     * IE fails when this property is set via the widget's fillInTemplate 
     * function. In this case, DOM nodes shall be removed manually using the 
     * Node APIs.
     * </p>
     * @param {Node} domNode The DOM node to remove child nodes.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _removeChildNodes: function(domNode) {
        if (domNode == null) {
            return false;
        }
        try {
            domNode.innerHTML = ""; // Cannot be null on IE.
        } catch (e) {
            // Iterate over child nodes.
            while (domNode.hasChildNodes()) {
                var node = domNode.childNodes[0];
                domNode.removeChild(node);
            }
        }
        return true;
    },

    /**
     * This function sleeps for specified milli seconds.
     * 
     * @param {int} delay The amount to delay.
     * @return {boolean} true if current time is greater than the exit time.
     * @private
     */
    _sleep: function(delay) {
        var start = new Date();
        var exitTime = start.getTime() + delay;

        while (true) {
            start = new Date();
            if (start.getTime() > exitTime) {
                return true;
            }
        }
        return false;
    },

    /**
     * Subscribe to an event topic.
     *
     * @param {String} topic The event topic to subscribe to.
     * @param {Object} obj The object in which a function will be invoked, or
     * null for default scope.
     * @param {String|Function} func The name of a function in context, or a 
     * function reference to invoke when topic is published. 
     * @return {boolean} true if successful; otherwise, false.
     */
    subscribe: function(topic, obj, func) {
        woodstock4_3._dojo.subscribe(topic, obj, func);
        return true;
    },

    /**
     * This function is used to update a widget, HTML fragment, or static
     * string for the given domNode.
     * <p>
     * Note: If the widget associated with props.id already exists, the widget's 
     * setProps() function is invoked with the given props param. If the widget 
     * does not exist, the widget object is instantiated via the _addFragment()
     * function -- all params are passed through.
     * </p><p>
     * See woodstock4_3.widget.label._setProps for example.
     * </p>
     * @param {Node} domNode The DOM node used to add widget (default).
     * @param {String} id The id of the widget to update, if available.
     * @param {Object} props Key-Value pairs of properties.
     * @param {String} position The position (e.g., "first", "last", etc.) to add widget.
     * @param {boolean} escape HTML escape static strings -- default is true.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _updateFragment: function(domNode, id, props, position, escape) {
        if (props == null) {
            return false;
        }
        // Ensure props is not a string.
        var common = woodstock4_3.widget.common;
        var widget = (typeof props != 'string') 
            ? common.getWidget(id) : null;

        // Update widget or add fragment.
        if (widget && typeof widget.setProps == "function") {
            widget.setProps(props);
        } else {
            common._addFragment(domNode, props, position, escape);
        }
        return true;
    }
};

// Initialize widgets.
woodstock4_3._dojo.addOnLoad(function() {
    var common = woodstock4_3.widget.common;

    // Defer widget creation until the window.onLoad event.
    if (new Boolean(woodstock4_3._base.config.parseOnLoad).valueOf() == true) {
        common._parseMarkup(woodstock4_3._dojo.body());
    } else {
        common._loaded(); // Widgets created in-line.
    }
});

/** 
 * @class This class contains functions for table components.
 * <p>
 * Once the table is rendered, you will be able to invoke functions directly on 
 * the HTML element. For example:
 * </p><p><pre>
 * var table = document.getElementById("form1:table1");
 * var count = table.getAllSelectedRowsCount();
 * </pre></p><p>
 * Note: It is assumed that formElements.js has been included in the page. In
 * addition, all given HTML element IDs are assumed to be the outter most tag
 * enclosing the component.
 * </p>
 * @static
 * @private
 */
woodstock4_3._html.table = {
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // DOM functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * This function is used to confirm the number of selected components (i.e., 
     * checkboxes or radiobuttons used to de/select rows of the table), affected
     * by a delete action. This functionality requires the selectId property of
     * the tableColumn component and hiddenSelectedRows property of the
     * tableRowGroup component to be set.
     * <p>
     * If selections are hidden from view, the confirmation message indicates the
     * number of selections not displayed in addition to the total number of
     * selections. If selections are not hidden, the confirmation message indicates
     * only the total selections.
     * </p>
     *
     * @return {boolean} A value of true if the user clicks the "OK" button, or 
     * false if the user clicks the "Cancel" button.
     */
    confirmDeleteSelectedRows: function() {
        return this.confirmSelectedRows(this.deleteSelectionsMsg);
    },

    /**
     * This function is used to confirm the number of selected components (i.e., 
     * checkboxes or radiobuttons used to de/select rows of the table), affected by
     * an action such as edit, archive, etc. This functionality requires the 
     * selectId property of the tableColumn component and hiddenSelectedRows
     * property of the tableRowGroup component to be set.
     * <p>
     * If selections are hidden from view, the confirmation message indicates the
     * number of selections not displayed in addition to the total number of
     * selections. If selections are not hidden, the confirmation message indicates
     * only the total selections.
     * </p>
     *
     * @param {String} message The confirmation message (e.g., Archive all selections?).
     * @return {boolean} A value of true if the user clicks the "OK" button, or 
     * false if the user clicks the "Cancel" button.
     */
    confirmSelectedRows: function(message) {
        // Get total selections message.
        var totalSelections = this.getAllSelectedRowsCount();
        var totalSelectionsArray = this.totalSelectionsMsg.split("{0}");
        var totalSelectionsMsg = totalSelectionsArray[0] + totalSelections;

        // Append hidden selections message.
        var hiddenSelections = this.getAllHiddenSelectedRowsCount();
        if (hiddenSelections > 0) {
            // Get hidden selections message.
            var hiddenSelectionsArray = this.hiddenSelectionsMsg.split("{0}");
            var hiddenSelectionsMsg = hiddenSelectionsArray[0] + hiddenSelections;

            totalSelectionsMsg = hiddenSelectionsMsg + totalSelectionsMsg;
        }
        return (message != null)
            ? confirm(totalSelectionsMsg + message)
            : confirm(totalSelectionsMsg);
    },

    /**
     * This function is used to toggle the filter panel from the filter menu. This
     * functionality requires the filterId of the table component to be set. In 
     * addition, the selected value must be set as well to restore the default
     * selected value when the embedded filter panel is closed.
     * <p>
     * If the "Custom Filter" option has been selected, the table filter panel is 
     * toggled. In this scenario, false is returned indicating the onChange event,
     * generated by the table filter menu, should not be allowed to continue.
     * </p><p>
     * If the "Custom Filter Applied" option has been selected, no action is taken.
     * Instead, the is selected filter menu is reverted back to the "Custom Filter" 
     * selection. In this scenario, false is also returned indicating the onChange 
     * event, generated by the table filter menu, should not be allowed to continue.
     * </p><p>
     * For all other selections, true is returned indicating the onChange event, 
     * generated by the table filter menu, should be allowed to continue.
     * </p>
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    filterMenuChanged: function() {
        // Validate panel IDs.
        if (this.panelToggleIds == null || this.panelToggleIds.length == 0) {
            return false;
        }

        // Get filter menu.
        var menu = this._getSelectElement(this.panelToggleIds[this.FILTER]);
        if (menu == null) {
            return true;
        }

        // Test if table filter panel should be opened.
        if (menu.options[menu.selectedIndex].value == this.customFilterOptionValue) {
            this.toggleFilterPanel();
            return false;
        } else if (menu.options[menu.selectedIndex].
                value == this.customFilterAppliedOptionValue) {
            // Set selected option.
            menu.selectedIndex = 0;
            for (var i = 0; i < menu.options.length; i++) {
                if (menu.options[i].value == this.customFilterOptionValue) {
                    menu.options[i].selected = true;
                    break;
                }
            }
            return false;
        }
        return true;
    },

    /**
     * This function is used to get the number of selected components in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select 
     * rows of the table). This functionality requires the selectId property of the
     * tableColumn component and hiddenSelectedRows property of the table
     * component to be set.
     *
     * @return {int} The number of components selected in the current page.
     */
    getAllSelectedRowsCount: function() {
        return this.getAllHiddenSelectedRowsCount() +
            this.getAllRenderedSelectedRowsCount();
    },

    /**
     * This function is used to get the number of selected components, in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table), currently hidden from view. This functionality requires 
     * the selectId property of the tableColumn component and hiddenSelectedRows
     * property of the table component to be set.
     *
     * @return {int} The number of selected components hidden from view.
     */
    getAllHiddenSelectedRowsCount: function() {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return count;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            count = count + this._getGroupHiddenSelectedRowsCount(this.groupIds[i]);
        }
        return count;
    },

    /**
     * This function is used to get the number of selected components, in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table), currently rendered. This functionality requires 
     * the selectId property of the tableColumn component to be set.
     *
     * @return {int} The number of selected components hidden from view.
     */
    getAllRenderedSelectedRowsCount: function() {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return count;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            count = count + this._getGroupRenderedSelectedRowsCount(this.groupIds[i]);
        }
        return count;
    },

    /**
     * This function is used to initialize all rows displayed in the table when the
     * state of selected components change (i.e., checkboxes or radiobuttons used to
     * de/select rows of the table). This functionality requires the selectId
     * property of the tableColumn component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    initAllRows: function() {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return false;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            this._initGroupRows(this.groupIds[i]);
        }
        return true;
    },

    /**
     * This function is used to toggle the filter panel open or closed. This
     * functionality requires the filterId of the table component to be set. In 
     * addition, the selected value must be set as well to restore the default
     * selected value when the embedded filter panel is closed.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    toggleFilterPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Toggle filter panel.
        this._togglePanel(this.panelIds[this.FILTER],
        this.panelFocusIds[this.FILTER], this.panelToggleIds[this.FILTER]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    /**
     * This function is used to toggle the preferences panel open or closed. This
     * functionality requires the filterId of the table component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    togglePreferencesPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Toggle preferences panel.
        this._togglePanel(this.panelIds[this.PREFERENCES],
        this.panelFocusIds[this.PREFERENCES], this.panelToggleIds[this.PREFERENCES]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Internal functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element ID for the component.
     *
     * // Panel Properties
     * @config {Array} panelIds An array of embedded panel IDs.</li>
     * @config {Array} panelFocusIds An array of IDs used to set focus for open panels.</li>
     * @config {Array} panelToggleIds An array of IDs used to toggle embedded panels.</li>
     * @config {Array} panelToggleIconsOpen An array of toggle icons for open panels.</li>
     * @config {Array} panelToggleIconsClose An array of toggle icons for closed panels.</li>
     *
     * // Filter Properties
     * @config {String} basicFilterStyleClass The style class for basic or no filters.</li>
     * @config {String} customFilterStyleClass The style class for custom filters.</li>
     * @config {String} customFilterOptionValue The custom filter menu option value.</li>
     * @config {String} customFilterAppliedOptionValue The custom filter applied menu option value.</li>
     *
     * // Sort Panel Properties
     * @config {Array} sortColumnMenuIds An array of HTML element IDs for sort column menu components.</li>
     * @config {Array} sortOrderMenuIds An array of HTML element IDs for sort order menu components.</li>
     * @config {Array} sortOrderToolTips An array of tool tips used for sort order menus.</li>
     * @config {Array} sortOrderToolTipsAscending An array of ascending tool tips used for sort order menus.</li>
     * @config {Array} sortOrderToolTipsDescending An array of descending tool tips used for sort order menus.</li>
     * @config {String} duplicateSelectionMsg The message displayed for duplicate menu selections.</li>
     * @config {String} missingSelectionMsg The message displayed for missing menu selections.</li>
     * @config {String} selectSortMenuOptionValue The sort menu option value for the select column.</li>
     * @config {boolean} hiddenSelectedRows Flag indicating that selected rows might be currently hidden from view.</li>
     * @config {boolean} paginated Flag indicating table is in pagination mode.</li>
     *
     * // Group Properties
     * @config {String} selectRowStylClass The style class for selected rows.</li>
     * @config {Array} selectIds An arrary of component IDs used to select rows of the table.</li>
     * @config {Array} groupIds An array of TableRowGroup IDs rendered for the table.</li>
     * @config {Array} rowIds An array of row IDs for rendered for each TableRowGroup.</li>
     * @config {Array} hiddenSelectedRowCounts An array of selected row counts hidden from view.</li>
     * @config {String} hiddenSelectionsMsg The hidden selections message for confirm dialog.</li>
     * @config {String} totalSelectionsMsg The total selections message for confirm dialog.</li>
     * @config {String} deleteSelectionsMsg The delete selections message for confirm dialog.</li>
     *
     * // Group Panel Properties
     * @param {String} columnFooterId ID for column footer.</li>
     * @param {String} columnHeaderId ID for column header.</li>
     * @param {String} tableColumnFooterId ID for table column footer.</li>
     * @param {String} groupFooterId ID for group footer.</li>
     * @param {String} groupPanelToggleButtonId ID for group panel toggle button.</li>
     * @param {String} groupPanelToggleButtonToolTipOpen tool tip for open row group.</li>
     * @param {String} groupPanelToggleButtonToolTipClose tool tip for closed row group.</li>
     * @param {String} groupPanelToggleIconOpen The toggle icon for open row group.</li>
     * @param {String} groupPanelToggleIconClose The toggle icon for closed row group.</li>
     * @param {String} warningIconId ID for warning icon.</li>
     * @param {String} warningIconOpen The warning icon for open row group.</li>
     * @param {String} warningIconClosed The warning icon for closed row group.</li>
     * @param {String} warningIconToolTipOpen The warning icon tool tip for open row group.</li>
     * @param {String} warningIconToolTipClose The warning icon tool tip for closed row group.</li>
     * @param {String} collapsedHiddenFieldId ID for collapsed hidden field.</li>
     * @param {String} selectMultipleToggleButtonId ID for select multiple toggle button.</li>
     * @param {String} selectMultipleToggleButtonToolTip The select multiple toggle button tool tip.</li>
     * @param {String} selectMultipleToggleButtonToolTipSelected The select multiple toggle button tool tip when selected.</li>
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize table.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

        // Misc properties.
        domNode.SEPARATOR = ":";   // NamingContainer separator.

        // Panel toggle keys.
        domNode.SORT        = 0;
        domNode.PREFERENCES = 1;
        domNode.FILTER      = 2;

        // Sort keys.
        domNode.PRIMARY   = 0;
        domNode.SECONDARY = 1;
        domNode.TERTIARY  = 2;

        // Replace extra backslashes, JSON escapes new lines (e.g., \\n).
	domNode.hiddenSelectionsMsg = (props.hiddenSelectionsMsg != null)
            ? props.hiddenSelectionsMsg.replace(/\\n/g, "\n") : "";
        domNode.totalSelectionsMsg = (props.totalSelectionsMsg != null)
            ? props.totalSelectionsMsg.replace(/\\n/g, "\n") : "";
	domNode.missingSelectionMsg = (props.missingSelectionMsg != null)
            ? props.missingSelectionMsg.replace(/\\n/g, "\n") : "";
	domNode.duplicateSelectionMsg = (props.duplicateSelectionMsg != null)
            ? props.duplicateSelectionMsg.replace(/\\n/g, "\n") : "";
        domNode.deleteSelectionsMsg = (props.deleteSelectionsMsg != null)
            ? props.deleteSelectionsMsg.replace(/\\n/g, "\n") : "";

        // Private functions.
        domNode._getGroupSelectedRowsCount = woodstock4_3._html.table._getGroupSelectedRowsCount;
        domNode._getGroupHiddenSelectedRowsCount = woodstock4_3._html.table._getGroupHiddenSelectedRowsCount;
        domNode._getGroupRenderedSelectedRowsCount = woodstock4_3._html.table._getGroupRenderedSelectedRowsCount;
        domNode._getSelectElement = woodstock4_3._html.table._getSelectElement;
        domNode._initGroupRows = woodstock4_3._html.table._initGroupRows;
        domNode._initPrimarySortOrderMenu = woodstock4_3._html.table._initPrimarySortOrderMenu;
        domNode._initPrimarySortOrderMenuToolTip = woodstock4_3._html.table._initPrimarySortOrderMenuToolTip;
        domNode._initSecondarySortOrderMenu = woodstock4_3._html.table._initSecondarySortOrderMenu;
        domNode._initSecondarySortOrderMenuToolTip = woodstock4_3._html.table._initSecondarySortOrderMenuToolTip;
        domNode._initSortColumnMenus = woodstock4_3._html.table._initSortColumnMenus;
        domNode._initSortOrderMenu = woodstock4_3._html.table._initSortOrderMenu;
        domNode._initSortOrderMenus = woodstock4_3._html.table._initSortOrderMenus;
        domNode._initSortOrderMenuToolTip = woodstock4_3._html.table._initSortOrderMenuToolTip;
        domNode._initTertiarySortOrderMenu = woodstock4_3._html.table._initTertiarySortOrderMenu;
        domNode._initTertiarySortOrderMenuToolTip = woodstock4_3._html.table._initTertiarySortOrderMenuToolTip;
        domNode._menuChanged = woodstock4_3._html.table._menuChanged;
        domNode._resetFilterMenu = woodstock4_3._html.table._resetFilterMenu;
        domNode._selectAllRows = woodstock4_3._html.table._selectAllRows;
        domNode._selectGroupRows = woodstock4_3._html.table._selectGroupRows;
        domNode._toggleGroupPanel = woodstock4_3._html.table._toggleGroupPanel;
        domNode._togglePanel = woodstock4_3._html.table._togglePanel;
        domNode._toggleSortPanel = woodstock4_3._html.table._toggleSortPanel;
        domNode._validateSortPanel = woodstock4_3._html.table._validateSortPanel;

        // Public functions.
        domNode.confirmDeleteSelectedRows = woodstock4_3._html.table.confirmDeleteSelectedRows;
        domNode.confirmSelectedRows = woodstock4_3._html.table.confirmSelectedRows;
        domNode.filterMenuChanged = woodstock4_3._html.table.filterMenuChanged;
        domNode.getAllSelectedRowsCount = woodstock4_3._html.table.getAllSelectedRowsCount;
        domNode.getAllHiddenSelectedRowsCount = woodstock4_3._html.table.getAllHiddenSelectedRowsCount;
        domNode.getAllRenderedSelectedRowsCount = woodstock4_3._html.table.getAllRenderedSelectedRowsCount;
        domNode.initAllRows = woodstock4_3._html.table.initAllRows;
        domNode.toggleFilterPanel = woodstock4_3._html.table.toggleFilterPanel;
        domNode.togglePreferencesPanel = woodstock4_3._html.table.togglePreferencesPanel;

        return true;
    },

    /**
     * This function is used to get the number of selected components for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table).
     * This functionality requires the selectId property of the tableColumn component
     * and the hiddenSelectedRows property of the table component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of components selected in the current page.
     * @private
     */
    _getGroupSelectedRowsCount: function(groupId) {
        return this._getGroupHiddenSelectedRowsCount(groupId) + 
            this._getGroupRenderedSelectedRowsCount(groupId);
    },

    /**
     * This function is used to get the number of selected components, for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table),
     * currently hidden from view. This functionality requires the selectId property
     * of the tableColumn component and hiddenSelectedRows property of the table
     * component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of selected components hidden from view.
     * @private
     */
    _getGroupHiddenSelectedRowsCount: function(groupId) {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return count;
        }

        // Find the given group Id in the groupIds array.
        for (var i = 0; i < this.groupIds.length; i++) {
            // Get selectId and rowIds array associated with groupId.
            if (groupId == this.groupIds[i]) {
                count = this.hiddenSelectedRowCounts[i];
                break;
            }
        }
        return count;
    },

    /**
     * This function is used to get the number of selected components, for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table),
     * currently rendered. This functionality requires the selectId property of the
     * tableColumn component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of components selected in the current page.
     * @private
     */
    _getGroupRenderedSelectedRowsCount: function(groupId) {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return count;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + 
                this.SEPARATOR + selectId);
            if (select == null) {
                continue;
            }
            var props = select.getProps();
            if (props.disabled != true && props.checked == true) {
                count++;
            }
        }
        return count;
    },

    /**
     * Use this function to access the HTML select element that makes up
     * the dropDown.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the dropDown).
     * @return {Node} a reference to the select element.
     */
    _getSelectElement: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * This function is used to initialize rows for the given group when the state
     * of selected components change (i.e., checkboxes or radiobuttons used to
     * de/select rows of the table). This functionality requires the selectId
     * property of the tableColumn component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initGroupRows: function(groupId) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        var checked = true; // Checked state of multiple select toggle button.
        var selected = false; // At least one component is selected.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + 
                this.SEPARATOR + selectId);
            if (select == null) {
                continue;
            }

            // Set row style class.
            var row = document.getElementById(this.groupIds[i] + 
                this.SEPARATOR + rowIds[k]);
            var props = select.getProps();
            if (props.disabled == true) {
                // Don't highlight row or set checked/selected properties.
                woodstock4_3._base.common._stripStyleClass(row, 
                    this.selectRowStyleClass);
            } else if (props.checked == true) {
                woodstock4_3._base.common._addStyleClass(row, 
                    this.selectRowStyleClass);
                selected = true;
            } else {
                woodstock4_3._base.common._stripStyleClass(row, 
                    this.selectRowStyleClass);
                checked = false;
            }
        }

        // Set multiple select toggle button state.
        var title;
        var checkbox = document.getElementById(groupId + this.SEPARATOR + 
            this.selectMultipleToggleButtonId);
        if (checkbox != null) {
            title = (checked) 
                ? this.selectMultipleToggleButtonToolTipSelected
                : this.selectMultipleToggleButtonToolTip;
            checkbox.setProps({
                "checked": checked,
                "title": title
            });
        }

        // Get flag indicating groupis collapsed.
        var prefix = groupId + this.SEPARATOR;
        var collapsed = !woodstock4_3._base.common._isVisible(prefix + rowIds[0]);

        // Set next warning image.
        var image = document.getElementById(prefix + this.warningIconId);
        if (image != null) {
            var src = this.warningIconOpen;
            title = this.warningIconToolTipOpen;

            // Don't show icon when multiple select is checked.
            if (collapsed && selected && !checked) {
                src = this.warningIconClose;
                title = this.warningIconToolTipClose;
            }
            image.setProps({
               "src": src,
               "title": title
            });
        }
        return true;
    },

    /**
     * This function is used to initialize the primary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initPrimarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.PRIMARY], 
            this.sortOrderMenuIds[this.PRIMARY]);
    },  

    /**
     * This function is used to initialize the primary sort order menu tool tips 
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initPrimarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(
            this.sortOrderMenuIds[this.PRIMARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.PRIMARY], 
            this.sortOrderMenuIds[this.PRIMARY]);
    },

    /**
     * This function is used to initialize the secondary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSecondarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.SECONDARY], 
            this.sortOrderMenuIds[this.SECONDARY]);
    },

    /**
     * This function is used to initialize the secondary sort order menu tool tips
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSecondarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(
            this.sortOrderMenuIds[this.SECONDARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.SECONDARY], 
            this.sortOrderMenuIds[this.SECONDARY]);
    },

    /**
     * This function is used to initialize the primary, secondary, and tertiary 
     * sort column menus used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortColumnMenus: function() {
        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Set initial selected option for all sort column menus.
        for (var i = 0; i < this.sortColumnMenuIds.length; i++) {
            // Get sort column menu.
            var sortColumnMenu = this._getSelectElement(this.sortColumnMenuIds[i]);
            if (sortColumnMenu == null) {
                continue;
            }

            // Set default selected value.
            sortColumnMenu.selectedIndex = 0;

            // Set selected option.
            for (var k = 0; k < sortColumnMenu.options.length; k++) {
                if (sortColumnMenu.options[k].defaultSelected == true) {
                    sortColumnMenu.options[k].selected = true;
                    break;
                }
            }
            // Ensure hidden filed values are updated.
            this._menuChanged(this.sortColumnMenuIds[i]);
        }
        return true;
    },

    /**
     * This function is used to initialize the primary, secondary, and tertiary 
     * sort order menus used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenus: function() {
        // Validate sort order menu IDs.
        if (this.sortOrderMenuIds == null || this.sortOrderMenuIds.length == 0) {
            return false;
        }

        // Set initial selected option for all sort column menus.
        for (var i = 0; i < this.sortOrderMenuIds.length; i++) {
            // Get sort order menu.
            var sortOrderMenu = this._getSelectElement(this.sortOrderMenuIds[i]);
            if (sortOrderMenu == null) {
                continue;
            }

            // Set default selected value.
            sortOrderMenu.selectedIndex = 0;

            // Get sort column menu.
            var sortColumnMenu = this._getSelectElement(this.sortColumnMenuIds[i]);
            if (sortColumnMenu != null) {
                // If the table is paginated and there are no hidden selected rows, the select
                // column cannot be sorted descending.
                if (sortColumnMenu.options[sortColumnMenu.selectedIndex].
                        value == this.selectSortMenuOptionValue
                        && !this.hiddenSelectedRows && this.paginated) {
                    sortOrderMenu.options[1].disabled = true;
                } else {
                    sortOrderMenu.options[1].disabled = false;
                }
            }

            // Set selected option.
            for (var k = 0; k < sortOrderMenu.options.length; k++) {
                if (sortOrderMenu.options[k].defaultSelected == true) {
                    sortOrderMenu.options[k].selected = true;
                    break;
                }
            }
            // Ensure hidden filed values and styles are updated.
            this._menuChanged(this.sortOrderMenuIds[i]);

            // Initialize tool tip.
            this._initSortOrderMenuToolTip(this.sortColumnMenuIds[i], this.sortOrderMenuIds[i]);
        }
        return true;
    },

    /**
     * This function is used to initialize sort order menus used in the
     * sort panel. When a sort column menu changes, the given sort order 
     * menu is initialized based on the the selected value of the given
     * sort column menu.
     *
     * @param {String} sortColumnMenuId The HTML element ID for the sort column menu component.
     * @param {String} sortOrderMenuId The HTML element ID for the sort order menu component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenu: function(sortColumnMenuId, sortOrderMenuId) {
        if (sortColumnMenuId == null || sortOrderMenuId == null) {
            return false;
        }

        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Validate sort order menu IDs.
        if (this.sortOrderMenuIds == null || this.sortOrderMenuIds.length == 0) {
            return false;
        }

        // Get sort column menu.
        var sortColumnMenu = this._getSelectElement(sortColumnMenuId);
        if (sortColumnMenu == null) {
            return false;
        }

        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(sortOrderMenuId);
        if (sortOrderMenu == null) {
            return false;
        }

        // Reset selected option.
        sortOrderMenu.selectedIndex = 0; // Default ascending.

        // Get sort column menu selected index.            
        var selectedIndex = (sortColumnMenu.selectedIndex > -1)
            ? sortColumnMenu.selectedIndex : 0; // Default to first option.

        // If the table is paginated and there are no hidden selected rows, the select
        // column cannot be sorted descending.
        if (sortColumnMenu.options[selectedIndex].value == this.selectSortMenuOptionValue
                && !this.hiddenSelectedRows && this.paginated) {
            sortOrderMenu.options[1].disabled = true;
        } else {
            sortOrderMenu.options[1].disabled = false;
        }

        // Attempt to match the selected index of the given sort column menu with the
        // default selected value from either sort column menu. If a match is found, the 
        // default selected value of the associated sort order menu is retrieved. This
        // default selected value is set as the current selection of the given sort 
        // order menu.
        for (var i = 0; i < this.sortColumnMenuIds.length; i++) {
            // Get the current sort column menu to test the default selected value.
            var currentSortColumnMenu = this._getSelectElement(
                this.sortColumnMenuIds[i]);
            if (currentSortColumnMenu == null) {
                continue;
            }

            // Find default selected value for the current sort column menu.
            var defaultSelected = null;
            for (var k = 0; k < currentSortColumnMenu.options.length; k++) {
                if (currentSortColumnMenu.options[k].defaultSelected == true) {
                    defaultSelected = currentSortColumnMenu.options[k].value;
                    break;
                }
            }

            // Match default selected value with selected index value.
            if (defaultSelected != null && defaultSelected ==
                    sortColumnMenu.options[selectedIndex].value) {
                // Get current sort order menu to test the default selected value.
                var currentSortOrderMenu = this._getSelectElement(
                    this.sortOrderMenuIds[i]);
                if (currentSortOrderMenu == null) {
                    continue;
                }

                // Find default selected index for the current sort order menu.
                var defaultSelectedIndex = -1;
                for (var c = 0; c < currentSortOrderMenu.options.length; c++) {
                    if (currentSortOrderMenu.options[c].defaultSelected == true) {
                        defaultSelectedIndex = c;
                        break;
                    }
                }

                // Set selected value for given sort order menu.
                if (defaultSelectedIndex > -1) {
                    sortOrderMenu.options[defaultSelectedIndex].selected = true;
                } else {
                    sortOrderMenu.options[0].selected = true; // Default ascending.
                }
                break;
            }
        }
        // Ensure hidden field values and styles are updated.
        this._menuChanged(sortOrderMenuId);

        // Set sort order menu tool tip.
        return this._initSortOrderMenuToolTip(sortColumnMenuId, sortOrderMenuId);
    },

    /**
     * This function is used to initialize sort order menu tool tips used in the 
     * sort panel. When a sort column menu changes, the given sort order 
     * menu tool tip is initialized based on the the selected value of the given
     * sort column menu.
     *
     * @param {String} sortColumnMenuId The HTML element ID for the sort column menu component.
     * @param {String} sortOrderMenuId The HTML element ID for the sort order menu component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenuToolTip: function(sortColumnMenuId, sortOrderMenuId) {
        if (sortColumnMenuId == null || sortOrderMenuId == null) {
            return false;
        }

        // Get sort column menu.
        var sortColumnMenu = this._getSelectElement(sortColumnMenuId);
        if (sortColumnMenu == null) {
            return false;
        }

        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(sortOrderMenuId);
        if (sortOrderMenu == null) {
            return false;
        }

        // Get tool tip associated with given sort order menu.
        var toolTip = "";
        if (this.sortOrderToolTips != null && this.sortOrderToolTips.length != 0
                && this.sortOrderMenuIds != null) {
            for (var i = 0; i < this.sortOrderMenuIds.length; i++) {
                // The tool tip is at index zero, after splitting the message.
                if (sortOrderMenuId == this.sortOrderMenuIds[i]) {
                    toolTip = this.sortOrderToolTips[i].split("{0}")[0];
                    break;
                }
            }
        }

        // Get sort column menu selected index.            
        var selectedIndex = (sortColumnMenu.selectedIndex > -1)
            ? sortColumnMenu.selectedIndex : 0; // Default to first option.

        // Set tool tip.
        if (sortOrderMenu.options[sortOrderMenu.selectedIndex].value == "true") {
            sortOrderMenu.title = toolTip + this.sortOrderToolTipsDescending[selectedIndex];
        } else {
            // Default ascending.
            sortOrderMenu.title = toolTip + this.sortOrderToolTipsAscending[selectedIndex];
        }
        return true;
    },

    /**
     * This function is used to initialize the tertiary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initTertiarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.TERTIARY], 
            this.sortOrderMenuIds[this.TERTIARY]);
    },

    /**
     * This function is used to initialize the tertiary sort order menu tool tips
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initTertiarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(
            this.sortOrderMenuIds[this.TERTIARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.TERTIARY], 
            this.sortOrderMenuIds[this.TERTIARY]);
    },

    /**
     * This function is invoked by the choice onselect action to set the
     * selected, and disabled styles.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     */
    _menuChanged: function(elementId) {         
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget._changed();
        }
        return false;
    },

    /**
     * This function is used to reset filter drop down menu. This functionality 
     * requires the filterId of the table component to be set. In addition,
     * the selected value must be set as well to restore the default selected
     * value when the embedded filter panel is closed.
     *
     * @param {String} filterId The HTML element ID of the filter menu.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _resetFilterMenu: function(filterId) {
        if (filterId == null) {
            return false;
        }

        // Get filter menu.
        var menu = this._getSelectElement(filterId);
        if (menu == null) {
            return true;
        }

        // Get div element associated with the filter panel.
        var div = document.getElementById(this.panelIds[this.FILTER]);
        if (div == null) {
            return false;
        }

        // Set selected style.
        if (woodstock4_3._base.common._isVisibleElement(div)) {
            woodstock4_3._base.common._stripStyleClass(menu, this.basicFilterStyleClass);
            woodstock4_3._base.common._addStyleClass(menu, this.customFilterStyleClass);
        } else {
            // Reset default selected option.
            menu.selectedIndex = 0;
            for (var i = 0; i < menu.options.length; i++) {
                if (menu.options[i].defaultSelected == true) {
                    menu.options[i].selected = true;
                    break;
                }
            }
            woodstock4_3._base.common._stripStyleClass(menu, this.customFilterStyleClass);
            woodstock4_3._base.common._addStyleClass(menu, this.basicFilterStyleClass);
        }
        return true;
    },

    /**
     * This function is used to set the selected state components in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table). This functionality requires the selectId property of
     * the tableColumn component to be set.
     *
     * @param {boolean} selected Flag indicating components should be selected.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _selectAllRows: function(selected) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return false;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            this._selectGroupRows(this.groupIds[i], selected);
        }
        return true;
    },

    /**
     * This function is used to set the selected state components for the given row group
     * (i.e., checkboxes or radiobuttons used to de/select rows of the table). This 
     * functionality requires the selectId property of the tableColumn component to be
     * set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @param {boolean} selected Flag indicating components should be selected.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _selectGroupRows: function(groupId, selected) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + this.SEPARATOR + selectId);
            if (select == null || select.getProps().disabled == true) {
                continue;
            }
            select.setProps({checked: new Boolean(selected).valueOf()});
        }
        return this._initGroupRows(groupId); // Set row highlighting.
    },

    /**
     * This function is used to toggle row group panels open or closed.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _toggleGroupPanel: function(groupId) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get rowIds array associated with groupId.
        var rowIds = null;
        for (var c = 0; c < this.groupIds.length; c++) {
            if (groupId == this.groupIds[c]) {
                rowIds = this.rowIds[c];
                break;
            }
        }

        // If row IDs could not be found, do not continue.
        if (rowIds == null) {
            return false;
        }

        // Get flag indicating group is collapsed.
        var prefix = groupId + this.SEPARATOR;
        var collapsed = !woodstock4_3._base.common._isVisible(prefix + rowIds[0]);

        // Get the number of column headers and table column footers for all 
        // TableRowGroup children.
        var _prefix;
        var columnHeaderId;
        var tableColumnFooterId;
        var columnHeadersCount = 0;
        var tableColumnFootersCount = 0;
        for (var i = 0; i < this.groupIds.length; i++) {
            // Only need to test first nested column header/footer; thus, index 0 is used.        
            _prefix = this.groupIds[i] + this.SEPARATOR;
            columnHeaderId = _prefix + this.columnHeaderId + this.SEPARATOR + "0";
            tableColumnFooterId = _prefix + this.tableColumnFooterId + this.SEPARATOR + "0";
            if (document.getElementById(columnHeaderId) != null) {
                columnHeadersCount++;
            }
            if (document.getElementById(tableColumnFooterId) != null) {
                tableColumnFootersCount++;
            }
        }

        // Toggle nested column footer.
        var rowIndex = 0;
        var columnFooterId;
        while (true) {
            columnFooterId = prefix + this.columnFooterId + 
                this.SEPARATOR + rowIndex++;
            if (document.getElementById(columnFooterId) == null) {
                break;
            }
            woodstock4_3._base.common._setVisible(columnFooterId, collapsed);
        }

        // Toggle column header only if multiple column headers are shown.
        if (columnHeadersCount > 1) {
            rowIndex = 0;
            while (true) {
                columnHeaderId = prefix + this.columnHeaderId + 
                    this.SEPARATOR + rowIndex++;
                if (document.getElementById(columnHeaderId) == null) {
                    break;
                }            
                woodstock4_3._base.common._setVisible(columnHeaderId, collapsed);
            }
        }

        // Toggle table column footer only if multiple column footers are shown.
        if (tableColumnFootersCount > 1) {
            rowIndex = 0;
            while (true) {
                tableColumnFooterId = prefix + this.tableColumnFooterId + 
                    this.SEPARATOR + rowIndex++;
                if (document.getElementById(tableColumnFooterId) == null) {
                    break;
                }
                woodstock4_3._base.common._setVisible(tableColumnFooterId, collapsed);
            }
        }

        // Toggle group rows.
        var rowId;
        for (var k = 0; k < rowIds.length; k++) {
            rowId = prefix + rowIds[k];
            woodstock4_3._base.common._setVisible(rowId, collapsed);
        }

        // Toggle group footers.
        woodstock4_3._base.common._setVisible(prefix + this.groupFooterId, collapsed);

        // Set next toggle button image.
        var groupPanelToggleButtonId = prefix + this.groupPanelToggleButtonId;
        var imgHyperlink = document.getElementById(groupPanelToggleButtonId);
        if (imgHyperlink != null) {
            if (collapsed) {
                imgHyperlink.setProps({
                    enabledImage: {
                        src: this.groupPanelToggleIconOpen  
                    },
                    title: this.groupPanelToggleButtonToolTipOpen
                });
            } else {
                imgHyperlink.setProps({
                    enabledImage: {
                        src: this.groupPanelToggleIconClose
                    },
                    title: this.groupPanelToggleButtonToolTipClose
                });
            }
        }

        // Set collapsed hidden field.
        var hiddenField = document.getElementById(prefix + this.collapsedHiddenFieldId);
        if (hiddenField != null) {
            hiddenField.value = !collapsed;
        }
        return this._initGroupRows(groupId); // Set next warning image.
    },

    /**
     * This function is used to toggle embedded panels.
     *
     * @param {String} panelId The panel ID to toggle.
     * @param {String} panelFocusIdOpen The ID used to set focus when panel is opened.
     * @param {String} panelFocusIdClose The ID used to set focus when panel is closed.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _togglePanel: function(panelId, panelFocusIdOpen, panelFocusIdClose) {
        if (panelId == null) {
            return false;
        }

        // Toggle the given panel, hide all others.
        for (var i = 0; i < this.panelIds.length; i++) {
            // Get div element associated with the panel.
            var div = document.getElementById(this.panelIds[i]);
            if (div == null) {
                continue;
            }

            // Set display value. Alternatively, we could set div.style.display
            // equal to "none" or "block" (i.e., hide/show).
            if (this.panelIds[i] == panelId) {
                // Set focus when panel is toggled -- bugtraq 6316565.
                var focusElement = null;

                if (woodstock4_3._base.common._isVisibleElement(div)) {
                    woodstock4_3._base.common._setVisibleElement(div, false); // Hide panel.
                    focusElement = document.getElementById(panelFocusIdClose);
                } else {
                    woodstock4_3._base.common._setVisibleElement(div, true); // Show panel.
                    focusElement = document.getElementById(panelFocusIdOpen);
                }

                // Set focus.
                if (focusElement != null) {
                    focusElement.focus();
                }
            } else {
                // Panels are hidden by default.
                woodstock4_3._base.common._setVisibleElement(div, false);
            }

            // Get image from icon hyperlink component.
            var imgHyperlink = document.getElementById(this.panelToggleIds[i]);
            if (imgHyperlink == null || imgHyperlink.getProps().enabledImage == null) {
                continue; // Filter panel facet may be in use.
            }

            // Set image.
            if (woodstock4_3._base.common._isVisibleElement(div)) {
                imgHyperlink.setProps({
                    enabledImage: {
                        src: this.panelToggleIconsOpen[i]
                    }
                });
            } else {
                imgHyperlink.setProps({
                    enabledImage: {
                        src: this.panelToggleIconsClose[i]
                    }
                });
            }
        }
        return true;
    },

    /**
     * This function is used to toggle the sort panel open or closed. This
     * functionality requires the filterId of the table component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _toggleSortPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Initialize sort column and order menus.
        this._initSortColumnMenus(); 
        this._initSortOrderMenus();

        // Toggle sort panel.
        this._togglePanel(this.panelIds[this.SORT], 
        this.panelFocusIds[this.SORT], this.panelToggleIds[this.SORT]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    /**
     * This function is used to validate sort column menu selections 
     * for the sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _validateSortPanel: function() {
        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Get sort column menus.
        var primarySortColumnMenu = this._getSelectElement(
            this.sortColumnMenuIds[this.PRIMARY]);
        var secondarySortColumnMenu = this._getSelectElement(
            this.sortColumnMenuIds[this.SECONDARY]);
        var tertiarySortColumnMenu = this._getSelectElement(
            this.sortColumnMenuIds[this.TERTIARY]);

        // Test primary and secondary menu selections.
        if (primarySortColumnMenu != null && secondarySortColumnMenu != null) {
            // Test if secondary sort is set, but primary is not.
            if (primarySortColumnMenu.selectedIndex < 1 
                    && secondarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (primarySortColumnMenu.selectedIndex > 0
                  && primarySortColumnMenu.selectedIndex == secondarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }

        // Test primary and tertiary menu selections.
        if (primarySortColumnMenu != null && tertiarySortColumnMenu != null) {
            // Test if tertiary sort is set, but primary is not.
            if (primarySortColumnMenu.selectedIndex < 1 
                    && tertiarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (primarySortColumnMenu.selectedIndex > 0
                    && primarySortColumnMenu.selectedIndex == tertiarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }

        // Test secondary and tertiary menu selections.
        if (secondarySortColumnMenu != null && tertiarySortColumnMenu != null) {
            // Test if tertiary sort is set, but secondary is not.
            if (secondarySortColumnMenu.selectedIndex < 1 
                    && tertiarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (secondarySortColumnMenu.selectedIndex > 0
                    && secondarySortColumnMenu.selectedIndex == tertiarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }
        return true;
    }
};


woodstock4_3._dojo.provide("woodstock4_3.formElements");



woodstock4_3._dojo.provide("woodstock4_3._html.button");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for button components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.button
 * @private
 */
woodstock4_3._html.button = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated See woodstock4_3.widget.button
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize button.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var widget = woodstock4_3.widget.common.getWidget(props.id);
        if (widget == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set functions
        widget._domNode.isSecondary = woodstock4_3._html.button.isSecondary;
        widget._domNode.setSecondary = woodstock4_3._html.button.setSecondary;
        widget._domNode.isPrimary = woodstock4_3._html.button.isPrimary;
        widget._domNode.setPrimary = woodstock4_3._html.button.setPrimary;
        widget._domNode.isMini = woodstock4_3._html.button.isMini;
        widget._domNode.setMini = woodstock4_3._html.button.setMini;
        widget._domNode.getDisabled = woodstock4_3._html.button.getDisabled;
        widget._domNode.setDisabled = woodstock4_3._html.button.setDisabled;
        widget._domNode.getVisible = woodstock4_3._html.button.getVisible;
        widget._domNode.setVisible = woodstock4_3._html.button.setVisible;
        widget._domNode.getText = woodstock4_3._html.button.getText;
        widget._domNode.setText = woodstock4_3._html.button.setText;
        widget._domNode.doClick = woodstock4_3._html.button.click;

        return true;
    },

    /**
     * Simulate a mouse click in a button. 
     *
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).click();
     */
    click: function() {
        return this.click();
    },

    /**
     * Get the textual label of a button. 
     *
     * @return {String} The element value.
     * @deprecated Use document.getElementById(id).getProps().value;
     */
    getText: function() {
        return this.getProps().value;
    },

    /**
     * Set the textual label of a button. 
     *
     * @param {String} text The element value
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({value: "text"});
     */
    setText: function(text) {
        return this.setProps({value: text});
    },

    /**
     * Use this function to show or hide a button. 
     *
     * @param {boolean} show true to show the element, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({visible: boolean});
     */
    setVisible: function(show) {
        if (show == null) {
            return null;
        }
        return this.setProps({visible: show});
    },

    /**
     * Use this function to find whether or not this is visible according to our
     * spec.
     *
     * @return {boolean} true if visible; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().visible;
     */
    getVisible: function() {
        return this.getProps().visible;
    },

    /**
     * Test if button is set as "primary".
     *
     * @return {boolean} true if primary; otherwise, false for secondary
     * @deprecated Use document.getElementById(id).getProps().primary;
     */
    isPrimary: function() {
        return this.getProps().primary;
    },

    /**
     * Set button as "primary".
     *
     * @param {boolean} primary true for primary, false for secondary
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({primary: boolean});
     */
    setPrimary: function(primary) {
        if (primary == null) {
            return null;
        }
        return this.setProps({primary: primary});
    },

    /**
     * Test if button is set as "secondary".
     *
     * @return {boolean} true if secondary; otherwise, false for primary
     * @deprecated Use !(document.getElementById(id).getProps().primary);
     */
    isSecondary: function() {
        return !(this.getProps().primary);
    },

    /**
     * Set button as "secondary".
     *
     * @param {boolean} secondary true for secondary, false for primary
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({primary: false});
     */
    setSecondary: function(secondary) {
        if (secondary == null) {
            return null;
        }
        return this.setProps({primary: !secondary});
    },

    /**
     * Test if button is set as "mini".
     *
     * @return {boolean} true if mini; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().mini;
     */
    isMini: function() {
        return this.getProps().mini;
    },

    /**
     * Set button as "mini".
     *
     * @param {boolean} mini true for mini, false for standard button
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({mini: boolean});
     */
    setMini: function(mini) {
        if (mini == null) {
            return null;
        }
        return this.setProps({mini: mini});
    },

    /**
     * Test disabled state of button.
     *
     * @return {boolean} true if disabled; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().disabled;
     */
    getDisabled: function() {
        return this.getProps().disabled;
    },

    /**
     * Test disabled state of button.
     *
     * @param {boolean} disabled true if disabled; otherwise, false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(disabled) {
        if (disabled == null) {
            return null;
        }
        return this.setProps({disabled: disabled});
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.button = woodstock4_3._html.button;


woodstock4_3._dojo.provide("woodstock4_3._html.checkbox");



woodstock4_3._dojo.provide("woodstock4_3._html.rbcb");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for checkbox and radio button components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.rbcbGroup
 * @private
 */
woodstock4_3._html.rbcb = {
    /**
     * 
     * @param {String} elementId The element Id.
     * @param {boolean} checked true or false
     * @param {String} type
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */ 
    setChecked: function(elementId, checked, type) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({checked: checked});
        }
        return null; 
    },

    /**
     *
     * @param {String} elementId The element Id.
     * @param {boolean} disabled true or false
     * @param {String} type
     * @param {String} enabledStyle
     * @param {String} disabledStyle
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */ 
    setDisabled: function(elementId, disabled, type, enabledStyle,
            disabledStyle) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null; 
    },

    /** 
     * Set the disabled state for all radio buttons with the given controlName.
     * If disabled is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id
     * @param {String} formName The name of the form containing the element
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled, type, enabledStyle,
            disabledStyle) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.rbcb = woodstock4_3._html.rbcb;

/**
 * @class This class contains functions for checkbox components.
 * @static
 * 
 * @deprecated See woodstock4_3.widget.checkbox
 * @private
 */
woodstock4_3._html.checkbox = {
    /**
     * Set the disabled state for the given checkbox element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) {
        return woodstock4_3._html.rbcb.setDisabled(elementId, disabled,
            "checkbox", "Cb", "CbDis");
    },

    /** 
     * Set the disabled state for all the checkboxes in the check box
     * group identified by controlName. If disabled
     * is set to true, the check boxes are shown with disabled styles.
     *
     * @param {String} controlName The checkbox group control name
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled) {    
        return woodstock4_3._html.rbcb.setGroupDisabled(controlName,
            disabled, "checkbox", "Cb", "CbDis");
    },

    /**
     * Set the checked property for a checkbox with the given element Id.
     *
     * @param {String} elementId The element Id
     * @param checked true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */
    setChecked: function(elementId, checked) {
        return woodstock4_3._html.rbcb.setChecked(elementId, checked,
            "checkbox");
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.checkbox = woodstock4_3._html.checkbox;


woodstock4_3._dojo.provide("woodstock4_3._html.dropDown");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for dropDown components.
 * @static
 * 
 * @deprecated See woodstock4_3.widget.dropDown
 * @private
 */
woodstock4_3._html.dropDown = {
    /**
     * This function is invoked by the choice onselect action to set the
     * selected, and disabled styles.
     *
     * Page authors should invoke this function if they set the 
     * selection using JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated
     */
    changed: function(elementId) {         
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget._changed();
        }
        return false;
    },

    /**
     * Use this function to access the HTML select element that makes up
     * the dropDown.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the dropDown).
     * @return {Node} a reference to the select element. 
     * @deprecated Use document.getElementById(elementId).setSelectElement()
     */
    getSelectElement: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the label of the first
     * selected option on the dropDown. If no option is selected, this
     * function returns null.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The label of the selected option, or null if none is
     * selected. 
     * @deprecated Use document.getElementById(elementId).getSelectedLabel();
     */
    getSelectedLabel: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectedLabel();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the value of the first
     * selected option on the dropDown. If no option is selected, this
     * function returns null. 
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The value of the selected option, or null if none is
     * selected. 
     * @deprecated Use document.getElementById(elementId).getSelectedValue();
     */
    getSelectedValue: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectedValue();
        }
        return null;
    },

    /**
     * Set the disabled state for given dropdown element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * Page authors should invoke this function if they dynamically
     * enable or disable a dropdown using JavaScript.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({ disabled: disabled});
        }
        return null;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.dropDown = woodstock4_3._html.dropDown;


woodstock4_3._dojo.provide("woodstock4_3._html.field");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for field components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.field
 * @private
 */
woodstock4_3._html.field = {
    /**
     * Use this function to get the HTML input or textarea element
     * associated with a TextField, PasswordField, HiddenField or TextArea
     * component.
     *
     * @param {String} elementId The element ID of the field 
     * @return {Node} the input or text area element associated with the field component
     * @deprecated Use document.getElementById(elementId).getInputElement()
     */
    getInputElement: function(elementId) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getInputElement();
        }
        return null;
    },

    /**
     * Use this function to get the value of the HTML element 
     * corresponding to the Field component.
     *
     * @param {String} elementId The element ID of the Field component
     * @return {String} the value of the HTML element corresponding to the Field component 
     * @deprecated Use document.getElementById(id).getProps().value;
     */
    getValue: function(elementId) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getProps().value;
        }
        return null;
    },

    /**
     * Use this function to set the value of the HTML element 
     * corresponding to the Field component
     *
     * @param {String} elementId The element ID of the Field component
     * @param {String} newValue The new value to enter into the input element Field component 
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({value: "text"});
     */
    setValue: function(elementId, newValue) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({value: newValue});
        }
        return null;
    },

    /** 
     * Use this function to get the style attribute for the field. 
     * The style retrieved will be the style on the span tag that 
     * encloses the (optional) label element and the input element.
     *
     * @param {String} elementId The element ID of the Field component
     * @return {String} The style property of the field.
     * @deprecated Use document.getElementById(id).getProps().style;
     */
    getStyle: function(elementId) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getProps().style;
        }
        return null;
    },

    /**
     * Use this function to set the style attribute for the field. 
     * The style will be set on the <span> tag that surrounds the field.
     *
     * @param {String} elementId The element ID of the Field component
     * @param {String} newStyle The new style to apply
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({style: newStyle});
     */
    setStyle: function(elementId, newStyle) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({style: newStyle});
        }
        return null;
    },

    /**
     * Use this function to disable or enable a field. As a side effect
     * changes the style used to render the field. 
     *
     * @param {String} elementId The element ID of the field 
     * @param {boolean} newDisabled true to disable the field, false to enable the field
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, newDisabled) {  
        if (newDisabled == null) {
            return null;
        }
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({disabled: newDisabled});
        }
        return null;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.field = woodstock4_3._html.field;


woodstock4_3._dojo.provide("woodstock4_3._html.hyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for hyperlink components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.hyperlink
 * @private
 */
woodstock4_3._html.hyperlink = {
    /**
     * This function is used to submit a hyperlink.
     * <p>
     * Note: Params are name value pairs but all one big string array so 
     * params[0] and params[1] form the name and value of the first param.
     * </p>
     *
     * @params {Object} hyperlink The hyperlink element
     * @params {String} formId The form id
     * @params {Object} params Name value pairs
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated See woodstock4_3.widget.hyperlink
     */
    submit: function(hyperlink, formId, params) {
        // Need to test widget for tab and common task components. If a widget 
        // does not exist, fall back to the old code.
	var widget = woodstock4_3.widget.common.getWidget(hyperlink.id);
	if (widget == null) {
            // If a widget does not exist, we shall create one in order to call
            // the submit function directly.
            woodstock4_3._dojo.require("woodstock4_3.widget.hyperlink");
            widget = new woodstock4_3.widget.hyperlink({id: hyperlink.id});
	}
        return widget._submitFormData(formId, params);
    },

    /**
     * Use this function to access the HTML img element that makes up
     * the icon hyperlink.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the outter most tag enclosing the HTML img element).
     * @return {Node} The HTML image element.
     * @deprecated Use document.getElementById(elementId).getProps().enabledImage;
     */
    getImgElement: function(elementId) {
        // Need to test widget for alarmStatus, jobstatus, and notification phrase
        // components. If a widget does not exist, fall back to the old code.
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        var props = (widget) ? widget.getProps() : null;
        if (props && props.enabledImage) {
            var imgWidget = woodstock4_3.widget.common.getWidget(props.enabledImage.id);
            if (imgWidget != null) {
                return imgWidget._domNode;    
            }
        }

        // Image hyperlink is now a naming container and the img element id 
        // includes the ImageHyperlink parent id.
        if (elementId != null) {
            var parentid = elementId;
            var colon_index = elementId.lastIndexOf(":");
            if (colon_index != -1) {
                parentid = elementId.substring(colon_index + 1);
            }
            return document.getElementById(elementId + ":" + parentid + "_image");
        }
        return document.getElementById(elementId + "_image");
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.hyperlink = woodstock4_3._html.hyperlink;


woodstock4_3._dojo.provide("woodstock4_3._html.jumpDropDown");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for jumpDropDown components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.dropDown
 * @private
 */
woodstock4_3._html.jumpDropDown = {
    /**
     * This function is invoked by the jumpdropdown onchange action to set the
     * form action and then submit the form.
     *
     * Page authors should invoke this function if they set the selection using 
     * JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed()
     */
    changed: function(elementId) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.changed();
        }
        return false;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.jumpDropDown = woodstock4_3._html.jumpDropDown;


woodstock4_3._dojo.provide("woodstock4_3._html.listbox");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for listbox components.
 * @static
 * 
 * @deprecated See woodstock4_3.widget.listbox
 * @private
 */
woodstock4_3._html.listbox = {
    /**
     * This function is invoked by the list onselect action to set the selected, 
     * and disabled styles.
     *
     * Page authors should invoke this function if they set the selection
     * using JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed();
     */
    changed: function(elementId) {         
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget._changed();
        }
        return false;
    },

    /**
     * Use this function to access the HTML select element that makes up
     * the list. 
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the list).
     * @return {Node} The HTML select element.
     * @deprecated Use document.getElementById(elementId).getSelectElement()
     */
    getSelectElement: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the label of the first
     * selected option on the listbox. If no option is selected, this
     * function returns null. 
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The label of the selected option, or null if none is selected.
     * @deprecated Use document.getElementById(elementId).getSelectedLabel();
     */
    getSelectedLabel: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectedLabel();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the value of the first
     * selected option on the listbox. If no option is selected, this
     * function returns null. 
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The value of the selected option, or null if none is
     * selected.
     * @deprecated Use document.getElementById(elementId).getSelectedValue();
     */
    getSelectedValue: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectedValue();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to set the enabled/disabled state
     * of the listbox component. In addition to disabling the list, it
     * also changes the styles used when rendering the component. 
     *
     * Page authors should invoke this function if they dynamically
     * enable or disable a list using JavaScript.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.listbox = woodstock4_3._html.listbox;
woodstock4_3._dojo.require("woodstock4_3._html.rbcb");


woodstock4_3._dojo.provide("woodstock4_3._html.radiobutton");

woodstock4_3._dojo.require("woodstock4_3._html.rbcb");

/**
 * @class This class contains functions for radioButton components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.radioButton
 * @private
 */
woodstock4_3._html.radiobutton = {
    /**
     * Set the disabled state for the given radiobutton element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id.
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) {    
        return woodstock4_3._html.rbcb.setDisabled(elementId, disabled, 
            "radio", "Rb", "RbDis");
    },

    /**
     * Set the disabled state for all the radio buttons in the radio button
     * group identified by controlName. If disabled
     * is set to true, the check boxes are displayed with disabled styles.
     *
     * @param {String} controlName The radio button group control name
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled) {    
        return woodstock4_3._html.rbcb.setGroupDisabled(controlName, disabled, 
            "radio", "Rb", "RbDis");
    },

    /**
     * Set the checked property for a radio button with the given element Id.
     *
     * @param {String} elementId The element Id
     * @param {boolean} checked true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */
    setChecked: function(elementId, checked) {
        return woodstock4_3._html.rbcb.setChecked(elementId, checked, "radio");
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.radiobutton = woodstock4_3._html.radiobutton;


woodstock4_3._dojo.provide("woodstock4_3.json");

//
// Copyright (c) 2005 JSON.org
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The Software shall be used for Good, not Evil.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//

/**
 * @class This class contains functions for parsing JSON.
 * @static
 */
woodstock4_3.json = {
    /**
     * JSON escape chars.
     * @private
     */
    _m: {
        '\b': '\\b',
        '\t': '\\t',
        '\n': '\\n',
        '\f': '\\f',
        '\r': '\\r',
        '"' : '\\"',
        '\\': '\\\\'
    },

    /**
     * JSON parsor.
     * @private
     */
    _s: {
        /** @ignore */
        'boolean': function (x) {
            return String(x);
        },
        /** @ignore */
        number: function (x) {
            return isFinite(x) ? String(x) : null;
        },
        /** @ignore */
        string: function (x) {
            if (/["\\\x00-\x1f]/.test(x)) {
                x = x.replace(/([\x00-\x1f\\"])/g, function(a, b) {
                    var c = woodstock4_3.json._m[b];
                    if (c) {
                        return c;
                    }
                    c = b.charCodeAt();
                    return '\\u00' +
                        Math.floor(c / 16).toString(16) +
                        (c % 16).toString(16);
                });
            }
            return '"' + x + '"';
        },
        /** @ignore */
        object: function (x) {
            if (x) {
                var a = [], b, f, i, l, v;
                if (x instanceof Array) {
                    a[0] = '[';
                    l = x.length;
                    for (i = 0; i < l; i += 1) {
                        v = x[i];
                        f = woodstock4_3.json._s[typeof v];
                        if (f) {
                            v = f(v);
                            if (typeof v == 'string') {
                                if (b) {
                                    a[a.length] = ',';
                                }
                                a[a.length] = v;
                                b = true;
                            }
                        }
                    }
                    a[a.length] = ']';
                } else if (typeof x.hasOwnProperty === 'function') {
                    a[0] = '{';
                    for (i in x) {
                        if (x.hasOwnProperty(i)) {
                            v = x[i];
                            f = woodstock4_3.json._s[typeof v];
                            if (f) {
                                v = f(v);
                                if (typeof v == 'string') {
                                    if (b) {
                                        a[a.length] = ',';
                                    }
                                    a.push(woodstock4_3.json._s.string(i), ':', v);
                                    b = true;
                                }
                            }
                        }
                    }
                    a[a.length] = '}';
                } else {
                    return null;
                }
                return a.join('');
            }
            return null;
        }
    },

    /**
     * Stringify a JavaScript value, producing JSON text. 
     *
     * @param {Object} v A non-cyclical JSON object.
     * @return {boolean} true if successful; otherwise, false.
     */
    stringify: function (v) {
        var f = woodstock4_3.json._s[typeof v];
        if (f) {
            v = f(v);
            if (typeof v == 'string') {
                return v;
            }
        }
        return null;
    },

    /**
     * Parse a JSON text, producing a JavaScript object.
     *
     * @param {String} text The string containing JSON text.
     * @return {boolean} true if successful; otherwise, false.
     */
    parse: function (text) {
        try {
            return !(/[^,:{}\[\]0-9.\-+Eaeflnr-u \n\r\t]/.test(
                text.replace(/"(\\.|[^"\\])*"/g, ''))) && eval('(' + text + ')');
        } catch (e) {
            return false;
        }
    }
};


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.anchor");



woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.common");

woodstock4_3._dojo.require("woodstock4_3.json");
woodstock4_3._dojo.require("woodstock4_3.widget.common");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.dynaFaces");

woodstock4_3._dojo.require("woodstock4_3.theme.common");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions to obtain JSF Extensions resources.
 * @static
 * @private
 */
woodstock4_3.widget._jsfx.dynaFaces = {
    /**
     * This function is used to initialize the environment with Object literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {Object} ajax Key-Value pairs of Ajax properties.     
     * @config {boolean} ajax.isJsfx Flag indicating to include JSF Extensions.
     * @config {boolean} webuiJsfx Flag indicating to include default Ajax functionality.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        if (props == null) {
            console.debug("Cannot initialize JSF Extensions."); // See Firebug console.
            return false;
        }

        // Don't load JSF Extensions.
        if (props.ajax && props.ajax.isJsfx != null) {
            if (new Boolean(props.ajax.isJsfx).valueOf() == false) {
                return false;
            }
        }
        
        // Load JSF Extensions immediately.
        if (new Boolean(props.webuiJsfx).valueOf() == true) {
            woodstock4_3.widget._jsfx.dynaFaces._loadJsfx();
        } else {
            // Override default publish functionality to lazily load JSFX.
            woodstock4_3.widget.common.publish = function(topic, props) {
                // Load JSF Extensions and publish event via a callback, 
                // ensuring all resources have been loaded.
                woodstock4_3.widget._jsfx.dynaFaces._loadJsfx(function() {
                    // Publish an event for custom AJAX implementations to listen for.
                    return woodstock4_3._dojo.publish(topic, props);
                });
            };
        }
        return true;
    },

    /**
     * Load JSF Extensions.
     *
     * @param {Function} callback The JavaScript function to invoke on load.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _loadJsfx: function(callback) {
        if (typeof DynaFaces != "undefined") {
            return callback();
        }
        var bootstrap = woodstock4_3._base.bootstrap;
        var theme = woodstock4_3.theme.common;
        var isDebug = new Boolean(woodstock4_3._base.config.isDebug).valueOf();

        // Get script URLs.
        var pUrl = theme._getJavaScript((isDebug == true)
            ? "prototypeUncompressed" : "prototype");
        var jUrl = theme._getJavaScript((isDebug == true)
            ? "jsfxUncompressed" : "jsfx");

        // Ensure Prototype is loaded first using a callback.
        var jsfxCallback = function() {
            bootstrap._loadScript(jUrl, callback);
        };

        // Load Prototype and DynaFaces.
        //
        // Note: Prototype must be added to the global scope. Therefore,
        // dojo._loadPath() and/or dojo._loadUri() will not work here.
        if (typeof Prototype == "undefined") {
            bootstrap._loadScript(pUrl, jsfxCallback);
        } else {
            jsfxCallback();
        }
        return true;
    }
};

// Initialize the environment.
woodstock4_3.widget._jsfx.dynaFaces._init(woodstock4_3._base.config);

/**
 * @class This class contains functions to obtain data asynchronously using JSF
 * Extensions as the underlying transfer protocol.
 * @static
 * @private
 */
woodstock4_3.widget._jsfx.common = {
    /**
     * This function is used to process refresh events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The widget id.
     * @config {String} endTopic The event topic to publish.
     * @config {String} execute The string containing a comma separated list 
     * of client ids against which the execute portion of the request 
     * processing lifecycle must be run.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processRefreshEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);

        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: (props.execute) ? props.execute : "none",
            closure: {
                endTopic: props.endTopic
            },
            render: props.id,
            replaceElement: woodstock4_3.widget._jsfx.common._refreshCallback,
            xjson: {
                id: props.id,
                event: "refresh"
            }
        });
        return true;
    },

    /**
     * This function is used to process state change events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The widget id.
     * @config {String} endTopic The event topic to publish.
     * @config {Object} props Key-Value pairs of widget properties to update.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processStateEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);

        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: "none",
            closure: {
                endTopic: props.endTopic
            },
            render: props.id,
            replaceElement: woodstock4_3.widget._jsfx.common._stateCallback,
            xjson: {
                id: props.id,
                event: "state",
                props: props.props // Widget properties to update.
            }
        });
        return true;
    },

    /**
     * This function is used to process submit events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The widget id.
     * @config {String} endTopic The event topic to publish.
     * @config {String} execute The string containing a comma separated list 
     * of client ids against which the execute portion of the request 
     * processing lifecycle must be run.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processSubmitEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);

        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: (props.execute) ? props.execute : props.id,
            closure: {
                endTopic: props.endTopic
            },
            render: props.id,
            replaceElement: woodstock4_3.widget._jsfx.common._submitCallback,
            xjson: {
                id: props.id,
                event: "submit"
            }
        });
        return true;
    }, 
   
    /**
     * This function is used to refresh widgets.
     *
     * @param {String} id The widget id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson argument provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _refreshCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = woodstock4_3.json.parse(content);

        // Add rows.
        var widget = woodstock4_3.widget.common.getWidget(id);
        widget.setProps(props);

        // Publish an event for custom AJAX implementations to listen for.
        if (closure.endTopic) {
            woodstock4_3._dojo.publish(closure.endTopic, [props]);
        }
        return true;
    },

    /**
     * This function is a callback to respond to the end of state request.
     * It will only publish submit end event without updating the widget itself.
     *
     * @param {String} id The widget id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson argument provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _stateCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = woodstock4_3.json.parse(content);
            
        // Publish an event for custom AJAX implementations to listen for.
        if (closure.endTopic) {
            woodstock4_3._dojo.publish(closure.endTopic, [props]);
        }
        return true;
    },

    /**
     * This function is a callback to respond to the end of submit request.
     * It will only publish submit end event without updating the widget itself.
     *
     * @param {String} id The widget id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson argument provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _submitCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = woodstock4_3.json.parse(content);
            
        // Publish an event for custom AJAX implementations to listen for.
        if (closure.endTopic) {
            woodstock4_3._dojo.publish(closure.endTopic, [props]);
        }
        return true;
    }
};


woodstock4_3._dojo.provide("woodstock4_3.widget.anchor");



woodstock4_3._dojo.provide("woodstock4_3.widget._base.refreshBase");

woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/** 
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.refreshBase
 * @class This class contains functions for widgets that extend refreshBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.refreshBase");

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget._base.refreshBase.event =
        woodstock4_3.widget._base.refreshBase.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         */
        beginTopic: undefined,

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         */
        endTopic: undefined
    }
};

/**
 * Initialize public functions.
 * <p>
 * Note: If this.event.refresh.beginTopic and this.event.refresh.endTopic are
 * null, a public function shall be added to the DOM node.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.refreshBase.prototype._initRefreshFunc = function () {
    if (this.event == null || this.event.refresh == null
            || this.event.refresh.beginTopic == null
            || this.event.refresh.endTopic == null) {
        return false;
    }
    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Set public functions.
    /** @ignore */
    this._domNode.refresh = function(execute) {
        return woodstock4_3.widget.common.getWidget(_id).refresh(execute);
    };
    return true;
};

/**
 * Process refresh event.
 * <p>
 * Note: If this.event.refresh.beginTopic and this.event.refresh.endTopic are 
 * not null, an event is published for custom Ajax implementations to listen 
 * for. If event topics are not implemented for this widget, the function 
 * returns and a message is output to the console.
 * </p>
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.refreshBase.prototype.refresh = function(execute) {
    if (this.event == null || this.event.refresh == null
            || this.event.refresh.beginTopic == null
            || this.event.refresh.endTopic == null) {
        console.debug("Error: Refresh event topics not implemented for " + 
            this._widgetType); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this._publish(this.event.refresh.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: this.event.refresh.endTopic
    }]);
    return true;
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.stateBase");

woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/** 
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.stateBase
 * @class This class contains functions for widgets that extend stateBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.stateBase");

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget._base.stateBase.event =
        woodstock4_3.widget._base.stateBase.prototype.event = {
    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         */
        beginTopic: undefined,

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         */
        endTopic: undefined
    }
};

/**
 * Process state event.
 * <p>
 * Note: If this.event.state.beginTopic and this.event.state.endTopic are 
 * not null, an event is published for custom Ajax implementations to listen 
 * for. If event topics are not implemented for this widget, the function 
 * returns and a message is output to the console.
 * </p>
 * @param {Object} props Key-Value pairs of widget properties to update.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.stateBase.prototype._stateChanged = function(props) {
    if (this.event == null || this.event.state == null
            || this.event.state.beginTopic == null
            || this.event.state.endTopic == null) {
        console.debug("Error: State event topics not implemented for " + 
            this._widgetType); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this._publish(this.event.state.beginTopic, [{
        id: this.id,
        endTopic: this.event.state.endTopic,
        props: props
    }]);
    return true;
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.anchorBase");



woodstock4_3._dojo.provide("woodstock4_3.widget._base.widgetBase");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3._dijit._Templated");
woodstock4_3._dojo.require("woodstock4_3._dijit._Widget"); 
woodstock4_3._dojo.require("woodstock4_3.theme.common");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.widgetBase
 * @class This class contains functions used for base functionality in all 
 * widgets. 
 * <p>
 * The widgetBase class inherits from woodstock4_3._dijit._Widget and 
 * woodstock4_3._dijit._Templated. The woodstock4_3._dijit._Widget class
 * is responsible for calling the _buildRendering() and _postCreate() functions 
 * in that order. The dijit_Templated function overrides the _buildRendering() 
 * functon to fill in template properties.
 * <p></p>
 * The _postCreate() function is responible for initializing CSS selectors, 
 * events, and public functions. Commonly used functions (e.g., getProps(), 
 * setProps(), and refresh() are set on the outermost DOM node via the 
 * "superclass" function of widgetBase. This inherited function is also 
 * responsible for invoking the private _setProps() function. 
 * <p></p>
 * The private _setProps() function is used to set widget properties that can be
 * updated by a web app developer. This helps encapsolate functionality and 
 * brand changes while providing a common API for all widgets. In addition, the 
 * widget is selctively updated (i.e., if and only if a key-value pair has been 
 * given). Saving given properties is deferred to the public setProps() function
 * which allows _setProps() to be used during initialization.
 * <p></p>
 * The private _setProps() function is also responsible for invoking 
 * _setCommonProps() and _setEventProps(). These properties may not always be
 * set on the outermost DOM node; however, core (i.e., id, class, style, etc.) 
 * properties are. Core properties are set on the DOM via the "superclass" 
 * function of widgetBase which invokes the _setCoreProps() function.
 * <p></p>
 * The _getClassName() function is responsible for obtaining the selector that
 * will be set on the outermost DOM node. The private _setProps() function 
 * of widgetBase ensures that the _getClassName() function is called prior to 
 * invoking _setCoreProps(). In most cases, this function will be overridded in
 * order to apply widget specific selectors. However, selectors should be 
 * concatinated in order of precedence (e.g., the user's className property is 
 * always appended last).
 * <p></p>
 * The public setProps() function is responsible for extending the widget class
 * with properties so they can be used during later updates. After extending the
 * widget, the private _setProps() function is called. In some cases, the public
 * setProps() function may be overridden. For example, the label clears the
 * contents property from the widget because that is something we do not want to
 * extend.
 * <p></p>
 * The start() function is typically called after the widget has been 
 * instantiated. For example, a progressBar might start a timer to periodically
 * refresh. 
 * <p></p>
 * Warning: It's not possible to append HTML elements from script that is 
 * not a direct child of the BODY element. If there is any Javascript
 * running inside the body that is a direct child of body, IE will throw
 * an "Internet Explorer cannot open the Internet site" error. For example,
 * woodstock4_3._dijit._Templated._createNodesFromText generates such an 
 * error by calling appendChild(). Therefore, widget creation must be deferred
 * to the window.onLoad event. See http://trac.dojotoolkit.org/ticket/4631
 * </p>
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.widgetBase", [
        woodstock4_3._dijit._Widget, woodstock4_3._dijit._Templated ], {
    // Note: If your class contains arrays or other objects, they should be
    // declared in the constructor function so that each instance gets it's own
    // copy. Simple types (literal strings and numbers) are fine to declare in 
    // the class directly. Also note that superclass constructors are always 
    // called automatically, and always before the subclass constructor.
    constructor: function() {
        this._started = false;
        this._templateType = this._widgetType;
    },

    // Set defaults.
    _common: woodstock4_3._base.common, // Common utils.
    _dojo: woodstock4_3._dojo, // Dojo utils.
    _proto: woodstock4_3._base.proto, // Prototype utils.
    _theme: woodstock4_3.theme.common, // Theme utils.
    _widget: woodstock4_3.widget.common // Widget utils.
});

// This function is not public and should not appear in the jsDoc.
/** @ignore */
woodstock4_3.widget._base.widgetBase.prototype.buildRendering = function () {
    // Template must be set prior to calling "superclass".
    this._buildRendering();
    return this._inherited("buildRendering", arguments);
};

/**
 * This function is used to render the widget from a template.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._buildRendering = function () {
    // Get default template path.
    if (this.templatePath == null) {
        this.templatePath = this._theme._getTemplatePath(this._templateType);
    }

    // Get default template string.
    if (this.templatePath == null && this.templateString == null) {
        var browser = woodstock4_3._base.browser;

        // Load browser specific template, if any.
        if (browser._isFirefox()) {
            this.templateString = this._theme._getTemplateString(
                this._widgetType + "_firefox");
        } else if (browser._isMozilla()) {
            this.templateString = this._theme._getTemplateString(
                this._widgetType + "_mozilla");
        } else if (browser._isIe()) {
            this.templateString = this._theme._getTemplateString(
                this._widgetType + "_ie");
        } else if (browser._isSafari()) {
            this.templateString = this._theme._getTemplateString(
                this._widgetType + "_safari");
        }
        // Get default template.
        if (this.templateString == null) {
            this.templateString = this._theme._getTemplateString(this._templateType);
        }
    }

    // The templatePath should have precedence. Therefore, in order for the 
    // templatePath to be used, templateString must be null.
    if (this.templatePath != null) {
        this.templateString = null;
    }
    return true;
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._getClassName = function() {
    // Save template selector, if any.
    if (typeof this._className == "undefined") {
        this._className = this._domNode.className;
    }
    // Append template selector to given className property.
    var newClassName = this.className;
    if (this.className && this._className) {
        newClassName += " " + this._className;
    } else if (this._className) {
        newClassName = this._className;
    }
    return newClassName;
};

/**
 * This function is used to get common properties from the widget. Please see
 * the _setCommonProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._getCommonProps = function() {
    var props = {};

    // Set properties.
    if (this.accessKey != null) { props.accessKey = this.accessKey; }
    if (this.dir != null) { props.dir = this.dir; }
    if (this.lang != null) { props.lang = this.lang; }
    if (this.tabIndex != null) { props.tabIndex = this.tabIndex; }
    if (this.title != null) { props.title = this.title; }

    return props;
};

/**
 * This function is used to get core properties from the widget. Please see
 * the _setCoreProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._getCoreProps = function() {
    var props = {};

    // Set properties.
    if (this.className != null) { props.className = this.className; }
    if (this.id != null) { props.id = this.id; }
    if (this.style != null) { props.style = this.style; }
    if (this.visible != null) { props.visible = this.visible; }

    return props;
};

/**
 * This function is used to get event properties from the widget. Please
 * see the _setEventProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._getEventProps = function() {
    var props = {};

    // Set properties.
    if (this.onBlur != null) { props.onBlur = this.onBlur; }
    if (this.onChange != null) { props.onChange = this.onChange; }
    if (this.onClick != null) { props.onClick = this.onClick; }
    if (this.onDblClick != null) { props.onDblClick = this.onDblClick; }
    if (this.onFocus != null) { props.onFocus = this.onFocus; }
    if (this.onKeyDown != null) { props.onKeyDown = this.onKeyDown; }
    if (this.onKeyPress != null) { props.onKeyPress = this.onKeyPress; }
    if (this.onKeyUp != null) { props.onKeyUp = this.onKeyUp; }
    if (this.onMouseDown != null) { props.onMouseDown = this.onMouseDown; }
    if (this.onMouseOut != null) { props.onMouseOut = this.onMouseOut; }
    if (this.onMouseOver != null) { props.onMouseOver = this.onMouseOver; }
    if (this.onMouseUp != null) { props.onMouseUp = this.onMouseUp; }
    if (this.onMouseMove != null) { props.onMouseMove = this.onMouseMove; }
    if (this.onSelect != null) { props.onSelect = this.onSelect; }

    return props;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @id woodstock4_3.widget._base.widgetBase.getProps
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.widgetBase.prototype.getProps = function() {
    var props = {};

    // Set properties.
    this._proto._extend(props, this._getCommonProps(), false);
    this._proto._extend(props, this._getCoreProps(), false);
    this._proto._extend(props, this._getEventProps(), false);

    return props;
};

/**
 * The inherited function will climb up the scope chain, from superclass to 
 * superclass and through mixin classes as well, until it finds "someMethod",
 * then it will invoke that method.
 * <p>
 * Note: The argument is always literally arguments, a special Javascript array 
 * variable which holds all the arguments (like argv in C). There are a few 
 * variations to inherited() for special cases. If you have a method that was 
 * put into your object outside of declare, you need to specify the name of the
 * calling function like this:
 * </p><p><code>
 * this.inherited("someMethod", arguments);
 * </code></p><p>
 * You can also send custom parameters to the ancestor function. Just place the
 * extra arguments in array literal notation with brackets:
 * </p><p><code>
 * this.inherited(arguments, [ customArg1, customArg2 ]);
 * </code></p>
 *
 * @param {String} name The name of the inherited function.
 * @param {Object} args The function arguments.
 * @param {Object} newArgs Custom function arguments.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._inherited = function(name, args, newArgs) {
    return this.inherited(name, args, newArgs);
};

/**
 * This function is used to test if widget has been initialized.
 * <p>
 * Note: It is assumed that an HTML element is used as a place holder for the
 * document fragment.
 * </p>
 * @return {boolean} true if widget is initialized.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._isInitialized = function() {
    // Testing if the outermost DOM node has been added to the document and
    // ensuring a Dojo attach point exists works fine for JSP. However, the 
    // following code always returns null for facelets.
    //
    // var domNode = document.getElementById(this.id);
    // if (domNode && domNode.getAttribute("dojoattachpoint")) {
    if (this.initialized == true) {
        return true;
    }
    return false;
};

// This function is not public and should not appear in the jsDoc.
/** @ignore */
woodstock4_3.widget._base.widgetBase.prototype.postCreate = function () {
    return this._postCreate();
};

/**
 * This is called after the _buildRendering() function.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._postCreate = function () {
    // In order to register widgets properly, the DOM node id must be set prior 
    // to creating any widget children. Otherwise, widgets may not be destroyed.
    this._domNode.id = this.id;

    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Set public functions.
    /** @ignore */
    this._domNode.getProps = function() { 
        return woodstock4_3.widget.common.getWidget(_id).getProps();
    };
    /** @ignore */
    this._domNode.setProps = function(props, notify) { 
        return woodstock4_3.widget.common.getWidget(_id).setProps(props, notify);
    };
    /** @ignore */
    this._domNode.subscribe = function(topic, obj, func) {
        return woodstock4_3.widget.common.subscribe(topic, obj, func);
    };

    // Initialize public functions.
    if (typeof this._initRefreshFunc == "function") {
        this._initRefreshFunc();
    }
    if (typeof this._initSubmitFunc == "function") {
        this._initSubmitFunc();
    }

    // Set event topics.
    this._domNode.event = this.event;

    // Set properties.
    this._setProps(this.getProps());

    // All widget properties have been set.
    return this.initialized = true;
};

/**
 * Publish an event topic.
 * <p>
 * Note: In order to obtain Ajax modules dynamically, this function shall be 
 * overridden by a custom AJAX implementation.
 * </p>
 * @param {String} topic The event topic to publish.
 * @param {Object} props Key-Value pairs of properties. This will be applied
 * to each topic subscriber.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._publish = function(topic, props) {
    // Obtain the Ajax module associated with this widget.
    var config = woodstock4_3._base.config;
    if (new Boolean(config.ajax.isAjax).valueOf() == true && config.ajax.module) {
        woodstock4_3._dojo.require(config.ajax.module + "." + this._widgetType);
    }
    return woodstock4_3.widget.common.publish(topic, props);
};

/**
 * This function is used to set common properties for the given domNode.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey Shortcut key.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._setCommonProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }
    if (props.accessKey != null) { 
        domNode.accessKey = props.accessKey;
    }
    if (props.dir != null) {
        domNode.dir = props.dir;
    }
    if (props.lang != null) {
        domNode.lang = props.lang;
    }
    if (props.tabIndex > -1 && props.tabIndex < 32767) {
        domNode.tabIndex = props.tabIndex;
    }
    if (props.title != null) {
        domNode.title = props.title;
    }
    return true;
};

/**
 * This function is used to set core properties for the given domNode. These
 * properties are typically set on the outermost element.
 * <p>
 * Note: The className is typically provided by a web app developer. If 
 * the widget has a default className, it should be added to the DOM node
 * prior to calling this function. For example, the "myCSS" className would
 * be appended to the existing "Tblsun4" className (e.g., "Tbl_sun4 myCSS").
 * </p>
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} style Specify style rules inline.
 * @config {boolean} visible Hide or show element.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._setCoreProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }
    if (props.className != null) {
        domNode.className = props.className;
    }
    if (props.id != null) { 
        domNode.id = props.id;
    }
    if (props.style != null) { 
        domNode.style.cssText = props.style;
    }
    if (props.visible != null) {
        this._common._setVisibleElement(domNode, 
            new Boolean(props.visible).valueOf());
    }
    return true;
};

/**
 * This function is used to set event properties for the given domNode.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Element value changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Element text selected.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._setEventProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }

    // Note: JSON strings are not recognized as JavaScript. In order for
    // events to work properly, an anonymous function must be created.
    if (props.onBlur != null) { 
        domNode.onblur = (typeof props.onBlur == 'string')
            ? new Function("event", props.onBlur)
            : props.onBlur;
    }
    if (props.onClick != null) {
        domNode.onclick = (typeof props.onClick == 'string')
            ? new Function("event", props.onClick)
            : props.onClick;
    }
    if (props.onChange != null) {
        domNode.onchange = (typeof props.onChange == 'string')
            ? new Function("event", props.onChange)
            : props.onChange;
    }
    if (props.onDblClick != null) {
        domNode.ondblclick = (typeof props.onDblClick == 'string')
            ? new Function("event", props.onDblClick)
            : props.onDblClick;
    }
    if (props.onFocus != null) {
        domNode.onfocus = (typeof props.onFocus == 'string')
            ? new Function("event", props.onFocus)
            : props.onFocus;
    }
    if (props.onKeyDown != null) {
        domNode.onkeydown = (typeof props.onKeyDown == 'string')
            ? new Function("event", props.onKeyDown)
            : props.onKeyDown;
    }
    if (props.onKeyPress != null) {
        domNode.onkeypress = (typeof props.onKeyPress == 'string')
            ? new Function("event", props.onKeyPress)
            : props.onKeyPress;
    }
    if (props.onKeyUp != null) {
        domNode.onkeyup = (typeof props.onKeyUp == 'string')
            ? new Function("event", props.onKeyUp)
            : props.onKeyUp;
    }
    if (props.onMouseDown != null) {
        domNode.onmousedown = (typeof props.onMouseDown == 'string')
            ? new Function("event", props.onMouseDown)
            : props.onMouseDown;
    }
    if (props.onMouseOut != null) {
        domNode.onmouseout = (typeof props.onMouseOut == 'string')
            ? new Function("event", props.onMouseOut)
            : props.onMouseOut;
    }
    if (props.onMouseOver != null) {
        domNode.onmouseover = (typeof props.onMouseOver == 'string')
            ? new Function("event", props.onMouseOver)
            : props.onMouseOver;
    }
    if (props.onMouseUp != null) {
        domNode.onmouseup = (typeof props.onMouseUp == 'string')
            ? new Function("event", props.onMouseUp)
            : props.onMouseUp;
    }
    if (props.onMouseMove != null) {
        domNode.onmousemove = (typeof props.onMouseMove == 'string')
            ? new Function("event", props.onMouseMove)
            : props.onMouseMove;
    }
    if (props.onSelect != null) {
        domNode.onselect = (typeof props.onSelect == 'string')
            ? new Function("event", props.onSelect)
            : props.onSelect;
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @id woodstock4_3.widget._base.widgetBase.setProps
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.widgetBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Extend widget object for later updates.
    this._proto._extend(this, props);

    // Create a clone which can be safely modified in order to update 
    // subwidgets more efficiently.
    var _props = {};
    this._proto._extend(_props, props);

    // Set properties.
    this._setProps(_props);

    // Notify listeners state has changed.
    if (new Boolean(notify).valueOf() == true &&
            typeof this._stateChanged == "function") {
        this._stateChanged(props);
    }
    return true;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set style class -- must be set before calling _setCoreProps().
    props.className = this._getClassName();
    // The visible selector must be set otherwise, className may wipe it out.
    props.visible = (props.visible != null) ? props.visible : this.visible;

    // Set more properties.
    return this._setCoreProps(this._domNode, props);
};

// This function is not public and should not appear in the jsDoc.
/** @ignore */
woodstock4_3.widget._base.widgetBase.prototype.startup = function () {
    return this._startup();
};

/**
 * This function is used to "start" the widget, after the widget has been
 * instantiated.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.widgetBase.prototype._startup = function () {
    if (this._started == true) {
        return false;
    }
    return this._started = true;
};

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.anchorBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class contains functions for widgets that extend anchorBase.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.anchorBase",
        woodstock4_3.widget._base.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
    }
});

/**
 * Helper function to add children.
 *
 * @param props Key-Value pairs of properties.
 * @config {Array} contents The contents of the anchor body.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.anchorBase.prototype._addContents = function(props) {
    if (props.contents == null) {
        return false;
    }

    // Remove child nodes.
    this._widget._removeChildNodes(this._domNode);

    // Add contents.
    for (i = 0; i < props.contents.length; i++) {
        this._widget._addFragment(this._domNode, props.contents[i], "last");
    }
    return true;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.anchorBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.hrefLang != null) { props.hrefLang = this.hrefLang; }
    if (this.target != null) { props.target = this.target; }
    if (this.type != null) { props.type = this.type; }
    if (this.rev != null) { props.rev = this.rev; }
    if (this.rel != null) { props.rel = this.rel; }
    if (this.shape != null) { props.shape = this.shape; }
    if (this.coords != null) { props.coords = this.coords; }
    if (this.charset != null) { props.charset = this.charset; }
    if (this.accessKey != null) { props.accesskey = this.accessKey; }
    if (this.href != null) { props.href = this.href; }
    if (this.name != null) { props.name = this.name; } 
    if (this.contents != null) { props.contents = this.contents; }
    if (this.disabled != null) { props.disabled = this.disabled; }

    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.anchorBase.prototype._onClickCallback = function(event) {
    if (this.disabled == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the request.
    var result = (this._domNode._onclick)
        ? this._domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.anchorBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.anchorBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Add contents.
    this._addContents(props);

    // Set properties.
    if (props.accessKey != null) { this._domNode.accesskey = props.accessKey; }
    if (props.charset != null) { this._domNode.charset = props.charset; }
    if (props.coords != null) { this._domNode.coords = props.coords; }
        if (props.href) {
            // If context path is provided, then check whether the image has
            // context path already appended and if not, append it.
            if (this.prefix) {
                props.href = 
                    woodstock4_3.widget.common._appendPrefix(this.prefix, props.href);
            }
            this._domNode.href = props.href; 
        }
    if (props.hrefLang != null) { this._domNode.hrefLang =  props.hrefLang; }
    if (props.name != null) { this._domNode.name = props.name; }
    if (props.rev != null) { this._domNode.rev = props.rev; }
    if (props.rel != null) { this._domNode.rel = props.rel; }
    if (props.shape != null) { this._domNode.shape = props.shape; }
    if (props.target != null) { this._domNode.target = props.target; }
    if (props.type != null) { this._domNode.type = props.type; }

    // Set id -- anchors must have the same id and name on IE.
    if (props.name != null) {
        props.id = props.name;
    }

    // A web app devleoper could return false in order to cancel the 
    // submit. Thus, we will handle this event via the onClick call back.
    if (props.onClick) {
        // Set private function scope on DOM node.
        this._domNode._onclick = (typeof props.onClick == 'string')
            ? new Function("event", props.onClick) : props.onClick;

        // Must be cleared before calling _setEventProps() below.
        props.onClick = null;
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * This function is used to construct an anchor widget.
 *
 * @constructor
 * @name woodstock4_3.widget.anchor
 * @extends woodstock4_3.widget._base.anchorBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the anchor widget. 
 * <p>
 * The anchor widget creates an HTML anchor that traverses to the specified URL.
 * Or, it can be used to Anchor a position in the page.
 * </p><p>
 * <h3>Example 1a: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 1b: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget which outputs a named anchor.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       name: "anchorName",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * anchor is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Anchor State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("anc1"); // Get anchor
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the anchor is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Anchor" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("anc1"); // Get anchor
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an anchor using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the anchor text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Anchor Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("anc1"); // Get anchor
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "anc1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.anchor.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.anchor", [
        woodstock4_3.widget._base.refreshBase,
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.anchorBase ], {
    // Set defaults.
    _widgetType: "anchor" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.anchor.event =
        woodstock4_3.widget.anchor.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.anchor.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_anchor_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.anchor.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_anchor_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.anchor.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_anchor_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.anchor.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_anchor_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.anchor.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.href && this.disabled == false)
        ? this._theme.getClassName("ANCHOR","")
        : this._theme.getClassName("ANCHOR_DISABLED","");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.anchor.prototype._postCreate = function () {
    // Create callback function for onclick event.
    this._dojo.connect(this._domNode, "onclick", this, "_onClickCallback");

    return this._inherited("_postCreate", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.anchor.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.button");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.button");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a button widget.
 * 
 * @constructor
 * @name woodstock4_3.widget.button
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the button widget.
 * <p>
 * The button widget can render a button as a primary button or a secondary
 * button. A primary button is intended to be used for buttons that are the most
 * commonly used on a page or section of a page, and are rendered to be more
 * visually prominent than secondary buttons. The primary attribute is false by
 * default, which renders a secondary button.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "btn1",
 *       value: "This is a button",
 *       widgetType: "button"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * button is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "btn1",
 *       value: "This is a button",
 *       widgetType: "button"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Button State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("btn1"); // Get button
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the button is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "btn1",
 *       value: "This is a button",
 *       widgetType: "button"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Button" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("btn1"); // Get button
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a button using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the button
 * label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "btn1",
 *       value: "This is a button",
 *       widgetType: "button"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Button Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("btn1"); // Get button
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "btn1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.button.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} escape HTML escape value (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} mini Set button as mini if true (N/A with src property).
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} prefix The application context path of image.
 * @config {boolean} primary Set button as primary if true (N/A with src property).
 * @config {boolean} reset Set button as reset if true (overrides submit type).
 * @config {boolean} submit Set button as submit if true (default). If false,
 * the underlying HTML input type will be button.
 * @config {String} src Source for image (overrides submit and reset types).
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.button", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
        this.escape = true;
        this.mini = false;
        this.primary = true;
        this.reset = false;
        this.submit = true;
    },
    _widgetType: "button" // Required for theme properties.
});

/**
 * This function is used to render the widget from a template.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._buildRendering = function () {
    // On IE, the HTML input type property must be set when the DOM node is 
    // created. Therefore, we're using HTML templates to define the type.
    if (this.src != null) {
        this._templateType = "imageButton";
    } else if (new Boolean(this.reset).valueOf() == true) {
        this._templateType = "resetButton";
    } else if (new Boolean(this.submit).valueOf() == true) {
        this._templateType = "submitButton";
    }
    return this._inherited("_buildRendering", arguments);
}

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.button.event =
        woodstock4_3.widget.button.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.button.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_button_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.button.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_button_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.button.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_button_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.button.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_button_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.button.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var key = null;

    // If dealing with an image button, only the BUTTON3 selectors are used.
    // Note that the "mini" and "primary" values can still be set but
    // have no effect on image buttons by policy Vs by theme.
    if (this.src != null) {
        key = (this.disabled == true)
            ? "BUTTON3_DISABLED"
            : "BUTTON3";
    } else if (this.mini == true && this.primary == true) {
        key = (this.disabled == true)
            ? "BUTTON1_MINI_DISABLED" // primaryMiniDisabledClassName
            : "BUTTON1_MINI";         // primaryMiniClassName;
    } else if (this.mini == true) {
        key = (this.disabled == true)
            ? "BUTTON2_MINI_DISABLED" // secondaryMiniDisabledClassName
            : "BUTTON2_MINI";         // secondaryMiniClassName;
    } else if (this.primary == true) {
        key = (this.disabled == true)
            ? "BUTTON1_DISABLED"      // primaryDisabledClassName
            : "BUTTON1";              // primaryClassName
    } else {
        key = (this.disabled == true)
            ? "BUTTON2_DISABLED"      // secondaryDisabledClassName
            : "BUTTON2";	      // secondaryClassName
    }

    var newClassName = this._theme.getClassName(key, "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to obtain the outermost HTML element class name during
 * an onFocus or onMouseOver event.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.button.prototype._getHoverClassName = function() {
    // Cannot call this._inherited("_getClassName", arguments) here.
    var className = woodstock4_3.widget.button.superclass._getClassName.apply(this, arguments);
    var key = null;

    // If dealing with an image button, only the BUTTON3 selectors are used.
    // Note that the "mini" and "primary" values can still be set but
    // have no effect on image buttons by policy Vs by theme.
    if (this.src != null) {
        key = "BUTTON3_HOVER";
    } else if (this.mini == true && this.primary == true) {
        key = "BUTTON1_MINI_HOVER"; 	// primaryMiniHovClassName;
    } else if (this.mini == true) {
        key = "BUTTON2_MINI_HOVER"; 	// secondaryMiniHovClassName;
    } else if (this.primary == true) {
        key = "BUTTON1_HOVER"; 		// primaryHovClassName;
    } else {
        key = "BUTTON2_HOVER";		// secondaryHovClassName;
    }

    var newClassName = this._theme.getClassName(key, "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};
    
/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @id woodstock4_3.widget.button.getProps
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.button.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.alt != null) { props.alt = this.alt; }
    if (this.align != null) { props.align = this.align; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.escape != null) { props.escape = this.escape; }
    if (this.mini != null) { props.mini = this.mini; }
    if (this.primary != null) { props.primary = this.primary; }
    if (this.src != null) { props.src = this.src; }
    if (this.value != null) { props.value = this.value; }

    return props;
};

/**
 * Helper function to create callback for onBlur event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._onBlurCallback = function(event) {
    if (this.disabled == true) {
        return true;
    }
    // Prevent errors during page submit, when modules have not been loaded.
    try {
        // Set style class.
        this._domNode.className = this._getClassName();
    } catch (err) {}
    return true;
};

/**
 * Helper function to create callback for onFocus event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._onFocusCallback = function(event) {
    if (this.disabled == true) {
        return true;
    }
    // Prevent errors during page submit, when modules have not been loaded.
    try {
        // Set style class.
        this._domNode.className = this._getHoverClassName();
    } catch (err) {}
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._domNode.name = this.id;
    }

    // Initialize deprecated public functions. 
    //
    // Note: Although we now have a setProps function to update properties,
    // these functions were previously added to the DOM node; thus, we must
    // continue to be backward compatible.
    
    /** @ignore */
    this._domNode.isSecondary = function() { return !(woodstock4_3.widget.common.getWidget(this.id).getProps().primary); };
    /** @ignore */
    this._domNode.setSecondary = function(secondary) { return woodstock4_3.widget.common.getWidget(this.id).setProps({primary: !secondary}); };
    /** @ignore */
    this._domNode.isPrimary = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().primary; };
    /** @ignore */
    this._domNode.setPrimary = function(primary) { return woodstock4_3.widget.common.getWidget(this.id).setProps({primary: primary}); };
    /** @ignore */
    this._domNode.isMini = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().mini; };
    /** @ignore */
    this._domNode.setMini = function(mini) { return woodstock4_3.widget.common.getWidget(this.id).setProps({mini: mini}); };
    /** @ignore */
    this._domNode.getDisabled = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().disabled; };
    /** @ignore */
    this._domNode.setDisabled = function(disabled) { return woodstock4_3.widget.common.getWidget(this.id).setProps({disabled: disabled}); };
    /** @ignore */
    this._domNode.getVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().visible; };
    /** @ignore */
    this._domNode.setVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setProps({visible: show}); };
    /** @ignore */
    this._domNode.getText = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().value; };
    /** @ignore */
    this._domNode.setText = function(text) { return woodstock4_3.widget.common.getWidget(this.id).setProps({value: text}); };
    this._domNode.doClick = this._domNode.click;

    // Set events.
    this._dojo.connect(this._domNode, "onblur", this, "_onBlurCallback");
    this._dojo.connect(this._domNode, "onfocus", this, "_onFocusCallback");
    this._dojo.connect(this._domNode, "onmouseout", this, "_onBlurCallback");
    this._dojo.connect(this._domNode, "onmouseover", this, "_onFocusCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.alt != null) { this._domNode.alt = props.alt; }
    if (props.align != null) { this._domNode.align = props.align; }

    // Set disabled.
    if (props.disabled != null) { 
        this._domNode.disabled = new Boolean(props.disabled).valueOf();
    }

    // Set image source.
    if (props.src) {
        // If context path is provided, then check whether the image has
        // context path already appended and if not, append it.
        if (this.prefix) {
            props.src = 
                woodstock4_3.widget.common._appendPrefix(this.prefix, props.src);                
        }
        this._domNode.src = props.src; 
    }

    // Set value -- an empty string is valid.
    if (props.value != null) {
        // If escape is true, we want the text to be displayed literally. To 
        // achieve this behavior, do nothing.
        //
        // If escape is false, we want any special sequences in the text 
        // (e.g., "&nbsp;") to be displayed as evaluated (i.e., unescaped).
        this._domNode.value = (new Boolean(this.escape).valueOf() == false)
            ? this._proto._unescapeHTML(props.value)
            : props.value;
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.button.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.checkbox");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.checkbox");



woodstock4_3._dojo.provide("woodstock4_3.widget._base.checkedBase");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget.common");


woodstock4_3._dojo.provide("woodstock4_3.widget._base.labeledBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.labeledBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class defines functions and properties 
 * for widgets that have label subcomponents and implements 
 * the <code>required</code> and <code>valid</code> properties which
 * control the indicators on the label.
 *
 * <h3>Dojo attach points</h3>
 * A <code>labeledBase</code> subclass's templates are expected to
 * define following attachpoint identifiers.
 * <ul>
 * <li>_labelContainer - the attachpoint for the label. (mandatory)</li>
 * <li>_brNode - the attachpoint for a <code>br</code> element to
 * implement <code>labelOnTop</code> behavior. (optional)</li>
 * </ul>
 * </p>
 * <h3>The <code>label</code> object property</h3>
 * The <code>label</code> property is an object that defines the properties
 * for a widget that is rendered to represent a label. Minimally, only the
 * <code>value</code> property of the label object property must be non 
 * null for <code>labeledBase</code> to render a label. If only the
 * <code>value</code> property is specified the following default
 * values will be used to create an instance of 
 * <code>woodstock4_3.widget.label</code>.
 * <p>
 * <ul>
 * <li><code>widgetType</code> -
 * <code>woodstock4_3.widget.label</code></li>
 * <li><code>id</code> - this.id + "_label"</li>
 * <li><code>htmlFor</code> - subclasses determine the appropriate value
 * for this property in their implementation of <code>_getLabelProps</code>.
 * </li>
 * </ul>
 * <p>See <code>_postCreate</code> and <code>_getLabelProps</code>
 * </p>
 * <p>
 * This class is meant to be subclassed and not instantiated.
 * Therefore it has no <code>_widgetType</code> property.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} required If true the widget that is labeled requires
 * input, and indicator will appear near the label.
 * @config {boolean} valid If true the widget that is labeled has valid 
 * input. If false an indicator will appear near the label.
 * @config {boolean} labelOnTop If true the label appears above the
 * widget that is labeled. Subclasses may or may not implement this property.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.labeledBase",
        woodstock4_3.widget._base.widgetBase, {
    // Set defaults.
    constructor: function() {
	// If true the label appears above the owning widget.
	this.lableOnTop = false,
	// If true an indicator appears near the label to indicate input
	// for the labeled widget is required.
	this.required = false;
	// If false an indicator appears near the label to indicate input
	// for the labeled widget is invalid.
	this.valid = true;

	// Private flag to remember that last label style class.
	this._lastLabelOnTopClassName =  null;

	// Private flag to remember that last disabled label style class.
	this._lastLabelDisabledClassName =  null;
    }
});

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.labeledBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    if (this.label != null) { props.label = this.label; };
    if (this.labelOnTop != null) { props.labelOnTop = this.labelOnTop; };
    if (this.required != null) { props.required = this.required; };
    if (this.valid != null) { props.valid = this.valid; };

    return props;
};

/**
 * Return a CSS selector to be used for the label widget when the widget
 * is disabled.
 * <p>
 * This implementation returns null. This method should be implemented
 * in subclasses to return the appropriate selector desired by the subclass.
 * </p>
 * @param {boolean} disabled If <code>disabled</code> is true, return the 
 * selector for the label when the widget is "disabled", else return the
 * selector for the label when widget is enabled.
 * @return {String} This implementation returns null;
 */
woodstock4_3.widget._base.labeledBase.prototype._getLabelDisabledClassName = function(disabled) {
    return null;
};

/**
 * Return a CSS selector to be used for the label widget.
 * <p>
 * This implementation returns null. This method should be implemented
 * in subclasses to return the appropriate selector desired by the subclass.
 * </p>
 * @param {boolean} ontop If <code>ontop</code> is true, return the selector
 * for the label when the label is "ontop", else return the selector for the
 * label when it is not "ontop".
 * @return {String} This implementation returns null;
 * @private
 */
woodstock4_3.widget._base.labeledBase.prototype._getLabelOnTopClassName = function(ontop) {
    return null;
};

/**
 * Return label properties desired by this widget.
 * <p>
 * This implementation returns null. This method should be implemented
 * in subclasses to return label properties desired by the subclass.
 * </p>
 * @return {Object} Key-Value pairs of label properties.
 * @private
 */
woodstock4_3.widget._base.labeledBase.prototype._getLabelProps = function() {
    var props = {};
    
    var cn = this._getLabelOnTopClassName(this.labelOnTop);
    if (cn != null) {
	this._common._addStyleClass(props, cn);
    }
    cn = this._getLabelDisabledClassName(this.disabled);
    if (cn != null) {
	this._common._addStyleClass(props, cn);
    }
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.labeledBase.prototype._postCreate = function () {
    // A widget that has inherited from labeledBase must have
    // "this._labelContainer", but check anyway.
    //
    if (!this._labelContainer) {
       return this._inherited("_postCreate", arguments);
    }

    // We should probably set and id anyway, even if it is just "_label".
    //
    if (this.id) {
	this._labelContainer.id = this.id + "_label";
    }

    // If the application is creating a label on construction
    // this.label will be non-null. If it is a widget fragment 
    // (provides a widgetType and an id) then assum it is fully
    // configured. If it is not a fragment then get the desired
    // label properties to create the widget.
    // 
    if (this.label && this.label.value != null
            && !this._widget._isFragment(this.label)) {
	this.label.id = this.id + "_label";
        this.label.widgetType = "label";

	// Get subclass label preferences
	//
	var props = this._getLabelProps();
	this._proto._extend(props, this.label);
	this._proto._extend(this.label, props);
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.labeledBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Always update the _brNode state, even if there isn't a label.
    // If there is no _brNode the subclass does not support labelOnTop.
    //
    if (this._brNode && props.labelOnTop != null) {
	this._common._setVisibleElement(this._brNode, props.labelOnTop);

	// Always remove the last label on top selector.
	// 
	this._common._stripStyleClass(this._labelContainer,
	    this._lastLabelOnTopClassName);

	// Get the label selector from the subclass
	// and remember the new ontop selector.
	//
	this._lastLabelOnTopClassName = 
	    this._getLabelOnTopClassName(props.labelOnTop);

	// Add the "ontop" selector.
	//
	this._common._addStyleClass(this._labelContainer,
	    this._lastLabelOnTopClassName);
    }

    // Cases:
    // - called from _postCreate
    //   + creating a label
    //       this.label and this.label.id will not be null,
    //       props.label will not be null
    //   + no label
    //       this.label and this.label.id and props.label will be null
    //
    // - called from setProps
    //   + creating a label 
    //       same as called from _postCreate
    //   + updating a label (label subcomponent must exist)
    //       props.label and this.label and this.label.id will not be null
    //   + updating the widget's required or valid properties
    //       props.label may be null, this.label and this.label.id will
    //       not be null because there must be can existing label
    //       subcomponent.
    //
    // If the label is being updated then props.label will not be null
    // and hence "this.label" will not be null, but "this.label.id"
    // may be null if there is no existing subcomponent label and props is 
    // "{ label : { value : 'foo'}}" i.e. no id is specified.
    // 
    if (this.label != null && this.label.id != null && (props.label != null 
            || props.required != null || props.valid != null
	    || props.disabled != null)) {

	if (props.disabled != null) {
	    this._common._stripStyleClass(this._labelContainer,
		this._lastLabelDisabledClassName);
	    this._lastLabelDisabledClassName =
		this._getLabelDisabledClassName(props.disabled);
	    this._common._addStyleClass(this._labelContainer,
		this._lastLabelDisabledClassName);
	}

	// Make sure we have an object to hang properties on.
	// This is the case where the widget's "required" and "valid"
	// properties are being updated.
	//
	if (props.label == null) {
	    props.label = {};
	}
	if (props.required != null) {
	    props.label.required = props.required;
	}
	if (props.valid != null) {
	    props.label.valid = props.valid;
	}

	this._widget._updateFragment(this._labelContainer, this.label.id,
            props.label);
    }

    // Set more properties.
    return this._inherited("_setProps", arguments);
};

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.checkedBase
 * @extends woodstock4_3.widget._base.labeledBase
 * @class This class contains functions for widgets that extend checkedBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.checkedBase",
        woodstock4_3.widget._base.labeledBase, {
    // Set defaults.
    _idSuffix: "" // Overridden by subclass
});

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._getImageClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._getInputClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * Returns the HTML input element that makes up the chekcbox.
 *
 * @return {Node} The HTML input element. 
 */
woodstock4_3.widget._base.checkedBase.prototype.getInputElement = function() {
    return this._inputNode;
};

/**
 * Return an Object Literal of label properties desired
 * by the checkedBase widget.
 * <p>
 * This implementation adds the <code>htmlFor</code> property with
 * <code>this._inputNode.id</code>.
 * </p>
 * @return {Object} object with the <code>htmlFor</code>
 * property set.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._getLabelProps = function() {
    // Let the super class contribute
    //
    var props = this.inherited("_getLabelProps", arguments);
    // Subclasses can override this value
    //
    props.level = 3;
    props.htmlFor = this._inputNode.id;
    return props;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.checkedBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.  
    if (this.align != null) { props.align = this.align; }
    if (this.disabled != null) { props.disabled = this.disabled; }   
    if (this.image != null) { props.image = this.image; }
    if (this.name != null) { props.name = this.name; }        
    if (this.readOnly != null) { props.readOnly = this.readOnly; }
    if (this.value != null) { props.value = this.value; }

    // After widget has been initialized, get user's input.
    if (this._isInitialized() == true && this._inputNode.checked != null) {
        props.checked = this._inputNode.checked;
    } else if (this.checked != null) {
        props.checked = this.checked;
    }
    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._onClickCallback = function(event) {
    if (this.readOnly == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the request.
    var result = (this._domNode._onclick)
        ? this._domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._inputNode.id = this.id + this._idSuffix;
        this._imageContainer.id = this.id + "_imageContainer";

        // If null, use HTML input id.
        if (this.name == null) {
            this.name = this._inputNode.id;
        }
    }

    // Set public functions.

    /** @ignore */
    this._domNode.getInputElement = function() {
	var widget = woodstock4_3.widget;
	return widget.common.getWidget(this.id).getInputElement();
    }
    
    // Create callback function for onclick event.
    this._dojo.connect(this._domNode, "onclick", this, "_onClickCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.value != null) { 
        this._inputNode.value = props.value;
    }
    if (props.readOnly != null) { 
        this._inputNode.readOnly = new Boolean(props.readOnly).valueOf();       
    }
    if (props.disabled != null) {
        this._inputNode.disabled = new Boolean(props.disabled).valueOf();        
    }
    
    // Set HTML input element class name.
    this._inputNode.className = this._getInputClassName();
    
    if (props.name != null) { 
        // IE does not support the name attribute being set dynamically as 
        // documented at:
        //
        // http://msdn.microsoft.com/workshop/author/dhtml/reference/properties/name_2.asp
        //
        // In order to create an HTML element with a name attribute, the name
        // and value must be provided when using the inner HTML property or the
        // document.createElement() function. As a work around, we shall set the
        // attribute via the HTML template using name="${this.name}". In order
        // to obtain the correct value, the name property must be provided to 
        // the widget. Although we're resetting the name below, as the default,
        // this has no affect on IE. 
        this._inputNode.name = props.name;
    }

    if (props.checked != null) {
        var checked = new Boolean(props.checked).valueOf();

        // Dynamically setting the checked attribute on IE 6 does not work until
        // the HTML input element has been added to the DOM. As a work around, 
        // we shall use a timeout to set the property during initialization.
        if (this._isInitialized() != true &&
                woodstock4_3._base.browser._isIe()) {
            var _id = this.id;
            setTimeout(function() {
                // New literals are created every time this function
                // is called, and it's saved by closure magic.
                var widget = woodstock4_3.widget.common.getWidget(_id);
                widget._inputNode.checked = checked;
            }, 0); // (n) milliseconds delay.
        } else {
            this._inputNode.checked = checked;
        }
    }

    // Set image properties.
    if (props.image || props.disabled != null && this.image) {     
        // Ensure property exists so we can call setProps just once.
        if (props.image == null) {
            props.image = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.image.className = this._getImageClassName();

        // Update/add fragment.
        this._widget._updateFragment(this._imageContainer, this.image.id, props.image);
    } 

    // Set more properties.
    this._setCommonProps(this._inputNode, props);
    this._setEventProps(this._inputNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");


woodstock4_3._dojo.provide("woodstock4_3.widget._base.submitBase");

woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/** 
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.submitBase
 * @class This class contains functions for widgets that extend submitBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.submitBase");

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget._base.submitBase.event =
        woodstock4_3.widget._base.submitBase.prototype.event = {
    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** 
         * Submit event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         */
        beginTopic: undefined,

        /** 
         * Submit event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         */
        endTopic: undefined
    }
};

/**
 * Initialize public functions.
 * <p>
 * Note: If this.event.submit.beginTopic and this.event.submit.endTopic are
 * not null, a public function shall be added to the DOM node.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.submitBase.prototype._initSubmitFunc = function () {
    if (this.event == null || this.event.submit == null
            || this.event.submit.beginTopic == null
            || this.event.submit.endTopic == null) {
        return false;
    }
    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Set public functions.
    /** @ignore */
    this._domNode.submit = function(execute) {
        return woodstock4_3.widget.common.getWidget(_id).submit(execute);    
    };
    return true;
};

/**
 * Process submit event.
 * <p>
 * Note: If this.event.submit.beginTopic and this.event.submit.endTopic are not
 * null, an event is published for custom Ajax implementations to listen for. If
 * event topics are not implemented for this widget, the function returns and a 
 * message is output to the console.
 * </p>
 * @param {String} execute Comma separated string containing a list of 
 * client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.submitBase.prototype.submit = function(execute) {
    if (this.event == null || this.event.submit == null
            || this.event.submit.beginTopic == null
            || this.event.submit.endTopic == null) {
        console.debug("Error: Submit event topics not implemented for " + 
            this._widgetType); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this._publish(this.event.submit.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: this.event.submit.endTopic
    }]);
    return true;
};

/**
 * This function is used to construct a checkbox widget.
 *
 * @constructor
 * @name woodstock4_3.widget.checkbox
 * @extends woodstock4_3.widget._base.checkedBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the checkbox widget.
 * <p>
 * The widget can be used as a single checkbox or as one checkbox among a group
 * of checkboxes. A group of checkboxes represents a multiple selection list 
 * which can have any number of checkboxes selected, or none selected.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cb1",
 *       label: { value: "This is a checkbox" },
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the radio button, the
 * checkbox is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "b1",
 *       label: { value: "This is a checkbox" },
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "rb1",
 *       label: { value: "Toggle Checkbox State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "radioButton"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cb1"); // Get checkbox
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the radio button, the checkbox is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cb1",
 *       label: { value: "This is a checkbox" },
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "rb1",
 *       label: { value: "Refresh Checkbox" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "radioButton"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cb1"); // Get checkbox
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a checkbox using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the checkbox 
 * label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cb1",
 *       label: { value: "This is a checkbox" },
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Checkbox Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cb1"); // Get checkbox
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "cb1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.checkbox.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {boolean} checked 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} image 
 * @config {String} label 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect 
 * @config {boolean} readOnly 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.checkbox", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.checkedBase ], {
    // Set defaults.
    _idSuffix: "_cb",
    _widgetType: "checkbox" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.checkbox.event =
        woodstock4_3.widget.checkbox.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkbox_event_refresh_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkbox_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkbox_event_state_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkbox_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkbox_event_submit_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkbox_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.disabled == true)
        ? this._theme.getClassName("CHECKBOX_SPAN_DISABLED", "")
        : this._theme.getClassName("CHECKBOX_SPAN", ""); 

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getImageClassName = function() {
    return (this.disabled == true)
        ? this._theme.getClassName("CHECKBOX_IMAGE_DISABLED", "")
        : this._theme.getClassName("CHECKBOX_IMAGE", "");  
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getInputClassName = function() {
    // readOnly style.
    if (this.readOnly == true) {
        return this._theme.getClassName("CHECKBOX_READONLY", "");        
    }

    // disabled style.
    return (this.disabled == true)
        ? this._theme.getClassName("CHECKBOX_DISABLED", "")
        : this._theme.getClassName("CHECKBOX", "");  
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getLabelDisabledClassName = function(disabled) {
    return (disabled == true)
        ? this._theme.getClassName("CHECKBOX_LABEL_DISABLED", "")
        : this._theme.getClassName("CHECKBOX_LABEL", "");  
};

/**
 * Return an Object Literal of label properties desired
 * by the checkbox widget.
 * <p>
 * This implementation sets
 * the <code>checkbox.labelLevel</code> 
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively.
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} label properties.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getLabelProps = function() {

    // First see if the super class wants to contribute to the props.
    // Let selectBase add the htmlFor property
    //
    var props = this._inherited("_getLabelProps", arguments);
    props.level = this._theme.getMessage("checkbox.labelLevel", null, 3);
    return props;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.checkbox.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.checkbox.event.submit.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.checkboxGroup");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.checkboxGroup");

/**
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */

woodstock4_3._dojo.provide("woodstock4_3.widget._base.checkedGroupBase");

woodstock4_3._dojo.require("woodstock4_3.widget._base.labeledBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.checkedGroupBase
 * @extends woodstock4_3.widget._base.labeledBase
 * @class This class contains functions for widgets that extend checkedGroupBase.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {int} columns 
 * @config {Array} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {boolean} readOnly Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.checkedGroupBase",
    woodstock4_3.widget._base.labeledBase);

/**
 * Helper function to add elements with Object literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled 
 * @config {Array} columns 
 * @config {Array} contents 
 * @config {Object} label
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedGroupBase.prototype._addContents = function(props) {   

    // The label needs special handling because the labeledBase super
    // class is handling the update and creation of the label. The dependence
    // is on "this._labelContainer". We do not want to delete it and
    // create or clone a new one. We need to save the current 
    // "_labelContainer" instance and add it to the new contents.
    //
    // Obviously the whole table layout of rb's and cb's is fragile,
    // and inefficient, especially with respect to the label which
    // resides in the first cell of the first column.
    // Use the "rowspan" attribute to simplify the problem.
    // 
    // This implementation is inefficient if the only properties 
    // of one of the content objects is changing. Therefore 
    // individual rb's and cb's  should be manipulated directly and 
    // not by a
    //
    // "{contents : [ ...,{ ...checked: false...},...]"
    //
    if (props != null && props.contents != null) {

	// Get rid of the current contents
	// An empty "props.content", "{ content : [] }" this will remove
	// all the contents.
	//
	this._widget._removeChildNodes(this._tbodyContainer);

	// In case we are updating, use the columns on the group
	// instance, it may not be specified in props in a call
	// to "setProps".
	//
	var columns = this.columns != null && this.columns > 0
		? this.columns
		: 1;
	if (props.columns != null && props.columns > 0) {
	    columns = props.columns;
	}
	    
        var length = props.contents.length;
        var rows = (length + (columns-1))/columns;

	// Remember if the group is disabled or not.
	// We need to check "this" in case we are being called from
	// "setProps".
	//
        var disabled = this.disabled == null ? false : this.disabled;
	if (props.disabled != null) {
	    disabled = props.disabled;
	}
        var itemN = 0;
	var row = 0;

	// If we have content and there is a label, create the first
	// row with the label. This will set "row" to 1, so the
	// following loop over rows and columns starts at the second
	// row.
	//
	// Note that the label will not exist as a widget the first time
	// through when the group widget is being created.
	//
	var labelNode = null;
	if (rows > 0 && this.label) {
	    var rowContainer = this._rowContainer.cloneNode(false);
	    labelNode = this._labelNode.cloneNode(false);
	    labelNode.id = "label";
	    rowContainer.appendChild(labelNode);
	    labelNode.setAttribute("rowSpan", rows);
	    labelNode.appendChild(this._labelContainer);

	    // Special case for first row, when we have a label.
	    //
	    this._tbodyContainer.appendChild(rowContainer);
	    for (var i = 0; i < columns; ++i, ++itemN) {
		var content = this._contentNode.cloneNode(false);
		rowContainer.appendChild(content);
		// Set disabled.                   
		//
		props.contents[itemN].disabled = disabled;;
		// Add child to the group.
		//
		this._widget._addFragment(
		    content, props.contents[itemN], "last");
	    }
	    row = 1;
	}

        for (; row < rows; row++) {
	    var rowContainer = this._rowContainer.cloneNode(false);
	    this._tbodyContainer.appendChild(rowContainer);
            for (var column = 0; column < columns; column++, itemN++) {
		// Clone <td> node.
		// There could be issues here by not doing a deep clone.
		// There could be "decorative" elements in the td node
		// in the template, but the shallow clone, ignores that.
		//
		var content = this._contentNode.cloneNode(false);
		// In case we have a sparse last row
		//
                if (itemN < length) {
                    rowContainer.appendChild(content);
                    // Set disabled.                   
		    //
                    props.contents[itemN].disabled = disabled;
                   
                    // Add child to the group.
		    //
                    this._widget._addFragment(
			content, props.contents[itemN], "last");
                } else {
		    // We could try using "colspan" on a sparse last row.
		    //
                    rowContainer.appendChild(content);
		}
            }
        }
    }
    return true;
};

/**
 * Return an Object Literal of label properties desired
 * by the checkedGroupBase widget.
 * <p>
 * This implementation adds the <code>htmlFor</code> property assigning
 * it the input element in the first row and the first column of the group.
 * </p>
 * @return {Object} object with the <code>htmlFor</code>
 * property set.
 * @private
 */
woodstock4_3.widget._base.checkedGroupBase.prototype._getLabelProps = function() {
    // Let the super class contribute
    //
    var props = this.inherited("_getLabelProps", arguments);
    // Subclasses can override this value
    //
    props.level = 2;
    return props;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.checkedGroupBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.     
    if (this.columns != null) { props.columns = this.columns; }
    if (this.contents != null) { props.contents = this.contents; }    
    if (this.disabled != null) { props.disabled = this.disabled; }   
    if (this.id != null) { props.id = this.id; }
    if (this.name != null) { props.name = this.name; }
    if (this.readOnly != null) { props.readOnly = this.readOnly; }  

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedGroupBase.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {                    
        this._contentNode.id = this.id + "_contentNode";
        this._divContainer.id = this.id + "_divContainer";
        this._rowContainer.id = this.id + "_rowContainer";
        this._labelNode.id = this.id + "_labelNode";
        this._tableContainer.id = this.id + "_tableContainer";   
        this._tbodyContainer.id = this.id + "_tbodyContainer";     
    }

    // Show label.
    if (this.label) {
        this._common._setVisibleElement(this._labelNode, true);
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.checkedGroupBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedGroupBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set contents.    
    if (props.contents) { // || props.disabled != null) {              
        this._addContents(props);

	// We have a control in the first cell and
	// the label's htmlFor value is not being updated explicityly and
	// we have or are creating a label, update the htmlFor 
	// label property.
	//
	if (props.contents.length != 0 && props.contents[0].id != null &&
		(props.label == null || props.label.htmlFor == null) &&
		this.label != null && this.label.id != null) {
	    if (props.label == null) {
		props.label = {};
	    }
	    var widget = this._widget.getWidget(props.contents[0].id);
	    if (widget != null && widget.getInputElement) {
		props.label.htmlFor = widget.getInputElement().id;
	    }
	}
    } else 
    if (props.disabled != null && this.contents != null) {

	// If props.contents is not null, then disabled will
	// be handled when the contents are updated.
	// Otherwise iterate over the contents disabling them
	//
	for (var i = 0; i < this.contents.length; i++) {
	    var contentWidget = this._widget.getWidget(this.contents[i].id);
	    if (contentWidget) {
		contentWidget.setProps({disabled: props.disabled});
	    }
        }
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");

/**
 * This function is used to construct a checkboxGroup widget.
 *
 * @constructor
 * @name woodstock4_3.widget.checkboxGroup
 * @extends woodstock4_3.widget._base.checkedGroupBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the checkboxGroup widget.
 * <p>
 * The widget is used to layout an array of checkboxes. To group checkboxes
 * properly, each widget id must be unique, but the name property value is the
 * same.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cbg1",
 *       contents: [{ 
 *         id: "cb1",
 *         label: { value: "This is checkbox 1" },
 *         widgetType: "checkbox"
 *       }, {
 *         id: "cb2",
 *         label: { value: "This is checkbox 2" },
 *         widgetType: "checkbox"
 *       }],
 *       label: { value: "This is a checkbox group" },
 *       widgetType: "checkboxGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the radio button, the
 * checkbox group is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cbg1",
 *       contents: [{ 
 *         id: "cb1",
 *         label: { value: "This is checkbox 1" },
 *         widgetType: "checkbox"
 *       }, {
 *         id: "cb2",
 *         label: { value: "This is checkbox 2" },
 *         widgetType: "checkbox"
 *       }],
 *       label: { value: "This is a checkbox group" },
 *       widgetType: "checkboxGroup"
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "rb1",
 *       label: { value: "Toggle Checkbox Group State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "radioButton"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cbg1"); // Get checkbox group
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the radio button, the checkbox group is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cbg1",
 *       contents: [{ 
 *         id: "cb1",
 *         label: { value: "This is checkbox 1" },
 *         widgetType: "checkbox"
 *       }, {
 *         id: "cb2",
 *         label: { value: "This is checkbox 2" },
 *         widgetType: "checkbox"
 *       }],
 *       label: { value: "This is a checkbox group" },
 *       widgetType: "checkboxGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "rb1",
 *       label: { value: "Refresh Checkbox Group" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "radioButton"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cb1"); // Get checkbox group
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a checkbox group using the 
 * refresh function. The execute property of the refresh function is used to 
 * define the client id which is to be submitted and updated server-side. As the
 * user types in the text field, the input value is updated server-side and the 
 * checkbox group label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cbg1",
 *       contents: [{ 
 *         id: "cb1",
 *         label: { value: "This is checkbox 1" },
 *         widgetType: "checkbox"
 *       }, {
 *         id: "cb2",
 *         label: { value: "This is checkbox 2" },
 *         widgetType: "checkbox"
 *       }],
 *       label: { value: "This is a checkbox group" },
 *       widgetType: "checkboxGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Checkbox Group Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cbg"); // Get checkbox group
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "cbg") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.checkboxGroup.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {int} columns 
 * @config {Array} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {boolean} readOnly Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.checkboxGroup", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.checkedGroupBase ], {
    // Set defaults.
    _widgetType: "checkboxGroup" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.checkboxGroup.event =
        woodstock4_3.widget.checkboxGroup.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkboxGroup_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkboxGroup_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkboxGroup_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkboxGroup_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.checkboxGroup.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.columns > 1)
        ? this._theme.getClassName("CBGRP_HORIZ", "")
        : this._theme.getClassName("CBGRP_VERT", "");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.checkboxGroup.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.dropDown");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.dropDown");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");


woodstock4_3._dojo.provide("woodstock4_3.widget._base.selectBase");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.labeledBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.selectBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class defines functions and properties for
 * widgets based on the "select" HTML element.
 *
 * <h3>Dojo attach points</h3>
 * A <code>selectBase</code>  subclass's template must
 * define the following attach point identifiers.
 * <ul>
 * <li><code>_listContainer</code> - the <code>select</code> element.</li>
 * <li><code>_optionNode</code> - the element to clone for an
 * <code>option</code> element.
 * </li>
 * <li><code>_optionGroupContainer</code> - the element to clone for an
 * <code>optGroup</code> element.</li>
 * <li><code>_memberOptionNode</code> - the element to clone for an
 * <code>option</code> in an <code>optGroup</code> element.</li>
 * </ul>
 * <h3>The <code>options</code> array property</h3>
 * The <code>options</code> array property defines the contents of the
 * <code>select</code> HTML element. The contents of the array are 
 * translated into <code>option</code> and <code>optGroup</code> HTML
 * elements. Each element of the <code>options</code> array property
 * is considered an object with the following properties. Note that
 * "NA" in the following table means "not applicable", meaning there
 * is no HTML element property counterpart.
 * <p>
 * <table border="1px">
 * <tr><th>Property</th><th>Type</th><th>Description</th>
 * <th>HTML option or optGroup element property</th></tr>
 * <tr><td>label</td><td>String</td><td>
 * The text that appears as the choice in the rendered <code>option</code>
 * HTML element. (See note[1] below)
 * </td><td>label</td></tr>
 * <tr><td>value</td><td>String</td><td>
 * The value for this option. It is submitted in a request if this option is
 * selected.</td><td>value</td></tr>
 * <tr><td>separator</td><td>boolean</td><td>
 * If true this option represents a separator and cannot be selected.
 * </td><td>NA</td></tr>
 * <tr><td>group</td><td>boolean</td><td>
 * If true this option represents a group and cannot be selected. An
 * <code>optGroup</code> HTML element is rendered to represent this option.
 * </td><td>NA</td></tr>
 * <tr><td>escape</td><td>boolean</td><td>
 * If false the label string will be evaluated as HTML markup.
 * </td><td>NA</td></tr>
 * <tr><td>selected</td><td>boolean</td><td>
 * If true this option will be initially selected. 
 * </td><td>selected</td></tr>
 * <tr><td>disabled</td><td>boolean</td><td>
 * If true this option will be initially disabled and cannot be selected.
 * </td><td>disabled</td></tr>
 * <tr><td>title</td><td>String</td><td>
 * The HTML title attribute text.</td><td>title</td></tr>
 * </table>
 * </p>
 * <p>
 * The option object may also define the javascript event properties
 * suitable for an <code>option</code> or <code>optGroup</code> HTML
 * element and they will be assigned to the element's DOM node.
 * </p>
 * <p>
 * <ul>
 * <li>Note[1] Browser runtime behavior, contrary to the HTML and DOM
 * specifications, indicates that the DOM <code>label</code> property
 * is not rendered and only the <code>text</code> property is rendered.
 * This means that the <code>label</code> property of an
 * <code>options</code> array element will
 * be assigned to an <code>option</code> or <code>optGroup</code>
 * DOM node's <code>text</code> property.</li>
 * </ul>
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled If true the select element is disabled.
 * @config {Object} label The properties for a label widget.
 * @config {boolean} labelOnTop If false the label will appear to the left
 * of the <code>select</code> element, aligned with the first list option. 
 * @config {Object} options An array of Objects that represent the 
 * <code>select</code> element's <code>option</code> and 
 * <code>optgroup</code> elements. @see #_setOptions.
 * @config {boolean} required If true the a selection is required.
 * @config {boolean} valid If true the widget has a valid selection.
 * @config {String} width This value will be assigned to the 
 * <code>_listContainer.style.width</code> attribute to set the width of the
 * select element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.selectBase", 
	woodstock4_3.widget._base.labeledBase, {
    // Set defaults
    constructor: function() {
	this.disabled = false;
	// Flag to prevent blank entries in the drop down for the original empty
        // dojo attach point nodes. See "_setOptions".
	this._alreadyRemoved = false;
    }
});

/**
 * This function is called by the <code>_onChangeCallback</code>
 * function to set the option element's className attribute with appropriate
 * selected and disabled selectors.
 * <p>
 * This method calls <code>this._getOptionClassName</code> once for each
 * option. Subclasses should implement <code>_getOptionClassName</code>
 * in order to assign the appropriate CSS class name, to reflect a disabled or
 * selected option and any other properties due to an onChange event.
 * <p>
 * See <code>_getOptionClassName</code> for the default behavior.
 * </p>
 * <p>
 * Before this method calls <code>this._getOptionClassName</code>, it has
 * addressed an IE issue where disabled options can be selected, and has
 * deslected, selected disabled options.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._changed = function() {
    var options = this._listContainer.options;

    // IE allows disabled options to be selected. Ensure that
    // disabled options never appear as selected options.
    //
    if (woodstock4_3._base.browser._isIe()) {
	for (var i = 0; i < options.length; ++i) {
	    if (options[i].disabled == true && options[i].selected == true) {
		if (this._listContainer.multiple == true) {
		    options[i].selected = false;
		} else {
		    // Don't we need to set the disabled option's
		    // selected to false too ?
		    //
		    this._listContainer.selectedIndex = -1;
		}
	    }
	    // Only set the option if its value is different
	    // than what is returned. Note that if this method
	    // returns null, the option may not visually reflect
	    // the true state.
	    //
	    var cn = this._getOptionClassName(options[i]);
	    if (cn != null && options[i].className != cn) {
		options[i].className = cn;
	    }
	} 
    } else {
	for (var i = 0; i < options.length; ++i) { 
	    // Only set the option if its value is different
	    // than what is returned. Note that if this method
	    // returns null, the option may not visually reflect
	    // the true state.
	    //
	    var cn = this._getOptionClassName(options[i]);
	    if (cn != null && options[i].className != cn) {
		options[i].className = cn;
	    }
	}
    }
    return true;
};

/**
 * This method copies option properties from <code>fromOption</code> to 
 * <code>toOption</code>. Note that <code>fromOption</code> is assumed to
 * be a DOM <code>option</code> element. There is not a one to one mapping
 * of properties by name from a DOM <code>option</code> element to a
 * widget's <code>options</code> array <code>option</code> array element.
 * The properties that are set in <code>_setOptionProps</code> and
 * <code>_setGroupOptionProps</code> are copied.
 * @param {Object} toOption An <code>Object</code> to receive
 * <code>selectBase</code> option properties.
 * @param {Object} fromOption An <code>option</code> or <code>optgroup</code>
 * element DOM node to copy <code>selectBase</code> option properties from.
 * @return {boolean} Returns true.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._copyOption = function(toOption, fromOption) {
    var domhandlers = [ "onblur", "onchange", "onclick", "ondblclick",
	"onfocus", "onkeydown", "onkeypress", "onkeyup", "onmousedown",
	"onmouseout", "onmouseover", "onmouseup", "onmousemove", "onresize"];

    var widgethandlers = [ "onBlur", "onChange", "onClick", "onDblClick",
	"onFocus", "onKeyDown", "onKeyPress", "onKeyUp", "onMouseDown",
	"onMouseOut", "onMouseOver", "onMouseUp", "onMouseMove", "onResize"];

    for (var i = 0; i < domhandlers.length; ++i) {
	if (fromOption[domhandlers[i]]) {
	    toOption[widgethandlers[i]] = fromOption[domhandlers[i]];
	}
    }
    toOption.label = fromOption.label;
    toOption.value = fromOption.value;
    toOption.escape = fromOption.escape;
    toOption.disabled = fromOption.disabled;
    toOption.defaultSelected = fromOption.defaultSelected;
    toOption.selected = fromOption.selected;
    toOption.title = fromOption.title;
    toOption.className = fromOption.className;
    toOption.group = fromOption.group;
    if (toOption.group == true) {
	toOption.options = this._copyOptions(fromOption);
    }
    return true;
};

/**
 * Returns an <code>options</code> array of <code>option</code> objects
 * as defined by <code>selectBase</code> (see selectBase._setOptions).
 * <code>domNode</code> is assumed to be a <code>select</code> element
 * or an <code>optgroup</code>.
 * <p>
 * There is not a one to one mapping of properties by name from a DOM
 * <code>option</code> or <code>optgroup</code> element to a
 * widget's <code>options</code> array <code>option</code> object array element.
 * The properties that are set in <code>_setOptionProps</code> and 
 * <code>_setGroupOptionProps</code> are copied to <code>Object</code>s
 * and returned as the elements of the returned array. The returned
 * array is equivalent to the <code>options</code> array used to create
 * a selectBase subclass, like listbox or dropdown.
 * @param {Object} domNode A <code>select</code> element DOM node or an
 * <code>optgroup</code> DOM node.
 * @return {Object} An array of <code>selectBase</code> options. If 
 * <code>domNode</code> has no child nodes, an empty array is returned.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._copyOptions = function(domNode) {
    var newoptions = [];
    if (!domNode.hasChildNodes()) {
	return newoptions;
    }
    var len = domNode.childNodes.length;
    for (var j = 0; j < len; ++j) {
	newoptions[j] = new Object();
	if (domNode.childNodes != null) {
	    this._copyOption(newoptions[j], domNode.childNodes[j]);
	}
    }
    return newoptions;
};

/**
 * Return an Object Literal of label properties desired by this widget.
 * <p>
 * This implementation adds the <code>htmlFor</code> property with
 * <code>this._listContainer.id</code>.
 * </p>
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._getLabelProps = function() {
    var props = this._inherited("_getLabelProps", arguments);

    props.htmlFor = this._listContainer.id;
    return props;
};

/**
 * This function returns the CSS class name for an HTML <code>option</code> or
 * <code>optgroup</code> element.
 * <p>
 * This implementation returns <code>option.className</code>
 * This method should be overridden by subclasses.
 * </p>
 *
 * @param {Object} option Key-Value pairs of properties.
 * @return {String} The HTML option element class name.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._getOptionClassName = function(option) {
    // Make sure that if a subclass does not implement this method
    // that it causes no change to option.
    //
    return option.className;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.selectBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Get properties.
    // Should we return the default theme value for labelOnTop
    // if "this.labelOnTop" has not been set ?
    //
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.width != null) { props.width = this.width; }

    // After widget has been initialized, get actual select element state
    // Note that the options that exist on the select element will
    // not include the original "options" props that are not
    // HTML option or HTML optgroup attributes.
    //
    if (this._isInitialized() == true && this._listContainer != null) {
	// We need to copy the options from this._listContainer
	// or else they will be destroyed if passed back to addProps.
	//
	var opts = this._copyOptions(this._listContainer);
        props.options = opts;
    } else if (this.options != null) {
        props.options = this.options;
    }
    return props;
};

/**
 * This function is used to obtain the underlying HTML select element.
 *
 * @return {Node} The HTML select element.
 */
woodstock4_3.widget._base.selectBase.prototype.getSelectElement = function() {
    return this._listContainer;
};

/**
 * This function is used to obtain the index of the selected option.
 *
 * @return {int} The selected index of underlying HTML select element.
 */
woodstock4_3.widget._base.selectBase.prototype.getSelectedIndex = function() {
    return this._listContainer.selectedIndex;
};

/**
 * This function is used to obtain the <code>label</code> attribute of
 * the selected option.
 * <p>
 * If no option is selected, this function returns null.
 * </p>
 * <p>
 * If the underlying select element's <code>multiple</code> attribute is true,
 * this method returns the first selected option's label.
 * </p>
 * 
 * @return The label of the selected option, or null if none is selected. 
 * @return {String} The label attribute of the selected option.
 */
woodstock4_3.widget._base.selectBase.prototype.getSelectedLabel = function() { 
    var index = this._listContainer.selectedIndex; 

    if (index == -1) { 
        return null; 
    } else { 
	return this._listContainer.options[index].label;
    }
};

/**
 * This function is used to obtain the <code>value</code> attribute
 * of the selected option.
 * <p>
 * If no option is selected, this function returns null. 
 * </p>
 * <p>
 * If the underlying select element's <code>multiple</code>" attribute is true,
 * this method returns the first selected option's value.
 * </p>
 *
 * @return {String} The selected option value or null if none is selected. 
 */
woodstock4_3.widget._base.selectBase.prototype.getSelectedValue = function() { 
    var index = this._listContainer.selectedIndex; 
    if (index == -1) { 
        return null; 
    } else { 
        return this._listContainer.options[index].value; 
    }
};

/**
 * This function is invoked for the select element's <code>onchange</code>
 * event.
 * <p>
 * If <code>this.disabled</code> is true, return false.<br/>
 * This method calls the handler defined by the <code>props.onChange</code>
 * property. If that handler returns false, false is returned. If the
 * <code>props.onChanged</code> handler returns true, then
 * <code>this._changed</code> is called and its return value is returned.
 * </p>
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._onChangeCallback = function(event) {
    if (this.disabled == true) {
        return false;
    }

    // If function returns false, we must prevent the auto-submit.
    //
    var result = (this._listContainer._onchange)
        ? this._listContainer._onchange(event) : true;
    if (result == false) {
        return false;
    }

    // Set style classes.
    return this._changed(event);
};

/**
 * This function is used to fill in remaining template properties, after the
 * <code>_buildRendering</code> function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._listContainer.id = this.id + "_list";
	this._listContainer.name = this._listContainer.id;
    }

    // Set public functions.

    /** @ignore */
    this._domNode.getSelectedValue = function() { 
	return woodstock4_3.widget.common.getWidget(this.id).getSelectedValue();
    };
    /** @ignore */
    this._domNode.getSelectedLabel = function() { 
	return woodstock4_3.widget.common.getWidget(this.id).getSelectedLabel();
    };
    /** @ignore */
    this._domNode.getSelectElement = function() { 
	return woodstock4_3.widget.common.getWidget(this.id).getSelectElement();
    };

    // Set events.
    this._dojo.connect(this._listContainer, "onchange", this, 
	"_onChangeCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set the <code>option</code> properties on the
 * <code>optgroup</code> HTML DOM element, <code>element</code>.
 * <p>
 * This method assigns the <code>option.label</code>,
 * <code>option.disabled</code>, <code>option.title</code> properties
 * to <code>element</code>, non HTML properties defined in 
 * <code>option</code>, calls <code>this._setEventProps</code>
 * and then calls <code>this._getOptionClassName</code>. 
 * Subclasses should implement <code>_getOptionClassName</code> if the default
 * behavior of <code>selectBase._getOptionClassName</code> is not appropriate.
 * Subclasses usually subclass this method to provide subclass specific 
 * CSS class names and properties. For all the properties defined in the 
 * option array see <code>_setOptions</code>.
 * </p>
 * @param {Node} element The optgroup DOM node
 * @param {Object} option Key-Value pairs of properties for the optgroup node.
 * @config {boolean} disabled If true the optgroup is disabled.
 * @config {String} label The optgroup choice text.
 * @config {String} title The optgrp HTML title attribute value.
 * @config {boolean} escape If false the label string will be evaluated
 * as HTML markup.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._setGroupOptionProps =
	function(element, option) {
    element.label = option.label;
  
    if (option.disabled != null) {
        element.disabled = option.disabled;
    }

    if (option.title != null) {
	element.title = option.title;
    }

    // optGroup elements can have events.
    this._setEventProps(element, option);

    // Non HTML properties
    //
    if (option.escape != null) {
	element.escape = option.escape;
    }
    element.group = true;

    // Do this last so that all other properties will have been set.
    // Note that if this does return null, which it shouldn't
    // then the visual appearance may not reflect the 
    // actual state.
    //
    var cn =  this._getOptionClassName(element);
    if (cn != null) {
	element.className = cn;
    }
    return true;
};

/**
 * This function is used to set the <code>option</code> properties on the
 * <code>option</code> HTML DOM element, <code>element</code>.
 * <p>
 * This method assigns the <code>option.label</code>,
 * <code>option.disabled</code>, <code>option.title</code> properties
 * to <code>element</code>, non HTML properties defined in 
 * <code>option</code>, calls <code>this._setEventProps</code>
 * and then calls <code>this._getOptionClassName</code>. 
 * Subclasses should implement <code>_getOptionClassName</code> if the default
 * behavior of <code>selectBase._getOptionClassName</code> is not appropriate.
 * Subclasses usually subclass this method to provide subclass specific CSS
 * class names and properties. For all the properties defined in the option
 * array see <code>_setOptions</code>
 * </p>
 * @param {Node} element The <code>option</code> HTML element DOM node
 * @param {Object} option Key-Value pairs of properties for the option node.
 * @config {boolean} disabled If true the option is disabled.
 * @config {String} label The option choice text.
 * @config {String} title The value of the option's HTML <code>title</code>
 * attribute.
 * @config {boolean} escape If false the label string will be evaluated
 * as HTML markup.
 * @config {boolean} separator If true this option acts as a separator.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._setOptionProps = function(element, option) {
    element.value = option.value;

    // If option.escape is true, we want the text to be displayed
    // literally. To achieve this behavior, do nothing.
    // If option.escape is false, we want any special sequences in the text 
    // (e.g., "&nbsp;") to be displayed as evaluated (i.e., unescaped).
    //
    // This is because it is assumed that the value of the property has 
    // already been escaped. This may be true if the value was set in the 
    // HTML response. This is why we call "unescapeHTML".
    //
    // However it may not be true if the text was set in javascript.
    //

    // Note that the HTML and ECMA JS DOM language binding indicates
    // that the "label" property should be used. However browsers do not
    // respect those specs. In fact label does not even work at all.
    // Setting the string on label results in no string being displayed.
    //
    var textToUse = option.label;
    if (option.escape != null && option.escape == false) {
	// Prototype method.
	element.text = this._proto._unescapeHTML(textToUse);
    } else {
	element.text = textToUse;
    }
    // We must record the state of the original option
    //
    element.label = option.label;

    // Never allow a disabled option to be spedified as selected
    // This is only a problem if an option is initially selected
    // and disbled on most browsers since once disabled it cannot
    // be selected except for IE. For IE it is ensured that a
    // disabled option cannot be "selected" in the
    // "change" method.
    //
    if (option.disabled != null && option.disabled == true) {
	option.selected = false;
    }

    element.defaultSelected = option.selected != null ? option.selected : false;

    // When creating new options, defaultSelected should also be set.
    // Actually only "defaultSelected" should be set and set only once
    // since it is supposed to be the "initial" value.
    // However all options are always recreated in this implementation
    // when this method is called by _setOptions.
    //
    element.selected = element.defaultSelected;
	
    element.disabled = option.disabled != null ? option.disabled : false;

    if (option.title != null) {
	element.title = option.title;
    }

    // option elements can have events.
    this._setEventProps(element, option);

    // In order for subclasses to return the appropriate
    // selector all the properties in the option properties array must
    // be placed on the HTML element option and optgroup instances.
    //
    // This is because "_getOptionClassName" is called in two different
    // contexts. It is called here where both the HTML option or
    // optgroup element is available and the original option properties 
    // are known, and during the "onchange" event where only the 
    // HTML option element is known. Therefore if the choice of selector 
    // is based on whether or not an option is a "separator" then, 
    // the separator state must be available from the HTML option or optgroup
    // element instance.
    //
    // This issue could be mitigated by using descendant selectors
    // vs. literal selectors. In other words, if every time an option
    // became disabled, the className attribute would be set (or merged) with 
    // Dis. A selector could be defined as ".Lst option.Dis" which would
    // implicitly match. If attribute selectors worked we could use
    // that instead. Either strategy would eliminate the need for
    // methods like "_getOptionClassName".
    //
    // Only set non HTML element props if we have to.
    //
    if (option.separator != null && option.separator == true) {
	element.separator = option.separator;
	element.disabled = true;
    }
    if (option.escape != null && option.escape == false) {
	element.escape = option.escape;
    }
    element.group = false;

    // Call _getOptionClassName only after setting HTML props
    // and option props on element.  Note that if this method
    // returns null, the option may not visually reflect
    // the true state.
    //
    var cn = this._getOptionClassName(element);
    if (cn != null) {
	element.className = cn;
    }
    return true;
};

/**
 * This function replaces all <code>option</code> and <code>optgroup</code>
 * elements in the HTML <code>select</code> element,
 * <code>this._listContainer</code>.
 * <p>
 * All options are replaced with the options specified in <code>props</code>.
 * If <code>props</code> is null, <code>false</code>
 * is returned and no change to the <code>select</code> element occurs. If
 * <code>props</code> contains no properties all options in the 
 * <code>select</code> element are removed.
 * </p>
 * @param {Object} props Key-Value pairs of option properties.
 * @config {String} label The text that appears as the choice in the 
 * rendered <code>option</code>.
 * @config {String} value The value for this option, which is submitted
 * in a request if this option is selected.
 * @config {boolean} separator If true this option represents a separator and
 * cannot be selected.
 * @config {boolean} group If true this option represents a group and
 * cannot be selected.
 * @config {boolean} escape If false the label string will be evaluated as
 * HTML markup.
 * @config {boolean} selected If true this option will be initially
 * selected.
 * @config {boolean} disabled If true this option will be initially 
 * disabled and cannot be selected.
 * @config {String} title The HTML title attribute text.
 * @return {boolean} true if successful; otherwise, false.
 * </p>
 * <p>
 * The option object may also define javascript event properties suitable for
 * an <code>option</code> or <code>optGroup</code> HTML element.
 * </p>
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._setOptions = function(props) {
    if (props == null) {
	return false;
    }

    // Remove all existing options
    //
    // Note: this has to be done. Otherwise, you'll get blank entries in the 
    // drop down for the original empty dojo attach point nodes.
    //

    if (!this._alreadyRemoved) {
        this._listContainer.removeChild(this._optionNode); 
        this._optionGroupContainer.removeChild(this._memberOptionNode);
        this._listContainer.removeChild(this._optionGroupContainer);
        this._alreadyRemoved = true;
    }

    // Cleaning up the old options
    //
    while (this._listContainer.firstChild) {
        this._listContainer.removeChild(this._listContainer.firstChild);
    }

    var thisNode;
    for (var i = 0; i < props.options.length; i++) {

	var pOption = props.options[i];

	var isie = woodstock4_3._base.browser._isIe();
	if (pOption.group == null || pOption.group == false) {

	    // For some reason, ie is prone to painting problems (esp. after a 
	    // style change) when using the DOM to populate options, but 
	    // apparently not when the options are in a group
	    //
	    // There is no working "clone" on "option" nodes in IE
	    // Manually get the attributes and then the "text"
	    // DOM attribute. I don't think an option can have HTML
	    // markup in the body, at least literally in the template
	    // so this strategy should be be sufficient enough to get
	    // the template's option attributes
	    //
	    if (isie) {
		thisNode = new Option();
		var len = this._optionNode.attributes.length;
		for (var j = 0; j < len; ++j) {
		    thisNode.setAttribute(this._optionNode.attributes[j].name,
			    this._optionNode.attributes[j].value);
		}
		// Need to manually do text, although this is likely null
		// in the template.
		//
		thisNode.text = this._optionNode.text;

	    } else {
		thisNode = this._optionNode.cloneNode(true);
	    }

	    // Set the properties on this option element
	    this._setOptionProps(thisNode, pOption);
	    
	    // Would it be better to create all the nodes and then
	    // add them to the "options" array or append then ?
	    // Granted this would mean iterating twice.
	    // But it might be better if something fails
	    // and if so leave the original list alone.
	    //

	    // Append this option node to the select
	    //
	    if (isie) {
		var idx = this._listContainer.options.length;
		var isSelected = thisNode.selected;
		this._listContainer.options[idx] = thisNode;

		// VERIFY THAT WE STILL NEED THIS
		//

		// explicitly set the selected property again!
		// this is necessary to work around a defect in 
		// some versions of IE6
		//
		this._listContainer.options[idx].selected = isSelected;
	    } else {
		this._listContainer.appendChild(thisNode);
	    }
	} else {
	    // group option optgroup
	    //
	    thisNode = this._optionGroupContainer.cloneNode(true);
	    
	    // Set the properties on this optgroup element
	    //
	    this._setGroupOptionProps(thisNode, pOption);
	    
	    // Add the option elements to this group
	    // Supports only one level
	    //
	    var thisSubNode;
	    for (var ix = 0; ix < pOption.options.length; ix++) {
		thisSubNode = this._memberOptionNode.cloneNode(true);
		this._setOptionProps(thisSubNode, pOption.options[ix]);
		thisNode.appendChild(thisSubNode); 
	    }

	    // Append this optgroup node to the select element
	    // only after constructing its option children
	    //
	    this._listContainer.appendChild(thisNode);
	}
    }
    return true;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._setProps = function(props) {
    if (props == null) {
	return null;
    }

    // A web app devleoper could return false in order to cancel the 
    // auto-submit. Thus, we will handle this event via the onChange
    // call back.
    //
    if (props.onChange) {
        // Set private function scope on DOM node.
        this._listContainer._onchange = (typeof props.onChange == 'string')
	    ? new Function("event", props.onChange) 
	    : props.onChange;

        // Must be cleared before calling _setEventProps() below.
        props.onChange = null;
    }

    if (props.disabled != null) {
	if (this._listContainer.disabled != props.disabled) {
	    this._listContainer.disabled = props.disabled;
	}
    }

    // Add option and optgroup elements in the select element
    //
    if (props.options) {
        this._setOptions(props);
    }

    if (props.width != null && props.width != "") {
	this._listContainer.style.width = props.width;
    }
    // Set more properties.
    this._setCommonProps(this._listContainer, props);
    this._setEventProps(this._listContainer, props);

    // Set remaining properties.
    //
    return this._inherited("_setProps", arguments);
};

/** 
 * This function is used to set the selected option.
 *
 * @param {int} index The selected index of underlying HTML select element.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.selectBase.prototype.setSelectedIndex = function(index) {
    if (index >= 0 && index < this._listContainer.options.length) {
        this._listContainer.selectedIndex = index;
    }
    return true;
};
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");

/**
 * This function is used to construct a dropDown widget.
 *
 * @constructor
 * @name woodstock4_3.widget.dropDown
 * @extends woodstock4_3.widget._base.selectBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the dropDown widget.
 * <p>
 * A dropDown widget can be configured as a jump menu. Jump menus immediately 
 * perform an action, such as opening a window or navigating to another page, 
 * when the user selects an item. This action is accomplished by submitting the 
 * form immediately when the user changes the selection. By default, the 
 * dropDown widget is not a jump menu. The submission of the form does not 
 * occur until the user activates another component such as a button in order 
 * to submit the form. To configure the dropDown widget to render a jump menu, 
 * set the submitForm property to true.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mnu1",
 *       label: { value: "This is a menu" },
 *       options: [{
 *         value: "BOS",
 *         label: "Boston"
 *       }, {
 *         value: "SFO",
 *         label: "San Francisco"
 *       }],
 *       widgetType: "dropDown"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * menu is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mnu1",
 *       label: { value: "This is a menu" },
 *       options: [{
 *         value: "BOS",
 *         label: "Boston"
 *       }, {
 *         value: "SFO",
 *         label: "San Francisco"
 *       }],
 *       widgetType: "dropDown"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Menu State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("mnu1"); // Get menu
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the menu is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mnu1",
 *       label: { value: "This is a menu" },
 *       options: [{
 *         value: "BOS",
 *         label: "Boston"
 *       }, {
 *         value: "SFO",
 *         label: "San Francisco"
 *       }],
 *       widgetType: "dropDown"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Menu" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("mnu1"); // Get menu
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a menu using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the menu
 * label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mnu1",
 *       label: { value: "This is a menu" },
 *       options: [{
 *         value: "BOS",
 *         label: "Boston"
 *       }, {
 *         value: "SFO",
 *         label: "San Francisco"
 *       }],
 *       widgetType: "dropDown"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Menu Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("mnu1"); // Get menu
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "mnu1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.dropDown.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} label Defines the widget and its properties to use for
 * a label.
 * @config {boolean} labelOnTop If true the label appears above the dropdown. 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Option selection is changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Option text is highlighted.
 * @config {Array} options See <code>selectBase._setOptions</code> or
 * <code>selectBase</code> overview for details on the elements of this array.
 * @config {int} size The number of option rows to display.
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm If true the dropDown performs like a
 * "jump menu" and submits the entire form when an option is selected.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title The a HTML title attribue for the <code>select</code> element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.dropDown", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.selectBase ], {
    // Set defaults.
    constructor : function() {
	this.labelOnTop = this._theme._getMessageBoolean("dropDown.labelOnTop", false);
	this.submitForm =  false;
	this.width = this._theme.getMessage("dropDown.width", null);
    },
    _widgetType: "dropDown" // Required for theme properties.
});

/**
 * This function is called by the <code>selectBase::onChangeCallback</code>
 * method in respsonse to an onchange event.
 * <p>
 * If this instance is configured as a "jump" menu,
 * <code>submitForm</code> is true, the containing form is submitted.
 * </p>
 * <p>
 * If this instance is configured as a standard "dropdown" menu, then
 * the superclass's <code>_changed</code> method is called.
 * See <code>selectBase._changed</code>.<br/>
 * See <code>selectBase.onChangeCallback</code>
 * </p>
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._changed = function() {

    // A null submitForm is the same as a standard menu.
    //
    if (this.submitForm != true) {
        // Drop down changed.
        return this._inherited("_changed", arguments);
    } else {

        // Jump drop down changed.
        var jumpDropdown = this._listContainer; 

        // Find the <form> for this drop down
        var form = jumpDropdown.form;
        if (form != null) { 
            this._submitterHiddenNode.value = "true";

            // Set style classes.
            var options = jumpDropdown.options;
            for (var i = 0; i < options.length; i++) { 
                options[i].className = this._getOptionClassName(options[i]);
            }
            form.submit();
        }
    }
    return true; 
};

/**
 * The dropDown defines the following event topics. The submit topics
 * are only valid if the <code>submitForm</code> property is true.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.dropDown.event =
        woodstock4_3.widget.dropDown.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for.*/
        beginTopic: "woodstock4_3_widget_dropDown_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for.*/
        endTopic: "woodstock4_3_widget_dropDown_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_dropDown_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_dropDown_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for.
	 * This topic is only valid if the <code>submitForm</code>
	 * property is <code>true</code>
	 */
        beginTopic: "woodstock4_3_widget_dropDown_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for.
	 * This topic is only valid if the <code>submitForm</code>
	 * property is <code>true</code>
	 */
        endTopic: "woodstock4_3_widget_dropDown_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element CSS class name.
 * <p>
 * Note: Selectors should be concatenated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var newClassName = this._theme.getClassName("DROPDOWN", "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Return an Object Literal of label properties desired by this widget.
 * <p>
 * This implementation sets
 * the <code>dropDown.labelLevel</code> and <code>dropDown.labelAlign</code>
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively. It sets the <code>htmlFor</code> attribute to 
 * <code>_listContainer.id</code>.
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getLabelProps = function() {

    var props = this._inherited("_getLabelProps", arguments);
    props.level = this._theme.getMessage("dropDown.labelLevel", null, 2);
    return props;
};

/**
 * Return a CSS selector for the dropDown label.
 * The chosen CSS selector is a function of 
 * the <code>labelOnTop</code> property and the whether or not the
 * dropdown is a "jump" dropDown. If <code>labelOnTop</code>
 * property is <code>true</code> null is returned. If the 
 * <code>labelOnTop</code> property is true, then the value of the
 * the of the theme keys, <code>MENU_JUMP_LABEL_ALIGN</code> or 
 * <code>MENU_STANDARD_LABEL_ALIGN</code> from the 
 * <code>styles</code> category are returned if the drop down is
 * a "jump" dropDown or plain dropDown, respectively.
 * @param {boolean} ontop The label on top state. In some contexts this state
 * must be passed to the method, for example when called from the 
 * <code>selectBase.setProps</code> method.
 * @return {String} A CSS selector for the dropDown label.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getLabelOnTopClassName = function(ontop) {
    if (ontop != null && ontop == true) {
	return null;
    }
    // Jumpmenu ?
    //
    return (this.submitForm != null && this.submitForm == true)
	? this._theme.getClassName("MENU_JUMP_LABEL_ALIGN", null)
        : this._theme.getClassName("MENU_STANDARD_LABEL_ALIGN", null);
};

/**
 * Return a CSS selector for _listContainer HTML element.
 * The chosen CSS selector is a function of 
 * the <code>disabled</code> property and whether or not the dropDown
 * is a "jump" dropDown. The returned selector is the value of a 
 * property defined in the <code>styles</code> category, as shown 
 * in the following table.
 * <table border="1px">
 * <tr><th>key</th><th>disabled</th><th>jump dropDown</th></tr>
 * <tr>
 * <td>MENU_STANDARD</td>
 * <td>false</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>MENU_STANDARD_DISABLED</td>
 * <td>true</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>MENU_JUMP</td>
 * <td>false</td>
 * <td>true</td>
 * </tr>
 * <tr>
 * <td>MENU_JUMP_DISABLED</td>
 * <td>true</td>
 * <td>true</td>
 * </tr>
 * </table>
 * @return {String} A CSS selector for the _listContainer HTML element.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getListContainerClassName =
        function(disabled, jumpmenu) {
    var key = "MENU_STANDARD";
    if (disabled == true) {
	if (jumpmenu == true) {
	    key = "MENU_JUMP_DISABLED";
	} else {
	    key = "MENU_STANDARD_DISABLED";
	}
    } else if (jumpmenu == true) {
	key = "MENU_JUMP";
    }
    return this._theme.getClassName(key, null);
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.dropDown.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Get properties.
    if (this.submitForm != null) { props.submitForm = this.submitForm; }

    return props;
};

/**
 * This function returns the CSS class name for an HTML option element,
 * based on the state of <code>element</code>. The CSS class name defined by 
 * the following theme keys from the <code>styles</code> theme category
 * is returned for the given states, based on 
 * the properties in <code>element</code>.
 * <p>
 * <table border="1px">
 * <tr><th>state</th><th>key</th></tr>
 * <tr><td>
 * separator == true</td><td>*_OPTION_SEPARATOR</td>
 * </tr><tr><td>
 * group == true</td><td>*_OPTION_GROUP</td>
 * </tr><tr><td>
 * disabled == true</td><td>*_OPTION_DISABLED</td>
 * </tr><tr><td>
 * selected == true and disabled == false</td>
 * <td>*_OPTION_SELECTED</td>
 * </tr><tr><td>
 * separator == false and group false<br/> 
 * 	and disabled == false<br/>
 *	and selected == false</td><td>*_OPTION</td>
 * </tr>
 * </table><br/>
 * Note:  * is <code>MENU_JUMP</code> for a jump menu and
 * <code>MENU_STANDARD</code> if not.
 * </p>
 * <p>
 * <p>
 * If <code>element</code> is null, the CSS class name for the theme
 * key "MENU_STANDARD_OPTION" is returned.
 * </p>
 * @param {Object} element An HTML option DOM node instance.
 * @config {boolean} separator If true, element is a option separator.
 * @config {boolean} group If true the element is a optGroup instance.
 * @config {boolean} disabled If true the element is disabled.
 * @config {boolean} selected If true the element is selected.
 * @return {String} The HTML option element CSS class name.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getOptionClassName = function(element) {

    var jumpmenu = this.submitForm != null && this.submitForm == true;

    var key = null;
    if (jumpmenu) {
	key = "MENU_JUMP_OPTION";
	if (element != null) {
	    if (element.separator == true) {
		key = "MENU_JUMP_OPTION_SEPARATOR";
	    } else
	    if (element.group == true) {
		key = "MENU_JUMP_OPTION_GROUP";
	    } else
	    if (element.selected == true) {
		key = "MENU_JUMP_OPTION_SELECTED";
	    }

	    // Disabled wins
	    //
	    if (element.disabled == true) {
		key = "MENU_JUMP_OPTION_DISABLED";
	    }
	}
    } else {
	key = "MENU_STANDARD_OPTION";
	if (element != null) {
	    if (element.separator == true) {
		key = "MENU_STANDARD_OPTION_SEPARATOR";
	    } else
	    if (element.group == true) {
		key = "MENU_STANDARD_OPTION_GROUP";
	    } else
	    if (element.selected == true) {
		key = "MENU_STANDARD_OPTION_SELECTED";
	    }

	    // Disabled wins
	    //
	    if (element.disabled == true) {
		key = "MENU_STANDARD_OPTION_DISABLED";
	    }
	}
    }
    return this._theme.getClassName(key);
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._submitterHiddenNode.id = this.id + "_submitter";
    }

    // Set the CSS class name for the select element.
    //
    var jumpmenu = this.submitForm != null && this.submitForm == true;
    var disabled = this.disabled != null && this.disabled == true;

    this._common._addStyleClass(this._listContainer, 
	this._getListContainerClassName(disabled, jumpmenu));
    
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.dropDown.prototype.setProps = function(props, notify) {
    if (props == null) {
	return this._inherited("setProps", arguments);
    }
	
    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    // We need to clear the selector of the _listContainer
    // when state changes from disabled to non disabled state
    // or from submitForm to standard dropdown.
    //
    var isjumpmenu = this.submitForm == true;
    var toggleJumpmenu = props.submitForm != null
        && props.submitForm != isjumpmenu;

    var isdisabled = this.disabled == true;
    var toggleDisabled = props.disabled != null
        && props.disabled != isdisabled;

    // Get current state of the classsname, and strip it 
    // and then add the classname for the new state.
    //
    if (toggleJumpmenu || toggleDisabled) {
	var cn = this._getListContainerClassName(isdisabled, isjumpmenu);
	this._common._stripStyleClass(this._listContainer, cn);

	cn = this._getListContainerClassName(
	    toggleDisabled ? props.disabled == true : isdisabled,
	    toggleJumpmenu ? props.jumpmenu == true : isjumpmenu);
	this._common._addStyleClass(this._listContainer, cn);
    }
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._setProps = function(props) {
    // If props is null, create one to pass default theme values
    // to the superClass via props.
    //
    if (props == null) {
	return this._inherited("_setProps", arguments);
    }

    var jumpmenu = props.submitForm != null && props.submitForm == true;

    // Add attributes to the hidden input for jump drop down.
    //
    if ((jumpmenu && this._submitterHiddenNode.name == null) || (jumpmenu &&
	    this._submitterHiddenNode.name != this._submitterHiddenNode.id)) {

	this._submitterHiddenNode.name = this._submitterHiddenNode.id;
	this._submitterHiddenNode.value = "false";
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.dropDown.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.dropDown.event.submit.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.hiddenField");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.hiddenField");

woodstock4_3._dojo.require("woodstock4_3.widget._base.selectBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a hiddenField widget.
 *
 * @constructor
 * @name woodstock4_3.widget.hiddenField
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the hiddenField widget.
 * <p>
 * TDB...
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a hidden field",
 *       widgetType: "hiddenField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * hidden field is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a hidden field",
 *       widgetType: "hiddenField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Hidden Field State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get hidden field
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the hidden field is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a hidden field",
 *       widgetType: "hiddenField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Hidden Field" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get hidden field
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a hidden field using the 
 * refresh function. The execute property of the refresh function is used to
 * define the client id which is to be submitted and updated server-side. When
 * the user clicks on the checkbox, the input value is updated server-side and 
 * the hidden field is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a hidden field",
 *       widgetType: "hiddenField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Change Hidden Field Value" },
 *       onClick="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get hidden field
 *       return widget.refresh("cb1"); // Asynchronously refresh while submitting checkbox value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "field1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.hiddenField.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} name
 * @config {String} value Value of input.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.hiddenField", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
    },
    _widgetType: "hiddenField" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.hiddenField.event =
        woodstock4_3.widget.hiddenField.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hiddenField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hiddenField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hiddenField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hiddenField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hiddenField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hiddenField_event_submit_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.hiddenField.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.name != null) { props.name = this.name; }
    if (this.value != null) { props.value = this.value; }

    return props;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.hiddenField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.name != null) { this._domNode.name = props.name; }
    if (props.value != null) {
        // An empty string is valid.
        this._domNode.value = props.value;
    }
    if (props.disabled != null) { 
        this._domNode.disabled = new Boolean(props.disabled).valueOf();
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.hiddenField.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.hiddenField.event.submit.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.hyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.hyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget._base.anchorBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");

/**
 * This function is used to construct a hyperlink widget.
 *
 * @constructor
 * @name woodstock4_3.widget.hyperlink
 * @extends woodstock4_3.widget._base.anchorBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the hyperlink widget.
 * <p>
 * The hyperlink widget creates an HTML anchor that submits form data. If the 
 * disabled attribute of the hyperlink is set to true, clicking on the hyperlink
 * will not generate a request and hence the form will not be submitted or the 
 * page will not navigate to the specified url.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "hyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * hyperlink is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "hyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Hyperlink State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get hyperlink
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the hyperlink is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "hyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Hyperlink" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get hyperlink
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an hyperlink using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the hyperlink text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "hyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Hyperlink Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get hyperlink
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "hyp1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.hyperlink.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} formId The id of the HTML form element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {Array} params The parameters to be passed during request.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.hyperlink", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.anchorBase ], {
    // Set defaults.
    _widgetType: "hyperlink" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.hyperlink.event =
        woodstock4_3.widget.hyperlink.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hyperlink_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hyperlink_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hyperlink_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hyperlink_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.hyperlink.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.disabled == true)
        ? this._theme.getClassName("HYPERLINK_DISABLED","")
        : this._theme.getClassName("HYPERLINK","");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.hyperlink.prototype._onClickCallback = function(event) {
    if (this.disabled == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the submit.
    var result = (this._domNode._onclick)
        ? this._domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    if (this.href) {
        return false;
    }
    event.preventDefault();
    
    // If a form id isnt provided, use the utility function to
    // obtain the form id.
    if (this.formId == null) {
        var form = this._widget._getForm(this._domNode);
        this.formId = (form) ? form.id : null;
    }
    return this._submitFormData(this.formId, this.params);
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.hyperlink.prototype._postCreate = function () {
    // If the href attribute does not exist, set "#" as the default value of the
    // DOM node.
    this._domNode.href = "#";

    // Create callback function for onClick event.
    this._dojo.connect(this._domNode, "onclick", this, "_onClickCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function submits the hyperlink if the hyperlink is enabled.
 *
 * @param {String} formId The id of the HTML form element.
 * @param {Array} params The parameters to be passed during request.
 * @return {boolean} false to cancel the JavaScript event.
 * @private
 */
woodstock4_3.widget.hyperlink.prototype._submitFormData = function (formId, params) {
    if (formId == null) {
        return false;
    }
    var form = document.getElementById(formId);
    if (form == null) {
        return false;
    }
    var oldTarget = form.target;
    var oldAction = form.action;

    // Obtain HTML element for tab and common task components.
    var link = document.getElementById(this.id);
    if (link == null) {
        link = this._domNode;
    }

    // Set new action URL.
    var prefix;
    if (form.action) {
        prefix = (form.action.indexOf("?") == -1) ? "?" : "&";
        form.action += prefix + link.id + "_submittedLink=" + link.id;
    }
        
    // Set new target.
    if (link.target && link.target.length > 0) {
        form.target = link.target;
    } else {
        form.target = "_self";
    }

    // Append query params to new action URL.
    if (params != null) {
        var x;
        for (var i = 0; i < params.length; i++) {
            x = params[i];
            form.action += "&" + params[i] + "=" + params[i + 1]; 
            i++;
        }
    }

    // Submit form.
    form.submit(); 

    // Restore target and action URL.
    if (link.target != null) {
        form.target = oldTarget;
        form.action = oldAction;
    }
    return false;        
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.hyperlink.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.image");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.image");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a image widget.
 *
 * @name woodstock4_3.widget.image
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the image widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} border
 * @config {String} className CSS selector.
 * @config {String} prefix The application context path of image.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} height 
 * @config {String} hspace 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} longDesc 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} src 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @config {String} vspace
 * @config {String} width
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.image", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.border = 0;
    },
    _highContrastMode: null, // Must be available to all image widgets.
    _widgetType: "image" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.image.event =
        woodstock4_3.widget.image.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_image_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_image_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_image_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_image_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.image.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.alt != null) { props.alt = this.alt; }
    if (this.icon != null) { props.icon = this.icon; }
    if (this.align != null) { props.align = this.align; }
    if (this.border != null) { props.border = this.border; }
    if (this.height != null) { props.height = this.height; }
    if (this.hspace != null) { props.hspace = this.hspace; }
    if (this.longDesc != null) { props.longDesc = this.longDesc; }
    if (this.src != null) { props.src = this.src; }
    if (this.vspace != null) { props.vspace = this.vspace; }
    if (this.width != null) { props.width = this.width; }

    return props;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.image.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Style must be set first (e.g., for absolute positioning) -- some style
    // properties may be overridden later by the combined image.
    this._setCommonProps(this._domNode, props);

    // Clear style properties if icon was previously set.
    if (props.src && this.icon) {
        this._domNode.style.border = "";
        this._domNode.style.backgroundImage = "";
        this._domNode.style.backgroundPosition = "";
        this._domNode.style.height = "";
        this._domNode.style.width = "";
    }

    // Set properties.
    if (props.icon) {
        // IE6 has issues with "png" images. IE6 png issue can be fixed but that
        // needs an outermost <span> tag. 
        //
        // <span style="overflow: hidden; width:13px;height:13px; padding: 0px;zoom: 1";>
        // <img src="dot.gif"
        //  style="filter:progid:DXImageTransform.Microsoft.AlphaImageLoader(src='testImage.png',sizingMethod='crop');
        //  margin-left:-0px;
        //  margin-top:-26px; border: none; height:39px;width:13px;"/>
        // </span>
        //
        // For now, skipping the combined image approach for IE6. Also need fix 
        // for Safari.
        var iconProps = this._theme.getImage(props.icon);
        if (iconProps == null) {
            console.debug("Error: theme icon '" + props.icon + "' not found.");
        } else {
            var mapKey = iconProps['map_key'];
            if (mapKey != null && !woodstock4_3._base.browser._isIe6()
                    && !this._widget._isHighContrastMode()) {
                // Note: Comparing height/width against "actual" properties is not a
                // valid test -- DOT images don't have a default size, for example.
                if (iconProps['top'] != null && iconProps['actual_height'] != null 
                        && iconProps['actual_width'] != null) {               
                    var transImage = this._theme.getImage("DOT");
                    var combinedImage = this._theme.getImage(mapKey);

                    // Set style properties.
                    this._domNode.style.border = "0";
                    this._domNode.style.backgroundImage = "url(" + combinedImage['src'] + ")";
                    this._domNode.style.backgroundPosition = "0px " + iconProps['top'] + "px";
                    this._domNode.style.height = iconProps['actual_height'] + "px";
                    this._domNode.style.width = iconProps['actual_width'] + "px";

                    // Set transparent image.
                    iconProps.src = transImage['src'];
                }
            }
            
            // Assign icon properties, even if combined image is not used.
            if (iconProps.alt != null) { this._domNode.alt = iconProps.alt; }
            if (iconProps.height != null) { this._domNode.height = iconProps.height; }
            if (iconProps.width != null) { this._domNode.width = iconProps.width; }            
            if (iconProps.src) {
                
                // Apply IE6 specific fix for png images if needed and obtain 
                // the modified style.
                props.style = this._setSrcProperty(iconProps);                                                
            }
        }
    } else {
        // Icon properties take precedence.
        if (props.alt != null) { this._domNode.alt = props.alt; }
        if (props.height != null) { this._domNode.height = props.height; }
        if (props.width != null) { this._domNode.width = props.width; }
        if (props.src) {
            
            // If context path is provided, then check whether the image has
            // context path already appended and if not, append it.            
            if (this.prefix) {
                props.src = 
                    woodstock4_3.widget.common._appendPrefix(this.prefix, props.src);                
            }                
            
            // Apply IE6 specific fix for png images if needed and obtain the 
            // modified style.
            props.style = this._setSrcProperty(props);
        }
    }
    if (props.align != null) { this._domNode.align = props.align; }
    if (props.border != null) { this._domNode.border = props.border; }
    if (props.hspace != null) { this._domNode.hspace = props.hspace; }
    if (props.longDesc != null) { this._domNode.longDesc = props.longDesc; }    
    if (props.vspace != null) { this._domNode.vspace = props.vspace; }

    // Set more properties.
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * Apply the src property on the dom element. If the image is an png image and 
 * the browser is IE 6 an IE6 specific fix needs to be applied and the style 
 * property has to be modified. If the png and IE6 combination is not true, then
 * no modification is done for the src property of the image and the style 
 * property.
 * 
 * @param {Object} props Key-Value pairs of properties.
 * @return (String} styleProp Modified value of the style property if the image
 * is a png image rendered on IE6.
 * @private
 */
woodstock4_3.widget.image.prototype._setSrcProperty = function(props) {    
    var styleProp = null;

    //IE6 PNG fix to handle transparent PNGs.
    if (props.src.indexOf(".png") > 0 && woodstock4_3._base.browser._isIe6()) {
        var width = (props.width > 0)
            ? props.width : this._theme.getMessage("Image.defaultWidth");
        var height =(props.height > 0)
            ? props.height : this._theme.getMessage("Image.defaultHeight"); 
        var styleMsg = this._theme.getMessage("Image.IEPngCSSStyleQuirk", 
          [width, height, props.src]);

        this._domNode.src = this._theme.getImage("DOT").src;

        if (props.style != null) {
            styleProp = props.style +";"+ styleMsg;
        } else {
            styleProp = (this.style != null) 
                ? this.style + ";" + styleMsg : styleMsg;                                          
        }
    } else {
        // While toggling between png and non-png type images need to adjust the 
        // style set on the dom element, since that is the property that 
        // contains the source for png images.
        if (woodstock4_3._base.browser._isIe6() 
                && this._domNode.style.cssText.indexOf(".png")>0) {
            styleProp = this.style;
        }
        this._domNode.src = props.src;  
    }
    return styleProp;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.image.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.imageHyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.imageHyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget.hyperlink");

/**
 * This function is used to construct a imageHyperlink widget.
 *
 * @constructor
 * @name woodstock4_3.widget.imageHyperlink
 * @extends woodstock4_3.widget.hyperlink
 * @class This class contains functions for the imageHyperlink widget.
 * <p>
 * The imageHyperlink widget creates an image surrounded by an HTML anchor that
 * submits form data. If the disabled attribute of the hyperlink is set to true, 
 * clicking the on the image will not generate a request and hence the form will
 * not be submitted or the page will not navigate to the specified url.
 * </p><p>
 * <h3>Example 1a: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       disabledImage: {
 *         id: "img1",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/disabled.gif"
 *         widgetType: "image"
 *       },
 *       enabledImage: {
 *         id: "img2",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/enabled.gif"
 *         widgetType: "image"
 *       },
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "imageHyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * image hyperlink is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       disabledImage: {
 *         id: "img1",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/disabled.gif"
 *         widgetType: "image"
 *       },
 *       enabledImage: {
 *         id: "img2",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/enabled.gif"
 *         widgetType: "image"
 *       },
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "imageHyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Image Hyperlink State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get image hyperlink
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the image hyperlink is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       disabledImage: {
 *         id: "img1",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/disabled.gif"
 *         widgetType: "image"
 *       },
 *       enabledImage: {
 *         id: "img2",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/enabled.gif"
 *         widgetType: "image"
 *       },
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "imageHyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Image Hyperlink" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get image hyperlink
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an image hyperlink using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the image hyperlink text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       disabledImage: {
 *         id: "img1",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/disabled.gif"
 *         widgetType: "image"
 *       },
 *       enabledImage: {
 *         id: "img2",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/enabled.gif"
 *         widgetType: "image"
 *       },
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "imageHyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Image Hyperlink Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get image hyperlink
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "hyp1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.imageHyperlink.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {Object} disabledImage
 * @config {Object} enabledImage
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} imagePosition
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.imageHyperlink",
        woodstock4_3.widget.hyperlink, {
    // Set defaults.
    _widgetType: "imageHyperlink" // Required for theme properties.
});

/**
 * Helper function to add children.
 *
 * @param props Key-Value pairs of properties.
 * @config {Array} contents The contents of the anchor body.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.imageHyperlink.prototype._addContents = function(props) {
    if (props.contents == null) {
        return false;
    }

    // Remove child nodes.
    this._widget._removeChildNodes(this._leftContentsContainer);
    this._widget._removeChildNodes(this._rightContentsContainer);

    // Add contents.
    for (i = 0; i <props.contents.length; i++) {
        this._widget._addFragment(this._leftContentsContainer, props.contents[i], "last");
        this._widget._addFragment(this._rightContentsContainer, props.contents[i], "last");
    }
    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.imageHyperlink.event =
        woodstock4_3.widget.imageHyperlink.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_imageHyperlink_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_imageHyperlink_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_imageHyperlink_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_imageHyperlink_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.imageHyperlink.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.enabledImage != null) { props.enabledImage = this.enabledImage; }
    if (this.disabledImage != null) { props.disabledImage = this.disabledImage; }
    if (this.imagePosition != null) { props.imagePosition = this.imagePosition; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.imageHyperlink.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._enabledImageContainer.id = this.id + "_enabled";
        this._disabledImageContainer.id = this.id + "_disabled";
        this._leftContentsContainer.id = this.id + "_leftContents";
        this._rightContentsContainer.id = this.id + "_rightContents";
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.imageHyperlink.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Show en/disabled images.
    if (props.disabled != null) {
        var disabled = new Boolean(props.disabled).valueOf();

        // We need to hide/show images only when the disabed image is specified.
        if (this.disabledImage) { 
            this._common._setVisibleElement(this._enabledImageContainer, !disabled);
            this._common._setVisibleElement(this._disabledImageContainer, disabled);
        }
    }

    // Add enabled image.
    if (props.enabledImage) {
        // Update/add fragment.
        this._widget._updateFragment(this._enabledImageContainer, this.enabledImage.id, 
            props.enabledImage);
    }

    // Add disabled image.
    if (props.disabledImage) {
        // Update/add fragment.
        this._widget._updateFragment(this._disabledImageContainer, this.disabledImage.id, 
            props.disabledImage);
    }

    // Set image position.
    if (props.imagePosition) {
        var left = (props.imagePosition == "left");
        this._common._setVisibleElement(this._leftContentsContainer, !left);
        this._common._setVisibleElement(this._rightContentsContainer, left);    
    }

    // Add contents.
    this._addContents(props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.imageHyperlink.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.label");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.label");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a label widget.
 *
 * @name woodstock4_3.widget.label
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the label widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey
 * @config {String} className CSS selector.
 * @config {String} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {Object} errorImage 
 * @config {String} htmlFor 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} level 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} primary Set button as primary if true.
 * @config {boolean} required
 * @config {Object} requiredImage
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.label", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.level = this._theme.getMessage("label.level", null, 2);
        this.required = false;
        this.valid = true;
    },
    _widgetType: "label" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.label.event =
        woodstock4_3.widget.label.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_label_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_label_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_label_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_label_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.label.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var key = "LABEL_LEVEL_TWO_TEXT";

    if (this.valid == false) {
        key = "CONTENT_ERROR_LABEL_TEXT";
    } else if (this.level == 1) {
        key = "LABEL_LEVEL_ONE_TEXT";
    } else if (this.level == 3) {
        key = "LABEL_LEVEL_THREE_TEXT";
    }

    // Get theme property.
    var newClassName = this._theme.getClassName(key, "");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to process notification events with Object
 * literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} detail Message detail text.
 * @config {boolean} valid Flag indicating validation state.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.label.prototype._notify = function(props) {
    if (props == null) {
        return false;
    }
    return this.setProps({
        valid: props.valid,
        errorImage: {
            title: props.detail
        }
    });
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.label.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.contents != null) { props.contents = this.contents; }
    if (this.errorImage != null) { props.errorImage = this.errorImage; }
    if (this.htmlFor != null) { props.htmlFor = this.htmlFor; }
    if (this.level != null) { props.level = this.level; }
    if (this.required != null) { props.required = this.required; }
    if (this.requiredImage) { props.requiredImage = this.requiredImage; }
    if (this.valid != null) { props.valid = this.valid; }
    if (this.value != null) { props.value = this.value; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.label.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._contentsContainer.id = this.id + "_contentsContainer";
        this._errorImageContainer.id = this.id + "_errorImageContainer";
        this._requiredImageContainer.id = this.id + "_requiredImageContainer";
        this._valueContainer.id = this.id + "_valueContainer";
    }

    // If errorImage or requiredImage are null, create images from the theme.
    // When the _setProps() function is called, image widgets will be
    // instantiated via the props param. 
    if (this.errorImage == null) {
	this.errorImage = {
            icon: "LABEL_INVALID_ICON",
            id: this.id + "_error",
	    className: this._theme.getClassName("LABEL_INVALID_IMAGE", null),
            widgetType: "image"
        };
    }
    if (this.requiredImage == null) {
        this.requiredImage = {
            icon: "LABEL_REQUIRED_ICON",
            id: this.id + "_required",
	    className: this._theme.getClassName("LABEL_REQUIRED_IMAGE", null),
            widgetType: "image"
        };
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.label.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.label.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.htmlFor != null) {
	this._domNode.htmlFor = props.htmlFor;
    }
    if (props.valid != null) {
	this.valid = new Boolean(props.valid).valueOf();
    }
    if (props.required != null) {
	this.required = new Boolean(props.required).valueOf();
    }
    if (props.value != null) {
        // An empty string is valid.
	this._widget._addFragment(this._valueContainer, props.value);
    }

    // Set error image properties.
    if (props.errorImage || props.valid != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.errorImage == null) {
            props.errorImage = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.errorImage.visible = !this.valid;

        // Update/add fragment.
        this._widget._updateFragment(this._errorImageContainer, this.errorImage.id, 
            props.errorImage);
    }

    // Set required image properties.
    if (props.requiredImage || props.required != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.requiredImage == null) {
            props.requiredImage = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.requiredImage.visible = this.required;

        // Update/add fragment.
        this._widget._updateFragment(this._requiredImageContainer, 
            this.requiredImage.id, props.requiredImage);
    }

    // Set contents.
    if (props.contents) {
        // Remove child nodes.
        this._widget._removeChildNodes(this._contentsContainer);

	for (var i = 0; i < props.contents.length; i++) {
            this._widget._addFragment(this._contentsContainer, props.contents[i], "last");
        }
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.label.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.listbox");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.listbox");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.selectBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");


woodstock4_3._dojo.provide("woodstock4_3.widget.textField");

woodstock4_3._dojo.require("woodstock4_3.widget.common");


woodstock4_3._dojo.provide("woodstock4_3.widget._base.fieldBase");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.fieldBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class contains functions for widgets that extend fieldBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.fieldBase",
        woodstock4_3.widget._base.labeledBase, {
    // Set defaults.
    constructor: function() {
	this.disabled = false;
	this.size = 20;
    }
});

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._getInputClassName = function() {   
    return null; // Overridden by subclass.
};

/**
 * Returns the HTML input element that makes up the text field.
 *
 * @return {Node} The HTML input element.
 */
woodstock4_3.widget._base.fieldBase.prototype.getInputElement = function() {
    return this._fieldNode;
};

/**
 * Return an Object Literal of label properties desired by this widget.
 * <p>
 * This implementation adds the <code>htmlFor</code> property with
 * <code>this._fieldNode.id</code>.
 * </p>
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._getLabelProps = function() {
    var props = this._inherited("_getLabelProps", arguments);

    props.htmlFor = this._fieldNode.id;
    return props;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.fieldBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);
    
    // Set properties.
    if (this.alt != null) { props.alt = this.alt; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.maxLength > 0) { props.maxLength = this.maxLength; }    
    if (this.notify != null) { props.notify = this.notify; }
    if (this.submitForm != null) { props.submitForm = this.submitForm; }
    if (this.text != null) { props.text = this.text; }
    if (this.title != null) { props.title = this.title; }
    if (this.type != null) { props.type= this.type; }
    if (this.readOnly != null) { props.readOnly = this.readOnly; }
    if (this.size > 0) { props.size = this.size; }
    if (this.style != null) { props.style = this.style; }
    
    // After widget has been initialized, get user's input.
    if (this._isInitialized() == true && this._fieldNode.value != null) {
        props.value = this._fieldNode.value;
    } else if (this.value != null) {
        props.value = this.value;
    }
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._fieldNode.id = this.id + "_field";
        this._fieldNode.name = this.id + "_field";
    }
    
    // Set public functions.

    /** @ignore */
    this._domNode.getInputElement = function() { return woodstock4_3.widget.common.getWidget(this.id).getInputElement(); };
    
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.submitForm == false || props.submitForm == true) { 
        // connect the keyPress event
        this._dojo.connect(this._fieldNode, "onkeypress", this, "_submitFormData");
    }
    if (props.maxLength > 0) { this._fieldNode.maxLength = props.maxLength; }
    if (props.size > 0) { this._fieldNode.size = props.size;  }
    if (props.value != null) { this._fieldNode.value = props.value; }
    if (props.title != null) { this._fieldNode.title = props.title; }   
    if (props.disabled != null) { 
        this._fieldNode.disabled = new Boolean(props.disabled).valueOf();
    }
    if (props.readOnly != null) { 
        this._fieldNode.readOnly = new Boolean(props.readOnly).valueOf();
    }
    
    // Set HTML input element class name.
    this._fieldNode.className = this._getInputClassName();
    
    // Set more properties.
    this._setCommonProps(this._fieldNode, props);
    this._setEventProps(this._fieldNode, props);
    
    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * Process keyPress events on the field, which enforces/disables 
 * submitForm behavior.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._submitFormData = function(event) {
    if (event == null) {
        return false;
    }
    if (event.keyCode == event.KEY_ENTER) {               
        if (this.submitForm == false) {
            // Disable form submission.
            if (window.event) {
                event.cancelBubble = true;
                event.returnValue = false;
            } else{
                event.preventDefault();
                event.stopPropagation();
            }
            return false;
        } else {
            // Submit the form                    
            if (event.currentTarget.form) {
                event.currentTarget.form.submit();
            }
        }
    }
    return true;    
};
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");

/**
 * This function is used to construct a textField widget.
 *
 * @constructor
 * @name woodstock4_3.widget.textField
 * @extends woodstock4_3.widget._base.fieldBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the textField widget.
 * <p>
 * The submitForm property can be used to alter the default browser behavior on
 * form submission when Enter key is pressed within a text field. A value of 
 * true will force the form submission while a value of false will prevent it. 
 * If submitForm is not specified, the default browser behavior will take effect.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text field",
 *       widgetType: "textField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * text field is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text field",
 *       widgetType: "textField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Text Field State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text field
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the text field is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text field",
 *       widgetType: "textField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Text Field" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text field
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a text field using the 
 * refresh function. The execute property of the refresh function is used to
 * define the client id which is to be submitted and updated server-side. When
 * the user clicks on the checkbox, the input value is updated server-side and 
 * the text field is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text field",
 *       widgetType: "textField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Change Text Field Value" },
 *       onClick="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text field
 *       return widget.refresh("cb1"); // Asynchronously refresh while submitting checkbox value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "field1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.textField.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoValidate
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} size 
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.textField", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.fieldBase ], {
    // Set defaults.
    constructor: function() {
        // Array of list values; may be empty; if null - then no autocomplete 
        // functionality is provided
        this.autoCompleteOptions = null; 
        this.autoCompleteSize = 15;
        this.autoCompleteCloseDelayTime = 100;
    },                  
    _widgetType: "textField" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.textField.event =
        woodstock4_3.widget.textField.prototype.event = {
    /**
     * This object contains filter event topics.
     * @ignore
     */
    autoComplete: {
        /** Filter event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_autoComplete_begin",

        /** Filter event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_autoComplete_end"
    },

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_submit_end"
    },

    /**
     * This object contains validation event topics.
     * @ignore
     */
    validation: {
        /** Validation event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_validation_begin",

        /** Validation event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_validation_end"
    }
};

/**
 * Utility function to adjust input and list widths to match
 * the one of surrounding domNode node    
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._adjustListGeometry = function () {
    this._listContainer.style.width = this._fieldNode.offsetWidth +"px";
   // this._listContainer.style.left = this._fieldNode.offsetLeft +"px";
    this._listContainer.style.left = "0px";
    this._listContainer.style.top = this._fieldNode.offsetTop + this._fieldNode.offsetHeight+"px";
    this._listNode.style.width = this._fieldNode.offsetWidth+"px";
    this._listContainer.style.zIndex = "999";
    return true;
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget.textField.prototype._getInputClassName = function() {          
    // Set readOnly style.
    if (this._fieldNode.readOnly) {
        return this._theme.getClassName("TEXT_FIELD_READONLY", "");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this._theme.getClassName("TEXT_FIELD_INVALID", "")
        : " " + this._theme.getClassName("TEXT_FIELD_VALID", "");
    
    // Set default style.    
    return (this.disabled == true)
        ? this._theme.getClassName("TEXT_FIELD_DISABLED", "") 
        : this._theme.getClassName("TEXT_FIELD", "") + validStyle;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.textField.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.autoValidate != null) { props.autoValidate = this.autoValidate; }
    if (this.autoComplete != null) { props.autoComplete = this.autoComplete; } 
    if (this.autoCompleteOptions != null) { props.autoCompleteOptions = this.autoCompleteOptions; } //TODO clone array?
        
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._postCreate = function () {
    // Set events.
    if (this.autoValidate == true) {
        // Generate the following event ONLY when 'autoValidate' == true.
        this._dojo.connect(this._fieldNode, "onblur", this, "_validate");
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * Helper function to create callback for autoComplete list closing event.
 *
 * @return {Function} The callback function.
 * @private
 */
woodstock4_3.widget.textField.prototype._createCloseListCallback = function() {
    var _id = this.id;
    return function(event) { 
        var widget = woodstock4_3.widget.common.getWidget(_id);
        if (widget == null) {
            return false;
        }
        widget.showAutoComplete = false;
        widget._updateListView();
        return true;
    };
};

/**
 * Publishes event to filter the options list according to specified 
 * filter string. Note that in default implementation such options will be 
 * updated remotely by Ajax call and then are automatically refreshed on the page.
 *
 * @param {String} filter
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._filterOptions = function() {    
    // Publish the event for custom AJAX implementations to listen for.
    // The implementation of this Ajax call will retrieve the value of the filter
    // and will obtain an updated lookup list ( either locally, or by submit to the server)
    // Data returned from Ajax call will be pushed into this._listWidget.
    this._publish(woodstock4_3.widget.textField.event.autoComplete.beginTopic, [{
        id: this.id
    }]);
    return true;        
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
    
    //we can receive updated autoComplete flag from ajax call back
    if (props.autoComplete != null) { this.autoComplete = new Boolean(props.autoComplete).valueOf();  }
    
    //initialize list, if required    
    //autocomplete list may be initiated only on widget start, and cannot 
    //be introduced as part of dynamic options on setProps
    if (this.autoComplete && this._listWidget == null) {
        //create and populate props for listbox
        var listWidgetProps = {
           id: this.id + "_list",
           onFocus: "woodstock4_3.widget.common.getWidget('" + this.id + 
                "')._processFocusEvent(this.event);", 
           onBlur: "woodstock4_3.widget.common.getWidget('" + this.id + 
                "')._processBlurEvent(this.event);",
           widgetType: "listbox"
        };

        //?? use of event registration as in following disables field processing keys 
        //onChange: "woodstock4_3.widget.common.getWidget('" + this.id + "')._processListChange(this.event);"

        // Fix: Properties should be initialized via postCreate. Then, the list
        // can be updated here instead of recreating the widget each time.
        this._widget._addFragment(this._listContainer, listWidgetProps);
        
        //store reference to the list
        this._listWidget = this._widget.getWidget(listWidgetProps.id);
        this._listNode = this._listWidget.getSelectElement();
        
        //since original list box is created empty, make sure it is not shown
        this._updateListView();
  
         //disable browser autocomplete
         
         //we assume here that once the field is armed for autocomplete,
         //this mode will not get disabled. Thus there is no logic provided
         //for restoration of "autocomplete" attribute
         //
        this._fieldNode.setAttribute("autocomplete", "off");
        
        //use focus event to open the list
        this._dojo.connect(this._fieldNode, "onfocus", this, "_processFocusEvent");
 
        //use blur events to close the list
        this._dojo.connect(this._fieldNode, "onblur", this, "_processBlurEvent");        
         
        // onChange event of the list will change the content of the input field
        this._dojo.connect(this._listNode, "onchange", this, "_processListChange");
 
        //onclick will allow to reopen options list after it has been closed with ESC
        this._dojo.connect(this._fieldNode, "onclick", this, "_processFocusEvent");

        // input field changes will trigger updates to the autoComplete list options
        this._dojo.connect(this._fieldNode, "onkeyup", this, "_processFieldKeyUpEvent");
        
        //additional logic applied to ENTER, ESCAPE, ARROWs on keydown in order to cancel the event bubble
        this._dojo.connect(this._fieldNode, "onkeydown", this, "_processFieldKeyDownEvent");
    }        
    
    if (this.autoComplete && props.autoCompleteOptions != null && this._listWidget != null ) {
        //autoComplete param has changed
        
        //we can receive new options from ajax call
        this.autoCompleteOptions = props.autoCompleteOptions;
        
        // Push properties into listbox.  
        var listProps = {};
        listProps.options = props.autoCompleteOptions;
        
        //change list box size up to autoCompleteSize elem
        if (this.autoCompleteOptions.length > this.autoCompleteSize) {
            listProps.size = this.autoCompleteSize;
        } else if (this.autoCompleteOptions.length == 1) {
            //provide extra space for one-line
            listProps.size = 2;
        } else {
            listProps.size = this.autoCompleteOptions.length;
        }

        this._listWidget.setProps(listProps);
        
        /* // display list on initiation
        this.showAutoComplete = true;        
        */
        this._updateListView();
    }   

    // Set remaining properties.
    var ret = this._inherited("_setProps", arguments);
    
    if (props.autoComplete && props.autoCompleteOptions != null && this._listWidget != null ) {
        this._adjustListGeometry();  //even if autocomplete options are not defined in this set of props
    }
    return ret;
};

/**
 * Process the blur  event - activate delayed timer to close the list.
 * Such timer will be canceled if either list of text field receive 
 * the focus back within a delay period.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processBlurEvent = function(event) {    
    //clear timeout for list closing, thereby preventing list from being closed    
    if (this.closingTimerId) {
        clearTimeout(this.closingTimerId);
        this.closingTimerId = null;
    }
    this.closingTimerId = setTimeout( 
        this._createCloseListCallback(), this.autoCompleteCloseDelayTime);   

    return true;
};

/**
 * Process keyPress events on the field, which enforces/disables 
 * submitForm behavior.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processFieldKeyDownEvent = function(event) {
    event = this._widget._getEvent(event);
    if (event == null) {
        return false;
    }

    if (event.keyCode == this._widget._keyCodes.ENTER && this.autoCompleteIsOpen) {               
        // Disable form submission. Note that form submission is disabled
        // only once when listbox is open. If it closed, form will be submitted
        // according to submitForm flag
        if (window.event) {
            event.cancelBubble = true;
            event.returnValue = false;
        } else{
            event.preventDefault();
            event.stopPropagation();
        } 
        this._fieldNode.value = this._listWidget.getSelectedValue();

        //force close the list box
        this.showAutoComplete = false;
        this._updateListView();    
        
        return true;
    }   

    //close the list in case of ESCAPE pressed
    if (event.keyCode == this._widget._keyCodes.ESCAPE  ) {               
        this._fieldNode.value = ""; //Warning: IE empties all fields on the page on 2nd ESC independently of setting value here
        this.showAutoComplete = false;
        this._updateListView();     
        this._filterOptions();
        return true;
    }
      
    //even when the text field is in focus, we want arrow keys ( up and down)
    //navigate through options in the select list
    if (event.keyCode == this._widget._keyCodes.DOWN_ARROW 
            && this._listWidget.getSelectedIndex() < this._listNode.options.length) {               
        try {
            this.showAutoComplete = true;

            if (!this.autoCompleteIsOpen || this._listNode.options.length <=1) {
                this._filterOptions();
                this._listWidget.setSelectedIndex(0) ;
            } else {     
                //list already open
                this._listWidget.setSelectedIndex(this._listWidget.getSelectedIndex() + 1) ;
                this._updateListView();
            }
            this._processListChange(event);
            return true;
       } catch (doNothing) {}
    }
    if (event.keyCode == this._widget._keyCodes.UP_ARROW && this._listWidget.getSelectedIndex() > 0) {               
        try {
            this.showAutoComplete = true;
            this._listWidget.setSelectedIndex(this._listWidget.getSelectedIndex() - 1) ;
            this._processListChange(event);
            return true;
        } catch (doNothing) {}
    }
    return true;
};

/**
 * Process keyPress events on the filter, which chages the filter
 * and submits a request for updated list.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processFieldKeyUpEvent = function(event) {    
    event = this._widget._getEvent(event);

    if (event != null &&
        ( event.keyCode == this._widget._keyCodes.ESCAPE ||
          event.keyCode == this._widget._keyCodes.ENTER ||
          event.keyCode == this._widget._keyCodes.DOWN_ARROW ||
          event.keyCode == this._widget._keyCodes.UP_ARROW  ||
          event.keyCode == this._widget._keyCodes.SHIFT  ||
          event.keyCode == this._widget._keyCodes.TAB
        )) {
        //these special keys are processed somewhere else - no filtering
        return false; 
    }
    this.showAutoComplete = true;
    this._filterOptions();
    return true;
};

/**
 * Process the focus event - turn on the flag for autoComplete list display
 * <p>
 * Displays autoComplete box on focus 
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processFocusEvent = function(event) {
    //clear timeout for list closing, thereby preventing list from being closed
    if (this.closingTimerId) {
        clearTimeout(this.closingTimerId);
        this.closingTimerId = null;
    }
    return true;
};

/**
 * Process onChange event on the select list, which results in filter being
 * updated with the new value. Note that no data submission is initiated here.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processListChange = function(event) {    
    event = this._widget._getEvent(event);

    if (event.type == "change") {               
        try {
            this._fieldNode.value = this._listWidget.getSelectedValue();
            
            //close the list
            this.showAutoComplete = false;
            this._updateListView();  
            this._fieldNode.focus(); 
                           
            return true;
       } catch (doNothing) {}
    }    

    // else - usually from selection movement with the arrow keys 
    this._fieldNode.value = this._listWidget.getSelectedValue();
    this._fieldNode.focus(); //keep the focus on filter field so that user can incrementally type additional data
    return true;
};

/**
 * Helper function to update the view of the component ( what is commonly known
 * in UI world as /refresh/ - but renamed not to be confused with Ajax refresh)
 *  - if size of the list box >=1, shows autocomplete list box.
 *  - if size of the list box <1, hides autocomplete list box.
 * 
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._updateListView = function() {
    if ( this.showAutoComplete == true && this.autoCompleteOptions.length >= 1) {
        //TODO a place for optimization here - not to adjust geometry each time
        this._adjustListGeometry();    

        //optionally we could add check for this._listNode.options.length >0
        this._listNode.className = this._theme.getClassName("TEXT_FIELD_AUTO_COMPLETE_LIST", "");

        // the following is preferred way of setting class, but it does not work
        //this._listWidget.setProps({className: this._theme.getClassName("TEXT_FIELD_AUTO_COMPLETE_LIST", "")}) ;
        
        //this.autoCompleteIsOpen flag indicates whether list box is open or not
        this.autoCompleteIsOpen = true;
    } else {
        this._listNode.className = this._theme.getClassName("HIDDEN");
        //this._listWidget.setProps(visible: 'false');
        this.autoCompleteIsOpen = false;
    }
    return true;
};

/**
 * Process validation event.
 * <p>
 * This function interprets an event (one of onXXX events, such as onBlur,
 * etc) and extracts data needed for subsequent Ajax request generation. 
 * Specifically, the widget id that has generated the event. If widget
 * id is found, publishBeginEvent is called with extracted data.
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._validate = function(event) {
    // Publish an event for custom AJAX implementations to listen for.
    this._publish(woodstock4_3.widget.textField.event.validation.beginTopic, [{
        id: this.id
    }]);
    return true;
};

/**
 * This function is used to construct a listbox widget.
 *
 * @name woodstock4_3.widget.listbox
 * @extends woodstock4_3.widget._base.selectBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the listbox widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} label Defines the widget and its properties to use for a
 * label.
 * @config {String} labelOnTop If true the label appears above the listbox.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} multiple If true allow multiple selections
 * @config {boolean} monospace If true use monospace font styling.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Option selection is changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Option text is highlighted.
 * @config {Array} options See selectBase._setOptions or selectBase overview
 * for details on the elements of this array.
 * @config {int} size The number of option rows to display.
 * @config {String} style Specify style rules inline. 
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.listbox", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.selectBase ], {
    // Set defaults.
    constructor : function() {
	this.labelOnTop = this._theme._getMessageBoolean("listbox.labelOnTop", false);
	this.monospace = this._theme._getMessageBoolean("listbox.monospace", false);
	this.multiple = this._theme._getMessageBoolean("listbox.multiple", false);
	this.size = this._theme.getMessage("listbox.size", null, 12);
	this.width = this._theme.getMessage("listbox.width", null);
    },
    _widgetType: "listbox" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.listbox.event =
        woodstock4_3.widget.listbox.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for.*/
        beginTopic: "woodstock4_3_widget_listbox_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for.*/
        endTopic: "woodstock4_3_widget_listbox_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_listbox_event_state_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_listbox_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_listbox_event_submit_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_listbox_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element CSS class name.
 * <p>
 * Note: Selectors should be concatenated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var newClassName = this._theme.getClassName("LISTBOX", "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Return an Object Literal of label properties desired by this widget.
 * <p>
 * This implementation sets
 * the <code>listbox.labelLevel</code> and <code>listbox.labelAlign</code>
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively. It sets the <code>htmlFor</code> attribute to 
 * <code>_listContainer.id</code>
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} Key-Value pairs of label properties.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getLabelProps = function() {

    var props = this._inherited("_getLabelProps", arguments);
    props.level = this._theme.getMessage("listbox.labelLevel", null, 2);
    return props;
};

/**
 * Return a CSS selector for the listbox label.
 * The chosen CSS selector is a function of 
 * the <code>labelOnTop</code> property. If <code>labelOnTop</code>
 * property is <code>false</code> the value of the theme key 
 * <code>LISTBOX_LABEL_ALIGN</code> from the 
 * <code>styles</code> category is returned, else <code>null</code>.
 * @param {boolean} ontop The label on top state. In some contexts this state
 * must be passed to the method, for example when called from the 
 * <code>selectBase.setProps</code> method.
 * @return {String} A CSS selector for the listbox label.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getLabelOnTopClassName = function(ontop) {
    //var labelontop = ontop != null ? ontop : this.labelOnTop;
    //return labelontop == true
    return ontop != null && ontop == true
	? null
	: this._theme.getClassName("LISTBOX_LABEL_ALIGN", null);
};

/**
 * Return a CSS selector for _listContainer HTML element.
 * The chosen CSS selector is a function of 
 * the <code>disabled</code> and <code>monospace</code>properties.
 * The returned selector is the value of a property defined in the
 * <code>styles</code> category, as shown in the following table.
 * <table border="1px">
 * <tr><th>key</th><th>disabled</th><th>monospace</th></tr>
 * <tr>
 * <td>LIST</td>
 * <td>false</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>LIST_DISABLED</td>
 * <td>true</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>LIST_MONOSPACE</td>
 * <td>false</td>
 * <td>true</td>
 * </tr>
 * <tr>
 * <td>LIST_MONOSPACE_DISABLED</td>
 * <td>true</td>
 * <td>true</td>
 * </tr>
 * </table>
 * @return {String} A CSS selector for the _listContainer HTML element.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getListContainerClassName =
	function(disabled, monospace) {
    var key = "LIST";
    if (disabled == true) {
	if (monospace == true) {
	    key = "LIST_MONOSPACE_DISABLED";
	} else {
	    key = "LIST_DISABLED";
	}
    } else if (monospace == true) {
	key = "LIST_MONOSPACE";
    }
    return this._theme.getClassName(key, null);
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.listbox.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Get properties.
    //
    if (this.size != null) {
	props.size = this.size;
    }
    if (this.multiple != null) {
	props.multiple = this.multiple;
    }
    if (this.monospace != null) {
	props.monospace = this.monospace;
    }
    return props;
};

/**
 * This function returns the CSS class name for an HTML option element,
 * based on the state of <code>element</code>. The CSS class name for 
 * the following theme keys is returned for the given states, based on 
 * the properties
 * in "option".
 * <p>
 * <table border="1px">
 * <tr><th>state</th><th>key</th></tr>
 * <tr><td>
 * separator == true</td><td>LIST_OPTION_SEPARATOR</td>
 * </tr><tr><td>
 * group == true</td><td>LIST_OPTION_GROUP</td>
 * </tr><tr><td>
 * disabled == true</td><td>LIST_OPTION_DISABLED</td>
 * </tr><tr><td>
 * selected == true and disabled == false</td><td>LIST_OPTION_SELECTED</td>
 * </tr><tr><td>
 * separator == false and group false<br/> 
 * 	and disabled == false<br/>
 *	and selected == false</td><td>LIST_OPTION</td>
 * </tr>
 * </table>
 * </p>
 * <p>
 * If <code>element</code> is null, the CSS class name for the theme
 * key "LIST_OPTION" is returned.
 * </p>
 * @param {Object} element An HTML option DOM node instance.
 * @config {boolean} separator If true, element is a option separator.
 * @config {boolean} group If true the element is a optGroup instance.
 * @config {boolean} disabled If true the element is disabled.
 * @config {boolean} selected If true the element is selected.
 * @return {String} The HTML option element CSS class name.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getOptionClassName = function(element) {
    var key = "LIST_OPTION";
    if (element == null) {
	return this._theme.getClassName(key, null);
    }

    if (element.separator == true) {
        key = "LIST_OPTION_SEPARATOR";
    } else
    if (element.group == true) {
        key = "LIST_OPTION_GROUP";
    } else
    if (element.selected == true) {
        key = "LIST_OPTION_SELECTED";
    }

    // disabled wins
    //
    if (element.disabled == true) {
	key = "LIST_OPTION_DISABLED";
    }
    return this._theme.getClassName(key, null);
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.listbox.prototype._postCreate = function () {
    // Don't trash the template.
    this._common._addStyleClass(this._listContainer,
	this._getListContainerClassName(
          this.disabled == true, this.monospace == true));

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.listbox.prototype.setProps = function(props, notify) {
    if (props == null) {
	return this._inherited("setProps", arguments);
    }

    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    var ismonospace = this.monospace == true;
    var toggleMonospace = props.monospace != null && 
	props.monospace != ismonospace;

    var isdisabled = this.disabled == true;
    var toggleDisabled = props.disabled != null && 
	props.disabled != isdisabled;

    // Get current state of the classsname, and strip it 
    // and then add the classname for the new state.
    //
    if (toggleMonospace || toggleDisabled) {
	var cn = this._getListContainerClassName(isdisabled, ismonospace);
	this._common._stripStyleClass(this._listContainer, cn);

	cn = this._getListContainerClassName(
	    toggleDisabled ? props.disabled == true : isdisabled,
	    toggleMonospace ? props.monospace == true : ismonospace);
	this._common._addStyleClass(this._listContainer, cn);
    }
    return this._inherited("setProps", arguments);
};

/**
 * This function sets properties specific to the listbox, the superclass
 * will set properties specific to the <code>select</code> HTML element
 * and handle <code>labelOnTop</code> and the label subcomponent.
 * <p>
 * If the <code>size</code>, <code>multiple</code>, or <code>monospace</code>
 * properties are not set in <code>props</code> or <code>this</code>,
 * the values are obtained from the theme using the keys 
 * <code>listbox.size</code>, <code>listbox.multiple</code>,
 * <code>listbox.monospace</code>, and <code>listbox.labelOnTop</code>
 * respectively, from the theme "messages" category. If the theme does 
 * not define these values then the values <code>12</code>,
 * <code>false</code>, <code>false</code>, and <code>false</code>
 * are used, respectively.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} multiple If true allow multiple selections
 * @config {boolean} monospace  If true the <code>LIST_MONOSPACE</code> and
 * <code>LIST_MONOSPACE_DISABLED</code>CSS class names are used, else
 * <code>LIST</code> and <code>LIST_DISABLED</code>.
 * @config {int} size The number of rows to display.
 * @config {boolean} labelOnTop If true render the label on top of the
 * list box, else to the upper left.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.listbox.prototype._setProps = function(props) {
    // If props == null create one so that the superclass can 
    // handle "labelOnTop" or anything else that needs to be
    // set in props and handled in the superclass
    //
    if (props == null) {
	return this._inherited("_setProps", arguments);
	//props = {};
    }

    if (props.size != null) {
	var size = props.size;
	if (size < 1) {
	    size = this._theme.getMessage("listbox.size", null, 12);
	}
	if (this._listContainer.size != size) {
	    this._listContainer.size = size;
	}
    }

    if (props.multiple != null && 
	    this._listContainer.multiple != props.multiple) {
	this._listContainer.multiple = props.multiple;
    }
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.listbox.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.listbox.event.submit.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.radioButton");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.radioButton");

woodstock4_3._dojo.require("woodstock4_3.widget._base.checkedBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");

/**
 * This function is used to construct a radioButton widget.
 *
 * @constructor
 * @name woodstock4_3.widget.radioButton
 * @extends woodstock4_3.widget._base.checkedBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the radioButton widget.
 * <p>
 * The widget can be used as a single radio button or as one radio button among
 * a group of radio buttons. A group of radio buttons represents a multiple 
 * selection list which can have any number of radio button selected, or none
 * selected.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "rb1",
 *       label: { value: "This is a radio button" },
 *       widgetType: "radioButton"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the radio button, the
 * radio button is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "rb1",
 *       label: { value: "This is a radio button" },
 *       widgetType: "radioButton"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Radio Button State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("rb1"); // Get radio button
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the radio button is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "rb1",
 *       label: { value: "This is a radio button" },
 *       widgetType: "radioButton"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Radio Button" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("rb1"); // Get radio button
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a radio button using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the radio button 
 * label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "rb1",
 *       label: { value: "This is a radio button" },
 *       widgetType: "radioButton"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Radio Button Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("rb1"); // Get radio button
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "rb1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.radioButton.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {boolean} checked 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} image 
 * @config {String} label 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect 
 * @config {boolean} readOnly 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.radioButton", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.checkedBase ], {
    // Set defaults.
    _idSuffix: "_rb",
    _widgetType: "radioButton" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.radioButton.event =
        woodstock4_3.widget.radioButton.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_radioButton_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_radioButton_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_radioButton_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_radioButton_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_radioButton_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_radioButton_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.radioButton.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.disabled == true)
        ? this._theme.getClassName("RADIOBUTTON_SPAN_DISABLED", "")
        : this._theme.getClassName("RADIOBUTTON_SPAN", "");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 * @private
 */
woodstock4_3.widget.radioButton.prototype._getImageClassName = function() {
    return (this.disabled == true)
        ? this._theme.getClassName("RADIOBUTTON_IMAGE_DISABLED", "")
        : this._theme.getClassName("RADIOBUTTON_IMAGE", "");  
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget.radioButton.prototype._getInputClassName = function() {
    // Set readOnly style.
    if (this.readOnly == true) {
        return this._theme.getClassName("RADIOBUTTON_READONLY", "");
    }

    // Disabled style.
    return (this.disabled == true)
        ? this._theme.getClassName("RADIOBUTTON_DISABLED", "")
        : this._theme.getClassName("RADIOBUTTON", "");  
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 * @private
 */
woodstock4_3.widget.radioButton.prototype._getLabelDisabledClassName = function(disabled) {
    return (disabled == true)
        ? this._theme.getClassName("RADIOBUTTON_LABEL_DISABLED", "")
        : this._theme.getClassName("RADIOBUTTON_LABEL", "");  
};

/**
 * Return an Object Literal of label properties desired
 * by the radioButton widget.
 * <p>
 * This implementation sets
 * the <code>radioButton.labelLevel</code> 
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively.
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} label properties.
 * @private
 */
woodstock4_3.widget.radioButton.prototype._getLabelProps = function() {

    // First see if the super class wants to contribute to the props.
    // Let selectBase add the htmlFor property
    //
    var props = this._inherited("_getLabelProps", arguments);
    props.level = this._theme.getMessage("radioButton.labelLevel", null, 3);
    return props;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.radioButton.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.radioButton.event.submit.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.radioButtonGroup");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.radioButtonGroup");

woodstock4_3._dojo.require("woodstock4_3.widget._base.checkedGroupBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");

/**
 * This function is used to construct a radioButtonGroup widget.
 *
 * @constructor
 * @name woodstock4_3.widget.radioButtonGroup
 * @extends woodstock4_3.widget._base.checkedGroupBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the radioButtonGroup widget.
 * <p>
 * The widget is used to layout an array of radio buttons. To group radio 
 * buttons properly, each widget id must be unique, but the name property value
 * is the same.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "rbg1",
 *       contents: [{ 
 *         id: "rb1",
 *         label: { value: "This is radio button 1" },
 *         widgetType: "radioButton"
 *       }, {
 *         id: "rb2",
 *         label: { value: "This is radio button 2" },
 *         widgetType: "radioButton"
 *       }],
 *       label: { value: "This is a radio button group" },
 *       widgetType: "radioButtonGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * radio button group is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "rbg1",
 *       contents: [{ 
 *         id: "rb1",
 *         label: { value: "This is radio button 1" },
 *         widgetType: "radioButton"
 *       }, {
 *         id: "rb2",
 *         label: { value: "This is radio button 2" },
 *         widgetType: "radioButton"
 *       }],
 *       label: { value: "This is a radio button group" },
 *       widgetType: "radioButtonGroup"
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Radio Button Group State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("rbg1"); // Get radio button group
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the radio button group is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "rbg1",
 *       contents: [{ 
 *         id: "rb1",
 *         label: { value: "This is radio button 1" },
 *         widgetType: "radioButton"
 *       }, {
 *         id: "rb2",
 *         label: { value: "This is radio button 2" },
 *         widgetType: "radioButton"
 *       }],
 *       label: { value: "This is a radio button group" },
 *       widgetType: "radioButtonGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh RadioButton Group" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("rb1"); // Get radio button group
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a radio button group using the
 * refresh function. The execute property of the refresh function is used to 
 * define the client id which is to be submitted and updated server-side. As the
 * user types in the text field, the input value is updated server-side and the 
 * radio button group label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "rbg1",
 *       contents: [{ 
 *         id: "rb1",
 *         label: { value: "This is radio button 1" },
 *         widgetType: "radioButton"
 *       }, {
 *         id: "rb2",
 *         label: { value: "This is radio button 2" },
 *         widgetType: "radioButton"
 *       }],
 *       label: { value: "This is a radio button group" },
 *       widgetType: "radioButtonGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Radio Button Group Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("rbg"); // Get radio button group
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "rbg") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.radioButtonGroup.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {int} columns 
 * @config {Array} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {boolean} readOnly Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.radioButtonGroup", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.checkedGroupBase ], {
    // Set defaults.
    _widgetType: "radioButtonGroup" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.radioButtonGroup.event =
        woodstock4_3.widget.radioButtonGroup.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_radioButtonGroup_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */        
        endTopic: "woodstock4_3_widget_radioButtonGroup_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_radioButtonGroup_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_radioButtonGroup_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.radioButtonGroup.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.columns > 1)
        ? this._theme.getClassName("RBGRP_HORIZ", "")
        : this._theme.getClassName("RBGRP_VERT", "");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.radioButtonGroup.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.text");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.text");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a text widget.
 *
 * @name woodstock4_3.widget.text
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the text widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} escape HTML escape value (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.text", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.escape = true;
    },
    _widgetType: "text" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.text.event =
        woodstock4_3.widget.text.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_text_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_text_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_text_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_text_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.text.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.escape != null) { props.escape = this.escape; }
    if (this.value != null) { props.value = this.value; }

    return props;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.text.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
      
    // Set text value -- an empty string is valid.
    if (props.value != null) {
        // NOTE: If you set this value manually, text must be HTML escaped.
        this._widget._addFragment(this._domNode, props.value, null, this.escape);
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.text.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.textArea");

woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.textArea");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget.textField");

/**
 * This function is used to construct a textArea widget.
 *
 * @constructor
 * @name woodstock4_3.widget.textArea
 * @extends woodstock4_3.widget.textField
 * @class This class contains functions for the textArea widget.
 * <p>
 * The autoSave property will submit the content of the modified text area for
 * server side processing. The text area value will be submitted every autoSave
 * milliseconds only if it has been modified within last interval. 
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text area",
 *       widgetType: "textArea"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * text area is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text area",
 *       widgetType: "textArea"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Text Area State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text area
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the text area is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text area",
 *       widgetType: "textArea"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Text Area" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text area
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a text area using the 
 * refresh function. The execute property of the refresh function is used to
 * define the client id which is to be submitted and updated server-side. When
 * the user clicks on the checkbox, the input value is updated server-side and 
 * the text area is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text area",
 *       widgetType: "textArea"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Change Text Area Value" },
 *       onClick="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text area
 *       return widget.refresh("cb1"); // Asynchronously refresh while submitting checkbox value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "field1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.textArea.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoSave 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} rows 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.textArea",
        woodstock4_3.widget.textField, {
    // Set defaults.
    constructor: function() {
        this.autoSave = 0;
        this.cols = 20;
        this.rows = 3;
    },
    _widgetType: "textArea" // Required for theme properties.
});

/**
 * Helper function to create callback for timer event.
 *
 * @return {Function} The callback function.
 * @private
 */
woodstock4_3.widget.textArea.prototype._createSubmitCallback = function() {
    var _id = this.id;

    // New literals are created every time this function
    // is called, and it's saved by closure magic.
    return function(event) { 
        var widget = woodstock4_3.widget.common.getWidget(_id);
        if (widget == null) {
            return false;
        }
        //Create a submit request only if field has been modified
        if (widget.lastSaved != widget._fieldNode.value) {
            widget.lastSaved = widget._fieldNode.value;
            widget.submit();
        }
        return true;
    };
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.textArea.event =
        woodstock4_3.widget.textArea.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textArea_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textArea_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textArea_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textArea_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textArea_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textArea_event_submit_end"
    }
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget.textArea.prototype._getInputClassName = function() {
    // Set readOnly style
    if (this._fieldNode.readOnly) {
        return this._theme.getClassName("TEXT_AREA_READONLY", "");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this._theme.getClassName("TEXT_AREA_INVALID", "")
        : " " + this._theme.getClassName("TEXT_AREA_VALID", "");

    // Set default style.    
    return (this.disabled == true)
        ? this._theme.getClassName("TEXT_AREA_DISABLED", "") 
        : this._theme.getClassName("TEXT_AREA", "") + validStyle;    
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.textArea.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);
    
    // Set properties.
    if (this.cols > 0 ) { props.cols = this.cols; }
    if (this.rows > 0) { props.rows = this.rows; }
    if (this.autoSave > 0 ) { props.autoSave = this.autoSave; }
   
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textArea.prototype._postCreate = function () {
    // Set events.                
    if (this.autoSave > 0) {
        this.autoSaveTimerId = setInterval(this._createSubmitCallback(), 
            this.autoSave);  
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textArea.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.   
    if (props.cols > 0) { this._fieldNode.cols = props.cols; }
    if (props.rows > 0) { this._fieldNode.rows = props.rows; }
    
    // Cancel autosave if it has been changed to <=0
    if (props.autoSave <= 0 && this.autoSaveTimerId && this.autoSaveTimerId != null ) {
        clearTimeout(this.autoSaveTimerId);
        this.autoSaveTimerId = null;
    }

    // Set label className -- must be set before calling superclass.
    if (props.label) {
        props.label.className = (props.label.className)
            ? this._theme.getClassName("TEXT_AREA_TOPLABELALIGN", "")  + " " + props.label.className
            : this._theme.getClassName("TEXT_AREA_TOPLABELALIGN", "") ;
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.textArea.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.textArea.event.submit.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._jsfx.textField");

woodstock4_3._dojo.require("woodstock4_3.json");
woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.common");
woodstock4_3._dojo.require("woodstock4_3.widget._jsfx.dynaFaces");
woodstock4_3._dojo.require("woodstock4_3.widget.textField");

/**
 * @class This class contains functions to obtain data asynchronously using JSF
 * Extensions as the underlying transfer protocol.
 * @static
 * @private
 */
woodstock4_3.widget._jsfx.textField = {
    /**
     * This function is used to update widgets with new autoComplete options list.
     *
     * @param {String} id The widget id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson argument provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _autoCompleteCallback: function(elementId, content, closure, xjson) {
        if (elementId == null || content == null) {
            return false;
        }
        
        // Parse JSON text.
        var props = woodstock4_3.json.parse(content);

        // Update text field.
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        widget.setProps(props);       


        // Publish an event for custom AJAX implementations to listen for.
        woodstock4_3._dojo.publish(
            woodstock4_3.widget.textField.event.autoComplete.endTopic, [props]);
        return true;
    },

    /**
     * This function is used to process autoComplete events.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The widget id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processAutoCompleteEvent: function(props) {
        if (props == null) {
            return false;
        }
        
        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);
        
        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: props.id,
            render: props.id,
            replaceElement: woodstock4_3.widget._jsfx.textField._autoCompleteCallback,
            xjson: {
                id : props.id,
                event: "autocomplete"
            }
        });
        return true;
    },

    /**
     * This function is used to process validation events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The widget id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processValidationEvent: function(props) {
        if (props == null) {
            return false;
        }
        
        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);
        
        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: props.id,
            render: props.id,
            replaceElement: woodstock4_3.widget._jsfx.textField._validationCallback,
            xjson: {
                id : props.id,
                event: "validate"
            }
        });
        return true;
    },  

    /**
     * This function is used to update widgets.
     *
     * @param {String} id The widget id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson argument provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _validationCallback: function(elementId, content, closure, xjson) {
        if (elementId == null || content == null) {
            return false;
        }
        
        // Parse JSON text.
        var props = woodstock4_3.json.parse(content);

        // Update text field.
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        widget.setProps({
            valid: props.valid,
            errorImage: {
                title: props.detail
            }
        });

        // Notify decoupled widgets.
        if (widget.notify) {
            // Update each given client ID.
            for (var i = 0; i < widget.notify.length; i++) {
                // Get widget associated with client ID.
                var curWidget = woodstock4_3.widget.common.getWidget(widget.notify[i]);
                if (curWidget && typeof curWidget._notify == "function") {
                    curWidget._notify(props);
                }
            }
        }

        // Publish an event for custom AJAX implementations to listen for.
        woodstock4_3._dojo.publish(
            woodstock4_3.widget.textField.event.validation.endTopic, [props]);
        return true;
    }    
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.textField.event.refresh.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.textField.event.state.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processStateEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.textField.event.submit.beginTopic,
    woodstock4_3.widget._jsfx.common, "_processSubmitEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.textField.event.validation.beginTopic,
    woodstock4_3.widget._jsfx.textField, "_processValidationEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.textField.event.autoComplete.beginTopic,
    woodstock4_3.widget._jsfx.textField, "_processAutoCompleteEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget.passwordField");

woodstock4_3._dojo.require("woodstock4_3.widget._base.fieldBase");

/**
 * This function is used to construct a passwordField widget.
 *
 * @constructor
 * @name woodstock4_3.widget.passwordField
 * @extends woodstock4_3.widget._base.fieldBase
 * @class This class contains functions for the passwordField widget.
 * <p>
 * The passwordField widget create an HTML input field where the characters 
 * entered are echoed back with a replacement character in order to mask the
 * input.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a password field",
 *       widgetType: "passwordField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * password field is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a Password Field",
 *       widgetType: "passwordField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Password Field State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get password field
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} size 
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.passwordField",
        woodstock4_3.widget._base.fieldBase, {
    // Set defaults.
    _widgetType: "passwordField" // Required for theme properties.
});

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget.passwordField.prototype._getInputClassName = function() {
    if (this._fieldNode.readOnly) {
        return this._theme.getClassName("PASSWORD_FIELD_READONLY", "");
    }

    //invalid style
    var validStyle =  (this.valid == false) 
        ? " " + this._theme.getClassName("PASSWORD_FIELD_INVALID", "")
        : " " + this._theme.getClassName("PASSWORD_FIELD_VALID", "");
    
    // Set default style.    
    return (this.disabled == true)
        ? this._theme.getClassName("PASSWORD_FIELD_DISABLED", "") 
        : this._theme.getClassName("PASSWORD_FIELD", "") + validStyle;
};
