/**
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */


woodstock4_3._dojo.provide("woodstock4_3._base.body");



woodstock4_3._dojo.provide("woodstock4_3._base.cookie");

/**
 * @class This class contains functions to manipulate cookies.
 * @static
 * @private
 */
woodstock4_3._base.cookie = {
    /**
     * This function will get the cookie value.
     *
     * @return {String} The cookie value.
     * @private
     */
    _get: function() {
        // Get document cookie.
        var cookie = document.cookie;

        // Parse cookie value.
        var pos = cookie.indexOf(this._cookieName + "=");
        if (pos == -1) {
            return null;
        }

        var start = pos + this._cookieName.length + 1;
        var end = cookie.indexOf(";", start);
        if (end == -1) {
            end = cookie.length;
        }

        // return cookie value
        return cookie.substring(start, end);
    },

    /**
     * This function will load the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _load: function() {
        // Get document cookie.
        var cookieVal = this._get();
        if (cookieVal == null) {
            return false;
        }

        // Break cookie into names and values.
        var a = cookieVal.split('&');

        // Break each pair into an array.
        for (var i = 0; i < a.length; i++) {
            a[i] = a[i].split(':');
        }

        // Set name and values for this object.
        for (i = 0; i < a.length; i++) {
            this[a[i][0]] = unescape(a[i][1]);
        }
        return true;
    },

    /**
     * This function will reset the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _reset: function() {
        // Clear cookie value.
        document.cookie = this._cookieName + "=";
        return true;
    },

    /**
     * This function will store the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _store: function() {
        // Create cookie value by looping through object properties
        var cookieVal = "";

        // Since cookies use the equals and semicolon signs as separators,
        // we'll use colons and ampersands for each variable we store.
        for (var prop in this) {
            // Ignore properties that begin with '$' and methods.
            if (prop.charAt(0) == '$' || typeof this[prop] == 'function') {
                continue;
            }
            if (cookieVal != "") {
                cookieVal += '&';
            }
            cookieVal += prop + ':' + escape(this[prop]);
        }
        var cookieString = this._cookieName + "=" + cookieVal;
        if (this._path != null) {
            cookieString += ";path=" + this._path;
        }
        // Store cookie value.
        document.cookie = cookieString;
        return true;
    }
};

/**
 * @class This class contains functionality for scrolling.
 * @constructor This function is used to construct a javascript object for
 * maintaining scroll position via cookie.
 * @param {String} viewId
 * @param {String} path
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3._base.scrollCookie = function(viewId, path) {    
    // All predefined properties of this object begin with '$' because
    // we don't want to store these values in the cookie.
    this._cookieName = viewId;
    this._path = path;

    // Default properties.
    this._left = "0";
    this._top  = "0";

    /**
     * This function will load the cookie and restore scroll position.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._restore = function() {
        // Load cookie value.
        this._load();
        scrollTo(this._left, this._top);
        return true;
    };

    /**
     * This function will set the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._set = function() {
        var documentElement = window.document.documentElement;
        if (documentElement && documentElement.scrollTop) {
            this._left = documentElement.scrollLeft;
            this._top  = documentElement.scrollTop;
        } else {
            this._left = window.document.body.scrollLeft;
            this._top  = window.document.body.scrollTop;
        }
        // if the left and top scroll values are still null
        // try to extract it assuming the browser is IE.
        if (this._left == null && this._top == null) {
            this._left = window.pageXOffset;
            this._top = window.pageYOffset;
        }
        // Store cookie value.
        this._store();
        return true;
    };
    return true;
};

// Inherit cookie properties.
woodstock4_3._base.scrollCookie.prototype = woodstock4_3._base.cookie;
woodstock4_3._dojo.require("woodstock4_3._base.browser");


woodstock4_3._dojo.provide("woodstock4_3._base.common");

woodstock4_3._dojo.require("woodstock4_3.theme.common");

/**
 * @class This class contains functions common to HTML elements.
 * @static
 * @private
 */
woodstock4_3._base.common = {
    /**
     * Variable needed when submitting form so timeout will work properly.
     * @private
     */
    _formToSubmit: null,

    /**
     * Variable needed when submitting form so timeout will work properly.
     * @private
     */
    _submissionComponentId: null,

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // String functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Replace occurences of delimiter with the escapeChar and the
     * delimiter. For example replace "," with "/," if delimiter == "," and
     * escapeChar is "/".
     *
     * @param {String} s The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {String} The escaped string.
     * @private
     */
    _escapeString: function(s, delimiter, escapeChar) {
        if (s == null) {
            return null;
        }
        if (delimiter == null) {
            return s;
        }
        if (escapeChar == null) {
            return null;
        }
        
        // Escape occurrences of delimiter with 
        // escapeChar and the delimiter.
        //
        // First escape the escape char.
        //
        var escape_escapeChar = escapeChar;
        if (escapeChar == "\\") {
            escape_escapeChar = escapeChar + escapeChar;
        }
        
        var rx = new RegExp(escape_escapeChar, "g");
        var s1 = s.replace(rx, escapeChar + escapeChar);
        
        rx = new RegExp(delimiter, "g");
        return s1.replace(rx, escapeChar + delimiter);
    },
    
    /**
     * Replace occurences of a sequence of 2 instances of delimiter 
     * with 1 instance of the delimiter. For example replace ",," with "," if 
     * delimiter == ","
     *
     * @param {String} s The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {String} The unescaped string.
     * @private
     */
    _unescapeString: function(s, delimiter, escapeChar) {
        if (s == null) {
            return null;
        }
        if (delimiter == null) {
            return s;
        }
        if (escapeChar == null) {
            return null;
        }
        
        // UnEscape occurrences of delimiter with 
        // single instance of the delimiter
        //
        var escape_escapeChar = escapeChar;
        if (escapeChar == "\\") {
            escape_escapeChar = escapeChar + escapeChar;
        }
        
        // First unescape the escape char.
        //
        var rx = new RegExp(escape_escapeChar + escape_escapeChar, "g");
        var s1 = s.replace(rx, escapeChar);
        
        // Now replace escaped delimters
        //
        rx = new RegExp(escape_escapeChar + delimiter, "g");
        return s1.replace(rx, delimiter);
    },
    
    /**
     * Return an array of unescaped strings from escapedString
     * where the escaped character is delimiter.
     * If delimiter is "," escapedString might have the form
     * <p><pre>
     * XX\,XX,MM\,MM
     *
     * where "\" is the escape char.
     * 
     * and is returned as an array
     * array[0] == "XX,XX"
     * array[1] == "MM,MM"
     * </pre><p>
     *
     * @param {String} escapedString The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {Array} An array of unescaped strings.
     * @private
     */
    _unescapeStrings: function(escapedString, delimiter, escapeChar) {
        if (escapedString == null || escapedString == "") {
            return null;
        }
        if (delimiter == null || delimiter == "") {
            return escapedString;
        }
        if (escapeChar == null || escapeChar == "") {
            return null;
        }
        
        // Need to do this character by character.
        var selections = new Array();
        var index = 0;
        var escseen = 0;
        var j = 0;
        
        for (var i = 0; i < escapedString.length; ++i) {
            if (escapedString.charAt(i) == delimiter) {
                if (escseen % 2 == 0) {
                    selections[index++] = escapedString.slice(j, i);
                    j = i + 1;
                }
            }
            if (escapedString.charAt(i) == escapeChar) {
                ++escseen;
                continue;
            } else {
                escseen = 0;
            }
        }
        
        // Capture the last split.
        selections[index] = escapedString.slice(j);
        
        // Now unescape each selection
        var unescapedArray = new Array();
        
        // Now replace escaped delimiters
        // i.e.  "\," with ","
        for (i = 0; i < selections.length; ++i) {
            unescapedArray[i] = woodstock4_3._base.common._unescapeString(
            selections[i], delimiter, escapeChar);
        }
        return unescapedArray;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Style functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Use this function add any styleClass to an html tag
     *
     * @param {Node} element the dom html tag element
     * @param {String} styleClass the name of the class to add
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _addStyleClass: function(element, styleClass) {
        // routine protection in javascript
        if (element == null || styleClass == null) {
            return false;
        }
        
        // handle easy case first
        if (element.className == null) {
            element.className = styleClass;
            return true;
        }
        
        // break out style classes into an array  
        var classes = woodstock4_3._base.common._splitStyleClasses(element);
        if (classes == null) {
            return false;
        }
        
        // For each element of classes, compare it to styleClass.
        // If it is not in the array, append it to the end.
        for (var i = 0; i < classes.length; i++) {
            if (classes[i] != null && classes[i] == styleClass) {
                return true;
            }
        }
        element.className = element.className + " " + styleClass;
        return true;
    },
    
    /**
     * Use this function to check if an array has a style class
     *
     * @param {Array} styleArray of style classes to check
     * @param {String} styleClass the styleClass to check
     * @return {Array} An array of classes.
     * @private
     */
    _checkStyleClasses: function(styleArray, styleClass) {
        if (styleArray == null || styleClass == null) {
            return false;
        }
        for (var i = 0; i < styleArray.length; i++) {
            if (styleArray[i] != null && styleArray[i] == styleClass) {
                return true;
            }
        }   
        return false;
    },
    
    /**
     * Use this function to get array of style classes
     *
     * @param {Node} element the dom html tag element
     * @return {Array} An array of classes.
     * @private
     */
    _splitStyleClasses: function(element) {
        if (element != null && element.className != null) {
            return element.className.split(" ");
        } else {
            return null;
        }
    },
    
    /**
     * Use this function remove any styleClass for an html tag
     *
     * @param {Node} element the dom html tag element
     * @param {String} styleClass the name of the class to remove
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _stripStyleClass: function(element, styleClass) {
        // routine protection in javascript
        if (element == null || styleClass == null || element.className == null) {
            return false;
        }
        
        // break out style classes into an array  
        var classes = woodstock4_3._base.common._splitStyleClasses(element);
        if (classes == null) {
            return false;
        }
        
        // For each styleClass, check if it's hidden and remove otherwise write
        // it back out to the class
        for (var i = 0; i < classes.length; i++) {
            if (classes[i] != null && classes[i] == styleClass) {
                classes.splice(i,1);  	
            }
        }
        element.className = classes.join(" ");
        return true;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Submit functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function to insert a hidden field element in the page.
     *
     * @param {String} elementId The element ID of the html tag 
     * @param {String} elementValue The value of the html tag.
     * @param {Node} parentForm The parent form of the html tag.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _insertHiddenField: function(elementId, elementValue, parentForm) {
        // We have to assume that there is only one element
        // with elementId. document.getElementById, returns
        // the first one it finds, which appears to be the 
        // first one created dynamically, if more than one 
        // element is created dynamically with the same id.
        //
        // appendChild just appends even if there is an element
        // with the same id that exists.
        //
        // The assumption is that there should only be 
        // one element in the document with such an id.
        //
        // If the elementId exists just modifiy its value
        // instead of creating and appending.
        //
        
        var element = document.forms[parentForm.id].elements[elementId];
        if (element != null) {
            element.value = elementValue;            
            return true;            
        } 
        
        var newElement = document.createElement('input');
        newElement.type = 'hidden';
        newElement.id = elementId;
        newElement.value = elementValue;
        newElement.name = elementId;
        parentForm.appendChild(newElement);

        return true;
    },
    
    /**
     * Use this function to submit a virtual form.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     * @deprecated Virtual forms only supported by JSF based form component.
     */
    _submitForm: function() {
        // "formToSubmit" is a literal (not virtual) form.
        // "submissionComponentId" is a component id (not client id).
        // the virtual form implementation uses _submissionComponentId
        // to determine which virtual form (if any) was submitted.
        if (woodstock4_3._base.common._formToSubmit == null) {
            return false;
        }
        if (woodstock4_3._base.common._submissionComponentId != null &&
                woodstock4_3._base.common._submissionComponentId.length > 0) {
            woodstock4_3._base.common._insertHiddenField('_submissionComponentId', 
                woodstock4_3._base.common._submissionComponentId,
                woodstock4_3._base.common._formToSubmit);
        }
        woodstock4_3._base.common._formToSubmit.submit();
        return false;
    },
    
    /**
     * Helper function to submit a virtual form.
     *
     * @param {Node} form The HTML form element to submit.
     * @param {String} submissionComponentId The Id of the component submitting the form.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     * @deprecated Virtual forms only supported by JSF based form component.
     */
    _timeoutSubmitForm: function(form, submissionComponentId) {
        woodstock4_3._base.common._formToSubmit = form;
        woodstock4_3._base.common._submissionComponentId = submissionComponentId;
        setTimeout('woodstock4_3._base.common._submitForm()', 0);
        return true;
    },
    
    /**
     * Helper function to submit a virtual form.
     *
     * @param {Node} form The HTML form element to submit.
     * @param {String} submissionComponentId The Id of the component submitting the form.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     * @deprecated Virtual forms only supported by JSF based form component.
     */
    _leaveSubmitterTrace: function(form, submissionComponentId) {
        // This function only needs to be called in the onclick handler of 
        // an ActionSource component that appears within a -standard- table.
        // Under those circumstances, if this function is not called, then when
        // the component is clicked, the virtual form implementation will have 
        // no way of knowing that a virtual form was submitted.
        if (form != null && submissionComponentId != null && 
                submissionComponentId.length > 0) {
            woodstock4_3._base.common._insertHiddenField('_submissionComponentId',
            submissionComponentId, form);
        }
        return true;
    },
    
    /**
     * delete a previously created element by _createSubmittableArray.
     *
     * @param {String} name The element ID of the html tag 
     * @param {Node} form The HTML form element to submit.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     * @deprecated Virtual forms only supported by JSF based form component.
     */
    _deleteSubmittableArray: function(name, parentForm) {
        try {
            var submittableArray  = document.getElementById(name);
            if (submittableArray != null) {
                parentForm.removeChild(submittableArray);
            }
        } catch (e) {}
        return true;
    },
    
    /**
     * This method creates a hidden "select" element with id 
     * and name attributes set name, values taken from the values
     * array argument, and display labels from the labels array.
     * It adds the element to the parentForm argument.
     * <p>
     * The pupose of this method is to create an array of values
     * that can be decoded using "name" as the key from a FacesContext
     * external context's "getRequestParameterValuesMap" as an
     * array of Strings. This reduces the need of rendering hidden input
     * field and delimiting several strings so that a multiple selection
     * can be returned.
     * </p><p>
     * The labels array provides an additional piece of data
     * for use on the client, but it is not contained in the submit.
     * All values added to the select are selected so that the
     * values will be submitted.
     * </p>
     *
     * @param {String} name The element ID of the html tag 
     * @param {Node} form The HTML form element to submit.
     * @param {Array} labels
     * @param {Array} values
     * @return {Node} The newly created select element.
     * @private
     */
    _createSubmittableArray: function(name, parentForm, labels, values) {
        // An attempt is made to remove a possibly previously created element
        // by this name. It always deletes an element of name from parentForm.
        woodstock4_3._base.common._deleteSubmittableArray(name, parentForm);
        
        if (values == null || values.length <= 0) {
            return null;
        }
        
        var selections = document.createElement('select');
        selections.className = woodstock4_3.theme.common.getClassName("HIDDEN");
        selections.name = name;
        selections.id = name;
        selections.multiple = true;
        
        // Start from the end of the array because
        // add puts things in at the head.
        //
        for (var i = 0; i < values.length; ++i) {
            var opt = document.createElement('option');
            opt.value = values[i];
            if (labels != null) {
                opt.label = labels[i];
            }
            opt.defaultSelected = true;
            selections.add(opt, null);
        }
        parentForm.appendChild(selections);
        return selections;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Visible functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function to test if the specified element is visible (i.e., it
     * does not contain the hidden className).
     *
     * @param {String} elementId The element ID of the html tag 
     * @return {boolean} true if visible; otherwise, false
     * @private
     */
    _isVisible: function(elementId) {
        if (elementId == null) {
            return false;
        }
        // Get element.
        var element = document.getElementById(elementId);
        return woodstock4_3._base.common._isVisibleElement(element);
    },
    
    /**
     * Use this function to test if the given element is visible (i.e., it
     * does not contain the hidden className).
     *
     * @param {Node} element The HTML element
     * @return {boolean} true if visible; otherwise, false
     * @private
     */
    _isVisibleElement: function(element) {
        if (element == null) {
            return false;
        }
        // Test for the hidden style class.
        var styleClasses = woodstock4_3._base.common._splitStyleClasses(element); 
        return !woodstock4_3._base.common._checkStyleClasses(styleClasses,
            woodstock4_3.theme.common.getClassName("HIDDEN"));
    },
    
    /**
     * Use this function to show or hide any html element in the page
     *
     * @param {String} elementId The element ID of the html tag 
     * @param {boolean} visible true to make the element visible, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _setVisible: function(elementId, visible) {
        if (elementId == null || visible == null ) {
            return false;
        }
        // Get element.
        var element = document.getElementById(elementId);
        return woodstock4_3._base.common._setVisibleElement(element, visible);
    },
    
    /**
     * Use this function to show or hide any html element in the page
     *
     * @param {Node} element The HTML element
     * @param {boolean} visible true to make the element visible, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _setVisibleElement: function(element, visible) {
        if (element == null || visible == null) {
            return false;
        }
        if (visible) {
            return woodstock4_3._base.common._stripStyleClass(element,
                woodstock4_3.theme.common.getClassName("HIDDEN"));
        } else {
            return woodstock4_3._base.common._addStyleClass(element,
                woodstock4_3.theme.common.getClassName("HIDDEN"));
        }
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.common = {
    escapeString: woodstock4_3._base.common._escapeString,
    unescapeString: woodstock4_3._base.common._unescapeString,
    unescapeStrings: woodstock4_3._base.common._unescapeStrings,
    addStyleClass: woodstock4_3._base.common._addStyleClass,
    checkStyleClasses: woodstock4_3._base.common._checkStyleClasses,
    splitStyleClasses: woodstock4_3._base.common._splitStyleClasses,
    stripStyleClass: woodstock4_3._base.common._stripStyleClass,
    insertHiddenField: woodstock4_3._base.common._insertHiddenField,
    submitForm: woodstock4_3._base.common._submitForm,
    timeoutSubmitForm: woodstock4_3._base.common._timeoutSubmitForm,
    leaveSubmitterTrace: woodstock4_3._base.common._leaveSubmitterTrace,
    deleteSubmittableArray: woodstock4_3._base.common._deleteSubmittableArray,
    createSubmittableArray: woodstock4_3._base.common._createSubmittableArray,
    isVisible: woodstock4_3._base.common._isVisible,
    isVisibleElement: woodstock4_3._base.common._isVisibleElement,
    setVisible: woodstock4_3._base.common._setVisible,
    setVisibleElement: woodstock4_3._base.common._setVisibleElement
};

/**
 * @class This class contains functions used to maintain focus and scroll position.
 * <p>
 * There can be an initial focus element and a default focus element. The
 * initial focus element is identifed by the "focusElementId" argument.
 * This argument is typically null on the first display of the page. If
 * the Body component is not preserving the focus then it may also be null,
 * at other times, since it represents the element id that last received
 * the focus before the request.
 * </p><p>
 * Whenever the page is displayed and "focusElementId" is null
 * "defaultFocusElementId" will receive the focus if it is not null. This id is
 * defined by the application using the Body "focus" attribute. If the
 * application wants to control the focus in all cases then it should set
 * the Body component "preserveFocus" attribute to "false". The application then
 * explicitly sets the Body "focus" attribute to the element id to receive
 * focus on every request/response.
 * </p><p>
 * In order to preserve focus across requests, the "focusElementFieldId"
 * element is used to preserve the id of the element that receives the focus
 * last. It is a hidden field that is submitted in the
 * request. See the "com.sun.webui.jsf.util.FocusManager" class for
 * information on how the focus is managed. This field exists in all
 * forms, since that it is the only way to guarantee that the last
 * element that received the focus is sent to the server. Which form
 * is submitted can never be known.
 * </p>
 * @constructor This function is used to construct a body object.
 * @param {String} viewId Used to name the scroll cookie
 * @param {String} path A member of the scroll cookie
 * @param {String} defaultFocusElementId The HTML element id that will receive focus.
 * @param {String} focusElementId The id of the element to receive the initial focus.
 * @param {String} focusElementFieldId The id of a hidden field to maintain
 * the id of the last element to have the focus.
 * @param {boolean} preserveScroll if true or not defined the scroll position is 
 * maintained else scroll is not maintained.
 * @private
 */
woodstock4_3._base.body = function(viewId, path, defaultFocusElementId, 
	focusElementId, focusElementFieldId, preserveScroll)  {
    /**
     * The id of the HTML element to receive the focus, if the
     * element identified in focusElementFieldId cannot receive the focus.
     * @private
     */
    this._defaultFocusId = defaultFocusElementId;

    /**
     * The id of a hidden input element whose value is the id
     * of the HTML element to receive the focus. It is set by
     * the _focusListener and calls to setFocusBy{Id,Element}.
     * @private
     */
    this._focusElementFieldId = focusElementFieldId;

    /**
     * The element id to receive the preserved, or initial focus.
     * This member should not be referenced once the onload listener
     * has been invoked. After that point the hidden field will have
     * have the element with the focus. We do this so that two locations
     * do not have to be kept in sync. Developers can just set the focus
     * to the element itself and the focus handler will manage the
     * focus persisitence.
     * @private
     */
    this._focusElementId = focusElementId;

    /**
     * Flag indicating to preserve scroll.
     * @private
     */
    this._preserveScroll = (preserveScroll == null)
        ? true : new Boolean(preserveScroll).valueOf();

    /**
     * Create the scroll cookie object.
     */
    if (this._preserveScroll == true) {
	this._scrollCookie = new woodstock4_3._base.scrollCookie(viewId, path);
    }

    /**
     * According to HTML spec only these elements have
     * "onfocus" which we will equate to "focus".
     * A, AREA, BUTTON, INPUT, LABEL, SELECT, TEXTAREA
     * However just check for a non null "focus" or 
     * catch the exception when trying to reference it.
     * Returns true if the element has a "focus" method, is not
     * disabled, and is visible, else false.
     *
     * @param {Node} element The DOM node to have focus.
     * @return {boolean} true if DOM Node can have focus.
     * @private
     */
    this._canAcceptFocus = function(element) {
	var result = false;
	try {
	    result = element != null && element.focus && !element.disabled
		&& element.type != "hidden"
		&& woodstock4_3._base.common._isVisible(element.id);
	} catch(err) {}
	return result;
    };

    /**
     * Record the id of the element that has just receivied focus.
     * This is called whenever an element receives the focus.
     * This is set on the document so that the cursor entering the
     * window does not trigger this listener.
     *
     * @param {Event} event The object generated by the focus event.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._focusListener = function(event) {
	// If it's not an element node just return
	//
	var node = null;
	var isElementNode = false;
	
	// is IE 
	//
	if (document.attachEvent) {
	    node = event.srcElement;
	
	    // We have to hard code "1" as the Node.ELEMENT_NODE in
	    // ie, because ie does not make the constant accessible.
	    //
	    isElementNode = (node == null ? false : node.nodeType == 1);
	} else {
	    node = event.target;
	    isElementNode = node.nodeType == Node.ELEMENT_NODE;
	}

	if (isElementNode) {
	    // Note that there is no reliable way to set
	    // focus to some other element if the event element
	    // deemed to receive the focus can't accept the focus.
	    //
	    this._updateFocusElementField(node);
	}
	return true;
    };

    /**
     * Set the initial focus and the scroll position.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._onLoadListener = function() {
	// register the focus listener first.
	// Then call "_setDefaultFocus" using "setTimeout" to
	// allow javascript processing to complete, probably
	// to allow "onload" to complete. The focus listener
	// will update the hidden focus fields as a result
	// of the call to "focus" in _setDefaultFocus.
	//

	// Add the focus listener, in the onload to prevent
	// recursive calls from calling _setDefaultFocus.
	//
        if (woodstock4_3._base.browser._isIe()) {
            woodstock4_3._dojo.connect(document, "onfocusin", this, "_focusListener");
        } else {
            woodstock4_3._dojo.connect(window, "onfocus", this, "_focusListener");
        }

        // Rely on the focus listener to update the focus
        // hidden fields by catching the 'element.focus()' in
        // _setDefaultFocus
        //
        this._setInitialFocus();

	// Set up the scroll position after the focus has been
	// restored. Need to make sure that this takes into
	// account the default focus that was just set.
	//
	return this._setDefaultScrollPosition();
    };

    /**
     * Update the page's scroll position.
     *
     * @param {Event} event The object generated by the onUnload event.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._onUnloadListener = function(event) {
	return this._storeScrollPosition();
    };

    /**
     * Set the default focus to the application's chosen default focus element.
     * This method should only be called once to prevent recursive
     * calls since it calls "focus()" on the focus element.
     * Called currently from the onload listener.
     * <p>
     * If "this._defaultFocusId" is not null it will receive the focus; 
     * otherwise, no focus is set.
     * </p>
     * @return {boolean} false if a default focus cannot be established, else true.
     * @private
     */
    this._setDefaultFocus = function() {
        // HTML elements may not have been added, yet.
        if (this._defaultFocusId != null) {
            var domNode = document.getElementById(this._defaultFocusId);
            if (domNode == null) {
                var _this = this; // Closure magic.
                return setTimeout(function() { _this._setDefaultFocus(); }, 10);
            }

            // Focus not set try the default.
            //
            if (this._setFocusById(this._defaultFocusId)) {
                return true;
            }
        }

	/* For now it doesn't sound like a good idea to ever set
	 * a "heuristic" default focus. It is better for screen readers to start
	 * from the top of the page which we will assume that that
	 * browser starts from there when focus is not set explicitly.
	 * This code can be removed, but left it in case we want to
	 * for some reason.

	// No previously set focus element and no default.
	// Find the first focusable element in the first available form.
	//
	for each (var f in window.document.forms) {
	    for each (var e in f.elements) {
		if (this._setFocusByElement(e)) {
		    return true;
		}
	    }
	}
	// If there is no form, set on the first available link
	//
	for each (var l in window.document.links) {
	    if (this._setFocusByElement(l)) {
		return true;
	    }
	}
	*/
	return false;
    };

    /**
     * This method is invoked in the onload listener, body._onLoadListener.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._setDefaultScrollPosition = function() {
	if (!this._preserveScroll) {
	    return false;
	}
	// # char found, anchor being used. forego scrolling.
	// CR 6342635. 
	//
        if (window.location.href.indexOf('#') == -1) {
	    this._scrollCookie._restore(); 
	} else {
	    // destroy the recent scroll data
	    //
	    this._scrollCookie._reset();
	}
        return true;
    };

    /**
     * Set the initial focus by restoring the focus from a previous
     * request or to the application's chosen default focus element.
     * This method should only be called once to prevent recursive
     * calls since it calls "focus()" on the focus element.
     * Called currently from the onload listener.
     * <p>
     * If "this._focusElementId" is not null it will receive the focus.
     * If that element can't receive the focus then the application defined 
     * "this._defaultFocusId" receives the focus. If that element cannot receive 
     * the focus, no focus is set.
     * </p>
     * @return {boolean} false if focus cannot be established, else true.
     * @private
     */
    this._setInitialFocus = function() {
        // HTML elements may not have been added, yet.
        if (this._focusElementId != null) {
            var domNode = document.getElementById(this._focusElementId);
            if (domNode == null) {
                var _this = this; // Closure magic.
                return setTimeout(function() { _this._setInitialFocus(); }, 10);
            }

            // Try to set focus to "this._focusElementId". If this fails
            // fallback to the app defined default 
            // "this.defaultFocusElementId", if there is one.
            //
            if (this._setFocusById(this._focusElementId)) {
                return true;
            }
        }
        return this._setDefaultFocus();
    };

    /**
     * Set the focus on "focusElement".
     * If focus can be set returns true else false.
     *
     * @param {Node} focusElement The DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._setFocusByElement = function(focusElement) {
	if (focusElement == null || !this._canAcceptFocus(focusElement)) {
	    return false;
	}

	// _canAcceptFocus tests the existence of the "focus" handler.
	// So it is safe to call it outside of a try/catch statement.
	// This should trigger the focus listener.
        try {
            // Still need try/catch because _canAcceptFocus doesn't account for 
            // when parent is invisible. For example, the table's sort panel 
            // closes during page submit making focus element invisible.
            focusElement.focus();
        } catch(err) {}

	// Assume that this update is performed by the 
	// focus listener. This policy was changed in order to 
	// call "_setDefaultFocus" using "setTimeout" in order for
	// javascript to have time to be evaluated, probably for
	// on load processing to complete.
	//this._updateFocusElementField(focusElement);
	return true;
    };

    /**
     * Set the focus on element with id "fid".
     * If focus can be set returns true else false.
     *
     * @param {String} fid The id of the DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._setFocusById = function(fid) {
	if (fid == null || fid.length == 0) {
	    return false;
	}
	return this._setFocusByElement(document.getElementById(fid));
    };

    /**
     * This method is invoked in the onunload event listener
     * body._onUnloadListener
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._storeScrollPosition = function() {
	if (!this._preserveScroll) {
	    return false;
	}
	try {
	    this._scrollCookie._set(); 
	} catch (e) {
	}
        return true; 
    };

    /** 
     * Update the hidden field that maintains the last element to 
     * receive the focus. If the body has multiple forms every form's
     * hidden field is updated with the "focusElement".
     *
     * @param {Node} focusElement The DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    this._updateFocusElementField = function(focusElement) {
	// Don't know if we'll have issues if multiple forms contain
	// an element with the same id. I know getElementById gets
	// confused.
	//

        if (focusElement == null) {
	    return false;
	}
	// Get the form that contains the focus element.
	//
	for (var i = 0;  i < document.forms.length; ++i) {
	    var form = document.forms[i];
            var field = null;

	    // Get the hidden field that maintains the focus element id.
	    // If it exists return it. We know its name is the same
	    // as its id.
	    //
	    try {
		field = form.elements[this._focusElementFieldId];
		if (field != null) {
		    field.value = focusElement.id;
		    continue;
		}
	    } catch (e) {
		// the array access of a non existent element
		// probably threw exception so create the field.
	    }
		
	    // If it doesn't exist create it.
	    // and add it to the form.
	    //
	    field = document.createElement('input');
	    field.type = 'hidden';
	    field.id = this._focusElementFieldId;
	    field.name = this._focusElementFieldId;
	    field.value = focusElement.id;
	    form.appendChild(field);
	}
	return true;
    };

    // The focus listener is set on the document so that the cursor 
    // entering the window does not trigger this listener. 
    this._onLoadListener();

    // If we are not preserving scroll don't add the unload listener.
    if (this._preserveScroll == true) {
        woodstock4_3._dojo.addOnUnload(this, "_onUnloadListener");
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.addRemove");

woodstock4_3._dojo.require("woodstock4_3._base.proto");


woodstock4_3._dojo.provide("woodstock4_3._html.listbox");



woodstock4_3._dojo.provide("woodstock4_3.widget.common");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.theme.common");


woodstock4_3._dojo.provide("woodstock4_3.xhr");

/**
 * @class This class contains functions for XMLHTTP requests.
 * @static
 */
woodstock4_3.xhr = {
    /**
     * This function is used to generate a request using the XMLHTTP 
     * request object (XHR) the underlying protocol.
     *
     * @param {String} method Use "GET" on operations that are primarily data
     * retrieval requests (default). Use "POST" on operations that send data to
     * the server.
     * @param {Object} props Key-Value pairs of properties.
     * @config {boolean} async Flag indicating request should asynchronous (default).
     * @config {String} content Optional postable string or DOM object data.
     * @config {Object} headers Key-Value pairs of request header properties.
     * @config {Function} onError The callback function called in an error case.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {Function} onReady The callback function called on a successful response.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {String} url The URL may be either a complete or relative URL.
     * @private
     */
    _doRequest: function(method, props) {
        if (props == null || props.url == null) {
            console.debug("Error: _send has null props"); // See Firebug console.
            return false;
        }
        var xhr = woodstock4_3.xhr._getXhr();
        if (xhr == null) {
            return false;
        }

        // Set callback functions.
        var _onError = props.onError;
        var _onReady = props.onReady;
        xhr.onreadystatechange = function() {
            // State shows "loaded".
            if (xhr.readyState == 4) {
                // Status shows "OK".
                if (xhr.status == 200) {
                    if (typeof _onReady == "function") {
                        _onReady(xhr);
                    }
                } else {
                    if (typeof _onError == "function") {
                        _onError(xhr);
                    }
                }
            }
        }
        // Open XHR.
        xhr.open((method) ? method : "GET", props.url,
            (props.async != null) ? props.async : true);

        // Set default headers -- must set after XHR is open.
        xhr.setRequestHeader("X-Requested-With", "XMLHttpRequest");
        xhr.setRequestHeader("X-Woodstock-Version", "4.3");

        // Set headers.
        if (props.headers) {
            for (var property in props.headers) {
                xhr.setRequestHeader(property, props.headers[property]);
            }
        }
        // Send request.
        xhr.send((props.content) ? props.content : null);
        return true;
    },

    /**
     * This function is used to generate a "GET" request using the XMLHTTP 
     * request object (XHR) the underlying protocol.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {boolean} async Flag indicating request should asynchronous (default).
     * @config {String} content Optional postable string or DOM object data.
     * @config {Object} headers Key-Value pairs of request header properties.
     * @config {Function} onError The callback function called in an error case.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {Function} onReady The callback function called on a successful response.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {String} url The URL may be either a complete or relative URL.
     * @return {boolean} true if successful; otherwise, false.
     */
    get: function(props) {
        return woodstock4_3.xhr._doRequest("GET", props);
    },

    /**
     * Get either an XMLHttpRequest or ActiveXObject object.
     *
     * @private
     */
    _getXhr: function() {
        var xhr = null;
    
        // Use native XMLHttpRequest object, if possible.
        if (window.XMLHttpRequest && !(window.ActiveXObject)) {
            try {
                xhr = new XMLHttpRequest();
            } catch(e) {
                console.debug("Error: XMLHttpRequest not available"); // See Firebug console.
            }
        } else if (window.ActiveXObject) {
            // Use ActiveXObject for IE.
            try {
                xhr = new ActiveXObject("Msxml2.XMLHTTP");
            } catch(e) {
        	try {
                    xhr = new ActiveXObject("Microsoft.XMLHTTP");
        	} catch(e) {
                    console.debug("Error: ActiveXObject not available"); // See Firebug console.
        	}
            }
        }
        return xhr;
    },

    /**
     * This function is used to generate a "POST" request using the XMLHTTP 
     * request object (XHR) the underlying protocol.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {boolean} async Flag indicating request should asynchronous (default).
     * @config {String} content Optional postable string or DOM object data.
     * @config {Object} headers Key-Value pairs of request header properties.
     * @config {Function} onError The callback function called in an error case.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {Function} onReady The callback function called on a successful response.
     * Note: XMLHttpRequest or ActiveXObject will be provided as an argument.
     * @config {String} url The URL may be either a complete or relative URL.
     * @return {boolean} true if successful; otherwise, false.
     */
    post: function(props) {
        return woodstock4_3.xhr._doRequest("POST", props);
    }
};

/**
 * @class This class contains functions common to all widgets.
 * @static
 */
woodstock4_3.widget.common = {
    /**
     * Object used to store onLoad functions.
     * @private
     */
    _loaders: [],

    /**
     * Object used to store private onLoad functions.
     * @private
     */
    _preLoaders: [],

    /**
     * Object used to store widget properties.
     * @private
     */
    _props: new Object(),

    /**
     * This function is used to add a widget, HTML fragment, or static string to
     * the given domNode.
     * <p>
     * Note: If props is a string, it shall be added as the innerHTML of the 
     * given domNode. By default, all strings shall be HTML escaped.
     * <p></p>
     * If props is a JSON object, containing a fragment property instead of
     * widgetType, it shall be added as the innerHTML of the given domNode. A
     * fragment is also a string; however it is evaluated as JavaScript and not
     * HTML escaped.
     * <p></p>
     * If props is a JSON object, containing a widgetType property, a widget
     * shall be created. The newly created widget shall be added as a child of 
     * the given domNode.
     * <p></p>
     * Valid values for the position param consist of "last" or null. In 
     * general, position only applies when creating new widgets; however, if the 
     * position param is null, any existing child nodes are removed from the
     * domNode.
     * </p>
     *
     * @param {Node} domNode The DOM node to add widget.
     * @param {Object} props Key-Value pairs of properties.
     * @param {boolean} position The position to add widget.
     * @param {boolean} escape HTML escape strings (default).
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _addFragment: function(domNode, props, position, escape) {
        if (domNode == null || props == null) {
            return false;
        }
        var common = woodstock4_3.widget.common;

        // If position is null, remove existing nodes. The contents shall be
        // replaced by the newly created widget.
        if (position == null) {
            common._removeChildNodes(domNode);

            // Note: To ensure Dojo does not replace the given domNode, always
            // provide a default position to the _createWidget() function. The
            // domNode may be used as a place holder for later updates.
            position = "last";            
        }

        // Add include, refId, or html object -- do not HTML escape.
        if (common._isFragmentObject(props)) {
            // Add HTML string.
            if (typeof props.html == "string") {
                return common._addFragment(domNode, props.html, position, false);
            } else if (typeof props.include == "string") {
                // Add URL include.
                var _url = props.include;
                woodstock4_3.xhr.get({
                    onReady: function (response) {
                        woodstock4_3.widget.common._addFragment(domNode, 
                            response.responseText, position, false);
                    },
                    url: _url
                });
            } else if (typeof props.refId == "string") {
                // Add widget reference.
                var refIds = props.refId.split(",");
                for (var i = 0; i < refIds.length; i++) {
                    // Ensure nodes have not been added to DOM already.
                    if (document.getElementById(refIds[i]) != null) {
                        continue;
                    }
                    var widget = common.getWidget(refIds[i]);
                    if (widget) {
                        common._addWidget(domNode, widget, position);
                    }   
                }
            }
            return true;
        }
 
        // Add fragment.
        if (typeof props == 'string') {
            // Strip script fragments, set innerHTML property, and
            // eval scripts using a timeout.
            //
            // Note that using Dojo's ContentPane widget would have
            // been more preferable than creating a new dependency
            // based on Prototype. However, as of version .4.1, Dojo
            // still does not use a timeout to eval JavaScript; thus,
            // IE generates errors with innerHTML.
            //
            // "The problem has to do with the browser's poor
            // threading model. Basically, once some JavaScript code
            // is running, all other threads of execution are on hold
            // until the running thread is finished with it's
            // task. This includes whatever thread of execution that
            // updates the DOM model when new content is inserted via
            // innerHTML. For example if you do:
            //
            // foo.innerHTML = '<span id="bar">Bar</span>';
            // var bar = document.getElementById('bar');
            //
            // This code will sometimes fail because although you
            // added the content via innerHTML the DOM may not have
            // been updated and therefore the "bar" element does not
            // exist in the DOM yet. To work around this you can do:
            //
            // foo.innerHTML = '<span id="bar">Bar</span>';
            // setTimeout(function() {
            //     var bar = document.getElementById('bar');
            // }, 10);
            //
            // This will work because in 10 milliseconds whatever
            // event handler is running will have finished, the DOM
            // will update, then the callback will execute and have no
            // problem accessing the DOM element."
            //
            // The full discussion on this topic can be found at:
            //
            // http://www.ruby-forum.com/topic/73990
            //
            if (escape != null && new Boolean(escape).valueOf() == false) {
                // Note: IE does not insert script tags via innerHTML.
                common._appendHTML(domNode, 
                    woodstock4_3._base.proto._stripScripts(props));

                // Ensure _loaded() is not called before widget completion.
                common._props[domNode.id] = "_wait";

                // Evaluate JavaScript.
                setTimeout(function() {
                    var common = woodstock4_3.widget.common;

                    // Eval not required for Mozilla/Firefox, but consistent.
                    woodstock4_3._base.proto._evalScripts(props);
                    if (new Boolean(woodstock4_3._base.config.parseOnLoad).valueOf() == true) {
                        common._parseMarkup(domNode);
                    }
                    delete(common._props[domNode.id]); // Clean up.
                }, 10);
            } else {
                // Static strings must be HTML escaped by default.
                common._appendHTML(domNode,
                    woodstock4_3._base.proto._escapeHTML(props));
            }
        } else {
            // Create widget.
            common._createWidget(domNode, props, position, false);
        }
        return true;
    },

    /**
     * Register a function to be called after the DOM has finished loading 
     * and widgets declared in markup have been instantiated.
     * <p><pre>
     * 
     * woodstock4_3.widget.common.addOnLoad(function);
     * woodstock4_3.widget.common.addOnLoad(object, "name", [arg1, arg2...]);
     *
     * </pre></p>
     * @param {Object|Function} obj A Function or Object containing a function
     * by the given name.
     * @param {String} name The function name to call.
     * @return {boolean} true if successful; otherwise, false.
     */
    addOnLoad: function(obj, name) {
        if (obj == null) {
            return false;
        }

        var common = woodstock4_3.widget.common;
        if (arguments.length == 1){
            common._loaders.push(obj);
        } else if (arguments.length > 1){
            common._loaders.push(common._hitch.apply(this, arguments));
        }
        return true;
    },
 
     /**
     * Register a function to be called after the DOM has finished loading 
     * and widgets declared in markup have been instantiated.
     * <p>
     * Functions are called after widgets have been instantiated, but before
     * loaders set via the public addOnLoad function. This ensures JavaScript
     * APIs are available for JSF components (e.g., file chooser).
     * </p><p><pre>
     * 
     * woodstock4_3.widget.common._addOnLoad(function);
     * woodstock4_3.widget.common._addOnLoad(object, "name", [arg1, arg2...]);
     *
     * </pre></p>
     * @param {Object|Function} obj A Function or Object containing a function
     * by the given name.
     * @param {String} name The function name to call.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _addOnLoad: function(obj, name) {
        if (obj == null) {
            return false;
        }

        var common = woodstock4_3.widget.common;
        if (arguments.length == 1){
            common._preLoaders.push(obj);
        } else if (arguments.length > 1){
            common._preLoaders.push(common._hitch.apply(this, arguments));
        }
        return true;
    },

    /**
     * This function is used to add widgets to given the domNode.
     * <p>
     * Valid values for the position param consist of "before", "last", or null.
     * If the position is "last", the widget is appended as the last child of 
     * the given domNode. If the position is "before", the widget is inserted
     * before the given domNode. If the position is null, the given domNode is 
     * replaced by the resulting HTML.
     * </p>
     *
     * @param {Node} domNode The DOM node to add widget.
     * @param {Node} widget The widget object to add.
     * @param {String} position The position within given domNode.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _addWidget: function(domNode, widget, position) {
        if (domNode == null || widget == null) {
            return false;
        }
        // Add widget to DOM.
        if (position == "last") {
            // Append widget as the last child of the given DOM node.
            domNode.appendChild(widget._domNode);
        } else if (position == "before") {
            // Append widget before given DOM node.
            domNode.parentNode.insertBefore(widget._domNode, domNode);
        } else if (domNode.parentNode) {
            // Replace given DOM node with widget.
            domNode.parentNode.replaceChild(widget._domNode, domNode);
        }
        return true;
    },

    /**
     * This function is used to append HTML strings to the innerHTML property of
     * the given domNode.
     * <p>
     * Note: Concatenating innerHTML with new strings does not always work. When
     * adding multiple HTML elements to domNode, we can get into a situation
     * where domNode.innerHTML may not yet contain all the changes made to 
     * the previously added DOM node. Therefore, we shall wrap new strings in an
     * HTML span element so it may be added as a child of domNode.
     * </p>
     *
     * @param {Node} domNode The DOM node to append string.
     * @param {String} html The HTML string to append.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _appendHTML: function(domNode, html) {
        if (domNode.innerHTML != null && domNode.innerHTML.length > 0) {
            var span = document.createElement('span');            
            span.innerHTML = html;
            domNode.appendChild(span);
        } else {
            // Don't need span when innerHTML is empty.
            domNode.innerHTML = html;
        }
        return true;
    },
    
    /**
     * This function is used to add the context path to a given property.
     * The property could denote an url such as path to an image to be rendered
     * or location of a resource to navigate to when a link is clicked. The
     * property given has to start with a "/". Relative urls that start with 
     * ".." will not have the context path appended. It also checks whether the
     * context path has already been added to the property.
     *
     * @param (String} prefix The context path of the application.
     * @param (String} url The URL to which contextPath is to be appended.
     * @return {String} The property that has the contextPath appended.
     * @private
     */
    _appendPrefix: function(prefix, url) {
        if (prefix == null || url == null) {
            return null;
        }
        return (url.charAt(0) == "/" && url.indexOf(prefix + "/") == -1)
            ? prefix + url : url;
    },

    /**
     * This function is used to create and start a widget. If the parseOnLoad
     * property is true, widget creation is deferred to the window.onLoad event.
     * <p>
     * Typically, an HTML tag is used as a temporary place holder for the newly
     * created widget. This allows widgets to be added to the document in the 
     * proper location. For example, the expected HTML markup to create an image
     * widget looks like so:
     * </p><p><pre>
     * &lt;span id="j_id1"&gt;&lt;/span&gt;
     * &lt;script type="text/javascript"&gt;
     *   woodstock4_3.widget.common.createWidget('j_id1',{
     *     "id": "form:image1",
     *     "height": "32px",
     *     "width": "32px",
     *     "src": "/example/images/dice1.gif",
     *     "widgetType": "image",
     *   });
     * &lt;/script&gt;
     * </pre></p><p>
     * For better lookup performance, it is assumed that script tags are located
     * immediately after an HTML span element. Ultimately, the newly created
     * widget is added as a child of the HTML span element.
     * </p><p>
     * Performance testing shows this approach is quicker than using the 
     * document.getElementById() function or Level 0 DOM syntax, especially for 
     * large HTML tables. The underlying problem appears to be the extra step
     * taken to convert HTML element IDs to a UTF-16 character encoding.
     * </p>
     *
     * @param {String} elementId The HTML element id to replace (may be null).
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The ID of the widget to create.
     * @config {String} widgetType The widget type to create.
     * @return {Object} The widget object if parseOnLoad is false; otherwise, null.
     */
    createWidget: function(elementId, props) {
        if (props == null) {
            return false;
        }
        var common = woodstock4_3.widget.common;

        // Since there is only one window.onLoad event, the ajaxZone tag of JSF
        // Extensions cannot make use of the parseOnLoad feature while 
        // re-rendering widgets. In this case, we shall call 
        // document.getElementById() even though it is not as efficient.
        if (new Boolean(woodstock4_3._base.config.parseOnLoad).valueOf() == false) {
            var domNode = document.getElementById(elementId);
            return common._createWidget(domNode, props, "last");
        }
        // Store widget properties for window.onLoad event.
        common._props[elementId] = props;
        return null;
    },

    /**
     * This function is used to create and start a widget.
     * <p>
     * See the _addNode function for valid position values.
     * </p>
     *
     * @param {Node} domNode The DOM node to add widget.
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The ID of the widget to create.
     * @config {String} widgetType The widget type to create.
     * @param {String} position The widget position within given domNode.
     * @returns {Object} The newly created widget.
     * @private
     */
    _createWidget: function(domNode, props, position) {
        var widget = null;
        if (props == null || props.id == null || props.widgetType == null) {
            console.debug("Error: _createWidget has null props"); // See Firebug console.
            return widget;
        }

        // Destroy previously created widgets, events, etc.
        var common = woodstock4_3.widget.common;
        common.destroyWidget(props.id);

        // Retrieve required module.
        var _widgetType = "woodstock4_3.widget."  + props.widgetType;
        woodstock4_3._dojo.require(_widgetType);
        
        try {
            // Get widget object.
            var obj = woodstock4_3._dojo.getObject(_widgetType);

            // Instantiate widget. 
            // Note: Dojo mixes attributes, if domNode is provided.
            widget = new obj(props);
        } catch (err) {
            var message = "Error: _createWidget falied for id=" + props.id;
	    message = common._getExceptionString(err, message, true);
            console.debug(message); // See Firebug console.
            return null;
        }

        // Add widget to DOM.
        common._addWidget(domNode, widget, position);

        // Start widget.
        widget._startup();
        return widget;
    },
     
    /**
     * This function is used to create and start a widget from parsed markup. 
     * See the _parseMarkup() function.
     * <p>
     * Unlike the _createWidget() function, setTimeout() is called to allow for
     * progressive rendering. Performance testing shows that the download is
     * quicker with the setTimout() call than without. 
     * </p><p>
     * The setTimeout() also helps to display extremely large tables. As an 
     * example, consider a table that displays 1000 rows. Many browsers display 
     * warnings if JavaScript runs longer than 5 seconds. The setTimeout breaks 
     * up the amount of JavaScript run at any given time and completes each 
     * segment within the browser's alloted time.
     * </p>
     *
     * @param {Node} domNode The DOM node to append widget.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _createMarkupWidget: function(domNode) {
        // Ensure a value exists for given domNode id.
        if (domNode == null 
                || woodstock4_3.widget.common._props[domNode.id] == null) {
            return false;
        }
            
        // Set timeout to allow for progressive rendering.
        setTimeout(function() {
            var common = woodstock4_3.widget.common;
            common._createWidget(domNode, common._props[domNode.id], "last");
            delete(common._props[domNode.id]); // Clean up.

            // Test remaining widget properties.
            for (var property in common._props) {
                return; // At least one widget has not been created.
            }
            // Call after all widgets have been created.
            common._loaded();
        }, 0);
        return true;
    },

    /**
     * This function is used to detroy a widget.
     * <p>
     * Note: By default, all descendant widgets are destroyed as well.
     * </p>
     *
     * @param {String} id The widget id to destroy.
     * @return {boolean} true if successful; otherwise, false.
     */
    destroyWidget: function(id) {
        if (id == null) {
            return false;
        }
        // Destroy previously created widgets, events, etc.
        var widget = woodstock4_3.widget.common.getWidget(id);
        if (widget) {
            return widget.destroyRecursive();
        }
        return false;
    },

    /**
     * Return the appropriate event object depending on the browser.
     *
     * @param {Event} event The client side event generated
     * @return {Event} The appropriate event object
     * @private
     */
    _getEvent: function(event) {
        return (event) ? event : ((window.event) ? window.event : null);          
    },
    
    /**
     * Return a formatted string with the information about the exception error.
     * @param {Error} err An exception <code>Error</code> instance.
     * @param {String} synopsis A general message from the error call site.
     * @param {boolean} verbose If true all possible information from the
     * <code>err</code> is formatted into the returned String.
     * @return {String} Formatted information from <code>err</code>.
     * @private
     */
    _getExceptionString: function(err, synopsis, verbose) {
	var msg = (synopsis == null ? "" : synopsis) + "\n";
	msg = msg + (err.name != null ? err.name : "Unnamed Error") + "\n";
	msg = msg + (err.message != null ? err.message : "No message") + "\n";
	if (verbose) {
	    msg = msg + "\n" + 
		(err.fileName != null ? err.fileName : "No filename" + "::") +
		(err.lineNumber != null ? err.lineNumber : "No lineNumber");
	    if (err.stack != null) {
		var stack = err.stack.replace(/()@/g, "\n");
		msg = msg + "\n" + stack;
	    }
	}
	return msg;
    },

    /**
     * This function returns the closest form ancestor of the given DOM node.
     * <p>
     * Note: Traversing the DOM can be slow, but all HTML input elements have a
     * form property. Therefore, avoid using this function when the form can be
     * retrieved via an HTML input element.
     * </p>
     * @param {Node} domNode A DOM node contained in the form.
     * @return {Node} The HTML form element or null if not found.
     * @private
     */
    _getForm: function(domNode) {
        var form = null;
        var obj = domNode;
        while (obj != null) {
            if (obj.tagName == "FORM") {
                form = obj;
                break;
            }
            obj = obj.parentNode;
        }
        return form;
    },

    /**
     * Return the key code of the key which generated the event.
     *
     * @param {Event} event The client side event generated
     * @return {String} The key code of the key which generated the event
     * @private
     */
    _getKeyCode: function(event) {
        if (event == null) {
            return null;
        }
        return (event.keyCode) 
            ? event.keyCode 
            : ((event.which) ? event.which : event.charCode);              
    },

    /**
     * Get array containing the absolute left and top position of the given DOM
     * node relative to the browser window.
     *
     * @param {Node} domNode The DOM node compute position for.
     * @return {Array} Array containing the absolute left and top position.
     * @private
     */
    _getPosition: function(domNode) {
        var leftPos = topPos = 0;
        if (domNode && domNode.offsetParent) {
            leftPos = domNode.offsetLeft;
            topPos = domNode.offsetTop;
            while ((domNode = domNode.offsetParent) != null) {
                leftPos += domNode.offsetLeft;
                topPos += domNode.offsetTop;
            }
        }
        return [leftPos, topPos];
    },

    /**
     * Get the page height, handling standard noise to mitigate browser
     * differences.
     *
     * @return {int} The page height or null if not available.
     * @private
     */
    _getPageHeight: function() {
        // Mozilla browsers.
        if (window.innerHeight) {
            return window.innerHeight;
        }
        // IE strict mode
        if (document.documentElement.clientHeight > 0) {
            return document.documentElement.clientHeight; 
        }
        // IE quirks mode.
        if (document.body.clientHeight) {
            return document.body.clientHeight;
        }
        return null;
    },

    /**
     * Get the page width, handling standard noise to mitigate browser 
     * differences.
     *
     * @return {int} The page height or null if not available.
     * @private
     */
    _getPageWidth: function() {
        // Mozilla browsers.
        if (window.innerWidth) {
            return window.innerWidth;
        }
        // IE strict mode.
        if (document.documentElement.clientWidth > 0) {
            return document.documentElement.clientWidth; 
        }
        // IE quirks mode.
        if (document.body.clientWidth) {
            return document.body.clientWidth;
        }
        return null;
    },

    /**
     * This function is used to obtain a widget.
     *
     * @param {String} id The widget id.
     * @return {Object} The widget object.
     */
    getWidget: function(id) {
        if (id == null) {
            return null;
        }
        return woodstock4_3._dijit.byId(id);
    },

    /**  
     * Create a function call within the given scope. 
     * <p><pre> 
     *  
     * _hitch(object, name, [arg1, arg2...]); 
     * 
     * </pre></p> 
     * @param {Object} scope The scope in which to call the given method. 
     * @param {String} name The function name to call. 
     * @return {Function} The resulting function. 
     * @private 
     */ 
    _hitch: function(scope, name) {
        if (scope == null || name == null) {
            return null;
        }
        // Save given arguments via closure magic. 
        var pre = woodstock4_3._base.proto._toArray(arguments, 2); 
        return function() { 
            // Append new arguments, if any. 
            var args = woodstock4_3._base.proto._toArray(arguments); 
            return scope[name].apply(scope || this, pre.concat(args)); 
        }; 
    },

    /**
     * Return <code>true</code> if <code>props</code> defines a fragment.
     * A fragment may be a string or an object containing known properties.
     * See the _isFragmentObject and _isFragmentWidget functions.
     * 
     * @param {Object} props properties that may define a fragment.
     * @return {boolean} true of <code>props</code> is a fragment
     * else false. If <code>props</code> is null, false is returned.
     * @private
     */
    _isFragment: function(props) {
        var result = false;
        if (props && (typeof props == "string"
                || woodstock4_3.widget.common._isFragmentObject(props)
                || woodstock4_3.widget.common._isFragmentWidget(props))) {
            result = true;
        }
        return result;
    },

    /**
     * Return <code>true</code> if <code>props</code> defines a fragment object.
     * A fragment object defines either an html, include, or refId property and
     * is not a fragment widget -- see _isFragmentWidget.
     * 
     * @param {Object} props properties that may define a fragment object.
     * @return {boolean} true of <code>props</code> is a fragment object
     * else false. If <code>props</code> is null, false is returned.
     * @private
     */
    _isFragmentObject: function(props) {
        var result = false;
        if (props && !woodstock4_3.widget.common._isFragmentWidget(props)
                && (typeof props.html == "string"
                || typeof props.refId == "string"
                || typeof props.include == "string")) {
            result = true;
        }
        return result;
    },

    /**
     * Return <code>true</code> if <code>props</code> defines a fragment widget.
     * A fragment widget is an object which defines <code>widgetType</code> and
     * <code>id</code> properties.  
     * 
     * @param {Object} props properties that may define a fragment widget.
     * @return {boolean} true of <code>props</code> is a fragment widget
     * else false. If <code>props</code> is null, false is returned.
     * @private
     */
    _isFragmentWidget: function(props) {
        var result = false;
        if (props && (props.id && props.id != "")
                && (props.widgetType && props.widgetType != "")) {
            result = true;
        }
        return result;
    },

    /**
     * This function tests for high contrast mode.
     *  
     * @return {boolean} true if high contrast mode.
     * @private
     */
    _isHighContrastMode:  function() {
        var config = woodstock4_3._base.config;
        if (config._isHighContrastMode != undefined) {
            return config._isHighContrastMode;
        }

        // Dojo appends the following div tag in body tag for a11y support.
        //
        // <div style="border-style: solid; border-color: red green; border-width: 1px; 
        //  position: absolute; left: -999px; top: -999px; background-image: 
        //  url(.../blank.gif);" id="a11yTestNode">
        // </div>

        // Currently high contrast mode check is supported for firefox and ie on
        // windows. High contrast mode is not supported for Safari.
        if (woodstock4_3._base.browser._isSafari()) {
            return false;            
        }
        // Get icon properties.
        var props = woodstock4_3.theme.common.getImage("DOT");
        if (props == null) {
            return false;
        }

        // Create div for testing if high contrast mode is on or images are 
        // turned off.
        var domNode = document.createElement("div"); 
        domNode.style.cssText = 'border: 1px solid;' +
            'border-color:red green;' +
            'position: absolute;' +
            'height: 5px;' +
            'top: -999px;' +
            'background-image: url("' + props.src + '");';

        var body = woodstock4_3._dojo.body();
        body.appendChild(domNode);

        // Detect the high contrast mode.
        var bImg = null;
        if (window.getComputedStyle) {
            var styleValue = getComputedStyle(domNode, "");
            bImg = styleValue.getPropertyValue("background-image");
        } else {
            bImg = domNode.currentStyle.backgroundImage;
        }
        if (bImg != null && (bImg == "none" || bImg == "url(invalid-url:)" )) {
            config._isHighContrastMode = true; // High Contrast Mode
        } else {
            config._isHighContrastMode = false;
        }

        // IE throws security exception if domNode isn't removed.
        // This allows widgets to be created before the window.onLoad event.
        body.removeChild(domNode);
        return config._isHighContrastMode;    
    },

    /** 
     * Keyboard key code as stored in browser events 
     *
     * @private
     */
    _keyCodes: woodstock4_3._dojo.keys,
 
     /**
     * This function is called after the DOM has finished loading and widgets
     * declared in markup have been instantiated.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _loaded: function() {
        // After the page has been parsed, there is no need to perform this task
        // again. Setting the parseOnLoad flag to false will allow the ajaxZone
        // tag of JSF Extensions, for example, to re-render widgets properly. 
        // That is, considering there will only ever be one window.onLoad event.
        woodstock4_3._base.config.parseOnLoad = false;

        // Call loaders set via private addOnLoad function.
        var common = woodstock4_3.widget.common;
        for(var i = 0; i < common._preLoaders.length; i++){
            common._preLoaders[i]();
        }
        common._preLoaders = []; // Clean up.

        // Call loaders set via public addOnLoad function.
        for(var i = 0; i < common._loaders.length; i++){
            common._loaders[i]();
        }
        common._loaders = []; // Clean up.
        return true;
    },

    /**
     * This function is used to parse HTML markup in order to create widgets
     * more efficiently. See the createWidget() function.
     *
     * @param {Node} domNode The DOM node containing HTML elements to replace.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _parseMarkup: function(domNode) {
        if (domNode == null) {
            return false;
        }
        // Note: Using document.getElementById() results in poor perfromance. 
        var nodes = domNode.getElementsByTagName("script");
        var common = woodstock4_3.widget.common;

        // If dealing with JSF facet fragments, we must search for span 
        // elements because IE removes HTML script tags from strings.
        if (nodes.length == 0) {
            nodes = domNode.getElementsByTagName("span");
            if (nodes.length == 0) {
                return false;
            }
            // Match span id with props.
            for (var i = 0; i < nodes.length; i++) {
                common._createMarkupWidget(nodes[i]);
            }
        } else {
            // Match script parent id with props.
            for (var i = 0; i < nodes.length; i++) {
                common._createMarkupWidget(nodes[i].parentNode);
            }
        }
        return true;
    },

    /**
     * Publish an event topic.
     *
     * @param {String} topic The event topic to publish.
     * @param {Object} props Key-Value pairs of properties. This will be applied
     * to each topic subscriber.
     * @return {boolean} true if successful; otherwise, false.
     */
    publish: function(topic, props) {
        // Publish an event for custom AJAX implementations to listen for.
        woodstock4_3._dojo.publish(topic, props);
        return true;
    },

    /**
     * This function is used to remove child nodes from given DOM node.
     * <p>
     * Note: Child nodes may be cleared using the innerHTML property. However,
     * IE fails when this property is set via the widget's fillInTemplate 
     * function. In this case, DOM nodes shall be removed manually using the 
     * Node APIs.
     * </p>
     * @param {Node} domNode The DOM node to remove child nodes.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _removeChildNodes: function(domNode) {
        if (domNode == null) {
            return false;
        }
        try {
            domNode.innerHTML = ""; // Cannot be null on IE.
        } catch (e) {
            // Iterate over child nodes.
            while (domNode.hasChildNodes()) {
                var node = domNode.childNodes[0];
                domNode.removeChild(node);
            }
        }
        return true;
    },

    /**
     * This function sleeps for specified milli seconds.
     * 
     * @param {int} delay The amount to delay.
     * @return {boolean} true if current time is greater than the exit time.
     * @private
     */
    _sleep: function(delay) {
        var start = new Date();
        var exitTime = start.getTime() + delay;

        while (true) {
            start = new Date();
            if (start.getTime() > exitTime) {
                return true;
            }
        }
        return false;
    },

    /**
     * Subscribe to an event topic.
     *
     * @param {String} topic The event topic to subscribe to.
     * @param {Object} obj The object in which a function will be invoked, or
     * null for default scope.
     * @param {String|Function} func The name of a function in context, or a 
     * function reference to invoke when topic is published. 
     * @return {boolean} true if successful; otherwise, false.
     */
    subscribe: function(topic, obj, func) {
        woodstock4_3._dojo.subscribe(topic, obj, func);
        return true;
    },

    /**
     * This function is used to update a widget, HTML fragment, or static
     * string for the given domNode.
     * <p>
     * Note: If the widget associated with props.id already exists, the widget's 
     * setProps() function is invoked with the given props param. If the widget 
     * does not exist, the widget object is instantiated via the _addFragment()
     * function -- all params are passed through.
     * </p><p>
     * See woodstock4_3.widget.label._setProps for example.
     * </p>
     * @param {Node} domNode The DOM node used to add widget (default).
     * @param {String} id The id of the widget to update, if available.
     * @param {Object} props Key-Value pairs of properties.
     * @param {String} position The position (e.g., "first", "last", etc.) to add widget.
     * @param {boolean} escape HTML escape static strings -- default is true.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _updateFragment: function(domNode, id, props, position, escape) {
        if (props == null) {
            return false;
        }
        // Ensure props is not a string.
        var common = woodstock4_3.widget.common;
        var widget = (typeof props != 'string') 
            ? common.getWidget(id) : null;

        // Update widget or add fragment.
        if (widget && typeof widget.setProps == "function") {
            widget.setProps(props);
        } else {
            common._addFragment(domNode, props, position, escape);
        }
        return true;
    }
};

// Initialize widgets.
woodstock4_3._dojo.addOnLoad(function() {
    var common = woodstock4_3.widget.common;

    // Defer widget creation until the window.onLoad event.
    if (new Boolean(woodstock4_3._base.config.parseOnLoad).valueOf() == true) {
        common._parseMarkup(woodstock4_3._dojo.body());
    } else {
        common._loaded(); // Widgets created in-line.
    }
});

/**
 * @class This class contains functions for listbox components.
 * @static
 * 
 * @deprecated See woodstock4_3.widget.listbox
 * @private
 */
woodstock4_3._html.listbox = {
    /**
     * This function is invoked by the list onselect action to set the selected, 
     * and disabled styles.
     *
     * Page authors should invoke this function if they set the selection
     * using JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed();
     */
    changed: function(elementId) {         
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget._changed();
        }
        return false;
    },

    /**
     * Use this function to access the HTML select element that makes up
     * the list. 
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the list).
     * @return {Node} The HTML select element.
     * @deprecated Use document.getElementById(elementId).getSelectElement()
     */
    getSelectElement: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the label of the first
     * selected option on the listbox. If no option is selected, this
     * function returns null. 
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The label of the selected option, or null if none is selected.
     * @deprecated Use document.getElementById(elementId).getSelectedLabel();
     */
    getSelectedLabel: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectedLabel();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the value of the first
     * selected option on the listbox. If no option is selected, this
     * function returns null. 
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The value of the selected option, or null if none is
     * selected.
     * @deprecated Use document.getElementById(elementId).getSelectedValue();
     */
    getSelectedValue: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectedValue();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to set the enabled/disabled state
     * of the listbox component. In addition to disabling the list, it
     * also changes the styles used when rendering the component. 
     *
     * Page authors should invoke this function if they dynamically
     * enable or disable a list using JavaScript.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.listbox = woodstock4_3._html.listbox;

/** 
 * @class This class contains functions for addRemove components.
 * @static
 * @private
 */
woodstock4_3._html.addRemove = {
    /**
     * This function is used to initialize HTML element properties with Object 
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element id.
     * @config {String} separator The character deliminator for ordered options.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize addRemove.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Note that _available, and _selected are not a facets
        // and therefore do not require the use of "facetid" as discussed below

        // The select element from which selections are made 
        domNode.availableList = document.getElementById(props.id + "_available");

        // The select element in which selections are shown 
        domNode.selectedList = document.getElementById(props.id + "_selected");

        // Bug 6338492 -
        //     ALL: If a component supports facets or children is must be a
        //      NamingContainer
        // Since AddRemove has become a NamingContainer the id's for
        // the facet children are prefixed with the AddRemove id
        // in addition to their own id, which also has the 
        // AddRemove id, as has been the convention for facets. This introduces
        // a redundancy in the facet id so the add button now looks like
        //
        // "formid:addremoveid:addremoveid_addButton"
        //
        // It used to be "formid:addremoveid_addButton"
        // It would be better to encapsulate that knowledge in the
        // AddRemove renderer as does FileChooser which has the
        // same problem but because the select elements are not
        // facets in AddRemove they really do only have id's of the
        // form "formid:addremoveid_list_value". Note that 
        // in these examples the "id" parameter is "formid:addremoveid"
        //
        // Therefore for now, locate the additional prefix here as the
        // "facet" id. Assume that id never ends in ":" and if there is
        // no colon, id is the same as the component id.
        var componentid = props.id;
        var colon_index = componentid.lastIndexOf(':');
        if (colon_index != -1) {
            componentid = props.id.substring(colon_index + 1);
        }
        var facetid = props.id + ":" + componentid;

        domNode.addButton = document.getElementById(facetid + "_addButton");
        domNode.addAllButton = document.getElementById(facetid + "_addAllButton");
        domNode.removeButton = document.getElementById(facetid + "_removeButton");
        domNode.removeAllButton = document.getElementById(facetid + "_removeAllButton");
        domNode.moveUpButton = document.getElementById(facetid + "_moveUpButton");
        domNode.moveDownButton = document.getElementById(facetid + "_moveDownButton");

        // _list_value and _item_list are not facets and do not need facetid
        domNode.selectedValues = document.getElementById(props.id + "_list_value");

        // Calculate the value indices
        var itemString = document.getElementById(props.id + "_item_list");

        // HTML elements may not have been created, yet. The moveUp/Down and 
        // remove/All buttons may not exist at all.
        if (itemString == null
                || domNode.availableList == null 
                || domNode.selectedList == null
                || domNode.selectedValues == null
                || domNode.addButton == null
                || domNode.removeButton == null
                || (new Boolean(props.selectAll).valueOf() == true
                    && (domNode.addAllButton == null || domNode.removeAllButton == null))
                || (new Boolean(props.moveButtons).valueOf() == true
                    && (domNode.moveUpButton == null || domNode.moveDownButton == null))) {
            return setTimeout(function() {
                woodstock4_3._html.addRemove._init(props);
            }, 10);
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

        // Calculate the value indices
        if (itemString != null) {
            var string = new String(itemString.value);
            domNode.allValues = string.split(props.separator);	
        } else {
            domNode.allValues = new Array();
        }

        // The options of the select element from which selections are made 
        domNode.availableOptions = domNode.availableList.options;

        // The options of the select element in which selections are shown 
        domNode.selectedOptions = domNode.selectedList.options;

        // Set public functions.
        domNode.add = woodstock4_3._html.addRemove.add;
        domNode.addAll = woodstock4_3._html.addRemove.addAll;
        domNode.remove = woodstock4_3._html.addRemove.remove;
        domNode.removeAll = woodstock4_3._html.addRemove.removeAll;
        domNode.moveUp = woodstock4_3._html.addRemove.moveUp;
        domNode.moveDown = woodstock4_3._html.addRemove.moveDown;
        domNode.updateButtons = woodstock4_3._html.addRemove.updateButtons;

        // Set private functions.
        domNode.calculateIndex = woodstock4_3._html.addRemove.calculateIndex;
        domNode.moveOption = woodstock4_3._html.addRemove.moveOption;
        domNode.updateValue = woodstock4_3._html.addRemove.updateValue;
        domNode.allowMultipleAdditions = woodstock4_3._html.addRemove.allowMultipleAdditions;
        domNode.availableOnChange = woodstock4_3._html.addRemove.availableOnChange;
        domNode.selectedOnChange = woodstock4_3._html.addRemove.selectedOnChange;

        // Enable multiple buttons.
        if (new Boolean(props.duplicateSelections).valueOf() == true) {
            domNode.allowMultipleAdditions();
        }

        // Initialize buttons.
        domNode.updateButtons();
        return true;
    },

    /**
     * This function adds options to the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    add: function() {
        if (this.availableOptions.selectedIndex == -1) {
            return false;
        }

        var sort = this.sort && (this.moveUpButton == null);

        // deselect everything in the selected list
        this.selectedList.selectedIndex = -1;
        return this.moveOption(this.availableOptions, this.selectedOptions,
            this.selectedList, sort);
    },

    /**
     * This function removes options to the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    remove: function() {
        if (this.selectedOptions.selectedIndex == -1) {
            return false;
        }

        // deselect everything in the selected list
        this.availableList.selectedIndex = -1;
        return this.moveOption(this.selectedOptions, this.availableOptions,
            this.availableList, this.sort);
    },

    /**
     * This function moves options in the selected list.
     *
     * @param {Array} moveFromOptions
     * @param {Array} moveToOptions
     * @param {Array} moveToList
     * @param {boolean} sort
     * @return {boolean} true if successful; otherwise, false.
     */
    moveOption: function(moveFromOptions, moveToOptions, moveToList, sort) {
        var index = moveFromOptions.selectedIndex;
        if (index == -1) {
            return false;
        }

        // Keep moving selected items until there aren't any more valid ones
        while (index != -1 && index < moveFromOptions.length - 1) {
            var lastOption = moveToOptions.length - 1;

            // This is the option we're moving
            var curSelection = moveFromOptions[index];

            // This is the index where we insert the option...
            var insertionIndex = 0;
	
            // ...and this is the option at that index
            var insertionOption;
 
            if (sort) {
                // If there are no buttons to move the selected items up or
                // down, then we preserve the sorting order of the available
                // items. We calculate the index of the selected item (based 
                // on the indices assigned when parsing the allValues
                // variable), and then we check each selected item until we
                // reach an item with a higher index.
                var itemIndex = this.calculateIndex(curSelection.value);
                for (var counter = 0;counter < lastOption + 1;++counter) {
                    insertionOption = moveToOptions[counter];
                    if (itemIndex < this.calculateIndex(insertionOption.value)) {
                        insertionIndex = counter;
                        break;
                    }
                }
            } else {
                // If there are buttons to move the options around, then we
                // simply add the new items in the last position
                insertionIndex = lastOption;
                insertionOption = moveToOptions[lastOption];
            }

            // To insert the item, Mozilla works different from Windows
            // and Opera.
            if (moveFromOptions.remove == null) {
                // Case 1: Mozilla
                moveToList.add(curSelection, insertionOption);
            } else {
                // Case 2: Windows and Opera
                moveFromOptions.remove(index);
                moveToOptions.add(curSelection, insertionIndex);
            }
	
            // Make sure the item is selected (this is needed for Opera)
            moveToOptions[insertionIndex].selected = true;

            // Update the options
            lastOption++;

            // Get the next selected index. 
            index = moveFromOptions.selectedIndex;
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * This function adds all options to the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    addAll: function() {
        var numOptions = this.availableOptions.length - 1;
        for (var index = 0;index < numOptions;++index) {
            if (this.availableOptions[index].disabled == false) {
                this.availableOptions[index].selected = true;
            }
        }
        return this.add();
    },

    /**
     * This function removes all options from the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    removeAll: function() {
        var numOptions = this.selectedOptions.length - 1;
        for (var index = 0;index < numOptions;++index) {
            if (this.selectedOptions[index].disabled == false) {
                this.selectedOptions[index].selected = true;
            }
        }
        return this.remove();
    },

    /**
     * This function moves options up in the selected list.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveUp: function() {
        // The original allowed items to be moved on both lists. Surely we
        // only sort items on the selected list? 
        // This does not work on Mozilla.

        // We will not move the last item - it's the separator
        var numOptions = this.selectedOptions.length - 1;

        // If there aren't at least two more selected items, then there is
        // nothing to move 
        if (numOptions < 2) {
            return false;
        }

        // Start by examining the first item 
        var index = 0;

        // We're not going to move the first item. Instead, we will start
        // on the first selected item that is below an unselected
        // item. We identify the first unselected item on the list, and 
        // then we will start on next item after that
        while (this.selectedOptions[index].selected) {
            ++index;
            if (index == numOptions) {
                // We've reached the last item - no more items below it so
                // we return
                return false;
            }
        }

        // Start on the item below this one 
        ++index;

        for (index; index < numOptions; ++index) {
            if (this.selectedOptions[index].selected == true) {
                var curOption = this.selectedOptions[index];
                if (this.selectedOptions.remove == null) {
                    // For Mozilla
                    this.selectedOptions[index] = null;
                    this.selectedList.add(curOption,
                        this.selectedOptions[index - 1]);
                } else {
                    // Windows and Opera do
                    this.selectedOptions.remove(index);
                    this.selectedOptions.add(curOption, index - 1);
                }
                // This is needed for Opera only
                this.selectedOptions[index - 1].selected = true;
            }
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * This function moves options down in the selected list.
     *
     * @return {boolean} false to cancel JavaScript event.
     */
    moveDown: function() {
        // The original allowed items to be moved on both lists. Surely we
        // only sort items on the selected list? 
        // This does not work on Mozilla

        // Last option is numOption -1. That is the separator and we don't
        // move it. We start by examining the second to last item. 
        var index = this.selectedOptions.length - 2;

        // If this number is less than zero, there was nothing on the list
        // and we return
        if (index < 0) {
            return false;
        }

        // We're not going to move the last item. Instead, we will start
        // on the last selected item that is above an unselected
        // item. We identify the last unselected item before the separator
        // and then we start with the item above that one. 
        while (this.selectedOptions[index].selected) {
            --index;
            if (index == 0) {
                // We've reached the first item - no item above it so we
                // return 
                return false;
            }
        }

        // Start on the item above this one 
        --index;

        for (index;index > -1;--index) {
            if (this.selectedOptions[index].selected == true) {
                var curOption = this.selectedOptions[index];
                if (this.selectedOptions.remove == null) {
                    // For Mozilla
                    this.selectedOptions[index] = null;
                    this.selectedList.add(curOption, 
                        this.selectedOptions[index + 1]);
                } else {
                    // Windows and Opera do
                    this.selectedOptions.remove(index);
                    this.selectedOptions.add(curOption, index + 1);
                }
                // This is needed for Opera only
                this.selectedOptions[index + 1].selected = true;
            }
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * This function updates the state of all buttons.
     *
     * @return {boolean} false to cancel JavaScript event.
     */
    updateButtons: function() {
        var numOptions = this.availableOptions.length-1;
        var setting;
        
        // Disabled items should not be moved and buttons should not be enabled
        // for selected disabled items (IE problem)
        for (var i = 0; i < numOptions; ++i) {
            if (this.availableOptions[i].disabled == true) {
                this.availableOptions[i].selected = false;
            }
        }

        var index = this.availableOptions.selectedIndex;

        // The Add button is enabled if there is at least one option
        // to select from and at least one item is selected
        if (this.addButton != null) {
            setting = numOptions < 1 || index == -1;
            if (this.addButton.setDisabled != null) {
                this.addButton.setDisabled(setting);
            } else {
                this.addButton.disabled = setting;
            }
        }

        // The Add All button is enabled if there is at least one option
        // to select from, and disabled otherwise
        if (this.addAllButton != null) {
            var counter = 0;
            // If available item list is disabled then AddAll button should be disabled 
            // irrespective of options element in list.
            if (this.availableList.disabled == false) {
                for (index = 0; index < numOptions; ++index) {
                    if (this.availableOptions[index].disabled == false) {
                        ++counter;
                    }
                }
            } 
            setting = (counter < 1);            
            if (this.addAllButton.setDisabled != null) {
                this.addAllButton.setDisabled(setting);
            } else {
                this.addAllButton.disabled = setting;
            }
        }

        // The remaining buttons are enabled/disabled based on the 
        // items on the selected list
        index = this.selectedOptions.selectedIndex;
        numOptions = this.selectedOptions.length - 1;
        
        if (this.removeAllButton != null) {
            counter = 0;
            // If selected item list is disabled then RemoveAll button should be disabled 
            // irrespective of options element in list.  
            if (this.selectedList.disabled == false) {
                for (index = 0; index < numOptions; ++index) {
                     if (this.selectedOptions[index].disabled == false) {
                         ++counter;
                     }
                }
            } 
            setting = (counter < 1);
            if (this.removeAllButton.setDisabled != null) {
                this.removeAllButton.setDisabled(setting);
            } else {
                this.removeAllButton.disabled = setting;
            }
        }

        // If there are no selected items or if none of them are selected,
        // we disable Remove, Move Up, Move Down
        index = this.selectedOptions.selectedIndex;
        var noItems = numOptions < 1 || index == -1;
        if (this.removeButton != null) {
            if (this.removeButton.setDisabled != null) {
                this.removeButton.setDisabled(noItems);
            } else {
                this.removeButton.disabled = noItems;
            }
        }

        // The Move Up button is enabled (setting = false) provided that
        // there is at least one selected item that is below an unselected item 
        if (this.moveUpButton != null) {
            setting = true;
            if (noItems != true) {
                // Find the first un-selected option, then see if there is
                // a selected option below that one
                var found = false;
                var unselected = -1;
                for (index = 0; index < numOptions; ++index) {
                    if (unselected == -1) {
                        if (this.selectedOptions[index].selected == false) {
                            unselected = index;
                        }
                    } else {
                        if (this.selectedOptions[index].selected == true) {
                            setting = false;
                            break;
                        }
                    }
                }
            }
            if (this.moveUpButton.setDisabled != null) {
                this.moveUpButton.setDisabled(setting);
            } else {
                this.moveUpButton.disabled = setting;
            }
        }

        // The Move Down button is enabled (setting = false) provided that
        // there is at least one unselected item below a selected item.
        if (this.moveDownButton != null) {
            setting = true;
            if (noItems != true) {	     
                for (index = this.selectedOptions.selectedIndex; index < numOptions; ++index) {
                    if (this.selectedOptions[index].selected == false) {
                        setting = false;
                    }
                }
            }
            if (this.moveDownButton.setDisabled != null) {
                this.moveDownButton.setDisabled(setting);
            } else {
                this.moveDownButton.disabled = setting;
            }
        }
        // set the focus to the list which has some selected item(s).
        // this needs to be done to shift the focus from disabled button (Mozilla)
        if (this.selectedOptions.selectedIndex > -1) {
             this.selectedList.focus();
        } else if (this.availableOptions.selectedIndex > -1) {
             this.availableList.focus();
        }
        return false;
    },

    /**
     * Calculate the current index.
     *
     * @return {int} The current index.
     */
    calculateIndex: function(value, lastIndex) {
        var string = new String(value);
        for (var counter=0; counter < this.allValues.length; counter++) {
            if (string == this.allValues[counter]) {
                return counter;
            }
        }
        // Something went wrong. Return the index before the separator 
        return this.allValues.length - 2;
    },

    /**
     * Update current value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    updateValue: function() {
        // Remove the options from the select that holds the actual
        // selected values
        while (this.selectedValues.length > 0) {
            this.selectedValues.remove(0);
        }

        // Create a new array consisting of the options marked as selected
        // on the official list
        var newOptions = new Array();
        var cntr = 0;
        var newOption;

        while (cntr < this.selectedOptions.length-1) {
            newOption = document.createElement("option");
            if (this.selectedOptions[cntr].text != null) {
                newOption.text = this.selectedOptions[cntr].text;
            }
            if (this.selectedOptions[cntr].value != null) {
                newOption.value = this.selectedOptions[cntr].value;
            }
            newOption.selected = true;
            newOptions[newOptions.length] = newOption;
            ++ cntr;
        }

        cntr = 0;
        if (this.selectedOptions.remove == null) {
            // For Mozilla
            while (cntr < newOptions.length) {
                this.selectedValues.add(newOptions[cntr], null);
                ++cntr;
            }
        } else {
            // Windows and Opera do
            while (cntr < newOptions.length) {
                this.selectedValues.add(newOptions[cntr], cntr);
                ++cntr;
            }
        }
        return true;
    },

    /**
     * Initialize multiple additions.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    allowMultipleAdditions: function() {
        // Replace the add and remove functions with functions which 
        // leave the available items as they are
        this.add = woodstock4_3._html.addRemove._multipleAdd;
        this.remove = woodstock4_3._html.addRemove._multipleRemove;
        return true;
    },

    /**
     * Add multiple options.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _multipleAdd: function() {
        this.selectedList.selectedIndex = -1;
        var index = this.availableOptions.selectedIndex;
        if (index == -1) {
            return false;
        }
    
        // keep moving selected items until there aren't any more valid ones
        while (index != -1 && index < this.availableOptions.length - 1) {
            var lastOption = this.selectedOptions.length - 1;

            // This is the option we're moving
            var curSelection = this.availableOptions[index];
            curSelection.selected = false;
            var addSelection = new Option();
            addSelection.text = curSelection.text;
            addSelection.value = curSelection.value;

            // This is the index where we insert the option...
            var insertionIndex = 0;
            // ...and this is the option at that index
            var insertionOption;

            // If there are no buttons to move the selected items up or
            // down, then we preserve the sorting order of the available
            // items. We calculate the index of the selected item (based 
            // on the indices assigned when parsing the allValues
            // variable), and then we check each selected item until we
            // reach an item with a higher index. 

            // We sort if there are no move buttons
            var sort = (this.moveUpButton == null);

            if (sort) {
                var itemIndex = this.calculateIndex(curSelection.value);
                for (var counter = 0; counter < lastOption + 1; ++counter) {
                    insertionOption = this.selectedOptions[counter];
                    if (itemIndex < this.calculateIndex(insertionOption.value)) {
                        insertionIndex = counter;
                        break;
                    }
                }
            } else {
                // If there are buttons to move the options around, then we
                // simply add the new items in the last position
                insertionIndex = lastOption;
                insertionOption = this.selectedOptions[lastOption];
            }

            // To insert the item, Mozilla works different from Windows
            // and Opera. 
            if (this.selectedOptions.remove == null) {
                // Case 1: Mozilla
                this.selectedList.add(addSelection, insertionOption);
            } else {
                // Case 2: Windows and Opera
                this.selectedOptions.add(addSelection, insertionIndex);
            }
	
            // Make sure the item is selected (this is needed for Opera)
            this.selectedOptions[insertionIndex].selected = true;

            // Update the options
            lastOption++;

            // Get the next selected index. 
            index = this.availableOptions.selectedIndex;
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * Remove multiple options.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _multipleRemove: function() {
        this.availableList.selectedIndex = -1;
        var index = this.selectedOptions.selectedIndex;
        if (index == -1) {
            return false;
        }

        while (index < this.selectedOptions.length - 1) {
            if (this.selectedOptions[index].selected) {
                if (this.selectedOptions.remove == null) {
                    // Case 1: Mozilla
                    this.selectedOptions[index] = null;
                } else {
                    // Case 2: Windows and Opera
                    this.selectedOptions.remove(index);
                }
            } else {
                index++;
            }
        }
        this.updateValue();
        this.updateButtons();
        return false;
    },

    /**
     * Process available on change event.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    availableOnChange: function() {
        this.selectedList.selectedIndex = -1;
        this.updateButtons();
        return false;
    },

    /**
     * Process selected on change event.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    selectedOnChange: function() {
        this.availableList.selectedIndex = -1;
        this.updateButtons();
        return false;
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.button");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for button components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.button
 * @private
 */
woodstock4_3._html.button = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated See woodstock4_3.widget.button
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize button.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var widget = woodstock4_3.widget.common.getWidget(props.id);
        if (widget == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set functions
        widget._domNode.isSecondary = woodstock4_3._html.button.isSecondary;
        widget._domNode.setSecondary = woodstock4_3._html.button.setSecondary;
        widget._domNode.isPrimary = woodstock4_3._html.button.isPrimary;
        widget._domNode.setPrimary = woodstock4_3._html.button.setPrimary;
        widget._domNode.isMini = woodstock4_3._html.button.isMini;
        widget._domNode.setMini = woodstock4_3._html.button.setMini;
        widget._domNode.getDisabled = woodstock4_3._html.button.getDisabled;
        widget._domNode.setDisabled = woodstock4_3._html.button.setDisabled;
        widget._domNode.getVisible = woodstock4_3._html.button.getVisible;
        widget._domNode.setVisible = woodstock4_3._html.button.setVisible;
        widget._domNode.getText = woodstock4_3._html.button.getText;
        widget._domNode.setText = woodstock4_3._html.button.setText;
        widget._domNode.doClick = woodstock4_3._html.button.click;

        return true;
    },

    /**
     * Simulate a mouse click in a button. 
     *
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).click();
     */
    click: function() {
        return this.click();
    },

    /**
     * Get the textual label of a button. 
     *
     * @return {String} The element value.
     * @deprecated Use document.getElementById(id).getProps().value;
     */
    getText: function() {
        return this.getProps().value;
    },

    /**
     * Set the textual label of a button. 
     *
     * @param {String} text The element value
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({value: "text"});
     */
    setText: function(text) {
        return this.setProps({value: text});
    },

    /**
     * Use this function to show or hide a button. 
     *
     * @param {boolean} show true to show the element, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({visible: boolean});
     */
    setVisible: function(show) {
        if (show == null) {
            return null;
        }
        return this.setProps({visible: show});
    },

    /**
     * Use this function to find whether or not this is visible according to our
     * spec.
     *
     * @return {boolean} true if visible; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().visible;
     */
    getVisible: function() {
        return this.getProps().visible;
    },

    /**
     * Test if button is set as "primary".
     *
     * @return {boolean} true if primary; otherwise, false for secondary
     * @deprecated Use document.getElementById(id).getProps().primary;
     */
    isPrimary: function() {
        return this.getProps().primary;
    },

    /**
     * Set button as "primary".
     *
     * @param {boolean} primary true for primary, false for secondary
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({primary: boolean});
     */
    setPrimary: function(primary) {
        if (primary == null) {
            return null;
        }
        return this.setProps({primary: primary});
    },

    /**
     * Test if button is set as "secondary".
     *
     * @return {boolean} true if secondary; otherwise, false for primary
     * @deprecated Use !(document.getElementById(id).getProps().primary);
     */
    isSecondary: function() {
        return !(this.getProps().primary);
    },

    /**
     * Set button as "secondary".
     *
     * @param {boolean} secondary true for secondary, false for primary
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({primary: false});
     */
    setSecondary: function(secondary) {
        if (secondary == null) {
            return null;
        }
        return this.setProps({primary: !secondary});
    },

    /**
     * Test if button is set as "mini".
     *
     * @return {boolean} true if mini; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().mini;
     */
    isMini: function() {
        return this.getProps().mini;
    },

    /**
     * Set button as "mini".
     *
     * @param {boolean} mini true for mini, false for standard button
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({mini: boolean});
     */
    setMini: function(mini) {
        if (mini == null) {
            return null;
        }
        return this.setProps({mini: mini});
    },

    /**
     * Test disabled state of button.
     *
     * @return {boolean} true if disabled; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().disabled;
     */
    getDisabled: function() {
        return this.getProps().disabled;
    },

    /**
     * Test disabled state of button.
     *
     * @param {boolean} disabled true if disabled; otherwise, false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(disabled) {
        if (disabled == null) {
            return null;
        }
        return this.setProps({disabled: disabled});
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.button = woodstock4_3._html.button;


woodstock4_3._dojo.provide("woodstock4_3._html.checkbox");



woodstock4_3._dojo.provide("woodstock4_3._html.rbcb");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for checkbox and radio button components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.rbcbGroup
 * @private
 */
woodstock4_3._html.rbcb = {
    /**
     * 
     * @param {String} elementId The element Id.
     * @param {boolean} checked true or false
     * @param {String} type
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */ 
    setChecked: function(elementId, checked, type) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({checked: checked});
        }
        return null; 
    },

    /**
     *
     * @param {String} elementId The element Id.
     * @param {boolean} disabled true or false
     * @param {String} type
     * @param {String} enabledStyle
     * @param {String} disabledStyle
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */ 
    setDisabled: function(elementId, disabled, type, enabledStyle,
            disabledStyle) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null; 
    },

    /** 
     * Set the disabled state for all radio buttons with the given controlName.
     * If disabled is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id
     * @param {String} formName The name of the form containing the element
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled, type, enabledStyle,
            disabledStyle) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.rbcb = woodstock4_3._html.rbcb;

/**
 * @class This class contains functions for checkbox components.
 * @static
 * 
 * @deprecated See woodstock4_3.widget.checkbox
 * @private
 */
woodstock4_3._html.checkbox = {
    /**
     * Set the disabled state for the given checkbox element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) {
        return woodstock4_3._html.rbcb.setDisabled(elementId, disabled,
            "checkbox", "Cb", "CbDis");
    },

    /** 
     * Set the disabled state for all the checkboxes in the check box
     * group identified by controlName. If disabled
     * is set to true, the check boxes are shown with disabled styles.
     *
     * @param {String} controlName The checkbox group control name
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled) {    
        return woodstock4_3._html.rbcb.setGroupDisabled(controlName,
            disabled, "checkbox", "Cb", "CbDis");
    },

    /**
     * Set the checked property for a checkbox with the given element Id.
     *
     * @param {String} elementId The element Id
     * @param checked true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */
    setChecked: function(elementId, checked) {
        return woodstock4_3._html.rbcb.setChecked(elementId, checked,
            "checkbox");
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.checkbox = woodstock4_3._html.checkbox;


woodstock4_3._dojo.provide("woodstock4_3._html.commonTasksSection");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.theme.common");

/** 
 * @class This class contains functions for commonTasksSection components.
 * @static
 * @private
 */
woodstock4_3._html.commonTasksSection = {
    /**
     * This function is used to initialize HTML element properties with Object 
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element id.
     * @config {String} pic1URL Selected image.
     * @config {String} pic2URL Hover image.
     * @config {String} pic3URL Normal image.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize commonTasksSection.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

        // Set functions.
	domNode.captureCloseKey = woodstock4_3._html.commonTasksSection.captureCloseKey;
	domNode.captureBottomInfoKey = woodstock4_3._html.commonTasksSection.captureBottomInfoKey;
        domNode.hideAll = woodstock4_3._html.commonTasksSection.hideAll;
        domNode.addCommonTask = woodstock4_3._html.commonTasksSection.addCommonTask;
        domNode.addInfoPanel = woodstock4_3._html.commonTasksSection.addInfoPanel;
        domNode.windowResize = woodstock4_3._html.commonTasksSection.windowResize;
        domNode.onclick = domNode.hideAll;

        // Set task element array.
        domNode.taskElement = new Array();
        domNode.count = 0;

        // Hide panels on resize.
        woodstock4_3._dojo.connect(window, 'onresize', domNode, domNode.windowResize);

        return true;
    },

    /**
     * Hide all task sections.
     *
     * @param {Event} event The JavaScript event.
     * @return {boolean} true if successful; otherwise, false.
     */
    hideAll: function(event) {
        for (var i = 0; i < this.count; i++) {
            task = this.taskElement[i];
            if (task.infoPanel) {
               woodstock4_3._base.common._setVisibleElement(task.infoPanel.info, false);
               task.infoPanel.image.src = this.pic3URL;
            }
        }
        if (woodstock4_3._base.browser._isIe5up()) {
            window.event.cancelBubble = true;
        } else {
            event.stopPropagation();
        }
        return true;
    },

    /**
     * This function handles window resize events.
     *
     * @param {Event} event The JavaScript event.
     * @return {boolean} true if successful; otherwise, false.
     */
    windowResize: function(event) {
        for (var i = 0; i < this.count; i++) {
            task = this.taskElement[i];
            if (task.infoPanel) {
               woodstock4_3._base.common._setVisibleElement(task.infoPanel.info, false);
               task.infoPanel.image.src = this.pic3URL;
            }
        }
        return true;
    },

    /**
     * This function is used to set common task properties using Object literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} commonTaskId
     * @config {String} closeId
     * @config {String} spacerId
     * @config {String} infoIconId
     * @config {String} infoPanelVar
     * @config {String} imageLinkId
     * @return {boolean} true if successful; otherwise, false.
     */
    addCommonTask: function(props) {
        if (props == null) {
            return false;
        }

        // Get HTML elements.
        var info = document.getElementById(props.commonTaskId + props.infoPanelVar);  //id of the info panel box.
        var image = document.getElementById(props.infoIconId); // id of the "i" image .
        var imageLink = document.getElementById(props.imageLinkId);
        var close = document.getElementById(props.closeId); // id of the close button.	
        var parent = document.getElementById(this.id);
        var task = document.getElementById(props.commonTaskId);
        var bottomInfoLink = (props.bottomInfoLink)
            ? document.getElementById(props.bottomInfoLink) 
            : null; // The bottom info panel id.
        var spacer = props.commonTaskId + ":" + props.spacerId; // id of the spacer image.

        // HTML elements may not have been created, yet.
        if (parent == null
                || (props.bottomInfoLink && bottomInfoLink == null)
                || (props.closeId && close == null)
                || (props.commonTaskId && props.infoPanelVar && info == null)
                || (props.commonTaskId && task == null)
                || (props.infoIconId && image == null)
                || (props.imageLinkId && imageLink == null)) {
            return setTimeout(function() {
                parent.addCommonTask(props);
            }, 10);
        }

        // Set info panel.
        var taskElement = document.getElementById(props.commonTaskId);
        taskElement.infoPanel = new this.addInfoPanel(info, image, imageLink,
            close, parent, task, bottomInfoLink, spacer);

        // Add task element to domNode.
        this.taskElement[this.count] = taskElement;
        this.count++;
        return true;
    },
    
    /**
     * Add info panel to common task section.
     *
     * @param {Node} info The info panel box.
     * @param {Node} image The info panel icon
     * @param {Node} imageLink
     * @param {Node} close The close button.
     * @param {Node} parent
     * @param {Node} task
     * @param {Node} bottomInfoLink The bottom info panel link.
     * @param {String} spacer ID of the spacer image.
     * @return {boolean} true if successful; otherwise, false.
     */
    addInfoPanel: function(info, image, imageLink, close, parent, task, 
            bottomInfoLink, spacer) {
        // Set HTML elements.
        this.info = info;
        this.image = image;
        this.imageLink = imageLink;
        this.close = close;
        this.parent = parent;
        this.task = task;
        this.bottomInfoLink = bottomInfoLink;
        this.spacer = spacer;
        this._theme = woodstock4_3.theme.common;
        
        var that = this;

        // Handle the keypress event for the "more" link if one is present.
        // Tabbing out of the info panel should close the info panel whereas
        // pressing escape key should close the info panel tooo
        if (this.bottomInfoLink) {
            
            this.bottomInfoLink.setProps({
                
                // Only for IE.                
                onKeyDown:function(event) {
                    if (woodstock4_3._base.browser._isIe5up()) {

                        // For IE, while pressing the shift key along with the tab key
                        // the onkeydown seems to be called twice. To prevent this,
                        // check whether the shift key is the one thats being pressed
                        // before actually calling the event handling function.
                        if (!(window.event.keyCode == 16)) {
                            that.captureBottomInfoKey(window.event);
                        }
                    }
                    return false;
                },
                onKeyPress:function(event) {	    
                    var evt = (event) ? event : ((window.event) ? window.event : null);  
                    if (!woodstock4_3._base.browser._isIe5up()) {
                        that.captureBottomInfoKey(event);
                    }
                    return false;                                 
                }        
            });
        }
        
        // Function that gets invoked when keypress event happens on the bottom
        // portion of the info panel.
        this.captureBottomInfoKey = function(event) {
            if ((event.keyCode == 9 && !event.shiftKey)|| event.keyCode == 27) {
                // need to remove the focus off the link. Otherwise there seems
                // to be problems setting focus on another element in IE.
                that.bottomInfoLink.blur();

                woodstock4_3._base.common._setVisibleElement(that.info, false);
                that.image.src = that.parent.pic3URL;	
                that.imageLink.focus();
            }

            if (event.shiftKey && event.keyCode == 9) {
                that.close.focus();

                // If you dont do this, the info panel closes on IE
                // and the focus is set on the "i" icon.
                woodstock4_3._base.common._setVisibleElement(that.info, true);
            }
            return true;
        };

        // Function that is called when the key press event happens on the
        // close image of the info panel.
        this.captureCloseKey = function(event) {
            // We want to process only key press events which have the tab key pressed.
            if (event.keyCode == 9) {
                // If this is not done IE doesnt set focus on the next available
                // element properly if the info panel closes.
                that.close.blur();     

                // If the "more" link is present, shift focus to that
                // else close the info panel on blur.
                if (that.bottomInfoLink && event.shiftKey == false) {
                    that.bottomInfoLink.focus(); 

                    // If this is not done, the info panel closes
                    // after you tab to the element on IE
                    woodstock4_3._base.common._setVisibleElement(that.info, true);
                } else {
                    that.image.src = that.parent.pic3URL;	            
                    woodstock4_3._base.common._setVisibleElement(that.info, false);    
                    that.imageLink.focus();
                }                                      
            }
            return true;
        };
                
        // Events which handle the closing of the div.        
        this.close.setProps({
            onClick:function(event) {
                woodstock4_3._base.common._setVisibleElement(that.info, false);
                that.image.src = that.parent.pic3URL;	
                if (woodstock4_3._base.browser._isIe5up()) {
                    window. event.cancelBubble = true;
                } else {
                    event.stopPropagation();
                }
                that.task.focus();
                return false;
            },        
            
            // Need to do this only on IE. "Tab" key doesnt get registered
            // for keypress on IE.                        
            onKeyDown:function(event) {
                if (woodstock4_3._base.browser._isIe5up()) {

                    // this seems to be called once for the shift key and
                    // once for the tab key. Prevent calling the capture
                    // function when the shift key is pressed
                    if (!(window.event.keyCode == 16)) {
                        that.captureCloseKey(window.event);
                    }

                    // If escape key is pressed, the info panel must close.
                    if (window.event.keyCode == 27 || window.event.keyCode == 13) {
                        woodstock4_3._base.common._setVisibleElement(that.info, false);
                        that.image.src = that.parent.pic3URL;
                        that.imageLink.focus();
                    }                
                    return false;
                }
                return true;
            },
            
            // Handle the keypress event on the close imageHyperlink.
            // If tab key is pressed, the focus must either pass to
            // the "more" link if it is present or the infoPanel should close. 
            onKeyPress:function(event) {              
                var evt = (event) ? event : ((window.event) ? window.event : null);         
                if (!woodstock4_3._base.browser._isIe5up()) {
                    that.captureCloseKey(evt);
                }
                // If escape key is pressed, the info panel must close.
                if (evt.keyCode == 27 || evt.keyCode == 13) {
                    woodstock4_3._base.common._setVisibleElement(that.info, false);
                    that.image.src = that.parent.pic3URL;
                    that.imageLink.focus();
                }
                return false;
            }
        });
        
        this.info.onclick = function(event) {
            woodstock4_3._base.common._setVisibleElement(that.info, true);
            if (woodstock4_3._base.browser._isIe5up()) {
                window. event.cancelBubble = true;
            } else {
                event.stopPropagation();
            }
            return true;
        };
        
        // Events which handle the image changes for the "i" image.
        this.imageLink.setProps({
            onMouseOver:function() {
                if (!woodstock4_3._base.common._isVisibleElement(that.info)) {
                    that.image.src = that.parent.pic2URL;
                } else {
                    that.image.src = that.parent.pic1URL;
                }
                return true;
            },        
            onFocus:function() {
                if (!woodstock4_3._base.common._isVisibleElement(that.info)) {
                    that.image.src = that.parent.pic2URL;
                } else {
                    that.image.src = that.parent.pic1URL;
                }
                return true;
            },        
            onBlur:function() {
                  if (!woodstock4_3._base.common._isVisibleElement(that.info)) {
                    that.image.src = that.parent.pic3URL;
                } else {
                    that.image.src = that.parent.pic1URL;
                }
                return true;
            },
            onMouseOut:function() {
                if (!woodstock4_3._base.common._isVisibleElement(that.info)) {
                    that.image.src = that.parent.pic3URL;
                } else {
                    that.image.src = that.parent.pic1URL;
                }
                return true;
            },
            onKeyPress:function(event) {
                var evt = (event) ? event : ((window.event) ? window.event : null);            
                if (evt.keyCode == 13) {
                    that.showInfoPanel();
                    return false;                
                }
                if (woodstock4_3._base.browser._isIe5up()) {
                    window.event.cancelBubble = true;
                } else {
                    event.stopPropagation();
                }
                return true;
            }            
        });
        
       // Toggle functionality incorporated
        this.image.setProps({onClick:function(event){
            that.showInfoPanel();
            if (woodstock4_3._base.browser._isIe5up()) {
                window.event.cancelBubble = true;
            } else {
                event.stopPropagation();
            }
            return true;
        }});
        
        this.showInfoPanel = function() {
            var cts = this.parent;
            for (var i = 0; i < cts.count; i++) {
                task = cts.taskElement[i];
                if (task.infoPanel != null
                        && task.infoPanel.image.id != this.image.id) {
                    woodstock4_3._base.common._setVisibleElement(task.infoPanel.info, false);
                    task.infoPanel.image.src = cts.pic3URL;
                }
            }
 
            if (!woodstock4_3._base.common._isVisibleElement(this.info)) {
                woodstock4_3._base.common._setVisibleElement(this.info, true);
                this.getElementPosition2(this.image.id);
                this.getElementPosition(this.task.id);        
                this.info.style.top = (this.ttop + parseInt(this._theme.getMessage("commonTasks.infoPanelOffsetTop"))) +'px';
                this.info.style.left =  (this.tleft - 1) + 'px';
                this.info.style.width = (this.ileft - this.tleft) + 29 + 'px';
                this.close.focus();
                this.image.src = cts.pic1URL;
            } else {
                this.image.src = cts.pic3URL;
                woodstock4_3._base.common._setVisibleElement(this.info, false);
            }
            return true;
        };

        // Javascript for setting the common task page's look and feel.

        // The prized coordinate locating function - Thank you Danny Goodman...
        this.getElementPosition = function(elemID) {
            var offsetTrail = document.getElementById(elemID);
            var offsetLeft = 0;
            var offsetTop = 0;

            while (offsetTrail) {
                offsetLeft += offsetTrail.offsetLeft;
                offsetTop += offsetTrail.offsetTop;
                offsetTrail = offsetTrail.offsetParent;
            }
            if (navigator.userAgent.indexOf("Mac") != -1 
                    && typeof document.body.leftMargin != "undefined") {
                offsetLeft += document.body.leftMargin;
                offsetTop += document.body.topMargin;
            }
            this.tleft = offsetLeft;
            this.ttop = offsetTop;
            return true;
        };

        this.getElementPosition2 = function(elemID) {
            var offsetTrail = document.getElementById(elemID);
            var offsetLeft = 0;
            var offsetTop = 0;

            while (offsetTrail) {
                offsetLeft += offsetTrail.offsetLeft;
                offsetTop += offsetTrail.offsetTop;
                offsetTrail = offsetTrail.offsetParent;
            }
            if (navigator.userAgent.indexOf("Mac") != -1
                    && typeof document.body.leftMargin != "undefined") {
                offsetLeft += document.body.leftMargin;
                offsetTop += document.body.topMargin;
            }
            this.ileft = offsetLeft;
            return true;
        };
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.dropDown");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for dropDown components.
 * @static
 * 
 * @deprecated See woodstock4_3.widget.dropDown
 * @private
 */
woodstock4_3._html.dropDown = {
    /**
     * This function is invoked by the choice onselect action to set the
     * selected, and disabled styles.
     *
     * Page authors should invoke this function if they set the 
     * selection using JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated
     */
    changed: function(elementId) {         
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget._changed();
        }
        return false;
    },

    /**
     * Use this function to access the HTML select element that makes up
     * the dropDown.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the dropDown).
     * @return {Node} a reference to the select element. 
     * @deprecated Use document.getElementById(elementId).setSelectElement()
     */
    getSelectElement: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the label of the first
     * selected option on the dropDown. If no option is selected, this
     * function returns null.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The label of the selected option, or null if none is
     * selected. 
     * @deprecated Use document.getElementById(elementId).getSelectedLabel();
     */
    getSelectedLabel: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectedLabel();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the value of the first
     * selected option on the dropDown. If no option is selected, this
     * function returns null. 
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The value of the selected option, or null if none is
     * selected. 
     * @deprecated Use document.getElementById(elementId).getSelectedValue();
     */
    getSelectedValue: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectedValue();
        }
        return null;
    },

    /**
     * Set the disabled state for given dropdown element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * Page authors should invoke this function if they dynamically
     * enable or disable a dropdown using JavaScript.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({ disabled: disabled});
        }
        return null;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.dropDown = woodstock4_3._html.dropDown;


woodstock4_3._dojo.provide("woodstock4_3._html.editableList");

woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3._html.listbox"); // Required by renderer.

/** 
 * @class This class contains functions for editableList components.
 * @static
 * @private
 */
woodstock4_3._html.editableList = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize editableList.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Not a facet does not have "extra" editable list id.

        // child elements
        // Get the field by calling the field getInputMethod
        // because only it knows about the underlying structure
        // of the rendered field component
        //
        domNode.list = document.getElementById(props.id + "_list");

        // Bug 6338492 -
        //     ALL: If a component supports facets or children is must be a
        //      NamingContainer
        // Since EditableList has become a NamingContainer the id's for
        // the facet children are prefixed with the EditableList id
        // in addition to their own id, which also has the 
        // EditableList id, as has been the convention for facets. This introduces
        // a redundancy in the facet id so the add button now looks like
        //
        // "formid:editablelistid:editablelistid:editablelistid_addButton"
        //
        // It used to be "formid:editablelistid_addButton"
        // It would be better to encapsulate that knowledge in the
        // EditableList renderer as does FileChooser which has the
        // same problem but because the select elements are not
        // facets in EditableList they really do only have id's of the
        // form "formid:addremoveid_list". Note that 
        // in these examples the "id" parameter is "formid:editablelistid"
        //
        // Therefore for now, locate the additional prefix here as the
        // "facet" id. Assume that id never ends in ":" and if there is
        // no colon, id is the same as the component id.
        //
        var componentid = props.id;
        var colon_index = componentid.lastIndexOf(':');
        if (colon_index != -1) {
            componentid = props.id.substring(colon_index + 1);
        }
        var facetid = props.id + ":" + componentid;

        // Get the field by calling the field getInputMethod
        // because only it knows about the underlying structure
        // of the rendered field component
        //
        var widget = woodstock4_3.widget.common.getWidget(facetid + "_field");
        if (widget) {
            domNode.field = widget.getInputElement();
        }

        domNode.addButton = document.getElementById(facetid + "_addButton"); 
        domNode.removeButton = document.getElementById(facetid + "_removeButton"); 

        // HTML elements may not have been created, yet.
        if (domNode.list == null 
                || domNode.field == null 
                || domNode.addButton == null 
                || domNode.removeButton == null) {
            return setTimeout(function() {
                woodstock4_3._html.editableList._init(props);
            }, 10);
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

        // Set private functions.
        domNode.add = woodstock4_3._html.editableList.add;
        domNode.enableAdd = woodstock4_3._html.editableList.enableAdd;
        domNode.enableRemove = woodstock4_3._html.editableList.enableRemove;
        domNode.setAddDisabled = woodstock4_3._html.editableList.setAddDisabled;
        domNode.setRemoveDisabled = woodstock4_3._html.editableList.setRemoveDisabled; 
        domNode.updateButtons = woodstock4_3._html.editableList.updateButtons;
        domNode.setDisabled = woodstock4_3._html.editableList.setDisabled;

        // Initialize buttons.
        domNode.updateButtons();
        return true;
    },

    /**
     * Add HTML element.
     *
     * @param {String} elementId The HTML element id.
     * @return {boolean} true if successful; otherwise, false.
     */
    add: function(elementId) {
        this.enableAdd(); 
        this.addButton.click();
        return true;
    },

    /**
     * Enable add button.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    enableAdd: function() {
        var disabled = (this.field.value == ""); 
        return this.setAddDisabled(disabled);
    },

    /**
     * Set add button disabled.
     *
     * @param {boolean} disabled If true, disable element.
     * @return {boolean} true if successful; otherwise, false.
     */
    setAddDisabled: function(disabled) {
        if (this.addButton.setDisabled != null) {
            this.addButton.setDisabled(disabled); 
        } else {
            this.addButton.disabled = disabled; 
        }
        return true;
    },

    /**
     * Enable remove button.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    enableRemove: function() {
        var disabled = (this.list.selectedIndex == -1); 
        return this.setRemoveDisabled(disabled); 
    },

    /**
     * Set remove button disabled.
     *
     * @param {boolean} disabled If true, disable element.
     * @return {boolean} true if successful; otherwise, false.
     */
    setRemoveDisabled: function(disabled) {
        if (this.removeButton.setDisabled != null) {
            this.removeButton.setDisabled(disabled); 
        } else {
            this.removeButton.disabled = disabled; 
        }
        return true;
    },

    /**
     * Update add and remove buttons.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    updateButtons: function() {
        this.enableAdd(); 
        this.enableRemove(); 
        return true;
    },

    /**
     * Set buttons disabled.
     *
     * @param {boolean} disabled If true, disable element.
     * @return {boolean} true if successful; otherwise, false.
     */
    setDisabled: function(disabled) {
        if (this.addButton.setDisabled != null) {
            this.addButton.setDisabled(disabled); 
        } else {
            this.addButton.disabled = disabled; 
        }
        if (this.removeButton.setDisabled != null) {
            this.removeButton.setDisabled(disabled); 
        } else {
            this.removeButton.disabled = disabled; 
        }
        this.field.disabled = disabled; 
        this.list.disabled = disabled; 
        return true;
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.field");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for field components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.field
 * @private
 */
woodstock4_3._html.field = {
    /**
     * Use this function to get the HTML input or textarea element
     * associated with a TextField, PasswordField, HiddenField or TextArea
     * component.
     *
     * @param {String} elementId The element ID of the field 
     * @return {Node} the input or text area element associated with the field component
     * @deprecated Use document.getElementById(elementId).getInputElement()
     */
    getInputElement: function(elementId) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getInputElement();
        }
        return null;
    },

    /**
     * Use this function to get the value of the HTML element 
     * corresponding to the Field component.
     *
     * @param {String} elementId The element ID of the Field component
     * @return {String} the value of the HTML element corresponding to the Field component 
     * @deprecated Use document.getElementById(id).getProps().value;
     */
    getValue: function(elementId) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getProps().value;
        }
        return null;
    },

    /**
     * Use this function to set the value of the HTML element 
     * corresponding to the Field component
     *
     * @param {String} elementId The element ID of the Field component
     * @param {String} newValue The new value to enter into the input element Field component 
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({value: "text"});
     */
    setValue: function(elementId, newValue) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({value: newValue});
        }
        return null;
    },

    /** 
     * Use this function to get the style attribute for the field. 
     * The style retrieved will be the style on the span tag that 
     * encloses the (optional) label element and the input element.
     *
     * @param {String} elementId The element ID of the Field component
     * @return {String} The style property of the field.
     * @deprecated Use document.getElementById(id).getProps().style;
     */
    getStyle: function(elementId) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getProps().style;
        }
        return null;
    },

    /**
     * Use this function to set the style attribute for the field. 
     * The style will be set on the <span> tag that surrounds the field.
     *
     * @param {String} elementId The element ID of the Field component
     * @param {String} newStyle The new style to apply
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({style: newStyle});
     */
    setStyle: function(elementId, newStyle) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({style: newStyle});
        }
        return null;
    },

    /**
     * Use this function to disable or enable a field. As a side effect
     * changes the style used to render the field. 
     *
     * @param {String} elementId The element ID of the field 
     * @param {boolean} newDisabled true to disable the field, false to enable the field
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, newDisabled) {  
        if (newDisabled == null) {
            return null;
        }
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.setProps({disabled: newDisabled});
        }
        return null;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.field = woodstock4_3._html.field;


woodstock4_3._dojo.provide("woodstock4_3._html.fileChooser");

woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3._base.proto");

/** 
 * @class This class contains functions for fileChooser components.
 * <p>
 * The filechooser has several intermediate actions in addition to the selection
 * of one or more files of folders. For example, there are actions that are 
 * initiated by buttons and actions initiated by mouse clicks and key presses.
 * Some events generated from key presses and mouse clicks behave like 
 * accelerators for the button actions.
 * </p><p>
 * Some actions are client side only actions, such as placing a
 * selected file or folder into the selected file or folder field.
 * </p><p>
 * The server side events and how they are generated.
 * </p><p><pre>
 * 1. moveup - the moveup button is clicked
 * 2. openfolder - select a folder from the list and the openfolder
 *    button is clicked or press the return key with the selection in
 *    focus or double click the selection
 * 3. sort - a sort selection is made from the sort dropdown menu
 * 4. refresh the list - set focus in the look in field and press the return key
 * 5. change the directory listing - type a new value in the look in field
 *    and press the return key
 *    The directory listing is also changed when a folder is typed into
 *    the selected file field, and submitted.
 * 6. filter the list - the focus is placed in the filter field and the
 *    return key is pressed
 * 7. change filter value - a new value is entered into the filter field 
 *    and the return key is pressed
 * 8. commit the changes - the button assigned as the chooseButton is
 *    is clicked, or programmtically clicked when the return key is 
 *    pressed in the select file field or a file selection is double clicked.
 * </pre></p><p>
 * Mouse clicks and return key activations.
 * </p><p><pre>
 * - The mouse clicks and key presses that correspond to actions are
 *   exectuted by "clicking" the corresponding button action programmatically.
 *   For example, action #2 when activated by double clicking or the
 *   return key, programmatically clicks the open folder button.
 *
 * - Action #4 and #6 explcitly submit the form
 *
 * - Action #8 programmatically clicks the assigned chooserButton.
 * </pre></p><p>
 * Selections are made and submitted in the following ways
 * </p><p><pre>
 *   File chooser or folder chooser
 *
 *   - One or more absolute or relative paths are typed or placed
 *   into the select file or folder field the return key is pressed.
 *
 *   - An external submit button submits the form and there are selections
 *   in the selected file or folder field.
 *
 *   File chooser 
 *
 *   - A file selection is double clicked.
 * </pre></p><p>
 * Client side selections
 * </p><p><pre>
 * - A file or folder is single clicked and it is entered into the
 *   selected file or folders field, depending on whether the chooser
 *   is a file or folder chooser.
 *
 * - When a directory is selected and the open folder button is clicked
 *   the entry is set as the look in field value and the form is submitted.
 * 
 * - When the move up button is clicked the parent directory is placed
 *   into the look in field and the form is submitted.
 * </pre></p>
 * @static
 * @private
 */
woodstock4_3._html.fileChooser = {
    // FIXME: Note that the dependence on literal client id's is not sufficient
    // if these components are developer defined facets. The actual
    // literal id's cannot be guaranteed.

    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @config {String} chooserType 
     * @config {String} parentFolder 
     * @config {String} separatorChar 
     * @config {String} escapeChar 
     * @config {String} delimiter 
     * @config {String} currentDir
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize fileChooser.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // This is not a user defined facet. It is created dynamically
        // if the enter key is pressed in the selected file field.
        // This is the expected form of the request paramter name
        // in the renderers decode method.
        //
        // Consider passing the name as a parameter, or some other
        // more well defined manner.
        var idPrefix = props.id;
        var index = props.id.lastIndexOf(':');
        if (index != -1) {
             idPrefix += props.id.substring(index);
        }

        // Get HTML elements.
        var widget = woodstock4_3.widget.common.getWidget(idPrefix + "_lookinField");
        if (widget) {
            domNode.lookinfield = widget.getInputElement();
        }
        widget = woodstock4_3.widget.common.getWidget(idPrefix + "_filterField");
        if (widget) {
            domNode.filterfield = widget.getInputElement();
        }
        widget = woodstock4_3.widget.common.getWidget(idPrefix + "_selectedField");
        if (widget) {
            domNode.selectedfield = widget.getInputElement();
        }
        widget = woodstock4_3.widget.common.getWidget(idPrefix + "_listEntries");
        if (widget) {
            domNode.listentries = widget.getSelectElement();
        }
        widget = woodstock4_3.widget.common.getWidget(idPrefix + "_sortMenu");
        if (widget) {
            domNode.sortmenu = widget.getSelectElement();
        }
        domNode.upButton = document.getElementById(idPrefix + "_upButton");
        domNode.openFolderButton = document.getElementById(idPrefix + "_openButton");

        // HTML elements may not have been created, yet.
        if (domNode.lookinfield == null 
                || domNode.filterfield == null 
                || domNode.selectedfield == null 
                || domNode.upButton == null
                || domNode.openFolderButton == null 
                || domNode.listentries == null
                || domNode.sortmenu == null) {
            return setTimeout(function() {
                woodstock4_3._html.fileChooser._init(props);
            }, 10);
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

        // boolean identifying the chooser mode.
        domNode.folderChooser = (props.chooserType == "folderChooser");
        domNode.fileAndFolderChooser = (props.chooserType == "fileAndFolderChooser");
        domNode.chooseButton = null;
        domNode.selectionsId = idPrefix + "_selections";
        domNode.listOptions = domNode.listentries.options;

        // FIXME: This encoding needs to be generalized if this code is to
        // become more generic chooser-like.
        // In fact encoding entries this way in not ideal.
        // A more explicit typing needs to be developed if it is 
        // necessary, possible a data structure that maps type to entry.
        if (domNode.folderChooser) {
            domNode.chooser = 'folder';
        } else {
            domNode.chooser = 'file';
        }

        // Set public functions.
        domNode.enterKeyPressed = woodstock4_3._html.fileChooser.enterKeyPressed;
        domNode.handleDblClick = woodstock4_3._html.fileChooser.handleDblClick;
        domNode.handleOnChange = woodstock4_3._html.fileChooser.handleOnChange;
        domNode.openFolderClicked = woodstock4_3._html.fileChooser.openFolderClicked;
        domNode.moveUpButtonClicked = woodstock4_3._html.fileChooser.moveUpButtonClicked;
        domNode.setChooseButton = woodstock4_3._html.fileChooser.setChooseButton;
        domNode.getCurrentDirectory = woodstock4_3._html.fileChooser.getCurrentDirectory;
        domNode.getOptionElements = woodstock4_3._html.fileChooser.getOptionElements;
        domNode.getSelectedOptions = woodstock4_3._html.fileChooser.getSelectedOptions;
        domNode.isFolderChooser = woodstock4_3._html.fileChooser.isFolderChooser;
        domNode.isFolderSelected = woodstock4_3._html.fileChooser.isFolderSelected;
        domNode.getSelectionValue = woodstock4_3._html.fileChooser.getSelectionValue;
        domNode.getSelectionValueByIndex = woodstock4_3._html.fileChooser.getSelectionValueByIndex;
        domNode.getSelectionType = woodstock4_3._html.fileChooser.getSelectionType;
        domNode.getSelectionTypeByIndex = woodstock4_3._html.fileChooser.getSelectionTypeByIndex;
        domNode.getValueType = woodstock4_3._html.fileChooser.getValueType;
        domNode.itemSelected = woodstock4_3._html.fileChooser.itemSelected;
        domNode.getSelectedFolders = woodstock4_3._html.fileChooser.getSelectedFolders;
        domNode.getSelectedFiles = woodstock4_3._html.fileChooser.getSelectedFiles;
        domNode.getSelectedValuesByType = woodstock4_3._html.fileChooser.getSelectedValuesByType;
        domNode.setSelectedFieldValue = woodstock4_3._html.fileChooser.setSelectedFieldValue;
        domNode.clearSelections = woodstock4_3._html.fileChooser.clearSelections;
        domNode.deselectFolders = woodstock4_3._html.fileChooser.deselectFolders;
        domNode.deselectSelectionsByType = woodstock4_3._html.fileChooser.deselectSelectionsByType;
        domNode.setSelected = woodstock4_3._html.fileChooser.setSelected;
        domNode.clearSelectedField = woodstock4_3._html.fileChooser.clearSelectedField;
        domNode.armChooseButton = woodstock4_3._html.fileChooser.armChooseButton;
        domNode.getFileNameOnly = woodstock4_3._html.fileChooser.getFileNameOnly;
        domNode.setChooseButtonDisabled = woodstock4_3._html.fileChooser.setChooseButtonDisabled;

        // For supporting valid entries in look in field and filter field.
        //
        // It is imperative that the look in field and filter field
        // are never submitted if the value does not imply a valid action.
        // Not currently used.
        domNode.onFocus = woodstock4_3._html.fileChooser.onFocus;
        domNode.onBlur = woodstock4_3._html.fileChooser.onBlur;

        // Save the initial lookin and filter values.
        domNode.lastLookInValue = domNode.lookinfield.value;
        domNode.lastFilterValue = domNode.filterfield.value;
        domNode.lookinCommitted = false;
        domNode.filterCommitted = false;
        domNode.openFolderButton.setDisabled(true);

	if (props.currentFolder != null) {
	  if (props.parentFolder == props.currentFolder) {
	    domNode.upButton.setDisabled(true);
   	  } else {
	    domNode.upButton.setDisabled(false);
	  }
	}
        return true;
    },

    /**
     * Handler for enter key presses.
     * <p><pre>
     * - Enter key in LookInField
     * - Enter key in FilterField
     * - Enter key in SelectedFileField
     * - Enter key in Listbox with folder selection.
     * Submit the chooser from the various mouse clicks
     * key presses.
     * </pre></p><p>
     * Handles doubleclick on a file selection in the list box.
     * This is equivalent to an enter key press in the selected file field.
     * </p>
     *
     * @param {Node} element
     * @return {boolean} false to cancel JavaScript event.
     */
    enterKeyPressed: function(element) {
	// Return pressed in the list
	if (element.id == this.listentries.id) {
	    // If the selected item is a folder call the click method
	    // on the openFolderButton
	    if (this.isFolderSelected()) {
		// Done in openFolderClicked
		//
		//this.lookinfield.value = this.getSelectionValue();
		this.openFolderButton.click();
	    }
	    return false;
	}

	// The FileChooser's value must only change when selections
	// have been made, not from just intermediate operations.
	//
	// Enter key pressed in the selectedFileField
	// or dbl click in the list.
	if (this.selectedfield && element.id == this.selectedfield.id) {
	    var escapedSelections = this.selectedfield.value;
	    var selections = woodstock4_3._base.common._unescapeStrings(escapedSelections,
		    this.delimiter, this.escapeChar);

	    // If a choose button has been defined call its click method
	    // otherwise do nothing. This behavior allows the filechooser
	    // to behave like any other component on a page.
	    if (this.chooseButton) {
		// Make sure its enabled.
		this.chooseButton.setDisabled(false);
		this.chooseButton.click();
	    }
	    return false;
	}

	// Enter key pressed in the filter field
	// Call the open folder button's click method.
	// Since there is no JSF action for mouse clicks or key presses
	// overload the open folder action to ensure that the 
	// sort value is updated.
	if (element.id == this.filterfield.id) {
	    this.filterCommitted = true;
	    // Don't let "" get submitted.
	    var fv = this.filterfield.value;
	    if (fv == null || fv == "") {
		this.filterfield.value = this.lastFilterValue;
		return false;
	    }
	    this.lastFilterValue = fv;
	    this.clearSelections();
	    element.form.submit();
	    return false;
	}

	// Enter key pressed in the LookIn field.
	// Call the open folder button's click method.
	// Since there is no JSF action for mouse clicks or key presses
	// overload the open folder action to ensure that the 
	// look in value is updated. This is needed anyway to display
	// the new folder's content.
	if (element.id == this.lookinfield.id) {
	    this.lookinCommitted = true;
	    // Don't let "" get submitted.
	    var lv = this.lookinfield.value;
	    if (lv == null || lv == "") {
		this.lookinfield.value = this.lastLookInValue;
		return false;
	    }
	    this.lastLookInValue = lv;
	    this.clearSelections();
	    element.form.submit();
	    return false;
	}
	return false;
    },

    /**
     * In file chooser mode
     * <p><pre>
     *    - a file selection, call enterKeyPressed with selected file field
     *    - a folder selection, call open folder click handler
     * In folder chooser mode
     *    - a folder selection, call open folder click handler
     * </pre></p>
     * @return {boolean} true if successful; otherwise, false.
     */
    handleDblClick: function() {
	// Nothing selected. Not sure if this can happen since
	// doubleclick implies selection.
	if (!this.itemSelected()) {
	    return false; 
	}

	var fldrSelected = this.isFolderSelected();

	// If the selected item is a folder call the click method
	// on the openFolderButton
	if (fldrSelected) {
	    // Set the look in field, since the selected folder will be
	    // the new look in field value. Done in openFolderClicked.
	    // This only works now because values are full paths.
	    //
	    this.openFolderButton.click();
	    return true;
	}

	// doubleclick is not valid for file selections in a
	// folder chooser.
	// If a file chooser, this is equivalent to a return key
	// in the selected file field.
	if (this.isFolderChooser()) {
	    if (!fldrSelected) {
		return false;
	    }
	} else {
	    // file chooser
	    // double click a file in file chooser mode
	    if (!fldrSelected) {
		if (this.selectedfield) {
		    return this.enterKeyPressed(this.selectedfield);
		}

		// If a choose button has been defined call its click method
		// otherwise do nothing. This behavior allows the filechooser
		// to behave like any other component on a page.
		if (this.chooseButton) {
		    // Make sure its enabled.
		    //
		    this.chooseButton.setDisabled(false);
		    return this.chooseButton.click();
		}
	    }
	}
	return true;
    },

    /**
     * Set choose button disabled.
     *
     * @param {boolean} disabled
     * @return {boolean} true if successful; otherwise, false.
     */
    setChooseButtonDisabled: function(disabled) {
	if (this.chooseButton) {
	    this.chooseButton.setDisabled(disabled);
	}
        return true;
    },

    // Replaces entries in selectedFileField
    // Get the selected entries from the list box and place
    // them in the selected file field, as comma separated entries.
    //
    // Note that the listbox values are full paths and encoded with
    // a "folder" or "file" designation. They probably should
    // be relative paths. The open folder action now depends on the
    // fact that the value is a full path.
    // This will have an effect on the open folder
    // action when the selected value is placed into the look in field.
    // If relative paths are used for the values then
    // the relative path would need to be appended to the look in
    // field value.
    //
    // However it may be the case that the full paths are edited to
    // take off the last element in the full path and keep the 
    // full path list box entries. Full paths are generally more
    // convenient.
    //
    // Note that this handler should call any required handlers
    // needed by the list box, vs. placing the required listbox
    // handlers in a javascript statement as the value of the
    // onChange attribute.
    //
    // Note also that the SWAED guidelines say to place relavtive
    // paths into the selected file field, this probably means
    // just using the display name vs. the value. However note the
    // dependencies on the full paths as described above.

    /**
     * Handler placed on the list box onchange enent.
     * <p>
     * Place all currently selected entries in to the
     * selected file field. If the chooser mode is file, only
     * files are placed into the selected file field.
     * </p><p>
     * If the chooser mode is folder, only folders are placed in the
     * selected folder field.
     * </p><p>
     * If multiple selections are allowed the entries are separated
     * by the specified delimiter. Enteries are escaped appropriately
     * with the specified escape character.
     * </p>
     * @return {boolean} false to cancel JavaScript event.
     */
    handleOnChange: function() {
        var widget = woodstock4_3.widget.common.getWidget(this.listentries.id);
        if (widget) {
            widget._changed();
        }

	// If nothing is selected disable buttons.
	if (!this.itemSelected()) {
	    this.openFolderButton.setDisabled(true); 
	    if (this.selectedfield &&
		(this.selectedfield.value == null ||
		    this.selectedfield.value == '')) {
		this.setChooseButtonDisabled(true);
	    }
	    return false;
	}

	// This may not be sufficient.
	// The issue is, should a file be selectable in a folder
	// chooser, period. Are they disabled or read only ?
	// And ditto for a multiple selection in a file chooser.
	// Should a folder be selectable as a multiple selection
	// in a file chooser ?
	//
	// This could be made more efficient
	// by return both arrays at once and making only
	// one pass

	var folders = this.getSelectedFolders();
	var files = this.getSelectedFiles();
	var selections = null;

	// If a file chooser, deselect folders when mixed
	// with file selections and disable the openFolder button);
	if (this.fileAndFolderChooser) {
	    selections = new Array(files.length + folders.length);
	    var i = 0;
	    for (; i< files.length; i++) {
	        selections[i] = files[i];
	    } 
	    for (j=0; j< folders.length; j++) {
	        selections[i+j] = folders[j];
	    } 
	    if (files.length > 0) {
		this.openFolderButton.setDisabled(true);
	    } else if (folders.length > 1) {
		this.openFolderButton.setDisabled(true);
	    } else if ((files.length == 0) || (folders.length == 1)) {
		this.openFolderButton.setDisabled(false);
	    }
	} else if (!this.isFolderChooser()) {
	    if (files.length > 0) {
		this.openFolderButton.setDisabled(true);
		this.deselectFolders();
	    } else if (folders.length > 1) {
		this.openFolderButton.setDisabled(false);
		var index = this.listentries.selectedIndex;
		this.deselectFolders();
		this.setSelected(index, true);
		this.clearSelectedField();
                var widget = woodstock4_3.widget.common.getWidget(this.listentries.id);
                if (widget) {
                    widget._changed();
                }
	    } else if (folders.length == 1) {
		// Only allow one folder to be selected
		this.openFolderButton.setDisabled(false);
		this.clearSelectedField();
	    } else {
		this.openFolderButton.setDisabled(true);
		this.clearSelectedField();
	    }
	    selections = files;
	} else {
	    // If a folder chooser allow more than one folder
	    // to be selected
	    selections = folders;
	    if (selections.length == 1) {
		this.openFolderButton.setDisabled(false);
	    } else {
		this.openFolderButton.setDisabled(true);
	    }
	}

	// Make sure the hidden select option array is up
	// to date in case there isn't a selectedFileField.
	if (!this.setSelectedFieldValue(selections)) {
	    woodstock4_3._base.common._createSubmittableArray(
                this.selectionsId, this.listentries.form, null, selections);
	}

	var flag = ((selections!= null) && (selections.length > 0));
	this.armChooseButton(flag);
	return false;
    },

    /**
     * Clear selected field.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    clearSelectedField: function() {
	if (this.selectedfield) {
	    this.selectedfield.value = '';
	}
        return true;
    },

    /**
     * This function is the event handler for the onclick event
     * of the openFolder button.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    openFolderClicked: function() {
	if (!this.isFolderSelected()) {
	    return false;
	}
	this.clearSelectedField();

	// Only works because the value is a full path.
	this.lookinfield.value = this.getSelectionValue();
	return true;
    },

    /**
     * Test if folder is selected.
     *
     * @return {boolean} true if folder is selected.
     */
    isFolderSelected: function() {
	return this.getSelectionType() == 'folder';
    },

    /**
     * This function is the event handler for the moveUp button.
     * Set the look in field to contain the parent or move up directory.
     * This is imperative. Be careful of the corner case when the 
     * look in field is already the root directory.
     * @return {boolean} true if successful; otherwise, false.
     */
    moveUpButtonClicked: function() {
	this.clearSelections();
	this.lookinfield.value = this.parentFolder;
        return true;
    },

    /**
     * The values of the list options are encoded as
     * <p>
     * <type>=<value>
     * </p><p>
     * Where type is one of "file" or "folder"
     * </p>
     * @return {String} The selection value.
     */
    getSelectionValue: function() {
	var index = this.listentries.selectedIndex;
	return this.getSelectionValueByIndex(index);
    },

    /**
     * Get selection value by index.
     *
     * @return {String}The selection value.
     */
    getSelectionValueByIndex: function(index) {
	var selection = this.listOptions[index].value;
	var i = selection.indexOf('=', 0);
	if (i < 0) {
	    return null;
	}
	if (i != 0) {
	    i = i + 1;
	}
	return selection.substring(i, selection.length); 
    },

    /**
     * Get selection type.
     *
     * @return {String} The selection type.
     */
    getSelectionType: function() {
	var index = this.listentries.selectedIndex;
	return this.getSelectionTypeByIndex(index);
    },

    /**
     * Get selection type by index.
     *
     * @return {String} The selection type.
     */
    getSelectionTypeByIndex: function(index) {
	var selection = this.listOptions[index].value;
	return this.getValueType(selection);
    },

    /**
     * Get value type.
     *
     * @return {String} The value type.
     */
    getValueType: function(value) {
	var i = value.indexOf('=', 0);
	if (i <= 0) {
	    return null;
	}
	var type = value.substring(0, i); 
	return type;
    },

    /**
     * Test if folder chooser.
     *
     * @return {boolean} true if folder chooser.
     */
    isFolderChooser: function() {
	return this.folderChooser;
    },

    /**
     * Get selected item.
     *
     * @return {String} The selected item.
     */
    itemSelected: function() {
	return (this.listentries.selectedIndex != -1);
    },

    /**
     * Get selected folders.
     *
     * @return {Array} An array of selected folders.
     */
    getSelectedFolders: function() {
	return this.getSelectedValuesByType('folder');
    },

    /**
     * Get selected files.
     *
     * @return {Array} An array of selected files.
     */
    getSelectedFiles: function() {
	return this.getSelectedValuesByType('file');
    },

    /**
     * Return all selected options by type, file or folder.
     *
     * @param {String} type
     * @return {Array} An array of selected values.
     */
    getSelectedValuesByType: function(type) {
	var selections = new Array();
	var i = 0;
	for (var j = 0; j < this.listOptions.length; j++) {
	    if (this.listOptions[j].selected){
		if (this.getSelectionTypeByIndex(j) == type) {
		    selections[i++] = this.getSelectionValueByIndex(j);
		} 
	    } 
	} 
	return selections;
    },

    /**
     * Format the selected file field as a comma separated list.
     *
     * @param {Array} selections
     * @return {boolean} true if successful; otherwise, false.
     */
    setSelectedFieldValue: function(selections) {
	var value;
	if (this.selectedfield == null) {
	    return false;
	}

	if (selections == null || selections.length == 0) {
	    return false;
	} else {
	    value = woodstock4_3._base.common._escapeString(
                this.getFileNameOnly(selections[0]), this.delimiter,
                this.escapeChar);
	}

	for (var j = 1; j < selections.length; j++) {
	    value = value + ',' + 
                woodstock4_3._base.common._escapeString(
                    this.getFileNameOnly(selections[j]), this.delimiter,
                    this.escapeChar);
	} 

	if (value != null && value != '') { 
	    this.selectedfield.value = value;
	} else { 
	    this.selectedfield.value = '';
	} 
	return true;
    },

    /**
     *
     * @param {Node} element
     * @return {boolean} true if successful; otherwise, false.
     */
    onFocus: function(element) {
	if (element.id == this.lookinfield.id) {
	    this.lookinCommitted = false;
	    this.lastLookInValue = this.lookinfield.value;
	} else if (element.id == this.filterfield.id) {
	    this.filterCommitted = false;
	    this.lastFilterValue = this.filterfield.value;
	}
	return true;
    },

    /**
     *
     * @param {Node} element
     * @return {boolean} true if successful; otherwise, false.
     */
    onBlur: function(element) {
	if (element.id == this.lookinfield.id) {
	    if (this.lookinCommitted == false) {
		this.lookinfield.value = this.lastLookInValue;
	    }
	} else if (element.id == this.filterfield.id) {
	    if (this.filterCommitted == false) {
		this.filterfield.value = this.lastFilterValue;
	    }
	}
	return true;
    },

    /**
     * Clear the selections whenever the selectedFileField is cleared.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    clearSelections: function() {
	var i = 0;
	for (var j = 0; j < this.listOptions.length; j++) {
	    if (this.listOptions[j].selected){
		this.listOptions[j].selected = false;
	    } 
	} 
	// call listbox.changed to update the
	// private state
        var widget = woodstock4_3.widget.common.getWidget(this.listentries.id);
        if (widget) {
            widget._changed();
        }
	if (this.selectedfield != null) {
	    this.selectedfield.value = "";
	}
        return true;
    },

    /**
     * Set selected.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    setSelected: function(index, torf) {
	this.listOptions[index].selected = torf;
        var widget = woodstock4_3.widget.common.getWidget(this.listentries.id);
        if (widget) {
            return widget._changed();
        }
        return false;
    },

    /**
     * Deselect folders.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    deselectFolders: function() {
	return this.deselectSelectionsByType('folder');
    },

    /**
     * Deselect by type.
     *
     * @param {String} type
     * @return {boolean} true if successful; otherwise, false.
     */
    deselectSelectionsByType: function(type) {
	for (var j = 0; j < this.listOptions.length; j++) {
	    if (this.listOptions[j].selected &&
		    this.getValueType(this.listOptions[j].value) == type) {
		this.listOptions[j].selected = false;
	    } 
	} 
        var widget = woodstock4_3.widget.common.getWidget(this.listentries.id);
        if (widget) {
            return widget._changed();
        }
        return false;
    },

    /**
     * Enable the choose button.
     *
     * @param {boolean} flag
     * @return {boolean} true if successful; otherwise, false.
     */
    armChooseButton: function(flag) {
	var disabled = true;
	if (this.selectedfield == null) {
	    disabled = flag;
	} else if (this.selectedfield.value != null 
	    && this.selectedfield.value != '') {
	        disabled = false;
	} 
	return this.setChooseButtonDisabled(disabled);
    },

    // Note that this is the only way that the file chooser can control
    // the submit of selected values. If this button is not set then
    // only an external submit button can submit the selections.
    // That means that if there is no chooser button assigned, double clicking
    // a file entry or hitting the return key in the selected file field
    // will NOT submit the values.
    //
    // This "feature" may become configurable.

    /**
     * convenience function to allow developers to disable their
     * chooser button when no entries from the filechooser are
     * selected. This function is not yet complete.
     *
     * @param {String} buttonId
     * @return {boolean} true if successful; otherwise, false.
     */
    setChooseButton: function(buttonId) {
	this.chooseButton = document.getElementById(buttonId);
	// See if there are selections and if so 
	// enable the button. Needs to be after the assignment
	var selections = document.getElementById(this.selectionsId);
	var disabled = true;
	if ((selections != null) && (selections.length > 0)) {
	    disabled = false;
	}
	return this.armChooseButton(disabled);
    },

    /**
     * Convenience function to get the current directory without 
     * going to the server.
     *
     * @return {String} The current directory.
     */
    getCurrentDirectory: function() {
	if (this.lookinfield) {
	    return this.lookinfield.value;
	}
        return null;
    },

    /**
     * Convenience function returning the list of option elements.
     *
     * @return {Array} An array of list options.
     */
    getOptionElements: function() {
	return this.listOptions;
    },

    /**
     * Convenience function to get the list of selected option elements
     * Return an array of selected values or a 0 length array if there
     * are no selections.
     *
     * @return {Array} An array of selected options.
     */
    getSelectedOptions: function() {
	var selections = new Array();
	var i = 0;
	for (var j = 0; j < this.listOptions.length; j++) {
	    if (this.listOptions[j].selected){
		selections[i++] = this.getSelectionValueByIndex(j);
	    } 
	} 
	return selections;
    },

    /**
     * Convenience function to get the file or folder name when 
     * the entire path name is supplied.
     *
     * @param {String} absoluteFileName
     * @return {String} The file name.
     */
    getFileNameOnly: function(absoluteFileName) {
        arrayOfPaths = absoluteFileName.split(this.separatorChar);
	justTheFileName = arrayOfPaths[arrayOfPaths.length -1];
        return justTheFileName;
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.hyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for hyperlink components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.hyperlink
 * @private
 */
woodstock4_3._html.hyperlink = {
    /**
     * This function is used to submit a hyperlink.
     * <p>
     * Note: Params are name value pairs but all one big string array so 
     * params[0] and params[1] form the name and value of the first param.
     * </p>
     *
     * @params {Object} hyperlink The hyperlink element
     * @params {String} formId The form id
     * @params {Object} params Name value pairs
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated See woodstock4_3.widget.hyperlink
     */
    submit: function(hyperlink, formId, params) {
        // Need to test widget for tab and common task components. If a widget 
        // does not exist, fall back to the old code.
	var widget = woodstock4_3.widget.common.getWidget(hyperlink.id);
	if (widget == null) {
            // If a widget does not exist, we shall create one in order to call
            // the submit function directly.
            woodstock4_3._dojo.require("woodstock4_3.widget.hyperlink");
            widget = new woodstock4_3.widget.hyperlink({id: hyperlink.id});
	}
        return widget._submitFormData(formId, params);
    },

    /**
     * Use this function to access the HTML img element that makes up
     * the icon hyperlink.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the outter most tag enclosing the HTML img element).
     * @return {Node} The HTML image element.
     * @deprecated Use document.getElementById(elementId).getProps().enabledImage;
     */
    getImgElement: function(elementId) {
        // Need to test widget for alarmStatus, jobstatus, and notification phrase
        // components. If a widget does not exist, fall back to the old code.
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        var props = (widget) ? widget.getProps() : null;
        if (props && props.enabledImage) {
            var imgWidget = woodstock4_3.widget.common.getWidget(props.enabledImage.id);
            if (imgWidget != null) {
                return imgWidget._domNode;    
            }
        }

        // Image hyperlink is now a naming container and the img element id 
        // includes the ImageHyperlink parent id.
        if (elementId != null) {
            var parentid = elementId;
            var colon_index = elementId.lastIndexOf(":");
            if (colon_index != -1) {
                parentid = elementId.substring(colon_index + 1);
            }
            return document.getElementById(elementId + ":" + parentid + "_image");
        }
        return document.getElementById(elementId + "_image");
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.hyperlink = woodstock4_3._html.hyperlink;


woodstock4_3._dojo.provide("woodstock4_3._html.jumpDropDown");

woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions for jumpDropDown components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.dropDown
 * @private
 */
woodstock4_3._html.jumpDropDown = {
    /**
     * This function is invoked by the jumpdropdown onchange action to set the
     * form action and then submit the form.
     *
     * Page authors should invoke this function if they set the selection using 
     * JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed()
     */
    changed: function(elementId) {
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.changed();
        }
        return false;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.jumpDropDown = woodstock4_3._html.jumpDropDown;


woodstock4_3._dojo.provide("woodstock4_3._html.orderableList");

woodstock4_3._dojo.require("woodstock4_3._base.proto");

/** 
 * @class This class contains functions for orderableList components.
 * @static
 * @private
 */
woodstock4_3._html.orderableList = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @config {String} moveMessage
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize orderableList.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Not a facet does not have "extra" editable list id.

        // The select element from which selections are made 
        domNode.list = document.getElementById(props.id + "_list");

        // Bug 6338492 -
        //     ALL: If a component supports facets or children is must be a
        //      NamingContainer
        // Since OrderableList has become a NamingContainer the id's for
        // the facet children are prefixed with the OrderableList id
        // in addition to their own id, which also has the 
        // OrderableList id, as has been the convention for facets. This introduces
        // a redundancy in the facet id so the moveUp button now looks like
        //
        // "formid:orderablelistid:orderablelistid:orderablelistid_moveUpButton"
        //
        // It used to be "formid:orderablelistid_moveUpButton"
        // It would be better to encapsulate that knowledge in the
        // OrderableList renderer as does FileChooser which has the
        // same problem but because the select elements are not
        // facets in OrderableList they really do only have id's of the
        // form "formid:orderablelistid_list". Note that 
        // in these examples the "id" parameter is "formid:orderablelistid"
        //
        // Therefore for now, locate the additional prefix here as the
        // "facet" id. Assume that id never ends in ":" and if there is
        // no colon, id is the same as the component id.
        //
        var componentid = props.id;
        var colon_index = componentid.lastIndexOf(':');
        if (colon_index != -1) {
            componentid = props.id.substring(colon_index + 1);
        }
        var facetid = props.id + ":" + componentid;

        domNode.moveUpButton = document.getElementById(facetid + "_moveUpButton");
        domNode.moveDownButton = document.getElementById(facetid + "_moveDownButton");
        domNode.moveTopButton = document.getElementById(facetid + "_moveTopButton");
        domNode.moveBottomButton = document.getElementById(facetid + "_moveBottomButton");

        // Not a facet
        domNode.values = document.getElementById(props.id + "_list_value");

        // HTML elements may not have been created, yet.

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

        // The options of the select element from which selections are made 
        domNode.options = domNode.list.options;

        // The messages
        if (domNode.moveMessage == null) {
            "Select at least one item to remove";
        }

        // Set functions.
        domNode.moveUp = woodstock4_3._html.orderableList.moveUp;
        domNode.moveDown = woodstock4_3._html.orderableList.moveDown;
        domNode.moveTop = woodstock4_3._html.orderableList.moveTop;
        domNode.moveBottom = woodstock4_3._html.orderableList.moveBottom;
        domNode.updateButtons = woodstock4_3._html.orderableList.updateButtons;
        domNode.updateValue = woodstock4_3._html.orderableList.updateValue;
        domNode.onChange = woodstock4_3._html.orderableList.updateButtons;

        // Initialize buttons.
        domNode.updateButtons();
        return true;
    },

    /**
     * The original allowed items to be moved on both lists. Surely we
     * only sort items on the selected list? 
     * This does not work on Mozilla
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveUp: function() {
        var numOptions = this.options.length;
    
        // If there aren't at least two more selected items, then there is
        // nothing to move 
        if (numOptions < 2) {
            return false;
        }

        // Start by examining the first item 
        var index = 0;

        // We're not going to move the first item. Instead, we will start
        // on the first selected item that is below an unselected
        // item. We identify the first unselected item on the list, and 
        // then we will start on next item after that
        while (this.options[index].selected) {
            ++index;
            if (index == numOptions) {
                // We've reached the last item - no more items below it so
                // we return
                return false;
            }
        }

        // Start on the item below this one 
        ++index;

        for (index; index < numOptions; ++index) {
            if (this.options[index].selected == true) {
                var curOption = this.options[index];
                if (this.options.remove == null) {
                    // For Mozilla
                    this.options[index] = null;
                    this.list.add(curOption, this.options[index - 1]);
                } else {
                    // Windows and Opera do
                    this.options.remove(index);
                    this.options.add(curOption, index - 1);
                }
                // This is needed for Opera only
                this.options[index].selected = false;
                this.options[index - 1].selected = true;
            }
        }
        this.updateValue();
        return this.updateButtons();
    },

    /**
     * The original allowed items to be moved on both lists. Surely we
     * only sort items on the selected list? 
     * This does not work on Mozilla
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveTop: function() {
        var numOptions = this.options.length;
        // If there aren't at least two items, there is nothing to move  
        if (numOptions < 2) {
            return false;
        }

        // Find the first open spot 
        var openSpot = 0;
        while (this.options[openSpot].selected) {
            openSpot++;
        }

        // Find the first selected item below it
        var index = openSpot+1;

        for (index; index < numOptions; ++index) {
            if (this.options[index].selected == true) {
                var curOption = this.options[index];
                if (this.options.remove == null) {
                    // For Mozilla
                    this.options[index] = null;
                    this.list.add(curOption, this.options[openSpot]);
                } else {
                    // Windows and Opera do
                    this.options.remove(index);
                    this.options.add(curOption, openSpot);
                }

                // This is needed for Opera only
                this.options[index].selected = false;
                this.options[openSpot].selected = true;
                openSpot++;
            }
        }
        this.updateValue();
        return this.updateButtons();
    },

    /** 
     * The original allowed items to be moved on both lists. Surely we
     * only sort items on the selected list? 
     * This does not work on Mozilla
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveDown: function() {
        // Get the last item
        var index = this.options.length - 1;
    
        // If this number is less than zero, there was nothing on the list
        // and we return
        if (index < 0) {
            return false;
        }

        for (var i = index - 1; i >= 0; i--) {
            if (this.options[i].selected) {          
                var next = i + 1;
                if (this.options[next].selected == false) {
                    tmpText = this.options[i].text;
                    tmpValue = this.options[i].value;
                    this.options[i].text = this.options[next].text;
                    this.options[i].value = this.options[next].value;
                    this.options[i].selected = false;
                    this.options[next].text = tmpText;
                    this.options[next].value = tmpValue;
                    this.options[next].selected = true;
                }
            }
        }

        this.updateValue();
        return this.updateButtons();
    },

    /**
     * Move options to bottom.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    moveBottom: function() {
        var numOptions = this.options.length - 1;

        // If there aren't at least two items, there is nothing to move  
        if (numOptions < 1) {
            return false;
        }

        // Find the last open spot 
        var openSpot = numOptions;
        while (this.options[openSpot].selected) {
            openSpot--;
        }

        // Find the first selected item above it
        var index = openSpot-1;

        for (index; index > -1; --index) {
            if (this.options[index].selected == true) {
                var curOption = this.options[index];
	        if (this.options.remove == null) {
                    // For Mozilla
                    this.options[index] = null;
                    this.list.add(curOption, this.options[openSpot+1]);
                } else {
                    // Windows and Opera do
                    this.options.remove(index);
                    this.options.add(curOption, openSpot);
                }

                // This is needed for Opera only
                this.options[index].selected = false;
                this.options[openSpot].selected = true;
                openSpot--;
            }
        }
        this.updateValue();
        return this.updateButtons();
    },

    /**
     * Update buttons.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    updateButtons: function() {
        var numOptions = this.options.length;
        var selectedIndex = this.options.selectedIndex;
        var disabled = true;
        var index;

        // First, check if move down and move to bottom should be
        // enabled. These buttons should be enabled if and only if at
        // least one of the items are selected and there is at least one
        // open spot below a selected item. 
        if (selectedIndex > -1 && selectedIndex < numOptions -1) {
            index = selectedIndex+1;
            while (index < numOptions) {
                if (this.options[index].selected == false) {
                    disabled = false;
                    break;
                }
                index++;
            }
        }

        if (this.moveDownButton != null) {
            if (this.moveDownButton.setDisabled != null) {
                this.moveDownButton.setDisabled(disabled);
            } else {
                this.moveDownButton.disabled = disabled;
            }
        }

        if (this.moveBottomButton != null) {
            if (this.moveBottomButton.setDisabled != null) {
                this.moveBottomButton.setDisabled(disabled);
            } else {
                this.moveBottomButton.disabled = disabled;
            }
        }

        // First, check if move up and move to top should be
        // enabled. These buttons should be enabled if and only if at
        // least one of the items is selected and there is at least one
        // open spot above a selected item. 
        disabled = true;

        if (selectedIndex > -1) {
            index = numOptions - 1;
            while (index > 0) {
                if (this.options[index].selected) {
                    break;
                }
                index--;
            }
            index--;
            while (index > -1) {
                if (this.options[index].selected == false) {
                    disabled = false;
                    break;
                }
                index--;
            }
        }

        if (this.moveUpButton != null) {
            if (this.moveUpButton.setDisabled != null) {
                this.moveUpButton.setDisabled(disabled);
            } else {
                this.moveUpButton.disabled = disabled;
            }
        }

        if (this.moveTopButton != null) {
            if (this.moveTopButton.setDisabled != null) {
                this.moveTopButton.setDisabled(disabled);
            } else {
                this.moveTopButton.disabled = disabled;
            }
        }
        return true;
    },

    /**
     * Update value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    updateValue: function() {
        // Remove the options from the select that holds the actual
        // selected values
        while (this.values.length > 0) {
            this.values.remove(0);
        }

        // Create a new array consisting of the options marked as selected
        // on the official list
        var newOptions = new Array();
        var cntr = 0;
        var newOption;

        while (cntr < this.options.length) {
            newOption = document.createElement("option");
            if (this.options[cntr].text != null) {
                newOption.text = this.options[cntr].text;
            }
            if (this.options[cntr].value != null) {
                newOption.value = this.options[cntr].value;
            }
            newOption.selected = true;
            newOptions[newOptions.length] = newOption;
            ++ cntr;
        }
        cntr = 0;
        if (this.options.remove == null) {
            // For Mozilla
            while (cntr < newOptions.length) {
                this.values.add(newOptions[cntr], null);
                ++cntr;
            }
        } else {
            // Windows and Opera do
            while (cntr < newOptions.length) {
                this.values.add(newOptions[cntr], cntr);
                ++cntr;
            }
        }
        return true;
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.radiobutton");

woodstock4_3._dojo.require("woodstock4_3._html.rbcb");

/**
 * @class This class contains functions for radioButton components.
 * @static
 *
 * @deprecated See woodstock4_3.widget.radioButton
 * @private
 */
woodstock4_3._html.radiobutton = {
    /**
     * Set the disabled state for the given radiobutton element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id.
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) {    
        return woodstock4_3._html.rbcb.setDisabled(elementId, disabled, 
            "radio", "Rb", "RbDis");
    },

    /**
     * Set the disabled state for all the radio buttons in the radio button
     * group identified by controlName. If disabled
     * is set to true, the check boxes are displayed with disabled styles.
     *
     * @param {String} controlName The radio button group control name
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled) {    
        return woodstock4_3._html.rbcb.setGroupDisabled(controlName, disabled, 
            "radio", "Rb", "RbDis");
    },

    /**
     * Set the checked property for a radio button with the given element Id.
     *
     * @param {String} elementId The element Id
     * @param {boolean} checked true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */
    setChecked: function(elementId, checked) {
        return woodstock4_3._html.rbcb.setChecked(elementId, checked, "radio");
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.radiobutton = woodstock4_3._html.radiobutton;


woodstock4_3._dojo.provide("woodstock4_3._html.scheduler");
woodstock4_3._dojo.require("woodstock4_3._base.proto");

/** 
 * @class This class contains functions for scheduler components.
 * @static
 * @private
 */
woodstock4_3._html.scheduler = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @config {String} datePickerId
     * @config {String} dateFieldId
     * @config {String} dateClass
     * @config {String} selectedClass
     * @config {String} edgeClass
     * @config {String} edgeSelectedClass
     * @config {String} todayClass
     * @config {String} dateFormat
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize scheduler.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);
        domNode.dateLinkId = props.datePickerId + ":dateLink"; 

        // Set functions.
        domNode.setSelected = woodstock4_3._html.scheduler.setSelected;
        domNode.setDateValue = woodstock4_3._html.scheduler.setDateValue; 
        domNode.isToday = woodstock4_3._html.scheduler.isToday;

        return true;
    },

    /**
     * Set selected link.
     *
     * @param {String} value
     * @param {Node} link
     * @return {boolean} true if successful; otherwise, false.
     */
    setDateValue: function(value, link) {
        var widget = woodstock4_3.widget.common.getWidget(this.dateFieldId);
        if (widget) {
            widget.setProps({value: value});
        }
        return this.setSelected(link);	
    },

    /**
     * Set selected.
     *
     * @param {Node} link
     * @return {boolean} true if successful; otherwise, false.
     */
    setSelected: function(link) {
        if (link == null) {
            return false;
        }

        var dateLink;	
        var linkNum = 0;	

        // Remove any prior highlight 
        while (linkNum < 42) {
            dateLink = document.getElementById(this.dateLinkId + linkNum);  
            if (dateLink == null) {
                break;    
            }

            if (dateLink.className == this.edgeSelectedClass) {
                dateLink.className = this.edgeClass;
            } else if (dateLink.className == this.selectedClass) {
                if (this.isToday(dateLink.title)) {
                    dateLink.className = this.todayClass;
                } else {
                    dateLink.className = this.dateClass;
                }
            }
            linkNum++;
        }

        // apply the selected style to highlight the selected link
        if (link.className == this.dateClass || 
            link.className == this.todayClass) {	
            link.className = this.selectedClass;
        } else if (link.className == this.edgeClass) {
            link.className = this.edgeSelectedClass;
        }
        this.currentSelection = link;
        return true;
    },

    /**
     * Find out if date is today's date.
     *
     * @param {Object} date
     * @return {boolean} true if date is today.
     */
    isToday: function(date) {
        var todaysDate = new Date();
        var pattern = new String(this.dateFormat); 
        var yearIndex = pattern.indexOf("yyyy"); 
        var monthIndex = pattern.indexOf("MM"); 
        var dayIndex = pattern.indexOf("dd"); 
        var currYear = todaysDate.getFullYear(); 
        var currMonth = todaysDate.getMonth() + 1; 
        var currDay = todaysDate.getDate(); 

        if (currYear == parseInt(date.substr(yearIndex, 4))
            && currMonth == parseInt(date.substr(monthIndex, 2))
                && currDay == parseInt(date.substr(dayIndex, 2))) {
            return true;
        }
        return false;
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.table");

woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/** 
 * @class This class contains functions for table components.
 * <p>
 * Once the table is rendered, you will be able to invoke functions directly on 
 * the HTML element. For example:
 * </p><p><pre>
 * var table = document.getElementById("form1:table1");
 * var count = table.getAllSelectedRowsCount();
 * </pre></p><p>
 * Note: It is assumed that formElements.js has been included in the page. In
 * addition, all given HTML element IDs are assumed to be the outter most tag
 * enclosing the component.
 * </p>
 * @static
 * @private
 */
woodstock4_3._html.table = {
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // DOM functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * This function is used to confirm the number of selected components (i.e., 
     * checkboxes or radiobuttons used to de/select rows of the table), affected
     * by a delete action. This functionality requires the selectId property of
     * the tableColumn component and hiddenSelectedRows property of the
     * tableRowGroup component to be set.
     * <p>
     * If selections are hidden from view, the confirmation message indicates the
     * number of selections not displayed in addition to the total number of
     * selections. If selections are not hidden, the confirmation message indicates
     * only the total selections.
     * </p>
     *
     * @return {boolean} A value of true if the user clicks the "OK" button, or 
     * false if the user clicks the "Cancel" button.
     */
    confirmDeleteSelectedRows: function() {
        return this.confirmSelectedRows(this.deleteSelectionsMsg);
    },

    /**
     * This function is used to confirm the number of selected components (i.e., 
     * checkboxes or radiobuttons used to de/select rows of the table), affected by
     * an action such as edit, archive, etc. This functionality requires the 
     * selectId property of the tableColumn component and hiddenSelectedRows
     * property of the tableRowGroup component to be set.
     * <p>
     * If selections are hidden from view, the confirmation message indicates the
     * number of selections not displayed in addition to the total number of
     * selections. If selections are not hidden, the confirmation message indicates
     * only the total selections.
     * </p>
     *
     * @param {String} message The confirmation message (e.g., Archive all selections?).
     * @return {boolean} A value of true if the user clicks the "OK" button, or 
     * false if the user clicks the "Cancel" button.
     */
    confirmSelectedRows: function(message) {
        // Get total selections message.
        var totalSelections = this.getAllSelectedRowsCount();
        var totalSelectionsArray = this.totalSelectionsMsg.split("{0}");
        var totalSelectionsMsg = totalSelectionsArray[0] + totalSelections;

        // Append hidden selections message.
        var hiddenSelections = this.getAllHiddenSelectedRowsCount();
        if (hiddenSelections > 0) {
            // Get hidden selections message.
            var hiddenSelectionsArray = this.hiddenSelectionsMsg.split("{0}");
            var hiddenSelectionsMsg = hiddenSelectionsArray[0] + hiddenSelections;

            totalSelectionsMsg = hiddenSelectionsMsg + totalSelectionsMsg;
        }
        return (message != null)
            ? confirm(totalSelectionsMsg + message)
            : confirm(totalSelectionsMsg);
    },

    /**
     * This function is used to toggle the filter panel from the filter menu. This
     * functionality requires the filterId of the table component to be set. In 
     * addition, the selected value must be set as well to restore the default
     * selected value when the embedded filter panel is closed.
     * <p>
     * If the "Custom Filter" option has been selected, the table filter panel is 
     * toggled. In this scenario, false is returned indicating the onChange event,
     * generated by the table filter menu, should not be allowed to continue.
     * </p><p>
     * If the "Custom Filter Applied" option has been selected, no action is taken.
     * Instead, the is selected filter menu is reverted back to the "Custom Filter" 
     * selection. In this scenario, false is also returned indicating the onChange 
     * event, generated by the table filter menu, should not be allowed to continue.
     * </p><p>
     * For all other selections, true is returned indicating the onChange event, 
     * generated by the table filter menu, should be allowed to continue.
     * </p>
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    filterMenuChanged: function() {
        // Validate panel IDs.
        if (this.panelToggleIds == null || this.panelToggleIds.length == 0) {
            return false;
        }

        // Get filter menu.
        var menu = this._getSelectElement(this.panelToggleIds[this.FILTER]);
        if (menu == null) {
            return true;
        }

        // Test if table filter panel should be opened.
        if (menu.options[menu.selectedIndex].value == this.customFilterOptionValue) {
            this.toggleFilterPanel();
            return false;
        } else if (menu.options[menu.selectedIndex].
                value == this.customFilterAppliedOptionValue) {
            // Set selected option.
            menu.selectedIndex = 0;
            for (var i = 0; i < menu.options.length; i++) {
                if (menu.options[i].value == this.customFilterOptionValue) {
                    menu.options[i].selected = true;
                    break;
                }
            }
            return false;
        }
        return true;
    },

    /**
     * This function is used to get the number of selected components in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select 
     * rows of the table). This functionality requires the selectId property of the
     * tableColumn component and hiddenSelectedRows property of the table
     * component to be set.
     *
     * @return {int} The number of components selected in the current page.
     */
    getAllSelectedRowsCount: function() {
        return this.getAllHiddenSelectedRowsCount() +
            this.getAllRenderedSelectedRowsCount();
    },

    /**
     * This function is used to get the number of selected components, in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table), currently hidden from view. This functionality requires 
     * the selectId property of the tableColumn component and hiddenSelectedRows
     * property of the table component to be set.
     *
     * @return {int} The number of selected components hidden from view.
     */
    getAllHiddenSelectedRowsCount: function() {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return count;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            count = count + this._getGroupHiddenSelectedRowsCount(this.groupIds[i]);
        }
        return count;
    },

    /**
     * This function is used to get the number of selected components, in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table), currently rendered. This functionality requires 
     * the selectId property of the tableColumn component to be set.
     *
     * @return {int} The number of selected components hidden from view.
     */
    getAllRenderedSelectedRowsCount: function() {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return count;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            count = count + this._getGroupRenderedSelectedRowsCount(this.groupIds[i]);
        }
        return count;
    },

    /**
     * This function is used to initialize all rows displayed in the table when the
     * state of selected components change (i.e., checkboxes or radiobuttons used to
     * de/select rows of the table). This functionality requires the selectId
     * property of the tableColumn component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    initAllRows: function() {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return false;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            this._initGroupRows(this.groupIds[i]);
        }
        return true;
    },

    /**
     * This function is used to toggle the filter panel open or closed. This
     * functionality requires the filterId of the table component to be set. In 
     * addition, the selected value must be set as well to restore the default
     * selected value when the embedded filter panel is closed.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    toggleFilterPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Toggle filter panel.
        this._togglePanel(this.panelIds[this.FILTER],
        this.panelFocusIds[this.FILTER], this.panelToggleIds[this.FILTER]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    /**
     * This function is used to toggle the preferences panel open or closed. This
     * functionality requires the filterId of the table component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    togglePreferencesPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Toggle preferences panel.
        this._togglePanel(this.panelIds[this.PREFERENCES],
        this.panelFocusIds[this.PREFERENCES], this.panelToggleIds[this.PREFERENCES]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Internal functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element ID for the component.
     *
     * // Panel Properties
     * @config {Array} panelIds An array of embedded panel IDs.</li>
     * @config {Array} panelFocusIds An array of IDs used to set focus for open panels.</li>
     * @config {Array} panelToggleIds An array of IDs used to toggle embedded panels.</li>
     * @config {Array} panelToggleIconsOpen An array of toggle icons for open panels.</li>
     * @config {Array} panelToggleIconsClose An array of toggle icons for closed panels.</li>
     *
     * // Filter Properties
     * @config {String} basicFilterStyleClass The style class for basic or no filters.</li>
     * @config {String} customFilterStyleClass The style class for custom filters.</li>
     * @config {String} customFilterOptionValue The custom filter menu option value.</li>
     * @config {String} customFilterAppliedOptionValue The custom filter applied menu option value.</li>
     *
     * // Sort Panel Properties
     * @config {Array} sortColumnMenuIds An array of HTML element IDs for sort column menu components.</li>
     * @config {Array} sortOrderMenuIds An array of HTML element IDs for sort order menu components.</li>
     * @config {Array} sortOrderToolTips An array of tool tips used for sort order menus.</li>
     * @config {Array} sortOrderToolTipsAscending An array of ascending tool tips used for sort order menus.</li>
     * @config {Array} sortOrderToolTipsDescending An array of descending tool tips used for sort order menus.</li>
     * @config {String} duplicateSelectionMsg The message displayed for duplicate menu selections.</li>
     * @config {String} missingSelectionMsg The message displayed for missing menu selections.</li>
     * @config {String} selectSortMenuOptionValue The sort menu option value for the select column.</li>
     * @config {boolean} hiddenSelectedRows Flag indicating that selected rows might be currently hidden from view.</li>
     * @config {boolean} paginated Flag indicating table is in pagination mode.</li>
     *
     * // Group Properties
     * @config {String} selectRowStylClass The style class for selected rows.</li>
     * @config {Array} selectIds An arrary of component IDs used to select rows of the table.</li>
     * @config {Array} groupIds An array of TableRowGroup IDs rendered for the table.</li>
     * @config {Array} rowIds An array of row IDs for rendered for each TableRowGroup.</li>
     * @config {Array} hiddenSelectedRowCounts An array of selected row counts hidden from view.</li>
     * @config {String} hiddenSelectionsMsg The hidden selections message for confirm dialog.</li>
     * @config {String} totalSelectionsMsg The total selections message for confirm dialog.</li>
     * @config {String} deleteSelectionsMsg The delete selections message for confirm dialog.</li>
     *
     * // Group Panel Properties
     * @param {String} columnFooterId ID for column footer.</li>
     * @param {String} columnHeaderId ID for column header.</li>
     * @param {String} tableColumnFooterId ID for table column footer.</li>
     * @param {String} groupFooterId ID for group footer.</li>
     * @param {String} groupPanelToggleButtonId ID for group panel toggle button.</li>
     * @param {String} groupPanelToggleButtonToolTipOpen tool tip for open row group.</li>
     * @param {String} groupPanelToggleButtonToolTipClose tool tip for closed row group.</li>
     * @param {String} groupPanelToggleIconOpen The toggle icon for open row group.</li>
     * @param {String} groupPanelToggleIconClose The toggle icon for closed row group.</li>
     * @param {String} warningIconId ID for warning icon.</li>
     * @param {String} warningIconOpen The warning icon for open row group.</li>
     * @param {String} warningIconClosed The warning icon for closed row group.</li>
     * @param {String} warningIconToolTipOpen The warning icon tool tip for open row group.</li>
     * @param {String} warningIconToolTipClose The warning icon tool tip for closed row group.</li>
     * @param {String} collapsedHiddenFieldId ID for collapsed hidden field.</li>
     * @param {String} selectMultipleToggleButtonId ID for select multiple toggle button.</li>
     * @param {String} selectMultipleToggleButtonToolTip The select multiple toggle button tool tip.</li>
     * @param {String} selectMultipleToggleButtonToolTipSelected The select multiple toggle button tool tip when selected.</li>
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize table.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

        // Misc properties.
        domNode.SEPARATOR = ":";   // NamingContainer separator.

        // Panel toggle keys.
        domNode.SORT        = 0;
        domNode.PREFERENCES = 1;
        domNode.FILTER      = 2;

        // Sort keys.
        domNode.PRIMARY   = 0;
        domNode.SECONDARY = 1;
        domNode.TERTIARY  = 2;

        // Replace extra backslashes, JSON escapes new lines (e.g., \\n).
	domNode.hiddenSelectionsMsg = (props.hiddenSelectionsMsg != null)
            ? props.hiddenSelectionsMsg.replace(/\\n/g, "\n") : "";
        domNode.totalSelectionsMsg = (props.totalSelectionsMsg != null)
            ? props.totalSelectionsMsg.replace(/\\n/g, "\n") : "";
	domNode.missingSelectionMsg = (props.missingSelectionMsg != null)
            ? props.missingSelectionMsg.replace(/\\n/g, "\n") : "";
	domNode.duplicateSelectionMsg = (props.duplicateSelectionMsg != null)
            ? props.duplicateSelectionMsg.replace(/\\n/g, "\n") : "";
        domNode.deleteSelectionsMsg = (props.deleteSelectionsMsg != null)
            ? props.deleteSelectionsMsg.replace(/\\n/g, "\n") : "";

        // Private functions.
        domNode._getGroupSelectedRowsCount = woodstock4_3._html.table._getGroupSelectedRowsCount;
        domNode._getGroupHiddenSelectedRowsCount = woodstock4_3._html.table._getGroupHiddenSelectedRowsCount;
        domNode._getGroupRenderedSelectedRowsCount = woodstock4_3._html.table._getGroupRenderedSelectedRowsCount;
        domNode._getSelectElement = woodstock4_3._html.table._getSelectElement;
        domNode._initGroupRows = woodstock4_3._html.table._initGroupRows;
        domNode._initPrimarySortOrderMenu = woodstock4_3._html.table._initPrimarySortOrderMenu;
        domNode._initPrimarySortOrderMenuToolTip = woodstock4_3._html.table._initPrimarySortOrderMenuToolTip;
        domNode._initSecondarySortOrderMenu = woodstock4_3._html.table._initSecondarySortOrderMenu;
        domNode._initSecondarySortOrderMenuToolTip = woodstock4_3._html.table._initSecondarySortOrderMenuToolTip;
        domNode._initSortColumnMenus = woodstock4_3._html.table._initSortColumnMenus;
        domNode._initSortOrderMenu = woodstock4_3._html.table._initSortOrderMenu;
        domNode._initSortOrderMenus = woodstock4_3._html.table._initSortOrderMenus;
        domNode._initSortOrderMenuToolTip = woodstock4_3._html.table._initSortOrderMenuToolTip;
        domNode._initTertiarySortOrderMenu = woodstock4_3._html.table._initTertiarySortOrderMenu;
        domNode._initTertiarySortOrderMenuToolTip = woodstock4_3._html.table._initTertiarySortOrderMenuToolTip;
        domNode._menuChanged = woodstock4_3._html.table._menuChanged;
        domNode._resetFilterMenu = woodstock4_3._html.table._resetFilterMenu;
        domNode._selectAllRows = woodstock4_3._html.table._selectAllRows;
        domNode._selectGroupRows = woodstock4_3._html.table._selectGroupRows;
        domNode._toggleGroupPanel = woodstock4_3._html.table._toggleGroupPanel;
        domNode._togglePanel = woodstock4_3._html.table._togglePanel;
        domNode._toggleSortPanel = woodstock4_3._html.table._toggleSortPanel;
        domNode._validateSortPanel = woodstock4_3._html.table._validateSortPanel;

        // Public functions.
        domNode.confirmDeleteSelectedRows = woodstock4_3._html.table.confirmDeleteSelectedRows;
        domNode.confirmSelectedRows = woodstock4_3._html.table.confirmSelectedRows;
        domNode.filterMenuChanged = woodstock4_3._html.table.filterMenuChanged;
        domNode.getAllSelectedRowsCount = woodstock4_3._html.table.getAllSelectedRowsCount;
        domNode.getAllHiddenSelectedRowsCount = woodstock4_3._html.table.getAllHiddenSelectedRowsCount;
        domNode.getAllRenderedSelectedRowsCount = woodstock4_3._html.table.getAllRenderedSelectedRowsCount;
        domNode.initAllRows = woodstock4_3._html.table.initAllRows;
        domNode.toggleFilterPanel = woodstock4_3._html.table.toggleFilterPanel;
        domNode.togglePreferencesPanel = woodstock4_3._html.table.togglePreferencesPanel;

        return true;
    },

    /**
     * This function is used to get the number of selected components for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table).
     * This functionality requires the selectId property of the tableColumn component
     * and the hiddenSelectedRows property of the table component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of components selected in the current page.
     * @private
     */
    _getGroupSelectedRowsCount: function(groupId) {
        return this._getGroupHiddenSelectedRowsCount(groupId) + 
            this._getGroupRenderedSelectedRowsCount(groupId);
    },

    /**
     * This function is used to get the number of selected components, for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table),
     * currently hidden from view. This functionality requires the selectId property
     * of the tableColumn component and hiddenSelectedRows property of the table
     * component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of selected components hidden from view.
     * @private
     */
    _getGroupHiddenSelectedRowsCount: function(groupId) {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return count;
        }

        // Find the given group Id in the groupIds array.
        for (var i = 0; i < this.groupIds.length; i++) {
            // Get selectId and rowIds array associated with groupId.
            if (groupId == this.groupIds[i]) {
                count = this.hiddenSelectedRowCounts[i];
                break;
            }
        }
        return count;
    },

    /**
     * This function is used to get the number of selected components, for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table),
     * currently rendered. This functionality requires the selectId property of the
     * tableColumn component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of components selected in the current page.
     * @private
     */
    _getGroupRenderedSelectedRowsCount: function(groupId) {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return count;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + 
                this.SEPARATOR + selectId);
            if (select == null) {
                continue;
            }
            var props = select.getProps();
            if (props.disabled != true && props.checked == true) {
                count++;
            }
        }
        return count;
    },

    /**
     * Use this function to access the HTML select element that makes up
     * the dropDown.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the dropDown).
     * @return {Node} a reference to the select element.
     */
    _getSelectElement: function(elementId) { 
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * This function is used to initialize rows for the given group when the state
     * of selected components change (i.e., checkboxes or radiobuttons used to
     * de/select rows of the table). This functionality requires the selectId
     * property of the tableColumn component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initGroupRows: function(groupId) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        var checked = true; // Checked state of multiple select toggle button.
        var selected = false; // At least one component is selected.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + 
                this.SEPARATOR + selectId);
            if (select == null) {
                continue;
            }

            // Set row style class.
            var row = document.getElementById(this.groupIds[i] + 
                this.SEPARATOR + rowIds[k]);
            var props = select.getProps();
            if (props.disabled == true) {
                // Don't highlight row or set checked/selected properties.
                woodstock4_3._base.common._stripStyleClass(row, 
                    this.selectRowStyleClass);
            } else if (props.checked == true) {
                woodstock4_3._base.common._addStyleClass(row, 
                    this.selectRowStyleClass);
                selected = true;
            } else {
                woodstock4_3._base.common._stripStyleClass(row, 
                    this.selectRowStyleClass);
                checked = false;
            }
        }

        // Set multiple select toggle button state.
        var title;
        var checkbox = document.getElementById(groupId + this.SEPARATOR + 
            this.selectMultipleToggleButtonId);
        if (checkbox != null) {
            title = (checked) 
                ? this.selectMultipleToggleButtonToolTipSelected
                : this.selectMultipleToggleButtonToolTip;
            checkbox.setProps({
                "checked": checked,
                "title": title
            });
        }

        // Get flag indicating groupis collapsed.
        var prefix = groupId + this.SEPARATOR;
        var collapsed = !woodstock4_3._base.common._isVisible(prefix + rowIds[0]);

        // Set next warning image.
        var image = document.getElementById(prefix + this.warningIconId);
        if (image != null) {
            var src = this.warningIconOpen;
            title = this.warningIconToolTipOpen;

            // Don't show icon when multiple select is checked.
            if (collapsed && selected && !checked) {
                src = this.warningIconClose;
                title = this.warningIconToolTipClose;
            }
            image.setProps({
               "src": src,
               "title": title
            });
        }
        return true;
    },

    /**
     * This function is used to initialize the primary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initPrimarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.PRIMARY], 
            this.sortOrderMenuIds[this.PRIMARY]);
    },  

    /**
     * This function is used to initialize the primary sort order menu tool tips 
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initPrimarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(
            this.sortOrderMenuIds[this.PRIMARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.PRIMARY], 
            this.sortOrderMenuIds[this.PRIMARY]);
    },

    /**
     * This function is used to initialize the secondary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSecondarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.SECONDARY], 
            this.sortOrderMenuIds[this.SECONDARY]);
    },

    /**
     * This function is used to initialize the secondary sort order menu tool tips
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSecondarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(
            this.sortOrderMenuIds[this.SECONDARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.SECONDARY], 
            this.sortOrderMenuIds[this.SECONDARY]);
    },

    /**
     * This function is used to initialize the primary, secondary, and tertiary 
     * sort column menus used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortColumnMenus: function() {
        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Set initial selected option for all sort column menus.
        for (var i = 0; i < this.sortColumnMenuIds.length; i++) {
            // Get sort column menu.
            var sortColumnMenu = this._getSelectElement(this.sortColumnMenuIds[i]);
            if (sortColumnMenu == null) {
                continue;
            }

            // Set default selected value.
            sortColumnMenu.selectedIndex = 0;

            // Set selected option.
            for (var k = 0; k < sortColumnMenu.options.length; k++) {
                if (sortColumnMenu.options[k].defaultSelected == true) {
                    sortColumnMenu.options[k].selected = true;
                    break;
                }
            }
            // Ensure hidden filed values are updated.
            this._menuChanged(this.sortColumnMenuIds[i]);
        }
        return true;
    },

    /**
     * This function is used to initialize the primary, secondary, and tertiary 
     * sort order menus used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenus: function() {
        // Validate sort order menu IDs.
        if (this.sortOrderMenuIds == null || this.sortOrderMenuIds.length == 0) {
            return false;
        }

        // Set initial selected option for all sort column menus.
        for (var i = 0; i < this.sortOrderMenuIds.length; i++) {
            // Get sort order menu.
            var sortOrderMenu = this._getSelectElement(this.sortOrderMenuIds[i]);
            if (sortOrderMenu == null) {
                continue;
            }

            // Set default selected value.
            sortOrderMenu.selectedIndex = 0;

            // Get sort column menu.
            var sortColumnMenu = this._getSelectElement(this.sortColumnMenuIds[i]);
            if (sortColumnMenu != null) {
                // If the table is paginated and there are no hidden selected rows, the select
                // column cannot be sorted descending.
                if (sortColumnMenu.options[sortColumnMenu.selectedIndex].
                        value == this.selectSortMenuOptionValue
                        && !this.hiddenSelectedRows && this.paginated) {
                    sortOrderMenu.options[1].disabled = true;
                } else {
                    sortOrderMenu.options[1].disabled = false;
                }
            }

            // Set selected option.
            for (var k = 0; k < sortOrderMenu.options.length; k++) {
                if (sortOrderMenu.options[k].defaultSelected == true) {
                    sortOrderMenu.options[k].selected = true;
                    break;
                }
            }
            // Ensure hidden filed values and styles are updated.
            this._menuChanged(this.sortOrderMenuIds[i]);

            // Initialize tool tip.
            this._initSortOrderMenuToolTip(this.sortColumnMenuIds[i], this.sortOrderMenuIds[i]);
        }
        return true;
    },

    /**
     * This function is used to initialize sort order menus used in the
     * sort panel. When a sort column menu changes, the given sort order 
     * menu is initialized based on the the selected value of the given
     * sort column menu.
     *
     * @param {String} sortColumnMenuId The HTML element ID for the sort column menu component.
     * @param {String} sortOrderMenuId The HTML element ID for the sort order menu component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenu: function(sortColumnMenuId, sortOrderMenuId) {
        if (sortColumnMenuId == null || sortOrderMenuId == null) {
            return false;
        }

        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Validate sort order menu IDs.
        if (this.sortOrderMenuIds == null || this.sortOrderMenuIds.length == 0) {
            return false;
        }

        // Get sort column menu.
        var sortColumnMenu = this._getSelectElement(sortColumnMenuId);
        if (sortColumnMenu == null) {
            return false;
        }

        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(sortOrderMenuId);
        if (sortOrderMenu == null) {
            return false;
        }

        // Reset selected option.
        sortOrderMenu.selectedIndex = 0; // Default ascending.

        // Get sort column menu selected index.            
        var selectedIndex = (sortColumnMenu.selectedIndex > -1)
            ? sortColumnMenu.selectedIndex : 0; // Default to first option.

        // If the table is paginated and there are no hidden selected rows, the select
        // column cannot be sorted descending.
        if (sortColumnMenu.options[selectedIndex].value == this.selectSortMenuOptionValue
                && !this.hiddenSelectedRows && this.paginated) {
            sortOrderMenu.options[1].disabled = true;
        } else {
            sortOrderMenu.options[1].disabled = false;
        }

        // Attempt to match the selected index of the given sort column menu with the
        // default selected value from either sort column menu. If a match is found, the 
        // default selected value of the associated sort order menu is retrieved. This
        // default selected value is set as the current selection of the given sort 
        // order menu.
        for (var i = 0; i < this.sortColumnMenuIds.length; i++) {
            // Get the current sort column menu to test the default selected value.
            var currentSortColumnMenu = this._getSelectElement(
                this.sortColumnMenuIds[i]);
            if (currentSortColumnMenu == null) {
                continue;
            }

            // Find default selected value for the current sort column menu.
            var defaultSelected = null;
            for (var k = 0; k < currentSortColumnMenu.options.length; k++) {
                if (currentSortColumnMenu.options[k].defaultSelected == true) {
                    defaultSelected = currentSortColumnMenu.options[k].value;
                    break;
                }
            }

            // Match default selected value with selected index value.
            if (defaultSelected != null && defaultSelected ==
                    sortColumnMenu.options[selectedIndex].value) {
                // Get current sort order menu to test the default selected value.
                var currentSortOrderMenu = this._getSelectElement(
                    this.sortOrderMenuIds[i]);
                if (currentSortOrderMenu == null) {
                    continue;
                }

                // Find default selected index for the current sort order menu.
                var defaultSelectedIndex = -1;
                for (var c = 0; c < currentSortOrderMenu.options.length; c++) {
                    if (currentSortOrderMenu.options[c].defaultSelected == true) {
                        defaultSelectedIndex = c;
                        break;
                    }
                }

                // Set selected value for given sort order menu.
                if (defaultSelectedIndex > -1) {
                    sortOrderMenu.options[defaultSelectedIndex].selected = true;
                } else {
                    sortOrderMenu.options[0].selected = true; // Default ascending.
                }
                break;
            }
        }
        // Ensure hidden field values and styles are updated.
        this._menuChanged(sortOrderMenuId);

        // Set sort order menu tool tip.
        return this._initSortOrderMenuToolTip(sortColumnMenuId, sortOrderMenuId);
    },

    /**
     * This function is used to initialize sort order menu tool tips used in the 
     * sort panel. When a sort column menu changes, the given sort order 
     * menu tool tip is initialized based on the the selected value of the given
     * sort column menu.
     *
     * @param {String} sortColumnMenuId The HTML element ID for the sort column menu component.
     * @param {String} sortOrderMenuId The HTML element ID for the sort order menu component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenuToolTip: function(sortColumnMenuId, sortOrderMenuId) {
        if (sortColumnMenuId == null || sortOrderMenuId == null) {
            return false;
        }

        // Get sort column menu.
        var sortColumnMenu = this._getSelectElement(sortColumnMenuId);
        if (sortColumnMenu == null) {
            return false;
        }

        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(sortOrderMenuId);
        if (sortOrderMenu == null) {
            return false;
        }

        // Get tool tip associated with given sort order menu.
        var toolTip = "";
        if (this.sortOrderToolTips != null && this.sortOrderToolTips.length != 0
                && this.sortOrderMenuIds != null) {
            for (var i = 0; i < this.sortOrderMenuIds.length; i++) {
                // The tool tip is at index zero, after splitting the message.
                if (sortOrderMenuId == this.sortOrderMenuIds[i]) {
                    toolTip = this.sortOrderToolTips[i].split("{0}")[0];
                    break;
                }
            }
        }

        // Get sort column menu selected index.            
        var selectedIndex = (sortColumnMenu.selectedIndex > -1)
            ? sortColumnMenu.selectedIndex : 0; // Default to first option.

        // Set tool tip.
        if (sortOrderMenu.options[sortOrderMenu.selectedIndex].value == "true") {
            sortOrderMenu.title = toolTip + this.sortOrderToolTipsDescending[selectedIndex];
        } else {
            // Default ascending.
            sortOrderMenu.title = toolTip + this.sortOrderToolTipsAscending[selectedIndex];
        }
        return true;
    },

    /**
     * This function is used to initialize the tertiary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initTertiarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.TERTIARY], 
            this.sortOrderMenuIds[this.TERTIARY]);
    },

    /**
     * This function is used to initialize the tertiary sort order menu tool tips
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initTertiarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = this._getSelectElement(
            this.sortOrderMenuIds[this.TERTIARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.TERTIARY], 
            this.sortOrderMenuIds[this.TERTIARY]);
    },

    /**
     * This function is invoked by the choice onselect action to set the
     * selected, and disabled styles.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     */
    _menuChanged: function(elementId) {         
        var widget = woodstock4_3.widget.common.getWidget(elementId);
        if (widget) {
            return widget._changed();
        }
        return false;
    },

    /**
     * This function is used to reset filter drop down menu. This functionality 
     * requires the filterId of the table component to be set. In addition,
     * the selected value must be set as well to restore the default selected
     * value when the embedded filter panel is closed.
     *
     * @param {String} filterId The HTML element ID of the filter menu.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _resetFilterMenu: function(filterId) {
        if (filterId == null) {
            return false;
        }

        // Get filter menu.
        var menu = this._getSelectElement(filterId);
        if (menu == null) {
            return true;
        }

        // Get div element associated with the filter panel.
        var div = document.getElementById(this.panelIds[this.FILTER]);
        if (div == null) {
            return false;
        }

        // Set selected style.
        if (woodstock4_3._base.common._isVisibleElement(div)) {
            woodstock4_3._base.common._stripStyleClass(menu, this.basicFilterStyleClass);
            woodstock4_3._base.common._addStyleClass(menu, this.customFilterStyleClass);
        } else {
            // Reset default selected option.
            menu.selectedIndex = 0;
            for (var i = 0; i < menu.options.length; i++) {
                if (menu.options[i].defaultSelected == true) {
                    menu.options[i].selected = true;
                    break;
                }
            }
            woodstock4_3._base.common._stripStyleClass(menu, this.customFilterStyleClass);
            woodstock4_3._base.common._addStyleClass(menu, this.basicFilterStyleClass);
        }
        return true;
    },

    /**
     * This function is used to set the selected state components in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table). This functionality requires the selectId property of
     * the tableColumn component to be set.
     *
     * @param {boolean} selected Flag indicating components should be selected.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _selectAllRows: function(selected) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return false;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            this._selectGroupRows(this.groupIds[i], selected);
        }
        return true;
    },

    /**
     * This function is used to set the selected state components for the given row group
     * (i.e., checkboxes or radiobuttons used to de/select rows of the table). This 
     * functionality requires the selectId property of the tableColumn component to be
     * set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @param {boolean} selected Flag indicating components should be selected.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _selectGroupRows: function(groupId, selected) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + this.SEPARATOR + selectId);
            if (select == null || select.getProps().disabled == true) {
                continue;
            }
            select.setProps({checked: new Boolean(selected).valueOf()});
        }
        return this._initGroupRows(groupId); // Set row highlighting.
    },

    /**
     * This function is used to toggle row group panels open or closed.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _toggleGroupPanel: function(groupId) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get rowIds array associated with groupId.
        var rowIds = null;
        for (var c = 0; c < this.groupIds.length; c++) {
            if (groupId == this.groupIds[c]) {
                rowIds = this.rowIds[c];
                break;
            }
        }

        // If row IDs could not be found, do not continue.
        if (rowIds == null) {
            return false;
        }

        // Get flag indicating group is collapsed.
        var prefix = groupId + this.SEPARATOR;
        var collapsed = !woodstock4_3._base.common._isVisible(prefix + rowIds[0]);

        // Get the number of column headers and table column footers for all 
        // TableRowGroup children.
        var _prefix;
        var columnHeaderId;
        var tableColumnFooterId;
        var columnHeadersCount = 0;
        var tableColumnFootersCount = 0;
        for (var i = 0; i < this.groupIds.length; i++) {
            // Only need to test first nested column header/footer; thus, index 0 is used.        
            _prefix = this.groupIds[i] + this.SEPARATOR;
            columnHeaderId = _prefix + this.columnHeaderId + this.SEPARATOR + "0";
            tableColumnFooterId = _prefix + this.tableColumnFooterId + this.SEPARATOR + "0";
            if (document.getElementById(columnHeaderId) != null) {
                columnHeadersCount++;
            }
            if (document.getElementById(tableColumnFooterId) != null) {
                tableColumnFootersCount++;
            }
        }

        // Toggle nested column footer.
        var rowIndex = 0;
        var columnFooterId;
        while (true) {
            columnFooterId = prefix + this.columnFooterId + 
                this.SEPARATOR + rowIndex++;
            if (document.getElementById(columnFooterId) == null) {
                break;
            }
            woodstock4_3._base.common._setVisible(columnFooterId, collapsed);
        }

        // Toggle column header only if multiple column headers are shown.
        if (columnHeadersCount > 1) {
            rowIndex = 0;
            while (true) {
                columnHeaderId = prefix + this.columnHeaderId + 
                    this.SEPARATOR + rowIndex++;
                if (document.getElementById(columnHeaderId) == null) {
                    break;
                }            
                woodstock4_3._base.common._setVisible(columnHeaderId, collapsed);
            }
        }

        // Toggle table column footer only if multiple column footers are shown.
        if (tableColumnFootersCount > 1) {
            rowIndex = 0;
            while (true) {
                tableColumnFooterId = prefix + this.tableColumnFooterId + 
                    this.SEPARATOR + rowIndex++;
                if (document.getElementById(tableColumnFooterId) == null) {
                    break;
                }
                woodstock4_3._base.common._setVisible(tableColumnFooterId, collapsed);
            }
        }

        // Toggle group rows.
        var rowId;
        for (var k = 0; k < rowIds.length; k++) {
            rowId = prefix + rowIds[k];
            woodstock4_3._base.common._setVisible(rowId, collapsed);
        }

        // Toggle group footers.
        woodstock4_3._base.common._setVisible(prefix + this.groupFooterId, collapsed);

        // Set next toggle button image.
        var groupPanelToggleButtonId = prefix + this.groupPanelToggleButtonId;
        var imgHyperlink = document.getElementById(groupPanelToggleButtonId);
        if (imgHyperlink != null) {
            if (collapsed) {
                imgHyperlink.setProps({
                    enabledImage: {
                        src: this.groupPanelToggleIconOpen  
                    },
                    title: this.groupPanelToggleButtonToolTipOpen
                });
            } else {
                imgHyperlink.setProps({
                    enabledImage: {
                        src: this.groupPanelToggleIconClose
                    },
                    title: this.groupPanelToggleButtonToolTipClose
                });
            }
        }

        // Set collapsed hidden field.
        var hiddenField = document.getElementById(prefix + this.collapsedHiddenFieldId);
        if (hiddenField != null) {
            hiddenField.value = !collapsed;
        }
        return this._initGroupRows(groupId); // Set next warning image.
    },

    /**
     * This function is used to toggle embedded panels.
     *
     * @param {String} panelId The panel ID to toggle.
     * @param {String} panelFocusIdOpen The ID used to set focus when panel is opened.
     * @param {String} panelFocusIdClose The ID used to set focus when panel is closed.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _togglePanel: function(panelId, panelFocusIdOpen, panelFocusIdClose) {
        if (panelId == null) {
            return false;
        }

        // Toggle the given panel, hide all others.
        for (var i = 0; i < this.panelIds.length; i++) {
            // Get div element associated with the panel.
            var div = document.getElementById(this.panelIds[i]);
            if (div == null) {
                continue;
            }

            // Set display value. Alternatively, we could set div.style.display
            // equal to "none" or "block" (i.e., hide/show).
            if (this.panelIds[i] == panelId) {
                // Set focus when panel is toggled -- bugtraq 6316565.
                var focusElement = null;

                if (woodstock4_3._base.common._isVisibleElement(div)) {
                    woodstock4_3._base.common._setVisibleElement(div, false); // Hide panel.
                    focusElement = document.getElementById(panelFocusIdClose);
                } else {
                    woodstock4_3._base.common._setVisibleElement(div, true); // Show panel.
                    focusElement = document.getElementById(panelFocusIdOpen);
                }

                // Set focus.
                if (focusElement != null) {
                    focusElement.focus();
                }
            } else {
                // Panels are hidden by default.
                woodstock4_3._base.common._setVisibleElement(div, false);
            }

            // Get image from icon hyperlink component.
            var imgHyperlink = document.getElementById(this.panelToggleIds[i]);
            if (imgHyperlink == null || imgHyperlink.getProps().enabledImage == null) {
                continue; // Filter panel facet may be in use.
            }

            // Set image.
            if (woodstock4_3._base.common._isVisibleElement(div)) {
                imgHyperlink.setProps({
                    enabledImage: {
                        src: this.panelToggleIconsOpen[i]
                    }
                });
            } else {
                imgHyperlink.setProps({
                    enabledImage: {
                        src: this.panelToggleIconsClose[i]
                    }
                });
            }
        }
        return true;
    },

    /**
     * This function is used to toggle the sort panel open or closed. This
     * functionality requires the filterId of the table component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _toggleSortPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Initialize sort column and order menus.
        this._initSortColumnMenus(); 
        this._initSortOrderMenus();

        // Toggle sort panel.
        this._togglePanel(this.panelIds[this.SORT], 
        this.panelFocusIds[this.SORT], this.panelToggleIds[this.SORT]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    /**
     * This function is used to validate sort column menu selections 
     * for the sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _validateSortPanel: function() {
        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Get sort column menus.
        var primarySortColumnMenu = this._getSelectElement(
            this.sortColumnMenuIds[this.PRIMARY]);
        var secondarySortColumnMenu = this._getSelectElement(
            this.sortColumnMenuIds[this.SECONDARY]);
        var tertiarySortColumnMenu = this._getSelectElement(
            this.sortColumnMenuIds[this.TERTIARY]);

        // Test primary and secondary menu selections.
        if (primarySortColumnMenu != null && secondarySortColumnMenu != null) {
            // Test if secondary sort is set, but primary is not.
            if (primarySortColumnMenu.selectedIndex < 1 
                    && secondarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (primarySortColumnMenu.selectedIndex > 0
                  && primarySortColumnMenu.selectedIndex == secondarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }

        // Test primary and tertiary menu selections.
        if (primarySortColumnMenu != null && tertiarySortColumnMenu != null) {
            // Test if tertiary sort is set, but primary is not.
            if (primarySortColumnMenu.selectedIndex < 1 
                    && tertiarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (primarySortColumnMenu.selectedIndex > 0
                    && primarySortColumnMenu.selectedIndex == tertiarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }

        // Test secondary and tertiary menu selections.
        if (secondarySortColumnMenu != null && tertiarySortColumnMenu != null) {
            // Test if tertiary sort is set, but secondary is not.
            if (secondarySortColumnMenu.selectedIndex < 1 
                    && tertiarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (secondarySortColumnMenu.selectedIndex > 0
                    && secondarySortColumnMenu.selectedIndex == tertiarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }
        return true;
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.tree");

woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.theme.common");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/** 
 * @class This class contains functions for tree components.
 * @static
 * @private
 */ 
woodstock4_3._html.tree = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize tree.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

	// Set functions.
        domNode.clearAllHighlight = woodstock4_3._html.tree.clearAllHighlight;
        domNode.clearHighlight = woodstock4_3._html.tree.clearHighlight;
        domNode.expandCollapse = woodstock4_3._html.tree.expandCollapse;
        domNode.expandTurner = woodstock4_3._html.tree.expandTurner;
        domNode.findContainingTreeNode = woodstock4_3._html.tree.findContainingTreeNode;
        domNode.findNodeByTypeAndProp = woodstock4_3._html.tree.findNodeByTypeAndProp;
        domNode.getCookieValue = woodstock4_3._html.tree.getCookieValue;
        domNode.getHighlightTreeBgColor = woodstock4_3._html.tree.getHighlightTreeBgColor;
        domNode.getHighlightTreeTextColor = woodstock4_3._html.tree.getHighlightTreeTextColor;
        domNode.getNormalTreeTextColor = woodstock4_3._html.tree.getNormalTreeTextColor;
        domNode.getParentTreeNode = woodstock4_3._html.tree.getParentTreeNode;
        domNode.getSelectedTreeNode = woodstock4_3._html.tree.getSelectedTreeNode;
        domNode.getTree = woodstock4_3._html.tree.getTree;
        domNode.highlight = woodstock4_3._html.tree.highlight;
        domNode.highlightParent = woodstock4_3._html.tree.highlightParent;
        domNode.isAnHref = woodstock4_3._html.tree.isAnHref;
        domNode.isTreeHandle = woodstock4_3._html.tree.isTreeHandle;
        domNode.onTreeNodeClick = woodstock4_3._html.tree.onTreeNodeClick;
        domNode.selectTreeNode = woodstock4_3._html.tree.selectTreeNode;
        domNode.setCookieValue = woodstock4_3._html.tree.setCookieValue;
        domNode.treecontent_submit = woodstock4_3._html.tree.treecontent_submit;
        domNode.treeNodeIsExpanded = woodstock4_3._html.tree.treeNodeIsExpanded;
        domNode.unhighlightParent = woodstock4_3._html.tree.unhighlightParent;
        domNode.updateHighlight = woodstock4_3._html.tree.updateHighlight;

        return true;
    },

    /**
     * Set cookie.
     *
     * @param {String} cookieName
     * @param {String} val
     * @return {boolean} true if successful; otherwise, false.
     */
    setCookieValue: function(cookieName, val) {

	/*
	Fix for bug 6476859 :On initial visit to page, 
        getting selected node returns value from previous 
	session.
	The cookie value should be stored with the global path 
        so that it is applicable for all pages on site.
	*/

	document.cookie = cookieName + "=" + val +";path=/;";
        return true;
    },

    /**
     * Get cookie.
     *
     * @param {String} cookieName
     * @return {String} The cookie value.
     */
    getCookieValue: function(cookieName) {
        var docCookie = document.cookie;
        var pos= docCookie.indexOf(cookieName+"=");

        if (pos == -1) {
            return null;
        }

        var start = pos+cookieName.length+1;
        var end = docCookie.indexOf(";", start );

        if (end == -1) {
            end= docCookie.length;
        }
        return docCookie.substring(start, end);
    },

    /**
     * This function expands or collapses the given tree node.  It expects the
     * source of the given event object (if supplied) to be a tree handle
     * image.  It will change this image to point in the correct direction
     * (right or down).  This implementation depends on the tree handle image
     * names including "tree_handleright" and "tree_handledown" in them.
     * Swapping "right" and "down" in these names must change the handle
     * direction to right and down respectively.
     *
     * @param {Node} treeNode
     * @param {String} imageId
     * @param {Event} event
     * @return {boolean} true if successful; otherwise, false.
     */
    expandCollapse: function(treeNode, imageId, event) {
        var tree = this.getTree(treeNode);
        var childNodes = document.getElementById(treeNode.id+"_children");
        if (childNodes) {
            // Get the event source
            if (!event) {
                event = window.event;
            }
            
            var elt = document.getElementById(imageId);
            
            // get the image widget to compare the actual icon values
            // as opposed to checking the name of the final rendered image.
            
            var imgWidget = woodstock4_3.widget.common.getWidget(imageId);
            var imgProps = imgWidget.getProps();
            var nodeIcon = imgProps.icon;
            
            // First, unhighlight the parent if applicable
            this.unhighlightParent(this.getSelectedTreeNode(tree.id));

            // Change the style to cause the expand / collapse & switch the image
            var display = childNodes.style.display;
            
            if (display == "none") {
                childNodes.style.display = "block";
                if (nodeIcon == "TREE_HANDLE_RIGHT_MIDDLE") {
                    nodeIcon = "TREE_HANDLE_DOWN_MIDDLE";    
                } else if (nodeIcon == "TREE_HANDLE_RIGHT_TOP") {
                    nodeIcon = "TREE_HANDLE_DOWN_TOP";
                } else if (nodeIcon == "TREE_HANDLE_RIGHT_LAST") {
                    nodeIcon = "TREE_HANDLE_DOWN_LAST";
                } else if (nodeIcon == "TREE_HANDLE_RIGHT_TOP_NOSIBLING") {
                    nodeIcon = "TREE_HANDLE_DOWN_TOP_NOSIBLING";
                }
            } else {
                childNodes.style.display = "none";
                if (nodeIcon == "TREE_HANDLE_DOWN_MIDDLE" ) {
                    nodeIcon = "TREE_HANDLE_RIGHT_MIDDLE";    
                } else if (nodeIcon == "TREE_HANDLE_DOWN_TOP" ) {
                    nodeIcon = "TREE_HANDLE_RIGHT_TOP";
                } else if (nodeIcon == "TREE_HANDLE_DOWN_LAST" ) {
                    nodeIcon = "TREE_HANDLE_RIGHT_LAST";
                } else if (nodeIcon == "TREE_HANDLE_DOWN_TOP_NOSIBLING") {
                    nodeIcon = "TREE_HANDLE_RIGHT_TOP_NOSIBLING" ;
                }
            }
            // update the image property to reflect the icon change
            imgWidget.setProps({icon: nodeIcon});

            // Last, update the visible parent of the selected node if now hidden
            this.highlightParent(this.getSelectedTreeNode(tree.id));
        }
        return true;
    },

    /**
     * This function returns the Tree for the given TreeNode.  From
     * a DOM point of view, the tree directly contains all its children
     * (excluding *_children div tags.  This will return the first
     * parentNode that is a div w/ an id != "*_children".
     *
     * @param {Node} treeNode
     * @return {Node} The tree for the given TreeNode.
     */
    getTree: function(treeNode) {
        var tree = treeNode.parentNode;
        var SUFFIX = new String("_children");

        while (tree) {
            // Ignore all div's ending w/ SUFFIX
            if ((tree.nodeName == "DIV")
                    && (tree.id.substr(tree.id.length-SUFFIX.length) != SUFFIX)) {
		break;
            }
            tree = tree.parentNode;
        }
        return tree;
    },

    /**
     * This method handles TreeNode onClick events.  It takes the TreeNode
     * &lt;div&gt; object that was clicked on in order to process the
     * highlighting changes that are necessary.  This object may be obtained by
     * calling <code>getElementById("&lt;TreeNode.getClientId()&gt;")</code>.
     * If this function is invoked from the TreeNode &lt;div&gt; object itself
     * (as is the case when this method is implicitly called), the TreeNode
     * object is simply the <code>this</code> variable.
     *
     * @param {Node} treeNode
     * @param {String} imageId
     * @param {Event} event
     * @return {boolean} true if successful; otherwise, false.
     */
    onTreeNodeClick: function(treeNode, imageId, event) {
        // Check for Tree Handles
        // The handle image and its surrounding area represents the 
        // handler section of the tree node. Clicking on this area
        // alone should cause the node to toggle.
        if (this.isTreeHandle(event)) {
            this.expandCollapse(treeNode, imageId, event);
            return true;
        }

        // Make sure they clicked on an href
        if (!this.isAnHref(event)) {
            // Do nothing
            return true;
        }

        // If we're here, we should select the TreeNode
        return this.selectTreeNode(treeNode.id);
    },

    /**
     * This function may be used to select the given TreeNode. 
     * It will clear the previous TreeNode and select the 
     * given one. The parameter passed to the function is
     * the id of the currently selected treeNode - the
     * value of the tree component from a JSF perspective.
     * This node may not be availbe in the DOM tree as the
     * user may have clicked on the expanded parent node
     * and closed it. In this case the parent node needs
     * to show up in bold.
     *
     * @param {String} treeNodeId
     * @return {boolean} true if successful; otherwise, false.
     */
    selectTreeNode: function(treeNodeId) {
        // Find the top of the tree
	var treeNode = document.getElementById(treeNodeId);

	if (treeNode) {
            var tree = this.getTree(treeNode);

            // Clear the old highlighting
            this.clearAllHighlight(tree.id);

            // Mark the node as selected
            this.setCookieValue(tree.id+"-hi", treeNode.id);

            // first highlight is as a parent
            // when the left frame loads the nodes will 
            // be hightlighted correctly
            this.highlightParent(treeNode);
            this.highlight(treeNode);

            // onClick handler should proceed with hyperlink
            return true;
	} else {
	    // get the parent node ID and highlight the parent.
	    // var arr = treeNodeId.split(":");
	    // var lastBitOfId = arr[arr.length - 1].length;
	    // var lastIndex = treeNodeId.length - lastBitOfId.length;
	    var lastIndex = treeNodeId.lastIndexOf(":");
	    parentNodeId = treeNodeId.substr(0, lastIndex);
	    var parentNode = 
		document.getElementById(parentNodeId);
	    if (parentNode) {	
	        parentNode.style.fontWeight = "bold";
	    }
	    return false;
	}
    },

    /**
     * This function returns the selected TreeNode given the treeId of the
     * Tree.
     *
     * @param {String} treeId
     * @return {Node}The selected TreeNode.
     */
    getSelectedTreeNode: function(treeId) {
        var id = this.getCookieValue(treeId+"-hi");
        if (id) {
            return document.getElementById(id);
        }
        return null;
    },

    /**
     * Clear all highlighted nodes.
     *
     * @param {String} cookieId
     * @return {boolean} true if successful; otherwise, false.
     */
    clearAllHighlight: function(cookieId) {
        // Clear
        var selectedNode = this.getSelectedTreeNode(cookieId);
        this.clearHighlight(selectedNode);
        this.setCookieValue(cookieId+"-hi", "");

        // FIXME: Fix this...
        // this.clearHighlight(document.getElementById(currentHighlightParent));
        return true;
    },

    /**
     * Clear highlighted node
     *
     * @param {Node} node
     * @return {boolean} true if successful; otherwise, false.
     */
    clearHighlight: function(node) {
        if (node) {
	    node.className = 
                woodstock4_3.theme.common.getClassName("TREE_ROW");
        }
        return true;
    },

    /**
     * This function determines if the event source was a tree handle image.
     * This implementation depends on the tree handle image file name
     * containing "tree_handle" and no other images containing this
     * string.
     *
     * @param {Event} event
     * @return {boolean} true if event was generated by tree handle.
     */
    isTreeHandle: function(event) {
        if (!event) {
            event = window.event;
            if (!event) {
                return false;
            }
        }
        var elt = (event.target) ? event.target : event.srcElement;

        // Ignore Tree Handles b/c they should not update highlighting
        
        if (elt.nodeName == "IMG") {
            var imgWidget = woodstock4_3.widget.common.getWidget(elt.id);
            var imgProps = imgWidget.getProps();
            var nodeIcon = imgProps.icon;
            if (nodeIcon.indexOf("TREE_HANDLE_") != -1) {
                return true;
            }
        } else if (elt.nodeName == "A") {
            // User might have been pressing enter around an image.
            // Note: I have never managed to get control to come here.
            
            aID = elt.id;
            var lastIndex = aID.lastIndexOf("_handle");
            if (lastIndex == -1) {
                return false;
            }
            var result = aID.substring(lastIndex, aID.length - 1);
            if (result == "_handle") {
                return true; 
            }
        }
        // Not a tree handle
        return false;
    },

    /**
     * This method checks to see if the event.target is an href, or if any of
     * the parent nodes which contain it is an href.  To be an href, it must be
     * an "A" tag with an "href" attribute containing atleast 4 characters.
     * (Note: Browsers will add on the protocol if you supply a relative URL
     * such as one starting with a '#', '/', or filename).
     *
     * @param {Event} event
     * @return {boolean} true if event was generated by a link.
     */
    isAnHref: function(event) {
        if (!event) {
            event = window.event;
            if (!event) {
                return false;
            }
        }
        var elt = (event.target) ? event.target : event.srcElement;

        // Look for parent href
        while (elt != null) {
            if (elt.nodeName == "A") {
                // Creates a String containing the url
                var url = new String(elt);
                if (url.length > 4) {
                    // All URLs are atleast this long
                    return true;
                }
            }
            elt = elt.parentNode;
        }
        // Not an href
        return false;
    },

    /**
     * This function updates the highlighting for the given Tree client id.
     * This function provides a way to restore the highlighting when a Tree is
     * reloaded in a window (necessary each page load).
     *
     * @param {String} cookieId
     * @return {boolean} true if successful; otherwise, false.
     */
    updateHighlight: function(cookieId) {
        var selNode = this.getSelectedTreeNode(cookieId);
        this.highlight(selNode);

        // FIXME: This doesn't work if the TreeNode element doesn't exist 
        // (which is the case for the server-side tree)
        return this.highlightParent(selNode);
    },

    /**
     * This function highlights the given <code>TreeNode</code>.  The
     * <code>obj</code> passed in is actually the &lt;div&gt; around the html
     * for the <code>TreeNode</code> and may be obtained by calling
     * <code>getElementById("&lt;TreeNode.getClidentId()&gt;")</code>.
     *
     * @param {Node} node
     * @return {boolean} true if successful; otherwise, false.
     */
    highlight: function(node) {
        if (node) {
	    node.className = 
                woodstock4_3.theme.common.getClassName("TREE_SELECTED_ROW");
            return true;
        }
        return false;
    },

    /**
     * This function finds the handler image ICON associated with a given 
     * tree node. The ICON value is used to identify if the node in 
     * question is expanded or not. This is a private function and should
     * not be used by developers on the client side.
     *
     * @param {Node} node
     * @return {boolean} true if the node has an image whose ICON 
     *        indicates the node is expanded.
     */
    findNodeByTypeAndProp: function(node) {
        if (node == null) {        
            return true;
        }
        // First check to see if node is a handler image.
        // Then check if it is of the right type. "RIGHT" icon
        // type indicates the node is not expanded.
        if (node.nodeName == "IMG") {
        
            var imgWidget = woodstock4_3.widget.common.getWidget(node.id);
            var imgProps = imgWidget.getProps();
            var nodeIcon = imgProps.icon;

            if ((nodeIcon == "TREE_HANDLE_RIGHT_MIDDLE") ||
                (nodeIcon == "TREE_HANDLE_RIGHT_TOP") ||
                (nodeIcon == "TREE_HANDLE_RIGHT_LAST") ||
                (nodeIcon == "TREE_HANDLE_RIGHT_TOP_NOSIBLING")) {
                
                return false;
                
            } else if ((nodeIcon == "TREE_HANDLE_DOWN_MIDDLE") ||
                (nodeIcon == "TREE_HANDLE_DOWN_TOP") ||
                (nodeIcon == "TREE_HANDLE_DOWN_LAST") ||
                (nodeIcon == "TREE_HANDLE_DOWN_TOP_NOSIBLING")) {
                
                return true;
            }
        }        
        // Not what we want, walk its children if any
        // Return true for when null conditions arise.
        var nodeList = node.childNodes;
        if (!nodeList || (nodeList.length == 0)) {
            return true;
        }
        var result;
        for (var count = 0; count<nodeList.length; count++) {
            // Recurse
            result = this.findNodeByTypeAndProp(nodeList[count]);
            if (result) {
                // Propagate the result
                return result;
            }
        }
        // Not found
        return true;
    },

    /**
     * This function determines if the given TreeNode is expanded.  It returns
     * <code>true</code> if it is, <code>false</code> otherwise.
     *
     * @param {Node} treeNode
     * @return {boolean} true if TreeNode is expanded.
     */
    treeNodeIsExpanded: function(treeNode) {
        // Find the div containing the handler images for this TreeNode row
        // and pass it to a function that looks for the right image within
        // the div and returns true if the image has been found and is of the
        // type that indicates the node is expanded, false otherwise.
        var node = document.getElementById(treeNode.id + "LineImages");
        return this.findNodeByTypeAndProp(node);
        
    },

    /**
     * This function returns the parent TreeNode of the given TreeNode.
     *
     * @param {Node} treeNode
     * @return {Node} The parent TreeNode.
     */
    getParentTreeNode: function(treeNode) {
        // Get the parent id
        var parentId = treeNode.parentNode.id;
        var childrenIdx = parentId.indexOf("_children");
        if (childrenIdx == -1) {
            return null;
        }
        // This is really a peer div id to what we really want... remove _children
        parentId = parentId.substring(0, childrenIdx);
        // Return the parent TreeNode
        return document.getElementById(parentId);
    },

    /**
     * Unhighlight parent node.
     *
     * @param {Node} childNode
     * @return {boolean} true if successful; otherwise, false.
     */
    unhighlightParent: function(childNode) {
        if (!childNode) {
            return false;
        }

        // First find the parent node and make sure it is collapsed (we don't
        // highlight parent nodes when the selected node is visible)
        var parentNode = this.getParentTreeNode(childNode);
        var highlight = null;
        while (parentNode != null) {
            if (!this.treeNodeIsExpanded(parentNode)) {
                highlight = parentNode;
            }
            parentNode = this.getParentTreeNode(parentNode);
        }
        if (highlight) {
            highlight.style.fontWeight = "normal";
        }
        return true;
    },

    /**
     * Highlight parent node.
     *
     * @param {Node} childNode
     * @return {boolean} true if successful; otherwise, false.
     */
    highlightParent: function(childNode) {
        if (!childNode) {
            return false;
        }

        // First find the parent node and make sure it is collapsed (we don't
        // highlight parent nodes when the selected node is visible)
        var parentNode = this.getParentTreeNode(childNode);
        var highlight = null;
        while (parentNode != null) {
            if (!this.treeNodeIsExpanded(parentNode)) {
                highlight = parentNode;
            }
            parentNode = this.getParentTreeNode(parentNode);
        }
        if (highlight) {
            highlight.style.fontWeight = "bold";
        }
        return true;
    },

    /**
     * Get normal tree text color.
     *
     * @return {String} The text color.
     */
    getNormalTreeTextColor: function() {
        return "#003399";
    },

    /**
     * Get highlight background color.
     *
     * @return {String} The background color.
     */
    getHighlightTreeBgColor: function() {
        return "#CBDCAF";  // ~greenish color
    },

    /**
     * Get highlight tree text color.
     *
     * @return {String} The text color.
     */
    getHighlightTreeTextColor: function() {
        return "#000000";  // black
    },

    /**
     * Returns the TreeNode that contains the given link. This assumes the link
     * is a direct child of the node.
     *
     * @param {Node} link
     * @return {Node} The TreeNode containing the given link.
     */
    findContainingTreeNode: function(link) {     
        var linkId = link.id;
        var nodeId = linkId.substring(0, linkId.lastIndexOf(":"));
        return document.getElementById(nodeId);
    },

    /**
     * If the Tree's expandOnSelect property is true, this method is called to 
     * expand the turner of the tree node with the given labelLink.
     *
     * @param {Node} labelLink
     * @param {String} turnerId
     * @param {String} imageId
     * @param {boolean} isClientSide
     * @param {Event} event
     * @return {boolean} true if successful; otherwise, false.
     */
    expandTurner: function(labelLink, turnerId, imageId, isClientSide, event) {
        var labelLinkId = labelLink.id;
	var formId = labelLinkId.substring(0, labelLinkId.indexOf(":"));
	var node = this.findContainingTreeNode(labelLink);
	var turnerLink = document.getElementById(turnerId); 

	if (turnerLink == null) {
            return false;
	}
	if (!this.treeNodeIsExpanded(node)) {
            // folder is currently closed, expand it
            if (isClientSide) {
		this.expandCollapse(node, imageId, event);      
            } else {
		turnerLink.onclick();
            }    
        }
        return true;
    },

    /**
     * When the tree node link has an associated action, this method should
     * be called to ensure selection highlighting and (if necessary) node 
     * expansion occurs.
     * <p>
     * If the developer specifies the content facet for a given TreeNode, he 
     * should call this function from his facet hyperlink's onClick.
     * </p>
     *
     * @param {String} nodeId
     * @return {boolean} true if successful; otherwise, false.
     */
    treecontent_submit: function(nodeId) {
	if (nodeId == null) {
            return false;
        }
	var node = document.getElementById(nodeId);
	var tree = this.getTree(node);

	// update the current selection
	this.selectTreeNode(node.id);

	// set a cookie that the Tree's decode method will 
	// inspect and expand the corresponding node if necessary
	return this.setCookieValue(tree.id + "-expand", nodeId);
    }
};


woodstock4_3._dojo.provide("woodstock4_3._html.upload");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.common");

/**
 * @class This class contains functions for upload components.
 * @static
 * @private
 */
woodstock4_3._html.upload = {
    /**
     * Use this function to get the HTML input element associated with the
     * Upload component.  
     * @param {String} elementId The client id of the Upload component
     * @return {Node} the input element associated with the Upload component
     * else null if elementId is null or "".
     */
    getInputElement: function(elementId) { 
        if (elementId == null || elementId == "") {
	    return null;
	}

	// The upload component MUST always render the input element
	// with the following suffix on the id 
	// "_com.sun.webui.jsf.upload".
	// This "binds" this version of the component to this theme
	// version.
	// This will change when "field" becomes a widget.
	//
        var element = document.getElementById(elementId + 
            "_com.sun.webui.jsf.upload");
        if (element && element.tagName == "INPUT") { 
            return element; 
        } else {
	    return null;
	}
    },

    /**
     * Create a hidden field with id "preservePathId" and add a listener
     * to the upload's input element, "uploadId". The listener is
     * is added for the onchange event of the upload's input field,
     * see preservePathListener.
     *
     * @param {String} uploadId The client id of the upload component.
     * @param {String} preservePathId
     * @return {boolean} true if the hidden element is created and a listener is
     * added, else false.
     * @private
     */
    _preservePath: function(uploadId, preservePathId) {
	if (uploadId == null || uploadId == "" ||
		preservePathId == null || preservePathId == "") {
	    return false;
	}

	// If there is no upload component, don't do anything.
	// I'm not sure if there is a timing issue here.
	//
	var uploadElement = woodstock4_3._html.upload.getInputElement(uploadId);
	if (uploadElement == null) {
	    return false;
	}
	var theForm = uploadElement.form;

	// Create the change listener.
	// The event target/srcElement is the upload input element
	// its value is the changed value, save it in the 
	// preservePath hidden field.
	//
	var onChangeListener = function(evt) {
	    // Is IE
	    if (document.attachEvent) {
		node = evt.srcElement;
	    } else {
		node = evt.target;
	    }
	    // node is the upload input element
	    //
	    var preservePath = null;
	    try {
		preservePath = theForm.elements[preservePathId];
	    } catch (e) {
	    }

	    // If the hidden field isn't there create it and assign
	    // the node's value
	    //
	    if (preservePath != null) {
		preservePath.value = node.value;
	    } else {
		woodstock4_3._base.common._insertHiddenField(preservePathId, 
                    node.value, theForm);
	    }
	    return true;
	};

	if (uploadElement.addEventListener) {
	    uploadElement.addEventListener('change', onChangeListener, true);
	} else {
	    uploadElement.attachEvent('onchange', onChangeListener);
	}
	return true;
    },

    /**
     * Use this function to disable or enable a upload. As a side effect
     * changes the style used to render the upload. 
     *
     * @param {String} elementId The client id of the upload component.
     * @param {boolean} disabled true to disable the upload, false to enable the upload
     * @return {boolean} true if successful; otherwise, false.
     */
    setDisabled: function(elementId, disabled) {
        if (elementId == null || elementId == "" || 
		disabled == null || disabled == "") {
            // must supply an elementId && state
            return false;
        }
        var input = woodstock4_3._html.upload.getInputElement(elementId); 
        if (input == null) {
            // specified elementId not found
            return false;
        }
        input.disabled = disabled;
	return true;
    },

    /**
     * Set the encoding type of the form to "multipart/form-data".
     * 
     * @param {String} elementId The client id of the upload component.
     * @return {boolean} true if encoding type can be set, else false.
     * @private
     */
    _setEncodingType: function(elementId) { 
	if (elementId == null || elementId == "") {
	    return false;
	}

        var upload = woodstock4_3._html.upload.getInputElement(elementId); 
        var form = upload != null ? upload.form : null;
	if (form != null) {
            // form.enctype does not work for IE, but works Safari
            // form.encoding works on both IE and Firefox
	    //
            if (woodstock4_3._base.browser._isSafari()) {
                form.enctype = "multipart/form-data";
            } else {
                form.encoding = "multipart/form-data";
            }
	    return true;
        }
	return false;
    }
};

// Extend for backward compatibility with JSF based components.
woodstock4_3.upload = woodstock4_3._html.upload;


woodstock4_3._dojo.provide("woodstock4_3._html.wizard");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3._base.proto");

/** 
 * @class This class contains functions for wizard components.
 * <p>
 * The wizard JavaScript object is accessed using the getElementById()
 * function. Methods defined on that javascript object instance maybe
 * called using that identifier. For example, the following javascript
 * could be used to close and forward to a page when the wizard closes.
 * </p><p><code>
 *   <ui:wizard ...other attributes... 
 *	onPopupDismiss="document.getElementById('form1:wizard1').closeAndForward('launchform', '/faces/wizardData.jsp', true);" >
 *
 *	...wizard step tags...
 *
 *   </ui:wizard>
 * </code></p>
 * @static
 * @private
 */
woodstock4_3._html.wizard = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize wizard.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        woodstock4_3._base.proto._extend(domNode, props, false);

        // Set functions.
        domNode.nextClicked = woodstock4_3._html.wizard.nextClicked;
        domNode.previousClicked = woodstock4_3._html.wizard.previousClicked;
        domNode.cancelClicked = woodstock4_3._html.wizard.cancelClicked;
        domNode.finishClicked = woodstock4_3._html.wizard.finishClicked;
        domNode.closeClicked = woodstock4_3._html.wizard.closeClicked;
        domNode.gotoStepClicked = woodstock4_3._html.wizard.gotoStepClicked;
        domNode.closePopup = woodstock4_3._html.wizard.closePopup;
        domNode.closeAndForward = woodstock4_3._html.wizard.closeAndForward;
        domNode.wizOnLoad = woodstock4_3._html.wizard.wizOnLoad;
        domNode.resize_hack = woodstock4_3._html.wizard.resize_hack;

        return true;
    },

    /**
     * @ignore
     */
    nextClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    previousClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    cancelClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    closeClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    finishClicked: function() {
        return true;
    },

    /**
     * @ignore
     */
    gotoStepClicked: function() {
        return true;
    },

    /**
     * Close popup.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    closePopup: function() {
        window.close();
        return true;
    },

    /**
     * Close the wizard popup and forward to "submitPage" by submitting
     * "submitForm".
     * <p>
     * When the wizard closes it is often necessary to send
     * a request to a page that will make use of the data collected
     * during a wizard session. This method does this by obtaining the
     * form element "submitForm" from the page that launched the
     * the wizard. This means that the page that launched the wizard
     * popup must still be visible in a browser window. If that form
     * is found, the "action" property is set to "submitPage" and the
     * "submit" method of that "submitForm" is executed.
     * The popup window is then closed. 
     * </p><p>
     * However due to JSF's client side state saving mode an extra step
     * must be taken. If the application is operating with client side
     * state saving, JSF will ignore the "submitPage" value of the
     * submitted form's "action" property and will send the request to the
     * view defined in the saved state, saved in an element in "submitForm".
     * </p><p>
     * If the application is configured for client side state saving and
     * the "submitPage" is different from the page that lauched the wizard,
     * set "cleartState" to true. This method will clear the saved state 
     * before submitting the form. The "clearState" default value is false
     * and the saved state will not be cleared.
     * </p><p>
     * The "clearState" functionality only works with Sun's RI.
     * </p>
     *
     * @param {boolean} submitForm
     * @param {boolean} submitPage
     * @param {boolean} clearState
     * @return {boolean} true if successful; otherwise, false.
     */
    closeAndForward: function(submitForm, submitPage, clearState) {
        var f = window.opener.document.getElementById(submitForm);
        if (f == null) {
            console.debug("Can't find form " + submitForm); // See Firebug console.
            window.close();
        }

        if (clearState != null && clearState == true) {
            var elements = f.elements;
            var clientstate = null;
            for (var i = 0; i < elements.length; ++i) {
                // This only works for the Sun RI and is
                // dependent on the RIConstants.FACES_VIEW value
                // of "com.sun.faces.VIEW"
                //
                if (elements[i].name == this.facesViewState) {
                    clientstate = elements[i];
                    break;
                }
            }
            if (clientstate != null) {
                f.removeChild(clientstate);
            }
        }

        f.action = submitPage;
        f.submit();
        window.close();
        return true;
    }, 

    /**
     * This method must be assigned to the onload handler of the onLoad
     * attribute of the ui:body tag if the wizard is to operate properly on IE.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    wizOnLoad: function() {
        var stepsid = this.id + "_stepspane";
        var helpid = this.id + "_helppane";
        var wizbdyid = this.id + "_WizBdy";
        return this.resize_hack(helpid, stepsid, wizbdyid);
    },

    /**
     * used only for popup window and IE, and called by wizOnLoad.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    resize_hack: function(helpid, stepsid, wizbdyid) {
        if (woodstock4_3._base.browser._isIe5up()) {

            var bdy = document.getElementById(wizbdyid);
            if (bdy != null) {
		var newheight = document.documentElement.clientHeight;
                bdy.style.height = newheight - 145;
                if (helpid != null && helpid != '') {
                    var help = document.getElementById(helpid);
                    if (help != null) {
                        help.style.height = newheight - 90;
                    }
                }
                if (stepsid != null && stepsid != '') {
                    var steps = document.getElementById(stepsid);
                    if (steps != null) {
                        steps.style.height = newheight - 90;
                    }
                }
            }
        }
        return true;
    }
};


woodstock4_3._dojo.provide("woodstock4_3.formElements");

woodstock4_3._dojo.require("woodstock4_3._html.button");
woodstock4_3._dojo.require("woodstock4_3._html.checkbox");
woodstock4_3._dojo.require("woodstock4_3._html.dropDown");
woodstock4_3._dojo.require("woodstock4_3._html.field");
woodstock4_3._dojo.require("woodstock4_3._html.hyperlink");
woodstock4_3._dojo.require("woodstock4_3._html.jumpDropDown");
woodstock4_3._dojo.require("woodstock4_3._html.listbox");
woodstock4_3._dojo.require("woodstock4_3._html.rbcb");
woodstock4_3._dojo.require("woodstock4_3._html.radiobutton");


woodstock4_3._dojo.provide("woodstock4_3.json");

//
// Copyright (c) 2005 JSON.org
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The Software shall be used for Good, not Evil.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//

/**
 * @class This class contains functions for parsing JSON.
 * @static
 */
woodstock4_3.json = {
    /**
     * JSON escape chars.
     * @private
     */
    _m: {
        '\b': '\\b',
        '\t': '\\t',
        '\n': '\\n',
        '\f': '\\f',
        '\r': '\\r',
        '"' : '\\"',
        '\\': '\\\\'
    },

    /**
     * JSON parsor.
     * @private
     */
    _s: {
        /** @ignore */
        'boolean': function (x) {
            return String(x);
        },
        /** @ignore */
        number: function (x) {
            return isFinite(x) ? String(x) : null;
        },
        /** @ignore */
        string: function (x) {
            if (/["\\\x00-\x1f]/.test(x)) {
                x = x.replace(/([\x00-\x1f\\"])/g, function(a, b) {
                    var c = woodstock4_3.json._m[b];
                    if (c) {
                        return c;
                    }
                    c = b.charCodeAt();
                    return '\\u00' +
                        Math.floor(c / 16).toString(16) +
                        (c % 16).toString(16);
                });
            }
            return '"' + x + '"';
        },
        /** @ignore */
        object: function (x) {
            if (x) {
                var a = [], b, f, i, l, v;
                if (x instanceof Array) {
                    a[0] = '[';
                    l = x.length;
                    for (i = 0; i < l; i += 1) {
                        v = x[i];
                        f = woodstock4_3.json._s[typeof v];
                        if (f) {
                            v = f(v);
                            if (typeof v == 'string') {
                                if (b) {
                                    a[a.length] = ',';
                                }
                                a[a.length] = v;
                                b = true;
                            }
                        }
                    }
                    a[a.length] = ']';
                } else if (typeof x.hasOwnProperty === 'function') {
                    a[0] = '{';
                    for (i in x) {
                        if (x.hasOwnProperty(i)) {
                            v = x[i];
                            f = woodstock4_3.json._s[typeof v];
                            if (f) {
                                v = f(v);
                                if (typeof v == 'string') {
                                    if (b) {
                                        a[a.length] = ',';
                                    }
                                    a.push(woodstock4_3.json._s.string(i), ':', v);
                                    b = true;
                                }
                            }
                        }
                    }
                    a[a.length] = '}';
                } else {
                    return null;
                }
                return a.join('');
            }
            return null;
        }
    },

    /**
     * Stringify a JavaScript value, producing JSON text. 
     *
     * @param {Object} v A non-cyclical JSON object.
     * @return {boolean} true if successful; otherwise, false.
     */
    stringify: function (v) {
        var f = woodstock4_3.json._s[typeof v];
        if (f) {
            v = f(v);
            if (typeof v == 'string') {
                return v;
            }
        }
        return null;
    },

    /**
     * Parse a JSON text, producing a JavaScript object.
     *
     * @param {String} text The string containing JSON text.
     * @return {boolean} true if successful; otherwise, false.
     */
    parse: function (text) {
        try {
            return !(/[^,:{}\[\]0-9.\-+Eaeflnr-u \n\r\t]/.test(
                text.replace(/"(\\.|[^"\\])*"/g, ''))) && eval('(' + text + ')');
        } catch (e) {
            return false;
        }
    }
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.anchorBase");



woodstock4_3._dojo.provide("woodstock4_3.widget._base.widgetBase");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3._dijit._Templated");
woodstock4_3._dojo.require("woodstock4_3._dijit._Widget"); 
woodstock4_3._dojo.require("woodstock4_3.theme.common");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.widgetBase
 * @class This class contains functions used for base functionality in all 
 * widgets. 
 * <p>
 * The widgetBase class inherits from woodstock4_3._dijit._Widget and 
 * woodstock4_3._dijit._Templated. The woodstock4_3._dijit._Widget class
 * is responsible for calling the _buildRendering() and _postCreate() functions 
 * in that order. The dijit_Templated function overrides the _buildRendering() 
 * functon to fill in template properties.
 * <p></p>
 * The _postCreate() function is responible for initializing CSS selectors, 
 * events, and public functions. Commonly used functions (e.g., getProps(), 
 * setProps(), and refresh() are set on the outermost DOM node via the 
 * "superclass" function of widgetBase. This inherited function is also 
 * responsible for invoking the private _setProps() function. 
 * <p></p>
 * The private _setProps() function is used to set widget properties that can be
 * updated by a web app developer. This helps encapsolate functionality and 
 * brand changes while providing a common API for all widgets. In addition, the 
 * widget is selctively updated (i.e., if and only if a key-value pair has been 
 * given). Saving given properties is deferred to the public setProps() function
 * which allows _setProps() to be used during initialization.
 * <p></p>
 * The private _setProps() function is also responsible for invoking 
 * _setCommonProps() and _setEventProps(). These properties may not always be
 * set on the outermost DOM node; however, core (i.e., id, class, style, etc.) 
 * properties are. Core properties are set on the DOM via the "superclass" 
 * function of widgetBase which invokes the _setCoreProps() function.
 * <p></p>
 * The _getClassName() function is responsible for obtaining the selector that
 * will be set on the outermost DOM node. The private _setProps() function 
 * of widgetBase ensures that the _getClassName() function is called prior to 
 * invoking _setCoreProps(). In most cases, this function will be overridded in
 * order to apply widget specific selectors. However, selectors should be 
 * concatinated in order of precedence (e.g., the user's className property is 
 * always appended last).
 * <p></p>
 * The public setProps() function is responsible for extending the widget class
 * with properties so they can be used during later updates. After extending the
 * widget, the private _setProps() function is called. In some cases, the public
 * setProps() function may be overridden. For example, the label clears the
 * contents property from the widget because that is something we do not want to
 * extend.
 * <p></p>
 * The start() function is typically called after the widget has been 
 * instantiated. For example, a progressBar might start a timer to periodically
 * refresh. 
 * <p></p>
 * Warning: It's not possible to append HTML elements from script that is 
 * not a direct child of the BODY element. If there is any Javascript
 * running inside the body that is a direct child of body, IE will throw
 * an "Internet Explorer cannot open the Internet site" error. For example,
 * woodstock4_3._dijit._Templated._createNodesFromText generates such an 
 * error by calling appendChild(). Therefore, widget creation must be deferred
 * to the window.onLoad event. See http://trac.dojotoolkit.org/ticket/4631
 * </p>
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.widgetBase", [
        woodstock4_3._dijit._Widget, woodstock4_3._dijit._Templated ], {
    // Note: If your class contains arrays or other objects, they should be
    // declared in the constructor function so that each instance gets it's own
    // copy. Simple types (literal strings and numbers) are fine to declare in 
    // the class directly. Also note that superclass constructors are always 
    // called automatically, and always before the subclass constructor.
    constructor: function() {
        this._started = false;
        this._templateType = this._widgetType;
    },

    // Set defaults.
    _common: woodstock4_3._base.common, // Common utils.
    _dojo: woodstock4_3._dojo, // Dojo utils.
    _proto: woodstock4_3._base.proto, // Prototype utils.
    _theme: woodstock4_3.theme.common, // Theme utils.
    _widget: woodstock4_3.widget.common // Widget utils.
});

// This function is not public and should not appear in the jsDoc.
/** @ignore */
woodstock4_3.widget._base.widgetBase.prototype.buildRendering = function () {
    // Template must be set prior to calling "superclass".
    this._buildRendering();
    return this._inherited("buildRendering", arguments);
};

/**
 * This function is used to render the widget from a template.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._buildRendering = function () {
    // Get default template path.
    if (this.templatePath == null) {
        this.templatePath = this._theme._getTemplatePath(this._templateType);
    }

    // Get default template string.
    if (this.templatePath == null && this.templateString == null) {
        var browser = woodstock4_3._base.browser;

        // Load browser specific template, if any.
        if (browser._isFirefox()) {
            this.templateString = this._theme._getTemplateString(
                this._widgetType + "_firefox");
        } else if (browser._isMozilla()) {
            this.templateString = this._theme._getTemplateString(
                this._widgetType + "_mozilla");
        } else if (browser._isIe()) {
            this.templateString = this._theme._getTemplateString(
                this._widgetType + "_ie");
        } else if (browser._isSafari()) {
            this.templateString = this._theme._getTemplateString(
                this._widgetType + "_safari");
        }
        // Get default template.
        if (this.templateString == null) {
            this.templateString = this._theme._getTemplateString(this._templateType);
        }
    }

    // The templatePath should have precedence. Therefore, in order for the 
    // templatePath to be used, templateString must be null.
    if (this.templatePath != null) {
        this.templateString = null;
    }
    return true;
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._getClassName = function() {
    // Save template selector, if any.
    if (typeof this._className == "undefined") {
        this._className = this._domNode.className;
    }
    // Append template selector to given className property.
    var newClassName = this.className;
    if (this.className && this._className) {
        newClassName += " " + this._className;
    } else if (this._className) {
        newClassName = this._className;
    }
    return newClassName;
};

/**
 * This function is used to get common properties from the widget. Please see
 * the _setCommonProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._getCommonProps = function() {
    var props = {};

    // Set properties.
    if (this.accessKey != null) { props.accessKey = this.accessKey; }
    if (this.dir != null) { props.dir = this.dir; }
    if (this.lang != null) { props.lang = this.lang; }
    if (this.tabIndex != null) { props.tabIndex = this.tabIndex; }
    if (this.title != null) { props.title = this.title; }

    return props;
};

/**
 * This function is used to get core properties from the widget. Please see
 * the _setCoreProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._getCoreProps = function() {
    var props = {};

    // Set properties.
    if (this.className != null) { props.className = this.className; }
    if (this.id != null) { props.id = this.id; }
    if (this.style != null) { props.style = this.style; }
    if (this.visible != null) { props.visible = this.visible; }

    return props;
};

/**
 * This function is used to get event properties from the widget. Please
 * see the _setEventProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._getEventProps = function() {
    var props = {};

    // Set properties.
    if (this.onBlur != null) { props.onBlur = this.onBlur; }
    if (this.onChange != null) { props.onChange = this.onChange; }
    if (this.onClick != null) { props.onClick = this.onClick; }
    if (this.onDblClick != null) { props.onDblClick = this.onDblClick; }
    if (this.onFocus != null) { props.onFocus = this.onFocus; }
    if (this.onKeyDown != null) { props.onKeyDown = this.onKeyDown; }
    if (this.onKeyPress != null) { props.onKeyPress = this.onKeyPress; }
    if (this.onKeyUp != null) { props.onKeyUp = this.onKeyUp; }
    if (this.onMouseDown != null) { props.onMouseDown = this.onMouseDown; }
    if (this.onMouseOut != null) { props.onMouseOut = this.onMouseOut; }
    if (this.onMouseOver != null) { props.onMouseOver = this.onMouseOver; }
    if (this.onMouseUp != null) { props.onMouseUp = this.onMouseUp; }
    if (this.onMouseMove != null) { props.onMouseMove = this.onMouseMove; }
    if (this.onSelect != null) { props.onSelect = this.onSelect; }

    return props;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @id woodstock4_3.widget._base.widgetBase.getProps
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.widgetBase.prototype.getProps = function() {
    var props = {};

    // Set properties.
    this._proto._extend(props, this._getCommonProps(), false);
    this._proto._extend(props, this._getCoreProps(), false);
    this._proto._extend(props, this._getEventProps(), false);

    return props;
};

/**
 * The inherited function will climb up the scope chain, from superclass to 
 * superclass and through mixin classes as well, until it finds "someMethod",
 * then it will invoke that method.
 * <p>
 * Note: The argument is always literally arguments, a special Javascript array 
 * variable which holds all the arguments (like argv in C). There are a few 
 * variations to inherited() for special cases. If you have a method that was 
 * put into your object outside of declare, you need to specify the name of the
 * calling function like this:
 * </p><p><code>
 * this.inherited("someMethod", arguments);
 * </code></p><p>
 * You can also send custom parameters to the ancestor function. Just place the
 * extra arguments in array literal notation with brackets:
 * </p><p><code>
 * this.inherited(arguments, [ customArg1, customArg2 ]);
 * </code></p>
 *
 * @param {String} name The name of the inherited function.
 * @param {Object} args The function arguments.
 * @param {Object} newArgs Custom function arguments.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._inherited = function(name, args, newArgs) {
    return this.inherited(name, args, newArgs);
};

/**
 * This function is used to test if widget has been initialized.
 * <p>
 * Note: It is assumed that an HTML element is used as a place holder for the
 * document fragment.
 * </p>
 * @return {boolean} true if widget is initialized.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._isInitialized = function() {
    // Testing if the outermost DOM node has been added to the document and
    // ensuring a Dojo attach point exists works fine for JSP. However, the 
    // following code always returns null for facelets.
    //
    // var domNode = document.getElementById(this.id);
    // if (domNode && domNode.getAttribute("dojoattachpoint")) {
    if (this.initialized == true) {
        return true;
    }
    return false;
};

// This function is not public and should not appear in the jsDoc.
/** @ignore */
woodstock4_3.widget._base.widgetBase.prototype.postCreate = function () {
    return this._postCreate();
};

/**
 * This is called after the _buildRendering() function.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._postCreate = function () {
    // In order to register widgets properly, the DOM node id must be set prior 
    // to creating any widget children. Otherwise, widgets may not be destroyed.
    this._domNode.id = this.id;

    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Set public functions.
    /** @ignore */
    this._domNode.getProps = function() { 
        return woodstock4_3.widget.common.getWidget(_id).getProps();
    };
    /** @ignore */
    this._domNode.setProps = function(props, notify) { 
        return woodstock4_3.widget.common.getWidget(_id).setProps(props, notify);
    };
    /** @ignore */
    this._domNode.subscribe = function(topic, obj, func) {
        return woodstock4_3.widget.common.subscribe(topic, obj, func);
    };

    // Initialize public functions.
    if (typeof this._initRefreshFunc == "function") {
        this._initRefreshFunc();
    }
    if (typeof this._initSubmitFunc == "function") {
        this._initSubmitFunc();
    }

    // Set event topics.
    this._domNode.event = this.event;

    // Set properties.
    this._setProps(this.getProps());

    // All widget properties have been set.
    return this.initialized = true;
};

/**
 * Publish an event topic.
 * <p>
 * Note: In order to obtain Ajax modules dynamically, this function shall be 
 * overridden by a custom AJAX implementation.
 * </p>
 * @param {String} topic The event topic to publish.
 * @param {Object} props Key-Value pairs of properties. This will be applied
 * to each topic subscriber.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._publish = function(topic, props) {
    // Obtain the Ajax module associated with this widget.
    var config = woodstock4_3._base.config;
    if (new Boolean(config.ajax.isAjax).valueOf() == true && config.ajax.module) {
        woodstock4_3._dojo.require(config.ajax.module + "." + this._widgetType);
    }
    return woodstock4_3.widget.common.publish(topic, props);
};

/**
 * This function is used to set common properties for the given domNode.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey Shortcut key.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._setCommonProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }
    if (props.accessKey != null) { 
        domNode.accessKey = props.accessKey;
    }
    if (props.dir != null) {
        domNode.dir = props.dir;
    }
    if (props.lang != null) {
        domNode.lang = props.lang;
    }
    if (props.tabIndex > -1 && props.tabIndex < 32767) {
        domNode.tabIndex = props.tabIndex;
    }
    if (props.title != null) {
        domNode.title = props.title;
    }
    return true;
};

/**
 * This function is used to set core properties for the given domNode. These
 * properties are typically set on the outermost element.
 * <p>
 * Note: The className is typically provided by a web app developer. If 
 * the widget has a default className, it should be added to the DOM node
 * prior to calling this function. For example, the "myCSS" className would
 * be appended to the existing "Tblsun4" className (e.g., "Tbl_sun4 myCSS").
 * </p>
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} style Specify style rules inline.
 * @config {boolean} visible Hide or show element.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._setCoreProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }
    if (props.className != null) {
        domNode.className = props.className;
    }
    if (props.id != null) { 
        domNode.id = props.id;
    }
    if (props.style != null) { 
        domNode.style.cssText = props.style;
    }
    if (props.visible != null) {
        this._common._setVisibleElement(domNode, 
            new Boolean(props.visible).valueOf());
    }
    return true;
};

/**
 * This function is used to set event properties for the given domNode.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Element value changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Element text selected.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._setEventProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }

    // Note: JSON strings are not recognized as JavaScript. In order for
    // events to work properly, an anonymous function must be created.
    if (props.onBlur != null) { 
        domNode.onblur = (typeof props.onBlur == 'string')
            ? new Function("event", props.onBlur)
            : props.onBlur;
    }
    if (props.onClick != null) {
        domNode.onclick = (typeof props.onClick == 'string')
            ? new Function("event", props.onClick)
            : props.onClick;
    }
    if (props.onChange != null) {
        domNode.onchange = (typeof props.onChange == 'string')
            ? new Function("event", props.onChange)
            : props.onChange;
    }
    if (props.onDblClick != null) {
        domNode.ondblclick = (typeof props.onDblClick == 'string')
            ? new Function("event", props.onDblClick)
            : props.onDblClick;
    }
    if (props.onFocus != null) {
        domNode.onfocus = (typeof props.onFocus == 'string')
            ? new Function("event", props.onFocus)
            : props.onFocus;
    }
    if (props.onKeyDown != null) {
        domNode.onkeydown = (typeof props.onKeyDown == 'string')
            ? new Function("event", props.onKeyDown)
            : props.onKeyDown;
    }
    if (props.onKeyPress != null) {
        domNode.onkeypress = (typeof props.onKeyPress == 'string')
            ? new Function("event", props.onKeyPress)
            : props.onKeyPress;
    }
    if (props.onKeyUp != null) {
        domNode.onkeyup = (typeof props.onKeyUp == 'string')
            ? new Function("event", props.onKeyUp)
            : props.onKeyUp;
    }
    if (props.onMouseDown != null) {
        domNode.onmousedown = (typeof props.onMouseDown == 'string')
            ? new Function("event", props.onMouseDown)
            : props.onMouseDown;
    }
    if (props.onMouseOut != null) {
        domNode.onmouseout = (typeof props.onMouseOut == 'string')
            ? new Function("event", props.onMouseOut)
            : props.onMouseOut;
    }
    if (props.onMouseOver != null) {
        domNode.onmouseover = (typeof props.onMouseOver == 'string')
            ? new Function("event", props.onMouseOver)
            : props.onMouseOver;
    }
    if (props.onMouseUp != null) {
        domNode.onmouseup = (typeof props.onMouseUp == 'string')
            ? new Function("event", props.onMouseUp)
            : props.onMouseUp;
    }
    if (props.onMouseMove != null) {
        domNode.onmousemove = (typeof props.onMouseMove == 'string')
            ? new Function("event", props.onMouseMove)
            : props.onMouseMove;
    }
    if (props.onSelect != null) {
        domNode.onselect = (typeof props.onSelect == 'string')
            ? new Function("event", props.onSelect)
            : props.onSelect;
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @id woodstock4_3.widget._base.widgetBase.setProps
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.widgetBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Extend widget object for later updates.
    this._proto._extend(this, props);

    // Create a clone which can be safely modified in order to update 
    // subwidgets more efficiently.
    var _props = {};
    this._proto._extend(_props, props);

    // Set properties.
    this._setProps(_props);

    // Notify listeners state has changed.
    if (new Boolean(notify).valueOf() == true &&
            typeof this._stateChanged == "function") {
        this._stateChanged(props);
    }
    return true;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.widgetBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set style class -- must be set before calling _setCoreProps().
    props.className = this._getClassName();
    // The visible selector must be set otherwise, className may wipe it out.
    props.visible = (props.visible != null) ? props.visible : this.visible;

    // Set more properties.
    return this._setCoreProps(this._domNode, props);
};

// This function is not public and should not appear in the jsDoc.
/** @ignore */
woodstock4_3.widget._base.widgetBase.prototype.startup = function () {
    return this._startup();
};

/**
 * This function is used to "start" the widget, after the widget has been
 * instantiated.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.widgetBase.prototype._startup = function () {
    if (this._started == true) {
        return false;
    }
    return this._started = true;
};

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.anchorBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class contains functions for widgets that extend anchorBase.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.anchorBase",
        woodstock4_3.widget._base.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
    }
});

/**
 * Helper function to add children.
 *
 * @param props Key-Value pairs of properties.
 * @config {Array} contents The contents of the anchor body.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.anchorBase.prototype._addContents = function(props) {
    if (props.contents == null) {
        return false;
    }

    // Remove child nodes.
    this._widget._removeChildNodes(this._domNode);

    // Add contents.
    for (i = 0; i < props.contents.length; i++) {
        this._widget._addFragment(this._domNode, props.contents[i], "last");
    }
    return true;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.anchorBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.hrefLang != null) { props.hrefLang = this.hrefLang; }
    if (this.target != null) { props.target = this.target; }
    if (this.type != null) { props.type = this.type; }
    if (this.rev != null) { props.rev = this.rev; }
    if (this.rel != null) { props.rel = this.rel; }
    if (this.shape != null) { props.shape = this.shape; }
    if (this.coords != null) { props.coords = this.coords; }
    if (this.charset != null) { props.charset = this.charset; }
    if (this.accessKey != null) { props.accesskey = this.accessKey; }
    if (this.href != null) { props.href = this.href; }
    if (this.name != null) { props.name = this.name; } 
    if (this.contents != null) { props.contents = this.contents; }
    if (this.disabled != null) { props.disabled = this.disabled; }

    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.anchorBase.prototype._onClickCallback = function(event) {
    if (this.disabled == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the request.
    var result = (this._domNode._onclick)
        ? this._domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.anchorBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.anchorBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Add contents.
    this._addContents(props);

    // Set properties.
    if (props.accessKey != null) { this._domNode.accesskey = props.accessKey; }
    if (props.charset != null) { this._domNode.charset = props.charset; }
    if (props.coords != null) { this._domNode.coords = props.coords; }
        if (props.href) {
            // If context path is provided, then check whether the image has
            // context path already appended and if not, append it.
            if (this.prefix) {
                props.href = 
                    woodstock4_3.widget.common._appendPrefix(this.prefix, props.href);
            }
            this._domNode.href = props.href; 
        }
    if (props.hrefLang != null) { this._domNode.hrefLang =  props.hrefLang; }
    if (props.name != null) { this._domNode.name = props.name; }
    if (props.rev != null) { this._domNode.rev = props.rev; }
    if (props.rel != null) { this._domNode.rel = props.rel; }
    if (props.shape != null) { this._domNode.shape = props.shape; }
    if (props.target != null) { this._domNode.target = props.target; }
    if (props.type != null) { this._domNode.type = props.type; }

    // Set id -- anchors must have the same id and name on IE.
    if (props.name != null) {
        props.id = props.name;
    }

    // A web app devleoper could return false in order to cancel the 
    // submit. Thus, we will handle this event via the onClick call back.
    if (props.onClick) {
        // Set private function scope on DOM node.
        this._domNode._onclick = (typeof props.onClick == 'string')
            ? new Function("event", props.onClick) : props.onClick;

        // Must be cleared before calling _setEventProps() below.
        props.onClick = null;
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.checkedBase");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget.common");


woodstock4_3._dojo.provide("woodstock4_3.widget._base.labeledBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.labeledBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class defines functions and properties 
 * for widgets that have label subcomponents and implements 
 * the <code>required</code> and <code>valid</code> properties which
 * control the indicators on the label.
 *
 * <h3>Dojo attach points</h3>
 * A <code>labeledBase</code> subclass's templates are expected to
 * define following attachpoint identifiers.
 * <ul>
 * <li>_labelContainer - the attachpoint for the label. (mandatory)</li>
 * <li>_brNode - the attachpoint for a <code>br</code> element to
 * implement <code>labelOnTop</code> behavior. (optional)</li>
 * </ul>
 * </p>
 * <h3>The <code>label</code> object property</h3>
 * The <code>label</code> property is an object that defines the properties
 * for a widget that is rendered to represent a label. Minimally, only the
 * <code>value</code> property of the label object property must be non 
 * null for <code>labeledBase</code> to render a label. If only the
 * <code>value</code> property is specified the following default
 * values will be used to create an instance of 
 * <code>woodstock4_3.widget.label</code>.
 * <p>
 * <ul>
 * <li><code>widgetType</code> -
 * <code>woodstock4_3.widget.label</code></li>
 * <li><code>id</code> - this.id + "_label"</li>
 * <li><code>htmlFor</code> - subclasses determine the appropriate value
 * for this property in their implementation of <code>_getLabelProps</code>.
 * </li>
 * </ul>
 * <p>See <code>_postCreate</code> and <code>_getLabelProps</code>
 * </p>
 * <p>
 * This class is meant to be subclassed and not instantiated.
 * Therefore it has no <code>_widgetType</code> property.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} required If true the widget that is labeled requires
 * input, and indicator will appear near the label.
 * @config {boolean} valid If true the widget that is labeled has valid 
 * input. If false an indicator will appear near the label.
 * @config {boolean} labelOnTop If true the label appears above the
 * widget that is labeled. Subclasses may or may not implement this property.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.labeledBase",
        woodstock4_3.widget._base.widgetBase, {
    // Set defaults.
    constructor: function() {
	// If true the label appears above the owning widget.
	this.lableOnTop = false,
	// If true an indicator appears near the label to indicate input
	// for the labeled widget is required.
	this.required = false;
	// If false an indicator appears near the label to indicate input
	// for the labeled widget is invalid.
	this.valid = true;

	// Private flag to remember that last label style class.
	this._lastLabelOnTopClassName =  null;

	// Private flag to remember that last disabled label style class.
	this._lastLabelDisabledClassName =  null;
    }
});

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.labeledBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    if (this.label != null) { props.label = this.label; };
    if (this.labelOnTop != null) { props.labelOnTop = this.labelOnTop; };
    if (this.required != null) { props.required = this.required; };
    if (this.valid != null) { props.valid = this.valid; };

    return props;
};

/**
 * Return a CSS selector to be used for the label widget when the widget
 * is disabled.
 * <p>
 * This implementation returns null. This method should be implemented
 * in subclasses to return the appropriate selector desired by the subclass.
 * </p>
 * @param {boolean} disabled If <code>disabled</code> is true, return the 
 * selector for the label when the widget is "disabled", else return the
 * selector for the label when widget is enabled.
 * @return {String} This implementation returns null;
 */
woodstock4_3.widget._base.labeledBase.prototype._getLabelDisabledClassName = function(disabled) {
    return null;
};

/**
 * Return a CSS selector to be used for the label widget.
 * <p>
 * This implementation returns null. This method should be implemented
 * in subclasses to return the appropriate selector desired by the subclass.
 * </p>
 * @param {boolean} ontop If <code>ontop</code> is true, return the selector
 * for the label when the label is "ontop", else return the selector for the
 * label when it is not "ontop".
 * @return {String} This implementation returns null;
 * @private
 */
woodstock4_3.widget._base.labeledBase.prototype._getLabelOnTopClassName = function(ontop) {
    return null;
};

/**
 * Return label properties desired by this widget.
 * <p>
 * This implementation returns null. This method should be implemented
 * in subclasses to return label properties desired by the subclass.
 * </p>
 * @return {Object} Key-Value pairs of label properties.
 * @private
 */
woodstock4_3.widget._base.labeledBase.prototype._getLabelProps = function() {
    var props = {};
    
    var cn = this._getLabelOnTopClassName(this.labelOnTop);
    if (cn != null) {
	this._common._addStyleClass(props, cn);
    }
    cn = this._getLabelDisabledClassName(this.disabled);
    if (cn != null) {
	this._common._addStyleClass(props, cn);
    }
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.labeledBase.prototype._postCreate = function () {
    // A widget that has inherited from labeledBase must have
    // "this._labelContainer", but check anyway.
    //
    if (!this._labelContainer) {
       return this._inherited("_postCreate", arguments);
    }

    // We should probably set and id anyway, even if it is just "_label".
    //
    if (this.id) {
	this._labelContainer.id = this.id + "_label";
    }

    // If the application is creating a label on construction
    // this.label will be non-null. If it is a widget fragment 
    // (provides a widgetType and an id) then assum it is fully
    // configured. If it is not a fragment then get the desired
    // label properties to create the widget.
    // 
    if (this.label && this.label.value != null
            && !this._widget._isFragment(this.label)) {
	this.label.id = this.id + "_label";
        this.label.widgetType = "label";

	// Get subclass label preferences
	//
	var props = this._getLabelProps();
	this._proto._extend(props, this.label);
	this._proto._extend(this.label, props);
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.labeledBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Always update the _brNode state, even if there isn't a label.
    // If there is no _brNode the subclass does not support labelOnTop.
    //
    if (this._brNode && props.labelOnTop != null) {
	this._common._setVisibleElement(this._brNode, props.labelOnTop);

	// Always remove the last label on top selector.
	// 
	this._common._stripStyleClass(this._labelContainer,
	    this._lastLabelOnTopClassName);

	// Get the label selector from the subclass
	// and remember the new ontop selector.
	//
	this._lastLabelOnTopClassName = 
	    this._getLabelOnTopClassName(props.labelOnTop);

	// Add the "ontop" selector.
	//
	this._common._addStyleClass(this._labelContainer,
	    this._lastLabelOnTopClassName);
    }

    // Cases:
    // - called from _postCreate
    //   + creating a label
    //       this.label and this.label.id will not be null,
    //       props.label will not be null
    //   + no label
    //       this.label and this.label.id and props.label will be null
    //
    // - called from setProps
    //   + creating a label 
    //       same as called from _postCreate
    //   + updating a label (label subcomponent must exist)
    //       props.label and this.label and this.label.id will not be null
    //   + updating the widget's required or valid properties
    //       props.label may be null, this.label and this.label.id will
    //       not be null because there must be can existing label
    //       subcomponent.
    //
    // If the label is being updated then props.label will not be null
    // and hence "this.label" will not be null, but "this.label.id"
    // may be null if there is no existing subcomponent label and props is 
    // "{ label : { value : 'foo'}}" i.e. no id is specified.
    // 
    if (this.label != null && this.label.id != null && (props.label != null 
            || props.required != null || props.valid != null
	    || props.disabled != null)) {

	if (props.disabled != null) {
	    this._common._stripStyleClass(this._labelContainer,
		this._lastLabelDisabledClassName);
	    this._lastLabelDisabledClassName =
		this._getLabelDisabledClassName(props.disabled);
	    this._common._addStyleClass(this._labelContainer,
		this._lastLabelDisabledClassName);
	}

	// Make sure we have an object to hang properties on.
	// This is the case where the widget's "required" and "valid"
	// properties are being updated.
	//
	if (props.label == null) {
	    props.label = {};
	}
	if (props.required != null) {
	    props.label.required = props.required;
	}
	if (props.valid != null) {
	    props.label.valid = props.valid;
	}

	this._widget._updateFragment(this._labelContainer, this.label.id,
            props.label);
    }

    // Set more properties.
    return this._inherited("_setProps", arguments);
};

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.checkedBase
 * @extends woodstock4_3.widget._base.labeledBase
 * @class This class contains functions for widgets that extend checkedBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.checkedBase",
        woodstock4_3.widget._base.labeledBase, {
    // Set defaults.
    _idSuffix: "" // Overridden by subclass
});

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._getImageClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._getInputClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * Returns the HTML input element that makes up the chekcbox.
 *
 * @return {Node} The HTML input element. 
 */
woodstock4_3.widget._base.checkedBase.prototype.getInputElement = function() {
    return this._inputNode;
};

/**
 * Return an Object Literal of label properties desired
 * by the checkedBase widget.
 * <p>
 * This implementation adds the <code>htmlFor</code> property with
 * <code>this._inputNode.id</code>.
 * </p>
 * @return {Object} object with the <code>htmlFor</code>
 * property set.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._getLabelProps = function() {
    // Let the super class contribute
    //
    var props = this.inherited("_getLabelProps", arguments);
    // Subclasses can override this value
    //
    props.level = 3;
    props.htmlFor = this._inputNode.id;
    return props;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.checkedBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.  
    if (this.align != null) { props.align = this.align; }
    if (this.disabled != null) { props.disabled = this.disabled; }   
    if (this.image != null) { props.image = this.image; }
    if (this.name != null) { props.name = this.name; }        
    if (this.readOnly != null) { props.readOnly = this.readOnly; }
    if (this.value != null) { props.value = this.value; }

    // After widget has been initialized, get user's input.
    if (this._isInitialized() == true && this._inputNode.checked != null) {
        props.checked = this._inputNode.checked;
    } else if (this.checked != null) {
        props.checked = this.checked;
    }
    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._onClickCallback = function(event) {
    if (this.readOnly == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the request.
    var result = (this._domNode._onclick)
        ? this._domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._inputNode.id = this.id + this._idSuffix;
        this._imageContainer.id = this.id + "_imageContainer";

        // If null, use HTML input id.
        if (this.name == null) {
            this.name = this._inputNode.id;
        }
    }

    // Set public functions.

    /** @ignore */
    this._domNode.getInputElement = function() {
	var widget = woodstock4_3.widget;
	return widget.common.getWidget(this.id).getInputElement();
    }
    
    // Create callback function for onclick event.
    this._dojo.connect(this._domNode, "onclick", this, "_onClickCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.value != null) { 
        this._inputNode.value = props.value;
    }
    if (props.readOnly != null) { 
        this._inputNode.readOnly = new Boolean(props.readOnly).valueOf();       
    }
    if (props.disabled != null) {
        this._inputNode.disabled = new Boolean(props.disabled).valueOf();        
    }
    
    // Set HTML input element class name.
    this._inputNode.className = this._getInputClassName();
    
    if (props.name != null) { 
        // IE does not support the name attribute being set dynamically as 
        // documented at:
        //
        // http://msdn.microsoft.com/workshop/author/dhtml/reference/properties/name_2.asp
        //
        // In order to create an HTML element with a name attribute, the name
        // and value must be provided when using the inner HTML property or the
        // document.createElement() function. As a work around, we shall set the
        // attribute via the HTML template using name="${this.name}". In order
        // to obtain the correct value, the name property must be provided to 
        // the widget. Although we're resetting the name below, as the default,
        // this has no affect on IE. 
        this._inputNode.name = props.name;
    }

    if (props.checked != null) {
        var checked = new Boolean(props.checked).valueOf();

        // Dynamically setting the checked attribute on IE 6 does not work until
        // the HTML input element has been added to the DOM. As a work around, 
        // we shall use a timeout to set the property during initialization.
        if (this._isInitialized() != true &&
                woodstock4_3._base.browser._isIe()) {
            var _id = this.id;
            setTimeout(function() {
                // New literals are created every time this function
                // is called, and it's saved by closure magic.
                var widget = woodstock4_3.widget.common.getWidget(_id);
                widget._inputNode.checked = checked;
            }, 0); // (n) milliseconds delay.
        } else {
            this._inputNode.checked = checked;
        }
    }

    // Set image properties.
    if (props.image || props.disabled != null && this.image) {     
        // Ensure property exists so we can call setProps just once.
        if (props.image == null) {
            props.image = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.image.className = this._getImageClassName();

        // Update/add fragment.
        this._widget._updateFragment(this._imageContainer, this.image.id, props.image);
    } 

    // Set more properties.
    this._setCommonProps(this._inputNode, props);
    this._setEventProps(this._inputNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};
/**
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */

woodstock4_3._dojo.provide("woodstock4_3.widget._base.checkedGroupBase");

woodstock4_3._dojo.require("woodstock4_3.widget._base.labeledBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.checkedGroupBase
 * @extends woodstock4_3.widget._base.labeledBase
 * @class This class contains functions for widgets that extend checkedGroupBase.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {int} columns 
 * @config {Array} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {boolean} readOnly Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.checkedGroupBase",
    woodstock4_3.widget._base.labeledBase);

/**
 * Helper function to add elements with Object literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled 
 * @config {Array} columns 
 * @config {Array} contents 
 * @config {Object} label
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedGroupBase.prototype._addContents = function(props) {   

    // The label needs special handling because the labeledBase super
    // class is handling the update and creation of the label. The dependence
    // is on "this._labelContainer". We do not want to delete it and
    // create or clone a new one. We need to save the current 
    // "_labelContainer" instance and add it to the new contents.
    //
    // Obviously the whole table layout of rb's and cb's is fragile,
    // and inefficient, especially with respect to the label which
    // resides in the first cell of the first column.
    // Use the "rowspan" attribute to simplify the problem.
    // 
    // This implementation is inefficient if the only properties 
    // of one of the content objects is changing. Therefore 
    // individual rb's and cb's  should be manipulated directly and 
    // not by a
    //
    // "{contents : [ ...,{ ...checked: false...},...]"
    //
    if (props != null && props.contents != null) {

	// Get rid of the current contents
	// An empty "props.content", "{ content : [] }" this will remove
	// all the contents.
	//
	this._widget._removeChildNodes(this._tbodyContainer);

	// In case we are updating, use the columns on the group
	// instance, it may not be specified in props in a call
	// to "setProps".
	//
	var columns = this.columns != null && this.columns > 0
		? this.columns
		: 1;
	if (props.columns != null && props.columns > 0) {
	    columns = props.columns;
	}
	    
        var length = props.contents.length;
        var rows = (length + (columns-1))/columns;

	// Remember if the group is disabled or not.
	// We need to check "this" in case we are being called from
	// "setProps".
	//
        var disabled = this.disabled == null ? false : this.disabled;
	if (props.disabled != null) {
	    disabled = props.disabled;
	}
        var itemN = 0;
	var row = 0;

	// If we have content and there is a label, create the first
	// row with the label. This will set "row" to 1, so the
	// following loop over rows and columns starts at the second
	// row.
	//
	// Note that the label will not exist as a widget the first time
	// through when the group widget is being created.
	//
	var labelNode = null;
	if (rows > 0 && this.label) {
	    var rowContainer = this._rowContainer.cloneNode(false);
	    labelNode = this._labelNode.cloneNode(false);
	    labelNode.id = "label";
	    rowContainer.appendChild(labelNode);
	    labelNode.setAttribute("rowSpan", rows);
	    labelNode.appendChild(this._labelContainer);

	    // Special case for first row, when we have a label.
	    //
	    this._tbodyContainer.appendChild(rowContainer);
	    for (var i = 0; i < columns; ++i, ++itemN) {
		var content = this._contentNode.cloneNode(false);
		rowContainer.appendChild(content);
		// Set disabled.                   
		//
		props.contents[itemN].disabled = disabled;;
		// Add child to the group.
		//
		this._widget._addFragment(
		    content, props.contents[itemN], "last");
	    }
	    row = 1;
	}

        for (; row < rows; row++) {
	    var rowContainer = this._rowContainer.cloneNode(false);
	    this._tbodyContainer.appendChild(rowContainer);
            for (var column = 0; column < columns; column++, itemN++) {
		// Clone <td> node.
		// There could be issues here by not doing a deep clone.
		// There could be "decorative" elements in the td node
		// in the template, but the shallow clone, ignores that.
		//
		var content = this._contentNode.cloneNode(false);
		// In case we have a sparse last row
		//
                if (itemN < length) {
                    rowContainer.appendChild(content);
                    // Set disabled.                   
		    //
                    props.contents[itemN].disabled = disabled;
                   
                    // Add child to the group.
		    //
                    this._widget._addFragment(
			content, props.contents[itemN], "last");
                } else {
		    // We could try using "colspan" on a sparse last row.
		    //
                    rowContainer.appendChild(content);
		}
            }
        }
    }
    return true;
};

/**
 * Return an Object Literal of label properties desired
 * by the checkedGroupBase widget.
 * <p>
 * This implementation adds the <code>htmlFor</code> property assigning
 * it the input element in the first row and the first column of the group.
 * </p>
 * @return {Object} object with the <code>htmlFor</code>
 * property set.
 * @private
 */
woodstock4_3.widget._base.checkedGroupBase.prototype._getLabelProps = function() {
    // Let the super class contribute
    //
    var props = this.inherited("_getLabelProps", arguments);
    // Subclasses can override this value
    //
    props.level = 2;
    return props;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.checkedGroupBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.     
    if (this.columns != null) { props.columns = this.columns; }
    if (this.contents != null) { props.contents = this.contents; }    
    if (this.disabled != null) { props.disabled = this.disabled; }   
    if (this.id != null) { props.id = this.id; }
    if (this.name != null) { props.name = this.name; }
    if (this.readOnly != null) { props.readOnly = this.readOnly; }  

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedGroupBase.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {                    
        this._contentNode.id = this.id + "_contentNode";
        this._divContainer.id = this.id + "_divContainer";
        this._rowContainer.id = this.id + "_rowContainer";
        this._labelNode.id = this.id + "_labelNode";
        this._tableContainer.id = this.id + "_tableContainer";   
        this._tbodyContainer.id = this.id + "_tbodyContainer";     
    }

    // Show label.
    if (this.label) {
        this._common._setVisibleElement(this._labelNode, true);
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.checkedGroupBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.checkedGroupBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set contents.    
    if (props.contents) { // || props.disabled != null) {              
        this._addContents(props);

	// We have a control in the first cell and
	// the label's htmlFor value is not being updated explicityly and
	// we have or are creating a label, update the htmlFor 
	// label property.
	//
	if (props.contents.length != 0 && props.contents[0].id != null &&
		(props.label == null || props.label.htmlFor == null) &&
		this.label != null && this.label.id != null) {
	    if (props.label == null) {
		props.label = {};
	    }
	    var widget = this._widget.getWidget(props.contents[0].id);
	    if (widget != null && widget.getInputElement) {
		props.label.htmlFor = widget.getInputElement().id;
	    }
	}
    } else 
    if (props.disabled != null && this.contents != null) {

	// If props.contents is not null, then disabled will
	// be handled when the contents are updated.
	// Otherwise iterate over the contents disabling them
	//
	for (var i = 0; i < this.contents.length; i++) {
	    var contentWidget = this._widget.getWidget(this.contents[i].id);
	    if (contentWidget) {
		contentWidget.setProps({disabled: props.disabled});
	    }
        }
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.fieldBase");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.fieldBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class contains functions for widgets that extend fieldBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.fieldBase",
        woodstock4_3.widget._base.labeledBase, {
    // Set defaults.
    constructor: function() {
	this.disabled = false;
	this.size = 20;
    }
});

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._getInputClassName = function() {   
    return null; // Overridden by subclass.
};

/**
 * Returns the HTML input element that makes up the text field.
 *
 * @return {Node} The HTML input element.
 */
woodstock4_3.widget._base.fieldBase.prototype.getInputElement = function() {
    return this._fieldNode;
};

/**
 * Return an Object Literal of label properties desired by this widget.
 * <p>
 * This implementation adds the <code>htmlFor</code> property with
 * <code>this._fieldNode.id</code>.
 * </p>
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._getLabelProps = function() {
    var props = this._inherited("_getLabelProps", arguments);

    props.htmlFor = this._fieldNode.id;
    return props;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.fieldBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);
    
    // Set properties.
    if (this.alt != null) { props.alt = this.alt; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.maxLength > 0) { props.maxLength = this.maxLength; }    
    if (this.notify != null) { props.notify = this.notify; }
    if (this.submitForm != null) { props.submitForm = this.submitForm; }
    if (this.text != null) { props.text = this.text; }
    if (this.title != null) { props.title = this.title; }
    if (this.type != null) { props.type= this.type; }
    if (this.readOnly != null) { props.readOnly = this.readOnly; }
    if (this.size > 0) { props.size = this.size; }
    if (this.style != null) { props.style = this.style; }
    
    // After widget has been initialized, get user's input.
    if (this._isInitialized() == true && this._fieldNode.value != null) {
        props.value = this._fieldNode.value;
    } else if (this.value != null) {
        props.value = this.value;
    }
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._fieldNode.id = this.id + "_field";
        this._fieldNode.name = this.id + "_field";
    }
    
    // Set public functions.

    /** @ignore */
    this._domNode.getInputElement = function() { return woodstock4_3.widget.common.getWidget(this.id).getInputElement(); };
    
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.submitForm == false || props.submitForm == true) { 
        // connect the keyPress event
        this._dojo.connect(this._fieldNode, "onkeypress", this, "_submitFormData");
    }
    if (props.maxLength > 0) { this._fieldNode.maxLength = props.maxLength; }
    if (props.size > 0) { this._fieldNode.size = props.size;  }
    if (props.value != null) { this._fieldNode.value = props.value; }
    if (props.title != null) { this._fieldNode.title = props.title; }   
    if (props.disabled != null) { 
        this._fieldNode.disabled = new Boolean(props.disabled).valueOf();
    }
    if (props.readOnly != null) { 
        this._fieldNode.readOnly = new Boolean(props.readOnly).valueOf();
    }
    
    // Set HTML input element class name.
    this._fieldNode.className = this._getInputClassName();
    
    // Set more properties.
    this._setCommonProps(this._fieldNode, props);
    this._setEventProps(this._fieldNode, props);
    
    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * Process keyPress events on the field, which enforces/disables 
 * submitForm behavior.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.fieldBase.prototype._submitFormData = function(event) {
    if (event == null) {
        return false;
    }
    if (event.keyCode == event.KEY_ENTER) {               
        if (this.submitForm == false) {
            // Disable form submission.
            if (window.event) {
                event.cancelBubble = true;
                event.returnValue = false;
            } else{
                event.preventDefault();
                event.stopPropagation();
            }
            return false;
        } else {
            // Submit the form                    
            if (event.currentTarget.form) {
                event.currentTarget.form.submit();
            }
        }
    }
    return true;    
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.menuBase");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.menuBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class contains functions for widgets that extend menuBase.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} formId 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {Array} options 
 * @config {boolean} primary Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.menuBase",
        woodstock4_3.widget._base.widgetBase, {
    // Set defaults.
    constructor: function() {
        this._focusPosition = 0;
    }
});

/**
 * Add the specified options to the dom element.
 *
 * @param {Node} menuNode The node to which the menu items are to be added.
 * @param {Object} props Key-Value pairs of properties.
 * @config {Array} options 
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._addOptions = function(menuNode, props) {
    var groupNode, optionNode, separator, sepNode;
    for (var i = 0; i < props.options.length; i++) {
        
        // create an li node which will represent an option element.
        optionNode = this._optionContainer.cloneNode(false);   
        optionNode.id = this.id + "_" + props.options[i].label + "_container";                
        this._setOptionNodeProps(optionNode, props.options[i], i);

        // Append the li element to the menu element.        
        menuNode.appendChild(optionNode);
        if (props.options[i].group == true) {
            optionNode.group = true;
            groupNode = this._groupOptionContainer.cloneNode(false);
            menuNode.appendChild(groupNode);
            this._addOptions(groupNode, props.options[i]);
        }
        
        if (props.options[i].separator == true) {
            separator = this._menuSeparatorContainer.cloneNode(true);
            if (woodstock4_3._base.browser._isIe5up()) {
                var sep = this._menuSeparator.cloneNode(true);
                separator.appendChild(sep);
            }
            this._common._setVisibleElement(separator, true);             
            menuNode.appendChild(separator);
        }
    }
    return true;    
};

/**
 * The callback function for clicking on a menu item.
 *
 * @param {String} optionId The id of the option element that is clicked.
 * @return {Function} The callback function.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._createOnClickCallback = function(optionId) {
    var _id = this.id;

    /**
     * New literals are created every time this function is called, and it's 
     * saved by closure magic.
     *
     * @param {Event} event The JavaScript event.
     */
    return function(event) {
        // Get hold of the particular option element.
        var elem = document.getElementById(optionId);
        if (elem == null) {
            return;
        }        

        // Get hold of the menu element.
        var widget = woodstock4_3.widget.common.getWidget(_id);                
        var val = elem.selectValue;
        var dis = elem.disabled;       
        var group = elem.group;
        
        // process the action only if the menu is not disabled and the selected
        // option is not a group header.
        if (!dis && !group) {
            widget._processOnClickEvent(val);
        }         
    };
};

/**
 * The callback function for key press on a menu item.
 *
 * @param {String} nodeId The id of the option element that is clicked
 * @return {Function} The callback function.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._createOnKeyDownCallBack = function(nodeId) {
    if (nodeId == null) {
        return null;
    }
    var id = this.id;
    return function(event) {
        var elem = document.getElementById(nodeId);
        if (elem == null) {
            return false;
        }
        var common = woodstock4_3.widget.common;
        var widget = common.getWidget(id);

        // If the menu is not visible, we do not need to capture
        // key press events.
        if (!woodstock4_3._base.common._isVisibleElement(widget.domNode)) {
            return false;
        }
        event = common._getEvent(event);
        var keyCode = common._getKeyCode(event);
        
        // if onkeypress returns false, we do not traverse the menu.
        var keyPressResult = true;
        if (widget._onkeypress) {
            keyPressResult = (widget._onkeypress) ? widget._onkeypress() : true;
        }
        if (keyPressResult != false) {
            widget._traverseMenu(keyCode, event, nodeId);
        }
        return true;
    };
};

/**
 * Handles the on mouse out for each menuitem.
 *
 * @param {Node} menuItem The DOM node associated with the menu item.
 * @return {Function} The callback function.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._createOnMouseOutCallBack = function(menuItem) {
    if (menuItem == null) {
        return null;
    }
    var _id = this.id;
 
    // New literals are created every time this function is called, and it's 
    // saved by closure magic.
    return function(event) {
        var widget = woodstock4_3.widget.common.getWidget(_id);
        menuItem.className = widget._theme.getClassName("MENU_GROUP_CONTAINER");
    };
};

/**
 * Handles the on mouse over for each menuitem.
 *
 * @param {Node} menuItem The DOM node associated with the menu item.
 * @return {Function} The callback function.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._createOnMouseOverCallBack = function(menuItem) {
    if (menuItem == null) {
        return null;
    }
    var _id = this.id;

    /**
     * New literals are created every time this function is called, and it's 
     * saved by closure magic.
     *
     * @param {Event} event The JavaScript event.
     */
    return function(event) {
        var widget = woodstock4_3.widget.common.getWidget(_id);
        menuItem.className = menuItem.className + " " + 
            widget._theme.getClassName("MENU_ITEM_HOVER");            
        if (widget != null) {
            // Mozilla browser (not firefox/seamonkey) do not support focus/blur
            // for divs                
            if (document.getElementById(widget.menuId[widget._focusPosition]).blur) {
                document.getElementById(widget.menuId[widget._focusPosition]).blur();
            }                
            if (!(menuItem.id == widget.menuId[widget._focusPosition])) {
                document.getElementById(widget.menuId[widget._focusPosition]).className =
                    widget._theme.getClassName("MENU_GROUP_CONTAINER");
            }
        }
        if (woodstock4_3._base.browser._isIe5up() ) {
            menuItem.className = menuItem.className + " " + 
                widget._theme.getClassName("MENU_ITEM_HOVER");            
        }
    };
};

/**
 * Calculate the maximum width of menu to be set.
 * If a menu if at all contains a group, then set the containsGroup
 * attribute to true. This will help in adding an extra pixel width
 * while assigning the width of th menu. This is to account for the
 * indentation of the menu.
 *
 * @param {Array} props 
 * @return {int} The max menu width.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._getMaxWidth = function(props) {
    var menuWidth = 0;
    var maxWidth = 0;
    for (var i = 0; i < props.length; i++) {
         menuWidth = props[i].label.length;
         if (menuWidth > maxWidth) {
            maxWidth = menuWidth;
         }
         if (props[i].image != null) {
            this.hasImage = true;
         }
         if (props[i].group) {
            this.containsGroup = true;
            maxWidth = this._getMaxWidth(props[i].options);
         }
    }
    return maxWidth;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.menuBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    if (this.options != null) { props.options = this.options; }
    if (this.formId != null) { props.formId = this.formId; }
    if (this.submitForm != null) { props.submitForm = this.submitForm; }  

    return props;
};

/**
 * Returns the currently selected item in the menu.
 *
 * @return {String} The selected item.
 */
woodstock4_3.widget._base.menuBase.prototype.getSelectedValue = function() {
    if (this.clickedItem) {
        return this.clickedItem;
    } else {
        return null;
    }
};

/**
 * This function is used to obtain the outermost HTML element style.
 * <p>
 * Note: Styles should be concatinated in order of precedence (e.g., the 
 * user's style property is always appended last).
 * </p>
 * @return {String} The outermost HTML element style.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._getStyle = function() {
    var style = "width:" + this.maxWidth + "em;";

    // Since the style has to be recalculated each and every time the options
    // are changed, you cannot rely on the the existing style on the dom element
    // to see whether style was already defined. Hence use a separate variable
    // which sets itself to true if the widget sets the width/
    if (!this.widthSet) {
    	var st = this.style;
	var reg = ";?[\\t]*width[\\t]*:";
	if (st != null) {
	    var res = st.match(reg);

            // Return user's style if width is set already.
            if (this.style && res != null) {
                return this.style;
            }
	}
    }
    this.widthSet = true;
    return (this.style)
        ? style + this.style
        : style;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._postCreate = function () {
    // Set public functions.

    /** @ignore */
    this._domNode.getSelectedValue = function(props, optionNode) { return woodstock4_3.widget.common.getWidget(this.id).getSelectedValue(); };
        
    return this._inherited("_postCreate", arguments);
};

/**
 * Process the enter key press event.Evaluvate the keyPress/keyDown (for non-IE/IE browsers)
 * and traverse through the menu. Also, if onChange is specified, evaluvate that and 
 * submit the form if submitForm is specified to true.
 *
 * @param (String) value The "value" of the selected option. 
 * @return {boolean} true The enter key press event completed successfully
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._processEnterKeyPressEvent = function(value) {
    var changeResult = true;

    // Check whether the selected value is different than the one previously selected
    var bool = (value == this.getSelectedValue());
    this._setSelectedValue(value);

    if (this._onchange && !bool) {    
        // If function returns false, we must prevent the request.       
        changeResult = (this._onchange) ? this._onchange() : true;
    }
    
    // Functions may sometime return without a value in which case the value
    // of the boolean variable may become undefined. 
    if (changeResult != false) {
        if (this.submitForm == true) {
            this._submitFormData();
        }  
    }
    return true;
};
    
/**
 * This function executes the onchange and onclick event handlers if provided by 
 * the developer. It then either submits the form if submitForm attribute is 
 * specified to true.
 *
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._processOnClickEvent = function(value) {
    var clickResult = true;
    var changeResult = true;
    
    //Check if the selected value has changed from the previous selection.
    var bool = (value == this.getSelectedValue());
    this._setSelectedValue(value);

    if (this._onclick) {
        clickResult = (this._onclick) ? this._onclick() : true;
    }
    if (this._onchange && !bool) {    
        // If function returns false, we must prevent the request.       
        changeResult = (this._onchange) ? this._onchange() : true;
    }

    // Functions may sometime return without a value in which case the value
    // of the boolean variable may become undefined. 
    if (clickResult != false && changeResult != false) {
        if (this.submitForm == true) {
            this._submitFormData();
        }  
    }
    return true;
};

/**
 * Set the appropriate class name on the menu item container.
 *
 * @param {Node} menuItemContainer The container for the menu item.
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled 
 * @config {boolean} group
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._setMenuNodeClassName = function(
        menuItemContainer, props) {        
    if (new Boolean(props.group).valueOf() == true) {
        menuItemContainer.className = this._theme.getClassName("MENU_GROUP_HEADER");
    } else if (new Boolean(props.disabled).valueOf() == true) {
        menuItemContainer.className = this._theme.getClassName("MENU_ITEM_DISABLED");
    } else {
        menuItemContainer.className = this._theme.getClassName("MENU_GROUP_CONTAINER");        

        // Whenever mouse over/out happens, focus must be set on the menu accordingly.
        // Apply an hack for IE for mouse hover on the div element since div:hover type
        // of css declarations do not seem to work. onmouseover and onmouseout events
        // are attached with the div element and a style class is applied each time a
        // mouseover happens. This style represents the "hover" class.
        // Note that the "this" in these functions represent the menuItem's "div" element
        // and not the "menu" widget element.
        this._dojo.connect(menuItemContainer, "onmouseover",
            this._createOnMouseOverCallBack(menuItemContainer));
        this._dojo.connect(menuItemContainer, "onmouseout",
            this._createOnMouseOutCallBack(menuItemContainer));
    }
    return true;
};

/**
 * Helper function to set the properties of an option item. This is invoked
 * by the _addOptions function call.
 *
 * @param optionNode The node for which the option is to be added.
 * @param {Object} props Key-Value pairs of properties.
 * @param {String} number The position of the option item in the menu.
 * @config {boolean} disabled 
 * @config {boolean} escape
 * @config {Object} image
 * @config {String} label
 * @config {String} title
 * @config {String} value
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._setOptionNodeProps = function(optionNode, props, number) {
    optionNode.id = this.id + "_" + props.value + "_container";
    var menuItemContainer = this._menuItemContainer.cloneNode(false);
    menuItemContainer.id = optionNode.id + "_label";

    // depending on the kind of node, assign the appropriate style
    // for the menu node.
    this._setMenuNodeClassName(menuItemContainer, props);
    optionNode.appendChild(menuItemContainer);
    
    // valueNode contains a div element which will hold the option.
    var valueNode = this._menuItemNode.cloneNode(false);  
    valueNode.id = this.id + "_" + props.value;

    if (!(new Boolean(props.group).valueOf() == true) && 
            !(new Boolean(props.disabled).valueOf() == true)) {
        this.menuId[this.menuItemCount++] = menuItemContainer.id;   
    }

    menuItemContainer.tabIndex = -1;
    valueNode.tabIndex = -1;

    menuItemContainer.selectValue = props.value;
    menuItemContainer.disabled = props.disabled;
    menuItemContainer.group = props.group;
    if (props.title != null) {
        menuItemContainer.title = props.title;
        valueNode.title = props.title;
    }
    
    if (valueNode.setAttributeNS) {
        valueNode.setAttributeNS(
            "http://www.w3.org/2005/07/aaa", "posinset", number);
    }

    if (valueNode.setAttributeNS) {
        valueNode.setAttributeNS(
            "http://www.w3.org/2005/07/aaa", "disabled", props.disabled);
    }
        
    // Create callback function for onkeydown event.
    this._dojo.connect(menuItemContainer, "onkeydown", 
        this._createOnKeyDownCallBack(menuItemContainer.id));         
        
    // Create callback function for onClick event.
    this._dojo.connect(menuItemContainer, "onclick",
        this._createOnClickCallback(menuItemContainer.id));
        
    // Set label value.
    if (props.label) {
        this._widget._addFragment(valueNode, props.label, "last", props.escape);
    }    

    // Set title.
    if (props.title != null) {
        optionNode.title = props.title;
    }
    
    // By default have the no image container cloned and kept
    // If an image is present, then replace that with the span
    // placeholder for the image.
    if (new Boolean(this.hasImage).valueOf() == true) {
        var imageNode = this._menuItemNoImageContainer.cloneNode(false);
        if (props.image != null) {
            // Add the widget
            imageNode = this._menuItemImageContainer.cloneNode(false);
            props.image.className = this._theme.getClassName("MENU_ITEM_IMAGE");
            this._widget._addFragment(imageNode, props.image);
        } 
        menuItemContainer.appendChild(imageNode);
    }
    
    // Append the placeholder image node.
    menuItemContainer.appendChild(this._menuItemSubMenu.cloneNode(false));

    // Append the div element to the li element.           
    menuItemContainer.appendChild(valueNode);
    return true;
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.menuBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }
    
    // Replace options -- do not extend.
    if (props.options) {
        this.options = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._setProps = function(props){
    if (props == null) {
        return false;
    }

    // A web app devleoper could return false in order to cancel the 
    // submit. Thus, we will handle this event via the onClick call back.
    if (props.onChange) {
        // Set private function scope on widget.
        this._onchange = (typeof props.onChange == 'string')
            ? new Function(props.onChange) : props.onChange;

        // Must be cleared before calling setJavaScriptProps() below.
        props.onChange = null;
    }

    // A web app devleoper could return false in order to cancel the 
    // submit. Thus, we will handle this event via the onClick call back.
    if (props.onClick) {
        
        // Set private function scope on widget.
        this._onclick = (typeof props.onClick == 'string')
            ? new Function(props.onClick) : props.onClick;

        // Must be cleared before calling setJavaScriptProps() below.
        props.onClick = null;
    }
    
    // Add options
    if (props.options) {
        
        // Try to set the width of the menu here.
        this.maxWidth = this._getMaxWidth(props.options);        
        
        // Account for image width if one exists. This property can be got from the
	// theme
        if (this.hasImage) {
            var placeHolderWidth = parseFloat(this._theme.getMessage("Menu.placeHolderImageWidth"));
            this.maxWidth += placeHolderWidth; 
        }       
   
        // If an menuGroup exists, then add one character width. Otherwise menu
        // does not scale properly
        if (this.containsGroup) {
            this.maxWidth += 1;
        }
             
        this._widget._removeChildNodes(this._outerMenuContainer);
        this.menuId = [];
        this.menuItemCount = 0;
        
        // Clone the menu node and add it to the outer container.
        var menuNode = this._groupOptionContainer.cloneNode(false);
        menuNode.className = this._theme.getClassName("MENU_CONTAINER");
        this._outerMenuContainer.appendChild(menuNode);         
        this._addOptions(menuNode, props);
    }

    // Need to redo style calculation if style or options
    // have been specified.
    if (props.style || props.options) {
        props.style = this._getStyle();
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);
    
    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * Set the selected item on the widget.
 *
 * @param {String} item The selected value.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._setSelectedValue = function(item) {
    this.clickedItem = item;
    return true;
};

/**
 * Submits the form. Appends the value of the selected item in the request url.
 *
 * @return {boolean} false to cancel the JavaScript event.
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._submitFormData = function () {
    if (this.formId == null) {
        return false;
    }
    var form = document.getElementById(this.formId);
    if (form == null) {
        return false;
    }
    var oldAction = form.action;
    var oldTarget = form.target;

    // Set new action URL.
    var prefix;
    if (form.action) {
        prefix = (form.action.indexOf("?") == -1) ? "?" : "&";
        form.action += prefix + this.id + "_submittedValue=" + this.getSelectedValue();
    }

    form.target = "_self";
    form.submit();     
    form.action = oldAction;
    form.target = oldTarget;
    return false;
};

/**
 * This function takes care of traversing through the menu items depending
 * on which key is pressed.
 *
 * @param (String) keyCode The valye of the key which was pressed
 * @param (Event) event The key press event.
 * @param (String) nodeId The id of the menu item. 
 * @return {boolean} true Propagate the javascript event
 * @private
 */
woodstock4_3.widget._base.menuBase.prototype._traverseMenu = function(keyCode, event, nodeId) {
    var arr = this.menuId;
    var elem = document.getElementById(nodeId);
    var focusElem = document.getElementById(arr[this._focusPosition]);
    
    if (focusElem.blur) {
        focusElem.blur();
    }
    // Operations to be performed if the arrow keys are pressed.
    if (keyCode >= 37 && keyCode <= 40) {
        
        // Check whether up arrow was pressed.
        if (keyCode == 38) {
            focusElem.className = this._theme.getClassName("MENU_GROUP_CONTAINER");
            this._focusPosition--;
            if (this._focusPosition < 0) {
                this._focusPosition = arr.length-1;
            }
            focusElem = document.getElementById(arr[this._focusPosition]); 

        // Check whether down arrow was pressed
        } else if (keyCode == 40) {
            focusElem.className = 
                this._theme.getClassName("MENU_GROUP_CONTAINER");
            this._focusPosition++;
            if (this._focusPosition == arr.length) {
                this._focusPosition = 0;
            }
            focusElem = document.getElementById(arr[this._focusPosition]);
        }   
        
    if (focusElem.focus) {
        focusElem.focus();
    }        
        focusElem.className = focusElem.className + " "+
            this._theme.getClassName("MENU_FOCUS");           
    // Check if enter key was pressed    
    } else if(keyCode == 13){
        focusElem.className =
            this._theme.getClassName("MENU_GROUP_CONTAINER");
        var val = elem.selectValue;
        this._processEnterKeyPressEvent(val);
       
    }    
    if (woodstock4_3._base.browser._isIe5up()) {
        window.event.cancelBubble = true;
        window.event.returnValue = false;
    } else {
        event.stopPropagation();
        event.preventDefault();
    }
    return true;
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.refreshBase");

woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/** 
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.refreshBase
 * @class This class contains functions for widgets that extend refreshBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.refreshBase");

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget._base.refreshBase.event =
        woodstock4_3.widget._base.refreshBase.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         */
        beginTopic: undefined,

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         */
        endTopic: undefined
    }
};

/**
 * Initialize public functions.
 * <p>
 * Note: If this.event.refresh.beginTopic and this.event.refresh.endTopic are
 * null, a public function shall be added to the DOM node.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.refreshBase.prototype._initRefreshFunc = function () {
    if (this.event == null || this.event.refresh == null
            || this.event.refresh.beginTopic == null
            || this.event.refresh.endTopic == null) {
        return false;
    }
    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Set public functions.
    /** @ignore */
    this._domNode.refresh = function(execute) {
        return woodstock4_3.widget.common.getWidget(_id).refresh(execute);
    };
    return true;
};

/**
 * Process refresh event.
 * <p>
 * Note: If this.event.refresh.beginTopic and this.event.refresh.endTopic are 
 * not null, an event is published for custom Ajax implementations to listen 
 * for. If event topics are not implemented for this widget, the function 
 * returns and a message is output to the console.
 * </p>
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.refreshBase.prototype.refresh = function(execute) {
    if (this.event == null || this.event.refresh == null
            || this.event.refresh.beginTopic == null
            || this.event.refresh.endTopic == null) {
        console.debug("Error: Refresh event topics not implemented for " + 
            this._widgetType); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this._publish(this.event.refresh.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: this.event.refresh.endTopic
    }]);
    return true;
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.selectBase");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3._base.common");
woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.labeledBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.selectBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class defines functions and properties for
 * widgets based on the "select" HTML element.
 *
 * <h3>Dojo attach points</h3>
 * A <code>selectBase</code>  subclass's template must
 * define the following attach point identifiers.
 * <ul>
 * <li><code>_listContainer</code> - the <code>select</code> element.</li>
 * <li><code>_optionNode</code> - the element to clone for an
 * <code>option</code> element.
 * </li>
 * <li><code>_optionGroupContainer</code> - the element to clone for an
 * <code>optGroup</code> element.</li>
 * <li><code>_memberOptionNode</code> - the element to clone for an
 * <code>option</code> in an <code>optGroup</code> element.</li>
 * </ul>
 * <h3>The <code>options</code> array property</h3>
 * The <code>options</code> array property defines the contents of the
 * <code>select</code> HTML element. The contents of the array are 
 * translated into <code>option</code> and <code>optGroup</code> HTML
 * elements. Each element of the <code>options</code> array property
 * is considered an object with the following properties. Note that
 * "NA" in the following table means "not applicable", meaning there
 * is no HTML element property counterpart.
 * <p>
 * <table border="1px">
 * <tr><th>Property</th><th>Type</th><th>Description</th>
 * <th>HTML option or optGroup element property</th></tr>
 * <tr><td>label</td><td>String</td><td>
 * The text that appears as the choice in the rendered <code>option</code>
 * HTML element. (See note[1] below)
 * </td><td>label</td></tr>
 * <tr><td>value</td><td>String</td><td>
 * The value for this option. It is submitted in a request if this option is
 * selected.</td><td>value</td></tr>
 * <tr><td>separator</td><td>boolean</td><td>
 * If true this option represents a separator and cannot be selected.
 * </td><td>NA</td></tr>
 * <tr><td>group</td><td>boolean</td><td>
 * If true this option represents a group and cannot be selected. An
 * <code>optGroup</code> HTML element is rendered to represent this option.
 * </td><td>NA</td></tr>
 * <tr><td>escape</td><td>boolean</td><td>
 * If false the label string will be evaluated as HTML markup.
 * </td><td>NA</td></tr>
 * <tr><td>selected</td><td>boolean</td><td>
 * If true this option will be initially selected. 
 * </td><td>selected</td></tr>
 * <tr><td>disabled</td><td>boolean</td><td>
 * If true this option will be initially disabled and cannot be selected.
 * </td><td>disabled</td></tr>
 * <tr><td>title</td><td>String</td><td>
 * The HTML title attribute text.</td><td>title</td></tr>
 * </table>
 * </p>
 * <p>
 * The option object may also define the javascript event properties
 * suitable for an <code>option</code> or <code>optGroup</code> HTML
 * element and they will be assigned to the element's DOM node.
 * </p>
 * <p>
 * <ul>
 * <li>Note[1] Browser runtime behavior, contrary to the HTML and DOM
 * specifications, indicates that the DOM <code>label</code> property
 * is not rendered and only the <code>text</code> property is rendered.
 * This means that the <code>label</code> property of an
 * <code>options</code> array element will
 * be assigned to an <code>option</code> or <code>optGroup</code>
 * DOM node's <code>text</code> property.</li>
 * </ul>
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled If true the select element is disabled.
 * @config {Object} label The properties for a label widget.
 * @config {boolean} labelOnTop If false the label will appear to the left
 * of the <code>select</code> element, aligned with the first list option. 
 * @config {Object} options An array of Objects that represent the 
 * <code>select</code> element's <code>option</code> and 
 * <code>optgroup</code> elements. @see #_setOptions.
 * @config {boolean} required If true the a selection is required.
 * @config {boolean} valid If true the widget has a valid selection.
 * @config {String} width This value will be assigned to the 
 * <code>_listContainer.style.width</code> attribute to set the width of the
 * select element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.selectBase", 
	woodstock4_3.widget._base.labeledBase, {
    // Set defaults
    constructor: function() {
	this.disabled = false;
	// Flag to prevent blank entries in the drop down for the original empty
        // dojo attach point nodes. See "_setOptions".
	this._alreadyRemoved = false;
    }
});

/**
 * This function is called by the <code>_onChangeCallback</code>
 * function to set the option element's className attribute with appropriate
 * selected and disabled selectors.
 * <p>
 * This method calls <code>this._getOptionClassName</code> once for each
 * option. Subclasses should implement <code>_getOptionClassName</code>
 * in order to assign the appropriate CSS class name, to reflect a disabled or
 * selected option and any other properties due to an onChange event.
 * <p>
 * See <code>_getOptionClassName</code> for the default behavior.
 * </p>
 * <p>
 * Before this method calls <code>this._getOptionClassName</code>, it has
 * addressed an IE issue where disabled options can be selected, and has
 * deslected, selected disabled options.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._changed = function() {
    var options = this._listContainer.options;

    // IE allows disabled options to be selected. Ensure that
    // disabled options never appear as selected options.
    //
    if (woodstock4_3._base.browser._isIe()) {
	for (var i = 0; i < options.length; ++i) {
	    if (options[i].disabled == true && options[i].selected == true) {
		if (this._listContainer.multiple == true) {
		    options[i].selected = false;
		} else {
		    // Don't we need to set the disabled option's
		    // selected to false too ?
		    //
		    this._listContainer.selectedIndex = -1;
		}
	    }
	    // Only set the option if its value is different
	    // than what is returned. Note that if this method
	    // returns null, the option may not visually reflect
	    // the true state.
	    //
	    var cn = this._getOptionClassName(options[i]);
	    if (cn != null && options[i].className != cn) {
		options[i].className = cn;
	    }
	} 
    } else {
	for (var i = 0; i < options.length; ++i) { 
	    // Only set the option if its value is different
	    // than what is returned. Note that if this method
	    // returns null, the option may not visually reflect
	    // the true state.
	    //
	    var cn = this._getOptionClassName(options[i]);
	    if (cn != null && options[i].className != cn) {
		options[i].className = cn;
	    }
	}
    }
    return true;
};

/**
 * This method copies option properties from <code>fromOption</code> to 
 * <code>toOption</code>. Note that <code>fromOption</code> is assumed to
 * be a DOM <code>option</code> element. There is not a one to one mapping
 * of properties by name from a DOM <code>option</code> element to a
 * widget's <code>options</code> array <code>option</code> array element.
 * The properties that are set in <code>_setOptionProps</code> and
 * <code>_setGroupOptionProps</code> are copied.
 * @param {Object} toOption An <code>Object</code> to receive
 * <code>selectBase</code> option properties.
 * @param {Object} fromOption An <code>option</code> or <code>optgroup</code>
 * element DOM node to copy <code>selectBase</code> option properties from.
 * @return {boolean} Returns true.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._copyOption = function(toOption, fromOption) {
    var domhandlers = [ "onblur", "onchange", "onclick", "ondblclick",
	"onfocus", "onkeydown", "onkeypress", "onkeyup", "onmousedown",
	"onmouseout", "onmouseover", "onmouseup", "onmousemove", "onresize"];

    var widgethandlers = [ "onBlur", "onChange", "onClick", "onDblClick",
	"onFocus", "onKeyDown", "onKeyPress", "onKeyUp", "onMouseDown",
	"onMouseOut", "onMouseOver", "onMouseUp", "onMouseMove", "onResize"];

    for (var i = 0; i < domhandlers.length; ++i) {
	if (fromOption[domhandlers[i]]) {
	    toOption[widgethandlers[i]] = fromOption[domhandlers[i]];
	}
    }
    toOption.label = fromOption.label;
    toOption.value = fromOption.value;
    toOption.escape = fromOption.escape;
    toOption.disabled = fromOption.disabled;
    toOption.defaultSelected = fromOption.defaultSelected;
    toOption.selected = fromOption.selected;
    toOption.title = fromOption.title;
    toOption.className = fromOption.className;
    toOption.group = fromOption.group;
    if (toOption.group == true) {
	toOption.options = this._copyOptions(fromOption);
    }
    return true;
};

/**
 * Returns an <code>options</code> array of <code>option</code> objects
 * as defined by <code>selectBase</code> (see selectBase._setOptions).
 * <code>domNode</code> is assumed to be a <code>select</code> element
 * or an <code>optgroup</code>.
 * <p>
 * There is not a one to one mapping of properties by name from a DOM
 * <code>option</code> or <code>optgroup</code> element to a
 * widget's <code>options</code> array <code>option</code> object array element.
 * The properties that are set in <code>_setOptionProps</code> and 
 * <code>_setGroupOptionProps</code> are copied to <code>Object</code>s
 * and returned as the elements of the returned array. The returned
 * array is equivalent to the <code>options</code> array used to create
 * a selectBase subclass, like listbox or dropdown.
 * @param {Object} domNode A <code>select</code> element DOM node or an
 * <code>optgroup</code> DOM node.
 * @return {Object} An array of <code>selectBase</code> options. If 
 * <code>domNode</code> has no child nodes, an empty array is returned.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._copyOptions = function(domNode) {
    var newoptions = [];
    if (!domNode.hasChildNodes()) {
	return newoptions;
    }
    var len = domNode.childNodes.length;
    for (var j = 0; j < len; ++j) {
	newoptions[j] = new Object();
	if (domNode.childNodes != null) {
	    this._copyOption(newoptions[j], domNode.childNodes[j]);
	}
    }
    return newoptions;
};

/**
 * Return an Object Literal of label properties desired by this widget.
 * <p>
 * This implementation adds the <code>htmlFor</code> property with
 * <code>this._listContainer.id</code>.
 * </p>
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._getLabelProps = function() {
    var props = this._inherited("_getLabelProps", arguments);

    props.htmlFor = this._listContainer.id;
    return props;
};

/**
 * This function returns the CSS class name for an HTML <code>option</code> or
 * <code>optgroup</code> element.
 * <p>
 * This implementation returns <code>option.className</code>
 * This method should be overridden by subclasses.
 * </p>
 *
 * @param {Object} option Key-Value pairs of properties.
 * @return {String} The HTML option element class name.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._getOptionClassName = function(option) {
    // Make sure that if a subclass does not implement this method
    // that it causes no change to option.
    //
    return option.className;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.selectBase.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Get properties.
    // Should we return the default theme value for labelOnTop
    // if "this.labelOnTop" has not been set ?
    //
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.width != null) { props.width = this.width; }

    // After widget has been initialized, get actual select element state
    // Note that the options that exist on the select element will
    // not include the original "options" props that are not
    // HTML option or HTML optgroup attributes.
    //
    if (this._isInitialized() == true && this._listContainer != null) {
	// We need to copy the options from this._listContainer
	// or else they will be destroyed if passed back to addProps.
	//
	var opts = this._copyOptions(this._listContainer);
        props.options = opts;
    } else if (this.options != null) {
        props.options = this.options;
    }
    return props;
};

/**
 * This function is used to obtain the underlying HTML select element.
 *
 * @return {Node} The HTML select element.
 */
woodstock4_3.widget._base.selectBase.prototype.getSelectElement = function() {
    return this._listContainer;
};

/**
 * This function is used to obtain the index of the selected option.
 *
 * @return {int} The selected index of underlying HTML select element.
 */
woodstock4_3.widget._base.selectBase.prototype.getSelectedIndex = function() {
    return this._listContainer.selectedIndex;
};

/**
 * This function is used to obtain the <code>label</code> attribute of
 * the selected option.
 * <p>
 * If no option is selected, this function returns null.
 * </p>
 * <p>
 * If the underlying select element's <code>multiple</code> attribute is true,
 * this method returns the first selected option's label.
 * </p>
 * 
 * @return The label of the selected option, or null if none is selected. 
 * @return {String} The label attribute of the selected option.
 */
woodstock4_3.widget._base.selectBase.prototype.getSelectedLabel = function() { 
    var index = this._listContainer.selectedIndex; 

    if (index == -1) { 
        return null; 
    } else { 
	return this._listContainer.options[index].label;
    }
};

/**
 * This function is used to obtain the <code>value</code> attribute
 * of the selected option.
 * <p>
 * If no option is selected, this function returns null. 
 * </p>
 * <p>
 * If the underlying select element's <code>multiple</code>" attribute is true,
 * this method returns the first selected option's value.
 * </p>
 *
 * @return {String} The selected option value or null if none is selected. 
 */
woodstock4_3.widget._base.selectBase.prototype.getSelectedValue = function() { 
    var index = this._listContainer.selectedIndex; 
    if (index == -1) { 
        return null; 
    } else { 
        return this._listContainer.options[index].value; 
    }
};

/**
 * This function is invoked for the select element's <code>onchange</code>
 * event.
 * <p>
 * If <code>this.disabled</code> is true, return false.<br/>
 * This method calls the handler defined by the <code>props.onChange</code>
 * property. If that handler returns false, false is returned. If the
 * <code>props.onChanged</code> handler returns true, then
 * <code>this._changed</code> is called and its return value is returned.
 * </p>
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._onChangeCallback = function(event) {
    if (this.disabled == true) {
        return false;
    }

    // If function returns false, we must prevent the auto-submit.
    //
    var result = (this._listContainer._onchange)
        ? this._listContainer._onchange(event) : true;
    if (result == false) {
        return false;
    }

    // Set style classes.
    return this._changed(event);
};

/**
 * This function is used to fill in remaining template properties, after the
 * <code>_buildRendering</code> function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._listContainer.id = this.id + "_list";
	this._listContainer.name = this._listContainer.id;
    }

    // Set public functions.

    /** @ignore */
    this._domNode.getSelectedValue = function() { 
	return woodstock4_3.widget.common.getWidget(this.id).getSelectedValue();
    };
    /** @ignore */
    this._domNode.getSelectedLabel = function() { 
	return woodstock4_3.widget.common.getWidget(this.id).getSelectedLabel();
    };
    /** @ignore */
    this._domNode.getSelectElement = function() { 
	return woodstock4_3.widget.common.getWidget(this.id).getSelectElement();
    };

    // Set events.
    this._dojo.connect(this._listContainer, "onchange", this, 
	"_onChangeCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set the <code>option</code> properties on the
 * <code>optgroup</code> HTML DOM element, <code>element</code>.
 * <p>
 * This method assigns the <code>option.label</code>,
 * <code>option.disabled</code>, <code>option.title</code> properties
 * to <code>element</code>, non HTML properties defined in 
 * <code>option</code>, calls <code>this._setEventProps</code>
 * and then calls <code>this._getOptionClassName</code>. 
 * Subclasses should implement <code>_getOptionClassName</code> if the default
 * behavior of <code>selectBase._getOptionClassName</code> is not appropriate.
 * Subclasses usually subclass this method to provide subclass specific 
 * CSS class names and properties. For all the properties defined in the 
 * option array see <code>_setOptions</code>.
 * </p>
 * @param {Node} element The optgroup DOM node
 * @param {Object} option Key-Value pairs of properties for the optgroup node.
 * @config {boolean} disabled If true the optgroup is disabled.
 * @config {String} label The optgroup choice text.
 * @config {String} title The optgrp HTML title attribute value.
 * @config {boolean} escape If false the label string will be evaluated
 * as HTML markup.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._setGroupOptionProps =
	function(element, option) {
    element.label = option.label;
  
    if (option.disabled != null) {
        element.disabled = option.disabled;
    }

    if (option.title != null) {
	element.title = option.title;
    }

    // optGroup elements can have events.
    this._setEventProps(element, option);

    // Non HTML properties
    //
    if (option.escape != null) {
	element.escape = option.escape;
    }
    element.group = true;

    // Do this last so that all other properties will have been set.
    // Note that if this does return null, which it shouldn't
    // then the visual appearance may not reflect the 
    // actual state.
    //
    var cn =  this._getOptionClassName(element);
    if (cn != null) {
	element.className = cn;
    }
    return true;
};

/**
 * This function is used to set the <code>option</code> properties on the
 * <code>option</code> HTML DOM element, <code>element</code>.
 * <p>
 * This method assigns the <code>option.label</code>,
 * <code>option.disabled</code>, <code>option.title</code> properties
 * to <code>element</code>, non HTML properties defined in 
 * <code>option</code>, calls <code>this._setEventProps</code>
 * and then calls <code>this._getOptionClassName</code>. 
 * Subclasses should implement <code>_getOptionClassName</code> if the default
 * behavior of <code>selectBase._getOptionClassName</code> is not appropriate.
 * Subclasses usually subclass this method to provide subclass specific CSS
 * class names and properties. For all the properties defined in the option
 * array see <code>_setOptions</code>
 * </p>
 * @param {Node} element The <code>option</code> HTML element DOM node
 * @param {Object} option Key-Value pairs of properties for the option node.
 * @config {boolean} disabled If true the option is disabled.
 * @config {String} label The option choice text.
 * @config {String} title The value of the option's HTML <code>title</code>
 * attribute.
 * @config {boolean} escape If false the label string will be evaluated
 * as HTML markup.
 * @config {boolean} separator If true this option acts as a separator.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._setOptionProps = function(element, option) {
    element.value = option.value;

    // If option.escape is true, we want the text to be displayed
    // literally. To achieve this behavior, do nothing.
    // If option.escape is false, we want any special sequences in the text 
    // (e.g., "&nbsp;") to be displayed as evaluated (i.e., unescaped).
    //
    // This is because it is assumed that the value of the property has 
    // already been escaped. This may be true if the value was set in the 
    // HTML response. This is why we call "unescapeHTML".
    //
    // However it may not be true if the text was set in javascript.
    //

    // Note that the HTML and ECMA JS DOM language binding indicates
    // that the "label" property should be used. However browsers do not
    // respect those specs. In fact label does not even work at all.
    // Setting the string on label results in no string being displayed.
    //
    var textToUse = option.label;
    if (option.escape != null && option.escape == false) {
	// Prototype method.
	element.text = this._proto._unescapeHTML(textToUse);
    } else {
	element.text = textToUse;
    }
    // We must record the state of the original option
    //
    element.label = option.label;

    // Never allow a disabled option to be spedified as selected
    // This is only a problem if an option is initially selected
    // and disbled on most browsers since once disabled it cannot
    // be selected except for IE. For IE it is ensured that a
    // disabled option cannot be "selected" in the
    // "change" method.
    //
    if (option.disabled != null && option.disabled == true) {
	option.selected = false;
    }

    element.defaultSelected = option.selected != null ? option.selected : false;

    // When creating new options, defaultSelected should also be set.
    // Actually only "defaultSelected" should be set and set only once
    // since it is supposed to be the "initial" value.
    // However all options are always recreated in this implementation
    // when this method is called by _setOptions.
    //
    element.selected = element.defaultSelected;
	
    element.disabled = option.disabled != null ? option.disabled : false;

    if (option.title != null) {
	element.title = option.title;
    }

    // option elements can have events.
    this._setEventProps(element, option);

    // In order for subclasses to return the appropriate
    // selector all the properties in the option properties array must
    // be placed on the HTML element option and optgroup instances.
    //
    // This is because "_getOptionClassName" is called in two different
    // contexts. It is called here where both the HTML option or
    // optgroup element is available and the original option properties 
    // are known, and during the "onchange" event where only the 
    // HTML option element is known. Therefore if the choice of selector 
    // is based on whether or not an option is a "separator" then, 
    // the separator state must be available from the HTML option or optgroup
    // element instance.
    //
    // This issue could be mitigated by using descendant selectors
    // vs. literal selectors. In other words, if every time an option
    // became disabled, the className attribute would be set (or merged) with 
    // Dis. A selector could be defined as ".Lst option.Dis" which would
    // implicitly match. If attribute selectors worked we could use
    // that instead. Either strategy would eliminate the need for
    // methods like "_getOptionClassName".
    //
    // Only set non HTML element props if we have to.
    //
    if (option.separator != null && option.separator == true) {
	element.separator = option.separator;
	element.disabled = true;
    }
    if (option.escape != null && option.escape == false) {
	element.escape = option.escape;
    }
    element.group = false;

    // Call _getOptionClassName only after setting HTML props
    // and option props on element.  Note that if this method
    // returns null, the option may not visually reflect
    // the true state.
    //
    var cn = this._getOptionClassName(element);
    if (cn != null) {
	element.className = cn;
    }
    return true;
};

/**
 * This function replaces all <code>option</code> and <code>optgroup</code>
 * elements in the HTML <code>select</code> element,
 * <code>this._listContainer</code>.
 * <p>
 * All options are replaced with the options specified in <code>props</code>.
 * If <code>props</code> is null, <code>false</code>
 * is returned and no change to the <code>select</code> element occurs. If
 * <code>props</code> contains no properties all options in the 
 * <code>select</code> element are removed.
 * </p>
 * @param {Object} props Key-Value pairs of option properties.
 * @config {String} label The text that appears as the choice in the 
 * rendered <code>option</code>.
 * @config {String} value The value for this option, which is submitted
 * in a request if this option is selected.
 * @config {boolean} separator If true this option represents a separator and
 * cannot be selected.
 * @config {boolean} group If true this option represents a group and
 * cannot be selected.
 * @config {boolean} escape If false the label string will be evaluated as
 * HTML markup.
 * @config {boolean} selected If true this option will be initially
 * selected.
 * @config {boolean} disabled If true this option will be initially 
 * disabled and cannot be selected.
 * @config {String} title The HTML title attribute text.
 * @return {boolean} true if successful; otherwise, false.
 * </p>
 * <p>
 * The option object may also define javascript event properties suitable for
 * an <code>option</code> or <code>optGroup</code> HTML element.
 * </p>
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._setOptions = function(props) {
    if (props == null) {
	return false;
    }

    // Remove all existing options
    //
    // Note: this has to be done. Otherwise, you'll get blank entries in the 
    // drop down for the original empty dojo attach point nodes.
    //

    if (!this._alreadyRemoved) {
        this._listContainer.removeChild(this._optionNode); 
        this._optionGroupContainer.removeChild(this._memberOptionNode);
        this._listContainer.removeChild(this._optionGroupContainer);
        this._alreadyRemoved = true;
    }

    // Cleaning up the old options
    //
    while (this._listContainer.firstChild) {
        this._listContainer.removeChild(this._listContainer.firstChild);
    }

    var thisNode;
    for (var i = 0; i < props.options.length; i++) {

	var pOption = props.options[i];

	var isie = woodstock4_3._base.browser._isIe();
	if (pOption.group == null || pOption.group == false) {

	    // For some reason, ie is prone to painting problems (esp. after a 
	    // style change) when using the DOM to populate options, but 
	    // apparently not when the options are in a group
	    //
	    // There is no working "clone" on "option" nodes in IE
	    // Manually get the attributes and then the "text"
	    // DOM attribute. I don't think an option can have HTML
	    // markup in the body, at least literally in the template
	    // so this strategy should be be sufficient enough to get
	    // the template's option attributes
	    //
	    if (isie) {
		thisNode = new Option();
		var len = this._optionNode.attributes.length;
		for (var j = 0; j < len; ++j) {
		    thisNode.setAttribute(this._optionNode.attributes[j].name,
			    this._optionNode.attributes[j].value);
		}
		// Need to manually do text, although this is likely null
		// in the template.
		//
		thisNode.text = this._optionNode.text;

	    } else {
		thisNode = this._optionNode.cloneNode(true);
	    }

	    // Set the properties on this option element
	    this._setOptionProps(thisNode, pOption);
	    
	    // Would it be better to create all the nodes and then
	    // add them to the "options" array or append then ?
	    // Granted this would mean iterating twice.
	    // But it might be better if something fails
	    // and if so leave the original list alone.
	    //

	    // Append this option node to the select
	    //
	    if (isie) {
		var idx = this._listContainer.options.length;
		var isSelected = thisNode.selected;
		this._listContainer.options[idx] = thisNode;

		// VERIFY THAT WE STILL NEED THIS
		//

		// explicitly set the selected property again!
		// this is necessary to work around a defect in 
		// some versions of IE6
		//
		this._listContainer.options[idx].selected = isSelected;
	    } else {
		this._listContainer.appendChild(thisNode);
	    }
	} else {
	    // group option optgroup
	    //
	    thisNode = this._optionGroupContainer.cloneNode(true);
	    
	    // Set the properties on this optgroup element
	    //
	    this._setGroupOptionProps(thisNode, pOption);
	    
	    // Add the option elements to this group
	    // Supports only one level
	    //
	    var thisSubNode;
	    for (var ix = 0; ix < pOption.options.length; ix++) {
		thisSubNode = this._memberOptionNode.cloneNode(true);
		this._setOptionProps(thisSubNode, pOption.options[ix]);
		thisNode.appendChild(thisSubNode); 
	    }

	    // Append this optgroup node to the select element
	    // only after constructing its option children
	    //
	    this._listContainer.appendChild(thisNode);
	}
    }
    return true;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.selectBase.prototype._setProps = function(props) {
    if (props == null) {
	return null;
    }

    // A web app devleoper could return false in order to cancel the 
    // auto-submit. Thus, we will handle this event via the onChange
    // call back.
    //
    if (props.onChange) {
        // Set private function scope on DOM node.
        this._listContainer._onchange = (typeof props.onChange == 'string')
	    ? new Function("event", props.onChange) 
	    : props.onChange;

        // Must be cleared before calling _setEventProps() below.
        props.onChange = null;
    }

    if (props.disabled != null) {
	if (this._listContainer.disabled != props.disabled) {
	    this._listContainer.disabled = props.disabled;
	}
    }

    // Add option and optgroup elements in the select element
    //
    if (props.options) {
        this._setOptions(props);
    }

    if (props.width != null && props.width != "") {
	this._listContainer.style.width = props.width;
    }
    // Set more properties.
    this._setCommonProps(this._listContainer, props);
    this._setEventProps(this._listContainer, props);

    // Set remaining properties.
    //
    return this._inherited("_setProps", arguments);
};

/** 
 * This function is used to set the selected option.
 *
 * @param {int} index The selected index of underlying HTML select element.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.selectBase.prototype.setSelectedIndex = function(index) {
    if (index >= 0 && index < this._listContainer.options.length) {
        this._listContainer.selectedIndex = index;
    }
    return true;
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.stateBase");

woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/** 
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.stateBase
 * @class This class contains functions for widgets that extend stateBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.stateBase");

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget._base.stateBase.event =
        woodstock4_3.widget._base.stateBase.prototype.event = {
    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         */
        beginTopic: undefined,

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         */
        endTopic: undefined
    }
};

/**
 * Process state event.
 * <p>
 * Note: If this.event.state.beginTopic and this.event.state.endTopic are 
 * not null, an event is published for custom Ajax implementations to listen 
 * for. If event topics are not implemented for this widget, the function 
 * returns and a message is output to the console.
 * </p>
 * @param {Object} props Key-Value pairs of widget properties to update.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.stateBase.prototype._stateChanged = function(props) {
    if (this.event == null || this.event.state == null
            || this.event.state.beginTopic == null
            || this.event.state.endTopic == null) {
        console.debug("Error: State event topics not implemented for " + 
            this._widgetType); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this._publish(this.event.state.beginTopic, [{
        id: this.id,
        endTopic: this.event.state.endTopic,
        props: props
    }]);
    return true;
};


woodstock4_3._dojo.provide("woodstock4_3.widget._base.submitBase");

woodstock4_3._dojo.require("woodstock4_3._base.proto");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/** 
 * This function is used to construct a base class.
 *
 * @constructor
 * @name woodstock4_3.widget._base.submitBase
 * @class This class contains functions for widgets that extend submitBase.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.submitBase");

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget._base.submitBase.event =
        woodstock4_3.widget._base.submitBase.prototype.event = {
    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** 
         * Submit event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         */
        beginTopic: undefined,

        /** 
         * Submit event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         */
        endTopic: undefined
    }
};

/**
 * Initialize public functions.
 * <p>
 * Note: If this.event.submit.beginTopic and this.event.submit.endTopic are
 * not null, a public function shall be added to the DOM node.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget._base.submitBase.prototype._initSubmitFunc = function () {
    if (this.event == null || this.event.submit == null
            || this.event.submit.beginTopic == null
            || this.event.submit.endTopic == null) {
        return false;
    }
    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Set public functions.
    /** @ignore */
    this._domNode.submit = function(execute) {
        return woodstock4_3.widget.common.getWidget(_id).submit(execute);    
    };
    return true;
};

/**
 * Process submit event.
 * <p>
 * Note: If this.event.submit.beginTopic and this.event.submit.endTopic are not
 * null, an event is published for custom Ajax implementations to listen for. If
 * event topics are not implemented for this widget, the function returns and a 
 * message is output to the console.
 * </p>
 * @param {String} execute Comma separated string containing a list of 
 * client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.submitBase.prototype.submit = function(execute) {
    if (this.event == null || this.event.submit == null
            || this.event.submit.beginTopic == null
            || this.event.submit.endTopic == null) {
        console.debug("Error: Submit event topics not implemented for " + 
            this._widgetType); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this._publish(this.event.submit.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: this.event.submit.endTopic
    }]);
    return true;
};
/**
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */

woodstock4_3._dojo.provide("woodstock4_3.widget._base.treeBase");

/**
 * This function is used to construct an accordion widget.
 *
 * @name woodstock4_3.widget.treeBase
 * @extends woodstock4_3.widget._base.widgetBase
 * @class This class contains the core functions for a tree widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {Object} rootNode The DOM node representing the root of this tree.
 * @config {boolean} multipleSelect True if multiple nodes can be selected.
 * @config {boolean} loadOnSelect Set to true if nodes can be loaded dynamically.
 * @config {Object} focusElement The DOM Node representing the element currently in focus.
 * @config {Object} submittedValue An Object representing the state of the tree. This data
 *  is should be sent to the server to synchronize the server side representation of the
 *  tree with the client side state of the tree.    
 */
woodstock4_3._dojo.declare("woodstock4_3.widget._base.treeBase",
        woodstock4_3.widget._base.widgetBase, {
    // Set defaults.
    constructor: function() {
	// If true more than one node can be selected.
	this.multipleSelect = false,
	// If true tree nodes can be loaded dynamically
	this.loadOnSelect = false;
	
        // the node currently on focus
        this.focusElement = null;
        // the root node, each tree can have atmost one of these.
        this.rootNode = null;
        
        // the set of values that would be submitted to a server
        // this reflects the state of the tree on the client side.
        
        this.submittedValue = {"selectedNodes": [],
                               "_selectedNodes": [],
                               "isExpanded": []};
    }
});

/**
 * Function to deselect a tree node. The node is removed from the list
 * of selected nodes.
 *
 * @param {String} id ID of the node to be deselected.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.treeBase.prototype.deSelectNode = function(nodeId) {

    // if multiple nodes can be selected what does deselect mean?
    // may be clicking on a selected node deselects it.
    for (var i=0; i<this.submittedValue.selectedNodes.length; i++) {
      if (this.submittedValue.selectedNodes[i] == nodeId) {
        delete this.submittedValue.selectedNodes[i];
      }
    }
    
    if (this.submittedValue._selectedNodes[nodeId]) {
        delete this.submittedValue._selectedNodes[nodeId];
    } else {
        this.submittedValue._selectedNodes[nodeId] = "selected";
    }
    var selectionInfo = "";
    for (var info in this.submittedValue._selectedNodes) {
        selectionInfo += info + "\n";
    }
    this._nodeSelectionState.value = selectionInfo;
    return true;
};

/**
 * This function is used to get list of selected nodes of the tree.
 * An array of selected nodes is returned.
 *
 * @return {Array} An array of selected nodes found during a preorder traversal.
 */
woodstock4_3.widget._base.treeBase.prototype.getSelectedNodes = function() {
    
    return this.submittedValue.selectedNodes;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget._base.treeBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // get the root
    if (this.rootNode != null) { props.rootNode = this.rootNode; }
    props.expanded = this.expanded;
    
    // get all children
    
    // Set properties.
    if (this.style != null) { props.style = this.style; }
    if (this.className != null) {props.className = this.className}
    return props;
};

/**
 * Function to deselect a tree node. The node is removed from the list
 * of selected nodes and the styles changed appropriately.
 *
 * @param {String} id ID of the node to be deselected.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.treeBase.prototype.isNodeSelected = function(id) {
    
    for (var i=0; i<this.submittedValue.selectedNodes.length; i++) {
      if (this.submittedValue.selectedNodes[i] == id) {
        return true;
      }
    }
    return false;
};

/**
 * Function to select a tree node. A tree maintains a list of
 * selected nodes which is available via the getSelectedNodes() 
 * function.
 *
 * @param {String} id ID of the node to be selected.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.treeBase.prototype.selectNode = function(nodeId) {
    
    if (this.multipleSelect) {
        var index = this.submittedValue.selectedNodes.length;
        this.submittedValue.selectedNodes[index] = nodeId;
        this.submittedValue._selectedNodes[nodeId] = "selected";
    } else {
        var oldNode = this.submittedValue.selectedNodes[0];
        delete this.submittedValue._selectedNodes[oldNode];
        this.submittedValue.selectedNodes[0] = nodeId;
        this.submittedValue._selectedNodes[nodeId] = "selected";
    }
    
    var selectionInfo = "";
    for (var info in this.submittedValue._selectedNodes) {
        selectionInfo += info + "\n";
    }
    this._nodeSelectionState.value = selectionInfo;
    return true;
};


/**
 * Function to toggle a tree node. This function saves the state of the node
 * in a hidden field that can be used to reconstruct the tree after a page
 * submit.
 *
 * @param {String} id The id of the node that is to be expanded.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget._base.treeBase.prototype.toggleNode = function(id) {
    
    var stateInfo = "";
    for (var state in this.submittedValue.isExpanded) {
        stateInfo += state + "\n";
    }
    this._nodeToggleState.value = stateInfo;
    return true;   
};


woodstock4_3.widget._base.treeBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }
    
    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
}




woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.accordion");



woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.common");

woodstock4_3._dojo.require("woodstock4_3.json");
woodstock4_3._dojo.require("woodstock4_3.widget.common");

/**
 * @class This class contains functions to obtain data asynchronously using the
 * XMLHTTP request object (XHR) the underlying protocol.
 * @static
 * @private
 */
woodstock4_3.widget._xhr.common = {
    /**
     * This function is used to generate a request using the XMLHTTP 
     * request object (XHR) the underlying protocol.
     *
     * Note: Unlike the woodstock4_3.xhr._doRequest function, the generated request
     * is always a "GET" and form data is submitted.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @param {String} id The widget Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure object provided to the callback function.
     * @param {Object} xjson The xjson header provided to the Ajax transaction.
     * @private
     */
    _doRequest: function(props) {
        if (props == null || props.id == null) {
            return false;
        }

        // Get properties for JavaScript closures.
        var _callback = props.callback ? props.callback : null;
        var _closure = props.closure ? props.closure : null;
        var _id = props.id ? props.id : null;
        var _url = woodstock4_3._base.config.ajax.url;
        var _xjson = props.xjson ? props.xjson : null;

        // Get form.
        var _form = woodstock4_3.widget.common._getForm(
            document.getElementById(props.id));
        if (_form == null) {
            _form = document.forms[0];
        }

        // Ensure URL has been provided.
        if (_url == null) {
            // Attempt to obtain default action url from form.
            if (_form && _form.action) {
                _url = _form.action;
            } else {
                console.warn("URL for Ajax transaction not provided. " +
                    "Set form action or Ajax config property");
                return false;
            }
        }

        // Generate AJAX request.
        woodstock4_3._dojo.xhrGet({
            error: function(content, ioArgs) {
                console.error("HTTP status code: ", ioArgs.xhr.status);
                return content;
            },
            form: _form,
            headers: {
                "X-JSON": woodstock4_3.json.stringify(_xjson),
                "X-Requested-With": "XMLHttpRequest",
                "X-Woodstock-Version": "4.3"
            },
            load: function(content, ioArgs) {
                if (typeof _callback == "function") {
                    _callback(_id, content, _closure, _xjson);
                }
                return content;
            },
            timeout: 5000, // Time in milliseconds
            url: _url
        });
    },

    /**
     * This function is used to process refresh events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @config {String} endTopic The event topic to publish.
     * @config {String} execute The string containing a comma separated list 
     * of client ids against which the execute portion of the request 
     * processing lifecycle must be run.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processRefreshEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Generate AJAX request.
        woodstock4_3.widget._xhr.common._doRequest({
            id: props.id,
            callback: woodstock4_3.widget._xhr.common._refreshCallback,
            closure: {
                endTopic: props.endTopic
            },
            xjson: {
                id: props.id,
                event: "refresh",
                execute: (props.execute) ? props.execute : "none"
            }
        });
        return true;
    },

    /**
     * This function is used to process state change events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @config {String} endTopic The event topic to publish.
     * @config {Object} props Key-Value pairs of widget properties to update.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processStateEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Generate AJAX request.
        woodstock4_3.widget._xhr.common._doRequest({
            id: props.id,
            callback: woodstock4_3.widget._xhr.common._stateCallback,
            closure: {
                endTopic: props.endTopic
            },
            xjson: {
                id: props.id,
                event: "state",
                execute: "none",
                props: props.props // Widget properties to update.
            }
        });
        return true;
    },

    /**
     * This function is used to process submit events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @config {String} endTopic The event topic to publish.
     * @config {String} execute The string containing a comma separated list 
     * of client ids against which the execute portion of the request 
     * processing lifecycle must be run.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processSubmitEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Generate AJAX request.
        woodstock4_3.widget._xhr.common._doRequest({
            id: props.id,
            callback: woodstock4_3.widget._xhr.common._submitCallback,
            closure: {
                endTopic: props.endTopic
            },
            xjson: {
                id: props.id,
                event: "submit",
                execute: (props.execute) ? props.execute : props.id
            }
        });
        return true;
    }, 
   
    /**
     * This function is used to refresh widgets.
     *
     * @param {String} id The HTML widget Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson header provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _refreshCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = woodstock4_3.json.parse(content);

        // Add rows.
        var widget = woodstock4_3.widget.common.getWidget(id);
        widget.setProps(props);

        // Publish an event for custom AJAX implementations to listen for.
        if (closure.endTopic) {
            woodstock4_3._dojo.publish(closure.endTopic, [props]);
        }
        return true;
    },

    /**
     * This function is a callback to respond to the end of state request.
     * It will only publish submit end event without updating the widget itself.
     *
     * @param {String} id The HTML widget Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson header provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _stateCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = woodstock4_3.json.parse(content);
            
        // Publish an event for custom AJAX implementations to listen for.
        if (closure.endTopic) {
            woodstock4_3._dojo.publish(closure.endTopic, [props]);
        }
        return true;
    },

    /**
     * This function is a callback to respond to the end of submit request.
     * It will only publish submit end event without updating the widget itself.
     *
     * @param {String} id The HTML widget Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson header provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _submitCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = woodstock4_3.json.parse(content);
            
        // Publish an event for custom AJAX implementations to listen for.
        if (closure.endTopic) {
            woodstock4_3._dojo.publish(closure.endTopic, [props]);
        }
        return true;
    }
};


woodstock4_3._dojo.provide("woodstock4_3.widget.accordion");



woodstock4_3._dojo.provide("woodstock4_3.widget.accordionTab");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct an accordionTab widget.
 *
 * @constructor
 * @name woodstock4_3.widget.accordionTab
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the accordionTab widget.
 * <p>
 * The accordionTab widget represents a container within which an arbitrary 
 * number of components can be placed. These components will render within the 
 * content area of the given tab inside the accordion. The accordionTab widget 
 * should be configured to have a label that will appear in the accordion body
 * even when the tab is not selected. The height of the content area can also be
 * specified. Tabs with larger content can have greater height.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "acc1",
 *       toggleControls: true,
 *       tabs: [{
 *         id: "tab1",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st1",
 *           value: "This is tab one",
 *           widgetType: "text"
 *         }],
 *         title: "Tab One",
 *         widgetType: "accordionTab"
 *       }, {
 *         id: "tab2",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st2",
 *           value: "This is tab two",
 *           widgetType: "text"
 *         }],
 *         title: "Tab Two",
 *         widgetType: "accordionTab"
 *       }],
 *       widgetType: "accordion"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * visible state is either enabled or disabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "acc1",
 *       toggleControls: true,
 *       tabs: [{
 *         id: "tab1",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st1",
 *           value: "This is tab one",
 *           widgetType: "text"
 *         }],
 *         title: "Tab One",
 *         widgetType: "accordionTab"
 *       }, {
 *         id: "tab2",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st2",
 *           value: "This is tab two",
 *           widgetType: "text"
 *         }],
 *         title: "Tab Two",
 *         widgetType: "accordionTab"
 *       }],
 *       widgetType: "accordion"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Accordion Tab State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("tab1"); // Get accordionTab
 *       return widget.setProps({visible: !domNode.getProps().visible}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the accordionTab is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "acc1",
 *       toggleControls: true,
 *       tabs: [{
 *         id: "tab1",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st1",
 *           value: "This is tab one",
 *           widgetType: "text"
 *         }],
 *         title: "Tab One",
 *         widgetType: "accordionTab"
 *       }, {
 *         id: "tab2",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st2",
 *           value: "This is tab two",
 *           widgetType: "text"
 *         }],
 *         title: "Tab Two",
 *         widgetType: "accordionTab"
 *       }],
 *       widgetType: "accordion"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Accordion Tab" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("tab1"); // Get accordionTab
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a accordionTab using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the accordionTab text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "acc1",
 *       toggleControls: true,
 *       tabs: [{
 *         id: "tab1",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st1",
 *           value: "This is tab one",
 *           widgetType: "text"
 *         }],
 *         title: "Tab One",
 *         widgetType: "accordionTab"
 *       }, {
 *         id: "tab2",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st2",
 *           value: "This is tab two",
 *           widgetType: "text"
 *         }],
 *         title: "Tab Two",
 *         widgetType: "accordionTab"
 *       }],
 *       widgetType: "accordion"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Accordion Tab Title" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("tab1"); // Get accordionTab
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "acc1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.accordionTab.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {int} contentHeight CSS selector.
 * @config {String} hiddenField Field set to true if tab is selected.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Array} tabContent The content area of the tab.
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.accordionTab", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.isContainer = true;
        this.selected = false;
        this.focusState = "title";
    },
    _widgetType: "accordionTab" // Required for theme properties.    
});

/**
 * The callback function for key press on the accordion tabs.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._createOnKeyDownCallBack = function() {
    var _id = this.id;
    return function(event) {
        var elem = document.getElementById(_id);
        if (elem == null) {
            return false;
        }
        var common = woodstock4_3.widget.common;
        var widget = common.getWidget(_id);

        event = common._getEvent(event);
        var keyCode = common._getKeyCode(event);

        // if onkeypress returns false, we do not traverse the menu.
        var keyPressResult = true;
        if (this._onkeypress) {
            keyPressResult = (this._onkeypress) ? this._onkeypress() : true;
        }        
       
        if (keyPressResult != false) {
            widget._traverseMenu(keyCode, event, _id);
        }
        return true;
   };
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.accordionTab.event =
        woodstock4_3.widget.accordionTab.prototype.event = {
    /**
     * This object contains load event topics.
     * @ignore
     */
    load: {
        /** 
         * Load event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordionTab.event.load.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_accordionTab_event_load_begin",

        /** 
         * Load event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordionTab.event.load.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_accordionTab_event_load_end"
    },

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordionTab.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_accordionTab_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordionTab.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_accordionTab_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** 
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordionTab.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_accordionTab_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordionTab.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_accordionTab_event_state_end"
    },

    /**
     * This object contains title event topics.
     * @ignore
     */
    title: {
        /** 
         * Action event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordionTab.event.title.beginTopic
         * @property {String} beginTopic
         */
        selectedTopic: "woodstock4_3_widget_accordionTab_event_tab_selected"
    }
};

/**
 * Process load event.
 *
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._loadContent = function(execute) {
    // Publish event.
    this._publish(woodstock4_3.widget.accordionTab.event.load.beginTopic, [{
        id: this.id
    }]);
    return true;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.accordionTab.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.selected != null) { props.selected = this.selected; }
    if (this.title != null) { props.title = this.title; }
    if (this.tabContent != null) { props.tabContent = this.tabContent; }
    if (this.visible != null) { props.visible = this.visible; }
    if (this.actions != null) { props.actions = this.actions; }
    if (this.className != null) { props.className = this.className; }
    if (this.style != null) { props.style = this.style; }
    if (this.contentHeight != null) { props.contentHeight = this.contentHeight; }
    if (this.id != null) { props.id = this.id; }
    if (this.tabContent != null) { props.tabContent = this.tabContent; }
    if (this.focusId != null) { props.focusId = this.focusId; }
    if (this.type != null) { props.type = this.type; }

    return props;
};

/**
 * Handle menu onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._onMenuClickCallback = function(event) {
    this._dojo.stopEvent(event);
    return true;
};

/**
 * Handle title onClick event.
 * <p>
 * This function selects the child tab when the user clicks on its label. The 
 * actual behavior of the accordion depends on multipleSelect being enabled.
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._onTitleClickCallback = function (event) {
    this._publish(woodstock4_3.widget.accordionTab.event.title.selectedTopic, [{
        id: this.id
    }]);
    if (this._titleContainer.focus) {
        this._titleContainer.focus();
    }
    return true;
};

/**
 * Handle title onMouseOut event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._onTitleMouseOutCallback = function(event) {
    if (this.selected) {
        this._titleContainer.className = this._theme.getClassName("ACCORDION_TABEXPANDED");
        this._turnerContainer.className = this._theme.getClassName("ACCORDION_DOWNTURNER");
        if (this._titleContainer.focus) {
            this._titleContainer.focus();
        }
        return true;
    }
    this._titleContainer.className = this._theme.getClassName("ACCORDION_TABCOLLAPSED");
    this._turnerContainer.className = this._theme.getClassName("ACCORDION_RIGHTTURNER");
    return true;
};

/**
 * Handle title onMouseOver event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._onTitleMouseOverCallback = function(event) {
    if (this.selected) {
        this._turnerContainer.className = this._theme.getClassName("ACCORDION_DOWNTURNER");
    } else {
        this._turnerContainer.className = this._theme.getClassName("ACCORDION_RIGHTTURNER");
    }
    if (this._titleContainer.focus) {
        this._titleContainer.focus();
    }
    return true;
};

/**
 * Handle the case when the _contentNode is about to lose focus. The tabIndex for the tab 
 * and the _contentNode should be set to -1 and the focusElement should also be nulled.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._onContentEndCallback = function(event) {
    this.focusState = "end";
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._domNode.id = this.id;
        this._titleNode.id = this.id + "_tabTitle";
        this._turnerContainer.id = this.id + "_tabTitleTurner";
        this._menuContainer.id = this.id + "_tabMenu";
        this._hiddenFieldNode.id = this.id + ":selectedState";
        this._hiddenFieldNode.name = this._hiddenFieldNode.id;
        this._domNode.tabIndex = -1;
        this._contentEnd.id = this.id + "_contentEnd";
    }

    // Set style classes.
    this._titleContainer.className = this._theme.getClassName("ACCORDION_TABCOLLAPSED");
    this._turnerContainer.className = this._theme.getClassName("ACCORDION_RIGHTTURNER");
    this._menuContainer.className = this._theme.getClassName("HIDDEN");
    this._titleNode.className = this._theme.getClassName("ACCORDION_TABTITLE");
    this._contentNode.className = this._theme.getClassName("ACCORDION_TABCONTENT");

    // Set public functions.
    // TBD...

    // Set events.
    this._dojo.connect(this._titleContainer, "onclick", this, "_onTitleClickCallback");
    this._dojo.connect(this._titleContainer, "onmouseover", this, "_onTitleMouseOverCallback");
    this._dojo.connect(this._turnerContainer, "onmouseover", this, "_onTitleMouseOverCallback");
    this._dojo.connect(this._turnerContainer, "onclick", this, "_onTitleClickCallback");
    this._dojo.connect(this._menuContainer, "onmouseover", this, "_onTitleMouseOverCallback");
    this._dojo.connect(this._menuContainer, "onclick", this, "_onMenuClickCallback");
    this._dojo.connect(this._titleContainer, "onmouseout", this, "_onTitleMouseOutCallback");
    this._dojo.connect(this._turnerContainer, "onmouseout", this, "_onTitleMouseOutCallback");
    this._dojo.connect(this._menuContainer, "onmouseout", this, "_onTitleMouseOutCallback");
    this._dojo.connect(this._contentEnd, "onblur", this, "_onContentEndCallback");
    //Create callback function for onkeydown event.
    this._dojo.connect(this._domNode, "onkeydown", this._createOnKeyDownCallBack()); 
    
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.accordionTab.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.tabContent) {
        this.tabContent = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    if (props.tabIndex == null) {
        props.tabIndex = -1;
        this.tabIndex = -1;
    } else { //new code
        this._domNode.tabIndex = props.tabIndex;
        this._titleContainer.tabIndex = props.tabIndex;
        this._contentNode.tabIndex = props.tabIndex;
        this._contentEnd.tabIndex = this.tabIndex;
        this.tabIndex = props.tabIndex;
    }  // end of new code.

    // Set properties.
    if (props.contentHeight) {
        this._contentNode.style.height = props.contentHeight;
    }

    if (props.title != null) {
        this._setTitle(props.title);
    }

    if (props.tabContent) {
        this._setTabContent(props.tabContent);
        if (this.selected) {
            this._hiddenFieldNode.value = "true";
            this._titleContainer.className = this._theme.getClassName("ACCORDION_TABEXPANDED");
            this._turnerContainer.className = this._theme.getClassName("ACCORDION_DOWNTURNER");
            this._contentNode.style.display = "block";
        } else {
            this._hiddenFieldNode.value = "false";
            this._titleContainer.className = this._theme.getClassName("ACCORDION_TABCOLLAPSED");
            this._turnerContainer.className = this._theme.getClassName("ACCORDION_RIGHTTURNER");
            this._contentNode.style.display = "none";
        }
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    if (props.focusId != null) {
        this.focusId = props.focusId;
        if (this.selected) {
            if (this.focusState == "content") {
                var focusElem = document.getElementById(this.focusId);
                if (focusElem.focus) {
                    focusElem.focus();
                }
            }
        } else {
            this.focusState = "title";
        }
    }
    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * Set tab selected.
 *
 * @param {boolean} isSelected true if selected.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._setSelected = function (isSelected) {
    if (this.selected) {
        this.selected = false;
    } else {
        this.selected = isSelected;
    }

    if (this.selected) {
        this._hiddenFieldNode.value = "true";
        this._titleContainer.className = this._theme.getClassName("ACCORDION_TABEXPANDED");
        this._turnerContainer.className = this._theme.getClassName("ACCORDION_DOWNTURNER");
        this._contentNode.style.display = "block";

        // if the tab does not have content and "loadOnSelect" is set
        // to true go ahead and refresh the widget. 
        if (!this.tabContent) {
            if (this.parent.loadOnSelect) {
                this._loadContent();
            }
        }
    } else {
        this._hiddenFieldNode.value = "false";
        this._titleContainer.className = this._theme.getClassName("ACCORDION_TABCOLLAPSED");
        this._turnerContainer.className = this._theme.getClassName("ACCORDION_RIGHTTURNER");
        this._contentNode.style.display = "none";
    }
    return true;
};

/**
 * Set the contents of the accordion tab.
 *
 * @param {Array} content The Contents of the tab body.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._setTabContent = function(content) {
    if (content) {
        this._widget._removeChildNodes(this._contentNode);
        for (var i = 0; i < content.length; i++) {
            this._widget._addFragment(this._contentNode, content[i], "last");
        }
    }
    return true;
};

/**
 * Set the title associated with the accordion tab.
 *
 * @param {String} title Title property associated with the tab.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._setTitle = function (title) {
    if (title) {
        // NOTE: If you set this value manually, text must be HTML escaped.
        this._widget._addFragment(this._titleNode, {
                id: this.id + "_titleLink",
                title: title,
                onClick: "return false;",
                className: "",
                contents: [title],
                widgetType: "hyperlink"
        });
    }
    return true;
};

/**
 * This function takes care of traversing through the accordionTab depending
 * on which key is pressed. If tab is open and current focus is on the title, 
 * the focus should move to the first element in the the open tab. Else it 
 * should simply move out accordion.
 * @param (String) keyCode The value of the key which was pressed
 * @param (Event) event The key press event.
 * @param (String) nodeId The id of the accordionTab. 
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordionTab.prototype._traverseMenu = function(keyCode, event, nodeId) {
    // The focus can either be on the title or the content.
    // If tab is open and current focus is on the title, the focus should move
    // to the first element in the the open tab. Else it should simply move out 
    // accordion.

    // do nothing if only shift is pressed.
    if (event.shiftKey && keyCode != 9) {  
        return true;
    }

    // handle the tab case.
    if (keyCode == 9) {         

        // shift+tab is handled by the parent accordion, simply remove focus
        // from this tab and set tabIndex to -1.
        if (event.shiftKey) {
            
            this.focusState == "title";
            if (this._contentNode.blur) {
                this._contentNode.blur();
            }
            if (this._titleContainer.blur) {
                this._titleContainer.blur();
            }
            // this.focusId = null;
            // this._domNode.tabIndex = -1;  //new
            return true;
        }

        // If tab is open and focus is on the title section
        // shift focus to the content section, else move out of the
        // tab.
        if (this.selected) {
            if (this.focusState == "title") {
                this._contentNode.tabIndex = this.tabIndex;
                this._contentEnd.tabIndex = this.tabIndex;
                this._contentNode.focus();
                if (this.focusId == null) {
                    var fChild = this._contentNode.firstChild;
                    if (fChild) {
                        if (fChild.focus) {
                            fChild.focus();
                        }
                        this.focusId = fChild;
                    }
                    return true;
                } else {
                    var focusElem = document.getElementById(this.focusId);
                    if (focusElem) {
                        if (focusElem.focus) {
                            focusElem.focus();
                        }
                    }
                }
                this.focusState = "content";
            } else if (this.focusState == "content") {
                return true;
            } else {
                this._contentNode.focus();
                if (this.focusId == null) {
                    this._contentNode.tabIndex = this.tabIndex;
                    return true;
                } else {
                    var focusElem = document.getElementById(this.focusId);
                    if (focusElem) {
                        if (focusElem.focus) {
                            focusElem.focus();
                        }
                    }
                }
                this.focusState = "content";
            }
        } else {
            // reset the tabIndex as control is moving out of the tab.
            // this._domNode.tabIndex = -1; //new
            return true;
        }
    } else if (keyCode == 38 && event.ctrlKey) {  // handle ctrl + up

        // Move focus to the header of the accordion tab if its contents is
        // currently in focus. Else, do nothing. Also, stop event propagation.

        if (this.focusState == "content") {
            this.focusState == "title";
            if (this._contentNode.blur) {
                this._contentNode.blur();
            }
            if (this._titleContainer.focus) {
                this._titleContainer.focus();
            }
            if (woodstock4_3._base.browser._isIe5up()) {
                window. event.cancelBubble = true;
                window.event.returnValue = false;
            } else {
                event.stopPropagation();
                event.preventDefault();
            }
        }
    }
    return true;
};
woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct an accordion widget.
 *
 * @constructor
 * @name woodstock4_3.widget.accordion
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the accordion widget. 
 * <p>
 * The accordion widget can be thought of as a vertical tab set. This set can
 * contain one or more tabs each of which can contain virtually any HTML markup
 * and have the following characteristics:
 * <ul><li>
 * Each tab contains links to different sections(tasks) of the application.
 * </li><li>
 * Only one accordion tab can be open at any given time.
 * </li></ul>
 * The accordion widget allows muliple tabs to be open simultaneously. Consider 
 * using an alternate navigational component if the number of tabs in the 
 * accordion exceeds ten. The accordion tag also supports "expandAll" and 
 * "collapseAll" buttons. Unless otherwise specified the accordion dows not 
 * display the expand all, collapse all and refresh icons. To display these 
 * icons, set the toggleControls and refreshIcon attributes to true. Note that 
 * an accordion can be refreshed if the application using it supports this 
 * feature.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "acc1",
 *       toggleControls: true,
 *       tabs: [{
 *         id: "tab1",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st1",
 *           value: "This is tab one",
 *           widgetType: "text"
 *         }],
 *         title: "Tab One",
 *         widgetType: "accordionTab"
 *       }, {
 *         id: "tab2",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st2",
 *           value: "This is tab two",
 *           widgetType: "text"
 *         }],
 *         title: "Tab Two",
 *         widgetType: "accordionTab"
 *       }],
 *       widgetType: "accordion"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * multipleSelect state is either enabled or disabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "acc1",
 *       toggleControls: true,
 *       tabs: [{
 *         id: "tab1",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st1",
 *           value: "This is tab one",
 *           widgetType: "text"
 *         }],
 *         title: "Tab One",
 *         widgetType: "accordionTab"
 *       }, {
 *         id: "tab2",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st2",
 *           value: "This is tab two",
 *           widgetType: "text"
 *         }],
 *         title: "Tab Two",
 *         widgetType: "accordionTab"
 *       }],
 *       widgetType: "accordion"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Accordion State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("acc1"); // Get accordion
 *       return widget.setProps({multipleSelect: !domNode.getProps().multipleSelect}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the accordion is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "acc1",
 *       toggleControls: true,
 *       tabs: [{
 *         id: "tab1",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st1",
 *           value: "This is tab one",
 *           widgetType: "text"
 *         }],
 *         title: "Tab One",
 *         widgetType: "accordionTab"
 *       }, {
 *         id: "tab2",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st2",
 *           value: "This is tab two",
 *           widgetType: "text"
 *         }],
 *         title: "Tab Two",
 *         widgetType: "accordionTab"
 *       }],
 *       widgetType: "accordion"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Accordion" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("acc1"); // Get accordion
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a accordion using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the accordion text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "acc1",
 *       toggleControls: true,
 *       tabs: [{
 *         id: "tab1",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st1",
 *           value: "This is tab one",
 *           widgetType: "text"
 *         }],
 *         title: "Tab One",
 *         widgetType: "accordionTab"
 *       }, {
 *         id: "tab2",
 *         contentHeight: "50px",
 *         tabContent: [{
 *           id: "st2",
 *           value: "This is tab two",
 *           widgetType: "text"
 *         }],
 *         title: "Tab Two",
 *         widgetType: "accordionTab"
 *       }],
 *       widgetType: "accordion"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Accordion Title" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("acc1"); // Get accordion
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "acc1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.accordion.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {Object} collapseAllImage Image associated with collapse all icon
 * @config {Object} expandAllImage Image associated with expand all icon
 * @config {String} id Uniquely identifies an element within a document.
 * @config {boolean} isRefreshIcon True if refresh icon should be set
 * @config {boolean} multipleSelect Set to true if multiple tabs can be selected
 * @config {Object} refreshImage Image associated with refresh icon.
 * @config {String} style Specify style rules inline.
 * @config {Array} tabs Tabs constituting the accordion's children
 * @config {boolean} toggleControls Set to true if expand/collapse icons should be set.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.accordion", [
        woodstock4_3.widget._base.refreshBase,
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.duration = 250;
        this.isContainer = true;
        this.loadOnSelect = false;
        this.multipleSelect = false;
        this.focusElement = null;
    },
    _widgetType: "accordion" // Required for theme properties.
});

/**
 * Helper function to add accordion header controls
 *
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._addControls = function(props) {       
    // Add expand and collapse icons only if multiple select is set to
    // true and the icons have been supplied.
    if (props.toggleControls && props.multipleSelect) {
        // Set expand all image properties.
        if (props.expandAllImage) {
            // Update/add fragment.
            this._widget._updateFragment(this._expandAllImgContainer, 
                this.expandAllImage.id, props.expandAllImage);
        }

        // Set collapse all image properties.
        if (props.collapseAllImage) {
            // Update/add fragment.
            this._widget._updateFragment(this._collapseAllImgContainer,
                this.collapseAllImage.id, props.collapseAllImage);
        }
        
        // a divider should only be added if expand/collapse icons exist.
        this._dividerNodeContainer.className = this._theme.getClassName("ACCORDION_HDR_DIVIDER");
    }

    // Set refresh image properties.
    if (props.isRefreshIcon && props.refreshImage) {
        // Update/add fragment.
        this._widget._updateFragment(this._refreshImgContainer,
            this.refreshImage.id, props.refreshImage);
    }
    return true;
};

/**
 * Close all open accordions and leave the others as is.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._collapseAllTabs = function(event) {
    // Iterate over all tabs.
    for (var i = 0; i < this.tabs.length; i++) {
        var widget = this._widget.getWidget(this.tabs[i].id);
        if (widget && widget.selected) {
            widget._setSelected(false);
        }
    }
    this._updateFocus(this._collapseAllContainer);
    return true;
};

/**
 * The callback function for key press on the accordion header.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._createOnKeyDownCallBack = function() {
    var _id = this.id;
    return function(event) {
        var elem = document.getElementById(_id);
        if (elem == null) {
            return false;
        }
        var common = woodstock4_3.widget.common;
        var widget = common.getWidget(_id);

        event = common._getEvent(event);
        var keyCode = common._getKeyCode(event);
        
        // if onkeypress returns false, we do not traverse the accordion
        var keyPressResult = true;
        if (this._onkeypress) {
            keyPressResult = (this._onkeypress) ? this._onkeypress() : true;
        }        
       
        if (keyPressResult != false) {
            widget._traverseMenu(keyCode, event, _id);
        }
        return true;
   };
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.accordion.event = 
        woodstock4_3.widget.accordion.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordion.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_accordion_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordion.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_accordion_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordion.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_accordion_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.accordion.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_accordion_event_state_end"
    }
};

/**
 * Open all closed tabs and leave the others as is.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._expandAllTabs = function(event) {
    // Iterate over all tabs.
    for (var i = 0; i < this.tabs.length; i++) {
        var widget = this._widget.getWidget(this.tabs[i].id);
        if (widget && !widget.selected) {
            widget._setSelected(true);
        }
    }
    this._updateFocus(this._expandAllContainer);
    return true;
};

/**
 * Set the appropriate focus before invoking tabSelected. This
 * function has been put in place because tabSelected is called
 * from various places but the focus does not have to be set
 * in all the cases.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._focusAndSelectTab = function(props) {
    // Iterate over all tabs to ensure id is valid.
    for (var i = 0; i < this.tabs.length; i++) {
        if (props.id == this.tabs[i].id) {
            this.focusElement = this.tabs[i];
            break;   
        }
    }
    this.tabSelected(props);
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.accordion.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.collapseAllImage != null) { props.collapseAllImage = this.collapseAllImage; }
    if (this.expandAllImage != null) { props.expandAllImage = this.expandAllImage; }
    if (this.isRefreshIcon != null) { props.isRefreshIcon = this.isRefreshIcon; }
    if (this.loadOnSelect != null) { props.loadOnSelect = this.loadOnSelect; }
    if (this.multipleSelect != null) { props.multipleSelect = this.multipleSelect; }
    if (this.refreshImage != null) { props.refreshImage = this.refreshImage; }
    if (this.style != null) { props.style = this.style; }
    if (this.tabs != null) { props.tabs = this.tabs; }
    if (this.toggleControls != null) { props.toggleControls = this.toggleControls; }
    if (this.type != null) { props.type = this.type; }
 
    return props;
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.accordion.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var newClassName = this._theme.getClassName("ACCORDION_DIV", "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._domNode.id = this.id;
        this._headerContainer.id = this.id + "_accHeader";
        this._expandAllContainer.id = this.id + "_expandAllNode";
        this._expandAllImgContainer.id = this._expandAllContainer.id + "_expandAllImage";
        this._collapseAllContainer.id = this.id + "_collapseAllNode";
        this._collapseAllImgContainer.id = this._collapseAllImgContainer.id + "_collapseAllImage";
        this._dividerNodeContainer.id = this.id + "_dividerNode";
        this._refreshNodeContainer.id = this.id + "_refreshNode";
    }

    // Set class names.
    this._headerContainer.className = this._theme.getClassName("ACCORDION_HDR");
    this._collapseAllContainer.className = this._theme.getClassName("ACCORDION_HDR_CLOSEALL");
    this._expandAllContainer.className = this._theme.getClassName("ACCORDION_HDR_OPENALL");
    
    // the divider should initially be hidden
    this._dividerNodeContainer.className = this._theme.getClassName("HIDDEN");
    this._refreshNodeContainer.className = this._theme.getClassName("ACCORDION_HDR_REFRESH");

    // Set events.
    var _id = this.id;
    this._dojo.connect(this._collapseAllContainer, "onclick", this, "_collapseAllTabs");
    this._dojo.connect(this._expandAllContainer, "onclick", this, "_expandAllTabs");
    this._dojo.connect(this._refreshNodeContainer, "onclick", function(event) {
        // New literals are created every time this function is called, and it's 
        // saved by closure magic.
        var widget = woodstock4_3.widget.common.getWidget(_id);
        widget._updateFocus(this._refreshNodeContainer);
        widget.refresh(_id);
        return false;
    });
    
    // Create callback function for onkeydown event.
    this._dojo.connect(this._domNode, "onkeydown", this._createOnKeyDownCallBack());
    
    // Subscribe to the "tab selected" event present in the accordion widget.
    this._widget.subscribe(woodstock4_3.widget.accordionTab.event.title.selectedTopic,
        this, "_focusAndSelectTab");

    // Generate the accordion header icons on the client side.
    if (this.toggleControls && this.multipleSelect) {
        if (this.expandAllImage == null) {
            this.expandAllImage = {
                id: this.id + "_expandImageLink",
                onClick: "return false;",
                enabledImage: {
                    icon: "ACCORDION_EXPAND_ALL",
                    id: this.id + "_expandAll",
                    widgetType: "image"
                },
                title: this._theme.getMessage("Accordion.expandAll"),
                widgetType: "imageHyperlink"
            };
        }
        
        if (this.collapseAllImage == null) {
            this.collapseAllImage = {
                id: this.id + "_collapseImageLink",
                onClick: "return false;",
                enabledImage: {
                    icon: "ACCORDION_COLLAPSE_ALL",
                    id: this.id + "_collapseAll",
                    widgetType: "image"
                },
                title: this._theme.getMessage("Accordion.collapseAll"),
                widgetType: "imageHyperlink"
            };
        }
    }

    // Set refresh image hyperlink properties.
    if (this.isRefreshIcon) {
        if (this.refreshImage == null) {
            this.refreshImage = {
                id: this.id + "_refreshImageLink",
                onClick: "return false;",
                enabledImage: {
                    icon: "ACCORDION_REFRESH",
                    id: this.id + "_refresh",
                    widgetType: "image"
                },
                title: this._theme.getMessage("Accordion.refresh"),
                widgetType: "imageHyperlink"
            };
         }
    }
    
    if (this.isRefreshIcon) {
        this._refreshNodeContainer.tabIndex = this.tabIndex;
    }
    if (this.toggleControls && this.multipleSelect) {
            this._expandAllContainer.tabIndex = this.tabIndex;
            this._collapseAllContainer.tabIndex = this.tabIndex;
    }
    if (this.tabs.length > 0) {
        for (var i=0; i< this.tabs.length; i++) {
            this.tabs[i].tabIndex = this.tabIndex;
        }
    }

    with (this._domNode.style) {
        if (position != "absolute") {
            position = "relative";
        }
        overflow = "hidden";
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.accordion.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace tabs -- do not extend.
    if (props.tabs) {
        this.tabs = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    if (props.tabIndex == null) {
        props.tabIndex = -1;
        this.tabIndex = -1;
    }

    // add control icons - refresh, expandall, collapseall.
    this._addControls(props); 

    // If we are coming here for
    // the first time there will be no children. The other case is when 
    // the accordion is being rerendered because of a refresh in which 
    // we want to use the latest set of children. _addFragment is supposed
    // to do that.
    if (props.tabs) {
        // Remove child nodes.
        this._widget._removeChildNodes(this._tabsContainer);

        // set the tab focus 
        var tabFocus = true;
        
        if (props.tabs.length > 0) {
            for (var i=0; i< props.tabs.length; i++) {
                props.tabs[i].tabIndex = this.tabIndex;
            }
        }
       
        // Add tabs.
        for (var i = 0; i < props.tabs.length; i++) {
            this._widget._addFragment(this._tabsContainer, props.tabs[i], "last");
        }
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * Set a different look for refresh/expand/collapse icons when focus is set
 * on them.
 *
 * @param {String} nodeId The node ID of the icon.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._setFocusStyleClass = function(nodeId) {
    if (nodeId == this._collapseAllContainer.id) {
        //set focus style on collapseNode
        this._collapseAllContainer.className = this._theme.getClassName("ACCORDION_HDR_CLOSEALL_FOCUS");
    } else if (nodeId == this._expandAllContainer.id) {
        //set focus style on expandNode
        this._expandAllContainer.className = this._theme.getClassName("ACCORDION_HDR_OPENALL_FOCUS");
    } else if (nodeId == this._refreshNodeContainer.id) {
        //set focus style on refreshNode
        this._refreshNodeContainer.className = this._theme.getClassName("ACCORDION_HDR_REFRESH_FOCUS");
    }
    return true;
};

/**
 * Reset the styles asscociated with refresh/expand/collapse icons when these 
 * icons are blurred.
 *
 * @param {String} nodeId The node ID of the icon.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._setBlurStyleClass = function(nodeId) {
    if (nodeId == this._collapseAllContainer.id) {
        //set normal className on collapseNode
        this._collapseAllContainer.className = this._theme.getClassName("ACCORDION_HDR_CLOSEALL");
    } else if (nodeId == this._expandAllContainer.id) {
        //set normal className on expandNode
        this._expandAllContainer.className = this._theme.getClassName("ACCORDION_HDR_OPENALL");
    } else if (nodeId == this._refreshNodeContainer.id) {
        //set normal className on refreshNode
        this._refreshNodeContainer.className = this._theme.getClassName("ACCORDION_HDR_REFRESH");
    }
    return true;
};

/**
 * Set appropriate styles when a tab is in focus.
 *
 * @param {String} nodeId The node ID of the icon.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._setTabFocus = function(nodeId) {
    // update the tab with the appropriate tabIndex
    var tabWidget = this._widget.getWidget(nodeId);
    var props = {tabIndex: this.tabIndex};
    this._widget._updateFragment(this._tabsContainer, nodeId, props);

    // set the style class to indicate that the tab is in focus.
    if (tabWidget.selected) {
        tabWidget._titleContainer.className = this._theme.getClassName("ACCORDION_TABEXPANDED_FOCUS");
    } else {
        tabWidget._titleContainer.className = this._theme.getClassName("ACCORDION_TABCOLLAPSED_FOCUS");
    }
    tabWidget._domNode.focus();
    return true;
};

/**
 * Set appropriate styles when a tab loses focus.
 *
 * @param {String} nodeId The node ID of the icon.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._setTabBlur = function(nodeId) {
    // update the tab with the appropriate tabIndex
    var tabWidget = this._widget.getWidget(nodeId);
    
    if (tabWidget.selected) {
        tabWidget._titleContainer.className = this._theme.getClassName("ACCORDION_TABEXPANDED");
        tabWidget._turnerContainer.className = this._theme.getClassName("ACCORDION_DOWNTURNER");
    } else { 
        tabWidget._titleContainer.className = this._theme.getClassName("ACCORDION_TABCOLLAPSED");
        tabWidget._turnerContainer.className = this._theme.getClassName("ACCORDION_RIGHTTURNER");
    }    

    if (tabWidget) {
        tabWidget._titleContainer.blur();
    }
    return true;
};

/**
 * Process tab selected events.
 *
 * @param props Key-Value pairs of properties.
 * @config {String} id The id of the selected tab.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype.tabSelected = function(props) {
    var widget = null;

    // Iterate over all tabs to ensure id is valid.
    for (var i = 0; i < this.tabs.length; i++) {
        if (props.id == this.tabs[i].id) {
            widget = this._widget.getWidget(this.tabs[i].id);
            break;   
        }
    }
    
    // Return if id was not valid.
    if (widget == null) {
            return false;
    }
    
    if (this.multipleSelect) {
        widget._setSelected(true);
    } else {
        for (var i = 0; i < this.tabs.length; i++) {
            widget = this._widget.getWidget(this.tabs[i].id);
            if (widget) {
                widget._setSelected(props.id == this.tabs[i].id);
                this.tabs[i] = widget.getProps();
            }
        }
    }
    return true;
};

/**
 * This function traverses through the accordion depending
 * on which key is pressed. If the down or right arrow key is pressed the
 * next focusable element of the accordion is focused on. If the last element 
 * was on focus this changes to the first element. If the enter or space key
 * was pressed the onclick function associated with the focused element is 
 * activated. 
 * @param (String) keyCode The valye of the key which was pressed
 * @param (Event) event The key press event.
 * @param (String) nodeId The id of the accordion item. 
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._traverseMenu = function(keyCode, event, nodeId) {
    var savedFocusElement;

    if (this.focusElement != null) {
        savedFocusElement = this.focusElement;
    }

    // Operations to be performed if the arrow keys are pressed.
    // down or right arrow key ==> forward
    // up or left arrow key ==> backward
    // During forward traversal control should go from the refresh icon
    // to the collapseAll icon to the expandAll icon to the tabs from
    // top to bottom. If control is on the last tab, clicking a forward
    // key will cause control to come back to the first tab and not the
    // accordion control icons. A similar but opposite behavior is 
    // exhibited when the up or left keys are pressed. Once control is
    // on the first tab backward traversal will take control to the last 
    // tab in the accordion.

    if (keyCode >= 37 && keyCode <= 40) {
        var forward = true;
        if (keyCode == 37 || keyCode == 38) {
            forward = false;
        }
        var focusSet = false;
        if (forward) {  // handling the down and right arrow keys
            if (this.isRefreshIcon) {
                if (this.focusElement.id == this._refreshNodeContainer.id) {
                    if (this.toggleControls && this.multipleSelect) {
                        this._updateFocus(this._collapseAllContainer);
                        focusSet = true;
                    } else {
                        this._updateFocus(this.tabs[0]);
                        focusSet = true;
                    }
                }
            }
            
            if (!focusSet && this.toggleControls && this.multipleSelect) {
                
                if (this.focusElement.id == this._collapseAllContainer.id) {
                    this._updateFocus(this._expandAllContainer);
                    focusSet = true;

                } else if (this.focusElement.id == this._expandAllContainer.id) {
                    this._updateFocus(this.tabs[0]);
                    focusSet = true;
                
                } else {
                    for (var i = 0; i < this.tabs.length; i++) {
                        if (this.tabs[i].id == this.focusElement.id) {
                            var newIndex = (i + 1) % this.tabs.length;
                            this._updateFocus(this.tabs[newIndex]);
                            focusSet = true;
                            break;
                        }
                    }
                }
            }
            if (!focusSet) {
                for (var i = 0; i < this.tabs.length; i++) {
                    if (this.tabs[i].id == this.focusElement.id) {
                        var newIndex = (i + 1) % this.tabs.length;
                        this._updateFocus(this.tabs[newIndex]);
                        focusSet = true;
                        break;
                    }
                }
                if (this.focusElement == null) {
                    this._updateFocus(this.tabs[0]);
                }
            }

         } else {  // traverse backward

            if (this.isRefreshIcon) {
                if (this.focusElement.id == this._refreshNodeContainer.id) {
                    var index = this.tabs.length;
                    this._updateFocus(this.tabs[index-1]);
                    focusSet = true;
                }
            }
            
            if (!focusSet && this.toggleControls && this.multipleSelect) {
                if (this.focusElement.id == this._collapseAllContainer.id) {
                    if (this.isRefreshIcon) {
                        this._updateFocus(this._refreshNodeContainer);
                        focusSet = true;
                    } else {
                        var index = this.tabs.length;
                        focusSet = true;
                        this._updateFocus(this.tabs[index-1]);
                    }
 
                } else if (this.focusElement.id == this._expandAllContainer.id) {
                    this._updateFocus(this._collapseAllContainer);
                    focusSet = true;
                }
            }
            if (!focusSet) {
                for (var i = 0; i < this.tabs.length; i++) {
                    if (this.tabs[i].id == this.focusElement.id) {
                        if (i == 0) {
                            var index = this.tabs.length;
                            this._updateFocus(this.tabs[index-1]);
                            focusSet = true;
                            break;
                            
                        } else {
                            this._updateFocus(this.tabs[i-1]);
                            focusSet = true;
                        }
                        break;
                    }
                }
                if (this.focusElement == null) { //focus on the last tab
                    var index = this.tabs.length;
                    this._updateFocus(this.tabs[index - 1]);
                    focusSet = true;
                }
            }
        }

        if (woodstock4_3._base.browser._isIe5up()) {
            window.event.cancelBubble = true;
            window.event.returnValue = false;
        } else {
            event.stopPropagation();
            event.preventDefault();
        }
    } else if(keyCode == 13 || keyCode == 32) {  // handle enter and space bar
        var actionComplete = false;
        if (this.isRefreshIcon) {
            if (this.focusElement.id == this._refreshNodeContainer.id) {
                var accWidget = this._widget.getWidget(nodeId);
                accWidget.refresh(nodeId);
                actionComplete = true;
            }    
        }
        if (!actionComplete && this.toggleControls && this.multipleSelect) {
            if (this.focusElement.id == this._collapseAllContainer.id) {
                this._collapseAllTabs();
                actionComplete = true;
                
            } else if (this.focusElement.id == this._expandAllContainer.id) {
                this._expandAllTabs();
                actionComplete = true;
            }
        } 
        if (!actionComplete) { // has to be an accordion tab
            if (this.focusElement) {
                var props;
                if (this.focusElement.selected) {
                    var widget = this._widget.getWidget(this.focusElement.id);
                    widget._setSelected(false);
                } else {
                    var widget = this._widget.getWidget(this.focusElement.id);
                    widget._setSelected(true);
                }
             
            }
        }

        if (woodstock4_3._base.browser._isIe5up()) {
            window.event.cancelBubble = true;
            window.event.returnValue = false;
        } else {
            event.stopPropagation();
            event.preventDefault();
        }
    } else if (keyCode == 9) {  // handle tabbing
        
        // Tabbing order:  refresh -> expand/collapse -> first tab header
        // -> first tabable element in tab content (if tab is open)
        // -> tab out of the accordion completely. Reverse tabbing order
        // is also supported.
        
        var forward = true;
        if (event.shiftKey) {
            forward = false;
        }
        if (this.focusElement) {
            var focusSet = false;
            if (this.isRefreshIcon) {
                if (this.focusElement.id == this._refreshNodeContainer.id) {
                    if (forward) {
                        if (this.toggleControls && this.multipleSelect) {
                            this.focusElement = this._collapseAllContainer;
                            this._setFocusStyleClass(this._collapseAllContainer.id);
                            this._setBlurStyleClass(this._refreshNodeContainer.id);
                            focusSet = true;
                            return true;
                        } else {
                            this.focusElement = this.tabs[0];
                            this._setBlurStyleClass(this._refreshNodeContainer.id);
                            this._setTabFocus(this.focusElement.id);
                            focusSet = true;
                            return true;
                        }
                    } else {
                        this.focusElement = null;
                        this._setBlurStyleClass(this._refreshNodeContainer.id);
                        focusSet = true;
                        return true;
                    }
                }
            }

            if (!focusSet && this.toggleControls && this.multipleSelect) {

                if (this.focusElement.id == this._collapseAllContainer.id) {
                    if (forward) {
                        this.focusElement = this._expandAllContainer;
                        this._setBlurStyleClass(this._collapseAllContainer.id);
                        this._setFocusStyleClass(this._expandAllContainer.id);
                        focusSet = true;
                    } else {
                        this.focusElement = this._refreshNodeContainer;
                        this._setFocusStyleClass(this._refreshNodeContainer.id);
                        this._setBlurStyleClass(this._collapseAllContainer.id);
                        focusSet = true;
                    }
                } else if (this.focusElement.id == this._expandAllContainer.id) {
                    if (forward) {
                        this.focusElement = this.tabs[0];
                        this._setTabFocus(this.focusElement.id);
                        this._setBlurStyleClass(this._expandAllContainer.id);
                        focusSet = true;
                    } else {
                        this.focusElement = this._collapseAllContainer;
                        this._setFocusStyleClass(this._collapseAllContainer.id);
                        this._setBlurStyleClass(this._expandAllContainer.id);
                        focusSet = true;
                    }
                } 
            }

            if (!focusSet) { // focus is on a tab
                if (forward) {  
                    var widget = this._widget.getWidget(this.focusElement.id);
                    if ((widget.getProps().selected == false) ||
                        (widget.focusState == "end")) {
                        
                        // simply move out of the accordion if tab is closed
                        // or tab is open but user has navigated to the end of
                        // the content section.
                        
                        for (var i = 0; i < this.tabs.length; i++) {
                            if (this.tabs[i].id == this.focusElement.id) {
                                var newIndex = (i + 1);
                                if (newIndex == this.tabs.length) {
                                    this._updateFocus(null);
                                    return true;
                                }
                                this._updateFocus(this.tabs[newIndex]);
                                focusSet = true;
                                return true;
                            }
                        }

                    }
                    focusSet = true;
                    
                } else {    // move to the expand/collapse/refresh icons
                    
                    this._setTabBlur(this.focusElement.id);
                    if (this.toggleControls && this.multipleSelect) {
                        this.focusElement = this._expandAllContainer;
                        this._setFocusStyleClass(this._expandAllContainer.id);
                    } else { 
                        this.focusElement = this._refreshNodeContainer;
                        this._setFocusStyleClass(this._refreshNodeContainer.id);
                    }
                    focusSet = true;
                    return true;
                }
            }

        } else { //get focus to the first element
            var focusSet = false;
            if (this.isRefreshIcon) {
               this.focusElement = this._refreshNodeContainer;
               this._setFocusStyleClass(this._refreshNodeContainer.id);
               focusSet = true;
            }
            if (!focusSet && this.toggleControls && this.multipleSelect) {
                this.focusElement = this._collapseAllContainer;
                this._setFocusStyleClass(this._collapseAllContainer.id);
                focusSet = true;
            } 
            if (!focusSet) {
                this.focusElement = this.tabs[0];
                this._setTabFocus(this.focusElement.id);
            }
            
        } 
    } 
    return true;
};

/**
 * This function updates the focused node within the accordion.
 * The old focus element is blurred and the new focus element is
 * set to focus. Appropriate style selectors are applied. 
 * @param (String) newFocusNode The new focus node.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.accordion.prototype._updateFocus = function(newFocusNode) {
    if (this.focusElement) {
        if (this.focusElement.id == this._refreshNodeContainer.id) {
            this._setBlurStyleClass(this._refreshNodeContainer.id);
        } else if (this.focusElement.id == this._collapseAllContainer.id) {
            this._setBlurStyleClass(this._collapseAllContainer.id);
        } else if (this.focusElement.id == this._expandAllContainer.id) {
            this._setBlurStyleClass(this._expandAllContainer.id);
        } else {
            for (var i = 0; i < this.tabs.length; i++) {
                if (this.tabs[i].id == this.focusElement.id) {
                    this._setTabBlur(this.tabs[i].id);
                    break;
                }
            }
        }
    }
    
    // set the new focusElement and then the associate syles etc.
    if (newFocusNode) {
        this.focusElement = newFocusNode;
        if (this.focusElement.id == this._refreshNodeContainer.id) {
                this._setFocusStyleClass(this._refreshNodeContainer.id);
        } else if (this.focusElement.id == this._collapseAllContainer.id) {
                this._setFocusStyleClass(this._collapseAllContainer.id);
        } else if (this.focusElement.id == this._expandAllContainer.id) {
                this._setFocusStyleClass(this._expandAllContainer.id);
        } else {
            for (var i = 0; i < this.tabs.length; i++) {
                if (this.tabs[i].id == this.focusElement.id) {
                    this._setTabFocus(this.tabs[i].id);
                    break;
                }
            }
        }
    }
    return true;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.accordion.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.accordionTab");

woodstock4_3._dojo.require("woodstock4_3.json");
woodstock4_3._dojo.require("woodstock4_3.widget.accordionTab");
woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");

/**
 * @class This class contains functions to obtain data asynchronously using JSF
 * Extensions as the underlying transfer protocol.
 * @static
 * @private
 */
woodstock4_3.widget._xhr.accordionTab = {
    /**
     * This function is used to load a tab content asynchronously.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processLoadContentEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Generate AJAX request.
        woodstock4_3.widget._xhr.common._doRequest({
            id: props.id,
            callback: woodstock4_3.widget._xhr.accordionTab._loadContentCallback,
            xjson: {
                id: props.id,
                event: "loadContent"
            }
        });
        return true;
    },

    /**
     * This function is used to update tab with the loaded content.
     *
     * @param {String} id The HTML widget Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson header provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _loadContentCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var json = woodstock4_3.json.parse(content);

        // Set progress.
        var widget = woodstock4_3.widget.common.getWidget(id);
        widget.setProps(json);

        // Publish an event for custom AJAX implementations to listen for.
        woodstock4_3._dojo.publish(woodstock4_3.widget.accordionTab.event.load.endTopic, [json]);
        return true;
    }
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.accordionTab.event.load.beginTopic,
    woodstock4_3.widget._xhr.accordionTab, "_processLoadContentEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.accordionTab.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.alarm");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.alarm");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct an alarm widget.
 *
 * @constructor
 * @name woodstock4_3.widget.alarm
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the alarm widget.
 * <p>
 * Use the type property to specify the alarm severity, which determines the 
 * alarm to display. The Alarm also supports a set of indicators which allows 
 * custom types and associated images. The text property is used to specify the
 * text to be displayed next to the alarm. The textPosition property specifies 
 * whether the text should be displayed to the left or right of the alarm.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "alarm1",
 *       text: "This is an alarm",
 *       textPosition: "right",
 *       type: "critical"
 *       widgetType: "alarm"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * alarm is either hidden or shown.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "alarm1",
 *       text: "This is an alarm",
 *       textPosition: "right",
 *       type: "critical"
 *       widgetType: "alarm"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Alarm State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("alarm1"); // Get alarm
 *       return widget.setProps({visible: !domNode.getProps().visible}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the alarm is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "alarm1",
 *       text: "This is an alarm",
 *       textPosition: "right",
 *       type: "critical"
 *       widgetType: "alarm"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Alarm" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("alarm1"); // Get alarm
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an alarm using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the alarm text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "alarm1",
 *       text: "This is an alarm",
 *       textPosition: "right",
 *       type: "critical"
 *       widgetType: "alarm"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Alarm Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("alarm1"); // Get alarm
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "btn1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.alarm.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Array} indicators Array of Key-Value pairs, containing type and 
 * image properties.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} style Specify style rules inline.
 * @config {String} text The text to display with alarm.
 * @config {String} textPosition Display text left or right of alarm.
 * @config {String} title Provides a title for element.
 * @config {String} type The type of alarm to display.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.alarm", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    _widgetType: "alarm" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.alarm.event =
        woodstock4_3.widget.alarm.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.alarm.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_alarm_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.alarm.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_alarm_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.alarm.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_alarm_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.alarm.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_alarm_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.alarm.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.text != null) { props.text = this.text; }
    if (this.indicators != null) { props.indicators = this.indicators; }
    if (this.textPosition != null) { props.textPosition = this.textPosition; }
    if (this.type != null) { props.type = this.type; }
    
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.alarm.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._rightText.id = this.id + "_rightText";
        this._leftText.id = this.id + "_leftText";
        this._imageContainer.id = this.id + "_imageContainer";        
    }
    // default set of indicators
    var  defaultIndicators = [{
        "type": "down",
        "image": {
            id: this.id + "_down",
            icon: "DOWN_ALARM_INDICATOR",
            widgetType: "image"
        }
    }, {
        "type": "critical",
        "image": {
            id: this.id + "_critical",
            icon: "CRITICAL_ALARM_INDICATOR",
            widgetType: "image"
        }
    }, {
        "type": "major",
        "image": {
            id: this.id + "_major",
            icon: "MAJOR_ALARM_INDICATOR",
            widgetType: "image"
        }
    }, {
        "type": "minor",
        "image": {
            id: this.id + "_minor",
            icon: "MINOR_ALARM_INDICATOR",
            widgetType: "image"
        }
    }, {
        "type": "ok",
        "image": {
            id: this.id + "_ok",
            icon: "OK_ALARM_INDICATOR",
            widgetType: "image"
        }
    }];

    if (this.indicators == null) {
        this.indicators = defaultIndicators;    
    } else {
        for (var i = 0; i < this.indicators.length; i++) {          
            for (var j = 0; j < defaultIndicators.length; j++) {
                if (this.indicators[i].type == defaultIndicators[j].type) {
                    defaultIndicators[j].image = this.indicators[i].image;
                    this.indicators[i] = null;                  
                    break;
                }
            }
        }
      
        // merge the indicators (defaultset + custom set)
        for (var i = 0; i < this.indicators.length; i++) {   
            if (this.indicators[i] != null) {                         
                defaultIndicators = defaultIndicators.concat(this.indicators[i]);
            }
        }
        this.indicators = defaultIndicators;     
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.alarm.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.dir != null) { this._domNode.dir = props.dir; }
    if (props.lang != null) { this._domNode.lang = props.lang; }    
    
    // Set right text.
    if (props.textPosition == "right" || props.textPosition == null && props.text != null) {
        this._common._setVisibleElement(this._leftText, false);
        this._widget._addFragment(this._rightText, props.text);
    }

    // Set left text.
    if (props.textPosition == "left" && props.text != null) {
        this._common._setVisibleElement(this._rightText, false);
        this._widget._addFragment(this._leftText, props.text);
    }    
    
    // Set indicator properties.
    if (props.indicators || props.type != null && this.indicators) {
        // Iterate over each indicator.
        for (var i = 0; i < this.indicators.length; i++) {
            // Ensure property exists so we can call setProps just once.
            var indicator = this.indicators[i]; // get current indicator.
            if (indicator == null) {
                indicator = {}; // Avoid updating all props using "this" keyword.
            }

            // Set properties.
            indicator.image.visible = (indicator.type == this.type) ? true: false;

            // Update/add fragment.
            this._widget._updateFragment(this._imageContainer, indicator.image.id, 
                indicator.image, "last");
        }
    }

    // Do not call _setCommonProps() here. 

    // Set more properties.
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.alarm.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.alert");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.alert");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct an alert widget.
 *
 * @constructor
 * @name woodstock4_3.widget.alert
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the alert widget.
 * <p>
 * The alert widget is comprised of an image, summary message, and an optional 
 * detail message. The image shown is determined by the type property, which 
 * must be set to "information", "success", "warning", "error" or a custom type.
 * The alert widget also supports a set of indicators which allows custom types
 * and associated images. The summary message is specified with the summary 
 * property and is displayed prominently next to the image. The optional detail
 * message is specified with the detail property, and is displayed in less 
 * prominent text following the summary message. The detail messge provides more
 * information about the alert. You can also include a link to more information
 * or another window using the moreInfo property. The link is displayed below 
 * the detail message.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "alert1",
 *       detail: "Detailed message",
 *       summary: "Summary message",
 *       type: "error",
 *       widgetType: "alert"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * alert is either hidden or shown.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "alert1",
 *       detail: "Detailed message",
 *       summary: "Summary message",
 *       type: "error",
 *       widgetType: "alert"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Alert State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("alert1"); // Get alert
 *       return widget.setProps({visible: !domNode.getProps().visible}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the alert is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "alert1",
 *       detail: "Detailed message",
 *       summary: "Summary message",
 *       type: "error",
 *       widgetType: "alert"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Alert" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("alert1"); // Get alert
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an alert using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the alert text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "alert1",
 *       detail: "Detailed message",
 *       summary: "Summary message",
 *       type: "error",
 *       widgetType: "alert"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Alert Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("alert1"); // Get alert
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "btn1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.alert.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} detail The alert detail message.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {Array} indicators Array of Key-Value pairs, containing type and 
 * image properties.
 * @config {Object} moreInfo Key-Value pairs of properties for more info link.
 * @config {String} spacerImage 
 * @config {String} summary The alert summary message.
 * @config {String} type The type of alert to display.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.alert", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    _widgetType: "alert" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.alert.event =
        woodstock4_3.widget.alert.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.alert.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_alert_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.alert.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_alert_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.alert.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_alert_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.alert.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_alert_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.alert.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.detail != null) { props.detail = this.detail; }
    if (this.indicators != null) { props.indicators = this.indicators; }
    if (this.summary != null) { props.summary = this.summary; }
    if (this.type != null) { props.type = this.type; }
    if (this.moreInfo != null) { props.moreInfo = this.moreInfo; }
    if (this.spacerImage != null) { props.spacerImage = this.spacerImage; }

    return props;
};

/**
 * This function is used to process notification events with Object
 * literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} detail Message detail text.
 * @config {String} summary Message summary text.
 * @config {boolean} valid Flag indicating validation state.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.alert.prototype._notify = function(props) {
    if (props == null) {
        return false;
    }
    return this.setProps({
        summary: props.summary,
        detail: props.detail,
        type: "error",
        visible: !props.valid
    });
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.alert.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._bottomLeftContainer.id = this.id + "_bottomLeftContainer";
        this._bottomMiddleContainer.id = this.id + "_bottomMiddleContainer";
        this._bottomRightContainer.id = this.id + "_bottomRightContainer";
        this._detailContainer.id = this.id + "_detailContainer";
        this._imageContainer.id = this.id + "_imageContainer";
        this._leftMiddleContainer.id = this.id + "_leftMiddleContainer";
        this._rightMiddleContainer.id = this.id + "_rightMiddleContainer";
        this._summaryContainer.id = this.id + "_summaryContainer";
        this._topLeftContainer.id = this.id + "_topLeftContainer";
        this._topMiddleContainer.id = this.id + "_topMiddleContainer";
        this._topRightContainer.id = this.id + "_topRightContainer";
        this._detailContainerLink.id = this.id + "_detailContainerLink";
    }

    // Create default indicators.   
    var  defaultIndicators = [{
        "type": "error",
        "image": {
            id: this.id + "_error",
            icon: "ERROR_ALERT_INDICATOR",
            widgetType: "image"
        }
    }, {
        "type": "warning",
        "image": {
            id: this.id + "_warning",
            icon: "WARNING_ALERT_INDICATOR",
            widgetType: "image"
        }
    }, {
        "type": "success",
        "image": {
            id: this.id + "_success",
            icon: "SUCCESS_ALERT_INDICATOR",
            widgetType: "image"
        }
    }, {
        "type": "information",
        "image": {
            id: this.id + "_info",
            icon: "INFORMATION_ALERT_INDICATOR",
            widgetType: "image"
        }
    }];
        
    if (this.indicators == null) {
        this.indicators = defaultIndicators;    
    } else {
      for (var i = 0; i < this.indicators.length; i++) {          
          for (var j = 0; j < defaultIndicators.length; j++) {
              if (this.indicators[i].type == defaultIndicators[j].type) {
                  defaultIndicators[j].image = this.indicators[i].image;
                  this.indicators[i] = null;                  
                  break;
              }
          }          
      }
      
      // merge the indicators (defaultset + custom set)
      for (var i = 0; i < this.indicators.length; i++) {   
          if (this.indicators[i] != null) {                         
                defaultIndicators = defaultIndicators.concat(this.indicators[i]);
          }      
      }
      this.indicators = defaultIndicators;     
    }
    
    // spacer image
    if (this.spacerImage == null) {
        this.spacerImage = {
             icon: "DOT",
             id: this.id + "_dot",
             widgetType: "image"
        };
    }
    // moreInfo link
    // To do: Test isFragment instead.
    if (this.moreInfo != null && this.moreInfo.id == null 
            && this.moreInfo.widgetType == null) {
        this.moreInfo = {
            id: this.id + "_" + "alertLink",
            enabledImage:  {
                id: this.id + "_moreInfoLinkImg",
                icon: "HREF_LINK",
                widgetType: "image"
            },     
            target: this.moreInfo.target,
            href: this.moreInfo.url,
            contents: [this.moreInfo.value],
            imagePosition: "right",                   
            title: this.moreInfo.tooltip,
            widgetType: "imageHyperlink"
        };
    }    
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.alert.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.dir != null) { this._domNode.dir = props.dir; }
    if (props.lang != null) { this._domNode.lang = props.lang; }    
    
    // Set summary.
    if (props.summary != null) {
        this._widget._addFragment(this._summaryContainer, props.summary);
    }

    // Set detail.
    if (props.detail != null) {
        this._widget._addFragment(this._detailContainer, props.detail);
    }

    // Set moreInfo.
    if (props.moreInfo) {
        this._widget._addFragment(this._detailContainerLink, props.moreInfo);
    }

    // Set spacer image.
    if (props.spacerImage) {
        var containers = [
            this._bottomLeftContainer,
            this._bottomMiddleContainer,
            this._bottomRightContainer,
            this._leftMiddleContainer,
            this._rightMiddleContainer,
            this._topLeftContainer,
            this._topMiddleContainer,
            this._topRightContainer];

        // Avoid widget ID collisions.
        for (var i = 0; i < containers.length; i++) {
            if (typeof props != 'string') {
                props.spacerImage.id = this.id + "_spacerImage" + i;
            }
            // Replace container with image.
            if (!this._widget.getWidget(props.spacerImage.id)) {
                this._widget._addFragment(containers[i], props.spacerImage);
            }
        }
    }

    // Set indicator properties.
    if (props.indicators || props.type != null && this.indicators) {
        // Iterate over each indicator.
        for (var i = 0; i < this.indicators.length; i++) {
            // Ensure property exists so we can call setProps just once.
            var indicator = this.indicators[i]; // get current indicator.
            if (indicator == null) {
                indicator = {}; // Avoid updating all props using "this" keyword.
            }

            // Set properties.
            indicator.image.visible = (indicator.type == this.type) ? true: false;
            indicator.image.tabIndex = this.tabIndex;

            // Update/add fragment.
            this._widget._updateFragment(this._imageContainer, indicator.image.id, 
                indicator.image, "last");
        }
    }

    // Do not call _setCommonProps() here. 

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.alert.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.anchor");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.anchor");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.anchorBase");

/**
 * This function is used to construct an anchor widget.
 *
 * @constructor
 * @name woodstock4_3.widget.anchor
 * @extends woodstock4_3.widget._base.anchorBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the anchor widget. 
 * <p>
 * The anchor widget creates an HTML anchor that traverses to the specified URL.
 * Or, it can be used to Anchor a position in the page.
 * </p><p>
 * <h3>Example 1a: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 1b: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget which outputs a named anchor.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       name: "anchorName",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * anchor is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Anchor State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("anc1"); // Get anchor
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the anchor is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Anchor" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("anc1"); // Get anchor
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an anchor using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the anchor text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "anc1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "anchor"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Anchor Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("anc1"); // Get anchor
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "anc1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.anchor.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.anchor", [
        woodstock4_3.widget._base.refreshBase,
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.anchorBase ], {
    // Set defaults.
    _widgetType: "anchor" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.anchor.event =
        woodstock4_3.widget.anchor.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.anchor.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_anchor_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.anchor.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_anchor_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.anchor.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_anchor_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.anchor.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_anchor_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.anchor.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.href && this.disabled == false)
        ? this._theme.getClassName("ANCHOR","")
        : this._theme.getClassName("ANCHOR_DISABLED","");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.anchor.prototype._postCreate = function () {
    // Create callback function for onclick event.
    this._dojo.connect(this._domNode, "onclick", this, "_onClickCallback");

    return this._inherited("_postCreate", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.anchor.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.bubble");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.bubble");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a bubble widget.
 *
 * @constructor
 * @name woodstock4_3.widget.bubble
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the bubble widget.
 * <p>
 * The bubble widget displays a pop up window which appears on screen in 
 * response to certain mouse interactions. The purpose of the bubble widget is
 * to provide detailed information to the end user when a mouse hovers on a 
 * particular HTML element. The bubble widget exposes two main JavaScript 
 * functions; "open()" and "close()". These two functions are invoked by the 
 * element that requires bubble help.
 * </p><p>
 * <h3>Example 1a: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "bub1",
 *       contents: [{
 *         id: "st1",
 *         value: "Bubble Help Text",
 *         widgetType: "text",
 *       }],
 *       title: "Bubble Title",
 *       widgetType: "bubble"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 1b: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using relative positioning.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "bub1",
 *       contents: [{
 *         id: "st1",
 *         value: "Bubble Help Text",
 *         widgetType: "text",
 *       }],
 *       style: "position:relative;top:200px;left:480px;width:20em;z-index:99;",
 *       title: "Bubble Title",
 *       widgetType: "bubble"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 1c: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget which opens and closes the bubble
 * window.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "bub1",
 *       contents: [{
 *         id: "st1",
 *         value: "Bubble Help Text",
 *         widgetType: "text",
 *       }],
 *       title: "Bubble Title",
 *       widgetType: "bubble"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Show Bubble" },
 *       onMouseOver: "woodstock4_3.widget.common.getWidget('bub1').open(event);"
 *       onMouseOut: "woodstock4_3.widget.common.getWidget('bub1').close();"
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to set the title of a widget using the getProps and 
 * setProps functions. When the user clicks the checkbox, the bubble title is
 * updated.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "bub1",
 *       contents: [{
 *         id: "st1",
 *         value: "Bubble Help Text",
 *         widgetType: "text",
 *       }],
 *       title: "Bubble Title",
 *       widgetType: "bubble"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Change Bubble Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("bub1"); // Get bubble
 *       var newTitle = "New " + widget.getProps().title;
 *       return widget.setProps({title: newTitle}); // Update bubble title
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the bubble is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "bub1",
 *       contents: [{
 *         id: "st1",
 *         value: "Bubble Help Text",
 *         widgetType: "text",
 *       }],
 *       title: "Bubble Title",
 *       widgetType: "bubble"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh bubble" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("bub1"); // Get bubble
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an bubble using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the bubble text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "bub1",
 *       contents: [{
 *         id: "st1",
 *         value: "Bubble Help Text",
 *         widgetType: "text",
 *       }],
 *       title: "Bubble Title",
 *       widgetType: "bubble"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change bubble Title" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("bub1"); // Get bubble
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "bub1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.bubble.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} autoClose 
 * @config {Object} closeButton 
 * @config {Array} contents 
 * @config {int} duration 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {int} openDelay 
 * @config {String} title Provides a title for element.
 * @config {int} width 
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.bubble", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.defaultTime = 2000;
        this.openDelayTime = 500;
        this.bubbleLeftConst = 5;
        this.topConst = 2;
    },   
    _widgetType: "bubble" // Required for theme properties.
});

/**
 * This function is used to close bubble help.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.bubble.prototype.close = function() {
    if (this.openTimerId != null) {
        clearTimeout(this.openTimerId);
    }
    if (this.visible == false) {
        return false;
    }
     
    var _id = this.id;
    this.timerId = setTimeout(function() {
        // New literals are created every time this function is called, and it's 
        // saved by closure magic.
        var getWidget = woodstock4_3.widget.common.getWidget;
        getWidget(_id).setProps({visible: false});
        getWidget(_id).srcElm.focus();
    }, this.defaultTime);

    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.bubble.event =
        woodstock4_3.widget.bubble.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.bubble.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_bubble_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.bubble.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_bubble_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.bubble.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_bubble_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.bubble.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_bubble_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.bubble.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.title != null) { props.title = this.title; }
    if (this.contents != null) { props.contents = this.contents; }
    if (this.height != null) { props.height = this.height; }
    if (this.width != null) { props.width = this.width; }
    if (this.autoClose != null) { props.autoClose = this.autoClose; }
    if (this.duration != null) { props.duration = this.duration; }
    if (this.closeButton != null) {props.closeButton = this.closeButton;}
    if (this.openDelay != null) {props.openDelay = this.openDelay;}
    if (this.focusId != null) {props.focusId = this.focusId;}
    if (this.tabIndex != null) {props.tabIndex = this.tabIndex;}
    
    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._onClickCallback = function(event) {
    // Close the popup if close button is clicked.
    event = this._widget._getEvent(event);

    var target = (event.target)
        ? event.target 
        : ((event.srcElement) 
            ? event.srcElement : null);

    if (woodstock4_3._base.browser._isIe5up()) {
        if (window.event != null) {
            window.event.cancelBubble = true;
        }
    } else {
        event.stopPropagation();
    }
    if (this._closeBtn == target) {
        clearTimeout(this.timerId);
        this.setProps({visible: false});
        this.srcElm.focus();
    }
    return true;
};

/**
 * Helper function to create callback for close event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._onCloseCallback = function(event) {
    if (event == null) {
        return false;
    }
    
    if ((event.type == "keydown" && event.keyCode == 27)
            || event.type == "click") {
        clearTimeout(this.timerId); 
        
        if (this.srcElm != null && this.visible) {
            if (this.srcElm.focus) {
                this.srcElm.focus();
            }
        }      
        this.setProps({visible: false});
    }
    return true;
};

/**
 * Helper function to create callback for shift + tab event.
 * Shift+Tab should not allow user to tab out of bubble component.
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._onShiftTabCallback = function(event) {
    if (event == null) {
        return false;
    }
    event = this._widget._getEvent(event);

    var target = (event.target)
        ? event.target 
        : ((event.srcElement) 
            ? event.srcElement : null);
    if (target == this._bubbleHeader) {                    
        if (woodstock4_3._base.browser._isFirefox() && (event.shiftKey && (event.keyCode == 9))) {
            if (this.focusId != null) {
                document.getElementById(this.focusId).focus();        
            } else {                
                 this._bubbleHeader.focus();
            }
            event.stopPropagation();
            event.preventDefault(); 
        }
     }
     return true;
};

/**
 * Helper function to create callback for tab event.
 * Cyclic tabbing behavior is implemented for bubble to prevent tab out of bubble component. 
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._onTabCallback = function(event) {
    if (event == null) {
        return false;
    }
    event = this._widget._getEvent(event);

    var target = (event.target)
        ? event.target 
        : ((event.srcElement) 
            ? event.srcElement : null);
    if (woodstock4_3._base.browser._isFirefox()) {        
        if (this._contentEnd == target) {
            this._bubbleHeader.focus();
        } else if (this._bubbleHeader == target && this.focusId != null && (event.keyCode == 9)) {
            document.getElementById(this.focusId).focus();            
        }
        event.stopPropagation();
        event.preventDefault(); 
    }
    return true;
};
    
/**
 * Helper function to create callback for onMouseOver event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._onMouseOverCallback = function(event) {
    clearTimeout(this.timerId);
    return true;
};

/**
 * Helper function to create callback for onMouseOut event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._onMouseOutCallback = function(event) {
    if (this.autoClose == true) {
        clearTimeout(this.timerId);            
        this.close();            
    }
    return true;
};

/**
 * This function is use to invoke buuble help.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.bubble.prototype.open = function(event) {
    // Get the absolute position of the target.
    var evt = this._widget._getEvent(event);
    // A11Y - open the bubble if its Ctrl key + F1
    if (evt.type == "keydown") {
        if (!(evt.ctrlKey && (evt.keyCode == 112))) {
            return false;
        }
        evt.stopPropagation();
        evt.preventDefault();  
    }
    this.srcElm = (evt.target) 
        ? evt.target : ((evt.srcElement) 
            ? evt.srcElement : null);

    var absPos = this._widget._getPosition(this.srcElm);
    this.srcElm.targetLeft = absPos[0];
    this.srcElm.targetTop = absPos[1];
   
    if (this.timerId != null) {
        clearTimeout(this.timerId);
        this.timerId = null;
    }
    
    if (this.openDelay != null && this.openDelay >= 0) {
        this.openDelayTime = this.openDelay;
    }

    // There should be delay before opening the bubble if open delay is specified.
    // If openDelay is less than zero then there will be dafault 0.5 sec delay.  
    
    var id = this.id; // Closure magic.
    this.openTimerId = setTimeout(function() {
        var getWidget = woodstock4_3.widget.common.getWidget;

        // Store the active bubble id to form element.
        // Check for the id if its available then close the pending bubble.
        if (woodstock4_3.widget.bubble.activeBubbleId && woodstock4_3.widget.bubble.activeBubbleId != id) {                
            clearTimeout(getWidget(woodstock4_3.widget.bubble.activeBubbleId).timerId);
            getWidget(woodstock4_3.widget.bubble.activeBubbleId).setProps({visible: false});
            woodstock4_3.widget.bubble.activeBubbleId = null;                
        }     
        woodstock4_3.widget.bubble.activeBubbleId = id;            
        getWidget(id).setProps({visible: true});
        getWidget(id)._setPosition();
    }, this.openDelayTime);           
    
    if (this.duration != null && this.duration >= 0) {
        this.defaultTime = this.duration;
    } 
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._bottomLeftArrow.id = this.id + "_bottomLeftArrow";
        this._bottomRightArrow.id = this.id + "_bottomRightArrow";
        this._topLeftArrow.id = this.id + "_topLeftArrow";
        this._topRightArrow.id = this.id + "_topRightArrow";
    }

    // Set public functions.

    /** @ignore */
    this._domNode.close = function() { return woodstock4_3.widget.common.getWidget(this.id).close(); };
    /** @ignore */
    this._domNode.open = function(event) { return woodstock4_3.widget.common.getWidget(this.id).open(event); };

    // Set events.

    // The onClick on window should close bubble.
    this._dojo.connect(document, "onclick", this, "_onCloseCallback");

    // The escape key should also close bubble.
    this._dojo.connect(document, "onkeydown", this, "_onCloseCallback");

    // The onClick event for component body. Closes the bubble only when
    // close button is clicked.
    this._dojo.connect(this._domNode, "onclick", this, "_onClickCallback");

    // Do not close the popup if mouseover on bubble if mouseover on bubble 
    // component then clear the timer and do not close bubble.
    this._dojo.connect(this._domNode, "onmouseover", this, "_onMouseOverCallback");

    // Close the popup if mouseout and autoClose is true if onmouseout and 
    // autoClose is true then close the bubble.
    this._dojo.connect(this._domNode, "onmouseout", this, "_onMouseOutCallback");
    
    // The onfocus event for _contentEnd. This is needed to handle tab event. 
    this._dojo.connect(this._contentEnd, "onfocus", this, "_onTabCallback");
    
    // The onkeydown event for _bubbleHeader. This is needed to handle tab event. 
    this._dojo.connect(this._bubbleHeader, "onkeydown", this, "_onTabCallback");
    
    // The onkeydown event for component body. This is needed to handle shift+tab event.
    this._dojo.connect(this._domNode, "onkeydown", this, "_onShiftTabCallback");
    
    // Initialize the bubble title width as a percentage of the bubble header.    
    if (this._bubbleTitle != null) {
        this._bubbleTitle.style.width = this._theme.getProperty("styles", 
            "BUBBLE_TITLEWIDTH") + "%";
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to position the bubble.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._setPosition = function() {
    // THIS CODE BLOCK IS NECESSARY WHEN THE PAGE FONT IS VERY SMALL,
    // AND WHICH OTHERWISE CAUSES THE PERCENTAGE OF THE HEADER WIDTH
    // ALLOCATED TO THE BUBBLE TITLE TO BE TOO LARGE SUCH THAT IT
    // ENCROACHES ON THE SPACE ALLOCATED FOR THE CLOSE BUTTON ICON,
    // RESULTING IN LAYOUT MISALIGNMENT IN THE HEADER.

    // Assume bubble title width max percentage of the bubble header.
    var maxPercent = this._theme.getProperty("styles", "BUBBLE_TITLEWIDTH");

    // Sum of widths of all elements in the header BUT the title.  This includes
    // the width of the close button icon, and the margins around the button and
    // the title.  This should be a themeable parameter that matches the left/right
    // margins specified in the stylesheet for "BubbleTitle" and "BubbleCloseBtn".
    var nonTitleWidth = this._theme.getProperty("styles", "BUBBLE_NONTITLEWIDTH");

    // Get the widths (in pixels) of the bubble header and title
    var headerWidth = this._bubbleHeader.offsetWidth;
    var titleWidth = this._bubbleTitle.offsetWidth;

    // Revise the aforementioned percentage downward until the title no longer
    // encroaches on the space allocated for the close button.  We decrement by
    // 5% each time because doing so in smaller chunks when the font gets very small so 
    // only results in unnecessary extra loop interations.
    //
    if (headerWidth > nonTitleWidth) {
        while ((maxPercent > 5) && (titleWidth > (headerWidth - nonTitleWidth))) {
            maxPercent -= 5;
            this._bubbleTitle.style.width = maxPercent + "%";
            titleWidth = this._bubbleTitle.offsetWidth;
        }
    }

    // Get DOM bubble object associated with this Bubble instance.
    var bubble = this._domNode;

    // If this.style is not null that means developer has specified positioning
    // for component. 
    if (this._domNode != null && this.style != null && this.style.length > 0) {
        if (bubble.style.length != null) {
            for (var i = 0; i < bubble.style.length; i++) {
                if (bubble.style[i] == "top") {
                    this.top = bubble.style.top;
                }
                if (bubble.style[i] == "left") {
                    this.left = bubble.style.left;
                }
            }
        } else {
            // For IE, simply query the style attributes.
            if (bubble.style.top != "") {
                this.top = bubble.style.top;
            }
            if (bubble.style.left != "") {
                this.left = bubble.style.left;
            }
        }
    }

    if ((this.top != null) && (this.left != null)) {
        bubble.style.left = this.left;
        bubble.style.top = this.top;    
    } else {
        // Why are we calling getElementById here instead of using the dojoAttachPoint?
        var topLeftArrow = document.getElementById(this._topLeftArrow.id);
        var topRightArrow = document.getElementById(this._topRightArrow.id);
        var bottomLeftArrow = document.getElementById(this._bottomLeftArrow.id);
        var bottomRightArrow = document.getElementById(this._bottomRightArrow.id);

        // hide all callout arrows.
        this._common._setVisible(bottomLeftArrow, false);
        this._common._setVisible(bottomRightArrow, false);
        this._common._setVisible(topLeftArrow, false);
        this._common._setVisible(topRightArrow, false);

        bottomLeftArrow.style.display = "none";
        bottomRightArrow.style.display = "none";
        topLeftArrow.style.display = "none";
        topRightArrow.style.display = "none";

        var slidLeft = false;

        // Assume default bubble position northeast of target, which implies a 
        // bottomLeft callout arrow
        this.arrow = bottomLeftArrow;

        // Try to position bubble to right of srcElm.
        var bubbleLeft = this.srcElm.targetLeft + this.srcElm.offsetWidth + this.bubbleLeftConst;

        // Check if right edge of bubble exceeds page boundary.
        var rightEdge = bubbleLeft + bubble.offsetWidth;
        if (rightEdge > this._widget._getPageWidth()) {

            // Shift bubble to left side of target;  implies a bottomRight arrow.
            bubbleLeft = this.srcElm.targetLeft - bubble.offsetWidth;
            this.arrow = bottomRightArrow;
            slidLeft = true;

            // If left edge of bubble crosses left page boundary, then
            // reposition bubble back to right of target and implies to go
            // back to bottomLeft arrow.  User will need to use scrollbars
            // to position bubble into view.
            if (bubbleLeft <= 0) {
                bubbleLeft = this.srcElm.targetLeft + this.srcElm.offsetWidth + this.bubbleLeftConst;
                this.arrow = bottomLeftArrow;
                slidLeft = false;
            }
        }

        // Try to position bubble above source element
        var bubbleTop = this.srcElm.targetTop - bubble.offsetHeight;

        // Check if top edge of bubble crosses top page boundary
        if (bubbleTop <= 0) {
            // Shift bubble to bottom of target.  User may need to use scrollbars
            // to position bubble into view.
            bubbleTop = this.srcElm.targetTop + this.srcElm.offsetHeight + this.bubbleLeftConst;

            // Use appropriate top arrow depending on left/right position.
            if (slidLeft == true)
                this.arrow = topRightArrow;
            else
                this.arrow = topLeftArrow;
        }
        // Adjust to account for parent container.
        var parentPos = this._widget._getPosition(this._domNode.offsetParent);
        bubbleLeft -= parentPos[0];
        bubbleTop -= parentPos[1]; 
        // Set new bubble position.
        bubble.style.left = bubbleLeft + "px";
        bubble.style.top = bubbleTop + "px";

        // If rendering a callout arrow, set it's position relative to the bubble.
        if (this.arrow != null) {
           this.arrow.style.display = "block";
           this._common._setVisible(this.arrow, true);

           if (this.arrow == topLeftArrow) {
               this.arrow.style.top = -(bubble.offsetHeight - this.topConst) + "px";               
           }
           if (this.arrow == topRightArrow) {
               this.arrow.style.top = -(bubble.offsetHeight - this.topConst) + "px";               
           }
        }
    }
    if (this.focusId != null) {
        document.getElementById(this.focusId).focus();        
    } else {
        if (woodstock4_3._base.browser._isFirefox()) {
            this._bubbleHeader.focus();
        }
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.bubble.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }
    
    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.bubble.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
    //Cyclic focus behavior is supported for firefox browser only.
    //If tabIndex values are provided for elements inside bubble then developer needs to set a valid tabIndex 
    //value for bubble component to achieve cyclic focus behavior. 
    if (woodstock4_3._base.browser._isFirefox()) {
        if (this.tabIndex >= 0) {
            this._contentEnd.tabIndex = this.tabIndex;
        } else {
            this._contentEnd.tabIndex = 0;
        }   
    }
    // Set title.
    if (props.title) {
        this._widget._addFragment(this._titleNode, props.title);
    }

    // hide/display close button
    if (props.closeButton != null) {
        var classNames = this._closeBtn.className.split(" ");
        var closeButtonClass = this._theme.getClassName("BUBBLE_CLOSEBTN");
        var noCloseButtonClass = this._theme.getClassName("BUBBLE_NOCLOSEBTN");

        if (props.closeButton == false) {
            this._common._stripStyleClass(this._closeBtn, closeButtonClass);
            if (!this._common._checkStyleClasses(classNames, noCloseButtonClass))
             this._common._addStyleClass(this._closeBtn, noCloseButtonClass);
        } else {          
          if (!this._common._checkStyleClasses(classNames, closeButtonClass))
             this._common._addStyleClass(this._closeBtn, closeButtonClass);
        }
    }

    // Set width.
    if (props.width > 0) {                    
        this._domNode.style.width = props.width + "px";        
    }

    // Set contents.
    if (props.contents) {
        // Remove child nodes.
        this._widget._removeChildNodes(this._childNode);

        for (var i = 0; i < props.contents.length; i++) {
            this._widget._addFragment(this._childNode, props.contents[i], "last");
        }
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.bubble.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.button");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.button");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a button widget.
 * 
 * @constructor
 * @name woodstock4_3.widget.button
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the button widget.
 * <p>
 * The button widget can render a button as a primary button or a secondary
 * button. A primary button is intended to be used for buttons that are the most
 * commonly used on a page or section of a page, and are rendered to be more
 * visually prominent than secondary buttons. The primary attribute is false by
 * default, which renders a secondary button.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "btn1",
 *       value: "This is a button",
 *       widgetType: "button"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * button is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "btn1",
 *       value: "This is a button",
 *       widgetType: "button"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Button State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("btn1"); // Get button
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the button is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "btn1",
 *       value: "This is a button",
 *       widgetType: "button"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Button" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("btn1"); // Get button
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a button using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the button
 * label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "btn1",
 *       value: "This is a button",
 *       widgetType: "button"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Button Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("btn1"); // Get button
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "btn1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.button.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} escape HTML escape value (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} mini Set button as mini if true (N/A with src property).
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} prefix The application context path of image.
 * @config {boolean} primary Set button as primary if true (N/A with src property).
 * @config {boolean} reset Set button as reset if true (overrides submit type).
 * @config {boolean} submit Set button as submit if true (default). If false,
 * the underlying HTML input type will be button.
 * @config {String} src Source for image (overrides submit and reset types).
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.button", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
        this.escape = true;
        this.mini = false;
        this.primary = true;
        this.reset = false;
        this.submit = true;
    },
    _widgetType: "button" // Required for theme properties.
});

/**
 * This function is used to render the widget from a template.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._buildRendering = function () {
    // On IE, the HTML input type property must be set when the DOM node is 
    // created. Therefore, we're using HTML templates to define the type.
    if (this.src != null) {
        this._templateType = "imageButton";
    } else if (new Boolean(this.reset).valueOf() == true) {
        this._templateType = "resetButton";
    } else if (new Boolean(this.submit).valueOf() == true) {
        this._templateType = "submitButton";
    }
    return this._inherited("_buildRendering", arguments);
}

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.button.event =
        woodstock4_3.widget.button.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.button.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_button_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.button.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_button_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * {Object} props Key-Value pairs of widget properties being updated.
         * </li></ul>
         *
         * @id woodstock4_3.widget.button.event.submit.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_button_event_state_begin",

        /**
         * State event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.button.event.submit.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_button_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.button.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var key = null;

    // If dealing with an image button, only the BUTTON3 selectors are used.
    // Note that the "mini" and "primary" values can still be set but
    // have no effect on image buttons by policy Vs by theme.
    if (this.src != null) {
        key = (this.disabled == true)
            ? "BUTTON3_DISABLED"
            : "BUTTON3";
    } else if (this.mini == true && this.primary == true) {
        key = (this.disabled == true)
            ? "BUTTON1_MINI_DISABLED" // primaryMiniDisabledClassName
            : "BUTTON1_MINI";         // primaryMiniClassName;
    } else if (this.mini == true) {
        key = (this.disabled == true)
            ? "BUTTON2_MINI_DISABLED" // secondaryMiniDisabledClassName
            : "BUTTON2_MINI";         // secondaryMiniClassName;
    } else if (this.primary == true) {
        key = (this.disabled == true)
            ? "BUTTON1_DISABLED"      // primaryDisabledClassName
            : "BUTTON1";              // primaryClassName
    } else {
        key = (this.disabled == true)
            ? "BUTTON2_DISABLED"      // secondaryDisabledClassName
            : "BUTTON2";	      // secondaryClassName
    }

    var newClassName = this._theme.getClassName(key, "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to obtain the outermost HTML element class name during
 * an onFocus or onMouseOver event.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.button.prototype._getHoverClassName = function() {
    // Cannot call this._inherited("_getClassName", arguments) here.
    var className = woodstock4_3.widget.button.superclass._getClassName.apply(this, arguments);
    var key = null;

    // If dealing with an image button, only the BUTTON3 selectors are used.
    // Note that the "mini" and "primary" values can still be set but
    // have no effect on image buttons by policy Vs by theme.
    if (this.src != null) {
        key = "BUTTON3_HOVER";
    } else if (this.mini == true && this.primary == true) {
        key = "BUTTON1_MINI_HOVER"; 	// primaryMiniHovClassName;
    } else if (this.mini == true) {
        key = "BUTTON2_MINI_HOVER"; 	// secondaryMiniHovClassName;
    } else if (this.primary == true) {
        key = "BUTTON1_HOVER"; 		// primaryHovClassName;
    } else {
        key = "BUTTON2_HOVER";		// secondaryHovClassName;
    }

    var newClassName = this._theme.getClassName(key, "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};
    
/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @id woodstock4_3.widget.button.getProps
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.button.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.alt != null) { props.alt = this.alt; }
    if (this.align != null) { props.align = this.align; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.escape != null) { props.escape = this.escape; }
    if (this.mini != null) { props.mini = this.mini; }
    if (this.primary != null) { props.primary = this.primary; }
    if (this.src != null) { props.src = this.src; }
    if (this.value != null) { props.value = this.value; }

    return props;
};

/**
 * Helper function to create callback for onBlur event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._onBlurCallback = function(event) {
    if (this.disabled == true) {
        return true;
    }
    // Prevent errors during page submit, when modules have not been loaded.
    try {
        // Set style class.
        this._domNode.className = this._getClassName();
    } catch (err) {}
    return true;
};

/**
 * Helper function to create callback for onFocus event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._onFocusCallback = function(event) {
    if (this.disabled == true) {
        return true;
    }
    // Prevent errors during page submit, when modules have not been loaded.
    try {
        // Set style class.
        this._domNode.className = this._getHoverClassName();
    } catch (err) {}
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._domNode.name = this.id;
    }

    // Initialize deprecated public functions. 
    //
    // Note: Although we now have a setProps function to update properties,
    // these functions were previously added to the DOM node; thus, we must
    // continue to be backward compatible.
    
    /** @ignore */
    this._domNode.isSecondary = function() { return !(woodstock4_3.widget.common.getWidget(this.id).getProps().primary); };
    /** @ignore */
    this._domNode.setSecondary = function(secondary) { return woodstock4_3.widget.common.getWidget(this.id).setProps({primary: !secondary}); };
    /** @ignore */
    this._domNode.isPrimary = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().primary; };
    /** @ignore */
    this._domNode.setPrimary = function(primary) { return woodstock4_3.widget.common.getWidget(this.id).setProps({primary: primary}); };
    /** @ignore */
    this._domNode.isMini = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().mini; };
    /** @ignore */
    this._domNode.setMini = function(mini) { return woodstock4_3.widget.common.getWidget(this.id).setProps({mini: mini}); };
    /** @ignore */
    this._domNode.getDisabled = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().disabled; };
    /** @ignore */
    this._domNode.setDisabled = function(disabled) { return woodstock4_3.widget.common.getWidget(this.id).setProps({disabled: disabled}); };
    /** @ignore */
    this._domNode.getVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().visible; };
    /** @ignore */
    this._domNode.setVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setProps({visible: show}); };
    /** @ignore */
    this._domNode.getText = function() { return woodstock4_3.widget.common.getWidget(this.id).getProps().value; };
    /** @ignore */
    this._domNode.setText = function(text) { return woodstock4_3.widget.common.getWidget(this.id).setProps({value: text}); };
    this._domNode.doClick = this._domNode.click;

    // Set events.
    this._dojo.connect(this._domNode, "onblur", this, "_onBlurCallback");
    this._dojo.connect(this._domNode, "onfocus", this, "_onFocusCallback");
    this._dojo.connect(this._domNode, "onmouseout", this, "_onBlurCallback");
    this._dojo.connect(this._domNode, "onmouseover", this, "_onFocusCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.button.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.alt != null) { this._domNode.alt = props.alt; }
    if (props.align != null) { this._domNode.align = props.align; }

    // Set disabled.
    if (props.disabled != null) { 
        this._domNode.disabled = new Boolean(props.disabled).valueOf();
    }

    // Set image source.
    if (props.src) {
        // If context path is provided, then check whether the image has
        // context path already appended and if not, append it.
        if (this.prefix) {
            props.src = 
                woodstock4_3.widget.common._appendPrefix(this.prefix, props.src);                
        }
        this._domNode.src = props.src; 
    }

    // Set value -- an empty string is valid.
    if (props.value != null) {
        // If escape is true, we want the text to be displayed literally. To 
        // achieve this behavior, do nothing.
        //
        // If escape is false, we want any special sequences in the text 
        // (e.g., "&nbsp;") to be displayed as evaluated (i.e., unescaped).
        this._domNode.value = (new Boolean(this.escape).valueOf() == false)
            ? this._proto._unescapeHTML(props.value)
            : props.value;
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.button.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.calendar");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.calendar");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a calendar widget.
 *
 * @constructor
 * @name woodstock4_3.widget.calendar
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the calendar widget.
 * <p>
 * The calendar widget displays an icon that displays a small calendar when 
 * clicked. Typically, this is used in conjuction with the calendarField widget
 * or a text field used to input a date. The user can either type directly into
 * the text field or select a date from the calendar display.
 * </p><p>
 * By default, the calendar widget accepts dates between the current date and 
 * hundred years out. The years shown in the calendar reflect this range. To
 * specify a different range of date, use the minDate and maxDate properties.
 * </p><p>
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       dateFormat: "MM/dd/yyyy",
 *       id: "cal1",
 *       maxDate: "05/12/2108",
 *       minDate: "05/12/1908",
 *       todayDateMsg: "Today: May 12, 2008",
 *       widgetType: "calendar"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * calendar is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       dateFormat: "MM/dd/yyyy",
 *       id: "cal1",
 *       maxDate: "05/12/2108",
 *       minDate: "05/12/1908",
 *       todayDateMsg: "Today: May 12, 2008",
 *       widgetType: "calendar"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Calendar State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cal1"); // Get calendar
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the calendar is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cal1",
 *       maxDate: "05/12/2108",
 *       minDate: "05/12/1908",
 *       todayDateMsg: "Today: May 12, 2008",
 *       widgetType: "calendar"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Calendar" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cal1"); // Get calendar
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a button using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. When the user 
 * clicks on the checkbox, the input value is updated server-side and the 
 * calendar is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cal1",
 *       maxDate: "05/12/2108",
 *       minDate: "05/12/1908",
 *       todayDateMsg: "Today: May 12, 2008",
 *       widgetType: "calendar"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Change Calendar Date" },
 *       onClick="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cal1"); // Get calendar
 *       return widget.refresh("cb1"); // Asynchronously refresh while submitting checkbox value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "cal1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.calendar.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {Object} closeButtonLink 
 * @config {String} date 
 * @config {String} dateFormat 
 * @config {Object} decreaseLink 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} increaseLink 
 * @config {Object} maxDate
 * @config {Object} minDate
 * @config {Object} monthMenu
 * @config {String} style Specify style rules inline.
 * @config {Object} todayDateMsg
 * @config {Object} toggleLink
 * @config {boolean} visible Hide or show element.
 * @config {Object} yearMenu
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.calendar", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    _widgetType: "calendar" // Required for theme properties.
});

/**
 * Helper function to add a day link in a cell.
 *
 * @param {Node} rowNodeClone
 * @param {Object} day
 * @param {String} id
 * @param {String} className
 * @param {boolean} setFocus
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._addDayLink = function(rowNodeClone, day, 
        id, className, setFocus) {
    // Clone <td> and <a> elements. 
    var colNodeClone = this._dayColumnContainer.cloneNode(false);
    rowNodeClone.appendChild(colNodeClone);    
    var linkNodeClone = this._dayLinkContainer.cloneNode(false);            
    colNodeClone.appendChild(linkNodeClone);
    
    // Format the date.      
    var formattedDate = this._formatDate(day.getMonth() + 1, day.getDate(), day.getFullYear()); 
  
    // set the link's properties for this day.
    linkNodeClone.title = formattedDate;
    linkNodeClone.id = id;
    linkNodeClone.href = "#";
    linkNodeClone.className = className;

    // NOTE: If you set this value manually, text must be HTML escaped.
    this._widget._addFragment(linkNodeClone, "" + day.getDate());

    var widgetId = this.id;
    linkNodeClone.onclick = function() { 
        woodstock4_3.widget.common.getWidget(widgetId)._daySelected(formattedDate); 
        return false;
    };  
    
    // If the setFocus is set to true, then when you tab out of the linkNode,
    // the focus should go to the close button. 
    if (setFocus) {
        linkNodeClone.onkeydown = function(event) {
            var widget = woodstock4_3.widget.common.getWidget(widgetId);
            
            // Get hold of the close button and set focus on it.
            var evt = (event) ? event : ((window.event) ? window.event : null);
            if (evt.keyCode == 9) {
                var elem = document.getElementById(widget.closeButtonLink.id);
                if (elem != null) {
                    if (elem.focus) {
                        elem.focus();
                    }
                    if (evt.preventDefault) {
                        evt.preventDefault();
                    } else {
                        evt.returnValue = false;
                    }
                    return false;                                  
                }
            }
            return true;
        };
    }
    return true;
};

/**
 * Helper function to add days in the month -- week data rows.
 *
 * @param {Object} currentValue The current value of the text field.
 * @param {boolean} initialize Flag indicating to initialze the year and month menus
 * with the current value. The value is true only when the calendar is opened.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._addDaysInMonth = function(currentValue, initialize) {
    // Date representing a day in a month.
    var day;    
    // Number of columns in a row.
    var column = 0;
    // Row number.    
    var rowNum = 0;    
    // Today's day.
    var today = 0;
    // Selected date.
    var selected = 0;     
    // Day link number
    var linkNum = 0; 
    // Prefix used for a day link id.
    var id = this.id + "_link:";
    // Day link id. ie, id + linkNum.
    var linkId;
    // One day in milliseconds -- 1000 * 60 * 60 * 24    
    var oneDayInMs = 86400000;     

    var todayDate = new Date();
    var todayYear = todayDate.getFullYear();
    var todayMonth = todayDate.getMonth() + 1;
    var todayDay = todayDate.getDate();                  
    
    // selectedYear, selectedMonth, selectedDay:
    // The date to show as highlighted (currentValue) provided
    // that the user is viewing that month and year.
    var selectedYear = null;
    var selectedMonth = null;
    var selectedDay = null;
    if (currentValue != null) {
        selectedYear = currentValue.getFullYear();
        selectedMonth = currentValue.getMonth() + 1;
        selectedDay = currentValue.getDate();
    }
    
    // Get month and year menu widgets.
    var monthMenuWidget = this._widget.getWidget(this.monthMenu.id);        
    var yearMenuWidget = this._widget.getWidget(this.yearMenu.id);
    if (monthMenuWidget == null || yearMenuWidget == null) {
        return false;
    }
               
    if (initialize) {
         // Set showMonth as selected in the month menu
	 // Set showYear as selected in the year menu
         // Use todayMonth and todayYear if currentValue is null.
	 var showMonth = todayMonth;
	 var showYear = todayYear;
	 if (currentValue != null) {
             // We have a currentValue, so use that for showMonth and showYear
             showMonth = selectedMonth;
	     showYear = selectedYear;
         }         
         this._setLimitedSelectedValue(monthMenuWidget.getSelectElement(), showMonth);
         this._setLimitedSelectedValue(yearMenuWidget.getSelectElement(), showYear);
    }
    
    var month = parseInt(monthMenuWidget.getSelectedValue());
    var year = parseInt(yearMenuWidget.getSelectedValue());
    
    //set selected
    if (currentValue != null && selectedYear == year && selectedMonth == month) {
        selected = selectedDay;
    }
        
    //set today
    if (todayYear == year && todayMonth == month) {
        today = todayDay;
    }
    
    // Add first week data row.
    var rowNodeClone = this._weekRowContainer.cloneNode(false);
    this._tbodyContainer.appendChild(rowNodeClone); 
    rowNodeClone.id = this.id + ":row" + rowNum;
    
    // Convert to javascript month numbering.
    month--;
    
    // Calculate the first of the main month to display in "first" row.
    var first = new Date(year, month, 1);                         
    var firstDay = first.getDay();    
    var className = this._theme.getClassName("DATE_TIME_OTHER_LINK");
    if (firstDay == this.firstDayOfWeek - 1) {
        // First cell on first row is the first of the current month
        day = first;
    } else {
        // First cell on first row is in previous month.
        var backDays = (firstDay - (this.firstDayOfWeek - 1) + 7) % 7;        
        
        // Calculate the date of first cell on first row in previous month.
        day = new Date(first.getTime() - backDays * oneDayInMs);        
        
        // Generate start of first row up to first of month
        while (day.getMonth() !=  first.getMonth()) {
            linkId = id + linkNum;
            this._addDayLink(rowNodeClone, day, linkId, className);
            day = new Date(day.getTime() + oneDayInMs);
            column++;
            linkNum++;            
        }
    }

    // Add any cells in the first row of the main month.
    while (column < 7) {
        // Set appropriate class name.
        if (day.getDate() == selected) {
            className = this._theme.getClassName("DATE_TIME_BOLD_LINK");
        } else if (day.getDate() == today) {
            className = this._theme.getClassName("DATE_TIME_TODAY_LINK");
        } else {
           className = this._theme.getClassName("DATE_TIME_LINK");
        }
            
        linkId = id + linkNum;
        this._addDayLink(rowNodeClone, day, linkId, className);        
        day = new Date(day.getTime() + oneDayInMs);
        column++;
        linkNum++;
    } 
    
    // This variable is used to decide whether the focus should be set
    // on the calendar's close button when tabbing    
    var setFocus = false;            
    
    // Add intermediate rows
    while (day.getDate() != 1) {
        rowNum++;
        // Clone a <tr> node
        rowNodeClone = this._weekRowContainer.cloneNode(false);
        this._tbodyContainer.appendChild(rowNodeClone); 
        rowNodeClone.id = this.id + ":row" + rowNum;

        column = 0;
        while (column < 7 && day.getDate() != 1) {            
            // Set appropriate class name.
            if (day.getDate() == selected) {
                className = this._theme.getClassName("DATE_TIME_BOLD_LINK");
            } else if (day.getDate() == today) {
                className = this._theme.getClassName("DATE_TIME_TODAY_LINK");
            } else {
                className = this._theme.getClassName("DATE_TIME_LINK");
            }
                 
            linkId = id + linkNum;
            var tmpDate = new Date(day.getTime() + oneDayInMs);

            // On some platforms, the date is not incremented correctly (e.g.,
            // October 28th 1990, 2007, and 2012). In this case, try again.
            if (tmpDate.getDate() == day.getDate()) {
                tmpDate = new Date(tmpDate.getTime() + oneDayInMs);
            } 
            
            // Check whether this is the last date in the calendar and if so
            // set the setFocus variable to true. This will mean that when the
            // user tabs away from this day, the focus will be set on the
            // close button.
            if (tmpDate.getDate() == 1 && column == 6) {
                setFocus = true;
            } else {
                setFocus = false;
            }
            this._addDayLink(rowNodeClone, day, linkId, className, setFocus);
            day = tmpDate;
            column++;
            linkNum++;
        }
    }
    
    // Add any cells in the last row of the following month
    while (column < 7) {
        var className = this._theme.getClassName("DATE_TIME_OTHER_LINK");
        linkId = id + linkNum;
        
        // Check whether this is the last date in the calendar and if so
        // set the setFocus variable to true. This will mean that when the
        // user tabs away from this day, the focus will be set on the
        // close button.        
        if (column == 6) {
            setFocus = true;
        } else {
            setFocus = false;
        }
        this._addDayLink(rowNodeClone, day, linkId, className, setFocus);                    
        day = new Date(day.getTime() + oneDayInMs);
        column++;
        linkNum++;
    }
    return true;
};

/**
 * Helper function to add the week day headers row.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._addWeekDays = function() {            
    var colNodeClone;
    var spanNodeClone;    
    var firstDay = this.firstDayOfWeek - 1;
    
    // Clone the <tr> node and append it to <tbody>
    var rowNodeClone = this._weekDayRow.cloneNode(false);
    this._tbodyContainer.appendChild(rowNodeClone);
        
    for (var i = 0; i < 7; i++) {
        // Clone the <th> node and append it to <tr>
        colNodeClone = this._weekDayColumn.cloneNode(false);
        rowNodeClone.appendChild(colNodeClone);
               
        // Clone the <span> node and append it to <th>
        spanNodeClone = this._weekDayContainer.cloneNode(false);
        colNodeClone.appendChild(spanNodeClone);
        
        // NOTE: If you set this value manually, text must be HTML escaped.
        this._widget._addFragment(spanNodeClone, this.weekDays[firstDay]);

        firstDay++;
        if (firstDay == 7) {
            firstDay = 0;
        }     
    }
    return true;
};

/**
 * Close the calendar if the enter key is pressed and set the initial focus
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */            
woodstock4_3.widget.calendar.prototype._closeCalendar = function(event) {
    var evt = (event) ? event : ((window.event) ? window.event : null);
     
    // If key pressed and enter, then close the menu
    if ((evt.type == "keydown") && (evt.keyCode == 13)) {
        this._toggleCalendar();
        document.getElementById(this.toggleLink.id).focus();        
        return false;
    } 
    this._setInitialFocus();
    return true;    
};

/**
 * Process day selected event.
 * <p>
 * When a day link is selected, an event is published which the 
 * calendarField widget will use to update its text field.
 * </p>
 * @param {String} Formatted date string.
 * @return {boolean} false to cancel JavaScript event.
 * @private
 */
woodstock4_3.widget.calendar.prototype._daySelected = function(formattedDate) {
    this._toggleCalendar();    
    this._publish(woodstock4_3.widget.calendar.event.day.selectedTopic, [{
        id: this.id,
        date:formattedDate
    }]);
    return false;
};

/**
 * This function is used to decrease the month by one.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._decreaseMonth = function() {
    var monthMenu = this._widget.getWidget(this.monthMenu.id).getSelectElement();
    // If the monthMenu has no value, set it to January (that's what
    // it will have appeared like in the browser). Can happen on IE.  
    if (monthMenu.value == null) {
        monthMenu.value = monthMenu.options[0].value;
    }
    
    var month = parseInt(monthMenu.value);
    if (month == 1) {
        var yearMenu = this._widget.getWidget(this.yearMenu.id).getSelectElement();        
         if (yearMenu.value == null) {
             // If the yearMenu has no value, set it to the first available year            
             // (that's what it will have appeared like in the browser). Can happen on IE.
             yearMenu.value = yearMenu.options[0].value;
         } else if (yearMenu.value == yearMenu.options[0].value) {
             // No need to update the calendar in this case,
             // we don't change anything.
             return false;           
         } else {
             // Decrease the year by one and set the month to December
             var year = parseInt(yearMenu.value);
             year--;
             yearMenu.value = year;
             month = 12;
         }
    } else {
        month--;
    }
    monthMenu.value = month;    
    return this._updateMonth(false);
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.calendar.event =
        woodstock4_3.widget.calendar.prototype.event = {
    /**
     * This object contains day event topics.
     * @ignore
     */
    day: {
        /** Day event topic for custom AJAX implementations to listen for. */
        selectedTopic: "woodstock4_3_widget_calendar_event_selected"
    },

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_calendar_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_calendar_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_calendar_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_calendar_event_state_end"
    },

    /**
     * This object contains toggle event topics.
     * @ignore
     */
    toggle: {
        /** Open event topic for custom AJAX implementations to listen for. */
        openTopic: "woodstock4_3_widget_calendar_event_toggle_open",

        /** Close event topic for custom AJAX implementations to listen for. */
        closeTopic: "woodstock4_3_widget_calendar_event_toggle_close"
    }
};

/**
 * Helper function to format the date.
 *
 * @param {String} month
 * @param {String} day
 * @param {String} year
 * @return {String} The date format.
 * @private
 */
woodstock4_3.widget.calendar.prototype._formatDate = function(month, day, year) {
    var date = new String(this.dateFormat);      
    date = date.replace("yyyy", new String(year));
    if (month < 10) {
        date = date.replace("MM", "0" + new String(month));
    } else {
        date = date.replace("MM", new String(month));
    }
    if (day < 10) {
        date = date.replace("dd", "0" + new String(day));
    } else {
        date = date.replace("dd", new String(day));
    }
    return date;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.calendar.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);
    
    // Set properties.
    if (this.todayDateMsg != null) { props.todayDateMsg = this.todayDateMsg; }
    if (this.spacerImage != null) { props.spacerImage = this.spacerImage; }
    if (this.topLeftImage != null) { props.topLeftImage = this.topLeftImage; }
    if (this.topRightImage != null) { props.topRightImage = this.topRightImage; }
    if (this.closeButtonLink != null) { props.closeButtonLink = this.closeButtonLink; }
    if (this.increaseLink != null) { props.increaseLink = this.increaseLink; }
    if (this.decreaseLink != null) { props.decreaseLink = this.decreaseLink; }
    if (this.monthMenu != null) { props.monthMenu = this.monthMenu; }
    if (this.yearMenu != null) { props.yearMenu = this.yearMenu; }   
    if (this.firstDayOfWeek != null) { props.firstDayOfWeek = this.firstDayOfWeek; }
    if (this.toggleLink != null) { props.toggleLink = this.toggleLink; }
    if (this.weekDays != null) { props.weekDays = this.weekDays; }    
    if (this.maxDate != null) { props.maxDate = this.maxDate; }
    if (this.minDate != null) { props.minDate = this.minDate; }
    
    return props;
};

/**
 * Workaround IE bug where popup calendar appears under other components.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._ieStackingContextFix = function() {
    var div = this._calendarContainer;
    if (div.style.display == "block") {
        // This popup should be displayed
        // Get the current zIndex for the div
        var divZIndex = div.currentStyle.zIndex;
        
        // Propogate the zIndex up the offsetParent tree
        var tag = div.offsetParent;
        while (tag != null) {
            var position = tag.currentStyle.position;
            if (position == "relative" || position == "absolute") {

                // Save any zIndex so it can be restored
                tag.raveOldZIndex = tag.style.zIndex;

                // Change the zIndex
                tag.style.zIndex = divZIndex;
            }
            tag = tag.offsetParent;
        }

        // Hide controls unaffected by z-index
        this._ieShowShim();
    } else {
        // This popup should be hidden so restore zIndex-s
        var tag = div.offsetParent;
        while (tag != null) {
            var position = tag.currentStyle.position;
            if (position == "relative" || position == "absolute") {
                if (tag.raveOldZIndex != null) {
                    tag.style.zIndex = tag.raveOldZIndex;
                }
            }
            tag = tag.offsetParent;
        }
        this._ieHideShim();
    }
    return true;
};

/**
 * Hides components unaffected by z-index.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._ieShowShim = function() {  
    var popup = this._calendarContainer;
    var shim = this._shimContainer;
    
    shim.style.position = "absolute";
    shim.style.left = popup.style.left;
    shim.style.top = popup.style.top;
    shim.style.width = popup.offsetWidth;
    shim.style.height = popup.offsetHeight;
    shim.style.zIndex = popup.currentStyle.zIndex - 1;
    shim.style.display = "block";

    return true;
};

/**
 * Hide the shim iframe.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._ieHideShim = function() {
    var shim = this._shimContainer;
    shim.style.display = "none";
    return true;
};

/**
 * This function is used to increment the current month.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._increaseMonth = function() {            
    var monthMenu = this._widget.getWidget(this.monthMenu.id).getSelectElement();          
    
    // If the monthMenu has no value, set it to January (that's what
    // it will have appeared like in the browser). Can happen on IE. 
    if (monthMenu.value == null) {
        monthMenu.value = monthMenu.options[0].value;
    }
    
    var month = parseInt(monthMenu.value);
    if (month == 12) {
        var yearMenu = this._widget.getWidget(this.yearMenu.id).getSelectElement();
        var numOptions = yearMenu.options.length;
        if (yearMenu.value == null) {
            // If the yearMenu has no value, set it to the first available year            
            // (that's what it will have appeared like in the browser). Can happen on IE.
            yearMenu.value = yearMenu.options[0].value;
        } else if (yearMenu.value == yearMenu.options[numOptions-1].value) {
            // No need to update the calendar in this case,
            // we don't change anything.
            return false;            
        } else {
            // Increase the year by one and set the month to January.
            var year = parseInt(yearMenu.value);
            year++;
            yearMenu.value = year;
            month = 1;
        }
    } else {
        month++;
    }
    monthMenu.value = month;   
    return this._updateMonth(false);    
};

/**
 * This function returns a JSON array of months to be displayed in the month 
 * drop down. 
 * return {Object} A JSON array of months.
 * @private
 */
woodstock4_3.widget.calendar.prototype._getMonthOptions = function() {
    var monthMenu = new Array();
    
    // Get the number of months in a calendar year.
    // Some calendars may have more than 12 months a year.
    var numOfMonths = parseInt(this._theme.getMessage("calendar.numOfMonths"));
    
    for ( var i = 0; i < numOfMonths; i++ ) {
        monthMenu[i] = {};
        monthMenu[i].value = i+1;
        monthMenu[i].disabled = false;
        monthMenu[i].separator = false;
        monthMenu[i].escape = true;
        monthMenu[i].group = false;
        monthMenu[i].label=this._theme.getMessage("calendar."+i);
    }    
    return monthMenu;
};

/**
 * This function returns a JSON array of years to be displayed in the year
 * drop down
 * @param {String} minYear the minimum year of the calendar display
 * @param {String} maxYear the maximum year of the calendar display
 * @return {Object} A JSON array of calendar years.
 * @private
 */
woodstock4_3.widget.calendar.prototype._getYearOptions = function(minYear, maxYear) {    
    var yearMenu =new Array();       
    var diff = maxYear - minYear;
    for ( var i = 0; i <= diff; i++ ) {
        yearMenu[i] = {};
        yearMenu[i].value = minYear;
        yearMenu[i].disabled = false;
        yearMenu[i].separator = false;
        yearMenu[i].escape = true;
        yearMenu[i].group = false;
        yearMenu[i].label=minYear;
        minYear++;
    }
    return yearMenu;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._postCreate = function () {
    // Set ids. 
    if (this.id) {
        this._calendarMenuContainer.id = this.id + "_calendarMenuContainer";
        this._linkNode.id = this.id + "_linkNodeContainer";
        this._todayDateContainer.id = this.id + "_todayDateContainer";
        this._closeButtonContainer.id = this.id + "_closeButtonContainer";
        this._previousLinkContainer.id = this.id + "_previousLinkContainer";
        this._monthMenuContainer.id = this.id + "_monthMenuContainer";
        this._nextLinkContainer.id = this.id + "_nextLinkContainer";
        this._yearMenuContainer.id = this.id + "_yearMenuContainer";
        this._shimContainer.id = this.id + "_shim";
    }

    // Create client side widgets for the calendar.
    // When the _setProps() function is called, these widgets will be
    // instantiated via the props param. 

    // If toggle link is null, create the image hyperlink.
    if (this.toggleLink == null) {
        this.toggleLink = {
            id: this.id + "_datePickerLink",
            contents: [],
            imagePosition: "left",
            title: this._theme.getMessage("calendar.popupImageAlt"),
            enabledImage: {
                border: 0,
                icon: "CALENDAR_BUTTON",
                id: this.id + "_datePickerLink_image",
                widgetType: "image"
            },
            disabledImage: {
                border: 0,
                icon: "CALENDAR_BUTTON_DISABLED",
                id: this.id + "_datePickerLink_image_disabled",
                widgetType: "image"
            },
            align:"middle",
            widgetType: "imageHyperlink"
        };
    }

    // Create the spacer image.
    if (this.spacerImage == null) {
        this.spacerImage = {
            icon: "DOT",
            id: this.id + ":DOT",
            widgetType: "image"
        };
    }
    
    // Create the top left image.
    if (this.topLeftImage == null) {
        this.topLeftImage = {
            icon: "SCHEDULER_TOP_LEFT",
            id: this.id + ":topLeft",
            widgetType: "image"
        };        
    }        
        
    //Create the top right image.
    if (this.topRightImage == null) {
        this.topRightImage = {
            icon: "SCHEDULER_TOP_RIGHT",
            id: this.id + ":topRight",
            widgetType: "image"
        };        
    }

    // Create the increase link imageHyperlink widget.
    if (this.increaseLink == null) {
        this.increaseLink = {
            id: this.id + ":nextMonthLink",
            enabledImage: {
                border: 0,
                icon: "SCHEDULER_FORWARD",
                id: this.id + ":nextMonthLink_image",
                widgetType: "image"
            },
            title: this._theme.getMessage("CalendarMonth.goForward"),
            widgetType: "imageHyperlink"
        };
    }   
    
    // Create the decrease link imageHyperlink widget.
    if (this.decreaseLink == null) {
        this.decreaseLink = {
            "id": this.id + ":previousMonthLink",
            enabledImage: {
                border: 0,
                icon: "SCHEDULER_BACKWARD",
                id: this.id + ":previousMonthLink_image",
                widgetType: "image"
            },
            title: this._theme.getMessage("CalendarMonth.goBack"),
            widgetType: "imageHyperlink"
        };
    }        
    
    // Create the close button link imageHyperlink widget
    if (this.closeButtonLink == null) {
        this.closeButtonLink = {
            id: this.id + ":closeButtonLink",
            enabledImage: {
                border: 0,
                icon: "CALENDAR_CLOSE_BUTTON",
                id: this.id + "closeButtonLink_close",
                widgetType: "image"
            },
            title: this._theme.getMessage("CalendarMonth.close"),
            className: this._theme.getClassName("CALENDAR_CLOSE_BUTTON"),
            widgetType: "imageHyperlink"
        };    
    }
    
    // If the dateFormatPattern is null, get one from the themes.
    if (this.dateFormat == null) {
        this.dateFormat = this._theme.getMessage("calendar.dateFormat");
    }
    
    // If the minDate and maxDate are not specified, create a default values.
    // The minDate's year is set to 100 years previous to the current year
    // and maxDate is set to 200 years forward from the minDate's year'
    var minDate = new Date();
    var maxDate = new Date();
    if (this.minDate == null) {
        minDate.setFullYear(minDate.getFullYear() - 100);
        this.minDate = this._formatDate(minDate.getMonth(), 
            minDate.getDate(), minDate.getFullYear());        
    } else {
        minDate = this._convertStringToDate(this.minDate);
    } 

    if (this.maxDate == null) {
        maxDate.setFullYear(minDate.getFullYear() + 200);
        this.maxDate = this._formatDate(maxDate.getMonth(), 
            maxDate.getDate(), maxDate.getFullYear());
    } else {
        maxDate = this._convertStringToDate(this.maxDate);
    }             
  
    // Initialize the days of the week.
    if (this.weekDays == null) {
        this.weekDays = new Array();
        this.weekDays[0] = this._theme.getMessage("CalendarMonth.weekdaySun");
        this.weekDays[1] = this._theme.getMessage("CalendarMonth.weekdayMon");
        this.weekDays[2] = this._theme.getMessage("CalendarMonth.weekdayTue");                
        this.weekDays[3] = this._theme.getMessage("CalendarMonth.weekdayWed");
        this.weekDays[4] = this._theme.getMessage("CalendarMonth.weekdayThu");
        this.weekDays[5] = this._theme.getMessage("CalendarMonth.weekdayFri");
        this.weekDays[6] = this._theme.getMessage("CalendarMonth.weekdaySat");
    }           
    
    // Get the first day of week for that particular locale.
    if (this.firstDayOfWeek == null) {
        this.firstDayOfWeek = parseInt(this._theme.getMessage("calendar.firstDayOfWeek"));
    }
    
    // This will append a localized string along with the
    // today's date
    if (this.todayDateMsg == null) {        
        var d = new Date();
        var todayDateMsg = this._theme.getMessage("CalendarMonth.todayIs");
        
        // Remove the "$0" argument used for the server side param
        var index = todayDateMsg.indexOf(":");
        this.todayDateMsg = todayDateMsg.substr(0, index+1);
        
        var month = this._theme.getMessage(
                        "calendar." + (d.getMonth()));
        month=month.substr(0,3);
        if (this.dateFormat.indexOf("MM") == 0) {
            this.todayDateMsg += " " + month + " " + d.getDay();
        } else {
            this.todayDateMsg += " " + d.getDay() + " " + month;        
        }
        this.todayDateMsg += ", "+d.getFullYear();
    }

    // Initialize the month menu if one does not exist.
    if (this.monthMenu == null) {                  
        this.monthMenu = {
            id: this.id + ":monthMenu",
            options:this._getMonthOptions(),
            title: this._theme.getMessage("CalendarMonth.selectMonth"),
            widgetType: "dropDown"
        };                  
    }
    
    // Initialize the year menu if one does not exist.
    if (this.yearMenu == null) {
        this.yearMenu = {
            id: this.id + ":yearMenu",
            options: this._getYearOptions(minDate.getYear(), maxDate.getYear()),
            title: this._theme.getMessage("CalendarMonth.selectYear"),
            widgetType: "dropDown"
        };          
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to obtain the Date object from the given string
 * date value
 *
 * @param {String} inputDate The date to be converted into Dataee object
 * @param {boolean} yearCheck Check whether the year falls within the specified range
 * @return {Object}  The Date object corresponding to the input String
 * @private
 */
woodstock4_3.widget.calendar.prototype._convertStringToDate = function(inputDate, yearCheck) {   
    if (inputDate == "") {
        property = null;
        return false;
    }
    
    var pattern = this.dateFormat;
    var yearIndex = pattern.indexOf("yyyy");
    var monthIndex = pattern.indexOf("MM");
    var dayIndex = pattern.indexOf("dd");

    // If the format is invalid, set the current value to null
    if (yearIndex < 0 || monthIndex < 0 || dayIndex < 0) {
        return null;
    } 
    
    var counter = 0;
    var number;
    var selectedDate = new Date();
    var found = 0;
    var dateString;

    while (counter < inputDate.length) {
        if (counter == yearIndex) {
            try {
                number = parseInt(inputDate.substr(counter, 4));
                if (isNaN(number)) {
                    property = null;
                    return false;
                }                
                // Check if the input date's year range is inbetween the 
                // allowed dates.   
                if (yearCheck == true) {
                    var index = 0;
                    var foundYear = false;                               
                    yearMenu = this._widget.getWidget(this.yearMenu.id).getSelectElement();
                    while (index < yearMenu.length) {
                        if (number == yearMenu.options[index].value) {
                            selectedDate.setFullYear(number);
                            ++found;
                            foundYear = true;
                            break;
                        }
                        index++;
                    }
                    if (!foundYear) {
                        break;
                    }
                } else {            
                    selectedDate.setFullYear(number);
                    ++found;
                }                    
            } catch(e) {}
        } else if (counter == monthIndex) {
            try {    
                dateString = inputDate.substr(counter, 2);
                // This is a workaround for Firefox! 
                // parseInt() returns 0 for values 08 and 09
                // while all other leading zeros work.
                if (dateString.charAt(0) == '0') {
                    dateString = dateString.substr(1, 1);
                }
                number = parseInt(dateString);
                if (isNaN(number)) {
                    property = null;
                    return false;
                }
                selectedDate.setMonth(number-1);
                ++found;
            } catch(e) {}
        } else if (counter == dayIndex) {
            try {
                dateString = inputDate.substr(counter, 2);
                // This is a workaround for Firefox! 
                // parseInt() returns 0 for values 08 and 09
                // while all other leading zeros work.
                if (dateString.charAt(0) == '0') {
                    dateString = dateString.substr(1, 1);
                }
                number = parseInt(dateString);
                if (isNaN(number)) {
                    return null;
                }
                selectedDate.setDate(number);
                ++found;
            } catch(e) {}
        }
        ++counter;
    }

    if (found == 3) {
        return selectedDate;
    } else {
        return null;
    }    
    return true;       
};

/**
 * Helper function to set the initial focus on the menus.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._setInitialFocus = function() {    
    var pattern = new String(this.dateFormat);
    var yearIndex = pattern.indexOf("yyyy");
    var monthIndex = pattern.indexOf("MM");
    
    // Moving the year menu around based on the date format is not supported yet.
    // So, the code for setting focus on the year menu has been commented out.
    // if (yearIndex < monthIndex) {        
    //    var yearMenu = document.getElementById(this.calendarMonth.yearMenu.id).getSelectElement();
    //    yearMenu.focus();                 
    // } else {
        var monthMenu = this._widget.getWidget(this.monthMenu.id).getSelectElement();          
        monthMenu.focus();
    // }
    return true;
};

/**
 * Set the value of an HTML select element, but limit value to min and max.
 *
 * @param {Node} select The HTML select element.
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._setLimitedSelectedValue = function(select, value) {
    var min = select.options[0].value;
    var max = select.options[select.length - 1].value;
    if (value < min) {        
        select.value = min;
    } else if ( value > max) {        
        select.value = max;
    } else {
        this._setSelectedValue(select, value);        
    }
    return true;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.        
    if (props.todayDateMsg) {
        // NOTE: If you set this value manually, text must be HTML escaped.
        this._widget._addFragment(this._todayDateContainer, props.todayDateMsg);
    }

    if (props.spacerImage) {
        if (!this._widget.getWidget(this.spacerImage.id)) {
            this._widget._addFragment(this._spacerImageContainer, props.spacerImage);
        }
    }

    if (props.topLeftImage) {
        if (!this._widget.getWidget(this.topLeftImage.id)) {
            this._widget._addFragment(this._topLeftImageContainer, props.topLeftImage);
        }
    }

    if (props.topRightImage) {
        if (!this._widget.getWidget(this.topRightImage.id)) {
            this._widget._addFragment(this._topRightImageContainer, props.topRightImage);
        }
    }

    if (props.date) {
        var selDate = this._convertStringToDate(props.date, true);
        if (selDate != null) {
            this.currentValue = selDate;
        }
    }

    // Set close link properties.
    if (props.closeButtonLink) {
        // Set properties.
        props.closeButtonLink.onClick = 
            "woodstock4_3.widget.common.getWidget('" + this.id + "')._toggleCalendar();return false;";
        props.closeButtonLink.onKeyDown = 
            "woodstock4_3.widget.common.getWidget('" + this.id + "')._closeCalendar(event);return false;";       

        // Update/add fragment.
        this._widget._updateFragment(this._closeButtonContainer, 
            this.closeButtonLink.id, props.closeButtonLink);
    }

    // Set decrease link properties.
    if (props.decreaseLink) {
        // Set properties.
        props.decreaseLink.onClick = 
            "woodstock4_3.widget.common.getWidget('" + this.id + "')._decreaseMonth();return false;";

        // Update/add fragment.
        this._widget._updateFragment(this._previousLinkContainer, 
            this.decreaseLink.id, props.decreaseLink);
    }

    // Set increase link properties.
    if (props.increaseLink) {
        // Set properties.
        props.increaseLink.onClick = 
            "woodstock4_3.widget.common.getWidget('" + this.id + "')._increaseMonth();return false;";

        // Update/add fragment.
        this._widget._updateFragment(this._nextLinkContainer, this.increaseLink.id, 
            props.increaseLink);
    }
    
    var minDate = null;
    var maxDate = null;    
    if (props.minDate || props.maxDate) {
        if (props.minDate) {
            
            // Convert the given string to a proper given date format pattern
            // and then store it.
            minDate = this._convertStringToDate(props.minDate);
            if (minDate != null) {
                this.minDate = this._formatDate(minDate.getMonth(), 
                    minDate.getDate(), minDate.getFullYear());
            }
        } 
        if (props.maxDate) {
            // Convert the given string to a proper given date format pattern
            // and then store it.            
            maxDate = this._convertStringToDate(props.maxDate);      
            if (maxDate != null) {
                this.maxDate = this._formatDate(maxDate.getMonth(), 
                    maxDate.getDate(), maxDate.getFullYear());
            }               
        } 
        
        //Recalculate the year options with new minDate and maxDate values.
        props.yearMenu = {
            id: this.id + ":yearMenu",
            options:this._getYearOptions(minDate.getFullYear(), maxDate.getFullYear()),
            title: this._theme.getMessage("CalendarMonth.selectYear"),
            widgetType: "dropDown"
        };  
        
        // update the value of yearMenu
        this.yearMenu = props.yearMenu;                                          
    }
        
    // Set month menu properties
    if (props.monthMenu) {                        
        // Set properties.
        props.monthMenu.onChange =
            "woodstock4_3.widget.common.getWidget('" + this.id + "')._updateMonth(false);return false;";
                         
        // Update/add fragment.
        this._widget._updateFragment(this._monthMenuContainer, this.monthMenu.id,
            props.monthMenu);
    }

    // Set year menu properties.
    if (props.yearMenu) {        
        // Set properties.
        props.yearMenu.onChange =
            "woodstock4_3.widget.common.getWidget('" + this.id + "')._updateMonth(false);return false;";

        // Update/add fragment.
        this._widget._updateFragment(this._yearMenuContainer, this.yearMenu.id,
            props.yearMenu);
    }

    // Set toggle link properties.
    if (props.disabled != null) {
        this.disabled = new Boolean(props.disabled).valueOf(); 
    }

    // If the popup calendar is still being shown, prevent disabling of the calendar.
    // The widget can only be disabled if the popup calendar is not shown.
    if (props.toggleLink || 
        (props.disabled != null && this._calendarContainer.style.display != "block")) {

        // Ensure property exists so we can call setProps just once.
        if (props.toggleLink == null) {
            props.toggleLink = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.toggleLink.disabled = this.disabled;
        props.toggleLink.onClick =
            "woodstock4_3.widget.common.getWidget('" + this.id + "')._toggleCalendar();return false;";

        // Update/add fragment.
        this._widget._updateFragment(this._linkNode, this.toggleLink.id, props.toggleLink); 
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * This function is used to set the value of a select element.
 *
 * @param {Node} select The HTML select element.
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._setSelectedValue = function(select, value) {
    for (var i = 0; i < select.length; i++) {
        if (select.options[i].value == value) {
            select.selectedIndex = i;
            return true;
        }
    }
    select.selectedIndex = -1;
    return true;
};

/**
 * Process toogle event.
 *
 * @return {boolean} false to cancel JavaScript event.
 * @private
 */
woodstock4_3.widget.calendar.prototype._toggleCalendar = function() {
    var topic = (this._calendarContainer.style.display != "block")
        ? woodstock4_3.widget.calendar.event.toggle.openTopic
        : woodstock4_3.widget.calendar.event.toggle.closeTopic;

    // Publish an event for other widgets to listen for.
    //
    // Note: This must be done before the calendar is opened so user
    // input can be applied to the current date.
    this._publish(woodstock4_3.widget.calendar.event.toggle.openTopic, [{
        id: this.id
    }]);

    // Open the calendar.
    if (this._calendarContainer.style.display != "block") {
        if (woodstock4_3.widget.calendar.activeCalendarId != null) {
            var cal = this._widget.getWidget(woodstock4_3.widget.calendar.activeCalendarId);
            cal._toggleCalendar();
        }
        woodstock4_3.widget.calendar.activeCalendarId = this.id;        
        this._calendarContainer.style.display = "block";
        this._setInitialFocus();
        this._updateMonth(true);    
    } else {
        // Hide the calendar popup
        this._calendarContainer.style.display = "none";
        woodstock4_3.widget.calendar.activeCalendarId = null;
    }

    // Test for IE 
    if (woodstock4_3._base.browser._isIe5up()) {
        this._ieStackingContextFix();
    }          
    return false;
};

/**
 * This function is used to update the calendar month.
 * It is called when the calendar is opened, the next or previous
 * links are clicked, or the month or year menus are changed.
 *
 * @param {boolean} initialize Flag indicating to initialze the year and month menus
 * with the current value. The value is true only when the calendar is opened.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendar.prototype._updateMonth = function(initialize) {
    // Remove all the nodes of <tbody> before cloning its children.
    this._widget._removeChildNodes(this._tbodyContainer);    
    // Add week days
    this._addWeekDays();    
    
    // Add days of the month
    this._addDaysInMonth(this.currentValue, initialize);  
    return true;     
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.calendar.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.calendarField");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.calendarField");

woodstock4_3._dojo.require("woodstock4_3.widget.calendar");


woodstock4_3._dojo.provide("woodstock4_3.widget.textField");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.fieldBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");

/**
 * This function is used to construct a textField widget.
 *
 * @constructor
 * @name woodstock4_3.widget.textField
 * @extends woodstock4_3.widget._base.fieldBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the textField widget.
 * <p>
 * The submitForm property can be used to alter the default browser behavior on
 * form submission when Enter key is pressed within a text field. A value of 
 * true will force the form submission while a value of false will prevent it. 
 * If submitForm is not specified, the default browser behavior will take effect.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text field",
 *       widgetType: "textField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * text field is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text field",
 *       widgetType: "textField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Text Field State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text field
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the text field is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text field",
 *       widgetType: "textField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Text Field" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text field
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a text field using the 
 * refresh function. The execute property of the refresh function is used to
 * define the client id which is to be submitted and updated server-side. When
 * the user clicks on the checkbox, the input value is updated server-side and 
 * the text field is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a text field",
 *       widgetType: "textField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Change Text Field Value" },
 *       onClick="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get text field
 *       return widget.refresh("cb1"); // Asynchronously refresh while submitting checkbox value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "field1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.textField.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoValidate
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} size 
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.textField", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.fieldBase ], {
    // Set defaults.
    constructor: function() {
        // Array of list values; may be empty; if null - then no autocomplete 
        // functionality is provided
        this.autoCompleteOptions = null; 
        this.autoCompleteSize = 15;
        this.autoCompleteCloseDelayTime = 100;
    },                  
    _widgetType: "textField" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.textField.event =
        woodstock4_3.widget.textField.prototype.event = {
    /**
     * This object contains filter event topics.
     * @ignore
     */
    autoComplete: {
        /** Filter event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_autoComplete_begin",

        /** Filter event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_autoComplete_end"
    },

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_submit_end"
    },

    /**
     * This object contains validation event topics.
     * @ignore
     */
    validation: {
        /** Validation event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_textField_event_validation_begin",

        /** Validation event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_textField_event_validation_end"
    }
};

/**
 * Utility function to adjust input and list widths to match
 * the one of surrounding domNode node    
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._adjustListGeometry = function () {
    this._listContainer.style.width = this._fieldNode.offsetWidth +"px";
   // this._listContainer.style.left = this._fieldNode.offsetLeft +"px";
    this._listContainer.style.left = "0px";
    this._listContainer.style.top = this._fieldNode.offsetTop + this._fieldNode.offsetHeight+"px";
    this._listNode.style.width = this._fieldNode.offsetWidth+"px";
    this._listContainer.style.zIndex = "999";
    return true;
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget.textField.prototype._getInputClassName = function() {          
    // Set readOnly style.
    if (this._fieldNode.readOnly) {
        return this._theme.getClassName("TEXT_FIELD_READONLY", "");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this._theme.getClassName("TEXT_FIELD_INVALID", "")
        : " " + this._theme.getClassName("TEXT_FIELD_VALID", "");
    
    // Set default style.    
    return (this.disabled == true)
        ? this._theme.getClassName("TEXT_FIELD_DISABLED", "") 
        : this._theme.getClassName("TEXT_FIELD", "") + validStyle;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.textField.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.autoValidate != null) { props.autoValidate = this.autoValidate; }
    if (this.autoComplete != null) { props.autoComplete = this.autoComplete; } 
    if (this.autoCompleteOptions != null) { props.autoCompleteOptions = this.autoCompleteOptions; } //TODO clone array?
        
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._postCreate = function () {
    // Set events.
    if (this.autoValidate == true) {
        // Generate the following event ONLY when 'autoValidate' == true.
        this._dojo.connect(this._fieldNode, "onblur", this, "_validate");
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * Helper function to create callback for autoComplete list closing event.
 *
 * @return {Function} The callback function.
 * @private
 */
woodstock4_3.widget.textField.prototype._createCloseListCallback = function() {
    var _id = this.id;
    return function(event) { 
        var widget = woodstock4_3.widget.common.getWidget(_id);
        if (widget == null) {
            return false;
        }
        widget.showAutoComplete = false;
        widget._updateListView();
        return true;
    };
};

/**
 * Publishes event to filter the options list according to specified 
 * filter string. Note that in default implementation such options will be 
 * updated remotely by Ajax call and then are automatically refreshed on the page.
 *
 * @param {String} filter
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._filterOptions = function() {    
    // Publish the event for custom AJAX implementations to listen for.
    // The implementation of this Ajax call will retrieve the value of the filter
    // and will obtain an updated lookup list ( either locally, or by submit to the server)
    // Data returned from Ajax call will be pushed into this._listWidget.
    this._publish(woodstock4_3.widget.textField.event.autoComplete.beginTopic, [{
        id: this.id
    }]);
    return true;        
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
    
    //we can receive updated autoComplete flag from ajax call back
    if (props.autoComplete != null) { this.autoComplete = new Boolean(props.autoComplete).valueOf();  }
    
    //initialize list, if required    
    //autocomplete list may be initiated only on widget start, and cannot 
    //be introduced as part of dynamic options on setProps
    if (this.autoComplete && this._listWidget == null) {
        //create and populate props for listbox
        var listWidgetProps = {
           id: this.id + "_list",
           onFocus: "woodstock4_3.widget.common.getWidget('" + this.id + 
                "')._processFocusEvent(this.event);", 
           onBlur: "woodstock4_3.widget.common.getWidget('" + this.id + 
                "')._processBlurEvent(this.event);",
           widgetType: "listbox"
        };

        //?? use of event registration as in following disables field processing keys 
        //onChange: "woodstock4_3.widget.common.getWidget('" + this.id + "')._processListChange(this.event);"

        // Fix: Properties should be initialized via postCreate. Then, the list
        // can be updated here instead of recreating the widget each time.
        this._widget._addFragment(this._listContainer, listWidgetProps);
        
        //store reference to the list
        this._listWidget = this._widget.getWidget(listWidgetProps.id);
        this._listNode = this._listWidget.getSelectElement();
        
        //since original list box is created empty, make sure it is not shown
        this._updateListView();
  
         //disable browser autocomplete
         
         //we assume here that once the field is armed for autocomplete,
         //this mode will not get disabled. Thus there is no logic provided
         //for restoration of "autocomplete" attribute
         //
        this._fieldNode.setAttribute("autocomplete", "off");
        
        //use focus event to open the list
        this._dojo.connect(this._fieldNode, "onfocus", this, "_processFocusEvent");
 
        //use blur events to close the list
        this._dojo.connect(this._fieldNode, "onblur", this, "_processBlurEvent");        
         
        // onChange event of the list will change the content of the input field
        this._dojo.connect(this._listNode, "onchange", this, "_processListChange");
 
        //onclick will allow to reopen options list after it has been closed with ESC
        this._dojo.connect(this._fieldNode, "onclick", this, "_processFocusEvent");

        // input field changes will trigger updates to the autoComplete list options
        this._dojo.connect(this._fieldNode, "onkeyup", this, "_processFieldKeyUpEvent");
        
        //additional logic applied to ENTER, ESCAPE, ARROWs on keydown in order to cancel the event bubble
        this._dojo.connect(this._fieldNode, "onkeydown", this, "_processFieldKeyDownEvent");
    }        
    
    if (this.autoComplete && props.autoCompleteOptions != null && this._listWidget != null ) {
        //autoComplete param has changed
        
        //we can receive new options from ajax call
        this.autoCompleteOptions = props.autoCompleteOptions;
        
        // Push properties into listbox.  
        var listProps = {};
        listProps.options = props.autoCompleteOptions;
        
        //change list box size up to autoCompleteSize elem
        if (this.autoCompleteOptions.length > this.autoCompleteSize) {
            listProps.size = this.autoCompleteSize;
        } else if (this.autoCompleteOptions.length == 1) {
            //provide extra space for one-line
            listProps.size = 2;
        } else {
            listProps.size = this.autoCompleteOptions.length;
        }

        this._listWidget.setProps(listProps);
        
        /* // display list on initiation
        this.showAutoComplete = true;        
        */
        this._updateListView();
    }   

    // Set remaining properties.
    var ret = this._inherited("_setProps", arguments);
    
    if (props.autoComplete && props.autoCompleteOptions != null && this._listWidget != null ) {
        this._adjustListGeometry();  //even if autocomplete options are not defined in this set of props
    }
    return ret;
};

/**
 * Process the blur  event - activate delayed timer to close the list.
 * Such timer will be canceled if either list of text field receive 
 * the focus back within a delay period.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processBlurEvent = function(event) {    
    //clear timeout for list closing, thereby preventing list from being closed    
    if (this.closingTimerId) {
        clearTimeout(this.closingTimerId);
        this.closingTimerId = null;
    }
    this.closingTimerId = setTimeout( 
        this._createCloseListCallback(), this.autoCompleteCloseDelayTime);   

    return true;
};

/**
 * Process keyPress events on the field, which enforces/disables 
 * submitForm behavior.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processFieldKeyDownEvent = function(event) {
    event = this._widget._getEvent(event);
    if (event == null) {
        return false;
    }

    if (event.keyCode == this._widget._keyCodes.ENTER && this.autoCompleteIsOpen) {               
        // Disable form submission. Note that form submission is disabled
        // only once when listbox is open. If it closed, form will be submitted
        // according to submitForm flag
        if (window.event) {
            event.cancelBubble = true;
            event.returnValue = false;
        } else{
            event.preventDefault();
            event.stopPropagation();
        } 
        this._fieldNode.value = this._listWidget.getSelectedValue();

        //force close the list box
        this.showAutoComplete = false;
        this._updateListView();    
        
        return true;
    }   

    //close the list in case of ESCAPE pressed
    if (event.keyCode == this._widget._keyCodes.ESCAPE  ) {               
        this._fieldNode.value = ""; //Warning: IE empties all fields on the page on 2nd ESC independently of setting value here
        this.showAutoComplete = false;
        this._updateListView();     
        this._filterOptions();
        return true;
    }
      
    //even when the text field is in focus, we want arrow keys ( up and down)
    //navigate through options in the select list
    if (event.keyCode == this._widget._keyCodes.DOWN_ARROW 
            && this._listWidget.getSelectedIndex() < this._listNode.options.length) {               
        try {
            this.showAutoComplete = true;

            if (!this.autoCompleteIsOpen || this._listNode.options.length <=1) {
                this._filterOptions();
                this._listWidget.setSelectedIndex(0) ;
            } else {     
                //list already open
                this._listWidget.setSelectedIndex(this._listWidget.getSelectedIndex() + 1) ;
                this._updateListView();
            }
            this._processListChange(event);
            return true;
       } catch (doNothing) {}
    }
    if (event.keyCode == this._widget._keyCodes.UP_ARROW && this._listWidget.getSelectedIndex() > 0) {               
        try {
            this.showAutoComplete = true;
            this._listWidget.setSelectedIndex(this._listWidget.getSelectedIndex() - 1) ;
            this._processListChange(event);
            return true;
        } catch (doNothing) {}
    }
    return true;
};

/**
 * Process keyPress events on the filter, which chages the filter
 * and submits a request for updated list.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processFieldKeyUpEvent = function(event) {    
    event = this._widget._getEvent(event);

    if (event != null &&
        ( event.keyCode == this._widget._keyCodes.ESCAPE ||
          event.keyCode == this._widget._keyCodes.ENTER ||
          event.keyCode == this._widget._keyCodes.DOWN_ARROW ||
          event.keyCode == this._widget._keyCodes.UP_ARROW  ||
          event.keyCode == this._widget._keyCodes.SHIFT  ||
          event.keyCode == this._widget._keyCodes.TAB
        )) {
        //these special keys are processed somewhere else - no filtering
        return false; 
    }
    this.showAutoComplete = true;
    this._filterOptions();
    return true;
};

/**
 * Process the focus event - turn on the flag for autoComplete list display
 * <p>
 * Displays autoComplete box on focus 
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processFocusEvent = function(event) {
    //clear timeout for list closing, thereby preventing list from being closed
    if (this.closingTimerId) {
        clearTimeout(this.closingTimerId);
        this.closingTimerId = null;
    }
    return true;
};

/**
 * Process onChange event on the select list, which results in filter being
 * updated with the new value. Note that no data submission is initiated here.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._processListChange = function(event) {    
    event = this._widget._getEvent(event);

    if (event.type == "change") {               
        try {
            this._fieldNode.value = this._listWidget.getSelectedValue();
            
            //close the list
            this.showAutoComplete = false;
            this._updateListView();  
            this._fieldNode.focus(); 
                           
            return true;
       } catch (doNothing) {}
    }    

    // else - usually from selection movement with the arrow keys 
    this._fieldNode.value = this._listWidget.getSelectedValue();
    this._fieldNode.focus(); //keep the focus on filter field so that user can incrementally type additional data
    return true;
};

/**
 * Helper function to update the view of the component ( what is commonly known
 * in UI world as /refresh/ - but renamed not to be confused with Ajax refresh)
 *  - if size of the list box >=1, shows autocomplete list box.
 *  - if size of the list box <1, hides autocomplete list box.
 * 
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._updateListView = function() {
    if ( this.showAutoComplete == true && this.autoCompleteOptions.length >= 1) {
        //TODO a place for optimization here - not to adjust geometry each time
        this._adjustListGeometry();    

        //optionally we could add check for this._listNode.options.length >0
        this._listNode.className = this._theme.getClassName("TEXT_FIELD_AUTO_COMPLETE_LIST", "");

        // the following is preferred way of setting class, but it does not work
        //this._listWidget.setProps({className: this._theme.getClassName("TEXT_FIELD_AUTO_COMPLETE_LIST", "")}) ;
        
        //this.autoCompleteIsOpen flag indicates whether list box is open or not
        this.autoCompleteIsOpen = true;
    } else {
        this._listNode.className = this._theme.getClassName("HIDDEN");
        //this._listWidget.setProps(visible: 'false');
        this.autoCompleteIsOpen = false;
    }
    return true;
};

/**
 * Process validation event.
 * <p>
 * This function interprets an event (one of onXXX events, such as onBlur,
 * etc) and extracts data needed for subsequent Ajax request generation. 
 * Specifically, the widget id that has generated the event. If widget
 * id is found, publishBeginEvent is called with extracted data.
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.textField.prototype._validate = function(event) {
    // Publish an event for custom AJAX implementations to listen for.
    this._publish(woodstock4_3.widget.textField.event.validation.beginTopic, [{
        id: this.id
    }]);
    return true;
};

/**
 * This function is used to construct a calendarField widget.
 *
 * @constructor
 * @name woodstock4_3.widget.calendarField
 * @extends woodstock4_3.widget.textField
 * @class This class contains functions for the calendarField widget.
 * <p>
 * The calendarField widget displays a text field that expects a date as input, 
 * together with an icon that displays a small calendar when clicked. The user
 * can either type directly into the text field or select a date from the 
 * calendar display.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "calField1",
 *       label: { value: "Start Date: " },
 *       size: 20,
 *       widgetType: "calendarField",
 *       calendar: {
 *         dateFormat: "MM/dd/yyyy",
 *         id: "calField1_cal",
 *         maxDate: "05/12/2108",
 *         minDate: "05/12/1908",
 *         todayDateMsg: "Today: May 12, 2008",
 *         widgetType: "calendar"
 *       }
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * calendar field is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "calField1",
 *       label: { value: "Start Date: " },
 *       size: 20,
 *       widgetType: "calendarField",
 *       calendar: {
 *         dateFormat: "MM/dd/yyyy",
 *         id: "calField1_cal",
 *         maxDate: "05/12/2108",
 *         minDate: "05/12/1908",
 *         todayDateMsg: "Today: May 12, 2008",
 *         widgetType: "calendar"
 *       }
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Calendar State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("calField1"); // Get calendar field
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the calendar field is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "calField1",
 *       label: { value: "Start Date: " },
 *       size: 20,
 *       widgetType: "calendarField",
 *       calendar: {
 *         dateFormat: "MM/dd/yyyy",
 *         id: "calField1_cal",
 *         maxDate: "05/12/2108",
 *         minDate: "05/12/1908",
 *         todayDateMsg: "Today: May 12, 2008",
 *         widgetType: "calendar"
 *       }
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Calendar" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("calField1"); // Get calendar field
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a calendar field using the 
 * refresh function. The execute property of the refresh function is used to 
 * define the client id which is to be submitted and updated server-side. As the
 * user types in the text field, the input value is updated server-side and the 
 * calendar field label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "calField1",
 *       label: { value: "Start Date: " },
 *       size: 20,
 *       widgetType: "calendarField",
 *       calendar: {
 *         dateFormat: "MM/dd/yyyy",
 *         id: "calField1_cal",
 *         maxDate: "05/12/2108",
 *         minDate: "05/12/1908",
 *         todayDateMsg: "Today: May 12, 2008",
 *         widgetType: "calendar"
 *       }
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Calendar Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("calField1"); // Get calendar field
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "calField1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.calendarField.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {Object} calendar 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} patternHelp 
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid 
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.calendarField",
        woodstock4_3.widget.textField, {
    // Set defaults.
    _widgetType: "calendarField" // Required for theme properties.
});

/**
 * This function is called when a day link is selected from the calendar.
 * It updates the field with the value of the clicked date.
 *
 * @param props Key-Value pairs of properties.
 * @config {String} id 
 * @config {String} date
 * @return {boolean} false to cancel JavaScript event.
 * @private
 */
woodstock4_3.widget.calendarField.prototype._dayClicked = function(props) {
    // Check whether the calendar associated with this particular calendarField
    // broadcasted the event.
    if (props.date != null && props.id == this.calendar.id) {
        // Set the selected date on the field.
        this._domNode.setProps({value: props.date});
    }
    return false;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.calendarField.event =
        woodstock4_3.widget.calendarField.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_calendarField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_calendarField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_calendarField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_calendarField_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.calendarField.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = this._theme.getClassName("CALENDAR_ROOT","");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.calendarField.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.  
    if (this.align != null) { props.align = this.align; }
    if (this.calendar != null) { props.calendar = this.calendar; }  
    if (this.patternHelp != null) { props.patternHelp = this.patternHelp; }   

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendarField.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._inlineHelpNode.id = this.id + "_pattern";
        this._linkContainer.id = this.id + "_linkContainer";
        this._calendarContainer.id = this.id + "_calendarContainer";
    }     

    // If a patternHelp is not specified by the developer
    // try to get one from the themes. 
    if (this.patternHelp == null) {
        
        var pattern;
        if (this.calendar != null && this.calendar.dateFormat != null) {
            pattern = this.calendar.dateFormat;
        } else {        
            pattern = this._theme.getMessage("calendar.dateFormat");
        }
        var help = this._theme.getMessage("calendar." + pattern);
        if (help != null) {
            this.patternHelp = help;
        } 
    }
    // Set events.

    // Subscribe to the "day clicked" event present in the calendar widget.
    this._widget.subscribe(woodstock4_3.widget.calendar.event.day.selectedTopic,
        this, "_dayClicked");
    // Subscribe to the "toggle" event that occurs whenever the calendar is opened.
    this._widget.subscribe(woodstock4_3.widget.calendar.event.toggle.openTopic,
        this, "_toggleCalendar");
        
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.calendarField.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }
    
    // If the popup calendar is visible, prevent disabling of the calendar.
    // The widget can only be disabled if the popup calendar is not visible.
    if (props.disabled != null) { 
        var widget = this._widget.getWidget(this.calendar.id); 
        if (widget != null && !(widget._calendarContainer.style.display != "block")) {
            props.disabled = this.disabled;
        }        
    }
    
    // Set remaining properties.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendarField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set disabled.
    if (props.disabled != null) { this.disabled = new Boolean(props.disabled).valueOf(); }

    // Set calendar.
    if (props.calendar || props.disabled != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.calendar == null) {
            props.calendar = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.calendar.disabled = this.disabled;
        
        // Update/add fragment.
        this._widget._updateFragment(this._calendarContainer, this.calendar.id,
            props.calendar); 
    }
    
    // Set date format pattern help.
    if (props.patternHelp) {                            
        // NOTE: If you set this value manually, text must be HTML escaped.
        this._widget._addFragment(this._inlineHelpNode, props.patternHelp);
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * This function is used to "start" the widget, after the widget has been
 * instantiated.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendarField.prototype._startup = function () {
    if (this._started == true) {
        return false;
    }

    // Adjust the size of the inline help text so that it fits to the
    // size of the text field.
    //
    // Note: Cannot do this in the _postCreate or setProps as nodes have not
    // been added to the DOM. So, offsetWidth would return zero. 
    var width = this._fieldNode.offsetWidth;
    this._inlineHelpNode.style.cssText = "width:" + width + "px;";

    return this._inherited("_startup", arguments);
};

/**
 * This function is used with the _toggleCalendar function of the calendar widget.
 * Whenever the calendar is opened, it updates the value of the calendar with
 * the value present in the field.
 * 
 * @param props Key-Value pairs of properties.
 * @config {String} id
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.calendarField.prototype._toggleCalendar = function(props) {   
    if (props.id != null && props.id == this.calendar.id) {
        var widget = this._widget.getWidget(props.id);
        widget.setProps({date: this.getProps().value});
    }
    return true;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.calendarField.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.checkbox");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.checkbox");

woodstock4_3._dojo.require("woodstock4_3.widget._base.checkedBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");

/**
 * This function is used to construct a checkbox widget.
 *
 * @constructor
 * @name woodstock4_3.widget.checkbox
 * @extends woodstock4_3.widget._base.checkedBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the checkbox widget.
 * <p>
 * The widget can be used as a single checkbox or as one checkbox among a group
 * of checkboxes. A group of checkboxes represents a multiple selection list 
 * which can have any number of checkboxes selected, or none selected.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cb1",
 *       label: { value: "This is a checkbox" },
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the radio button, the
 * checkbox is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "b1",
 *       label: { value: "This is a checkbox" },
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "rb1",
 *       label: { value: "Toggle Checkbox State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "radioButton"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cb1"); // Get checkbox
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the radio button, the checkbox is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cb1",
 *       label: { value: "This is a checkbox" },
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "rb1",
 *       label: { value: "Refresh Checkbox" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "radioButton"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cb1"); // Get checkbox
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a checkbox using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the checkbox 
 * label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cb1",
 *       label: { value: "This is a checkbox" },
 *       widgetType: "checkbox"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Checkbox Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cb1"); // Get checkbox
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "cb1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.checkbox.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {boolean} checked 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} image 
 * @config {String} label 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect 
 * @config {boolean} readOnly 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.checkbox", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.checkedBase ], {
    // Set defaults.
    _idSuffix: "_cb",
    _widgetType: "checkbox" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.checkbox.event =
        woodstock4_3.widget.checkbox.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkbox_event_refresh_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkbox_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkbox_event_state_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkbox_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkbox_event_submit_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkbox_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.disabled == true)
        ? this._theme.getClassName("CHECKBOX_SPAN_DISABLED", "")
        : this._theme.getClassName("CHECKBOX_SPAN", ""); 

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getImageClassName = function() {
    return (this.disabled == true)
        ? this._theme.getClassName("CHECKBOX_IMAGE_DISABLED", "")
        : this._theme.getClassName("CHECKBOX_IMAGE", "");  
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getInputClassName = function() {
    // readOnly style.
    if (this.readOnly == true) {
        return this._theme.getClassName("CHECKBOX_READONLY", "");        
    }

    // disabled style.
    return (this.disabled == true)
        ? this._theme.getClassName("CHECKBOX_DISABLED", "")
        : this._theme.getClassName("CHECKBOX", "");  
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getLabelDisabledClassName = function(disabled) {
    return (disabled == true)
        ? this._theme.getClassName("CHECKBOX_LABEL_DISABLED", "")
        : this._theme.getClassName("CHECKBOX_LABEL", "");  
};

/**
 * Return an Object Literal of label properties desired
 * by the checkbox widget.
 * <p>
 * This implementation sets
 * the <code>checkbox.labelLevel</code> 
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively.
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} label properties.
 * @private
 */
woodstock4_3.widget.checkbox.prototype._getLabelProps = function() {

    // First see if the super class wants to contribute to the props.
    // Let selectBase add the htmlFor property
    //
    var props = this._inherited("_getLabelProps", arguments);
    props.level = this._theme.getMessage("checkbox.labelLevel", null, 3);
    return props;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.checkbox.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.checkbox.event.submit.beginTopic,
    woodstock4_3.widget._xhr.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.checkboxGroup");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.checkboxGroup");

woodstock4_3._dojo.require("woodstock4_3.widget._base.checkedGroupBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");

/**
 * This function is used to construct a checkboxGroup widget.
 *
 * @constructor
 * @name woodstock4_3.widget.checkboxGroup
 * @extends woodstock4_3.widget._base.checkedGroupBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the checkboxGroup widget.
 * <p>
 * The widget is used to layout an array of checkboxes. To group checkboxes
 * properly, each widget id must be unique, but the name property value is the
 * same.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cbg1",
 *       contents: [{ 
 *         id: "cb1",
 *         label: { value: "This is checkbox 1" },
 *         widgetType: "checkbox"
 *       }, {
 *         id: "cb2",
 *         label: { value: "This is checkbox 2" },
 *         widgetType: "checkbox"
 *       }],
 *       label: { value: "This is a checkbox group" },
 *       widgetType: "checkboxGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the radio button, the
 * checkbox group is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cbg1",
 *       contents: [{ 
 *         id: "cb1",
 *         label: { value: "This is checkbox 1" },
 *         widgetType: "checkbox"
 *       }, {
 *         id: "cb2",
 *         label: { value: "This is checkbox 2" },
 *         widgetType: "checkbox"
 *       }],
 *       label: { value: "This is a checkbox group" },
 *       widgetType: "checkboxGroup"
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "rb1",
 *       label: { value: "Toggle Checkbox Group State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "radioButton"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cbg1"); // Get checkbox group
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the radio button, the checkbox group is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cbg1",
 *       contents: [{ 
 *         id: "cb1",
 *         label: { value: "This is checkbox 1" },
 *         widgetType: "checkbox"
 *       }, {
 *         id: "cb2",
 *         label: { value: "This is checkbox 2" },
 *         widgetType: "checkbox"
 *       }],
 *       label: { value: "This is a checkbox group" },
 *       widgetType: "checkboxGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "rb1",
 *       label: { value: "Refresh Checkbox Group" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "radioButton"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cb1"); // Get checkbox group
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a checkbox group using the 
 * refresh function. The execute property of the refresh function is used to 
 * define the client id which is to be submitted and updated server-side. As the
 * user types in the text field, the input value is updated server-side and the 
 * checkbox group label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "cbg1",
 *       contents: [{ 
 *         id: "cb1",
 *         label: { value: "This is checkbox 1" },
 *         widgetType: "checkbox"
 *       }, {
 *         id: "cb2",
 *         label: { value: "This is checkbox 2" },
 *         widgetType: "checkbox"
 *       }],
 *       label: { value: "This is a checkbox group" },
 *       widgetType: "checkboxGroup"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Checkbox Group Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("cbg"); // Get checkbox group
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "cbg") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.checkboxGroup.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {int} columns 
 * @config {Array} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {boolean} readOnly Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.checkboxGroup", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.checkedGroupBase ], {
    // Set defaults.
    _widgetType: "checkboxGroup" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.checkboxGroup.event =
        woodstock4_3.widget.checkboxGroup.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkboxGroup_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkboxGroup_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_checkboxGroup_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_checkboxGroup_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.checkboxGroup.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.columns > 1)
        ? this._theme.getClassName("CBGRP_HORIZ", "")
        : this._theme.getClassName("CBGRP_VERT", "");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.checkboxGroup.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.dropDown");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.dropDown");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.selectBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");

/**
 * This function is used to construct a dropDown widget.
 *
 * @constructor
 * @name woodstock4_3.widget.dropDown
 * @extends woodstock4_3.widget._base.selectBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the dropDown widget.
 * <p>
 * A dropDown widget can be configured as a jump menu. Jump menus immediately 
 * perform an action, such as opening a window or navigating to another page, 
 * when the user selects an item. This action is accomplished by submitting the 
 * form immediately when the user changes the selection. By default, the 
 * dropDown widget is not a jump menu. The submission of the form does not 
 * occur until the user activates another component such as a button in order 
 * to submit the form. To configure the dropDown widget to render a jump menu, 
 * set the submitForm property to true.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mnu1",
 *       label: { value: "This is a menu" },
 *       options: [{
 *         value: "BOS",
 *         label: "Boston"
 *       }, {
 *         value: "SFO",
 *         label: "San Francisco"
 *       }],
 *       widgetType: "dropDown"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * menu is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mnu1",
 *       label: { value: "This is a menu" },
 *       options: [{
 *         value: "BOS",
 *         label: "Boston"
 *       }, {
 *         value: "SFO",
 *         label: "San Francisco"
 *       }],
 *       widgetType: "dropDown"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Menu State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("mnu1"); // Get menu
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the menu is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mnu1",
 *       label: { value: "This is a menu" },
 *       options: [{
 *         value: "BOS",
 *         label: "Boston"
 *       }, {
 *         value: "SFO",
 *         label: "San Francisco"
 *       }],
 *       widgetType: "dropDown"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Menu" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("mnu1"); // Get menu
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a menu using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the menu
 * label is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mnu1",
 *       label: { value: "This is a menu" },
 *       options: [{
 *         value: "BOS",
 *         label: "Boston"
 *       }, {
 *         value: "SFO",
 *         label: "San Francisco"
 *       }],
 *       widgetType: "dropDown"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Menu Label" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("mnu1"); // Get menu
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "mnu1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.dropDown.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} label Defines the widget and its properties to use for
 * a label.
 * @config {boolean} labelOnTop If true the label appears above the dropdown. 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Option selection is changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Option text is highlighted.
 * @config {Array} options See <code>selectBase._setOptions</code> or
 * <code>selectBase</code> overview for details on the elements of this array.
 * @config {int} size The number of option rows to display.
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm If true the dropDown performs like a
 * "jump menu" and submits the entire form when an option is selected.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title The a HTML title attribue for the <code>select</code> element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.dropDown", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.selectBase ], {
    // Set defaults.
    constructor : function() {
	this.labelOnTop = this._theme._getMessageBoolean("dropDown.labelOnTop", false);
	this.submitForm =  false;
	this.width = this._theme.getMessage("dropDown.width", null);
    },
    _widgetType: "dropDown" // Required for theme properties.
});

/**
 * This function is called by the <code>selectBase::onChangeCallback</code>
 * method in respsonse to an onchange event.
 * <p>
 * If this instance is configured as a "jump" menu,
 * <code>submitForm</code> is true, the containing form is submitted.
 * </p>
 * <p>
 * If this instance is configured as a standard "dropdown" menu, then
 * the superclass's <code>_changed</code> method is called.
 * See <code>selectBase._changed</code>.<br/>
 * See <code>selectBase.onChangeCallback</code>
 * </p>
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._changed = function() {

    // A null submitForm is the same as a standard menu.
    //
    if (this.submitForm != true) {
        // Drop down changed.
        return this._inherited("_changed", arguments);
    } else {

        // Jump drop down changed.
        var jumpDropdown = this._listContainer; 

        // Find the <form> for this drop down
        var form = jumpDropdown.form;
        if (form != null) { 
            this._submitterHiddenNode.value = "true";

            // Set style classes.
            var options = jumpDropdown.options;
            for (var i = 0; i < options.length; i++) { 
                options[i].className = this._getOptionClassName(options[i]);
            }
            form.submit();
        }
    }
    return true; 
};

/**
 * The dropDown defines the following event topics. The submit topics
 * are only valid if the <code>submitForm</code> property is true.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.dropDown.event =
        woodstock4_3.widget.dropDown.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for.*/
        beginTopic: "woodstock4_3_widget_dropDown_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for.*/
        endTopic: "woodstock4_3_widget_dropDown_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_dropDown_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_dropDown_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for.
	 * This topic is only valid if the <code>submitForm</code>
	 * property is <code>true</code>
	 */
        beginTopic: "woodstock4_3_widget_dropDown_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for.
	 * This topic is only valid if the <code>submitForm</code>
	 * property is <code>true</code>
	 */
        endTopic: "woodstock4_3_widget_dropDown_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element CSS class name.
 * <p>
 * Note: Selectors should be concatenated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var newClassName = this._theme.getClassName("DROPDOWN", "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Return an Object Literal of label properties desired by this widget.
 * <p>
 * This implementation sets
 * the <code>dropDown.labelLevel</code> and <code>dropDown.labelAlign</code>
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively. It sets the <code>htmlFor</code> attribute to 
 * <code>_listContainer.id</code>.
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} Key-Value pairs of properties.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getLabelProps = function() {

    var props = this._inherited("_getLabelProps", arguments);
    props.level = this._theme.getMessage("dropDown.labelLevel", null, 2);
    return props;
};

/**
 * Return a CSS selector for the dropDown label.
 * The chosen CSS selector is a function of 
 * the <code>labelOnTop</code> property and the whether or not the
 * dropdown is a "jump" dropDown. If <code>labelOnTop</code>
 * property is <code>true</code> null is returned. If the 
 * <code>labelOnTop</code> property is true, then the value of the
 * the of the theme keys, <code>MENU_JUMP_LABEL_ALIGN</code> or 
 * <code>MENU_STANDARD_LABEL_ALIGN</code> from the 
 * <code>styles</code> category are returned if the drop down is
 * a "jump" dropDown or plain dropDown, respectively.
 * @param {boolean} ontop The label on top state. In some contexts this state
 * must be passed to the method, for example when called from the 
 * <code>selectBase.setProps</code> method.
 * @return {String} A CSS selector for the dropDown label.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getLabelOnTopClassName = function(ontop) {
    if (ontop != null && ontop == true) {
	return null;
    }
    // Jumpmenu ?
    //
    return (this.submitForm != null && this.submitForm == true)
	? this._theme.getClassName("MENU_JUMP_LABEL_ALIGN", null)
        : this._theme.getClassName("MENU_STANDARD_LABEL_ALIGN", null);
};

/**
 * Return a CSS selector for _listContainer HTML element.
 * The chosen CSS selector is a function of 
 * the <code>disabled</code> property and whether or not the dropDown
 * is a "jump" dropDown. The returned selector is the value of a 
 * property defined in the <code>styles</code> category, as shown 
 * in the following table.
 * <table border="1px">
 * <tr><th>key</th><th>disabled</th><th>jump dropDown</th></tr>
 * <tr>
 * <td>MENU_STANDARD</td>
 * <td>false</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>MENU_STANDARD_DISABLED</td>
 * <td>true</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>MENU_JUMP</td>
 * <td>false</td>
 * <td>true</td>
 * </tr>
 * <tr>
 * <td>MENU_JUMP_DISABLED</td>
 * <td>true</td>
 * <td>true</td>
 * </tr>
 * </table>
 * @return {String} A CSS selector for the _listContainer HTML element.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getListContainerClassName =
        function(disabled, jumpmenu) {
    var key = "MENU_STANDARD";
    if (disabled == true) {
	if (jumpmenu == true) {
	    key = "MENU_JUMP_DISABLED";
	} else {
	    key = "MENU_STANDARD_DISABLED";
	}
    } else if (jumpmenu == true) {
	key = "MENU_JUMP";
    }
    return this._theme.getClassName(key, null);
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.dropDown.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Get properties.
    if (this.submitForm != null) { props.submitForm = this.submitForm; }

    return props;
};

/**
 * This function returns the CSS class name for an HTML option element,
 * based on the state of <code>element</code>. The CSS class name defined by 
 * the following theme keys from the <code>styles</code> theme category
 * is returned for the given states, based on 
 * the properties in <code>element</code>.
 * <p>
 * <table border="1px">
 * <tr><th>state</th><th>key</th></tr>
 * <tr><td>
 * separator == true</td><td>*_OPTION_SEPARATOR</td>
 * </tr><tr><td>
 * group == true</td><td>*_OPTION_GROUP</td>
 * </tr><tr><td>
 * disabled == true</td><td>*_OPTION_DISABLED</td>
 * </tr><tr><td>
 * selected == true and disabled == false</td>
 * <td>*_OPTION_SELECTED</td>
 * </tr><tr><td>
 * separator == false and group false<br/> 
 * 	and disabled == false<br/>
 *	and selected == false</td><td>*_OPTION</td>
 * </tr>
 * </table><br/>
 * Note:  * is <code>MENU_JUMP</code> for a jump menu and
 * <code>MENU_STANDARD</code> if not.
 * </p>
 * <p>
 * <p>
 * If <code>element</code> is null, the CSS class name for the theme
 * key "MENU_STANDARD_OPTION" is returned.
 * </p>
 * @param {Object} element An HTML option DOM node instance.
 * @config {boolean} separator If true, element is a option separator.
 * @config {boolean} group If true the element is a optGroup instance.
 * @config {boolean} disabled If true the element is disabled.
 * @config {boolean} selected If true the element is selected.
 * @return {String} The HTML option element CSS class name.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._getOptionClassName = function(element) {

    var jumpmenu = this.submitForm != null && this.submitForm == true;

    var key = null;
    if (jumpmenu) {
	key = "MENU_JUMP_OPTION";
	if (element != null) {
	    if (element.separator == true) {
		key = "MENU_JUMP_OPTION_SEPARATOR";
	    } else
	    if (element.group == true) {
		key = "MENU_JUMP_OPTION_GROUP";
	    } else
	    if (element.selected == true) {
		key = "MENU_JUMP_OPTION_SELECTED";
	    }

	    // Disabled wins
	    //
	    if (element.disabled == true) {
		key = "MENU_JUMP_OPTION_DISABLED";
	    }
	}
    } else {
	key = "MENU_STANDARD_OPTION";
	if (element != null) {
	    if (element.separator == true) {
		key = "MENU_STANDARD_OPTION_SEPARATOR";
	    } else
	    if (element.group == true) {
		key = "MENU_STANDARD_OPTION_GROUP";
	    } else
	    if (element.selected == true) {
		key = "MENU_STANDARD_OPTION_SELECTED";
	    }

	    // Disabled wins
	    //
	    if (element.disabled == true) {
		key = "MENU_STANDARD_OPTION_DISABLED";
	    }
	}
    }
    return this._theme.getClassName(key);
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._submitterHiddenNode.id = this.id + "_submitter";
    }

    // Set the CSS class name for the select element.
    //
    var jumpmenu = this.submitForm != null && this.submitForm == true;
    var disabled = this.disabled != null && this.disabled == true;

    this._common._addStyleClass(this._listContainer, 
	this._getListContainerClassName(disabled, jumpmenu));
    
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.dropDown.prototype.setProps = function(props, notify) {
    if (props == null) {
	return this._inherited("setProps", arguments);
    }
	
    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    // We need to clear the selector of the _listContainer
    // when state changes from disabled to non disabled state
    // or from submitForm to standard dropdown.
    //
    var isjumpmenu = this.submitForm == true;
    var toggleJumpmenu = props.submitForm != null
        && props.submitForm != isjumpmenu;

    var isdisabled = this.disabled == true;
    var toggleDisabled = props.disabled != null
        && props.disabled != isdisabled;

    // Get current state of the classsname, and strip it 
    // and then add the classname for the new state.
    //
    if (toggleJumpmenu || toggleDisabled) {
	var cn = this._getListContainerClassName(isdisabled, isjumpmenu);
	this._common._stripStyleClass(this._listContainer, cn);

	cn = this._getListContainerClassName(
	    toggleDisabled ? props.disabled == true : isdisabled,
	    toggleJumpmenu ? props.jumpmenu == true : isjumpmenu);
	this._common._addStyleClass(this._listContainer, cn);
    }
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.dropDown.prototype._setProps = function(props) {
    // If props is null, create one to pass default theme values
    // to the superClass via props.
    //
    if (props == null) {
	return this._inherited("_setProps", arguments);
    }

    var jumpmenu = props.submitForm != null && props.submitForm == true;

    // Add attributes to the hidden input for jump drop down.
    //
    if ((jumpmenu && this._submitterHiddenNode.name == null) || (jumpmenu &&
	    this._submitterHiddenNode.name != this._submitterHiddenNode.id)) {

	this._submitterHiddenNode.name = this._submitterHiddenNode.id;
	this._submitterHiddenNode.value = "false";
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.dropDown.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.dropDown.event.submit.beginTopic,
    woodstock4_3.widget._xhr.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.editableField");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.editableField");

woodstock4_3._dojo.require("woodstock4_3.widget.textField");

/**
 * This function is used to construct a editableField widget.
 *
 * @constructor
 * @name woodstock4_3.widget.editableField
 * @extends woodstock4_3.widget.textField
 * @class This class contains functions for the editableField widget.
 * <p>
 * The editableField widget renders a borderless readonly text field that can be
 * edited inline. An initial read-only state resembles regular static text, and
 * it can be edited by using the onDblClick event or when the space key is 
 * pressed. Once the field looses the focus, it enters a read-only state again.
 * </p><p>
 * If the editable field is modified and if autoSave is set to its default 
 * value, which is true, the updated data will be automatically submitted to the
 * server by an Ajax request.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is an editable field",
 *       widgetType: "editableField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * editable field is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is an editable field",
 *       widgetType: "editableField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Editable Field State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get editable field
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the editable field is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is an editable field",
 *       widgetType: "editableField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Editable Field" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get editable field
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a editable field using the 
 * refresh function. The execute property of the refresh function is used to
 * define the client id which is to be submitted and updated server-side. When
 * the user clicks on the checkbox, the input value is updated server-side and 
 * the editable field is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is an editable field",
 *       widgetType: "editableField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Change Editable Field Value" },
 *       onClick="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get editable field
 *       return widget.refresh("cb1"); // Asynchronously refresh while submitting checkbox value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "field1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.editableField.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoSave
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} escape HTML escape button text (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} required 
 * @config {int} size
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} valid Value of input.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.editableField",
        woodstock4_3.widget.textField, {
    // Set defaults.
    constructor: function() {
        this.edit = false;
    },
    _widgetType: "editableField" // Required for theme properties.
});
   
/**
 * Helper function to disable edit mode.
 *
 * This function will commit or rollback the changes made in the field, and
 * setup appropriate style on the field.
 *
 * @param {boolean} acceptChanges Optional parameter. 
 * <p>
 * If true, the entered data (_fieldNode value) will be commited through the Ajax
 * submit() request. 
 * </p><p>
 * If false, the entered data (_fieldNode value) will be rolled back to previos 
 * state (value is saved before field enters editable state).
 * </p><p>
 * If not specified, no changes to the field value will made, and only styles 
 * will be modified.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.editableField.prototype._disableEdit = function(acceptChanges) {
    if (acceptChanges == true) {
        // If savedValue does not exist, we have not edited the field yet
        if (this.autoSave == true && this.savedValue && 
                this.savedValue != this._fieldNode.value) {
            this.submit();
        }
    } else if (acceptChanges == false) {
        if (this.savedValue)   {
            this._fieldNode.value = this.savedValue;
        }
    }
    this.edit = false;
    this.savedValue = null;
    this._fieldNode.className = this._getInputClassName();   
    this._fieldNode.readOnly = true;
    return true;
};

/**
 * Helper function to enable edit mode.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.editableField.prototype._enableEdit = function() {
    // Save the current value.
    this.savedValue = this._fieldNode.value;
        
    this.edit = true;
    this._fieldNode.className = this._getInputClassName();   
    this._fieldNode.readOnly = false;
    this._fieldNode.focus(); // In case function has been called programmatically, not by event.
    this._fieldNode.select();
    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.editableField.event =
        woodstock4_3.widget.editableField.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_editableField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_editableField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_editableField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_editableField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_editableField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_editableField_event_submit_end"
    }
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 * @private
 */
woodstock4_3.widget.editableField.prototype._getInputClassName = function() {    
    // Set default style.
    if (this.disabled == true) {
        return  this._theme.getClassName("EDITABLE_FIELD_DISABLED","");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this._theme.getClassName("EDITABLE_FIELD_INVALID","")
        : ""; 

    return (this.edit == true)
        ? this._theme.getClassName("EDITABLE_FIELD_EDIT","") + validStyle
        : this._theme.getClassName("EDITABLE_FIELD_DEFAULT","") + validStyle;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.editableField.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.autoSave!= null) { props.autoSave = this.autoSave; }

    return props;
};

/**
 * Process user gesture events on the field, such as dblclick, onblur, 
 * onkeyup, etc.
 *
 * @param {Event} event The JavaScript event
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.editableField.prototype._onEditCallback = function(event) {
    if (event == null) {
        return false;
    }
    if (event.type == "dblclick") {
        this._enableEdit();
        return true;
    }
    if (event.type == "blur") {
        this._disableEdit(true);
        return true;
    }
    if (event.type == "keyup") {
        if (this.edit == false) {
            // Currently in readOnly state.
            // Allow <SPACE> key.
            if (event.keyCode == 32) {
                this._enableEdit();
            }
        } else {
            // Currently in edit state.
            if (event.keyCode == 27) this._disableEdit(false); // ESC
        }
        // Otherwise do not do anything.
        return true;
    }
    return true;    
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.editableField.prototype._postCreate = function () {
    // Set Initial readOnly state.
    this._fieldNode.readOnly = true;

    // Set events.
    this._dojo.connect(this._fieldNode, "ondblclick", this, "_onEditCallback");
    this._dojo.connect(this._fieldNode, "onblur", this, "_onEditCallback");
    this._dojo.connect(this._fieldNode, "onkeyup", this, "_onEditCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.editableField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Explicitly provided readOnly property must be ignored.
    props.readOnly = null;

    // Set properties.
    if (props.autoSave != null) { this.autoSave = props.autoSave; }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.editableField.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.editableField.event.submit.beginTopic,
    woodstock4_3.widget._xhr.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.hiddenField");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.hiddenField");

woodstock4_3._dojo.require("woodstock4_3.widget._base.selectBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a hiddenField widget.
 *
 * @constructor
 * @name woodstock4_3.widget.hiddenField
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the hiddenField widget.
 * <p>
 * TDB...
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a hidden field",
 *       widgetType: "hiddenField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * hidden field is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a hidden field",
 *       widgetType: "hiddenField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Hidden Field State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get hidden field
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the hidden field is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a hidden field",
 *       widgetType: "hiddenField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Hidden Field" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get hidden field
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a hidden field using the 
 * refresh function. The execute property of the refresh function is used to
 * define the client id which is to be submitted and updated server-side. When
 * the user clicks on the checkbox, the input value is updated server-side and 
 * the hidden field is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "field1",
 *       value: "This is a hidden field",
 *       widgetType: "hiddenField"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Change Hidden Field Value" },
 *       onClick="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("field1"); // Get hidden field
 *       return widget.refresh("cb1"); // Asynchronously refresh while submitting checkbox value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "field1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.hiddenField.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} name
 * @config {String} value Value of input.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.hiddenField", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
    },
    _widgetType: "hiddenField" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.hiddenField.event =
        woodstock4_3.widget.hiddenField.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hiddenField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hiddenField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hiddenField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hiddenField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hiddenField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hiddenField_event_submit_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.hiddenField.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.name != null) { props.name = this.name; }
    if (this.value != null) { props.value = this.value; }

    return props;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.hiddenField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.name != null) { this._domNode.name = props.name; }
    if (props.value != null) {
        // An empty string is valid.
        this._domNode.value = props.value;
    }
    if (props.disabled != null) { 
        this._domNode.disabled = new Boolean(props.disabled).valueOf();
    }

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.hiddenField.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.hiddenField.event.submit.beginTopic,
    woodstock4_3.widget._xhr.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.hyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.hyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget._base.anchorBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");

/**
 * This function is used to construct a hyperlink widget.
 *
 * @constructor
 * @name woodstock4_3.widget.hyperlink
 * @extends woodstock4_3.widget._base.anchorBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the hyperlink widget.
 * <p>
 * The hyperlink widget creates an HTML anchor that submits form data. If the 
 * disabled attribute of the hyperlink is set to true, clicking on the hyperlink
 * will not generate a request and hence the form will not be submitted or the 
 * page will not navigate to the specified url.
 * </p><p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "hyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * hyperlink is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "hyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Hyperlink State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get hyperlink
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the hyperlink is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "hyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Hyperlink" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get hyperlink
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an hyperlink using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the hyperlink text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "hyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Hyperlink Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get hyperlink
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "hyp1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.hyperlink.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} formId The id of the HTML form element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {Array} params The parameters to be passed during request.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.hyperlink", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.anchorBase ], {
    // Set defaults.
    _widgetType: "hyperlink" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.hyperlink.event =
        woodstock4_3.widget.hyperlink.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hyperlink_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hyperlink_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_hyperlink_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_hyperlink_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.hyperlink.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);

    // Set default style.
    var newClassName = (this.disabled == true)
        ? this._theme.getClassName("HYPERLINK_DISABLED","")
        : this._theme.getClassName("HYPERLINK","");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.hyperlink.prototype._onClickCallback = function(event) {
    if (this.disabled == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the submit.
    var result = (this._domNode._onclick)
        ? this._domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    if (this.href) {
        return false;
    }
    event.preventDefault();
    
    // If a form id isnt provided, use the utility function to
    // obtain the form id.
    if (this.formId == null) {
        var form = this._widget._getForm(this._domNode);
        this.formId = (form) ? form.id : null;
    }
    return this._submitFormData(this.formId, this.params);
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.hyperlink.prototype._postCreate = function () {
    // If the href attribute does not exist, set "#" as the default value of the
    // DOM node.
    this._domNode.href = "#";

    // Create callback function for onClick event.
    this._dojo.connect(this._domNode, "onclick", this, "_onClickCallback");

    return this._inherited("_postCreate", arguments);
};

/**
 * This function submits the hyperlink if the hyperlink is enabled.
 *
 * @param {String} formId The id of the HTML form element.
 * @param {Array} params The parameters to be passed during request.
 * @return {boolean} false to cancel the JavaScript event.
 * @private
 */
woodstock4_3.widget.hyperlink.prototype._submitFormData = function (formId, params) {
    if (formId == null) {
        return false;
    }
    var form = document.getElementById(formId);
    if (form == null) {
        return false;
    }
    var oldTarget = form.target;
    var oldAction = form.action;

    // Obtain HTML element for tab and common task components.
    var link = document.getElementById(this.id);
    if (link == null) {
        link = this._domNode;
    }

    // Set new action URL.
    var prefix;
    if (form.action) {
        prefix = (form.action.indexOf("?") == -1) ? "?" : "&";
        form.action += prefix + link.id + "_submittedLink=" + link.id;
    }
        
    // Set new target.
    if (link.target && link.target.length > 0) {
        form.target = link.target;
    } else {
        form.target = "_self";
    }

    // Append query params to new action URL.
    if (params != null) {
        var x;
        for (var i = 0; i < params.length; i++) {
            x = params[i];
            form.action += "&" + params[i] + "=" + params[i + 1]; 
            i++;
        }
    }

    // Submit form.
    form.submit(); 

    // Restore target and action URL.
    if (link.target != null) {
        form.target = oldTarget;
        form.action = oldAction;
    }
    return false;        
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.hyperlink.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.image");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.image");

woodstock4_3._dojo.require("woodstock4_3._base.browser");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a image widget.
 *
 * @name woodstock4_3.widget.image
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the image widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} border
 * @config {String} className CSS selector.
 * @config {String} prefix The application context path of image.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} height 
 * @config {String} hspace 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} longDesc 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} src 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @config {String} vspace
 * @config {String} width
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.image", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.border = 0;
    },
    _highContrastMode: null, // Must be available to all image widgets.
    _widgetType: "image" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.image.event =
        woodstock4_3.widget.image.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_image_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_image_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_image_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_image_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.image.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.alt != null) { props.alt = this.alt; }
    if (this.icon != null) { props.icon = this.icon; }
    if (this.align != null) { props.align = this.align; }
    if (this.border != null) { props.border = this.border; }
    if (this.height != null) { props.height = this.height; }
    if (this.hspace != null) { props.hspace = this.hspace; }
    if (this.longDesc != null) { props.longDesc = this.longDesc; }
    if (this.src != null) { props.src = this.src; }
    if (this.vspace != null) { props.vspace = this.vspace; }
    if (this.width != null) { props.width = this.width; }

    return props;
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.image.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Style must be set first (e.g., for absolute positioning) -- some style
    // properties may be overridden later by the combined image.
    this._setCommonProps(this._domNode, props);

    // Clear style properties if icon was previously set.
    if (props.src && this.icon) {
        this._domNode.style.border = "";
        this._domNode.style.backgroundImage = "";
        this._domNode.style.backgroundPosition = "";
        this._domNode.style.height = "";
        this._domNode.style.width = "";
    }

    // Set properties.
    if (props.icon) {
        // IE6 has issues with "png" images. IE6 png issue can be fixed but that
        // needs an outermost <span> tag. 
        //
        // <span style="overflow: hidden; width:13px;height:13px; padding: 0px;zoom: 1";>
        // <img src="dot.gif"
        //  style="filter:progid:DXImageTransform.Microsoft.AlphaImageLoader(src='testImage.png',sizingMethod='crop');
        //  margin-left:-0px;
        //  margin-top:-26px; border: none; height:39px;width:13px;"/>
        // </span>
        //
        // For now, skipping the combined image approach for IE6. Also need fix 
        // for Safari.
        var iconProps = this._theme.getImage(props.icon);
        if (iconProps == null) {
            console.debug("Error: theme icon '" + props.icon + "' not found.");
        } else {
            var mapKey = iconProps['map_key'];
            if (mapKey != null && !woodstock4_3._base.browser._isIe6()
                    && !this._widget._isHighContrastMode()) {
                // Note: Comparing height/width against "actual" properties is not a
                // valid test -- DOT images don't have a default size, for example.
                if (iconProps['top'] != null && iconProps['actual_height'] != null 
                        && iconProps['actual_width'] != null) {               
                    var transImage = this._theme.getImage("DOT");
                    var combinedImage = this._theme.getImage(mapKey);

                    // Set style properties.
                    this._domNode.style.border = "0";
                    this._domNode.style.backgroundImage = "url(" + combinedImage['src'] + ")";
                    this._domNode.style.backgroundPosition = "0px " + iconProps['top'] + "px";
                    this._domNode.style.height = iconProps['actual_height'] + "px";
                    this._domNode.style.width = iconProps['actual_width'] + "px";

                    // Set transparent image.
                    iconProps.src = transImage['src'];
                }
            }
            
            // Assign icon properties, even if combined image is not used.
            if (iconProps.alt != null) { this._domNode.alt = iconProps.alt; }
            if (iconProps.height != null) { this._domNode.height = iconProps.height; }
            if (iconProps.width != null) { this._domNode.width = iconProps.width; }            
            if (iconProps.src) {
                
                // Apply IE6 specific fix for png images if needed and obtain 
                // the modified style.
                props.style = this._setSrcProperty(iconProps);                                                
            }
        }
    } else {
        // Icon properties take precedence.
        if (props.alt != null) { this._domNode.alt = props.alt; }
        if (props.height != null) { this._domNode.height = props.height; }
        if (props.width != null) { this._domNode.width = props.width; }
        if (props.src) {
            
            // If context path is provided, then check whether the image has
            // context path already appended and if not, append it.            
            if (this.prefix) {
                props.src = 
                    woodstock4_3.widget.common._appendPrefix(this.prefix, props.src);                
            }                
            
            // Apply IE6 specific fix for png images if needed and obtain the 
            // modified style.
            props.style = this._setSrcProperty(props);
        }
    }
    if (props.align != null) { this._domNode.align = props.align; }
    if (props.border != null) { this._domNode.border = props.border; }
    if (props.hspace != null) { this._domNode.hspace = props.hspace; }
    if (props.longDesc != null) { this._domNode.longDesc = props.longDesc; }    
    if (props.vspace != null) { this._domNode.vspace = props.vspace; }

    // Set more properties.
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * Apply the src property on the dom element. If the image is an png image and 
 * the browser is IE 6 an IE6 specific fix needs to be applied and the style 
 * property has to be modified. If the png and IE6 combination is not true, then
 * no modification is done for the src property of the image and the style 
 * property.
 * 
 * @param {Object} props Key-Value pairs of properties.
 * @return (String} styleProp Modified value of the style property if the image
 * is a png image rendered on IE6.
 * @private
 */
woodstock4_3.widget.image.prototype._setSrcProperty = function(props) {    
    var styleProp = null;

    //IE6 PNG fix to handle transparent PNGs.
    if (props.src.indexOf(".png") > 0 && woodstock4_3._base.browser._isIe6()) {
        var width = (props.width > 0)
            ? props.width : this._theme.getMessage("Image.defaultWidth");
        var height =(props.height > 0)
            ? props.height : this._theme.getMessage("Image.defaultHeight"); 
        var styleMsg = this._theme.getMessage("Image.IEPngCSSStyleQuirk", 
          [width, height, props.src]);

        this._domNode.src = this._theme.getImage("DOT").src;

        if (props.style != null) {
            styleProp = props.style +";"+ styleMsg;
        } else {
            styleProp = (this.style != null) 
                ? this.style + ";" + styleMsg : styleMsg;                                          
        }
    } else {
        // While toggling between png and non-png type images need to adjust the 
        // style set on the dom element, since that is the property that 
        // contains the source for png images.
        if (woodstock4_3._base.browser._isIe6() 
                && this._domNode.style.cssText.indexOf(".png")>0) {
            styleProp = this.style;
        }
        this._domNode.src = props.src;  
    }
    return styleProp;
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.image.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.imageHyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.imageHyperlink");

woodstock4_3._dojo.require("woodstock4_3.widget.hyperlink");

/**
 * This function is used to construct a imageHyperlink widget.
 *
 * @constructor
 * @name woodstock4_3.widget.imageHyperlink
 * @extends woodstock4_3.widget.hyperlink
 * @class This class contains functions for the imageHyperlink widget.
 * <p>
 * The imageHyperlink widget creates an image surrounded by an HTML anchor that
 * submits form data. If the disabled attribute of the hyperlink is set to true, 
 * clicking the on the image will not generate a request and hence the form will
 * not be submitted or the page will not navigate to the specified url.
 * </p><p>
 * <h3>Example 1a: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       disabledImage: {
 *         id: "img1",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/disabled.gif"
 *         widgetType: "image"
 *       },
 *       enabledImage: {
 *         id: "img2",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/enabled.gif"
 *         widgetType: "image"
 *       },
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "imageHyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 2: Update widget using the getProps and setProps functions</h3>
 * </p><p>
 * This example shows how to toggle the state of a widget using the
 * getProps and setProps functions. When the user clicks the checkbox, the
 * image hyperlink is either disabled or enabled.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       disabledImage: {
 *         id: "img1",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/disabled.gif"
 *         widgetType: "image"
 *       },
 *       enabledImage: {
 *         id: "img2",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/enabled.gif"
 *         widgetType: "image"
 *       },
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "imageHyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Toggle Image Hyperlink State" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get image hyperlink
 *       return widget.setProps({disabled: !domNode.getProps().disabled}); // Toggle state
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * <h3>Example 3a: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. When the user clicks on the checkbox, the image hyperlink is 
 * asynchronously updated with new data.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       disabledImage: {
 *         id: "img1",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/disabled.gif"
 *         widgetType: "image"
 *       },
 *       enabledImage: {
 *         id: "img2",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/enabled.gif"
 *         widgetType: "image"
 *       },
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "imageHyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "cb1",
 *       label: { value: "Refresh Image Hyperlink" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "checkbox"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get image hyperlink
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can take an optional list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,..."). When no parameter is given, 
 * the refresh function acts as a reset. That is, the widget will be redrawn 
 * using values set server-side, but not updated.
 * </p><p>
 * <h3>Example 3b: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update an image hyperlink using the refresh
 * function. The execute property of the refresh function is used to define the
 * client id which is to be submitted and updated server-side. As the user types
 * in the text field, the input value is updated server-side and the image hyperlink text
 * is updated client-side -- all without a page refresh.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "hyp1",
 *       contents: ["Click me"],
 *       disabledImage: {
 *         id: "img1",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/disabled.gif"
 *         widgetType: "image"
 *       },
 *       enabledImage: {
 *         id: "img2",
 *         height: 13,
 *         width: 13,
 *         src: "/myApp/images/enabled.gif"
 *         widgetType: "image"
 *       },
 *       href: "http://sun.com",
 *       target: "_blank",
 *       widgetType: "imageHyperlink"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * &lt;span id="sp2">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp2", {
 *       id: "field1",
 *       label: { value: "Change Image Hyperlink Text" },
 *       onKeyPress="setTimeout('updateWidget();', 0);",
 *       widgetType: "textField"
 *     });
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("hyp1"); // Get image hyperlink
 *       return widget.refresh("field1"); // Asynchronously refresh while submitting field value
 *     }
 *   &lt;/script>
 * &lt;/span>
 * </code></pre><p>
 * Note that the refresh function can optionally take a list of elements to 
 * execute. Thus, a comma-separated list of ids can be provided to update 
 * server-side components: refresh("id1,id2,...").
 * </p><p>
 * <h3>Example 4: Subscribing to event topics</h3>
 * </p><p>
 * When a widget is manipulated, some features may publish event topics for
 * custom AJAX implementations to listen for. For example, you may listen for
 * the refresh event topic using:
 * </p><pre><code>
 * &lt;script type="text/javascript">
 *    var foo = {
 *        // Process refresh event.
 *        //
 *        // @param {Object} props Key-Value pairs of properties.
 *        processRefreshEvent: function(props) {
 *            // Get the widget id.
 *            if (props.id == "hyp1") { // Do something... }
 *        }
 *    }
 *    // Subscribe to refresh event.
 *    woodstock.widget.common.subscribe(woodstock.widget.imageHyperlink.event.refresh.endTopic, 
 *      foo, "processRefreshEvent");
 * &lt;/script>
 * </code></pre>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {Object} disabledImage
 * @config {Object} enabledImage
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} imagePosition
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.imageHyperlink",
        woodstock4_3.widget.hyperlink, {
    // Set defaults.
    _widgetType: "imageHyperlink" // Required for theme properties.
});

/**
 * Helper function to add children.
 *
 * @param props Key-Value pairs of properties.
 * @config {Array} contents The contents of the anchor body.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.imageHyperlink.prototype._addContents = function(props) {
    if (props.contents == null) {
        return false;
    }

    // Remove child nodes.
    this._widget._removeChildNodes(this._leftContentsContainer);
    this._widget._removeChildNodes(this._rightContentsContainer);

    // Add contents.
    for (i = 0; i <props.contents.length; i++) {
        this._widget._addFragment(this._leftContentsContainer, props.contents[i], "last");
        this._widget._addFragment(this._rightContentsContainer, props.contents[i], "last");
    }
    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.imageHyperlink.event =
        woodstock4_3.widget.imageHyperlink.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_imageHyperlink_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_imageHyperlink_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_imageHyperlink_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_imageHyperlink_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.imageHyperlink.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.enabledImage != null) { props.enabledImage = this.enabledImage; }
    if (this.disabledImage != null) { props.disabledImage = this.disabledImage; }
    if (this.imagePosition != null) { props.imagePosition = this.imagePosition; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.imageHyperlink.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._enabledImageContainer.id = this.id + "_enabled";
        this._disabledImageContainer.id = this.id + "_disabled";
        this._leftContentsContainer.id = this.id + "_leftContents";
        this._rightContentsContainer.id = this.id + "_rightContents";
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.imageHyperlink.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Show en/disabled images.
    if (props.disabled != null) {
        var disabled = new Boolean(props.disabled).valueOf();

        // We need to hide/show images only when the disabed image is specified.
        if (this.disabledImage) { 
            this._common._setVisibleElement(this._enabledImageContainer, !disabled);
            this._common._setVisibleElement(this._disabledImageContainer, disabled);
        }
    }

    // Add enabled image.
    if (props.enabledImage) {
        // Update/add fragment.
        this._widget._updateFragment(this._enabledImageContainer, this.enabledImage.id, 
            props.enabledImage);
    }

    // Add disabled image.
    if (props.disabledImage) {
        // Update/add fragment.
        this._widget._updateFragment(this._disabledImageContainer, this.disabledImage.id, 
            props.disabledImage);
    }

    // Set image position.
    if (props.imagePosition) {
        var left = (props.imagePosition == "left");
        this._common._setVisibleElement(this._leftContentsContainer, !left);
        this._common._setVisibleElement(this._rightContentsContainer, left);    
    }

    // Add contents.
    this._addContents(props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.imageHyperlink.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.label");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.label");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a label widget.
 *
 * @name woodstock4_3.widget.label
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the label widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey
 * @config {String} className CSS selector.
 * @config {String} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {Object} errorImage 
 * @config {String} htmlFor 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} level 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} primary Set button as primary if true.
 * @config {boolean} required
 * @config {Object} requiredImage
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.label", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.level = this._theme.getMessage("label.level", null, 2);
        this.required = false;
        this.valid = true;
    },
    _widgetType: "label" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.label.event =
        woodstock4_3.widget.label.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_label_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_label_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_label_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_label_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.label.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var key = "LABEL_LEVEL_TWO_TEXT";

    if (this.valid == false) {
        key = "CONTENT_ERROR_LABEL_TEXT";
    } else if (this.level == 1) {
        key = "LABEL_LEVEL_ONE_TEXT";
    } else if (this.level == 3) {
        key = "LABEL_LEVEL_THREE_TEXT";
    }

    // Get theme property.
    var newClassName = this._theme.getClassName(key, "");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to process notification events with Object
 * literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} detail Message detail text.
 * @config {boolean} valid Flag indicating validation state.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.label.prototype._notify = function(props) {
    if (props == null) {
        return false;
    }
    return this.setProps({
        valid: props.valid,
        errorImage: {
            title: props.detail
        }
    });
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.label.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.contents != null) { props.contents = this.contents; }
    if (this.errorImage != null) { props.errorImage = this.errorImage; }
    if (this.htmlFor != null) { props.htmlFor = this.htmlFor; }
    if (this.level != null) { props.level = this.level; }
    if (this.required != null) { props.required = this.required; }
    if (this.requiredImage) { props.requiredImage = this.requiredImage; }
    if (this.valid != null) { props.valid = this.valid; }
    if (this.value != null) { props.value = this.value; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.label.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._contentsContainer.id = this.id + "_contentsContainer";
        this._errorImageContainer.id = this.id + "_errorImageContainer";
        this._requiredImageContainer.id = this.id + "_requiredImageContainer";
        this._valueContainer.id = this.id + "_valueContainer";
    }

    // If errorImage or requiredImage are null, create images from the theme.
    // When the _setProps() function is called, image widgets will be
    // instantiated via the props param. 
    if (this.errorImage == null) {
	this.errorImage = {
            icon: "LABEL_INVALID_ICON",
            id: this.id + "_error",
	    className: this._theme.getClassName("LABEL_INVALID_IMAGE", null),
            widgetType: "image"
        };
    }
    if (this.requiredImage == null) {
        this.requiredImage = {
            icon: "LABEL_REQUIRED_ICON",
            id: this.id + "_required",
	    className: this._theme.getClassName("LABEL_REQUIRED_IMAGE", null),
            widgetType: "image"
        };
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.label.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this._inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.label.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.htmlFor != null) {
	this._domNode.htmlFor = props.htmlFor;
    }
    if (props.valid != null) {
	this.valid = new Boolean(props.valid).valueOf();
    }
    if (props.required != null) {
	this.required = new Boolean(props.required).valueOf();
    }
    if (props.value != null) {
        // An empty string is valid.
	this._widget._addFragment(this._valueContainer, props.value);
    }

    // Set error image properties.
    if (props.errorImage || props.valid != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.errorImage == null) {
            props.errorImage = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.errorImage.visible = !this.valid;

        // Update/add fragment.
        this._widget._updateFragment(this._errorImageContainer, this.errorImage.id, 
            props.errorImage);
    }

    // Set required image properties.
    if (props.requiredImage || props.required != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.requiredImage == null) {
            props.requiredImage = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.requiredImage.visible = this.required;

        // Update/add fragment.
        this._widget._updateFragment(this._requiredImageContainer, 
            this.requiredImage.id, props.requiredImage);
    }

    // Set contents.
    if (props.contents) {
        // Remove child nodes.
        this._widget._removeChildNodes(this._contentsContainer);

	for (var i = 0; i < props.contents.length; i++) {
            this._widget._addFragment(this._contentsContainer, props.contents[i], "last");
        }
    }

    // Set more properties.
    this._setCommonProps(this._domNode, props);
    this._setEventProps(this._domNode, props);

    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.label.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.listbox");

woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.listbox");

woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.selectBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");
woodstock4_3._dojo.require("woodstock4_3.widget.textField");

/**
 * This function is used to construct a listbox widget.
 *
 * @name woodstock4_3.widget.listbox
 * @extends woodstock4_3.widget._base.selectBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the listbox widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} label Defines the widget and its properties to use for a
 * label.
 * @config {String} labelOnTop If true the label appears above the listbox.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} multiple If true allow multiple selections
 * @config {boolean} monospace If true use monospace font styling.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Option selection is changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Option text is highlighted.
 * @config {Array} options See selectBase._setOptions or selectBase overview
 * for details on the elements of this array.
 * @config {int} size The number of option rows to display.
 * @config {String} style Specify style rules inline. 
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.listbox", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.selectBase ], {
    // Set defaults.
    constructor : function() {
	this.labelOnTop = this._theme._getMessageBoolean("listbox.labelOnTop", false);
	this.monospace = this._theme._getMessageBoolean("listbox.monospace", false);
	this.multiple = this._theme._getMessageBoolean("listbox.multiple", false);
	this.size = this._theme.getMessage("listbox.size", null, 12);
	this.width = this._theme.getMessage("listbox.width", null);
    },
    _widgetType: "listbox" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.listbox.event =
        woodstock4_3.widget.listbox.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for.*/
        beginTopic: "woodstock4_3_widget_listbox_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for.*/
        endTopic: "woodstock4_3_widget_listbox_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_listbox_event_state_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_listbox_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_listbox_event_submit_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_listbox_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element CSS class name.
 * <p>
 * Note: Selectors should be concatenated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var newClassName = this._theme.getClassName("LISTBOX", "");
    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * Return an Object Literal of label properties desired by this widget.
 * <p>
 * This implementation sets
 * the <code>listbox.labelLevel</code> and <code>listbox.labelAlign</code>
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively. It sets the <code>htmlFor</code> attribute to 
 * <code>_listContainer.id</code>
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} Key-Value pairs of label properties.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getLabelProps = function() {

    var props = this._inherited("_getLabelProps", arguments);
    props.level = this._theme.getMessage("listbox.labelLevel", null, 2);
    return props;
};

/**
 * Return a CSS selector for the listbox label.
 * The chosen CSS selector is a function of 
 * the <code>labelOnTop</code> property. If <code>labelOnTop</code>
 * property is <code>false</code> the value of the theme key 
 * <code>LISTBOX_LABEL_ALIGN</code> from the 
 * <code>styles</code> category is returned, else <code>null</code>.
 * @param {boolean} ontop The label on top state. In some contexts this state
 * must be passed to the method, for example when called from the 
 * <code>selectBase.setProps</code> method.
 * @return {String} A CSS selector for the listbox label.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getLabelOnTopClassName = function(ontop) {
    //var labelontop = ontop != null ? ontop : this.labelOnTop;
    //return labelontop == true
    return ontop != null && ontop == true
	? null
	: this._theme.getClassName("LISTBOX_LABEL_ALIGN", null);
};

/**
 * Return a CSS selector for _listContainer HTML element.
 * The chosen CSS selector is a function of 
 * the <code>disabled</code> and <code>monospace</code>properties.
 * The returned selector is the value of a property defined in the
 * <code>styles</code> category, as shown in the following table.
 * <table border="1px">
 * <tr><th>key</th><th>disabled</th><th>monospace</th></tr>
 * <tr>
 * <td>LIST</td>
 * <td>false</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>LIST_DISABLED</td>
 * <td>true</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>LIST_MONOSPACE</td>
 * <td>false</td>
 * <td>true</td>
 * </tr>
 * <tr>
 * <td>LIST_MONOSPACE_DISABLED</td>
 * <td>true</td>
 * <td>true</td>
 * </tr>
 * </table>
 * @return {String} A CSS selector for the _listContainer HTML element.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getListContainerClassName =
	function(disabled, monospace) {
    var key = "LIST";
    if (disabled == true) {
	if (monospace == true) {
	    key = "LIST_MONOSPACE_DISABLED";
	} else {
	    key = "LIST_DISABLED";
	}
    } else if (monospace == true) {
	key = "LIST_MONOSPACE";
    }
    return this._theme.getClassName(key, null);
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.listbox.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Get properties.
    //
    if (this.size != null) {
	props.size = this.size;
    }
    if (this.multiple != null) {
	props.multiple = this.multiple;
    }
    if (this.monospace != null) {
	props.monospace = this.monospace;
    }
    return props;
};

/**
 * This function returns the CSS class name for an HTML option element,
 * based on the state of <code>element</code>. The CSS class name for 
 * the following theme keys is returned for the given states, based on 
 * the properties
 * in "option".
 * <p>
 * <table border="1px">
 * <tr><th>state</th><th>key</th></tr>
 * <tr><td>
 * separator == true</td><td>LIST_OPTION_SEPARATOR</td>
 * </tr><tr><td>
 * group == true</td><td>LIST_OPTION_GROUP</td>
 * </tr><tr><td>
 * disabled == true</td><td>LIST_OPTION_DISABLED</td>
 * </tr><tr><td>
 * selected == true and disabled == false</td><td>LIST_OPTION_SELECTED</td>
 * </tr><tr><td>
 * separator == false and group false<br/> 
 * 	and disabled == false<br/>
 *	and selected == false</td><td>LIST_OPTION</td>
 * </tr>
 * </table>
 * </p>
 * <p>
 * If <code>element</code> is null, the CSS class name for the theme
 * key "LIST_OPTION" is returned.
 * </p>
 * @param {Object} element An HTML option DOM node instance.
 * @config {boolean} separator If true, element is a option separator.
 * @config {boolean} group If true the element is a optGroup instance.
 * @config {boolean} disabled If true the element is disabled.
 * @config {boolean} selected If true the element is selected.
 * @return {String} The HTML option element CSS class name.
 * @private
 */
woodstock4_3.widget.listbox.prototype._getOptionClassName = function(element) {
    var key = "LIST_OPTION";
    if (element == null) {
	return this._theme.getClassName(key, null);
    }

    if (element.separator == true) {
        key = "LIST_OPTION_SEPARATOR";
    } else
    if (element.group == true) {
        key = "LIST_OPTION_GROUP";
    } else
    if (element.selected == true) {
        key = "LIST_OPTION_SELECTED";
    }

    // disabled wins
    //
    if (element.disabled == true) {
	key = "LIST_OPTION_DISABLED";
    }
    return this._theme.getClassName(key, null);
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.listbox.prototype._postCreate = function () {
    // Don't trash the template.
    this._common._addStyleClass(this._listContainer,
	this._getListContainerClassName(
          this.disabled == true, this.monospace == true));

    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals. Please
 * see the constructor detail for a list of supported properties.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.listbox.prototype.setProps = function(props, notify) {
    if (props == null) {
	return this._inherited("setProps", arguments);
    }

    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    var ismonospace = this.monospace == true;
    var toggleMonospace = props.monospace != null && 
	props.monospace != ismonospace;

    var isdisabled = this.disabled == true;
    var toggleDisabled = props.disabled != null && 
	props.disabled != isdisabled;

    // Get current state of the classsname, and strip it 
    // and then add the classname for the new state.
    //
    if (toggleMonospace || toggleDisabled) {
	var cn = this._getListContainerClassName(isdisabled, ismonospace);
	this._common._stripStyleClass(this._listContainer, cn);

	cn = this._getListContainerClassName(
	    toggleDisabled ? props.disabled == true : isdisabled,
	    toggleMonospace ? props.monospace == true : ismonospace);
	this._common._addStyleClass(this._listContainer, cn);
    }
    return this._inherited("setProps", arguments);
};

/**
 * This function sets properties specific to the listbox, the superclass
 * will set properties specific to the <code>select</code> HTML element
 * and handle <code>labelOnTop</code> and the label subcomponent.
 * <p>
 * If the <code>size</code>, <code>multiple</code>, or <code>monospace</code>
 * properties are not set in <code>props</code> or <code>this</code>,
 * the values are obtained from the theme using the keys 
 * <code>listbox.size</code>, <code>listbox.multiple</code>,
 * <code>listbox.monospace</code>, and <code>listbox.labelOnTop</code>
 * respectively, from the theme "messages" category. If the theme does 
 * not define these values then the values <code>12</code>,
 * <code>false</code>, <code>false</code>, and <code>false</code>
 * are used, respectively.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} multiple If true allow multiple selections
 * @config {boolean} monospace  If true the <code>LIST_MONOSPACE</code> and
 * <code>LIST_MONOSPACE_DISABLED</code>CSS class names are used, else
 * <code>LIST</code> and <code>LIST_DISABLED</code>.
 * @config {int} size The number of rows to display.
 * @config {boolean} labelOnTop If true render the label on top of the
 * list box, else to the upper left.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.listbox.prototype._setProps = function(props) {
    // If props == null create one so that the superclass can 
    // handle "labelOnTop" or anything else that needs to be
    // set in props and handled in the superclass
    //
    if (props == null) {
	return this._inherited("_setProps", arguments);
	//props = {};
    }

    if (props.size != null) {
	var size = props.size;
	if (size < 1) {
	    size = this._theme.getMessage("listbox.size", null, 12);
	}
	if (this._listContainer.size != size) {
	    this._listContainer.size = size;
	}
    }

    if (props.multiple != null && 
	    this._listContainer.multiple != props.multiple) {
	this._listContainer.multiple = props.multiple;
    }
    return this._inherited("_setProps", arguments);
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.listbox.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.listbox.event.submit.beginTopic,
    woodstock4_3.widget._xhr.common, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.login");

woodstock4_3._dojo.require("woodstock4_3.json");
woodstock4_3._dojo.require("woodstock4_3.widget.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.login");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a login widget.
 *
 * @name woodstock4_3.widget.login
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the login widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} id ID of the login widget.
 * @config {String} loginState State of the authentication process. Set to "INIT" at the beginning.
 * @config {boolean} autoStart Flag indicating if authentication process should start on page load.
 * @config {String} style Specify style rules inline.
 * @config {String} className CSS selector of outermost DOM node for this widget.
 * @config {String} tabIndex Attribute required to support tabbed navigation.
 * @config {Object} userData JSON object containing user data that needs to be displayed as user prompt.
 * @config {Object} keys JSON object representing the key value mapping for user data fields.
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.login", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.loginState = "INIT";
    },
    _widgetType: "login" // Required for theme properties.    
});

/**
 * This function publishes the event that kicks off the authentication process.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.login.prototype.authenticate = function() {
    if (this.keys) {
        for (var i = 0; i < this.keys.length; i++) {
            var widget = this._widget.getWidget(this.keys[i][0]);
            var keyVal;
            if (widget) {
                var name = widget._widgetType;
                if (name == "textField" || name == "passwordField") {
                    keyVal = widget.getProps().value;
                } else if (name == "dropDown" || name == "listbox") {
                    keyVal = widget.getSelectedValue();
                }
            }
            this.keys[i][1] = keyVal;
        }
    }
                    
    // Publish an event for custom AJAX implementations to listen for.
    this._publish(woodstock4_3.widget.login.event.authenticate.beginTopic, [{
        id: this.id,
        loginState: this.loginState,
        keys: this.keys,
        endTopic: woodstock4_3.widget.login.event.authenticate.endTopic
    }]);
    return true;
};

/**
 * This function executes the steps that need to be followed when the 
 * "Login" button is clicked.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.login.prototype._buttonClicked = function() {
    return this.authenticate();
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.login.event =
        woodstock4_3.widget.login.prototype.event = {
    /**
     * This closure is used to process authentication events.
     * @ignore
     */
    authenticate: {
        /**
         * Authentication begin event topics for custom AJAX implementations to listen for.
         */
        beginTopic: "woodstock4_3_widget_login_event_authenticate_begin",

        /**
         * Authentication end event topics for custom AJAX implementations to listen for.
         */
        endTopic: "woodstock4_3_widget_login_event_authenticate_end"

    },

    /**
     * This closure is used to process refresh events.
     * @ignore
     */
    refresh: {
        /**
         * Refresh event topics for custom AJAX implementations to listen for.
         */
        beginTopic: "woodstock4_3_widget_login_event_refresh_begin",

        /**
         * Refresh event topics for custom AJAX implementations to listen for.
         */
        endTopic: "woodstock4_3_widget_login_event_refresh_end"
    },

    /**
     * This closure is used to report success/failure during the authentication 
     * process.
     * @ignore
     */
    result: {
        /**
         * Successful authentication event topic for applications can listen for.
         */
        successTopic: "woodstock4_3_widget_login_event_result_success",

        /**
         * Authentication failure event topic that applications can listen for.
         */
        failureTopic: "woodstock4_3_widget_login_event_result_failure"
    },

    /**
     * This closure is used to process state change events.
     * @ignore
     */
    state: {
        /**
         * Event topics for custom AJAX implementations to listen for.
         */
        beginTopic: "woodstock4_3_widget_login_event_state_begin",

        /**
         * Event topics for custom AJAX implementations to listen for.
         */
        endTopic: "woodstock4_3_widget_login_event_state_end"
    }
};

/**
 * This function performs error handling on the client side
 * when authentication failure occurs. Typically this involves
 * updating the alter widget with the appropriate messages
 * sent across from the server.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} id ID of the login widget.
 * @config {String} loginState State of the authentication process. Set to "INIT" at the beginning.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.login.prototype._handleFailure = function(props) {
    this._setAlert(props.alert);
    this._publish(woodstock4_3.widget.login.event.result.failureTopic, [{
        id: props.id
    }]);

    // initialize the state so that subsequent requests can start afresh
    this.loginState = "INIT";
    props.loginState = "INIT";
    return true;
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.login.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var key = "LOGIN_DIV";

    // Get theme property.
    var newClassName = this._theme.getClassName(key, "");

    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.login.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // var props = woodstock4_3.widget.login.superclass.getProps.call(this);
    if (this.loginState != null) { props.loginState = this.loginState; }    
    if (this.autoStart != null) { props.autoStart = this.autoStart; }
    if (this.tabIndex != null) { props.tabIndex = this.tabIndex; }
    if (this.dotImage != null) { props.dotImage = this.dotImage; }
    if (this.loginButton != null) { props.loginButton = this.loginButton; }

    return props;
};

/**
 * This function handles the case where authentication is complete
 * and successful. The end result could involve storing a cookie
 * value somewhere or redirecting to some other page.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} id ID of the login widget.
 * @config {String} loginState State of the authentication process. Set to 
 * "INIT" at the beginning.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.login.prototype._handleSuccess = function(props) {
    // Publish the success event topic
    // Remove the alert message if props does not
    // contain any alert information.
    // Clear out the _loginTable.
    this._setAlert(props.alert);
    this._publish(woodstock4_3.widget.login.event.result.successTopic, [{
        id: props.id
    }]);

    this._widget._removeChildNodes(this._loginTable);
    this.loginState = "INIT";
    props.loginState = "INIT";

    if (props.redirectURL != null) {
        var loc = document.location;
        var newURL = loc.protocol + "//" + loc.host;
        newURL = newURL + props.redirectURL;
        document.location.href = newURL;
    }
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.login.prototype._postCreate = function () {
    if (this.tabIndex == null) {
        this.tabIndex = -1;
    }

    // If login button and dot image are empty generate them on the
    // client side.
    if (this.dotImage == null) {
	this.dotImage = {
            icon: "DOT",
            id: this.id + "_dotImage",
            widgetType: "image"
        };
    }
    
    if (this.loginButton == null) {
        this.loginButton = {
            alt: this._theme.getMessage("login.buttonAlt"),
            id: this.id + "_loginButton",
            tabIndex: this.tabIndex,
            value: this._theme.getMessage("login.buttonTitle"),
            widgetType: "button"
        };
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.login.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Handle the success and failure and continue states here.
    // The init state is handled in the _postCreate function
    // as there needs to be a minimal delay the first time this
    // widget is  being rendered - otherwise only the initial <span>
    // element representing the login widget gets transmitted causing
    // problems on the server side. 

    if (props.loginState == "SUCCESS") {
        this._handleSuccess(props);

    } else if (props.loginState == "FAILURE") {
        this._handleFailure(props);

    } else if (props.loginState == "CONTINUE") {
        this._updateLoginTable(props);
    }
    // Set remaining properties.
    return this._inherited("_setProps", arguments);
};

/**
 * This function adds or updates the alert widget displayed as part of the
 * login widget.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} type The alert type.
 * @config {String} summary The alert summary.
 * @config {String} detail The alert detail.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.login.prototype._setAlert = function(props) {
    if (props == null) {
        this._widget._removeChildNodes(this._alertContainer);
        return false;
    } else {
        var _props = {
            id: this.id + "_alert",
            widgetType: "alert"
        };

        // Add extra properties.
        if (props != null) {
            this._proto._extend(_props, props);
        }
        if (_props.summary == null) {
            _props.summary = this._theme.getMessage("login.errorSummary");
        }
        if (_props.detail == null) {
            _props.detail = this._theme.getMessage("login.errorDetail");
        }

        var tr = this._alertRowContainer.cloneNode(false);
        var td = this._alertCellContainer.cloneNode(false);
        this._loginTbody.appendChild(tr);
        tr.appendChild(td);

        this._widget._updateFragment(td, _props.id, _props);
    }
    return true;
};

/**
 * This function is used to "start" the widget, after the widget has been
 * instantiated.
 *
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.login.prototype._startup = function () {
    if (this._started == true) {
        return false;
    }
    // If widget set to "autoStart" fire up the authentication process.
    var props = this.getProps();
    if (this.loginState == "INIT") {
        var id = this.id;
        if (this.autoStart) {
            setTimeout(function() {
                woodstock4_3.widget.common.getWidget(id).authenticate();
            }, 10);    
        }
    }
    return this._inherited("_startup", arguments);
};

/**
 * This function is used to update the login section of the DOM tree with
 * the new set of widgets. This new set of widgets represents the new set
 * of prompts for entering authentication data. For example, the first set
 * of prompts could have been a textField and a password field. The second
 * set could be dropdown list containing a set of roles to choose from 
 * followed by a role password field.
 * 
 * @param {Object} props Key-Value pairs of properties.
 * @config {Object} userData JSON object containing user data that needs to be displayed as user prompt.
 * @config {Object} keys JSON object representing the key value mapping for user data fields.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.login.prototype._updateLoginTable = function(props) {
    // Remove existing data entries before adding the new ones.
    // This involves destroying the widgets and also deleting
    // the table rows.
    this._widget._removeChildNodes(this._loginTbody);
        
    var rowNum = 1;

    // add the alert row
    var alertAdded = this._setAlert(props.alert);
        
    // set up table rows for each of the user prompts
    if (props.userData) {
        var rowCount = props.userData.length;
        var idCount = 0;

        for (var i=0; i < rowCount; i++) {
            var dataRow = props.userData[i];
            var tr = this._inputDataRowContainer.cloneNode(false);
            this._loginTbody.appendChild(tr);
            for (var j=0; j<dataRow.length; j++) {
                var td;
                var divNode;
                if (j==0) {
                    td = this._labelContainerCell.cloneNode(false);
                    tr.appendChild(td);
                    if (i+1 == rowCount) {
                        divNode = this._lastLabelContainer.cloneNode(true);
                    } else {
                        divNode = this._labelContainer.cloneNode(true);
                    }
                    td.appendChild(divNode);
                    if (dataRow[j].widgetType != null) {
                        this._widget._addFragment(divNode, dataRow[j], "last"); 
                    }
                } else {
                    td = this._dataContainerCell.cloneNode(false);
                    tr.appendChild(td);
                    if (i+1 == rowCount) {
                        divNode = this._lastInputContainer.cloneNode(true);
                    } else if (dataRow[j].type == "text") {
                        divNode = this._stxtContainer.cloneNode(true);
                    } else if (dataRow[j].type == "textField") {
                        divNode = this._textContainer.cloneNode(true);
                    } else {
                        divNode = this._inputContainer.cloneNode(true);
                    }
                    td.appendChild(divNode);
                    if (dataRow[j].widgetType != null) {
                        this._widget._addFragment(divNode, dataRow[j], "last"); 
                    }
                }
            }
            rowNum++;
        }
        // add table row for spacer image followed by the login button.
        var buttonTR = this._buttonRowContainer.cloneNode(false);
        this._loginTbody.appendChild(buttonTR);
        var td1 = this._dotImageContainer.cloneNode(false);
        buttonTR.appendChild(td1);
        if (this.dotImage) {
            this._widget._addFragment(td1, this.dotImage, "last");
        }

        var td2 = this._buttonContainer.cloneNode(false);
        buttonTR.appendChild(td2);
        var spanNode = this._loginButtonContainer.cloneNode(true);
        var _this = this;

        /** @ignore */
        this.loginButton.onClick = function() {
            _this._buttonClicked(props);
            return false;
        };
        td2.appendChild(spanNode);
        this._widget._addFragment(spanNode, this.loginButton, "last");
    }
    return true;
};
woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");

/**
 * @class This class contains functions to authenticate data asynchronously 
 * using JSF Extensions as the underlying transfer protocol.
 * @static
 * @private
 */
woodstock4_3.widget._xhr.login =  {
    /**
     * This function is used to pass on the authentication data sent from 
     * the server callback object to the client. It calls the appropriate 
     * function in the login widget that updates the DOM tree to reflect 
     * the result of the authentication process to the end user.
     *
     * @param {String} id The HTML widget Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to the Ajax transaction.
     * @param {Object} xjson The xjson header provided to the Ajax transaction.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _loginCallback : function(id, content, closure, xjson) {
        // Parse JSON text and update login widget.
        var props = woodstock4_3.json.parse(content);

        // Publish an event for custom AJAX implementations 
	// to listen for.
        var widget = woodstock4_3.widget.common.getWidget(id);
        widget.setProps(props);
        
        // Publish an event for custom AJAX implementations to listen for.
        if (closure.endTopic) {
            woodstock4_3._dojo.publish(closure.endTopic, props);
        }
        return true;
    },

    /**
     * This function is used to send data from the
     * client to the server process where the actual
     * JAAS authentication is taking place.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @config {String} loginState
     * @config {String} endTopic
     * @config {String} keys
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processLoginEvent : function(props) {
        if (props == null) {
            return false;
        }

        // Generate AJAX request.
        woodstock4_3.widget._xhr.common._doRequest({
            id: props.id,
            callback: woodstock4_3.widget._xhr.login._loginCallback,
            closure: {
                endTopic: props.endTopic
            },
            xjson: {
                id: props.id,
                loginState: props.loginState,
                execute: props.id
            }
        });
        return true;
    }
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.login.event.authenticate.beginTopic,
    woodstock4_3.widget._xhr.login, "_processLoginEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.modalWindow");

/** 
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */

woodstock4_3._dojo.provide("woodstock4_3.widget.modalWindow");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");

/**
 * This function is used to construct a modalWindow widget.
 *
 * @constructor
 * @name woodstock4_3.widget.modalWindow
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @class This class contains functions for the modalWindow widget.
 * <p>ModalWindow is a popup window that is displayed over an optional mask overlay.
 * By putting an overlay over the whole page, ModalWindow disables user's mouse interaction
 * with the rest of the page. It remains possible though that user may still able to interact 
 * with the underlying page using assigned or standard keyboard access keys. User can also
 * scroll the window to reveal different portions of the screen, while modalWindow continues to 
 * maintain its visible relative position on the page.
 * 
 * </p>
 * <p>
 * ModalWindow consists of the header, content and footer. <br>
 * Buttons: ModalWindow provides a set of default buttons: close, ok, cancel, apply. Default buttons 
 * can be specified by providing the following properties: buttonOk, buttonOk, 
 * buttonCancel, buttonApply. With the exception of the close button, the default is false ( do not show)
 * the default buttons. Close button located on the window title and is shown by default.
 * <p>
 * Also, any number of custom buttons can be created within the command container of the window, with
 * the use of custom closures.
 * </p>

 * </p>
 * <p>ModalWindow displays arbitrary content. Content can be provided as an arbitrary HTML fragment,
 * inclusion of remote content, another widget, or a combination (array) of all of the above.
 * ModalWindow can also delegate content creation to a javascript method.
 * ModalWindow will dynamically size up to accomodate all of the content up to a specified maxWidth
 * and maxHeight.
 * </p>
 * 
 * <p>When created, ModalWindow is invisible,and can be open with the use of public open method.
 * </p>
 * 
 * <p>Before ModalWindow closes ( exits), it allows to execute a pre-exit method. If such method
 * returns false, ModalWindow will NOT be closed. Any other than false return will allow window to close
 * and the return will be passed as an exit code in close event. 
 * <ul>
 * <li>Any button can have a pre-exit method associated with its onClick event ( no additional args ).
 * Such method will be called prior to the exit. 
 * <ul>
 * <li>Custom actions can provide an onClick handler as part of action closure. 
 * <li>Standard buttons are equiped with the default close handler, or can provide their own with 
 * property [buttonName]ButtonOnClick ( okButtonOnClick, applyButtonOnClick, closeButtonOnClick, cancelButtonOnClick).
 * If no onClick handler has been provided for the button,
 * a default handler will simply close the window and publish a correposnding [buttonName]OnClickEvent.
 * If custom handler is provided, window will be closed, UNLESS the return value is false.
           <code>
            "closeButtonOnClick" : "return 888;",

            "actions" : [ {
                    id: "btn1",
                    value: "Close",  
                    onClick : "doSomething(); return true; "
                }   
            ]
          </code>
 * </ul>
 * </p>
 * <p>Events:<br>
 * ModalWindow publishes the following events:
 * <ul>
 * <li>woodstock4_3.widget.modalWindow.event.refresh.beginTopic - when dynamic refresh is requested
 * <li>woodstock4_3.widget.modalWindow.event.refresh.endTopic - when dyncamic refresh is complete
 * <li>woodstock4_3.widget.modalWindow.event.open - when window is open
 * <li>woodstock4_3.widget.modalWindow.event.close - when window is closed
 * </ul>
 * </p>
 * <p>
 * <h3>Example 1: Create widget</h3>
 * </p><p>
 * This example shows how to create a widget using a span tag as a place holder 
 * in the document. Minimally, the createWidget() function needs an id and 
 * widgetType properties.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mw1", *       
 *       widgetType: "modalWindow"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * </code></pre>
 * 
 * <h3>Example 2: Asynchronously update widget using refresh function</h3>
 * </p><p>
 * This example shows how to asynchronously update a widget using the refresh 
 * function. The modalWindow is asynchronously updated with new data when 
 * updateWidget() method is executed.
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mw1", *       
 *       widgetType: "modalWindow"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * ...
 *   &lt;script type="text/javascript">
 *     function updateWidget() {
 *       var widget = woodstock4_3.widget.common.getWidget("mw1"); // Get modalWindow
 *       return widget.refresh(); // Asynchronously refresh
 *     }
 *   &lt;/script>
 * </code></pre>
 * 
 * 
 * 
 * <h3>Example 3: Reuse widget instance to open different windows</h3>
 * </p><p>
 * This example shows how to reuse a single instance of the modalWindow with 
 * different properties over and over for different windows to be rendered.
 * It intstantiates the widget with minimal set of properties first, and then
 * uses public method to open windows for another set of properties.
 * 
 * </p><pre><code>
 * &lt;span id="sp1">
 *   &lt;script type="text/javascript">
 *     woodstock4_3.widget.common.createWidget("sp1", {
 *       id: "mw1", *       
 *       widgetType: "modalWindow"
 *     });
 *   &lt;/script>
 * &lt;/span>
 * ...
 *   &lt;script type="text/javascript">
 *     function openWindow1() {
 *       var widget = woodstock4_3.widget.common.getWidget("mw1"); // Get modalWindow
 *       //create property with standard Ok and Cancel buttons
 *        var propsDialogDefaultButtons = {
 *                        "widgetType" :"modalWindow",
 *                        "contents" :  [{html:"This window has no close button. It can be closed programmatically only."}],
 *                        "closeButton" : false //render no close button
 *                        };
 *        widget.open(propsDialogDefaultButtons);    
 *     }
 *     
 *     // openWindow2 opens window with one 'Close' button in action container
 *     function openWindow2() {
 *       var widget = woodstock4_3.widget.common.getWidget("mw1"); // Get modalWindow
 *       //create property with standard Ok and Cancel buttons
          var propsTooBigImage= {
                "title": "System Not Available",
                "widgetType" :"modalWindow",
                "contents" :  [{html:"Too big of an image: <br><img src='./images/mona-big.jpg'/>"}],
                "actions": [{
                        "id": "btn1",
                        "value": "Close"
                    }]
            };
 *        widget.open(propsTooBigImage);    
 *     }
   
 *     &lt;/script>
 * </code></pre>
 * 
 * 
 * 
 * <p>
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} maskColor color of the mask
 * @config {String} maskOpacity mask opacity
 * @config {String} top offset from the top, including units
 * @config {String} left offset from the left, including units; or "auto"
 * @config {String} maxWidth maximium width of the window contents including units, such as "200px"
 * @config {String} maxHeight maximium height of the window contents including units, such as "200px"
 * @config {String} title window title, or null for none
 * @config {Array} contents array of contents, or null for none
 * @config {Boolean} okButton true  or false to render default window ok button ( false by default)
 * @config {String} okButtonOnClick handler for ok button onClick event
 * @config {Boolean} applyButton true  or false to render default window apply button ( false by default)
 * @config {String} applyButtonOnClick handler for apply button onClick event
 * @config {Boolean} cancelButton true  or false to render default window cancel button ( false by default)
 * @config {String} cancelButtonOnClick handler for cancel button onClick event
 * @config {Boolean} closeButton true  or false to render default window close button ( true by default)
 * @config {String} closeButtonOnClick handler for close button onClick event
 * @config {Array} actions list of buttons descriptions
 * @config {boolean} visible Hide or show element.
 *
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.modalWindow", [
    woodstock4_3.widget._base.widgetBase,
    woodstock4_3.widget._base.refreshBase
], {
    // Set defaults.
    constructor: function() {
        this.closeButton =  true; 
        this.okButton =  false; 
        this.applyButton =  false; 
        this.cancelButton =  false; 
        this.actions = null;
        this.contents = null;
        this.left = "auto";
        this.mask = true;
        this.maxWidth = "1000px";
        this.maxHeight = "1000px";
        this.top = "100px";
        this.title = ""; // null disables title bar, empty string display bar with no title     
    },                  
    _widgetType: "modalWindow" // Required for theme properties.
});



/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.modalWindow.event =
    woodstock4_3.widget.modalWindow.prototype.event = {

    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} execute Comma separated list of IDs to be processed server
         * side along with this widget.
         * </li><li>
         * {String} id The widget ID to process the event for.
         * </li></ul>
         *
         * @id woodstock4_3.widget.modalWindow.event.refresh.beginTopic
         * @property {String} beginTopic
         */
        beginTopic: "woodstock4_3_widget_modalWindow_event_refresh_begin",

        /** 
         * Refresh event topic for custom AJAX implementations to listen for.
         * Key-Value pairs of properties to publish include:
         * <ul><li>
         * {String} id The widget ID to process the event for.
         * </li><li>
         * Please see the constructor detail for additional properties.
         * </li></ul>
         *
         * @id woodstock4_3.widget.modalWindow.event.refresh.endTopic
         * @property {String} endTopic
         */
        endTopic: "woodstock4_3_widget_modalWindow_event_refresh_end"
    },


    /**
     * okButtonOnClick event is publised when default ok button is clicked.
     * 
     * Key-Value pairs of properties to publish include:
     * <ul><li>
     * {String} id The widget ID to process the event for.
     * </li></ul>
     */
    okButtonOnClick: "woodstock4_3_widget_modalwindow_ok_clicked",

    /**
     * cancelButtonOnClick event is publised when default cancel button is clicked.
     * 
     * Key-Value pairs of properties to publish include:
     * <ul><li>
     * {String} id The widget ID to process the event for.
     * </li></ul>
     */
    cancelButtonOnClick: "woodstock4_3_widget_modalwindow_cancel_clicked",

    /**
     * applyButtonOnClick event is publised when default apply button is clicked.
     * 
     * Key-Value pairs of properties to publish include:
     * <ul><li>
     * {String} id The widget ID to process the event for.
     * </li></ul>
     */
    applyButtonOnClick: "woodstock4_3_widget_modalwindow_apply_clicked",

    /**
     * closeButtonOnClick event is publised when default close button is clicked.
     * 
     * Key-Value pairs of properties to publish include:
     * <ul><li>
     * {String} id The widget ID to process the event for.
     * </li></ul>
     */
    closeButtonOnClick: "woodstock4_3_widget_modalwindow_close_clicked",



    /**
     * Open event is publised when window is open.
     * 
     * State event topic for custom AJAX implementations to listen for.
     * Key-Value pairs of properties to publish include:
     * <ul><li>
     * {String} id The widget ID to process the event for.
     * </li></ul>
     *
     */
    open: "woodstock4_3_widget_modalwindow_event_open",

    /**
     * Close event is publised after window is closed.
     * 
     * State event topic for custom AJAX implementations to listen for.
     * Key-Value pairs of properties to publish include:
     * <ul>
     * <li>{String} id The widget ID to process the event for.
     * </ul>
     *
     */
    close: "woodstock4_3_widget_modalwindow_event_close"


};


/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.modalWindow.prototype._postCreate = function () {


    // Set public functions.

    /** @ignore */
    this._domNode.close = function(preExitAction) { return woodstock4_3.widget.common.getWidget(this.id).close(preExitAction); };
    /** @ignore */
    this._domNode.open = function(props) { return woodstock4_3.widget.common.getWidget(this.id).open(props); };

    // Set events.

    // The escape key should also close modalWindow.
    //this.keyConnection =  this._dojo.connect(document, "onkeydown", this, this._createKeyDownCallback());   
    
    return this._inherited("_postCreate", arguments);
};



/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.modalWindow.prototype.getProps = function() {
    
    var props = this._inherited("getProps", arguments);
     
    // Set properties.  
    if ( this.closeButton != null) { props.closeButton = this.closeButton; }
    if ( this.okButton != null) { props.okButton = this.okButton; }
    if ( this.cancelButton != null) { props.cancelButton = this.cancelButton; }
    if ( this.applyButton != null) { props.applyButton = this.applyButton; }
    
    if ( this.closeButtonOnClick != null) { props.closeButtonOnClick = this.closeButtonOnClick; }
    if ( this.okButtonOnClick != null) { props.okButtonOnClick = this.okButtonOnClick; }
    if ( this.cancelButtonOnClick != null) { props.cancelButtonOnClick = this.cancelButtonOnClick; }
    if ( this.applyButtonOnClick != null) { props.applyButtonOnClick = this.applyButtonOnClick; }

    if ( this.actions != null) { props.actions = this.actions; }    
    if ( this.contents != null) { props.contents = this.contents; }
    if ( this.left != null) { props.left = this.left; }
    if ( this.mask != null) { props.mask = this.mask; }
    if ( this.maskColor != null) { props.maskColor = this.maskColor; }
    if ( this.maskOpacity != null) { props.maskOpacity = this.maskOpacity; }
    if ( this.maxWidth != null) { props.maxWidth = this.maxWidth; }
    if ( this.maxHeight != null) { props.maxHeight = this.maxHeight; }
    if ( this.top != null) { props.top = this.top; }
    if ( this.title != null) { props.title = this.title; }
     
    return props;
};




/**
 * This function is used to set widget properties. Please see the constructor 
 * detail for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.modalWindow.prototype._setProps = function(props) {

    if (props == null) {
        return false;
    }

    if (props.mask != null) {
        this._setContainerVisible(this._overlay, props.mask);   
    } 
    if (props.maskColor) { this._overlay.style.backgroundColor = props.maskColor;}
    
    if (typeof props.maskOpacity == "number") { this._overlay.style.opacity = props.maskOpacity;}

    if (props.top) { 
        this._window.style.marginTop  = props.top;
        this.marginTop = props.top;
    }
    if (props.left) { 
        this._window.style.marginLeft  = props.left;
        this.marginLeft = props.left;
    }
    
    if (props.maxHeight) {
        this._window.style.maxHeight  = props.maxHeight;
        this.maxHeight = props.maxHeight;
        
    }

    if (props.maxWidth) {
        this._window.style.maxWidth  = props.maxWidth;
        this.maxWidth = props.maxWidth;
        
    }
    
   
    
    //subclasses can overwrite population of containers
    //body is populated first, as other containers may depend on its contents
    this._setContentsProps(props);   //i.e. contents
    this._setHeaderProps(props); //i.e. title, navigation, toolbar 
    this._setFooterProps(props); //i.e. command container


    //buttons can be added or removed dynamically. Connect/disconnect events appropriately
    this._updateEventsRegistration();
    
    // Set remaining properties.
    return this._inherited("_setProps", arguments);
 
};


 
/**
 * Sets the content of the modal window from the properties.
 * Each element of contents array can be either null | Object array | javascript method .
 * If Object array is provided, each fragement will be appended incrementally to the
 * innerHTML of the _contentContainer.<br>
 * If javascript method is provided, it will be called in order to populate the _contentContainer. Such 
 * method must be of signature:
 * <code>
    methodName(node, props)
 * </code>
 * where node passed will be the one of _contentContainer, and props is the pass through properties.
 * 
 * This method will remove the current content if provided contents is null or non-empty.
 * 
 * @param {Object} props Key-Value pairs of properties  ( see constructor for a list)
 * @return true if body is to be shown, false otherwise
 * @see constructor
 */
woodstock4_3.widget.modalWindow.prototype._setContentsProps = function(props)	{
    if (!props || typeof props.contents == "undefined") {
        return false;
    }                 
    
    this.contents = props.contents;
    
    var contents = props.contents;
                
    // Remove child nodes, even if contents is null
    this._widget._removeChildNodes(this._contentContainer);

    if (contents == null) {
        return true;
    }

    if (contents) {
       
        for (var i = 0; i < contents.length; i++) {
            if (typeof contents[i] == "function") {
                contents[i](this._contentContainer);
            } else {
                this._widget._addFragment(this._contentContainer, contents[i], "last");
            }
        }
        return true;
    }
    return false;
    
};


/** 
 *  Sets the content of the header.
 *  This implementation sets window title and window controls(close button).
 *  <br>
 *  The following properties are analyzed in this method:
 *  <ul>
 *  <li>title
 *  <li>closeButton
 *  <li>closeButtonOnClick
 *  </ul>
 *  
 * @param {Object} props Key-Value pairs of properties  
 * @return true if header is created, false otherwise
 */
woodstock4_3.widget.modalWindow.prototype._setHeaderProps  = function(props)	{
          
    if (props.title == null) {
        //do not display titlebar
        this._setContainerVisible(this._windowTitleBar, false);
    } else {
        this._setContainerVisible(this._windowTitleBar, true);        
    }

    this._widget._addFragment(this._windowTitleText, props.title);
    
    
    if (props.closeButton != null) {
        if (props.closeButton) {
            //button has been requested
            this._setContainerVisible(this._closeButton, true);
            //default close behavior
            this.closeButtonAction = new Function("return -1;"); //default exit code for close button            
        } else  {
            //request to disable button
            this._setContainerVisible(this._closeButton, false);
            this.closeButtonAction = null;
        }
    }

    if (props.closeButtonOnClick != null) {  
        //get a function for  a string or reference
        this.closeButtonAction = this._getFunction(props.closeButtonOnClick) ;                
    } 
       
    
    return true;
};



/** 
 *  Sets the content of the footer.
 *  
 *  This implementation will delegate command container creation to the method
 *  _setActionsContainerProps, then delegate footer creation to the footer function.
 *  At the end, if no command container or footer have been populated, this method 
 *  will hide the footer.
 *  
 *  <br>The following properties are analyzed in this method:
 *  <ul>
 *  <li>footer
 *  <li>all actions container props related properties ( see _setActionsContainerProps)
 *  </ul>
 *  
 * @param {Object} props Key-Value pairs of properties  
 * @return true if footer is created, false otherwise
 */
woodstock4_3.widget.modalWindow.prototype._setFooterProps  = function(props)	{
    if (!props) {
        return false;
    }
    
    var showFooter = false;
    
    showFooter = this._setActionsContainerProps(props);
    
    var footer = props.footer;
               
    if (footer != null) {
        this._widget._addFragment(this._footerContent, footer, "last");
        showFooter = true;
    }
        
    this._setContainerVisible(this._windowFooter, showFooter);
    return showFooter;
  
    
        
};

 
/* Populates actions container buttons.
 * All previous content is removed from the _actionContainer.
 * If default buttons have been requested, they will be rendered before
 * custom actions.
 * 
 *  <br>The following properties are analyzed in this method:
 *  <ul>
 *  <li>actions
 *  <li>default buttons: okButton, cancelButton, applyButton
 *  </ul>
 *  
 * @param {Object} props Key-Value pairs of properties  
 * @return true if command container has been created and needs to be displayed, false otherwise
 */
woodstock4_3.widget.modalWindow.prototype._setActionsContainerProps = function(props)	{
    if (!props) {
        return false;
    }

    // Remove child nodes.
    this._widget._removeChildNodes(this._actionContainer);

    var showThis = false;
    
    showThis = this._renderDefaultButton(props, "okButton", true) || showThis;
    showThis = this._renderDefaultButton(props, "applyButton", false)  || showThis;
    showThis = this._renderDefaultButton(props, "cancelButton", false)  || showThis;
    
    //render custom actions
    if (props.actions != null) {       
        //anticipated an array of controls
        for (i = 0; i < props.actions.length; i++) {            
            this._widget._addFragment(this._actionContainer, props.actions[i], "last");            
        }
        showThis = true;
    }
    
    if (showThis) {
        this._setContainerVisible(this._actionContainer, true);
        return true;
    
    } else  {
        this._setContainerVisible(this._actionContainer, false);
        return false;
    }
    return true;
     
};

/** Renders default button. This method checks for presence of 
 * button flag and onclick handler ( buttonName , buttonNameOnClick) and render default button widget
 * @param props properties being set
 * @param buttonName button name
 * @param primary true or false
 * @return true if successful
 */
woodstock4_3.widget.modalWindow.prototype._renderDefaultButton  = function(props, buttonName, primary)	{

    if (!props[buttonName]) {  
        return false;
    } else {
        this[buttonName] = props[buttonName];                
    }

    //render default buttons
    if (props[buttonName + "OnClick"] != null) {  
        //if custom handler has been provided, use it as a pre-Exit action, without any extra logic
        this[buttonName +"OnClickAction"] = this._createActionClickCallback(props[buttonName + "OnClick"]);
            //this._getFunction(props[buttonName + "OnClick"]) ;                
    } else  {
        //default pre-Exit action will publish the event
        this[buttonName +"OnClickAction"] = this._createActionClickCallback(this._createDefaultAction(buttonName));
    }
    
    var buttonProps = {
        widgetType : "button",
        value      : this._theme.getMessage("ModalWindow." + buttonName), 
        primary    : primary,
        id         : this.id + "_"+ buttonName,    
        onClick    : this[buttonName +"OnClickAction"]
    };
    this._widget._addFragment(this._actionContainer, buttonProps, "last");            
    
    return true;


};


/** Cleans up event connections
 * @return true  
 */
woodstock4_3.widget.modalWindow.prototype._cleanupEvents  = function()	{

    if (this.closeButtonConnection != null) {
        woodstock._dojo.disconnect(this.closeButtonConnection);
        this.closeButtonConnection = null;
    }

    if (this.keyConnection != null) {
        woodstock._dojo.disconnect(this.keyConnection);
        this.keyConnection = null;
    }
    return true;
};




/**
 *  Opens up the modalWindow.
 *  <br>
 *  User can supply optional properties. If properties are supplied, 
 *  the setProps(props) will be called prior to opening the widget.
 *  This enables to keep only one instance of the widget and easily re-use
 *  it without calling setProps() every time. 
 *  <br>
 *  If the properties are not specified, widget will open with the previous set of
 *  properties.
 *  
 *  @see close()
 *  
 *  @return true if successful, false otherwise
 */
woodstock4_3.widget.modalWindow.prototype.open  = function(props)	{
    if (props) {
        this.setProps(props);
    }

    this._setContainerVisible(this._domNode, true);
    
    // Publish a visible event 
    this._publish(woodstock4_3.widget.modalWindow.event.open, [{
            id: this.id
        }]);
    
    return true;
};
 



/** Closes the modalWindow.<br>
 *  If a preExitAction function is specified (typeof preExitAction == "function"), such function
 *  will be called and its return considered for exit decision. If return is false, 
 *  window will not exit. Otherwise, return will be passed with the visible.endTopic event as an exitCode.
 *  <br>
 *  If a preExitAction is NOT a function, it is considered to be an exit code.
 *  <br>
 *  If unconditionalExit is true, window will close no matter what return code of preExitAction is.
 *  <br>
 *  
 *      
 *  @param (Object) preExitAction pre-exit method that generates an exitCode
 *  @return true if window is closed, false otherwise
 */
woodstock4_3.widget.modalWindow.prototype.close  = function(preExitAction)	{
    
    //check for BEFORE-EXIT action
    if (typeof preExitAction == "function") {
        try {
            var allowToExit = preExitAction();
            if (allowToExit == false ) {
                //do not allow to close
                return false;
            }
        } catch (err) {
            console.debug(err.toString());
        }
    }
    
    this._setContainerVisible(this._domNode, false);
    this._cleanupEvents();
    
    // Publish close event 
    woodstock4_3.widget.common.publish(woodstock4_3.widget.modalWindow.event.close, [{
            id: this.id,
            exitCode: exitCode
        }]);

    return true;
};
 




    
/*
 * Cleans up if neccessary and updates event registration according to the 
 * current set of widget properties
 * @return true if successful, false otherwise
 */
woodstock4_3.widget.modalWindow.prototype._updateEventsRegistration  = function()	{
    this._cleanupEvents();
    
    if (this.closeButtonAction != null && this.closeButtonConnection == null) {
        // Generate the following event ONLY when closeButton is enabled
        var onClickFunc = this._createActionClickCallback(this.closeButtonAction );
        this.closeButtonConnection =  woodstock._dojo.connect(this._closeButton, "onclick", this, onClickFunc);
    }

    return true;
};


/**
 * Helper function to create callback for keydown event.
 *
 * @return {Function} The callback function.
 * @private
woodstock4_3.widget.modalWindow.prototype._createKeyDownCallback = function() {
    var _id = this.id;
    return function(event) { 
        //check event src
        
        var widget = woodstock4_3.widget.common.getWidget(_id);
        if (widget == null) {
            return false;
        }

        //check key
        if (event && event.keyCode == 27) {
            //simulate close as if triggered by close button click 
            //thus any event associated with the close button can be invoked
            widget.close(this.closeButtonAction);     
        }
        
        
        
        return true;
    };
};
*/


/** Creates a DEFAULT callback action to be assigned to the action or window controls buttons.
 *  Such callback will additionally publish an event.
 *  
 *  @param butName {String} name of the default button
 */
woodstock4_3.widget.modalWindow.prototype._createDefaultAction  = function(butName)	{
    if (butName != "okButton" && butName != "cancelButton" && butName != "applyButton" && butName != "closeButton") {
        return new Function("return true;"); //do nothing
    }
    var _id = this.id;
    
    return function() { 
        var widget = woodstock4_3.widget.common.getWidget(_id);
        if (widget == null) {
            return false;
        }
        widget._publish(woodstock4_3.widget.modalWindow.event[butName + "OnClick"], [{
          id: _id
        }]);
    
        return true;
    };
    
};
 

/** Creates a callback function to be assigned to the action or window controls buttons.
 *  Such callback will invoke close method.
 *  @param preExitAction {Function} optional method to invoke before window is closed
 */
woodstock4_3.widget.modalWindow.prototype._createActionClickCallback  = function(preExitAction)	{
    return this._widget._hitch(this, "close", this._getFunction(preExitAction));    
};
 
/** Utility method to show/hide a container (or any node)
 * @param (Object) node node to show/hide
 * @param (Boolean) visible true/false
 * @return if successfl, false otherwise
 */
woodstock4_3.widget.modalWindow.prototype._setContainerVisible  = function(node, visible)	{
    if (!node) {
        return false;
    }
    node.style.display = (visible == true) ?  "block" : "none";    
    return true;
};


/** Utility method that returns referenceable function from existing function
 * or a string. If argument provided is a string, an anonymous function in format
 * function(event) is created, otherwise the original reference is returned.
 * 
 * @param (String or Function) func function 
 * @return if successfl, false otherwise
 */
woodstock4_3.widget.modalWindow.prototype._getFunction  = function(func)	{

    if (func != null) {
        return (typeof func == 'string')
            ? new Function("event", func)
        : func;
    }
};


woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");


// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.modalWindow.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.popupMenu");

woodstock4_3._dojo.require("woodstock4_3.json");


woodstock4_3._dojo.provide("woodstock4_3.widget.popupMenu");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.menuBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.submitBase");

/**
 * This function is used to construct a popupMenu widget.
 *
 * @name woodstock4_3.widget.popupMenu
 * @extends woodstock4_3.widget._base.menuBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @extends woodstock4_3.widget._base.submitBase
 * @class This class contains functions for the popupMenu widget.
 * @constructor
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.popupMenu", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.submitBase,
        woodstock4_3.widget._base.menuBase ], {
    // Set defaults.
    _widgetType: "popupMenu" // Required for theme properties.
});

/**
 * Close the menu. Sets the visibility to false.
 *
 * @return {boolean} false to cancel the JavaScript event.
 */
woodstock4_3.widget.popupMenu.prototype.close = function() {
    if (this._common._isVisibleElement(this._domNode)) {
        if (woodstock4_3.widget.popupMenu.activeMenuId) {
            woodstock4_3.widget.popupMenu.activeMenuId = null;
        }
        if (this.target != null) {
            if (this.target.focus) {
                this.target.focus();
            }   
        }        
        return this.setProps({visible: false});
    }
    return false;    
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.popupMenu.event =
        woodstock4_3.widget.popupMenu.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_popupMenu_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_popupMenu_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_popupMenu_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_popupMenu_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_popupMenu_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_popupMenu_event_submit_end"
    }
};

/**
 * Helper function to create callback to close menu.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.popupMenu.prototype._onCloseMenuCallBack = function(event) {
    // Capture the click and see whether it falls within the boundary of the menu
    // if so do not close the menu.
    var evt = (event) 
        ? event : ((window.event) 
            ? window.event : null);

    var target = (evt.target) 
        ? evt.target 
        : ((evt.srcElement) 
            ? evt.srcElement : null);
        
    // If key pressed and it's NOT the escape key, do NOT cancel.
    if ((evt.type == "keydown") && (evt.keyCode != 27)) {
        return false;
    }
        
    // If the event occured on the menu, do NOT cancel.
    // Instead we let the event propagate to the MenuItem handler.
    // Cannot use 
    while (target != null) {
        if (target.className == "Menu_sun4") {
            return false;
        }
        target = target.parentNode;
    }

    // The above will not catch events on IE which occur on menuitem seperators
    // or empty space between menuitems.
    var menuLeft = this._domNode.offsetLeft;        
    var menuTop = this._domNode.offsetTop;        
    var tmp;

    var menuRight = menuLeft + this._domNode.offsetWidth - this.rightShadow;
    var menuBottom = menuTop + this._domNode.offsetHeight - this.bottomShadow;

    // Having problems with document.body.scrollTop/scrollLeft in firefox.
    // It always seems to return 0. But window.pageXOffset works fine.
    if (window.pageXOffset || window.pageYOffset) {
        var eventX = evt.clientX + window.pageXOffset;
        var eventY = evt.clientY + window.pageYOffset;
    } else if (document.documentElement.scrollLeft ||
            document.documentElement.scrollTop){
        var eventX = evt.clientX + document.documentElement.scrollLeft;
        var eventY = evt.clientY + document.documentElement.scrollTop;
    } else {
        var eventX = evt.clientX + document.body.scrollLeft;
        var eventY = evt.clientY + document.body.scrollTop;
    }
    if ((eventX >= menuLeft) && (eventX <= menuRight) && (eventY >= menuTop) && 
            (eventY <= menuBottom)) {
        return false;
    }
    if ((evt.type == "keydown" && evt.keyCode == 27) || evt.type == "click") {
        this.close();
    }
    return true;
};

/**
 * Use this function to make the menu visible. It takes an event parameter
 * as an argument.It calculates the position where the menu is to be displayed
 * at if one is not already provided by the developer.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.popupMenu.prototype.open = function(event) {
    var evt = this._widget._getEvent(event);
    var keyCode = this._widget._getKeyCode(evt);
    if (evt.type == "keydown" || evt.type == "keypress") {
        if (!(evt.shiftKey && keyCode == 121)) {
            return false;
        }
        if (woodstock4_3._base.browser._isIe5up()) {
            window.event.cancelBubble = true;
            window.event.returnValue = false;
        } else {
            evt.stopPropagation();
            evt.preventDefault();
        }        
    }
    // Only one menu can be open at a time. Hence, close the previous menu.
    var widget = this._widget.getWidget(woodstock4_3.widget.popupMenu.activeMenuId);
    if (widget) {
        widget.close();
    }
    woodstock4_3.widget.popupMenu.activeMenuId = this.id;

    // Note: Must test if event is null. Otherwise, pressing enter key while
    // link has focus generates an error on IE.
    if (evt) {
        evt.cancelBubble = true;
    }

    // If menu already rendered, do nothing.
    if (this._common._isVisibleElement(this._domNode)) {
        return false;
    }
        
    // Check if developer defined styles are set on the widget.
    if (this.style != null && this.style.length > 0) {
        // Mozilla browsers will tell us which styles are set.  If they're not
        // in the list, then the styles appear to be undefined.
        if (this._domNode.style.length != null) {
            for (var i = 0; i < this._domNode.style.length; i++) {
                var x = this._domNode.style[i];
                if (this._domNode.style[i] == "top") {
                    this.top = this._domNode.style.top;
                }                    
                if (this._domNode.style[i] == "left") {
                    this.left = this._domNode.style.left;
                }
            }
        } else {
            // For IE, simply query the style attributes.
            if (this._domNode.style.top != "") {
                this.top = this._domNode.style.top;
            }
            if (this._domNode.style.left != "") {
                this.left = this._domNode.style.left;
            }
        }
    }

    // Fix: Setting the menu visible here causes flashing. The menu is shown in
    // an old location until moved to the new location in the page.

    // Show the menu. Must do this here, else target properties referenced
    // below will not be valid.
    this.setProps({visible: true});

    // If specific positioning specified, then simply use it.  This means
    // no provisions are made to guarantee the menu renders in the viewable area.
    if ((this.top != null) && (this.left != null)) {
        this._domNode.style.left = this.left;
        this._domNode.style.top = this.top;
    } else {
        // No positioning specified, so we calculate the optimal position to
        // guarantee menu is fully viewable.
        if (evt == null) {
            return false;
        }
        // Note: It appears that the D.Edwards packer tool cannot resolve the
        // target object below. (See the example-html app -- only debug mode 
        // works.) Thus, we must rewrite the code using "if" statments.
        // 
        // var target = (evt.target) 
        //    ? evt.target 
        //    : (evt.srcElement)
        //        ? evt.srcElement : null;

        // Get the absolute position of the target.
        var target = null;
        if (evt.target) {
            target = evt.target;
        } else if (evt.srcElement) {
            target = evt.srcElement;
        }

        var absPos = this._widget._getPosition(target);
        var targetLeft = absPos[0];
        var targetTop = absPos[1];

        // Fix: Menu not positioned b/c target is null -- see example-html app.
        // Possible issue with packer tool encoding?
        var targetHeight = (target) ? target.offsetHeight : 0;

        // Assume default horizontal position is to align the left edge of the 
        // menu with the left edge of the target.
        var menuLeft = targetLeft + this.rightShadow;

        // But can be overridden to align right edges.
        // Check if right edge of menu exceeds page boundary.
        var rightEdge = menuLeft + this._domNode.offsetWidth;
        var pageWidth = this._widget._getPageWidth();
        if (rightEdge > pageWidth) {
            // Shift menu left just enough to bring it into view.
            menuLeft -= (rightEdge - pageWidth);
        }
        
        // If left edge of menu crosses left page boundary, then
        // shift menu right just enough to bring it into view.
        if (menuLeft < 0) {
            menuLeft = 0;
        }

        // Assume default vertical position is to position menu below target.
        var menuTop = targetTop + targetHeight + this.bottomShadow;
        
        // Check if bottom edge of menu exceeds page boundary.
        var bottomEdge = menuTop + this._domNode.offsetHeight - this.bottomShadow;
        if (bottomEdge > this._widget._getPageHeight()) {
            // Shift menu to top of target.
            menuTop = targetTop - this._domNode.offsetHeight;

            // If top edge of menu cross top page boundary, then
            // reposition menu back to below target.
            // User will need to use scrollbars to position menu into view.
            if (menuTop <= 0) {
                menuTop = targetTop + targetHeight - this.bottomShadow;
            }
        }
        // Adjust to account for parent container.
        var parentPos = this._widget._getPosition(this._domNode.offsetParent);
        menuLeft -= parentPos[0];
        menuTop -= parentPos[1]; 
        // Set new menu position.
        this._domNode.style.left = menuLeft + "px";
        this._domNode.style.top = menuTop + "px";
    }

    // Keep track of the element that opened the popup menu.
    // When the menu is closed, the focus is set back on this element.
    if (evt.target) {
        this.target = evt.target;
    } else if (evt.srcElement) {
        this.target = evt.srcElement;
    }
    if (this.target && this.target.blur) {
        this.target.blur();
    }
    
    // Always set the focus on the first element of the menu.
    if (this._focusPosition > 0) {
        var menuNode = document.getElementById(this.menuId[this._focusPosition]); 
        if (menuNode) {
            menuNode.className = this._theme.getClassName("MENU_GROUP_CONTAINER");
        }
    }
    this._focusPosition = 0;
    menuNode = document.getElementById(this.menuId[0]);
    menuNode.className = menuNode.className + " " + 
        this._theme.getClassName("MENU_FOCUS");  
        
    if (menuNode.focus) {
        menuNode.focus();
    }
    return true;        
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.popupMenu.prototype._postCreate = function () {
    // Set public functions.

    /** @ignore */
    this._domNode.open = function(event) { return woodstock4_3.widget.common.getWidget(this.id).open(event); };
    /** @ignore */
    this._domNode.close = function() { return woodstock4_3.widget.common.getWidget(this.id).close(); };

    // Set events.
    this._dojo.connect(document, "onclick", this, "_onCloseMenuCallBack"); 
            
    // escape key should also close menu.
    this._dojo.connect(document, "onkeydown", this, "_onCloseMenuCallBack");  

    // Default widths of the drop shadow on each side of the menu.  These MUST 
    // be in pixel units and MUST match the absolute values of the left/top 
    // styles of the "Menu" style class in the CSS.
    this.rightShadow = parseFloat(this._theme.getMessage("Menu.rightShadow"));
    this.bottomShadow = parseFloat(this._theme.getMessage("Menu.bottomShadow"));
    this._shadowContainer.className = this._theme.getClassName("MENU_SHADOW_CONTAINER"); 

    return this._inherited("_postCreate", arguments);
};

/**
 * Override the "super class" processKeyPressEvent functionality and close the menu.
 *
 * @param (String) value The "value" of the selected option.  
 * @return {boolean} true The enter key press event completed successfully
 * @private
 */
woodstock4_3.widget.popupMenu.prototype._processEnterKeyPressEvent = function(value) {
    this._inherited("_processEnterKeyPressEvent", arguments);
    this.close();
    return true;
};

/**
 * Override the "super class" _processOnClickEvent functionality and close the menu.
 *
 * @param {String} value The selected value.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.popupMenu.prototype._processOnClickEvent = function(value) {
    this._inherited("_processOnClickEvent", arguments);
    this.close();
    return true;
};

/**
 * Traverse through the menu items. This overrides the superclass implementation
 * and handles escape/tab/page up/page down/home/end key press events.
 *
 * @param (String) keyCode The valye of the key which was pressed
 * @param (Event) event The key press event.
 * @param (String) nodeId The id of the menu item. 
 * @return {boolean} true Propagate the javascript event
 * @private
 */
woodstock4_3.widget.popupMenu.prototype._traverseMenu = function(keyCode, event, nodeId) {
    // Handle the escape key and tab key press
    if (keyCode == 27 || keyCode == 9) {
        var focusElem = document.getElementById(this.menuId[this._focusPosition]);
        focusElem.className = this._theme.getClassName("MENU_GROUP_CONTAINER");        
        this.close();
        return true;
    } else if(keyCode >= 33 && keyCode <= 36) {
        focusElem = document.getElementById(this.menuId[this._focusPosition]);        
        focusElem.className = this._theme.getClassName("MENU_GROUP_CONTAINER");
        
        // Handle the home and page Up keys. Focus is set on the first element.
        if (keyCode == 33 || keyCode == 36) {
            this._focusPosition = 0;
            focusElem = document.getElementById(this.menuId[this._focusPosition]);        
        }
        
        // Handle Page Down and End keys. Focus is set on the last element.
        if (keyCode == 34 || keyCode == 35) {
            this._focusPosition = this.menuId.length - 1;
            focusElem = document.getElementById(this.menuId[this._focusPosition]);        
        }
        if (focusElem.focus) {
            focusElem.focus();
        }                        
        focusElem.className = focusElem.className + " " +
            this._theme.getClassName("MENU_FOCUS"); 
        if (woodstock4_3._base.browser._isIe5up()) {
            window. event.cancelBubble = true;
            window.event.returnValue = false;
        } else {
            event.stopPropagation();
            event.preventDefault();
        }   
        return true;                 
    }    
    this._inherited("_traverseMenu", arguments);
    return true;
};

/**
 * Process submit event.
 *
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.popupMenu.prototype.submit = function(execute) {
    // Publish an event for custom AJAX implementations to listen for.
    this._publish(woodstock4_3.widget.popupMenu.event.submit.beginTopic, [{
        id: this.id,
        execute: execute,
        value: this.getSelectedValue(),
        endTopic: woodstock4_3.widget.popupMenu.event.submit.endTopic
    }]);
    return true;
};
woodstock4_3._dojo.require("woodstock4_3.widget._xhr.common");

/**
 * @class This class contains functions to obtain data asynchronously using JSF
 * Extensions as the underlying transfer protocol.
 * @static
 * @private
 */
woodstock4_3.widget._xhr.popupMenu = {
    /**
     * This function is used to process submit events with Object literals. 
     * <p>
     * Note: Cannot use the _processSubmitEvent() function in the common.js file
     * as we need an extra attribute called value to be submitted for every request.
     * </p>
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @config {String} endTopic The event topic to publish.
     * @config {String} execute The string containing a comma separated list 
     * of client ids against which the execute portion of the request 
     * processing lifecycle must be run.
     * @config {String} value The selected menu option value.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _processSubmitEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Generate AJAX request.
        woodstock4_3.widget._xhr.common._doRequest({
            id: props.id,
            callback: woodstock4_3.widget._xhr.common._submitCallback,
            closure: {
                endTopic: props.endTopic
            },
            xjson: {
                id: props.id,
                event: "submit",
                execute: (props.execute) ? props.execute : "none",
                value: props.value
            }
        });
        return true;
    }
};

// Listen for Widget events.
woodstock4_3._dojo.subscribe(woodstock4_3.widget.popupMenu.event.refresh.beginTopic,
    woodstock4_3.widget._xhr.common, "_processRefreshEvent");
woodstock4_3._dojo.subscribe(woodstock4_3.widget.popupMenu.event.submit.beginTopic,
    woodstock4_3.widget._xhr.popupMenu, "_processSubmitEvent");


woodstock4_3._dojo.provide("woodstock4_3.widget._xhr.progressBar");

woodstock4_3._dojo.require("woodstock4_3.json");
woodstock4_3._dojo.require("woodstock4_3.widget.common");


woodstock4_3._dojo.provide("woodstock4_3.widget.progressBar");

woodstock4_3._dojo.require("woodstock4_3.widget.common");
woodstock4_3._dojo.require("woodstock4_3.widget._base.refreshBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.stateBase");
woodstock4_3._dojo.require("woodstock4_3.widget._base.widgetBase");

/**
 * This function is used to construct a progressBar widget.
 *
 * @name woodstock4_3.widget.progressBar
 * @extends woodstock4_3.widget._base.widgetBase
 * @extends woodstock4_3.widget._base.refreshBase
 * @extends woodstock4_3.widget._base.stateBase
 * @class This class contains functions for the progressBar widget.
 * @constructor
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} bottomText 
 * @config {Object} busyImage 
 * @config {String} failedStateText
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} logId 
 * @config {boolean} logMessage 
 * @config {String} overlayAnimation 
 * @config {String} percentChar 
 * @config {int} progress 
 * @config {String} progressImageUrl 
 * @config {String} progressControlBottom
 * @config {String} progressControlRight 
 * @config {int} refreshRate 
 * @config {String} taskState
 * @config {String} toolTip 
 * @config {String} topText 
 * @config {String} type 
 * @config {boolean} visible Hide or show element.
 * @config {int} width 
 */
woodstock4_3._dojo.declare("woodstock4_3.widget.progressBar", [
        woodstock4_3.widget._base.refreshBase, 
        woodstock4_3.widget._base.stateBase,
        woodstock4_3.widget._base.widgetBase ], {
    // Set defaults.
    constructor: function() {
        this.progress = 0;
        this.percentChar = "%";
        this.type = this.determinate;
        this.overlayAnimation = false;
    },
    busy: "BUSY",
    canceled: "canceled",
    completed: "completed",
    determinate: "DETERMINATE",
    failed: "failed",
    indeterminate: "INDETERMINATE",
    notstarted: "not_started",
    paused: "paused",
    resumed: "resumed",
    stopped: "stopped",
    running: "running",
    _widgetType: "progressBar"
});

/**
 * This function handles cancel progressBar event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.cancel = function() {
    clearTimeout(this.timeoutId);

    this._hiddenFieldNode.value = this.canceled;
    this.taskState = this.canceled;
    if (this.type == this.determinate) {
        this._innerBarContainer.style.width = "0%";
    }
    // publish cancel event
    this._publish(woodstock4_3.widget.progressBar.event.cancel.cancelTopic, [{
          id: this.id,
          taskState: this.taskState
        }]);
    return this._updateProgress();
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
woodstock4_3.widget.progressBar.event =
        woodstock4_3.widget.progressBar.prototype.event = {
    /**
     * This closure is used to publish progress events.
     * @ignore
     */
    progress: {
        /** Progress event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_progressBar_event_progress_begin",

        /** Progress event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_progressBar_event_progress_end"
    },
    
    /**
     * This object contains pause event topics.
     * @ignore
     */
    pause: {
        /** pause topic for custom AJAX implementations to listen for. */
        pauseTopic: "woodstock4_3_widget_progressBar_event_pause"
    },
    
    /**
     * This object contains resume event topics.
     * @ignore
     */
    resume: {
        /** resume topic for custom AJAX implementations to listen for. */
        resumeTopic: "woodstock4_3_widget_progressBar_event_resume"
    },
    
    /**
     * This object contains cancel event topics.
     * @ignore
     */
    cancel: {
        /** cancel topic for custom AJAX implementations to listen for. */
        cancelTopic: "woodstock4_3_widget_progressBar_event_cancel"
    },
    
    /**
     * This object contains stop event topics.
     * @ignore
     */
    stop: {
        /** stop topic for custom AJAX implementations to listen for. */
        stopTopic: "woodstock4_3_widget_progressBar_event_stop"
    },
        
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_progressBar_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_progressBar_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "woodstock4_3_widget_progressBar_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "woodstock4_3_widget_progressBar_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the constructor 
 * detail for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
woodstock4_3.widget.progressBar.prototype.getProps = function() {
    var props = this._inherited("getProps", arguments);

    // Set properties.
    if (this.height != null) { props.height = this.height; }
    if (this.width != null) { props.width = this.width; }
    if (this.bottomText != null) { props.bottomText = this.bottomText; }
    if (this.busyImage != null) { props.busyImage = this.busyImage; }
    if (this.failedStateText != null) { props.failedStateText = this.failedStateText; }
    if (this.id != null) { props.id = this.id; }
    if (this.log != null) { props.log = this.log; }
    if (this.logId != null) { props.logId = this.logId; }
    if (this.logMessage != null) { props.logMessage = this.logMessage; }
    if (this.overlayAnimation != null) { props.overlayAnimation = this.overlayAnimation; }
    if (this.percentChar != null) { props.percentChar = this.percentChar; }
    if (this.progress != null) { props.progress = this.progress; }
    if (this.progressImageUrl != null) { props.progressImageUrl = this.progressImageUrl; }
    if (this.progressControlBottom != null) { props.progressControlBottom = this.progressControlBottom; }
    if (this.progressControlRight != null) { props.progressControlRight = this.progressControlRight; }
    if (this.refreshRate != null) { props.refreshRate = this.refreshRate; }
    if (this.taskState != null) { props.taskState = this.taskState; }
    if (this.toolTip != null) { props.toolTip = this.toolTip; }
    if (this.topText != null) { props.topText = this.topText; }
    if (this.type != null) { props.type = this.type; }
    if (this.prefix != null) {props.prefix = this.prefix;}

    return props;
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 * @private
 */
woodstock4_3.widget.progressBar.prototype._getClassName = function() {
    var className = this._inherited("_getClassName", arguments);
    var key = "PROGRESSBAR"; 

    // Get theme property.
    var newClassName = this._theme.getClassName(key, "");
        
    return (className)
        ? newClassName + " " + className
        : newClassName;
};

/**
 * This method displays the Bottom Controls if it was hidden.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.isBottomControlVisible = function() {
    return this._common._isVisibleElement(this._bottomControlsContainer);
};

/**
 * This method displays the failed state message and icon if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.isFailedStateMessageVisible = function() {
    return this._common._isVisibleElement(this._failedStateContainer);
};

/**
 * This method displays the log message if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.isLogMsgVisible = function() {
    return this._common._isVisibleElement(this._logContainer);
};

/**
 * This method displays the operation text if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.isOperationTextVisible = function() {
    return this._common._isVisibleElement(this._topTextContainer);
};

/**
 * This method displays the ProgressBar Container if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.isProgressBarContainerVisible = function() {
    return this._common._isVisibleElement(this._barContainer);
};

/**
 * This method displays the ProgressBar if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.isProgressBarVisible = function() {
    return this._common._isVisibleElement(this); 
};

/**
 * This method displays the Right Controls if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.isRightControlVisible = function() {
    return this._common._isVisibleElement(this._rightControlsContainer);
};

/**
 * This method displays the status text if it was hidden.
 *
* @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.isStatusTextVisible = function() {
    return this._common._isVisibleElement(this._bottomTextContainer);
};

/**
 * This function handles pause button event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.pause = function() {
    clearTimeout(this.timeoutId);

    this._hiddenFieldNode.value = this.paused;
    this.taskState = this.paused;
    if (this.type == this.indeterminate) {
        this._innerBarContainer.className =
            this._theme.getClassName("PROGRESSBAR_INDETERMINATE_PAUSED");
    }
    // publish pause event
    this._publish(woodstock4_3.widget.progressBar.event.pause.pauseTopic, [{
          id: this.id,
          taskState: this.taskState
        }]);
    return this._updateProgress();
};

/**
 * This function is used to fill in remaining template properties, after the
 * _buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
woodstock4_3.widget.progressBar.prototype._postCreate = function () {
    // Set ids.
    if (this.id) {
        this._barContainer.id = this.id + "_barContainer";
        this._bottomControlsContainer.id = this.id + "_bottomControlsContainer";
        this._bottomTextContainer.id = this.id + "_bottomTextContainer"; 
        this._failedStateContainer.id = this.id + "_failedStateContainer";
        this._failedLabelContainer.id = this.id + "_failedLabelContainer";
        this._hiddenFieldNode.id = this.id + "_" + "controlType";
        this._hiddenFieldNode.name = this._hiddenFieldNode.id;
        this._innerBarContainer.id = this.id + "_innerBarContainer";
        this._innerBarOverlayContainer.id = this.id + "_innerBarOverlayContainer";
        this._logContainer.id = this.id + "_logContainer";
        this._rightControlsContainer.id = this.id + "_rightControlsContainer";
        this._topTextContainer.id = this.id + "_topTextContainer"; 
    }

    // Set public functions.

    /** @ignore */
    this._domNode.cancel = function() { return woodstock4_3.widget.common.getWidget(this.id).cancel(); };
    /** @ignore */
    this._domNode.isBottomControlVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).isBottomControlVisible(); };
    /** @ignore */
    this._domNode.isFailedStateMessageVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).isFailedStateMessageVisible(); };
    /** @ignore */
    this._domNode.isLogMsgVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).isLogMsgVisible(); };
    /** @ignore */
    this._domNode.isOperationTextVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).isOperationTextVisible(); };
    /** @ignore */
    this._domNode.isProgressBarContainerVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).isProgressBarContainerVisible(); };
    /** @ignore */
    this._domNode.isProgressBarVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).isProgressBarVisible(); };
    /** @ignore */
    this._domNode.isRightControlVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).isRightControlVisible(); };
    /** @ignore */
    this._domNode.isStatusTextVisible = function() { return woodstock4_3.widget.common.getWidget(this.id).isStatusTextVisible(); };
    /** @ignore */
    this._domNode.pause = function() { return woodstock4_3.widget.common.getWidget(this.id).pause(); };
    /** @ignore */
    this._domNode.resume = function() { return woodstock4_3.widget.common.getWidget(this.id).resume(); };
    /** @ignore */
    this._domNode.startProgress = function() { return woodstock4_3.widget.common.getWidget(this.id).startProgress(); }; 
    /** @ignore */
    this._domNode.stop = function() { return woodstock4_3.widget.common.getWidget(this.id).stop(); };     
    /** @ignore */
    this._domNode.setOnCancel = function(func) { return woodstock4_3.widget.common.getWidget(this.id).setOnCancel(func); };
    /** @ignore */
    this._domNode.setOnComplete = function(func) { return woodstock4_3.widget.common.getWidget(this.id).setOnComplete(func); };
    /** @ignore */
    this._domNode.setOnFail = function(func) { return woodstock4_3.widget.common.getWidget(this.id).setOnFail(func); };
    /** @ignore */
    this._domNode.setBottomControlVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setBottomControlVisible(show); };
    /** @ignore */
    this._domNode.setFailedStateMessageVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setFailedStateMessageVisible(show); };
    /** @ignore */
    this._domNode.setLogMsgVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setLogMsgVisible(show); };
    /** @ignore */
    this._domNode.setOperationTextVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setOperationTextVisible(show); };
    /** @ignore */
    this._domNode.setProgressBarContainerVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setProgressBarContainerVisible(show); };
    /** @ignore */
    this._domNode.setProgressBarVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setProgressBarVisible(show); };
    /** @ignore */
    this._domNode.setRightControlVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setRightControlVisible(show); };
    /** @ignore */
    this._domNode.setStatusTextVisible = function(show) { return woodstock4_3.widget.common.getWidget(this.id).setStatusTextVisible(show); };
    

    if (this.busyImage == null) {
	this.busyImage = {
            icon: "PROGRESS_BUSY",
            id: this.id + "_busy",
            widgetType: "image"
        };
    }
    return this._inherited("_postCreate", arguments);
};

/**
 * This function handles resume button event.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.resume = function() {
    clearTimeout(this.timeoutId);

    this._hiddenFieldNode.value = this.resumed;
    this.taskState = this.resumed;
    if (this.type == this.indeterminate) {
        this._innerBarContainer.className = 
            this._theme.getClassName("PROGRESSBAR_INDETERMINATE");            
    }
    // publish resume event
    this._publish(woodstock4_3.widget.progressBar.event.resume.resumeTopic, [{
          id: this.id,
          taskState: this.taskState
        }]);
    return this._updateProgress();
};

/**
 * This method hides the Bottom Control.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.setBottomControlVisible = function(show) {
    if (show == null) {
        return false;
    }
    this._common._setVisibleElement(this._bottomControlsContainer, show);
    return true;
};

/**
 * This method hides the failed state message and icon area.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.setFailedStateMessageVisible = function(show) {
    if (show == null) {
        return false;
    }
    this._common._setVisibleElement(this._failedStateContainer, show);
    return true;
};

/**
 * This method hides the log message area.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.setLogMsgVisible = function(show) {
    if (show == null) {
        return false;
    }
    this._common._setVisibleElement(this._logContainer, show);
    return true;
};

/**
 * This function invokes developer define function for cancel event.
 * 
 * @param {Function} func The JavaScript function.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.setOnCancel = function(func) {
    if (func) {
        this.funcCanceled = func;
    }
    return true;
};

/**
 * This function invokes developer define function for complete event.
 * 
 * @param {Function} func The JavaScript function.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.setOnComplete = function(func) {
    if (func) {
        this.funcComplete = func;
    }
    return true;
};

/**
 * This function invokes developer define function for failed event.
 * 
 * @param {Function} func The JavaScript function.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.setOnFail = function(func) {
    if (func) {
        this.funcFailed = func;
    }
    return true;
};

/**
 * This method hides the operation text.
 *
 * @param {boolean} show true to show the element, false to hide the element.
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.setOperationTextVisible = function(show) {
    if (show == null) {
        return false;
    }
    this._common._setVisibleElement(this._topTextContainer, show);
    return true;
};

/**
 * This function is used to set progress with Object literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} failedStateText
 * @config {String} logMessage
 * @config {int} progress
 * @config {String} status
 * @config {String} taskState
 * @config {String} topText
 * @config {String} type
 * @return {boolean} true if successful; otherwise, false.
 */
woodstock4_3.widget.progressBar.prototype.setProgress = function(props) {
    if (props == null) {
        return false;
    }
      
    // Adjust max value.
    if (props.progress > 99 
            || props.taskState == this.completed) {
        props.progress = 100;
    }

    // Save properties for later updates.
    this._proto._extend(this, props);    

    // Set status.
    if (props.status) {
        this._widget._addFragment(this._bottomTextContainer, props.status);
    }

    // If top text doesnt get change, dont update.
    if (props.topText) {
        if (props.topText != this.topText) {
            this._widget._addFragment(this._topTextContainer, props.topText);
        }
    }

    // Update log messages.
    if (this.type == this.determinate) { 
        if (props.progress != null && props.progress >= 0 ) {
            this._innerBarContainer.style.width = props.progress + '%';
        }

        if (props.logMessage) {
            var field = this._widget.getWidget(this.logId).getInputElement();
            if (field != null) {
                field.value = (field.value)
                   ? field.value + props.logMessage + "\n"
                   : props.logMessage + "\n";
            }
        }

        // Add overlay text.
        if (this.overlayAnimation == true) {
            // NOTE: If you set this value manually, text must be HTML escaped.
            var progress = props.progress;
            if (progress == null || progress < 0) {
                progress = 0;
            }
            this._widget._addFragment(this._innerBarOverlayContainer, progress + "%");
        }
    } 

    // Failed state.
    if (props.taskState == this.failed) {
        clearTimeout(this.timeoutId);
        this._widget._sleep(1000);
        this.setProgressBarContainerVisible(false);
        this.setBottomControlVisible(false);
        this.setRightControlVisible(false);
        this.setLogMsgVisible(false);

        if (props.failedStateText != null) {
            // NOTE: If you set this value manually, text must be HTML escaped.
            this._widget._addFragment(this._failedLabelContainer,
                props.failedStateText + " " + props.progress + this.percentChar);

            this._common._setVisibleElement(this._failedLabelContainer, true);
            this._common._setVisibleElement(this._failedStateContainer, true);
        }
        if (this.funcFailed != null) {
            (this.funcFailed)();
        }
        this._hiddenFieldNode.value = this.failed;
        return true;
    }

    // Cancel state.
    if (props.taskState == this.canceled) {
        clearTimeout(this.timeoutId);
        this._widget._sleep(1000);
        this.setOperationTextVisible(false);
        this.setStatusTextVisibl