/*
 * Decompiled with CFR 0.152.
 */
package com.twitter.common.thrift;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.twitter.common.base.Closure;
import com.twitter.common.base.Closures;
import com.twitter.common.base.MorePreconditions;
import com.twitter.common.net.pool.Connection;
import com.twitter.common.net.pool.ConnectionFactory;
import com.twitter.common.quantity.Amount;
import com.twitter.common.quantity.Time;
import com.twitter.common.quantity.Unit;
import com.twitter.common.thrift.TTransportConnection;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import org.apache.thrift.transport.TFramedTransport;
import org.apache.thrift.transport.TNonblockingSocket;
import org.apache.thrift.transport.TSocket;
import org.apache.thrift.transport.TTransport;
import org.apache.thrift.transport.TTransportException;

public class ThriftConnectionFactory
implements ConnectionFactory<Connection<TTransport, InetSocketAddress>> {
    private InetSocketAddress endpoint;
    private final int maxConnections;
    private final TransportType transportType;
    private final Amount<Long, Time> socketTimeout;
    private final Closure<Connection<TTransport, InetSocketAddress>> postCreateCallback;
    private boolean sslTransport = false;
    private final Set<Connection<TTransport, InetSocketAddress>> activeConnections = Sets.newSetFromMap((Map)Maps.newIdentityHashMap());
    private volatile int lastActiveConnectionsSize = 0;
    private final Lock activeConnectionsWriteLock = new ReentrantLock(true);

    private static InetSocketAddress asEndpoint(String host, int port) {
        MorePreconditions.checkNotBlank((String)host);
        Preconditions.checkArgument((port > 0 ? 1 : 0) != 0);
        return InetSocketAddress.createUnresolved(host, port);
    }

    public ThriftConnectionFactory(String host, int port, int maxConnections) {
        this(host, port, maxConnections, TransportType.BLOCKING);
    }

    public ThriftConnectionFactory(String host, int port, int maxConnections, boolean framedTransport) {
        this(ThriftConnectionFactory.asEndpoint(host, port), maxConnections, TransportType.get(framedTransport, false));
    }

    public ThriftConnectionFactory(InetSocketAddress endpoint, int maxConnections, boolean framedTransport) {
        this(endpoint, maxConnections, TransportType.get(framedTransport, false));
    }

    public ThriftConnectionFactory(String host, int port, int maxConnections, TransportType transportType) {
        this(host, port, maxConnections, transportType, null);
    }

    public ThriftConnectionFactory(String host, int port, int maxConnections, TransportType transportType, Amount<Long, Time> socketTimeout) {
        this(ThriftConnectionFactory.asEndpoint(host, port), maxConnections, transportType, socketTimeout);
    }

    public ThriftConnectionFactory(InetSocketAddress endpoint, int maxConnections, TransportType transportType) {
        this(endpoint, maxConnections, transportType, null);
    }

    public ThriftConnectionFactory(InetSocketAddress endpoint, int maxConnections, TransportType transportType, Amount<Long, Time> socketTimeout) {
        this(endpoint, maxConnections, transportType, socketTimeout, (Closure<Connection<TTransport, InetSocketAddress>>)Closures.noop(), false);
    }

    public ThriftConnectionFactory(InetSocketAddress endpoint, int maxConnections, TransportType transportType, Amount<Long, Time> socketTimeout, Closure<Connection<TTransport, InetSocketAddress>> postCreateCallback, boolean sslTransport) {
        Preconditions.checkArgument((maxConnections > 0 ? 1 : 0) != 0, (Object)"maxConnections must be at least 1");
        if (socketTimeout != null) {
            Preconditions.checkArgument(((Long)socketTimeout.as((Unit)Time.MILLISECONDS) >= 0L ? 1 : 0) != 0);
        }
        this.endpoint = (InetSocketAddress)Preconditions.checkNotNull((Object)endpoint);
        this.maxConnections = maxConnections;
        this.transportType = transportType;
        this.socketTimeout = socketTimeout;
        this.postCreateCallback = (Closure)Preconditions.checkNotNull(postCreateCallback);
        this.sslTransport = sslTransport;
    }

    public boolean mightCreate() {
        return this.lastActiveConnectionsSize < this.maxConnections;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Connection<TTransport, InetSocketAddress> create(Amount<Long, Time> timeout) throws TTransportException, IOException {
        Preconditions.checkNotNull(timeout);
        if ((Long)timeout.getValue() == 0L) {
            return this.create();
        }
        try {
            long timeRemainingNs = (Long)timeout.as((Unit)Time.NANOSECONDS);
            long start = System.nanoTime();
            if (!this.activeConnectionsWriteLock.tryLock(timeRemainingNs, TimeUnit.NANOSECONDS)) return null;
            try {
                if (!this.willCreateSafe()) {
                    Connection<TTransport, InetSocketAddress> connection = null;
                    return connection;
                }
                Connection<TTransport, InetSocketAddress> connection = this.createConnection((int)TimeUnit.NANOSECONDS.toMillis(timeRemainingNs -= System.nanoTime() - start));
                return connection;
            }
            finally {
                this.activeConnectionsWriteLock.unlock();
            }
        }
        catch (InterruptedException e) {
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Connection<TTransport, InetSocketAddress> create() throws TTransportException, IOException {
        this.activeConnectionsWriteLock.lock();
        try {
            if (!this.willCreateSafe()) {
                Connection<TTransport, InetSocketAddress> connection = null;
                return connection;
            }
            Connection<TTransport, InetSocketAddress> connection = this.createConnection(0);
            return connection;
        }
        finally {
            this.activeConnectionsWriteLock.unlock();
        }
    }

    private Connection<TTransport, InetSocketAddress> createConnection(int timeoutMillis) throws TTransportException, IOException {
        TTransport transport = this.createTransport(timeoutMillis);
        if (transport == null) {
            return null;
        }
        TTransportConnection connection = new TTransportConnection(transport, this.endpoint);
        this.postCreateCallback.execute((Object)connection);
        this.activeConnections.add(connection);
        this.lastActiveConnectionsSize = this.activeConnections.size();
        return connection;
    }

    private boolean willCreateSafe() {
        return this.activeConnections.size() < this.maxConnections;
    }

    @VisibleForTesting
    TTransport createTransport(int timeoutMillis) throws TTransportException, IOException {
        TSocket socket = null;
        if (this.transportType != TransportType.NONBLOCKING) {
            if (timeoutMillis <= 0) {
                return null;
            }
            if (this.sslTransport) {
                SSLSocketFactory factory = (SSLSocketFactory)SSLSocketFactory.getDefault();
                SSLSocket ssl_socket = (SSLSocket)factory.createSocket(this.endpoint.getHostName(), this.endpoint.getPort());
                ssl_socket.setSoTimeout(timeoutMillis);
                return new TSocket((Socket)ssl_socket);
            }
            socket = new TSocket(this.endpoint.getHostName(), this.endpoint.getPort(), timeoutMillis);
        }
        try {
            switch (this.transportType) {
                case BLOCKING: {
                    socket.open();
                    this.setSocketTimeout(socket);
                    return socket;
                }
                case FRAMED: {
                    TFramedTransport transport = new TFramedTransport(socket);
                    transport.open();
                    this.setSocketTimeout(socket);
                    return transport;
                }
                case NONBLOCKING: {
                    try {
                        return new TNonblockingSocket(this.endpoint.getHostName(), this.endpoint.getPort());
                    }
                    catch (IOException e) {
                        throw new IOException("Failed to create non-blocking transport to " + this.endpoint, e);
                    }
                }
            }
        }
        catch (TTransportException e) {
            throw new TTransportException("Failed to create transport to " + this.endpoint, (Throwable)e);
        }
        throw new IllegalArgumentException("unknown transport type " + (Object)((Object)this.transportType));
    }

    private void setSocketTimeout(TSocket socket) {
        if (this.socketTimeout != null) {
            socket.setTimeout(((Long)this.socketTimeout.as((Unit)Time.MILLISECONDS)).intValue());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void destroy(Connection<TTransport, InetSocketAddress> connection) {
        this.activeConnectionsWriteLock.lock();
        try {
            boolean wasActiveConnection = this.activeConnections.remove(connection);
            Preconditions.checkArgument((boolean)wasActiveConnection, (String)"connection %s not created by this factory", (Object[])new Object[]{connection});
            this.lastActiveConnectionsSize = this.activeConnections.size();
        }
        finally {
            this.activeConnectionsWriteLock.unlock();
        }
        connection.close();
    }

    public String toString() {
        return String.format("%s[%s]", this.getClass().getSimpleName(), this.endpoint);
    }

    public static enum TransportType {
        BLOCKING,
        FRAMED,
        NONBLOCKING;


        public static TransportType get(boolean framedTransport, boolean nonblocking) {
            if (nonblocking) {
                Preconditions.checkArgument((boolean)framedTransport, (Object)"nonblocking client requires a server running framed transport");
                return NONBLOCKING;
            }
            return framedTransport ? FRAMED : BLOCKING;
        }
    }
}

