/*
 * Decompiled with CFR 0.152.
 */
package com.twitter.common.thrift;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.twitter.common.base.ExceptionalFunction;
import com.twitter.common.net.loadbalancing.RequestTracker;
import com.twitter.common.net.monitoring.ConnectionMonitor;
import com.twitter.common.net.monitoring.TrafficMonitor;
import com.twitter.common.quantity.Amount;
import com.twitter.common.quantity.Time;
import com.twitter.common.quantity.Unit;
import com.twitter.common.stats.Stat;
import com.twitter.common.stats.StatImpl;
import com.twitter.common.stats.Stats;
import com.twitter.common.thrift.monitoring.TMonitoredProcessor;
import com.twitter.common.thrift.monitoring.TMonitoredServerSocket;
import com.twitter.thrift.Status;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.thrift.TProcessor;
import org.apache.thrift.TProcessorFactory;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TCompactProtocol;
import org.apache.thrift.protocol.TProtocolFactory;
import org.apache.thrift.server.THsHaServer;
import org.apache.thrift.server.TNonblockingServer;
import org.apache.thrift.server.TServer;
import org.apache.thrift.server.TThreadPoolServer;
import org.apache.thrift.transport.TFramedTransport;
import org.apache.thrift.transport.TNonblockingServerSocket;
import org.apache.thrift.transport.TNonblockingServerTransport;
import org.apache.thrift.transport.TServerSocket;
import org.apache.thrift.transport.TServerTransport;
import org.apache.thrift.transport.TTransportException;
import org.apache.thrift.transport.TTransportFactory;

public abstract class ThriftServer {
    private static final Logger LOG = Logger.getLogger(ThriftServer.class.getName());
    public static final Supplier<TProtocolFactory> BINARY_PROTOCOL = new Supplier<TProtocolFactory>(){

        public TProtocolFactory get() {
            return new TBinaryProtocol.Factory(false, true);
        }
    };
    public static final Supplier<TProtocolFactory> COMPACT_PROTOCOL = new Supplier<TProtocolFactory>(){

        public TProtocolFactory get() {
            return new TCompactProtocol.Factory();
        }
    };
    public static final ExceptionalFunction<ServerSetup, TServer, TTransportException> THREADPOOL_SERVER = new ExceptionalFunction<ServerSetup, TServer, TTransportException>(){

        public TServer apply(ServerSetup setup) throws TTransportException {
            TThreadPoolServer.Options options = new TThreadPoolServer.Options();
            if (setup.getNumThreads() > 0) {
                options.minWorkerThreads = setup.getNumThreads();
                options.maxWorkerThreads = setup.getNumThreads();
            }
            if (setup.getSocket() == null) {
                try {
                    setup.setSocket(new ServerSocket(setup.getPort()));
                }
                catch (IOException e) {
                    throw new TTransportException("Failed to create server socket on port " + setup.getPort(), (Throwable)e);
                }
            }
            TServerSocket unmonitoredSocket = null;
            TMonitoredServerSocket monitoredSocket = null;
            if (setup.isMonitored()) {
                monitoredSocket = new TMonitoredServerSocket(setup.getSocket(), (int)((Integer)setup.getSocketTimeout().as((Unit)Time.MILLISECONDS)), (ConnectionMonitor<InetSocketAddress>)setup.getMonitor());
            } else {
                unmonitoredSocket = new TServerSocket(setup.getSocket(), ((Integer)setup.getSocketTimeout().as((Unit)Time.MILLISECONDS)).intValue());
            }
            TTransportFactory transportFactory = new TTransportFactory();
            TProcessor processor = setup.getProcessor();
            if (setup.isMonitored()) {
                processor = new TMonitoredProcessor(processor, monitoredSocket, (RequestTracker<InetSocketAddress>)setup.getMonitor());
            }
            TMonitoredServerSocket socket = setup.isMonitored() ? monitoredSocket : unmonitoredSocket;
            return new TThreadPoolServer(processor, (TServerTransport)socket, transportFactory, transportFactory, setup.getProtoFactory(), setup.getProtoFactory(), options);
        }
    };
    public static final ExceptionalFunction<ServerSetup, TServer, TTransportException> NONBLOCKING_SERVER = new ExceptionalFunction<ServerSetup, TServer, TTransportException>(){

        public TServer apply(ServerSetup setup) throws TTransportException {
            TNonblockingServerSocket socket = setup.getSocketTimeout() == null ? new TNonblockingServerSocket(setup.getPort()) : new TNonblockingServerSocket(setup.getPort(), ((Integer)setup.getSocketTimeout().as((Unit)Time.MILLISECONDS)).intValue());
            setup.setSocket(ThriftServer.getServerSocketFor(socket));
            THsHaServer.Options options = new THsHaServer.Options();
            if (setup.getNumThreads() > 0) {
                options.workerThreads = setup.getNumThreads();
            }
            final ArrayBlockingQueue<Runnable> queue = new ArrayBlockingQueue<Runnable>(setup.getQueueSize() > 0 ? setup.getQueueSize() : options.workerThreads);
            final ThreadPoolExecutor invoker = new ThreadPoolExecutor(options.workerThreads, options.workerThreads, options.stopTimeoutVal, options.stopTimeoutUnit, queue);
            String serverName = setup.getName() != null ? setup.getName() : "no-name";
            Stats.export((Stat)new StatImpl<Integer>(serverName + "_thrift_server_active_threads"){

                public Integer read() {
                    return invoker.getActiveCount();
                }
            });
            Stats.export((Stat)new StatImpl<Integer>(serverName + "_thrift_server_queue_size"){

                public Integer read() {
                    return queue.size();
                }
            });
            return new THsHaServer(new TProcessorFactory(setup.getProcessor()), (TNonblockingServerTransport)socket, new TFramedTransport.Factory(), setup.getProtoFactory(), setup.getProtoFactory(), (ExecutorService)invoker, new TNonblockingServer.Options());
        }
    };
    private final String name;
    private final String version;
    private ServerSetup serverSetup = null;
    private TServer server = null;
    private Thread listeningThread;
    private Status status = Status.STARTING;
    private long serverStartNanos = -1L;
    private final Supplier<TProtocolFactory> protoFactorySupplier;
    private final ExceptionalFunction<ServerSetup, TServer, TTransportException> serverSupplier;

    @VisibleForTesting
    static ServerSocket getServerSocketFor(TNonblockingServerSocket thriftSocket) throws TTransportException {
        try {
            Field field = TNonblockingServerSocket.class.getDeclaredField("serverSocket_");
            field.setAccessible(true);
            return (ServerSocket)field.get(thriftSocket);
        }
        catch (NoSuchFieldException e) {
            throw new TTransportException("Couldn't get listening port", (Throwable)e);
        }
        catch (SecurityException e) {
            throw new TTransportException("Couldn't get listening port", (Throwable)e);
        }
        catch (IllegalAccessException e) {
            throw new TTransportException("Couldn't get listening port", (Throwable)e);
        }
    }

    public ThriftServer(String name, String version) {
        this(name, version, BINARY_PROTOCOL, THREADPOOL_SERVER);
    }

    public ThriftServer(String name, String version, Supplier<TProtocolFactory> protoFactorySupplier, ExceptionalFunction<ServerSetup, TServer, TTransportException> serverSupplier) {
        this.name = (String)Preconditions.checkNotNull((Object)name);
        this.version = (String)Preconditions.checkNotNull((Object)version);
        this.protoFactorySupplier = (Supplier)Preconditions.checkNotNull(protoFactorySupplier);
        this.serverSupplier = (ExceptionalFunction)Preconditions.checkNotNull(serverSupplier);
    }

    public void start(int port, TProcessor processor) {
        this.start(new ServerSetup(this.name, port, processor, (TProtocolFactory)this.protoFactorySupplier.get()));
    }

    public void start(ServerSetup setup) {
        Preconditions.checkNotNull((Object)setup.getProcessor());
        Preconditions.checkState((this.status != Status.ALIVE ? 1 : 0) != 0, (Object)"Server must only be started once.");
        this.setStatus(Status.ALIVE);
        try {
            this.doStart(setup);
        }
        catch (TTransportException e) {
            LOG.log(Level.SEVERE, "Failed to open thrift socket.", e);
            this.setStatus(Status.DEAD);
        }
    }

    @VisibleForTesting
    protected void doStart(ServerSetup setup) throws TTransportException {
        this.serverSetup = setup;
        this.server = (TServer)this.serverSupplier.apply((Object)setup);
        this.serverStartNanos = System.nanoTime();
        LOG.info("Starting thrift server on port " + this.getListeningPort());
        this.listeningThread = new ThreadFactoryBuilder().setDaemon(false).build().newThread(new Runnable(){

            @Override
            public void run() {
                try {
                    ThriftServer.this.server.serve();
                }
                catch (Throwable t) {
                    LOG.log(Level.WARNING, "Uncaught exception while attempting to handle service requests: " + t, t);
                    ThriftServer.this.setStatus(Status.DEAD);
                }
            }
        });
        this.listeningThread.start();
    }

    public int getListeningPort() {
        Preconditions.checkState((this.serverSetup != null ? 1 : 0) != 0);
        Preconditions.checkState((this.status == Status.ALIVE ? 1 : 0) != 0);
        Preconditions.checkState((this.serverSetup.getSocket() != null ? 1 : 0) != 0);
        return this.serverSetup.getSocket().getLocalPort();
    }

    public String getName() {
        return this.name;
    }

    public String getVersion() {
        return this.version;
    }

    public Status getStatus() {
        return this.status;
    }

    protected void setStatus(Status status) {
        LOG.info("Moving from status " + this.status + " to " + status);
        this.status = status;
    }

    public long uptime() {
        return TimeUnit.SECONDS.convert(System.nanoTime() - this.serverStartNanos, TimeUnit.NANOSECONDS);
    }

    protected void tryShutdown() throws Exception {
    }

    public void shutdown() {
        if (this.status == Status.STOPPED) {
            LOG.info("Server already stopped, shutdown request ignored.");
            return;
        }
        LOG.info("Received shutdown request, stopping server.");
        this.setStatus(Status.STOPPING);
        if (this.server != null) {
            this.server.stop();
        }
        this.server = null;
        try {
            this.tryShutdown();
        }
        catch (Exception e) {
            LOG.log(Level.WARNING, "Service handler vetoed shutdown request.", e);
            this.setStatus(Status.WARNING);
            return;
        }
        this.setStatus(Status.STOPPED);
    }

    public void awaitShutdown(Amount<Long, Time> timeout) throws InterruptedException {
        Preconditions.checkNotNull(timeout);
        this.shutdown();
        if (this.listeningThread != null) {
            this.listeningThread.join((Long)timeout.as((Unit)Time.MILLISECONDS));
        }
    }

    public static final class ServerSetup {
        private final String name;
        private final int port;
        private final TProcessor processor;
        private final TProtocolFactory protoFactory;
        private final int numThreads;
        private final int queueSize;
        private final TrafficMonitor<InetSocketAddress> monitor;
        private ServerSocket socket = null;
        private final Amount<Integer, Time> socketTimeout;

        public ServerSetup(int port, TProcessor processor, TProtocolFactory protoFactory) {
            this(port, processor, protoFactory, -1, (Amount<Integer, Time>)Amount.of((int)0, (Unit)Time.MILLISECONDS));
        }

        public ServerSetup(String name, int port, TProcessor processor, TProtocolFactory protoFactory) {
            this(name, port, processor, protoFactory, -1, (Amount<Integer, Time>)Amount.of((int)0, (Unit)Time.MILLISECONDS));
        }

        public ServerSetup(int port, TProcessor processor, TProtocolFactory protoFactory, TrafficMonitor<InetSocketAddress> monitor) {
            this(null, port, processor, protoFactory, -1, (Amount<Integer, Time>)Amount.of((int)0, (Unit)Time.MILLISECONDS), monitor);
        }

        public ServerSetup(int port, TProcessor processor, TProtocolFactory protoFactory, int numThreads, Amount<Integer, Time> socketTimeout) {
            this(null, port, processor, protoFactory, numThreads, socketTimeout, null);
        }

        public ServerSetup(String name, int port, TProcessor processor, TProtocolFactory protoFactory, int numThreads, Amount<Integer, Time> socketTimeout) {
            this(name, port, processor, protoFactory, numThreads, socketTimeout, null);
        }

        public ServerSetup(String name, int port, TProcessor processor, TProtocolFactory protoFactory, int numThreads, int queueSize, Amount<Integer, Time> socketTimeout) {
            this(name, port, processor, protoFactory, numThreads, queueSize, socketTimeout, null);
        }

        public ServerSetup(String name, int port, TProcessor processor, TProtocolFactory protoFactory, int numThreads, Amount<Integer, Time> socketTimeout, TrafficMonitor<InetSocketAddress> monitor) {
            this(name, port, processor, protoFactory, numThreads, -1, socketTimeout, monitor);
        }

        public ServerSetup(String name, int port, TProcessor processor, TProtocolFactory protoFactory, int numThreads, int queueSize, Amount<Integer, Time> socketTimeout, TrafficMonitor<InetSocketAddress> monitor) {
            Preconditions.checkArgument((port >= 0 && port < 65535 ? 1 : 0) != 0, (Object)("Invalid port: " + port));
            Preconditions.checkArgument((numThreads != 0 ? 1 : 0) != 0);
            Preconditions.checkArgument((queueSize != 0 ? 1 : 0) != 0);
            if (socketTimeout != null) {
                Preconditions.checkArgument(((Integer)socketTimeout.getValue() >= 0 ? 1 : 0) != 0);
            }
            this.name = name;
            this.port = port;
            this.processor = processor;
            this.protoFactory = protoFactory;
            this.numThreads = numThreads;
            this.queueSize = queueSize;
            this.socketTimeout = socketTimeout;
            this.monitor = monitor;
        }

        public String getName() {
            return this.name;
        }

        public int getPort() {
            return this.port;
        }

        public int getNumThreads() {
            return this.numThreads;
        }

        public int getQueueSize() {
            return this.queueSize;
        }

        public Amount<Integer, Time> getSocketTimeout() {
            return this.socketTimeout;
        }

        public TProcessor getProcessor() {
            return this.processor;
        }

        public TProtocolFactory getProtoFactory() {
            return this.protoFactory;
        }

        public ServerSocket getSocket() {
            return this.socket;
        }

        public void setSocket(ServerSocket socket) {
            this.socket = socket;
        }

        public boolean isMonitored() {
            return this.monitor != null;
        }

        public TrafficMonitor<InetSocketAddress> getMonitor() {
            return this.monitor;
        }
    }
}

