package com.vaadin.copilot;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.shared.util.SharedUtil;

import elemental.json.Json;
import elemental.json.JsonObject;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.RecordDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AnnotationExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.nodeTypes.NodeWithAnnotations;
import com.github.javaparser.ast.nodeTypes.NodeWithSimpleName;
import com.github.javaparser.ast.nodeTypes.NodeWithType;
import com.github.javaparser.resolution.UnsolvedSymbolException;
import com.github.javaparser.resolution.types.ResolvedType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Handles commands to parse Java source code. */
public class JavaParserHandler implements CopilotCommand {

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("parse-properties")) {
            String reqId = data.getString(KEY_REQ_ID);
            JsonObject respData = Json.createObject();
            respData.put(KEY_REQ_ID, reqId);
            try {
                String content = data.getString("source");
                List<PropertyData> propertyData = parseProperties(content);
                respData.put("properties", JsonUtils.listToJson(propertyData));
                devToolsInterface.send(command + "-response", respData);
                return true;
            } catch (Exception e) {
                getLogger().debug("Failed to parse properties for input {}", data.toJson(), e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Failed to parse properties", e);
            }
        }
        return false;
    }

    /**
     * Holds data for a Java annotation.
     *
     * @param name
     *            the name of the annotation
     * @param parameters
     *            the annotation parameters
     */
    public record AnnotationData(String name, Map<String, Object> parameters) {
    }

    /**
     * Holds data for a Java bean property.
     *
     * @param name
     *            the name of the property
     * @param humanFriendlyName
     *            a human friendly name of the property - can be used in the UI
     * @param type
     *            the type of the property
     * @param annotations
     *            the annotations set on the property
     */
    public record PropertyData(String name, String humanFriendlyName, String type, List<AnnotationData> annotations) {
    }

    static List<PropertyData> parseProperties(String source) {
        CompilationUnit compilationUnit = new JavaRewriter(source, true).getCompilationUnit();
        List<FieldDeclaration> fieldDeclarations = compilationUnit.findAll(FieldDeclaration.class);

        if (!fieldDeclarations.isEmpty()) {
            return parseFields(fieldDeclarations);
        }
        List<RecordDeclaration> records = compilationUnit.findAll(RecordDeclaration.class);
        if (!records.isEmpty()) {
            return parseRecord(records.get(0));
        }
        return Collections.emptyList();
    }

    private static List<PropertyData> parseFields(List<FieldDeclaration> fieldDeclarations) {
        return fieldDeclarations.stream().map(fieldDeclaration -> {
            VariableDeclarator variable = fieldDeclaration.getVariable(0);
            return parse(variable, fieldDeclaration);
        }).filter(Objects::nonNull).toList();
    }

    private static <T extends NodeWithSimpleName<?> & NodeWithType<?, ?> & NodeWithAnnotations<?>> PropertyData parse(
            T node) {
        return parse(node, node);
    }

    private static <T extends NodeWithSimpleName<?> & NodeWithType<?, ?>> PropertyData parse(T node,
            NodeWithAnnotations<?> nodeWithAnnotations) {
        String name = node.getNameAsString();
        try {
            String humanFriendlyName = SharedUtil.camelCaseToHumanFriendly(name);
            ResolvedType type = node.getType().resolve();
            String typeName;
            if (type.isReferenceType()) {
                typeName = type.asReferenceType().getQualifiedName();
            } else {
                typeName = type.describe();
            }
            List<AnnotationData> annotationData = nodeWithAnnotations.getAnnotations().stream()
                    .map(JavaParserHandler::parseAnnotation).toList();
            return new PropertyData(name, humanFriendlyName, typeName, annotationData);
        } catch (UnsolvedSymbolException e) {
            getLogger().debug("Ignoring property ({}} of type ({}) that could not be resolved", name,
                    node.getTypeAsString(), e);
            return null;
        }
    }

    private static List<PropertyData> parseRecord(RecordDeclaration recordDeclaration) {
        return recordDeclaration.getParameters().stream().map(JavaParserHandler::parse).toList();
    }

    private static AnnotationData parseAnnotation(AnnotationExpr annotationExpr) {

        if (annotationExpr.isNormalAnnotationExpr()) {
            return new AnnotationData(annotationExpr.getNameAsString(),
                    annotationExpr.asNormalAnnotationExpr().getPairs().stream().collect(
                            Collectors.toMap(pair -> pair.getNameAsString(), pair -> getValue(pair.getValue()))));
        } else if (annotationExpr.isSingleMemberAnnotationExpr()) {
            return new AnnotationData(annotationExpr.getNameAsString(),
                    Map.of("value", getValue(annotationExpr.asSingleMemberAnnotationExpr().getMemberValue())));
        } else {
            return new AnnotationData(annotationExpr.getNameAsString(), Map.of());
        }
    }

    private static Object getValue(Expression value) {
        try {
            return JavaRewriterUtil.fromExpression(value, null);
        } catch (IllegalArgumentException e) {
            return value.toString();
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JavaParserHandler.class);
    }
}
