package com.vaadin.copilot;

import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import elemental.json.JsonArray;
import elemental.json.JsonObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Detects the source folders in use by the project.
 */
public class JavaSourcePathDetector {

    public record PathInfo(List<Path> sourceFolders, List<Path> resourceFolders, Path projectRoot) {
    }

    private JavaSourcePathDetector() {
        // Utils only
    }

    /**
     * Detects the source folders in use by the project.
     *
     * @param applicationConfiguration
     *            the application configuration
     * @return the source folders in use by the project
     */
    public static PathInfo detectSourcePaths(ApplicationConfiguration applicationConfiguration) {
        return detectSourceFoldersUsingIDEPlugin()
                .orElseGet(() -> detectSingleModuleSourceFolders(applicationConfiguration));
    }

    private static Optional<PathInfo> detectSourceFoldersUsingIDEPlugin() {
        CopilotIDEPlugin plugin = CopilotIDEPlugin.getInstance();
        if (!plugin.isActive()) {
            return Optional.empty();
        }
        try {

            JsonObject sourcePathResponse = plugin.getSourcePaths();

            List<Path> source = new ArrayList<>();
            source.addAll(stringArrayToPathList(sourcePathResponse, "sourcePaths"));
            source.addAll(stringArrayToPathList(sourcePathResponse, "testSourcePaths"));

            List<Path> resource = new ArrayList<>();
            resource.addAll(stringArrayToPathList(sourcePathResponse, "resourcePaths"));
            resource.addAll(stringArrayToPathList(sourcePathResponse, "testResourcePaths"));

            return Optional.of(filterAndFindRoot(source, resource));
        } catch (CopilotIDEPlugin.UnsupportedOperationByPluginException e) {
            return Optional.empty();
        }
    }

    private static PathInfo filterAndFindRoot(List<Path> sourceFolders, List<Path> resourceFolders) {
        try {
            // We should be able to find the build folder from the classpath, e.g.
            // some/app/target/classes
            // If we do, we filter out irrelevant folders from the IDE project, i.e. folders
            // that are not actually deployed
            // but still exist in the same IDE project (such as the Hilla test project in
            // Copilot when running the Flow test project)
            List<Path> classpathRootFolders = getClasspathModuleRootFolders();
            if (!classpathRootFolders.isEmpty()) {
                sourceFolders = sourceFolders.stream()
                        .filter(sourceFolder -> isInAnyFolder(classpathRootFolders, sourceFolder)).toList();
                resourceFolders = resourceFolders.stream()
                        .filter(resourceFolder -> isInAnyFolder(classpathRootFolders, resourceFolder)).toList();
            }
        } catch (IOException | URISyntaxException e) {
            getLogger().error("Error filtering source folders using classpath", e);
        }

        return new PathInfo(sourceFolders, resourceFolders, getProjectRoot(sourceFolders, resourceFolders));
    }

    private static Path getProjectRoot(List<Path> sourceFolders, List<Path> resourceFolders) {
        List<Path> allPaths = Stream.concat(sourceFolders.stream(), resourceFolders.stream()).toList();
        Optional<Path> projectRoot = Util.findCommonAncestor(allPaths);
        if (projectRoot.isEmpty()) {
            throw new IllegalStateException("Unable to deduce project folder using source paths: " + allPaths);
        }
        return projectRoot.get();
    }

    private static List<Path> getClasspathModuleRootFolders() throws IOException, URISyntaxException {
        return getClasspathClassesFolders().stream().map(JavaSourcePathDetector::getProjectFolderFromClasspath)
                .filter(Objects::nonNull).toList();
    }

    private static Path getProjectFolderFromClasspath(Path classesFolder) {
        return classesFolder.endsWith(Path.of("target", "classes")) ? classesFolder.getParent().getParent() : null;
    }

    private static boolean isInAnyFolder(List<Path> folders, Path target) {
        return folders.stream().anyMatch(folder -> {
            try {
                return ProjectManager.isFileInside(target.toFile(), folder.toFile());
            } catch (IOException e) {
                getLogger().error("Error checking if {} is inside {}", target, folder, e);
                return false;
            }
        });
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JavaSourcePathDetector.class);
    }

    private static List<Path> getClasspathClassesFolders() throws IOException, URISyntaxException {
        Enumeration<URL> resources = JavaSourcePathDetector.class.getClassLoader().getResources(".");

        List<Path> paths = new ArrayList<>();
        while (resources.hasMoreElements()) {
            URL url = resources.nextElement();
            if (url.getProtocol().equals("file")) {
                Path path = Path.of(url.toURI());
                paths.add(path);
            }
        }
        return paths;
    }

    private static List<Path> stringArrayToPathList(JsonObject jsonObject, String key) {
        JsonArray pathsJson = jsonObject.getArray(key);
        List<Path> paths = new ArrayList<>();
        for (int i = 0; i < pathsJson.length(); i++) {
            paths.add(Path.of(pathsJson.getString(i)));
        }
        return paths;
    }

    static PathInfo detectSingleModuleSourceFolders(ApplicationConfiguration applicationConfiguration) {
        List<Path> sourcePaths = new ArrayList<>();
        List<Path> resourcePaths = new ArrayList<>();

        Path javaSourceFolder = applicationConfiguration.getJavaSourceFolder().toPath();
        Path javaResourceFolder = applicationConfiguration.getJavaResourceFolder().toPath();

        addIfExists(sourcePaths, javaSourceFolder);
        addIfExists(resourcePaths, javaResourceFolder);

        Path kotlinSourceFolder = Util.replaceFolderInPath(javaSourceFolder, "java", "kotlin", "main");
        addIfExists(sourcePaths, kotlinSourceFolder);

        addIfExists(sourcePaths, Util.replaceFolderInPath(javaSourceFolder, "main", "test", "src"));
        addIfExists(sourcePaths, Util.replaceFolderInPath(kotlinSourceFolder, "main", "test", "src"));

        addIfExists(resourcePaths, Util.replaceFolderInPath(javaResourceFolder, "main", "test", "src"));

        return new PathInfo(sourcePaths, resourcePaths, applicationConfiguration.getProjectFolder().toPath());
    }

    private static void addIfExists(List<Path> sourcePaths, Path path) {
        if (path.toFile().exists()) {
            sourcePaths.add(path);
        }

    }

}
