package com.vaadin.copilot;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;

import com.vaadin.flow.server.VaadinServletContext;

/**
 * Acts as a bridge between Copilot and Spring specific API. Can be imported
 * into copilot and must never itself import Spring classes.
 */
public class SpringBridge {

    public record EndpointInfo(Class<?> endpointClass, Method method) {
    }

    /**
     * Returns whether Spring is available in the classpath.
     *
     * @return true if Spring is available, false otherwise
     */
    public static boolean isSpringAvailable() {
        try {
            Class.forName("org.springframework.web.context.WebApplicationContext");
            return true;
        } catch (Throwable t) {
            return false;
        }
    }

    /**
     * Returns the value of the given property from the Spring environment of the
     * given context.
     *
     * @param context
     *            the Vaadin servlet context
     * @param property
     *            the property name
     * @return the property value or null if not found
     */
    public static String getPropertyValue(VaadinServletContext context, String property) {
        return (String) call("getPropertyValue", context, property);
    }

    private static Object call(String methodName, Object... parameters) {
        try {
            Method method = Arrays.stream(getIntegration().getMethods()).filter(m -> {
                if (!m.getName().equals(methodName)) {
                    return false;
                }
                if (parameters == null || parameters.length == m.getParameterCount()) {
                    return true;
                }
                return false;
            }).findFirst().orElseThrow(NoSuchMethodException::new);
            return method.invoke(null, parameters);
        } catch (NoSuchMethodException | ClassNotFoundException | IllegalAccessException
                | InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    private static Class<?> getIntegration() throws ClassNotFoundException {
        return Class.forName("com.vaadin.copilot.SpringIntegration");
    }

    /**
     * Returns the Spring Boot application class of the given context.
     *
     * @param context
     *            the Vaadin servlet context
     * @return the Spring Boot application class or null if not found
     */
    public static Class<?> getApplicationClass(VaadinServletContext context) {
        return (Class<?>) call("getApplicationClass", context);
    }

    /**
     * Returns whether Spring Security is enabled in the given context.
     *
     * @param context
     *            the Vaadin servlet context
     * @return true if Spring Security is enabled, false otherwise
     */
    public static boolean isViewSecurityEnabled(VaadinServletContext context) {
        return call("isViewSecurityEnabled", context) == Boolean.TRUE;
    }

    /**
     * Gets the prefix used for all route paths in the application
     *
     * @param context
     *            the Vaadin servlet context
     * @return the prefix used for all route paths in the application, never ending
     *         in a "/" or an empty string if no prefix is used
     */
    public static String getUrlPrefix(VaadinServletContext context) {
        String value = getPropertyValue(context, "vaadin.url-mapping");
        if (value == null) {
            return "";
        }
        if (value.endsWith("/*")) {
            value = value.substring(0, value.length() - 2);
        }
        if (value.endsWith("/")) {
            value = value.substring(0, value.length() - 1);
        }
        return value;
    }

    /**
     * Gets a list of all endpoints / browser callables in the application.
     *
     * @return a list of endpoint info objects
     */
    public static List<EndpointInfo> getEndpoints(VaadinServletContext context) {
        return (List<EndpointInfo>) call("getEndpoints", context);
    }
}
