package javatools.database;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Types;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import javatools.administrative.D;
import javatools.parsers.DateParser;
import javatools.parsers.NumberFormatter;
import javatools.parsers.NumberParser;

/** 
Copyright 2016 Fabian M. Suchanek

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 

This abstract class provides a Wrapper for SQL datatypes. A datatype has a type code
(from java.sql.Types) and a scale. For example, INTEGER(17) is a datatype. A datatype 
can convert a datum to a suitable string. E.g. a TIMESTAMP datatype will convert the 
datum "2006-09-18" to <code>TIMESTAMP '2006-09-18 00:00:00.00'</code>. Of course,
the same SQL datatype is different for different database systems, so that different
databases have to implement different classes for the same datatype.<BR>
Example:
<PRE>
     Database d=new OracleDatabase("user","password");     
     d.getSQLType(java.sql.Types.VARCHAR).format("Bobby's")
     -&gt; 'Bobby"s'
     d=new MySQLDatabase("user","password","database");     
     d.getSQLType(java.sql.Types.VARCHAR).format("Bobby's")
     -&gt; 'Bobby\'s'
</PRE>
See <A HREF=http://troels.arvin.dk/db/rdbms/>here</A> for a comparison of database systems.
This class provides the ANSI implementations with generous conversion capabilities.<BR>
Example:
<PRE>
    SQLType.ANSItimestamp.format("13th of May 1980")
    --&gt; TIMESTAMP '1980-05-13 00:00:00.00'
</PRE>

*/

public abstract class SQLType {

  /** Holds the type code as defined in java.sql.Types */
  protected int typeCode;

  /** Holds the scale as defined in java.sql.Types */
  protected int scale = 0;

  /** Formats an object to a valid SQL literal of the given type */
  public abstract String format(Object s);

  /** Returns the java.sql.Types type */
  public int getTypeCode() {
    return (typeCode);
  }

  /** Returns the scale java.sql.Types type */
  public int getScale() {
    return (scale);
  }

  //       ANSI Types
  public static class ANSIvarchar extends SQLType {

    public ANSIvarchar(int size) {
      typeCode = Types.VARCHAR;
      scale = size;
    }

    public ANSIvarchar() {
      this(255);
    }

    @Override
    public String format(Object o) {
      String s = o.toString().replace("'", "\\'");
      if (s.length() > scale) s = s.substring(0, scale);
      return ("'" + s + "'");
    }

    @Override
    public String toString() {
      return ("VARCHAR(" + scale + ")");
    }
  }

  public static ANSIvarchar ansivarchar = new ANSIvarchar();

  public static class ANSIblob extends SQLType {

    public ANSIblob(int size) {
      typeCode = Types.BLOB;
      scale = 0;
    }

    public ANSIblob() {
      this(0);
    }

    @Override
    public String format(Object o) {
      String s = o.toString().replace("'", "\\'");
      //if(s.length()>scale) s=s.substring(0,scale);
      return ("'" + s + "'");
    }

    @Override
    public String toString() {
      return ("BLOB");
    }
  }

  public static ANSIblob ansiblob = new ANSIblob();

  public static class ANSItext extends SQLType {

    public ANSItext(int size) {
      typeCode = Types.BLOB;
      scale = 0;
    }

    public ANSItext() {
      this(0);
    }

    @Override
    public String format(Object o) {
      String s = o.toString().replace("'", "\\'");
      //if(s.length()>scale) s=s.substring(0,scale);
      return ("'" + s + "'");
    }

    @Override
    public String toString() {
      return ("TEXT");
    }
  }

  public static ANSItext ansitext = new ANSItext();

  public static class ANSIchar extends SQLType {

    public ANSIchar(int size) {
      typeCode = Types.CHAR;
      scale = 0;
    }

    public ANSIchar() {
      this(0);
    }

    @Override
    public String format(Object o) {
      if (o == null) return ("null");
      String s = o.toString();
      if (s.length() == 0) return ("null");
      char c = s.charAt(0);
      if (c == '\'') return ("'\\''");
      else return ("'" + c + "'");
    }

    @Override
    public String toString() {
      return ("CHAR");
    }
  }

  public static ANSIchar ansichar = new ANSIchar();

  public static class ANSItimestamp extends SQLType {

    public ANSItimestamp() {
      typeCode = Types.TIMESTAMP;
    }

    @Override
    public String format(Object o) {
      if (o instanceof String) {
        o = DateParser.asCalendar(DateParser.normalize(o.toString()));
      }
      if (o instanceof Calendar) {
        Calendar c = (Calendar) o;
        // ISO time is YYYY-MM-DD 'T' HH:MM:SS.MMMM
        // SQL requires dropping the 'T'
        String s = NumberFormatter.ISOtime(c).replace("T ", "");
        s = s.substring(0, s.indexOf('.'));
        return ("TIMESTAMP '" + s + "'");
      }
      if (o instanceof Date) {
        Date d = (Date) o;
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
        return "'" + format.format(d) + "'";
      }
      return (null);
    }

    @Override
    public String toString() {
      return ("TIMESTAMP");
    }
  }

  public static ANSItimestamp ansitimestamp = new ANSItimestamp();

  public static class ANSIinteger extends SQLType {

    public ANSIinteger(int size) {
      typeCode = Types.INTEGER;
      scale = size;
    }

    public ANSIinteger() {
      this(0);
    }

    @Override
    public String format(Object o) {
      if (o instanceof Double) return ("" + Math.rint((Double) o));
      if (o instanceof Float) return ("" + Math.rint((Float) o));
      if (o instanceof Integer) return ("" + ((Integer) o).longValue());
      if (o instanceof Long) return ("" + ((Long) o).longValue());
      if (o instanceof String) return ("" + NumberParser.getLong(o.toString()));
      return (null);
    }

    @Override
    public String toString() {
      if (scale == 0) return ("INTEGER");
      return ("INTEGER(" + scale + ")");
    }
  }

  public static ANSIinteger ansiinteger = new ANSIinteger();

  public static class ANSIsmallint extends SQLType {

    public ANSIsmallint(int size) {
      typeCode = Types.SMALLINT;
      scale = size;
    }

    public ANSIsmallint() {
      this(0);
    }

    @Override
    public String format(Object o) {
      if (o instanceof Double) return ("" + Math.rint((Double) o));
      if (o instanceof Float) return ("" + Math.rint((Float) o));
      if (o instanceof Integer) return ("" + ((Integer) o).longValue());
      if (o instanceof Long) return ("" + ((Long) o).longValue());
      if (o instanceof String) return ("" + NumberParser.getLong(o.toString()));
      return (null);
    }

    @Override
    public String toString() {
      if (scale == 0) return ("SMALLINT");
      return ("SMALLINT(" + scale + ")");
    }
  }

  public static ANSIsmallint ansismallint = new ANSIsmallint();

  public static class ANSIfloat extends SQLType {

    public ANSIfloat(int size) {
      typeCode = Types.FLOAT;
      scale = size;
    }

    public ANSIfloat() {
      this(0);
    }

    @Override
    public String format(Object o) {
      if (o instanceof Double || o instanceof Float || o instanceof Integer || o instanceof Long) return (o.toString());
      if (o instanceof String) return (NumberParser.getNumber(NumberParser.normalize(o.toString())));
      return (null);
    }

    @Override
    public String toString() {
      if (scale == 0) return ("FLOAT");
      return ("FLOAT(" + scale + ")");
    }
  };

  public static ANSIfloat ansifloat = new ANSIfloat();

  public static class ANSIboolean extends SQLType {

    public ANSIboolean() {
      typeCode = Types.BOOLEAN;
    }

    @Override
    public String format(Object o) {
      if (o instanceof Boolean) return (o.toString());
      if (o instanceof Float) return (Boolean.toString(((Float) o) != 0.0).toString());
      if (o instanceof Double) return (Boolean.toString(((Double) o) != 0.0));
      if (o instanceof Integer) return (Boolean.toString(((Integer) o) != 0));
      if (o instanceof Long) return (Boolean.toString(((Long) o) != 0));
      if (o instanceof String) return (Boolean.toString(Boolean.parseBoolean(o.toString())));
      return (null);
    }

    @Override
    public String toString() {
      return ("BOOLEAN");
    }
  };

  public static ANSIboolean ansiboolean = new ANSIboolean();

  public static class ANSIBigint extends SQLType {

    public ANSIBigint() {
      typeCode = Types.BIGINT;
    }

    @Override
    public String format(Object o) {
      if (o instanceof Double) return ("" + Math.rint((Double) o));
      if (o instanceof Float) return ("" + Math.rint((Float) o));
      if (o instanceof Integer) return ("" + ((Integer) o).longValue());
      if (o instanceof Long) return ("" + ((Long) o).longValue());
      if (o instanceof BigInteger) return (o.toString());
      if (o instanceof BigDecimal) return (((BigDecimal) o).toBigInteger().toString());
      if (o instanceof String) return (o.toString());
      return (null);
    }

    @Override
    public String toString() {
      return ("BIGINT");
    }
  }

  public static ANSIBigint ansibigint = new ANSIBigint();

  public static void main(String[] args) {
    D.p(ansifloat.format(0.0067));
  }

}
