package javatools.datatypes;

import java.util.AbstractSet;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javatools.administrative.D;

/**
Copyright 2016 Fabian M. Suchanek

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 

This class implements a HashMap with integer values.
 */
public class IntHashMap<K> extends AbstractSet<K> {

  /** Holds the keys */
  protected Object[] keys;

  /** Holds the values */
  protected int[] values;

  /** Holds size */
  protected int size;

  /** Constructor */
  public IntHashMap() {
    clear();
  }

  /** Creates an intHashMap with these keys set to 1 */
  public IntHashMap(K... keys) {
    this();
    for (K k : keys)
      add(k);
  }

  /** Creates an intHashMap from a list that contains keys and values in alternation*/
  @SuppressWarnings("unchecked")
  public IntHashMap<K> putAll(Object... keyValuePairs) {
    for (int i = 0; i < keyValuePairs.length; i += 2) {
      Object value = keyValuePairs[i + 1];
      if (value instanceof Integer) put((K) keyValuePairs[i], (Integer) value);
      else if (value instanceof Character) put((K) keyValuePairs[i], ((Character) value).charValue());
      else if (value instanceof Byte) put((K) keyValuePairs[i], ((Byte) value).intValue());
      else if (value instanceof Short) put((K) keyValuePairs[i], ((Short) value).intValue());
      else throw new RuntimeException("Values have to be integers");
    }
    return (this);
  }

  /** Creates an intHashMap with these keys set to 1 */
  public IntHashMap(Collection<K> keys) {
    this();
    for (K k : keys)
      add(k);
  }

  /** Creates an intHashMap with the same keys and the sizes */
  public IntHashMap(Map<K, IntHashMap<K>> map) {
    this();
    for (Entry<K, IntHashMap<K>> entry : map.entrySet())
      put(entry.getKey(), entry.getValue().size());
  }

  /** Returns an index where to store the object */
  protected int index(Object key, int len) {
    return (Math.abs(key.hashCode()) % len);
  }

  /** Returns an index where to store the object */
  protected int index(Object key) {
    return (index(key, keys.length));
  }

  /** Retrieves a value */
  public int get(Object key) {
    return (get(key, -1));
  }

  /** Finds a key, keys[find] will be NULL if non-existent */
  protected int find(Object key) {
    int i = index(key);
    while (true) {
      if (keys[i] == null) return (i);
      if (keys[i].equals(key)) return (i);
      i++;
      if (i == keys.length) i = 0;
    }
  }

  /** Retrieves a value */
  public int get(Object key, int defaultValue) {
    int pos = find(key);
    if (keys[pos] == null) return (defaultValue);
    else return (values[pos]);
  }

  /** True if value is there */
  public boolean containsKey(Object key) {
    return (keys[find(key)] != null);
  }

  /**
   * Increases a value, true for 'added new key with delta as value', false
   * for 'increased existing value'
   */
  public boolean add(K key, int delta) {
    int pos = find(key);
    if (keys[pos] == null) {
      keys[pos] = key;
      values[pos] = delta;
      size++;
      if (size > keys.length * 3 / 4) rehash();
      return (true);
    }
    values[pos] += delta;
    return (false);
  }

  /**
   * Increases a value, true for 'added new key with value 1', false for
   * 'increased existing value'
   */
  public boolean increase(K key) {
    return (add(key, 1));
  }

  /** Returns keys. Can be used only once. */
  public PeekIterator<K> keys() {
    final Object[] e = keys;
    return (new PeekIterator<K>() {

      int pos = -1;

      @SuppressWarnings("unchecked")
      @Override
      protected K internalNext() throws Exception {
        pos++;
        for (; pos < keys.length; pos++) {
          if (e[pos] != null && e[pos] != DEL) {
            return ((K) e[pos]);
          }
        }
        return (null);
      }

    });
  }

  /**
   * Adds a key, true for 'added the key as new', false for 'overwrote
   * existing value'
   */
  public boolean put(K key, int value) {
    if (put(keys, values, key, value)) {
      size++;
      if (size > keys.length * 3 / 4) rehash();
      return (true);
    }
    return (false);
  }

  /**
   * Adds a key, true for 'added the key as new', false for 'overwrote
   * existing value'
   */
  protected boolean put(Object[] keys, int[] values, Object key, int value) {
    int i = index(key, keys.length);
    while (true) {
      if (keys[i] == null || keys[i] == DEL) {
        keys[i] = key;
        values[i] = value;
        return (true);
      }
      if (keys[i].equals(key)) {
        values[i] = value;
        return (false);
      }
      i++;
      if (i == keys.length) i = 0;
    }
  }

  protected Object DEL = new Object();

  @Override
  public boolean remove(Object arg0) {
    int pos = find(arg0);
    if (keys[pos] == null) return (false);
    keys[pos] = DEL;
    values[pos] = 0;
    size--;
    rehash();
    return (true);
  }

  /** Rehashes */
  protected void rehash() {
    Object[] newKeys = new Object[keys.length * 2];
    int[] newValues = new int[keys.length * 2];
    for (int i = 0; i < keys.length; i++) {
      if (keys[i] != null && keys[i] != DEL) put(newKeys, newValues, keys[i], values[i]);
    }
    keys = newKeys;
    values = newValues;
  }

  @Override
  public Iterator<K> iterator() {
    return keys().iterator();
  }

  @Override
  public int size() {
    return size;
  }

  @Override
  public boolean add(K e) {
    return (increase(e));
  };

  @Override
  public void clear() {
    size = 0;
    keys = new Object[10];
    values = new int[10];
  }

  @Override
  public boolean contains(Object o) {
    return containsKey(o);
  }

  /** Adds all integer values up */
  public void add(IntHashMap<K> countBindings) {
    for (K key : countBindings.keys()) {
      add(key, countBindings.get(key));
    }
  }

  /** Adds all integer values up */
  public void addAll(IntHashMap<K> countBindings) {
    add(countBindings);
  }

  /** increases the counters */
  public void add(Collection<K> set) {
    for (K k : set)
      add(k);
  }

  @Override
  public String toString() {
    if (isEmpty()) return ("{}");
    StringBuilder b = new StringBuilder("{");
    int counter = 30;
    for (K key : keys()) {
      if (counter-- == 0) {
        b.append("..., ");
        break;
      }
      b.append(key).append('=').append(get(key)).append(", ");
    }
    b.setLength(b.length() - 2);
    return (b.append("}").toString());
  }

  /** returns the keys in increasing order*/
  public List<K> increasingKeys() {
    List<K> result = keys().asList();
    Collections.sort(result, new Comparator<K>() {

      @Override
      public int compare(K o1, K o2) {
        int i1 = get(o1);
        int i2 = get(o2);
        return (i1 < i2 ? -1 : i1 > i2 ? 1 : 0);
      }
    });
    return (result);
  }

  /** returns the keys in decreasing order */
  public List<K> decreasingKeys() {
    List<K> result = keys().asList();
    Collections.sort(result, new Comparator<K>() {

      @Override
      public int compare(K o1, K o2) {
        int i1 = get(o1);
        int i2 = get(o2);
        return (i1 < i2 ? 1 : i1 > i2 ? -1 : 0);
      }
    });
    return (result);
  }

  @Override
  public boolean equals(Object o) {
    if (!(o instanceof IntHashMap<?>)) return (false);
    IntHashMap<?> other = (IntHashMap<?>) o;
    if (other.size() != this.size()) return (false);
    for (int i = 0; i < keys.length; i++) {
      if (keys[i] != null && values[i] != other.get(keys[i])) return (false);
    }
    return (true);
  }

  @Override
  public int hashCode() {
    return Arrays.hashCode(values);
  }

  /** Finds the maximum value*/
  public int findMax() {
    int max = Integer.MIN_VALUE;
    for (int i = 0; i < keys.length; i++) {
      if (keys[i] != null && values[i] > max) max = values[i];
    }
    return (max);
  }

  /** Computes the sum*/
  public long computeSum() {
    long sum = 0;
    for (int i = 0; i < keys.length; i++) {
      if (keys[i] != null) sum += values[i];
    }
    return (sum);
  }

  /** Test */
  public static void main(String[] args) throws Exception {
    IntHashMap<String> m = new IntHashMap<String>();
    for (int i = 1; i < 3000; i *= 2)
      m.put("#" + i, i);
    m.put("#0", 17);
    m.remove("#300000");
    m.remove("#32");
    for (String key : m.keys())
      D.p(key, m.get(key));
  }

}
