package edu.hm.hafner.analysis.parser;

import java.util.List;

import org.apache.commons.text.StringEscapeUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import edu.hm.hafner.analysis.IssueBuilder;
import edu.hm.hafner.analysis.IssueParser;
import edu.hm.hafner.analysis.ParsingException;
import edu.hm.hafner.analysis.ReaderFactory;
import edu.hm.hafner.analysis.Report;
import edu.hm.hafner.analysis.Severity;
import edu.hm.hafner.analysis.XmlElementUtil;

/**
 * A parser for IntelliJ IDEA inspections.
 *
 * @author Alex Lopashev, alexlopashev@gmail.com
 */
public class IdeaInspectionParser extends IssueParser {
    private static final long serialVersionUID = 3307389086106375473L;

    @Override
    public Report parse(final ReaderFactory readerFactory) throws ParsingException {
        Document document = readerFactory.readDocument();

        Element rootElement = (Element) document.getElementsByTagName("problems").item(0);
        return parseProblems(XmlElementUtil.getNamedChildElements(rootElement, "problem"));
    }

    private Report parseProblems(final List<Element> elements) {
        Report problems = new Report();
        for (Element element : elements) {
            String file = getChildValue(element, "file");
            Element problemClass = XmlElementUtil.getFirstElementByTagName(element, "problem_class");
            if (problemClass != null) {
                IssueBuilder builder = new IssueBuilder().setFileName(file)
                        .setLineStart(Integer.parseInt(getChildValue(element, "line")))
                        .setCategory(StringEscapeUtils.unescapeXml(getValue(problemClass)))
                        .setMessage(StringEscapeUtils.unescapeXml(getChildValue(element, "description")))
                        .setSeverity(getPriority(problemClass.getAttribute("severity")));
                problems.add(builder.build());
            }
        }
        return problems;
    }

    private Severity getPriority(final String severity) {
        Severity priority = Severity.WARNING_LOW;
        if ("WARNING".equals(severity)) {
            priority = Severity.WARNING_NORMAL;
        }
        else if ("ERROR".equals(severity)) {
            priority = Severity.WARNING_HIGH;
        }
        return priority;
    }

    private String getValue(final Element element) {
        return element.getFirstChild().getNodeValue();
    }

    private String getChildValue(final Element element, final String childTag) {
        Element firstElement = XmlElementUtil.getFirstElementByTagName(element, childTag);
        if (firstElement != null) {
            Node child = firstElement.getFirstChild();
            if (child != null) {
                return child.getNodeValue();
            }
        }
        return "-";
    }
}

